module interpol

   use fml

   implicit none

   real(kind=MF_DOUBLE) :: bi2ic_coeff(9), bi3ic_coeff(16)

contains
!______________________________________________________________________
!
   function fun_bi2ic_real( x, y ) result( z )
      real(kind=MF_DOUBLE), intent(in) :: x, y
      real(kind=MF_DOUBLE) :: z
      z = bi2ic_coeff(1)*x**2*y**2 + bi2ic_coeff(2)*x*y**2              &
        + bi2ic_coeff(3)*y**2      + bi2ic_coeff(4)*x**2*y              &
        + bi2ic_coeff(5)*x*y       + bi2ic_coeff(6)*y                   &
        + bi2ic_coeff(7)*x**2      + bi2ic_coeff(8)*x                   &
        + bi2ic_coeff(9)
   end function fun_bi2ic_real
!______________________________________________________________________
!
   function fun_bi2ic_mf( x, y ) result( z )
      type(mfArray) :: x, y
      type(mfArray) :: z

      integer :: i, j, m, n
      real(kind=MF_DOUBLE), pointer :: x_ptr(:,:), y_ptr(:,:), z_ptr(:,:)

      m = size(x,1)
      n = size(x,2)

      z = mfZeros(m,n)
      call msPointer( x, x_ptr ) ! x_ptr => x
      call msPointer( y, y_ptr ) ! y_ptr => y
      call msPointer( z, z_ptr ) ! z_ptr => z
      do i = 1, m
         do j = 1, n
            z_ptr(i,j) = fun_bi2ic_real( x_ptr(i,j), y_ptr(i,j) )
         end do
      end do

      call msFreePointer( x, x_ptr )
      call msFreePointer( y, y_ptr )
      call msFreePointer( z, z_ptr )
      call msReturnArray( z )

   end function fun_bi2ic_mf
!______________________________________________________________________
!
   function fun_bi3ic_real( x, y ) result( z )
      real(kind=MF_DOUBLE), intent(in) :: x, y
      real(kind=MF_DOUBLE) :: z
      z = bi3ic_coeff(1)*x**3*y**3 + bi3ic_coeff(2)*x**2*y**3           &
        + bi3ic_coeff(3)*x*y**3    + bi3ic_coeff(4)*y**3                &
        + bi3ic_coeff(5)*x**3*y**2 + bi3ic_coeff(6)*x**3*y              &
        + bi3ic_coeff(7)*x**3      + bi3ic_coeff(8)*x**2*y**2           &
        + bi3ic_coeff(9)*x*y**2    + bi3ic_coeff(10)*y**2               &
        + bi3ic_coeff(11)*x**2*y   + bi3ic_coeff(12)*x*y                &
        + bi3ic_coeff(13)*y        + bi3ic_coeff(14)*x**2               &
        + bi3ic_coeff(15)*x        + bi3ic_coeff(16)
   end function fun_bi3ic_real
!______________________________________________________________________
!
   function fun_bi3ic_mf( x, y ) result( z )
      type(mfArray) :: x, y
      type(mfArray) :: z

      integer :: i, j, m, n
      real(kind=MF_DOUBLE), pointer :: x_ptr(:,:), y_ptr(:,:), z_ptr(:,:)

      m = size(x,1)
      n = size(x,2)

      z = mfZeros(m,n)
      call msPointer( x, x_ptr ) ! x_ptr => x
      call msPointer( y, y_ptr ) ! y_ptr => y
      call msPointer( z, z_ptr ) ! z_ptr => z
      do i = 1, m
         do j = 1, n
            z_ptr(i,j) = fun_bi3ic_real( x_ptr(i,j), y_ptr(i,j) )
         end do
      end do

      call msFreePointer( x, x_ptr )
      call msFreePointer( y, y_ptr )
      call msFreePointer( z, z_ptr )
      call msReturnArray( z )

   end function fun_bi3ic_mf
!______________________________________________________________________
!
end module interpol
!______________________________________________________________________
!
program test_mod_polyfun

   use fml

   use test_aux
   use lib_aux

   use interpol

   implicit none

   type(mfArray) :: x, y, z, xx, yy, pp, x_vec, y_vec, z_vec
   type(mfArray) :: x_out, y_out, z_out, tri_out, tetra_out
   type(mfArray) :: tri, tetra
   type(mfArray) :: a, b, c, xi, yi, zi, f, fi
   type(mfArray) :: p, ran, r, normr, r2
   type(mfArray) :: y_sec, y_smooth, weights
   type(mfArray) :: indices
   type(mfTriConnect) :: tri_connect
   type(mfTetraConnect) :: tetra_connect
   type(mfPLdomain) :: PL_domain
   type(mfPLdomain3D) :: PL_domain3D

   integer :: num, status, n, i, j, k, convexity
   real(kind=MF_DOUBLE) :: x1, x2, y1, y2, error, err_max
   type(mf_Int_List), allocatable :: connected_nodes(:), connected_faces(:)
   integer, allocatable :: face_tri(:,:)

   type(mfVoronoiStruct) :: voronoi

   integer :: MsgLevel_save

   ! Msg level = 3 : all messages are printed (verbose mode)
   !             2 : messages of kind 'ERROR' and 'Warning' are
   !                 printed [default]
   !             1 : only messages of kind 'ERROR' are printed
   !             0 : nothing is printed (quiet mode)
   !                 (pause are ignored !)

!!   call msSetMsgLevel(3)
!!   call msSetTrbLevel("all")

   print "()"

   ! for TRACE_MEM_DEBUG, STDERR must be redirected to STDOUT;
   ! the same thing stands for 'make check'
   call msSetStdIO( stderr=STDOUT )

!!goto 10

   ! to avoid escape sequences of color setting in the output file
   ! (default in "on")
   call msSetColoredMsg( "off" )

   ! VERSIONs ----------------------------------------------------------
   print "()"
   print *, "MF_MUESLI_VERSION = '", MF_MUESLI_VERSION, "'"
   print "()"
   print *, "MF_COMPILER_VERSION = '", MF_COMPILER_VERSION, "'"
   print "()"
   print *, "MF_COMPILATION_CONFIG = '", MF_COMPILATION_CONFIG(), "'"
   print "()"

   ! mfPolyVal ---------------------------------------------------------
   call print_separation("mfPolyVal")

   ! coeff. of a polynomial, in descending powers
   print "(/,A)", "*** test number 1 ***"
   p = [ 1.0d0, 1.0d0, 0.0d0, 0.0d0, 1.0d0 ]
   x = 1.0d0
   call msDisplay(p, "p", x, "x")
   call msDisplay(mfPolyVal(p,x), "mfPolyVal(p,x)")

   ! coeff. of a polynomial, in descending powers
   print "(/,A)", "*** test number 2 ***"
   p = .t. mf( [ 1.0d0, 1.0d0, 0.0d0, 0.0d0, 1.0d0 ] )
   x = 1.0d0*MF_I
   call msDisplay(p, "p", x, "x")
   call msDisplay(mfPolyVal(p,x), "mfPolyVal(p,x)")

   ! coeff. of a polynomial, in descending powers
   print "(/,A)", "*** test number 3 ***"
   p = .h. mf( [ (1.0d0,0.0d0), (0.0d0,1.0d0), (0.0d0,0.0d0),           &
                 (0.0d0,1.0d0), (1.0d0,0.0d0) ] )*MF_I
   x = 1.0d0
   call msDisplay(p, "p", x, "x")
   call msDisplay(mfPolyVal(p,x), "mfPolyVal(p,x)")

   ! coeff. of a polynomial, in descending powers
   print "(/,A)", "*** test number 4 ***"
   p = .h. mf( [ (1.0d0,0.0d0), (0.0d0,1.0d0), (0.0d0,0.0d0),           &
                 (0.0d0,1.0d0), (1.0d0,0.0d0) ] )*MF_I
   x = mf( [ 1.0d0, -1.0d0 ] )*MF_I
   call msDisplay(p, "p", x, "x")
   call msDisplay(mfPolyVal(p,x), "mfPolyVal(p,x)")

   ! mfPolyFit ---------------------------------------------------------
   call print_separation("mfPolyFit")

   ! data generation
   n = 250
   x = .t. mfLinSpace(0.0d0,2.5d0,250)
   y = mfCos(x**2)
   call msDisplay(size(x),"size of data")

   print "(/,A)", "*** test number 5 ***"
   p = mfPolyFit( x, y, 4 ) ! non singular case (Normal Equations)
   call msDisplay(p, "polynomial coefficients for fourth degree")

   print "(/,A)", "*** test number 6 ***"
   p = mfPolyFit( x, y, 10 ) ! nearly singular case (QR used)
   call msDisplay(p, "polynomial coefficients for tenth degree")

   ! msPolyFit ---------------------------------------------------------
   call print_separation("msPolyFit")

   print "(/,A)", "*** test number 7 ***"
   call msPolyFit( mfOut(p,normr), x, y, 4 ) ! non singular case (Normal Equations)
   call msDisplay(p, "polynomial coefficients for fourth degree")
   call msDisplay(normr, "norm-2 of the residuals")

   print "(/,A)", "*** test number 8 ***"
   call msPolyFit( mfOut(p,normr,r2), x, y, 4 ) ! non singular case (Normal Equations)
   call msDisplay(p, "polynomial coefficients for fourth degree")
   call msDisplay(normr, "norm-2 of the residuals")
   call msDisplay(r2, "correlation coefficient: r2")

   print "(/,A)", "*** test number 9 ***"
   call msPolyFit( mfOut(p,normr,r2), x, y, 10 ) ! nearly singular case (QR used)
   call msDisplay(p, "polynomial coefficients for tenth degree")
   call msDisplay(normr, "norm-2 of the residuals")
   call msDisplay(r2, "correlation coefficient: r2")

   ! mfSpline ----------------------------------------------------------
   call print_separation("mfSpline")

   print "(/,A)", "*** test number 10 ***"
   x = .t. mfLinspace( 0.0d0, 1.0d0, 5 )
   y = mfZeros(5,1)
   do i = 1, 5
      call msSet( (-1.0d0)**i, y, i )
   end do
   call msDisplay(x, "x")
   call msDisplay(y, "y")

   call msDisplay( mfSpline( x, y ), "mfSpline( x, y )" )

   print "(/,A)", "*** test number 11 ***"
   call msDisplay( mfSpline( x, y, periodic=.true. ),                   &
                  "mfSpline( x, y, periodic=.true. )" )

   print "(/,A)", "*** test number 12 ***"

   call msDisplay( mfSpline( x, y, BC_type_1=1, BC_val_1=0.0d0 ),       &
                  "mfSpline( x, y, BC_type_1=1, BC_val_1=0.0d0 )" )

   print "(/,A)", "*** test number 13 ***"

   call msDisplay( mfSpline( x, y, BC_type_2=1, BC_val_2=0.0d0 ),       &
                  "mfSpline( x, y, BC_type_2=1, BC_val_2=0.0d0 )" )

   print "(/,A)", "*** test number 14 ***"

   call msDisplay( mfSpline( x, y, BC_type_1=1, BC_val_1=0.0d0,         &
                                   BC_type_2=1, BC_val_2=0.0d0 ),       &
                  "mfSpline( x, y, BC_type_1=1, BC_val_1=0.0d0, BC_type_2=1, BC_val_2=0.0d0 )" )

   print "(/,A)", "*** test number 15 ***"
   xx = .t. mfLinspace( -0.2d0, 1.2d0, 8 )
   call msDisplay(xx, "xx")
   call msDisplay( mfSpline( x, y, xx ), "mfSpline( x, y, xx )" )

   print "(/,A)", "*** test number 16 ***"
   call msDisplay( mfSpline( x, y, xx, periodic=.true. ),               &
                  "mfSpline( x, y, xx, periodic=.true. )" )

   ! calling with tempo args
   print "(/,A)", "*** test number 17 ***"

   call msDisplay( mfSpline( mf([0.0d0,1.0d0,2.0d0]), mf([0.0d0,1.0d0,0.0d0]) ), &
                  "mfSpline( mf([0.0d0,1.0d0,2.0d0]), mf([0.0d0,1.0d0,0.0d0]) )" )

   ! xx may have any shape
   print "(/,A)", "*** test number 18 ***"

   call msDisplay( mfSpline( mf([1.0d0,5.0d0,9.0d0]),                   &
                             mf([0.0d0,1.0d0,0.0d0]),                   &
                             mfMagic(3) ),                              &
                  "mfSpline( mf([1.0d0,5.0d0,9.0d0]), mf([0.0d0,1.0d0,0.0d0]), mfMagic(3) )" )

   ! msSpline ----------------------------------------------------------
   call print_separation("msSpline")

   ! using weights for smoothing data

   ! unique weight for all points
   print "(/,A)", "*** test number 19 ***"
   x = .t. mfLinspace( 0.0d0, 1.0d0, 5 )
   y = mfZeros(5,1)
   do i = 1, 5
      call msSet( (-1.0d0)**i, y, i )
   end do
   call msDisplay(x, "x")
   call msDisplay(y, "y")

   weights = 1000
   call msSpline( mfOut(y_smooth,y_sec), x, y, weights )
   call msDisplay( weights, "weights", y_sec, "y_sec", y_smooth, "y_smooth" )
   print *, "  -----"

   weights = 10
   call msSpline( mfOut(y_smooth,y_sec), x, y, weights )
   call msDisplay( weights, "weights", y_sec, "y_sec", y_smooth, "y_smooth" )
   print *, "  -----"

   weights = 0.1
   call msSpline( mfOut(y_smooth,y_sec), x, y, weights )
   call msDisplay( weights, "weights", y_sec, "y_sec", y_smooth, "y_smooth" )
   print *, "  -----"

   weights = 0.001
   call msSpline( mfOut(y_smooth,y_sec), x, y, weights )
   call msDisplay( weights, "weights", y_sec, "y_sec", y_smooth, "y_smooth" )

   ! unique weight for all points, but periodic spline
   print "(/,A)", "*** test number 20 ***"
   print "(/,A)", " periodic case"

   weights = 1000
   call msSpline( mfOut(y_smooth,y_sec), x, y, weights, periodic=.true. )
   call msDisplay( weights, "weights", y_sec, "y_sec", y_smooth, "y_smooth" )
   print *, "  -----"

   weights = 10
   call msSpline( mfOut(y_smooth,y_sec), x, y, weights, periodic=.true. )
   call msDisplay( weights, "weights", y_sec, "y_sec", y_smooth, "y_smooth" )
   print *, "  -----"

   weights = 0.1
   call msSpline( mfOut(y_smooth,y_sec), x, y, weights, periodic=.true. )
   call msDisplay( weights, "weights", y_sec, "y_sec", y_smooth, "y_smooth" )
   print *, "  -----"

   weights = 0.001
   call msSpline( mfOut(y_smooth,y_sec), x, y, weights, periodic=.true. )
   call msDisplay( weights, "weights", y_sec, "y_sec", y_smooth, "y_smooth" )

   ! different weights
   print "(/,A)", "*** test number 21 ***"
   x = [ 0.0d0, 1.0d0, 2.0d0, 3.0d0, 4.0d0,  5.0d0, 6.0d0 ]
   y = [ 0.0d0, 1.0d0, 0.0d0, 5.0d0, 0.0d0, 10.0d0, 0.0d0 ]
   weights = [ MF_INF, 1.0d3, MF_INF, 1.0d2, MF_INF,  1.0d1, MF_INF ]
   call msDisplay(x, "x")
   call msDisplay(y, "y")
   call msFormat("long")
   call msDisplay(weights, "weights")
   call msFormat()

   call msSpline( mfOut(y_smooth,y_sec), x, y, weights )
   call msDisplay( y_sec, "y_sec", y_smooth, "y_smooth" )

   ! different weights, but periodic spline
   print "(/,A)", "*** test number 22 ***"
   print "(/,A)", " periodic case"

   call msSpline( mfOut(y_smooth,y_sec), x, y, weights, periodic=.true. )
   call msDisplay( y_sec, "y_sec", y_smooth, "y_smooth" )

   ! mfPPVal -----------------------------------------------------------
   call print_separation("mfPPVal")

   x = .t. mfLinspace( 0.0d0, 1.0d0, 3 )
   y = .t. mf( [ 0.0d0, 1.0d0, 0.0d0 ] )
   call msDisplay(x, "x")
   call msDisplay(y, "y")

   pp = mfSpline( x, y )
   call msDisplay( pp, "pp = mfSpline( x, y )" )

   xx = .t. mfLinspace( -0.25d0, 1.25d0, 7 )
   call msDisplay(xx, "xx")

   print "(/,A)", "*** test number 23 ***"
   yy = mfPPVal( x, y, pp, xx )
   call msDisplay( yy, "yy = mfPPVal( x, y, pp, xx )" )

   ! calling with tempo args
   print "(/,A)", "*** test number 24 ***"
   yy = mfPPVal( x, y, pp, mf(0.5d0) )
   call msDisplay( yy, "yy = mfPPVal( x, y, pp, mf(0.5d0) )" )

   ! xx may have any shape
   print "(/,A)", "*** test number 25 ***"
   yy = mfPPVal( x, y, pp, (mfMagic(3)-4.0)/5.0 )
   call msDisplay( yy, "yy = mfPPVal( x, y, pp, mfMagic(3) )" )

   ! mfPPDer -----------------------------------------------------------
   call print_separation("mfPPDer")

   x = .t. mfLinspace( 0.0d0, 1.0d0, 3 )
   y = .t. mf( [ 0.0d0, 1.0d0, 0.0d0 ] )
   call msDisplay(x, "x")
   call msDisplay(y, "y")

   pp = mfSpline( x, y )
   call msDisplay( pp, "pp = mfSpline( x, y )" )

   xx = .t. mfLinspace( -0.25d0, 1.25d0, 7 )
   call msDisplay(xx, "xx")

   print "(/,A)", "*** test number 26 ***"
   yy = mfPPDer( x, y, pp, xx )
   call msDisplay( yy, "yy = mfPPDer( x, y, pp, xx )" )

   ! calling with tempo args
   print "(/,A)", "*** test number 27 ***"
   yy = mfPPDer( x, y, pp, mf(0.5d0) )
   call msDisplay( yy, "yy = mfPPDer( x, y, pp, mf(0.5d0) )" )

   ! xx may have any shape
   print "(/,A)", "*** test number 28 ***"
   yy = mfPPDer( x, y, pp, (mfMagic(3)-4.0)/5.0 )
   call msDisplay( yy, "yy = mfPPDer( x, y, pp, mfMagic(3) )" )

   ! mfInterp1 ---------------------------------------------------------
   call print_separation("mfInterp1")

   print *, " Linear interpolation"
   ! Linear interpolation (default: order=1)

   x = mfLinspace( 0.0d0, 1.0d0, 5 )
   call msDisplay( x, "x" )
   print *, "y = x**2"
   y = x**2

   ! xi is a point :
   print "(/,A)", "*** test number 29 ***"
   xi = 0.1234d0
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi )
   call msDisplay( yi, "interpolated data at a point" )
   print "()"

   ! xi is a point :
   print "(/,A)", "*** test number 30 ***"
   xi = -1.0d0
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi )
   call msDisplay( yi, "interpolated data at a point (out-of-range -> NaN expected)" )
   print "()"

   ! xi is a point :
   print "(/,A)", "*** test number 31 ***"
   xi = -1.0d0
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, extrapol=MF_INF )
   call msDisplay( yi, "interpolated data at a point (out-of-range -> Inf expected)" )
   print "()"

   ! xi is a point :
   print "(/,A)", "*** test number 32 ***"
   xi = -1.0d0
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, extrapol=.true. )
   call msDisplay( yi, "interpolated data at a point (using extrapolation)" )
   print "()"

   ! xi is a vector :
   print "(/,A)", "*** test number 33 ***"
   xi = mfLinspace( -0.5d0, 1.5d0, 4 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi )
   call msDisplay( yi, "interpolated data at many points (two NaNs expected)" )
   print "()"

   ! xi is a vector :
   print "(/,A)", "*** test number 34 ***"
   xi = mfLinspace( -0.5d0, 1.5d0, 4 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, extrapol=MF_INF )
   call msDisplay( yi, "interpolated data at many points (two Infs expected)" )
   print "()"

   ! xi is a vector :
   print "(/,A)", "*** test number 35 ***"
   xi = mfLinspace( -0.5d0, 1.5d0, 4 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, extrapol=.true. )
   call msDisplay( yi, "interpolated data at many points (using extrapolation)" )
   print "()"

   x = .t. mfLinspace( 0.0d0, 1.0d0, 5 )
   call msDisplay( x, "x" )
   print *, "y = x**2"
   y = x**2

   ! xi is a point :
   print "(/,A)", "*** test number 36 ***"
   xi = 0.1234d0
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi )
   call msDisplay( yi, "interpolated data at a point" )
   print "()"

   ! xi is a vector :
   print "(/,A)", "*** test number 37 ***"
   xi = .t. mfLinspace( -0.5d0, 1.5d0, 4 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi )
   call msDisplay( yi, "interpolated data at many points" )
   print "()"

   ! x contains decreasing values
   x = mfLinspace( 1.0d0, 0.0d0, 5 )
   call msDisplay( x, "x" )
   print *, "y = x**2"
   y = x**2

   ! xi is a point :
   print "(/,A)", "*** test number 38 ***"
   xi = 0.1234d0
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi )
   call msDisplay( yi, "interpolated data at a point" )
   print "()"

   ! xi is a point :
   print "(/,A)", "*** test number 39 ***"
   xi = -1.0d0
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, extrapol=.true. )
   call msDisplay( yi, "interpolated data at a point (using extrapolation)" )
   print "()"

   print *, " Nearest interpolation"
   ! Nearest interpolation (order=0)

   x = .t. mfLinspace( 0.0d0, 1.0d0, 5 )
   call msDisplay( x, "x" )
   print *, "y = x**2"
   y = x**2

   ! xi is a vector :
   print "(/,A)", "*** test number 40 ***"
   xi = .t. mfLinspace( 0.05d0, 0.95d0, 4 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, order=0 )
   call msDisplay( yi, "interpolated data at many points" )
   yi = xi**2
   call msDisplay( yi, "exact values" )
   print "()"

   ! xi is a vector :
   print "(/,A)", "*** test number 41 ***"
   xi = .t. mfLinspace( -0.15d0, 1.15d0, 7 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, order=0 )
   call msDisplay( yi, "interpolated data at many points (2 NaNs expected)" )
   yi = xi**2
   call msDisplay( yi, "exact values" )
   print "()"

   ! xi is a vector :
   print "(/,A)", "*** test number 42 ***"
   xi = .t. mfLinspace( -0.15d0, 1.15d0, 7 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, order=0, extrapol=MF_INF )
   call msDisplay( yi, "interpolated data at many points (2 Infs expected)" )
   yi = xi**2
   call msDisplay( yi, "exact values" )
   print "()"

   ! xi is a vector :
   print "(/,A)", "*** test number 43 ***"
   xi = .t. mfLinspace( -0.15d0, 1.15d0, 7 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, order=0, extrapol=.true. )
   call msDisplay( yi, "interpolated data at many points (using extrapolation)" )
   yi = xi**2
   call msDisplay( yi, "exact values" )
   print "()"

   print *, " Quadratic interpolation"
   ! Quadratic interpolation (order=2)

   x = .t. mfLinspace( 0.0d0, 1.0d0, 5 )
   call msDisplay( x, "x" )
   print *, "y = x**2"
   y = x**2

   ! xi is a vector :
   print "(/,A)", "*** test number 44 ***"
   xi = .t. mfLinspace( 0.05d0, 0.95d0, 4 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, order=2 )
   call msDisplay( yi, "interpolated data at many points" )
   yi = xi**2
   call msDisplay( yi, "exact values" )
   print "()"

   ! xi is a vector :
   print "(/,A)", "*** test number 45 ***"
   xi = .t. mfLinspace( -0.15d0, 1.15d0, 7 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, order=2 )
   call msDisplay( yi, "interpolated data at many points (2 NaNs expected)" )
   yi = xi**2
   call msDisplay( yi, "exact values" )
   print "()"

   ! xi is a vector :
   print "(/,A)", "*** test number 46 ***"
   xi = .t. mfLinspace( -0.15d0, 1.15d0, 7 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, order=2, extrapol=MF_INF )
   call msDisplay( yi, "interpolated data at many points (2 Infs expected)" )
   yi = xi**2
   call msDisplay( yi, "exact values" )
   print "()"

   ! xi is a vector :
   print "(/,A)", "*** test number 47 ***"
   xi = .t. mfLinspace( -0.15d0, 1.15d0, 7 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, order=2, extrapol=.true. )
   call msDisplay( yi, "interpolated data at many points (using extrapolation)" )
   yi = xi**2
   call msDisplay( yi, "exact values" )
   print "()"

   print *, " Cubic interpolation"
   ! Cubic interpolation (order=3)

   x = .t. mfLinspace( 0.0d0, 1.0d0, 6 )
   call msDisplay( x, "x" )
   print *, "y = x**3"
   y = x**3

   ! xi is a vector :
   print "(/,A)", "*** test number 48 ***"
   xi = .t. mfLinspace( 0.05d0, 0.95d0, 4 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, order=3 )
   call msDisplay( yi, "interpolated data at many points" )
   yi = xi**3
   call msDisplay( yi, "exact values" )
   print "()"

   ! xi is a vector :
   print "(/,A)", "*** test number 49 ***"
   xi = .t. mfLinspace( -0.15d0, 1.15d0, 7 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, order=3 )
   call msDisplay( yi, "interpolated data at many points (2 NaNs expected)" )
   yi = xi**3
   call msDisplay( yi, "exact values" )
   print "()"

   ! xi is a vector :
   print "(/,A)", "*** test number 50 ***"
   xi = .t. mfLinspace( -0.15d0, 1.15d0, 7 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, order=3, extrapol=MF_INF )
   call msDisplay( yi, "interpolated data at many points (2 Infs expected)" )
   yi = xi**3
   call msDisplay( yi, "exact values" )
   print "()"

   ! xi is a vector :
   print "(/,A)", "*** test number 51 ***"
   xi = .t. mfLinspace( -0.15d0, 1.15d0, 7 )
   call msDisplay( xi, "xi" )
   yi = mfInterp1( x, y, xi, order=3, extrapol=.true. )
   call msDisplay( yi, "interpolated data at many points (using extrapolation)" )
   yi = xi**3
   call msDisplay( yi, "exact values" )

   ! mfInterp2 ---------------------------------------------------------
   call print_separation("mfInterp2")

   print *, " Nearest interpolation"
   ! Nearest interpolation (order=0)

   a =     mfLinspace( 0.0d0, 1.0d0, 4 )
   b = .t. mfLinspace( 0.0d0, 1.0d0, 5 )
   call msMeshGrid( mfOut(x,y), a, b )
   call msDisplay( x, "x", y, "y" )
   print *, "z = x"
   z = x

   ! (xi,yi) is a point :
   print "(/,A)", "*** test number 52 ***"
   xi = 0.1234d0
   yi = 0.9876d0
   call msDisplay( xi, "xi", yi, "yi" )
   zi = mfInterp2( x, y, z, xi, yi, order=0 )
   call msDisplay( zi, "interpolated data at a point" )
   print "()"

   ! (xi,yi) are vectors :
   print "(/,A)", "*** test number 53 ***"
   xi = mfColon( -0.2d0, 1.21d0, 0.2d0, tol=0.001d0 )
   yi = mfColon( -0.2d0, 1.21d0, 0.2d0, tol=0.001d0 )
   call msDisplay( xi, "xi", yi, "yi" )
   zi = mfInterp2( x, y, z, xi, yi, order=0 )
   call msDisplay( zi, "interpolated data at points on a line" )

   ! (xi,yi) are matrices :
   print "(/,A)", "*** test number 54 ***"
   a =     mfColon( -0.2d0, 1.21d0, 0.2d0, tol=0.001d0 )
   b = .t. mfColon( -0.2d0, 1.21d0, 0.2d0, tol=0.001d0 )
   call msDisplay( a, "a", b, "b" )
   call msMeshGrid( mfOut(xi,yi), a, b )
   MsgLevel_save = mfGetMsgLevel()
   call msSetMsgLevel(1)
   zi = mfInterp2( x, y, z, xi, yi, order=0 )
   call msSetMsgLevel(MsgLevel_save)
   call msDisplay( zi, "interpolated data at points on a 2D grid" )

   print *, " Bilinear interpolation"
   ! Bilinear interpolation (default: order=1)

   a =     mfLinspace( 0.0d0, 1.0d0, 4 )
   b = .t. mfLinspace( 0.0d0, 1.0d0, 5 )
   call msMeshGrid( mfOut(x,y), a, b )
   call msDisplay( x, "x", y, "y" )
   print *, "z = x"
   z = x

   ! (xi,yi) is a point :
   print "(/,A)", "*** test number 55 ***"
   xi = 0.1234d0
   yi = 0.9876d0
   call msDisplay( xi, "xi", yi, "yi" )
   zi = mfInterp2( x, y, z, xi, yi )
   call msDisplay( zi, "interpolated data at a point" )
   print "()"

   ! (xi,yi) are vectors :
   print "(/,A)", "*** test number 56 ***"
   xi = mfColon( -0.2d0, 1.21d0, 0.2d0, tol=0.001d0 )
   yi = mfColon( -0.2d0, 1.21d0, 0.2d0, tol=0.001d0 )
   call msDisplay( xi, "xi", yi, "yi" )
   zi = mfInterp2( x, y, z, xi, yi )
   call msDisplay( zi, "interpolated data at points on a line" )

   ! (xi,yi) are matrices :
   print "(/,A)", "*** test number 57 ***"
   a =     mfColon( -0.2d0, 1.21d0, 0.2d0, tol=0.001d0 )
   b = .t. mfColon( -0.2d0, 1.21d0, 0.2d0, tol=0.001d0 )
   call msDisplay( a, "a", b, "b" )
   call msMeshGrid( mfOut(xi,yi), a, b )
   MsgLevel_save = mfGetMsgLevel()
   call msSetMsgLevel(1)
   zi = mfInterp2( x, y, z, xi, yi )
   call msSetMsgLevel(MsgLevel_save)
   call msDisplay( zi, "interpolated data at points on a 2D grid" )

   print *, " Biquadratic interpolation"
   ! Biquadratic interpolation (order=2)

   ! generate a random biquadratic function
   bi2ic_coeff = mfRand( 9, 1 )
   call msDisplay( .t.mf(bi2ic_coeff), "bi2ic_coeff" )

   x1 = mfRandN(1)
   x2 = mfRandN(1)
   y1 = mfRandN(1)
   y2 = mfRandN(1)
   x_vec =     mfLinSpace( x1, x2, 6 )
   y_vec = .t. mfLinSpace( y1, y2, 4 )
   call msMeshGrid( mfOut(x,y), x_vec, y_vec )
   call msDisplay( x, "x" )
   call msDisplay( y, "y" )
   z = fun_bi2ic_mf( x, y )
   call msDisplay( z, "z" )

   ! (xi,yi) is a point :
   print "(/,A)", "*** test number 58 ***"
   err_max = 0.0d0
   do k = 1, 100
      xi = (x2-x1)*mfRand(1) + x1
      yi = (y2-y1)*mfRand(1) + y1
      zi = mfInterp2( x, y, z, xi, yi, order=2 )
      error =  mfAbs( zi - fun_bi2ic_mf(xi,yi) )
      if( error > err_max ) err_max = error
   end do
   if( err_max < 1.0d-12 ) then
      print *, "   test ok"
   else
      print *, "   test FAILED    <-----------------------------"
   end if

   ! (xi,yi) are vectors :
   print "(/,A)", "*** test number 59 ***"
   err_max = 0.0d0
   do k = 1, 100
      xi = (x2-x1)*mfRand(1,5) + x1
      yi = (y2-y1)*mfRand(1,5) + y1
      zi = mfInterp2( x, y, z, xi, yi, order=2 )
      error =  mfNorm( zi - fun_bi2ic_mf(xi,yi) )
      if( error > err_max ) err_max = error
   end do
   if( err_max < 1.0d-12 ) then
      print *, "   test ok"
   else
      print *, "   test FAILED    <-----------------------------"
   end if

   ! (xi,yi) are matrices :
   print "(/,A)", "*** test number 60 ***"
   err_max = 0.0d0
   do k = 1, 100
      xi = (x2-x1)*mfRand(5,5) + x1
      yi = (y2-y1)*mfRand(5,5) + y1
      zi = mfInterp2( x, y, z, xi, yi, order=2 )
      error =  mfNorm( zi - fun_bi2ic_mf(xi,yi) )
      if( error > err_max ) err_max = error
   end do
   if( err_max < 1.0d-12 ) then
      print *, "   test ok"
   else
      print *, "   test FAILED    <-----------------------------"
   end if

   print *, " Bicubic interpolation"
   ! Bicubic interpolation (order=3)

   ! generate a random bicubic function
   bi3ic_coeff = mfRand( 16, 1 )
   call msDisplay( .t.mf(bi3ic_coeff), "bi3ic_coeff" )

   x1 = mfRandN(1)
   x2 = mfRandN(1)
   y1 = mfRandN(1)
   y2 = mfRandN(1)
   x_vec =     mfLinSpace( x1, x2, 7 )
   y_vec = .t. mfLinSpace( y1, y2, 5 )
   call msMeshGrid( mfOut(x,y), x_vec, y_vec )
   call msDisplay( x, "x" )
   call msDisplay( y, "y" )
   z = fun_bi3ic_mf( x, y )
   call msDisplay( z, "z" )

   ! (xi,yi) is a point :
   print "(/,A)", "*** test number 61 ***"
   err_max = 0.0d0
   do k = 1, 100
      xi = (x2-x1)*mfRand(1) + x1
      yi = (y2-y1)*mfRand(1) + y1
      zi = mfInterp2( x, y, z, xi, yi, order=3 )
      error =  mfAbs( zi - fun_bi3ic_mf(xi,yi) )
      if( error > err_max ) err_max = error
   end do
   if( err_max < 1.0d-12 ) then
      print *, "   test ok"
   else
      print *, "   test FAILED    <-----------------------------"
   end if

   ! (xi,yi) are vectors :
   print "(/,A)", "*** test number 62 ***"
   err_max = 0.0d0
   do k = 1, 100
      xi = (x2-x1)*mfRand(1,5) + x1
      yi = (y2-y1)*mfRand(1,5) + y1
      zi = mfInterp2( x, y, z, xi, yi, order=3 )
      error =  mfNorm( zi - fun_bi3ic_mf(xi,yi) )
      if( error > err_max ) err_max = error
   end do
   if( err_max < 1.0d-12 ) then
      print *, "   test ok"
   else
      print *, "   test FAILED    <-----------------------------"
   end if

   ! (xi,yi) are matrices :
   print "(/,A)", "*** test number 63 ***"
   err_max = 0.0d0
   do k = 1, 100
      xi = (x2-x1)*mfRand(5,5) + x1
      yi = (y2-y1)*mfRand(5,5) + y1
      zi = mfInterp2( x, y, z, xi, yi, order=3 )
      error =  mfNorm( zi - fun_bi3ic_mf(xi,yi) )
      if( error > err_max ) err_max = error
   end do
   if( err_max < 1.0d-12 ) then
      print *, "   test ok"
   else
      print *, "   test FAILED    <-----------------------------"
   end if
   print "()"

   ! mfDelaunay --------------------------------------------------------
   call print_separation("mfDelaunay")

   x = [ 0.0d0, 10.0d0, 10.0d0, 0.0d0 ]
   y = [ 0.0d0,  0.0d0,  1.0d0, 1.0d0 ]
   call msDisplay( .t.(x .vc. y), "nodes coord." )

   print "(/,A)", "*** test number 64 ***"
   print "(/,1X,A)", "Delaunay triangulation (no constraint)."
   tri = mfDelaunay( x, y )
   print "(/,1X,A,I0)", "nb of triangles = ", size(tri,1)
   call msDisplay( tri, "triangles" )

   ! msDelaunay --------------------------------------------------------
   call print_separation("msDelaunay")

   x = [ 0.0d0, 10.0d0, 10.0d0, 0.0d0 ]
   y = [ 0.0d0,  0.0d0,  1.0d0, 1.0d0 ]
   call msDisplay( .t.(x .vc. y), "nodes coord." )

   print "(/,A)", "*** test number 65 ***"
   print "(/,1X,A)", "Constrained Delaunay triangulation with minimum angle imposed."
   call msDelaunay( mfOut(x_out,y_out,tri_out), x, y, theta_min=20.0d0 )
   print "(/,1X,A,I0)", "nb of triangles = ", size(tri_out,1)
   call msDisplay( tri_out, "triangles" )

   print "(/,A)", "*** test number 66 ***"
   print "(/,1X,A)", "Constrained Delaunay triangulation with maximum area imposed."
   call msDelaunay( mfOut(x_out,y_out,tri_out), x, y, area_max=1.0d0 )
   print "(/,1X,A,I0)", "nb of triangles = ", size(tri_out,1)
   call msDisplay( tri_out, "triangles" )

   print "(/,A)", "*** test number 67 ***"
   print "(/,1X,A)", "Constrained Delaunay triangulation of a PL-domain."
   ! L-shaped domain
   allocate( PL_domain%n_xy(6,2) )
   PL_domain%n_xy(:,1) = [ 0.0d0, 2.0d0, 2.0d0, 1.0d0, 1.0d0, 0.0d0 ] ! x
   PL_domain%n_xy(:,2) = [ 0.0d0, 0.0d0, 1.0d0, 1.0d0, 2.0d0, 2.0d0 ] ! y
   allocate( PL_domain%edge_n(6,2) )
   PL_domain%edge_n(:,1) = [ 1, 2, 3, 4, 5, 6 ]
   PL_domain%edge_n(:,2) = [ 2, 3, 4, 5, 6, 1 ]
   allocate( PL_domain%holes_xy(1,2) )
   PL_domain%holes_xy(1,1) = 1.1d0 ! x
   PL_domain%holes_xy(1,2) = 1.1d0 ! y

   call msDisplay( PL_domain%n_xy, "(L-shaped domain) nodes coord." )

   call msDelaunay( mfOut(x_out,y_out,tri_out), PL_domain,            &
                    theta_min=0.0d0 ) ! means no constraint

   print "(/,1X,A,I0)", "nb of triangles = ", size(tri_out,1)
   call msDisplay( tri_out, "triangles" )

   ! msCheckPLDomainValidity -------------------------------------------
   call print_separation("msCheckPLDomainValidity")

   print "(/,A)", "*** test number 68 ***"
   call msCheckPLDomainValidity( PL_domain, status )
   if( status == 1 ) then
      print *, "  -> the provided PL domain seems to be valid."
   else
      print *, "  -> the provided PL domain is NOT valid."
      stop
   end if
   print *

   ! msPrintPLdomain ---------------------------------------------------
   call print_separation("msPrintPLdomain")

   print "(/,A)", "*** test number 69 ***"
   call msPrintPLdomain( PL_domain )

   call msRelease( PL_domain )
   print *

   ! msBuildTriConnect -------------------------------------------------
   call print_separation("msBuildTriConnect")

   x = [ 0.0d0, 1.0d0, 1.0d0, 0.0d0, 0.5d0 ]
   y = [ 0.0d0, 0.0d0, 1.0d0, 1.0d0, 0.5d0 ]
   tri = mfDelaunay( x, y )

   print "(/,A)", "*** test number 70 ***"
   call msBuildTriConnect( x, y, tri, tri_connect )
   print *, "TriSearch : Prep is ok."
   print "()"

   ! msPrintTriConnect -------------------------------------------------
   call print_separation("msPrintTriConnect")

   print "(/,A)", "*** test number 71 ***"
   call msPrintTriConnect( tri_connect, short_info=.true. )
   print "()"

   print "(/,A)", "*** test number 72 ***"
   call msCheckDomainConvexity( tri_connect )
   call msPrintTriConnect( tri_connect, short_info=.true. )
   print "()"

   print "(/,A)", "*** test number 73 ***"
   call msPrintTriConnect( tri_connect )
   print "()"

   ! msExtractTriConnect -------------------------------------------------
   call print_separation("msExtractTriConnect")

   ! Build a PL domain with three internal holes
   ! L-shaped domain
   allocate( PL_domain%n_xy(18,2) )
   PL_domain%n_xy(:,1) = [ 0.0d0, 2.0d0, 2.0d0, 1.0d0, 1.0d0, 0.0d0,    &
                           0.2d0, 0.8d0, 0.8d0, 0.2d0,                  &
                           0.2d0, 0.8d0, 0.8d0, 0.2d0,                  &
                           1.2d0, 1.8d0, 1.8d0, 1.2d0 ] ! x
   PL_domain%n_xy(:,2) = [ 0.0d0, 0.0d0, 1.0d0, 1.0d0, 2.0d0, 2.0d0,    &
                           1.2d0, 1.2d0, 1.8d0, 1.8d0,                  &
                           0.2d0, 0.2d0, 0.8d0, 0.8d0,                  &
                           0.2d0, 0.2d0, 0.8d0, 0.8d0 ] ! y
   allocate( PL_domain%edge_n(18,2) )
   PL_domain%edge_n(:,1) = [ 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18 ]
   PL_domain%edge_n(:,2) = [ 2, 3, 4, 5, 6, 1, 8, 9, 10, 7, 12, 13, 14, 11, 16, 17, 18, 15 ]
   allocate( PL_domain%holes_xy(4,2) )
   PL_domain%holes_xy(1,1) = 1.25d0 ! x
   PL_domain%holes_xy(1,2) = 1.25d0 ! y
   PL_domain%holes_xy(2,1) = 0.5d0 ! x
   PL_domain%holes_xy(2,2) = 1.5d0 ! y
   PL_domain%holes_xy(3,1) = 0.5d0 ! x
   PL_domain%holes_xy(3,2) = 0.5d0 ! y
   PL_domain%holes_xy(4,1) = 1.5d0 ! x
   PL_domain%holes_xy(4,2) = 0.5d0 ! y

   call msDelaunay( mfOut(x,y,tri), PL_domain, theta_min=28.6d0,        &
                    area_max=0.1d0 )

   call msBuildTriConnect( x, y, tri, tri_connect )

   print "(/,A)", "*** test number 74 ***"
   call msExtractTriConnect( tri_connect,                               &
                             face_tri=face_tri, convexity=convexity )

   print "(/,2X,A,I0)", "Domain convexity = ", convexity

   ! Get boundary faces
   indices = mfFind( mf(face_tri(:,2)) <= 0 )

   ! Show the virtual triangle out of the boundary
   call msDisplay( mfGet(mf(face_tri(:,2)),indices),                    &
                   "face_tri only for bondary faces" )

   call msCheckDomainConvexity( tri_connect )

   print "(/,A)", "*** test number 75 ***"
   call msExtractTriConnect( tri_connect,                               &
                             face_tri=face_tri, convexity=convexity )

   print "(/,2X,A,I0)", "Domain convexity = ", convexity

   ! Show the virtual triangle out of the boundary
   y = mfGet(mf(face_tri(:,2)),indices)
   call msDisplay( y, "face_tri only for bondary faces" )

   print "(/,2X,A,I0)", "Number of internal holes: ",                   &
                        mfInt( mfMax(mfAbs(y)) ) - 1

   print *

   ! msTriNodeNeighbors ----------------------------------------------
   call print_separation("msTriNodeNeighbors")

   print "(/,A)", "*** test number 76 ***"
   call msTriNodeNeighbors( tri_connect, connected_nodes=connected_nodes )
   call msDisplay( connected_nodes, "Connected nodes to each node" )

   print "(/,A)", "*** test number 77 ***"
   call msTriNodeNeighbors( tri_connect, connected_faces=connected_faces )
   call msDisplay( connected_faces, "Connected faces to each node" )
   print "()"

   call msRelease( connected_nodes )
   deallocate( connected_nodes )

   ! mfTriSearch -------------------------------------------------------
   call print_separation("mfTriSearch")

   print "(/,A)", "*** test number 78 ***"
   num = mfTriSearch( tri_connect, 0.25d0, 0.25d0 )
   print "(1X,A,I0)", "enclosing (or nearest) triangle is # ", num

   print "(/,A)", "*** test number 79 ***"
   num = mfTriSearch( tri_connect, 0.25d0, 0.25d0, strict=.false. )
   print "(1X,A,I0)", "enclosing (or nearest) triangle is # ", num

   print "(/,A)", "*** test number 80 ***"
   num = mfTriSearch( tri_connect, mf(0.25d0), mf(0.25d0) )
   print "(1X,A,I0)", "enclosing (or nearest) triangle is # ", num

   print "(/,A)", "*** test number 81 ***"
   num = mfTriSearch( tri_connect, mf(0.25d0), mf(0.25d0), strict=.false. )
   print "(1X,A,I0)", "enclosing (or nearest) triangle is # ", num
   print "()"

   ! mfNodeSearch -------------------------------------------------------
   call print_separation("mfNodeSearch")

   print "(/,A)", "*** test number 82 ***"
   num = mfNodeSearch( tri_connect, 0.25d0, 0.25d0 )
   print "(1X,A,I0)", "nearest node is # ", num

   print "(/,A)", "*** test number 83 ***"
   num = mfNodeSearch( tri_connect, 0.25d0, 0.25d0, strict=.false. )
   print "(1X,A,I0)", "nearest node is # ", num

   print "(/,A)", "*** test number 84 ***"
   num = mfNodeSearch( tri_connect, mf(0.25d0), mf(0.25d0) )
   print "(1X,A,I0)", "nearest node is # ", num

   print "(/,A)", "*** test number 85 ***"
   num = mfNodeSearch( tri_connect, mf(0.25d0), mf(0.25d0), strict=.false. )
   print "(1X,A,I0)", "nearest node is # ", num
   print "()"

   ! msRelease ---------------------------------------------------------
   call print_separation("msRelease")

   print "(/,A)", "*** test number 86 ***"
   call msRelease( tri_connect )
   print *, "TriSearch : End."
   print "()"

   ! mfVoronoi ---------------------------------------------------------
   call print_separation("mfVoronoi")

   x = [ 0.2d0, 0.8d0, 0.2d0, 0.8d0 ]
   y = [ 0.6d0, 0.8d0, 0.2d0, 0.4d0 ]

   print "(/,A)", "*** test number 87 ***"
   voronoi = mfVoronoi( x, y, what="vertices" )
   call msPrintVoronoi( voronoi )

   print "(/,A)", "*** test number 88 ***"
   voronoi = mfVoronoi( x, y, what="neighbors" )
   call msPrintVoronoi( voronoi )

   print "(/,A)", "*** test number 89 ***"
   voronoi = mfVoronoi( x, y, what="both" )
   call msPrintVoronoi( voronoi )

   ! msRelease ---------------------------------------------------------
   call print_separation("msRelease")

   print "(/,A)", "*** test number 90 ***"
   call msRelease( voronoi )
   print "()"

   ! mfGridData --------------------------------------------------------
   call print_separation("mfGridData")

   x = [ -0.15d0,  1.12d0, 1.10d0, -0.12d0, 0.5d0 ]
   y = [ -0.11d0, -0.13d0, 1.14d0,  1.13d0, 0.5d0 ]
   f = 2*x - 3*y
   call msDisplay( .t.((x .vc. y) .vc. f), "nodes and values [linear in (x,y)]" )

   print "(/,A)", "*** test number 91 ***"
   ! (xi,yi) is a point :
   xi = 0.12345d0
   yi = 0.98765d0
   call msDisplay( xi, "xi", yi, "yi" )
   fi = mfGridData( x, y, f, xi, yi )
   call msDisplay( fi, "interpolated data at a point" )
   call msDisplay( 2*xi - 3*yi, " exact value" )
   print "()"

   print "(/,A)", "*** test number 92 ***"
   ! (xi,yi) are vectors :
   xi = mfLinspace( -0.197d0, 1.215d0, 8 )
   yi = mfLinspace( -0.203d0, 1.223d0, 8 )
   call msDisplay( xi, "xi", yi, "yi" )
   fi = mfGridData( x, y, f, xi, yi )
   call msDisplay( fi, "interpolated data at points on a line" )
   call msDisplay( 2*xi - 3*yi, " exact value" )

   print "(/,A)", "*** test number 93 ***"
   ! (xi,yi) are matrices :
   a =     mfColon( -0.2d0, 1.21d0, 0.2d0 )
   b = .t. mfColon( -0.2d0, 1.21d0, 0.2d0 )
   call msDisplay( a, "a", b, "b" )
   call msMeshGrid( mfOut(xi,yi), a, b )
   fi = mfGridData( x, y, f, xi, yi )
   call msDisplay( fi, "interpolated data at points on a 2D grid" )
   call msDisplay( 2*xi - 3*yi, " exact value" )

   ! mfDelaunay3D ------------------------------------------------------
   call print_separation("mfDelaunay3D")

   x = [ 0.0d0, 1.0d0, 0.0d0, 1.0d0, 0.0d0, 1.0d0, 0.0d0, 1.0d0, 0.5d0 ]
   y = [ 0.0d0, 0.0d0, 1.0d0, 1.0d0, 0.0d0, 0.0d0, 1.0d0, 1.0d0, 0.5d0 ]
   z = [ 0.0d0, 0.0d0, 0.0d0, 0.0d0, 1.0d0, 1.0d0, 1.0d0, 1.0d0, 0.5d0 ]
   call msDisplay( .t.(x .vc. y .vc. z), "nodes coord." )

   print "(/,A)", "*** test number 94 ***"
   tetra = mfDelaunay3D( x, y, z )
   call msDisplay( tetra, "tetrahedra" )

   ! msBuildTetraConnect -----------------------------------------------
   call print_separation("msBuildTetraConnect")

   print "(/,A)", "*** test number 95 ***"
   call msBuildTetraConnect( x, y, z, tetra, tetra_connect )
   print "()"

   ! msPrintTetraConnect -----------------------------------------------
   call print_separation("msPrintTetraConnect")

   print "(/,A)", "*** test number 96 ***"
   call msPrintTetraConnect( tetra_connect )
   print "()"

   ! msDel3DNodeNeighbors ----------------------------------------------
   call print_separation("msDel3DNodeNeighbors")

   print "(/,A)", "*** test number 97 ***"
   call msDel3DNodeNeighbors( tetra, connected_nodes )
   call msDisplay( connected_nodes, "Connected nodes" )

   ! mfTetraSearch -----------------------------------------------------
   call print_separation("mfTetraSearch")

   print "(/,A)", "*** test number 98 ***"
   num = mfTetraSearch( tetra_connect, 0.75d0, 0.25d0, 0.10d0 )
   print "(1X,A,I0)", "enclosing tetrahedron is # ", num
   print "()"

   ! mfNodeSearch3D ----------------------------------------------------
   call print_separation("mfNodeSearch3D")

   print "(/,A)", "*** test number 99 ***"
   num = mfNodeSearch3D( tetra_connect, 0.75d0, 0.25d0, 0.10d0 )
   print "(1X,A,I0)", "nearest node is # ", num
   print "()"

   ! msEndDelaunay3D ---------------------------------------------------
   call print_separation("msEndDelaunay3D")

   print "(/,A)", "*** test number 100 ***"
   call msEndDelaunay3D()
   print "()"

   ! msRelease ---------------------------------------------------------
   call print_separation("msRelease")

   print "(/,A)", "*** test number 101 ***"
   call msRelease( tetra_connect )
   print "()"

   ! msCheckPLDomain3DValidity -----------------------------------------
   call print_separation("msCheckPLDomain3DValidity")

   ! unit cube domain
   allocate( PL_domain3D%n_xyz(8,3) )
   PL_domain3D%n_xyz(:,1) = [ 0.0d0, 1.0d0, 1.0d0, 0.0d0,               &
                              0.0d0, 1.0d0, 1.0d0, 0.0d0 ] ! x
   PL_domain3D%n_xyz(:,2) = [ 0.0d0, 0.0d0, 1.0d0, 1.0d0,               &
                              0.0d0, 0.0d0, 1.0d0, 1.0d0 ] ! y
   PL_domain3D%n_xyz(:,3) = [ 0.0d0, 0.0d0, 0.0d0, 0.0d0,               &
                              1.0d0, 1.0d0, 1.0d0, 1.0d0 ] ! z
   allocate( PL_domain3D%edge_n(12,2) )
   PL_domain3D%edge_n(:,1) = [ 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4 ]
   PL_domain3D%edge_n(:,2) = [ 2, 3, 4, 1, 6, 7, 8, 5, 5, 6, 7, 8 ]
   allocate( PL_domain3D%face_e(6) )
   ! face 1 (defined by 4 edges)
   allocate( PL_domain3D%face_e(1)%list(4) )
   PL_domain3D%face_e(1)%list(:) = [ 1, 2, 3, 4 ]
   ! face 2
   allocate( PL_domain3D%face_e(2)%list(4) )
   PL_domain3D%face_e(2)%list(:) = [ 5, 6, 7, 8 ]
   ! face 3
   allocate( PL_domain3D%face_e(3)%list(4) )
   PL_domain3D%face_e(3)%list(:) = [ 1, 10,  5,  9 ]
   ! face 4
   allocate( PL_domain3D%face_e(4)%list(4) )
   PL_domain3D%face_e(4)%list(:) = [ 2, 11,  6, 10 ]
   ! face 5
   allocate( PL_domain3D%face_e(5)%list(4) )
   PL_domain3D%face_e(5)%list(:) = [ 3, 12,  7, 11 ]
   ! face 6
   allocate( PL_domain3D%face_e(6)%list(4) )
   PL_domain3D%face_e(6)%list(:) = [ 4,  9,  8, 12 ]

   print "(/,A)", "*** test number 102 ***"
   call msCheckPLDomain3DValidity( PL_domain3D, status )
   if( status == 1 ) then
      print *, "  -> the provided 3D PL domain seems to be valid."
   else
      print *, "  -> the provided 3D PL domain is NOT valid."
      stop
   end if
   print *

   ! msPrintPLdomain3D -------------------------------------------------
   call print_separation("msPrintPLdomain3D")

   print "(/,A)", "*** test number 103 ***"
   call msPrintPLdomain3D( PL_domain3D )

   ! msDelaunay3D ------------------------------------------------------
   call print_separation("msDelaunay3D")

   print "(/,A)", "*** test number 104 ***"
   print "(/,1X,A)", "Constrained Delaunay tetrahedralization."
   call msDelaunay3D( mfOut(x_out,y_out,z_out,tetra_out),               &
                      PL_domain3D, quality=1.5d0, vol_max=0.2d0 )

   print *, "new nb of pts = ", size(x_out,1)
   print *, "nb of tetrahedra = ", size(tetra_out,1)

   call msDisplay( x_out .hc. y_out .hc. z_out, "new pts coords" )
   call msDisplay( tetra_out, "tetrahedra" )

   call msRelease( PL_domain3D )

   call msEndDelaunay3D()

   ! mfGridData3D ------------------------------------------------------
   call print_separation("mfGridData3D")

   x = [ 0.0d0, 1.0d0, 1.0d0, 0.0d0, 0.0d0, 1.0d0, 1.0d0, 0.0d0, 0.5d0 ]
   y = [ 0.0d0, 0.0d0, 1.0d0, 1.0d0, 0.0d0, 0.0d0, 1.0d0, 1.0d0, 0.5d0 ]
   z = [ 0.0d0, 0.0d0, 0.0d0, 0.0d0, 1.0d0, 1.0d0, 1.0d0, 1.0d0, 0.5d0 ]
   f = 2*x - 3*y + z
   call msDisplay( .t.(x .vc. y .vc. z .vc. f), "nodes and values [linear in (x,y,z)]" )

   print "(/,A)", "*** test number 105 ***"
   ! (xi,yi,zi) is a point :
   xi = 0.12345d0
   yi = 0.67676d0
   zi = 0.55555d0
   call msDisplay( xi, "xi", yi, "yi", zi, "zi" )
   fi = mfGridData3D( x, y, z, f, xi, yi, zi )
   call msDisplay( fi, "interpolated data at a point" )
   call msDisplay( 2*xi - 3*yi + zi, " exact value" )
   print "()"

   print "(/,A)", "*** test number 106 ***"
   ! (xi,yi,zi) are vectors :
   xi = mfLinspace( -0.097d0, 1.05d0, 8 )
   yi = mfLinspace( -0.103d0, 1.13d0, 8 )
   zi = mfLinspace( -0.100d0, 1.10d0, 8 )
   call msDisplay( xi, "xi", yi, "yi", zi, "zi" )
   fi = mfGridData3D( x, y, z, f, xi, yi, zi )
   call msDisplay( fi, "interpolated data at points on a line" )
   call msDisplay( 2*xi - 3*yi + zi, " exact value" )

   print "(/,A)", "*** test number 107 ***"
   ! (xi,yi,zi) are matrices :
   a =     mfColon( -0.1d0, 1.1d0, 0.2d0 )
   b = .t. mfColon( -0.1d0, 1.1d0, 0.2d0 )
   c = mfColon( -0.1d0, 1.1d0, 0.2d0 )
   call msDisplay( a, "a", b, "b" )
   call msMeshGrid( mfOut(xi,yi), a, b )
   zi = xi
   fi = mfGridData3D( x, y, z, f, xi, yi, zi )
   call msDisplay( fi, "interpolated data at points on a 3D grid" )
   call msDisplay( 2*xi - 3*yi + zi, " exact value" )

   ! mfLegendre --------------------------------------------------------
   call print_separation("mfLegendre")

   x = mfLinspace( -1.0d0, 1.0d0, 7 )
   call zeroes_small_values( x, 10.0d0*MF_EPS )
   call msDisplay( x, "x" )

   print "(/,A)", "*** test number 108 ***"
   call msDisplay( mfLegendre(-1,x), "mfLegendre(-1,x)" )
   print "(/,A)", "*** test number 109 ***"
   call msDisplay( mfLegendre(0,x), "mfLegendre(0,x)" )
   print "(/,A)", "*** test number 110 ***"
   call msDisplay( mfLegendre(1,x), "mfLegendre(1,x)" )
   print "(/,A)", "*** test number 111 ***"
   call msDisplay( mfLegendre(2,x), "mfLegendre(2,x)" )

   ! mfRoots -----------------------------------------------------------
   call print_separation("mfRoots")

   p = [ 1.0d0, 0.0d0, -7.0d0, 6.0d0 ]
   call msDisplay( p, "polynomial: p" )

   print "(/,A)", "*** test number 112 ***"
   r = mfRoots(p)
   call msDisplay( r, "roots: r = roots(p)" )

   p = .t. p
   call msDisplay( p, "polynomial: p" )

   print "(/,A)", "*** test number 113 ***"
   r = mfRoots(p)
   call msDisplay( r, "roots: r = roots(p)" )

   p = [ 0.0d0, 0.0d0, 1.0d0, 0.0d0, 2.0d0, 0.0d0, 3.0d0, 0.0d0, 0.0d0 ]
   call msDisplay( p, "polynomial: p" )

   print "(/,A)", "*** test number 114 ***"
   r = mfRoots(p)
   call msDisplay( r, "roots: r = roots(p)" )

   p = .t. p
   call msDisplay( p, "polynomial: p" )

   print "(/,A)", "*** test number 115 ***"
   r = mfRoots(p)
   call msDisplay( r, "roots: r = roots(p)" )


   ! mfPoly ------------------------------------------------------------
   call print_separation("mfPoly")

   r = [ -3.0d0, 1.0d0, 2.0d0 ]
   call msDisplay( r, "roots: r" )

   print "(/,A)", "*** test number 116 ***"
   p = mfPoly(r)
   call msDisplay( p, "polynomial: p = poly(r)" )

   r = .t. r
   call msDisplay( r, "roots: r" )

   print "(/,A)", "*** test number 117 ***"
   p = mfPoly(r)
   call msDisplay( p, "polynomial: p = poly(r)" )

   ! mfRoots and mfPoly should be inverse together
   p = [ 0.0d0, 0.0d0, 1.0d0, 0.0d0, 2.0d0, 0.0d0, 3.0d0, 0.0d0, 0.0d0 ]
   call msDisplay( p, "polynomial: p" )
   r = mfRoots(p)
   call msDisplay( r, "roots: r = roots(p)" )
   print "(/,A)", "*** test number 118 ***"
   p = mfPoly(r)
   call zeroes_small_values( p, 10.0d0*MF_EPS )
   call msDisplay( p, "polynomial: p = poly(r)" )

   p = [ 0.0d0, 0.0d0, 1.0d0, 0.0d0, 2.0d0, 0.0d0, 3.0d0, 0.0d0, 0.0d0 ]
   p = .t. p
   call msDisplay( p, "polynomial: p" )
   r = mfRoots(p)
   call msDisplay( r, "roots: r = roots(p)" )
   print "(/,A)", "*** test number 119 ***"
   p = mfPoly(r)
   call zeroes_small_values( p, 10.0d0*MF_EPS )
   call msDisplay( p, "polynomial: p = poly(r)" )

   ! End of tests ------------------------------------------------------
   print "(A)", "                         ┏━━━━━━━━━━━━━━━━━━━━━━━━━┓"
   print "(A)", "                         ┃ End of test_mod_polyfun ┃"
   print "(A)", "                         ┗━━━━━━━━━━━━━━━━━━━━━━━━━┛"

   !--------------------------------------------------------------------
99 continue

   print "()"
   call msSetMsgLevel(2)

   call msRelease( x, y, z, xx, yy, pp, tri )
   call msRelease( x_vec, y_vec, z_vec )
   call msRelease( x_out, y_out, z_out, tri_out, tetra_out )
   call msRelease( PL_domain )
   call msRelease( f, fi )
   call msRelease( a, b, c, xi, yi, zi )
   call msRelease( p, ran, r, normr, r2 )
   call msRelease( y_sec, y_smooth, weights )
   call msRelease( tri_connect )
   call msRelease( tetra )
   call msRelease( tetra_connect )
   if( allocated(connected_nodes) ) then
      call msRelease( connected_nodes )
      ! As connected_nodes(:) is allocatable, it should be deallocated
      ! at the end of the program. Here, we do such a deallocation because
      ! valgrind complains about memory leak.
      deallocate( connected_nodes )
   end if
   if( allocated(connected_faces) ) then
      call msRelease( connected_faces )
      ! As connected_nodes(:) is allocatable, it should be deallocated
      ! at the end of the program. Here, we do such a deallocation because
      ! valgrind complains about memory leak.
      deallocate( connected_faces )
   end if
   if( allocated(face_tri) ) then
      deallocate( face_tri )
   end if
   call msRelease( indices )

   ! non official Muesli routine: introduced here to clean memory
   call RngStream_end()

end program
