module optim_funs

   use fml

   implicit none

contains

   function boltzmann( x, p, n ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x
      real(kind=MF_DOUBLE), intent(in) :: p(n)
      integer,              intent(in) :: n
      real(kind=MF_DOUBLE) :: res
      res = 1.0d0 / ( 1.0d0 + exp((p(1)-x)/p(2)) )
   end function boltzmann

end module optim_funs
!_______________________________________________________________________
!
program test_mod_datafun

   use fml

   use test_aux
   use lib_aux

   use optim_funs

   implicit none

   type(mfArray) :: x, y, z, a, i, c, theta, res
   type(mfArray) :: x1, x2, p, r2, status, v, Fx, Fy
   integer :: j, k, n
   real(kind=MF_DOUBLE) :: tol

   ! Msg level = 3 : all messages are printed (verbose mode)
   !             2 : messages of kind 'ERROR' and 'Warning' are
   !                 printed [default]
   !             1 : only messages of kind 'ERROR' are printed
   !             0 : nothing is printed (quiet mode)
   !                 (pause are ignored !)

!!   call msSetMsgLevel(3)
!!   call msSetTrbLevel("all")

   print "()"

   ! for TRACE_MEM_DEBUG, STDERR must be redirected to STDOUT;
   ! the same thing stands for 'make check'
   call msSetStdIO( stderr=STDOUT )

!!goto 10

   ! to avoid escape sequences of color setting in the output file
   ! (default in "on")
   call msSetColoredMsg( "off" )

   ! VERSIONs ----------------------------------------------------------
   print "()"
   print *, "MF_MUESLI_VERSION = '", MF_MUESLI_VERSION, "'"
   print "()"
   print *, "MF_COMPILER_VERSION = '", MF_COMPILER_VERSION, "'"
   print "()"
   print *, "MF_COMPILATION_CONFIG = '", MF_COMPILATION_CONFIG(), "'"
   print "()"

   ! mfProd ------------------------------------------------------------
   call print_separation("mfProd")

   print "(/,A)", "*** test number 1 ***"
   x = mfMagic(3)
   call msDisplay(x,"mfMagic(3)")
   call msDisplay(mfProd(x),"product")
   print "(/,A)", "*** test number 2 ***"
   call msDisplay(mfProd(x,1),"col product")
   print "(/,A)", "*** test number 3 ***"
   call msDisplay(mfProd(x,2),"row product")
   print "(/,A)", "*** test number 4 ***"
   call msDisplay(mfProd(mfDiag(x)),"diag. product")

   ! mfSum -------------------------------------------------------------
   call print_separation("mfSum")

   print "(/,A)", "*** test number 5 ***"
   x = mfMagic(3)
   call msDisplay(x,"mfMagic(3)")
   call msDisplay(mfSum(x),"sum")
   print "(/,A)", "*** test number 6 ***"
   call msDisplay(mfSum(x,1),"col sum")
   print "(/,A)", "*** test number 7 ***"
   call msDisplay(mfSum(x,2),"row sum")
   print "(/,A)", "*** test number 8 ***"
   call msDisplay(mfSum(mfDiag(x)),"diag. sum")

   print "(/,A)", "*** test number 9 ***"
   x = mfZeros(3)
   call msSet(1.0d0,x,3,1)
   call msSet(1.0d0,x,2,2)
   call msSet(1.0d0,x,1,3)
   x = mfMul( mfMagic(3), x ) ! vertical flip
   call msDisplay(mfSum(mfDiag(x)),"diag. (2nd) sum")

   print "(/,A)", "*** test number 10 ***"
   x = mfMagic(4)
   call msDisplay(x,"mfMagic(4)")
   call msDisplay(mfSum(x),"sum")
   print "(/,A)", "*** test number 11 ***"
   call msDisplay(mfSum(x,1),"col sum")
   print "(/,A)", "*** test number 12 ***"
   call msDisplay(mfSum(x,2),"row sum")
   print "(/,A)", "*** test number 13 ***"
   call msDisplay(mfSum(mfDiag(x)),"diag. sum")

   print "(/,A)", "*** test number 14 ***"
   x = mfZeros(4)
   call msSet(1.0d0,x,4,1)
   call msSet(1.0d0,x,3,2)
   call msSet(1.0d0,x,2,3)
   call msSet(1.0d0,x,1,4)
   x = mfMul( mfMagic(4), x ) ! vertical flip
   call msDisplay(mfSum(mfDiag(x)),"diag. (2nd) sum")

   print "(/,A)", "*** test number 15 ***"
   x = mfMagic(5)
   call msDisplay(x,"mfMagic(5)")
   call msDisplay(mfSum(x,1),"col sum")
   print "(/,A)", "*** test number 16 ***"
   call msDisplay(mfSum(x,2),"row sum")
   print "(/,A)", "*** test number 17 ***"
   call msDisplay(mfSum(mfDiag(x)),"diag. sum")

   print "(/,A)", "*** test number 18 ***"
   x = mfZeros(5)
   call msSet(1.0d0,x,5,1)
   call msSet(1.0d0,x,4,2)
   call msSet(1.0d0,x,3,3)
   call msSet(1.0d0,x,2,4)
   call msSet(1.0d0,x,1,5)
   x = mfMul( mfMagic(5), x ) ! vertical flip
   call msDisplay(mfSum(mfDiag(x)),"diag. (2nd) sum")

   print "(/,A)", "*** test number 19 ***"
   x = mfMagic(6)
   call msDisplay(x,"mfMagic(6)")
   call msDisplay(mfSum(x,1),"col sum")
   print "(/,A)", "*** test number 20 ***"
   call msDisplay(mfSum(x,2),"row sum")
   print "(/,A)", "*** test number 21 ***"
   call msDisplay(mfSum(mfDiag(x)),"diag. sum")

   print "(/,A)", "*** test number 22 ***"
   x = mfZeros(6)
   call msSet(1.0d0,x,6,1)
   call msSet(1.0d0,x,5,2)
   call msSet(1.0d0,x,4,3)
   call msSet(1.0d0,x,3,4)
   call msSet(1.0d0,x,2,5)
   call msSet(1.0d0,x,1,6)
   x = mfMul( mfMagic(6), x ) ! vertical flip
   call msDisplay(mfSum(mfDiag(x)),"diag. (2nd) sum")

   ! mfMax -------------------------------------------------------------
   call print_separation("mfMax")

   print "(/,A)", "*** test number 23 ***"
   x = mf( [ 1.0, 2.0, 3.0, 4.0, 5.0 ] )
   call msSet( MF_NAN, x, 3 )
   call msDisplay(x,"x")
   call mf_save_and_disable_fpe( )
   call msDisplay(mfMax(x),"max")
   call mf_restore_fpe( )

   print "(/,A)", "*** test number 24 ***"
   x = .t. x
   call msDisplay(x,"x")
   call mf_save_and_disable_fpe( )
   call msDisplay(mfMax(x),"max")
   call mf_restore_fpe( )

   print "(/,A)", "*** test number 25 ***"
   x = mfMagic(3)
   call msSet( MF_NAN, x, 3, 3 )
   call msDisplay(x,"x [mfMagic(3) with one NaN]")
   call mf_save_and_disable_fpe( )
   call msDisplay(mfMax(x,1),"col max")
   print "(/,A)", "*** test number 26 ***"
   call msDisplay(mfMax(x,2),"row max")
   call mf_restore_fpe( )

   print "(/,A)", "*** test number 27 ***"
   call msDisplay(mfMax(x,6.0d0),"mfMax(x,6.0d0)")

   print "(/,A)", "*** test number 28 ***"
   a = mfEye(3)*6.0d0
   call msSet( MF_NAN, a, 1, 3 )
   call msDisplay(a,"a")
   call mf_save_and_disable_fpe( )
   call msDisplay(mfMax(x,a),"mfMax(x,a)")
   call mf_restore_fpe( )

   ! msMax -------------------------------------------------------------
   call print_separation("msMax")

   print "(/,A)", "*** test number 29 ***"
   x = mf( [ 1, 3, 5, 2, 4 ] )
   call msSet( MF_NAN, x, 2 )
   call msDisplay(x,"x")
   call mf_save_and_disable_fpe( )
   call msMax(mfOut(a,i),x)
   call mf_restore_fpe( )
   call msDisplay(a,"max")
   call msDisplay(i,"for index")

   print "(/,A)", "*** test number 30 ***"
   x = .t. x
   call msDisplay(x,"x")
   call mf_save_and_disable_fpe( )
   call msMax(mfOut(a,i),x)
   call mf_restore_fpe( )
   call msDisplay(a,"max")
   call msDisplay(i,"for index")

   print "(/,A)", "*** test number 31 ***"
   x = mfMagic(3)
   call msSet( MF_NAN, x, 3, 3 )
   call msDisplay(x,"mfMagic(3)")
   call mf_save_and_disable_fpe( )
   call msMax(mfOut(a,i),x,1)
   call mf_restore_fpe( )
   call msDisplay(a,"col max")
   call msDisplay(i,"for index")
   print "(/,A)", "*** test number 32 ***"
   call mf_save_and_disable_fpe( )
   call msMax(mfOut(a,i),x,2)
   call mf_restore_fpe( )
   call msDisplay(a,"row max")
   call msDisplay(i,"for index")

   ! mfMin -------------------------------------------------------------
   call print_separation("mfMin")

   print "(/,A)", "*** test number 33 ***"
   x = mf( [ 1.0, 2.0, 3.0, 4.0, 5.0 ] )
   call msSet( MF_NAN, x, 3 )
   call msDisplay(x,"x")
   call mf_save_and_disable_fpe( )
   call msDisplay(mfMin(x),"min")
   call mf_restore_fpe( )

   print "(/,A)", "*** test number 34 ***"
   x = .t. x
   call msDisplay(x,"x")
   call mf_save_and_disable_fpe( )
   call msDisplay(mfMin(x),"min")
   call mf_restore_fpe( )

   print "(/,A)", "*** test number 35 ***"
   x = mfMagic(3)
   call msSet( MF_NAN, x, 3, 3 )
   call msDisplay(x,"x [mfMagic(3) with one NaN]")
   call mf_save_and_disable_fpe( )
   call msDisplay(mfMin(x,1),"col min")
   print "(/,A)", "*** test number 36 ***"
   call msDisplay(mfMin(x,2),"row min")
   call mf_restore_fpe( )

   print "(/,A)", "*** test number 37 ***"
   call msDisplay(mfMin(x,4.0d0),"mfMin(x,4.0d0)")

   print "(/,A)", "*** test number 38 ***"
   a = mfEye(3)*4.0d0
   call msSet( MF_NAN, a, 1, 3 )
   call msDisplay(a,"a")
   call mf_save_and_disable_fpe( )
   call msDisplay(mfMin(x,a),"mfMin(x,a)")
   call mf_restore_fpe( )

   ! msMin -------------------------------------------------------------
   call print_separation("msMin")

   print "(/,A)", "*** test number 39 ***"
   x = mf( [ 1, 3, 5, 2, 4 ] )
   call msSet( MF_NAN, x, 2 )
   call msDisplay(x,"x")
   call mf_save_and_disable_fpe( )
   call msMin(mfOut(a,i),x)
   call mf_restore_fpe( )
   call msDisplay(a,"min")
   call msDisplay(i,"for index")

   print "(/,A)", "*** test number 40 ***"
   x = .t. x
   call msDisplay(x,"x")
   call mf_save_and_disable_fpe( )
   call msMin(mfOut(a,i),x)
   call mf_restore_fpe( )
   call msDisplay(a,"min")
   call msDisplay(i,"for index")

   print "(/,A)", "*** test number 41 ***"
   x = mfMagic(3)
   call msSet( MF_NAN, x, 3, 3 )
   call msDisplay(x,"x [mfMagic(3) with one NaN]")
   call mf_save_and_disable_fpe( )
   call msMin(mfOut(a,i),x,1)
   call mf_restore_fpe( )
   call msDisplay(a,"col min")
   call msDisplay(i,"for index")

   print "(/,A)", "*** test number 42 ***"
   call mf_save_and_disable_fpe( )
   call msMin(mfOut(a,i),x,2)
   call mf_restore_fpe( )
   call msDisplay(a,"row min")
   call msDisplay(i,"for index")

   ! mfSort ------------------------------------------------------------
   call print_separation("mfSort")

   print "(/,A)", "*** test number 43 ***"
   x = mf( [ 1, 3, 5, 2, 4 ] )
   call msDisplay(x,"x")
   call msDisplay(mfSort(x),"mfSort(x)")

   print "(/,A)", "*** test number 44 ***"
   x = mf( [ 1, 3, 5, 2, 4 ] )
   call msDisplay(x,"x")
   call msDisplay(mfSort(x,'des'),"mfSort(x,'des')")

   print "(/,A)", "*** test number 45 ***"
   x = .t. x
   call msDisplay(x,"x")
   call msDisplay(mfSort(x),"mfSort(x)")

   print "(/,A)", "*** test number 46 ***"
   x = mfMagic(3)
   call msDisplay(x,"mfMagic(3)")
   call msDisplay(mfSort(x,1),"mfSort(x,1)")
   print "(/,A)", "*** test number 47 ***"
   call msDisplay(mfSort(x,2),"mfSort(x,2)")

   print "(/,A)", "*** test number 48 ***"
   x = mfMagic(3)
   call msDisplay(x,"mfMagic(3)")
   call msDisplay(mfSort(x,1,'des'),"mfSort(x,1,'des')")
   print "(/,A)", "*** test number 49 ***"
   call msDisplay(mfSort(x,2,'des'),"mfSort(x,2,'des')")

   ! vector containing only NaNs
   print "(/,A)", "*** test number 50 ***"
   call mf_save_and_disable_fpe( )
   x = MF_NAN*mfOnes(1,6)
   call msDisplay(x,"x")
   call msDisplay(mfSort(x),"mfSort(x)")
   call mf_restore_fpe( )

   print "(/,A)", "*** test number 51 ***"
   call mf_save_and_disable_fpe( )
   x = MF_NAN*mfOnes(1,6)
   call msDisplay(x,"x")
   call msDisplay(mfSort(x,'des'),"mfSort(x,'des')")
   call mf_restore_fpe( )

   ! row vector containing some NaNs
   print "(/,A)", "*** test number 52 ***"
   call mf_save_and_disable_fpe( )
   x = mf( [ 1.0d0, MF_NAN, 5.0d0, MF_NAN, 2.0d0, 4.0d0 ] )
   call msDisplay(x,"x")
   call msDisplay(mfSort(x),"mfSort(x)")
   call mf_restore_fpe( )

   print "(/,A)", "*** test number 53 ***"
   call mf_save_and_disable_fpe( )
   x = mf( [ 1.0d0, MF_NAN, 5.0d0, MF_NAN, 2.0d0, 4.0d0 ] )
   call msDisplay(x,"x")
   call msDisplay(mfSort(x,'des'),"mfSort(x,'des')")
   call mf_restore_fpe( )

   ! column vector containing some NaNs
   print "(/,A)", "*** test number 54 ***"
   call mf_save_and_disable_fpe( )
   x = .t. mf( [ 1.0d0, MF_NAN, 5.0d0, MF_NAN, 2.0d0, 4.0d0 ] )
   call msDisplay(x,"x")
   call msDisplay(mfSort(x),"mfSort(x)")
   call mf_restore_fpe( )

   print "(/,A)", "*** test number 55 ***"
   call mf_save_and_disable_fpe( )
   x = .t. mf( [ 1.0d0, MF_NAN, 5.0d0, MF_NAN, 2.0d0, 4.0d0 ] )
   call msDisplay(x,"x")
   call msDisplay(mfSort(x,'des'),"mfSort(x,'des')")
   call mf_restore_fpe( )

   ! complex values (vector only !): double sort, first on module,
   ! then on phase-angle
   print "(/,A)", "*** test number 56 ***"
   x = mf( [ (1.0d0, 0.5d0), (0.0d0, 0.0d0), (0.0d0, 0.0d0),            &
             (0.0d0, -3.0d0), (-1.5d0, 0.0d0), (2.0d0, 0.0d0),          &
             (0.0d0, -1.0d0), (-2.0d0, 0.0d0), (0.0d0, 2.0d0) ] )
   call msDisplay(x,"x")
   call msDisplay(mfSort(x),"mfSort(x)")

   print "(/,A)", "*** test number 57 ***"
   x = mf( [ (1.0d0, 0.5d0), (0.0d0, 0.0d0), (0.0d0, -3.0d0),           &
             (-1.5d0, 0.0d0), (2.0d0, 0.0d0), (0.0d0, -1.0d0),          &
             (-2.0d0, 0.0d0), (0.0d0, 2.0d0) ] )
   call msDisplay(x,"x")
   call msDisplay(mfSort(x,'des'),"mfSort(x,'des')")

   ! complex values (vector only !)
   ! (due to the complex transpose in tests 58 and 59, the imaginary
   ! part of (-2,0) is actually a negative zero, and thus leads to a
   ! phase-angle equal to -Pi)
   print "(/,A)", "*** test number 58 ***"
   x = .h. mf( [ (1.0d0, 0.5d0), (0.0d0, 0.0d0), (0.0d0, -3.0d0),       &
                (-1.5d0, 0.0d0), (2.0d0, 0.0d0), (0.0d0, -1.0d0),       &
                (-2.0d0, 0.0d0), (0.0d0, 2.0d0) ] )
   call msDisplay(x,"x")
   call msDisplay(mfSort(x),"mfSort(x)")

   print "(/,A)", "*** test number 59 ***"
   x = .h. mf( [ (1.0d0, 0.5d0), (0.0d0, 0.0d0), (0.0d0, -3.0d0),       &
                (-1.5d0, 0.0d0), (2.0d0, 0.0d0), (0.0d0, -1.0d0),       &
                (-2.0d0, 0.0d0), (0.0d0, 2.0d0) ] )
   call msDisplay(x,"x")
   call msDisplay(mfSort(x,'des'),"mfSort(x,'des')")

   ! complex vector containing only NaNs
   print "(/,A)", "*** test number 60 ***"
   call mf_save_and_disable_fpe( )
   x = (MF_NAN*mfOnes(1,6))*MF_I
   call msDisplay(x,"x")
   call msDisplay(mfSort(x),"mfSort(x)")
   call mf_restore_fpe( )

   ! complex row vector containing some NaNs
   print "(/,A)", "*** test number 61 ***"
   call mf_save_and_disable_fpe( )
   x = mf( [ 1.0d0, MF_NAN, 0.0d0, 1.0d0,  0.0d0, -1.5d0,  0.0d0 ] ) +  &
       mf( [ 0.5d0, 0.0d0, -3.0d0, 0.0d0, MF_NAN,  0.0d0, -1.0d0 ] )*MF_I
   call msDisplay(x,"x")
   call msDisplay(mfSort(x),"mfSort(x)")
   call mf_restore_fpe( )

   print "(/,A)", "*** test number 62 ***"
   call mf_save_and_disable_fpe( )
   x = mf( [ 1.0d0, MF_NAN, 0.0d0, 1.0d0,  0.0d0, -1.5d0,  0.0d0 ] ) +  &
       mf( [ 0.5d0, 0.0d0, -3.0d0, 0.0d0, MF_NAN,  0.0d0, -1.0d0 ] )*MF_I
   call msDisplay(x,"x")
   call msDisplay(mfSort(x,'des'),"mfSort(x,'des')")
   call mf_restore_fpe( )

   ! complex column vector containing some NaNs
   print "(/,A)", "*** test number 63 ***"
   call mf_save_and_disable_fpe( )
   x = mf( [ 1.0d0, MF_NAN, 0.0d0, 1.0d0,  0.0d0, -1.5d0,  0.0d0 ] ) +  &
       mf( [ 0.5d0, 0.0d0, -3.0d0, 0.0d0, MF_NAN,  0.0d0, -1.0d0 ] )*MF_I
   x = .h. x
   call msDisplay(x,"x")
   call msDisplay(mfSort(x),"mfSort(x)")
   call mf_restore_fpe( )

   ! msSort ------------------------------------------------------------
   call print_separation("msSort")

   ! in-place sorting (not using mfOut -- reals only)

   print "(/,A)", "*** test number 64 ***"
   x = mf( [ 1, 3, 5, 2, 4 ] )
   call msDisplay(x,"x")
   call msSort(x)
   call msDisplay(x,"sorted vec")

   print "(/,A)", "*** test number 65 ***"
   x = mf( [ 5.0d0, MF_NAN, 3.0d0, 1.0d0, MF_NAN, 2.0d0, 4.0d0 ] )
   call msDisplay(x,"x")
   call msSort(x)
   call msDisplay(x,"sorted vec")

   print "(/,A)", "*** test number 66 ***"
   x = mfMagic(5)
   call msDisplay(x,"x")
   call msSort(x,dim=1)
   call msDisplay(x,"sorted matrix (by columns)")

   print "(/,A)", "*** test number 67 ***"
   x = mfMagic(5)
   call msDisplay(x,"x")
   call msSort(x,dim=2)
   call msDisplay(x,"sorted matrix (by rows)")

   ! using mfOut

   print "(/,A)", "*** test number 68 ***"
   x = mf( [ 1, 3, 5, 2, 4 ] )
   call msDisplay(x,"x")
   call msSort(mfOut(a,i),x)
   call msDisplay(a,"sorted vec")
   call msDisplay(i,"with index")

   print "(/,A)", "*** test number 69 ***"
   x = mf( [ 1, 3, 5, 2, 4 ] )
   call msDisplay(x,"x")
   call msSort(mfOut(a,i),x,'des')
   call msDisplay(a,"sorted vec (descending)")
   call msDisplay(i,"with index")

   print "(/,A)", "*** test number 70 ***"
   x = .t. x
   call msDisplay(x,"x")
   call msSort(mfOut(a,i),x)
   call msDisplay(a,"sorted vec")
   call msDisplay(i,"with index")

   print "(/,A)", "*** test number 71 ***"
   x = mfMagic(3)
   call msDisplay(x,"mfMagic(3)")
   call msSort(mfOut(a,i),x,1)
   call msDisplay(a,"col sort")
   call msDisplay(i,"with index")
   print "(/,A)", "*** test number 72 ***"
   call msSort(mfOut(a,i),x,2)
   call msDisplay(a,"row sort")
   call msDisplay(i,"with index")

   ! vector containing only NaNs
   print "(/,A)", "*** test number 73 ***"
   call mf_save_and_disable_fpe( )
   x = MF_NAN*mfOnes(1,6)
   call msDisplay(x,"x")
   call msSort(mfOut(a,i),x)
   call mf_restore_fpe( )
   call msDisplay(a,"sorted vec")
   call msDisplay(i,"with index")

   ! row vector containing some NaNs
   print "(/,A)", "*** test number 74 ***"
   call mf_save_and_disable_fpe( )
   x = mf( [ 1.0d0, MF_NAN, 5.0d0, MF_NAN, 2.0d0, 4.0d0 ] )
   call msDisplay(x,"x")
   call msSort(mfOut(a,i),x)
   call mf_restore_fpe( )
   call msDisplay(a,"sorted vec")
   call msDisplay(i,"with index")

   ! col vector containing some NaNs
   print "(/,A)", "*** test number 75 ***"
   call mf_save_and_disable_fpe( )
   x = .t. mf( [ 1.0d0, MF_NAN, 5.0d0, MF_NAN, 2.0d0, 4.0d0 ] )
   call msDisplay(x,"x")
   call msSort(mfOut(a,i),x)
   call mf_restore_fpe( )
   call msDisplay(a,"sorted vec")
   call msDisplay(i,"with index")

   print "(/,A)", "*** test number 76 ***"
   call mf_save_and_disable_fpe( )
   x = .t. mf( [ 1.0d0, MF_NAN, 5.0d0, MF_NAN, 2.0d0, 4.0d0 ] )
   call msDisplay(x,"x")
   call msSort(mfOut(a,i),x,'des')
   call mf_restore_fpe( )
   call msDisplay(a,"sorted vec (descending)")
   call msDisplay(i,"with index")

   ! complex values (vector only !) row
   x = mf( [ (1.0d0, 0.5d0), (0.0d0, 0.0d0), (0.0d0, -1.0d0),           &
             (-1.5d0, 0.0d0), (1.0d0, 0.0d0), (0.0d0, 1.0d0) ] )
   call msDisplay(x,"x")
   print "(/,A)", "*** test number 77 ***"
   call msSort(mfOut(a,i),x)
   call msDisplay(a,"sorted vec")
   call msDisplay(i,"with index")

   ! col
   x = .h. mf( [ (1.0d0, 0.5d0), (0.0d0, 0.0d0), (0.0d0, -1.0d0),       &
                 (-1.5d0, 0.0d0), (1.0d0, 0.0d0), (0.0d0, 1.0d0) ] )
   call msDisplay(x,"x")
   print "(/,A)", "*** test number 78 ***"
   call msSort(mfOut(a,i),x)
   call msDisplay(a,"sorted vec")
   call msDisplay(i,"with index")

   ! complex vector containing only NaNs
   print "(/,A)", "*** test number 79 ***"
   call mf_save_and_disable_fpe( )
   x = (MF_NAN*mfOnes(1,6))*MF_I
   call msDisplay(x,"x")
   call msSort(mfOut(a,i),x)
   call mf_restore_fpe( )
   call msDisplay(a,"sorted vec")
   call msDisplay(i,"with index")

   ! complex row vector containing some NaNs
   print "(/,A)", "*** test number 80 ***"
   call mf_save_and_disable_fpe( )
   x = mf( [ 1.0d0, MF_NAN, 0.0d0, 1.0d0,  0.0d0, -1.5d0,  0.0d0, 0.0d0 ] ) +  &
       mf( [ 0.5d0, 0.0d0, -3.0d0, 0.0d0, MF_NAN,  0.0d0, -1.0d0, 1.0d0 ] )*MF_I
   call msDisplay(x,"x")
   call msSort(mfOut(a,i),x)
   call mf_restore_fpe( )
   call msDisplay(a,"sorted vec")
   call msDisplay(i,"with index")

   ! complex column vector containing some NaNs
   print "(/,A)", "*** test number 81 ***"
   call mf_save_and_disable_fpe( )
   x = mf( [ 1.0d0, MF_NAN, 0.0d0, 1.0d0,  0.0d0, -1.5d0,  0.0d0, 0.0d0 ] ) +  &
       mf( [ 0.5d0, 0.0d0, -3.0d0, 0.0d0, MF_NAN,  0.0d0, -1.0d0, 1.0d0 ] )*MF_I
   x = .h. x
   call msDisplay(x,"x")
   call msSort(mfOut(a,i),x)
   call mf_restore_fpe( )
   call msDisplay(a,"sorted vec")
   call msDisplay(i,"with index")

   print "(/,A)", "*** test number 82 ***"
   call mf_save_and_disable_fpe( )
   x = mf( [ 1.0d0, MF_NAN, 0.0d0,  0.0d0, -1.5d0,  0.0d0 ] ) +         &
       mf( [ 0.5d0, 0.0d0, -3.0d0, MF_NAN,  0.0d0, -1.0d0 ] )*MF_I
   x = .h. x
   call msDisplay(x,"x")
   call msSort(mfOut(a,i),x,'des')
   call mf_restore_fpe( )
   call msDisplay(a,"sorted vec (descending)")
   call msDisplay(i,"with index")

   ! mfIsSorted --------------------------------------------------------
   call print_separation("mfIsSorted")

   ! only vectors
   print "(/,A)", "*** test number 83 ***"
   x = mf( [ 1, 3, 5, 2, 4 ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x) = ", mfIsSorted(x)

   print "(/,A)", "*** test number 84 ***"
   x = mf( [ 1, 2, 3, 4, 5 ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 85 ***"
   x = mf( [ 5, 4, 3, 2, 1 ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 86 ***"
   x = mf( [ 1.0d0, 3.0d0, 5.0d0, MF_NAN, 2.0d0, 4.0d0, MF_NAN ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 87 ***"
   x = mf( [ 1.0d0, 2.0d0, 3.0d0, 4.0d0, 5.0d0, MF_NAN, MF_NAN ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   ! columns vectors
   print "(/,A)", "*** test number 88 ***"
   x = .t. mf( [ 1, 3, 5, 2, 4 ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x) = ", mfIsSorted(x)

   print "(/,A)", "*** test number 89 ***"
   x = .t. mf( [ 1, 2, 3, 4, 5 ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 90 ***"
   x = .t. mf( [ 5, 4, 3, 2, 1 ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 91 ***"
   x = .t. mf( [ 1.0d0, 3.0d0, 5.0d0, MF_NAN, 2.0d0, 4.0d0, MF_NAN ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 92 ***"
   x = .t. mf( [ 1.0d0, 2.0d0, 3.0d0, 4.0d0, 5.0d0, MF_NAN, MF_NAN ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   ! complex values -- row vectors
   print "(/,A)", "*** test number 93 ***"
   x = mf( [ (1.0d0, 0.5d0), (0.0d0, 0.0d0), (0.0d0, 0.0d0),            &
             (0.0d0, -3.0d0), (-1.5d0, 0.0d0), (0.0d0, -2.0d0),         &
             (0.0d0, -1.0d0), (2.0d0, 0.0d0), (0.0d0, 2.0d0) ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 94 ***"
   x = mf( [ (0.0d0, 0.0d0), (0.0d0, 0.0d0), (0.0d0, -1.0d0),           &
             (1.0d0, 0.5d0), (-1.5d0, 0.0d0), (0.0d0, -2.0d0),          &
             (2.0d0, 0.0d0), (0.0d0, 2.0d0), (0.0d0, -3.0d0) ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 95 ***"
   x = mf( [ (0.0d0, -3.0d0), (0.0d0, 2.0d0), (2.0d0, 0.0d0),           &
             (0.0d0, -2.0d0), (-1.5d0, 0.0d0), (1.0d0, 0.5d0),          &
             (0.0d0, -1.0d0), (0.0d0, 0.0d0), (0.0d0, 0.0d0) ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 96 ***"
   x = mf( [ (1.0d0, 0.5d0), (0.0d0, 0.0d0), (MF_NAN, 0.0d0),           &
             (0.0d0, 0.0d0), (0.0d0, -3.0d0), (-1.5d0, 0.0d0),          &
             (1.0d0, MF_NAN), (0.0d0, -2.0d0), (0.0d0, -1.0d0),         &
             (2.0d0, 0.0d0), (0.0d0, 2.0d0) ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 97 ***"
   x = mf( [ (0.0d0, 0.0d0), (0.0d0, 0.0d0), (0.0d0, -1.0d0),           &
             (1.0d0, 0.5d0), (-1.5d0, 0.0d0), (0.0d0, -2.0d0),          &
             (2.0d0, 0.0d0), (0.0d0, 2.0d0), (0.0d0, -3.0d0),           &
             (MF_NAN, 0.0d0), (1.0d0, MF_NAN) ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 98 ***"
   x = mf( [ (0.0d0, -3.0d0), (0.0d0, 2.0d0), (2.0d0, 0.0d0),           &
             (0.0d0, -2.0d0), (-1.5d0, 0.0d0), (1.0d0, 0.5d0),          &
             (0.0d0, -1.0d0), (0.0d0, 0.0d0), (0.0d0, 0.0d0),           &
             (MF_NAN, 0.0d0), (1.0d0, MF_NAN) ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   ! complex values -- column vectors
   print "(/,A)", "*** test number 99 ***"
   x = .h. mf( [ (1.0d0, 0.5d0), (0.0d0, 0.0d0), (0.0d0, 0.0d0),        &
             (0.0d0, -3.0d0), (-1.5d0, 0.0d0), (0.0d0, -2.0d0),         &
             (0.0d0, -1.0d0), (2.0d0, 0.0d0), (0.0d0, 2.0d0) ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 100 ***"
   x = .h. mf( [ (0.0d0, 0.0d0), (0.0d0, 0.0d0), (0.0d0, -1.0d0),       &
             (1.0d0, 0.5d0), (-1.5d0, 0.0d0), (0.0d0, 2.0d0),           &
             (2.0d0, 0.0d0), (0.0d0, -2.0d0), (0.0d0, -3.0d0) ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 101 ***"
   x = .h. mf( [ (0.0d0, -3.0d0), (0.0d0, -2.0d0), (2.0d0, 0.0d0),      &
             (0.0d0, 2.0d0), (-1.5d0, 0.0d0), (1.0d0, 0.5d0),           &
             (0.0d0, -1.0d0), (0.0d0, 0.0d0), (0.0d0, 0.0d0) ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 102 ***"
   x = .h. mf( [ (1.0d0, 0.5d0), (0.0d0, 0.0d0), (MF_NAN, 0.0d0),       &
             (0.0d0, 0.0d0), (0.0d0, -3.0d0), (-1.5d0, 0.0d0),          &
             (1.0d0, MF_NAN), (0.0d0, -2.0d0), (0.0d0, -1.0d0),         &
             (2.0d0, 0.0d0), (0.0d0, 2.0d0) ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 103 ***"
   x = .h. mf( [ (0.0d0, 0.0d0), (0.0d0, 0.0d0), (0.0d0, -1.0d0),       &
             (1.0d0, 0.5d0), (-1.5d0, 0.0d0), (0.0d0, 2.0d0),           &
             (2.0d0, 0.0d0), (0.0d0, -2.0d0), (0.0d0, -3.0d0),          &
             (MF_NAN, 0.0d0), (1.0d0, MF_NAN) ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')

   print "(/,A)", "*** test number 104 ***"
   x = .h. mf( [ (0.0d0, -3.0d0), (0.0d0, -2.0d0), (2.0d0, 0.0d0),      &
             (0.0d0, 2.0d0), (-1.5d0, 0.0d0), (1.0d0, 0.5d0),           &
             (0.0d0, -1.0d0), (0.0d0, 0.0d0), (0.0d0, 0.0d0),           &
             (MF_NAN, 0.0d0), (1.0d0, MF_NAN) ] )
   call msDisplay(x,"x")
   print *, "mfIsSorted(x,'ascend') = ", mfIsSorted(x,'ascend')
   print *, "mfIsSorted(x,'descend') = ", mfIsSorted(x,'descend')
   print *, "mfIsSorted(x,'either') = ", mfIsSorted(x,'either')
   print "()"

   ! mfSortRows --------------------------------------------------------
   call print_separation("mfSortRows")

   print "(/,A)", "*** test number 105 ***"
   x = mfMagic(3)
   call msDisplay(x,"x")
   call msDisplay( mfSortRows(x,1), "sortrows(x,1)" )
   call msDisplay( mfSortRows(x,2), "sortrows(x,2)" )
   call msDisplay( mfSortRows(x,3), "sortrows(x,3)" )

   print "(/,A)", "*** test number 106 ***"
   x = mfMagic(3)
   call msDisplay(x,"x")
   call msDisplay( mfSortRows(x,1,"descending"), "sortrows(x,1,'descending')" )
   call msDisplay( mfSortRows(x,2,"descending"), "sortrows(x,2,'descending')" )
   call msDisplay( mfSortRows(x,3,"descending"), "sortrows(x,3,'descending')" )

   print "(/,A)", "*** test number 107 ***"
   x = mf( [ 3, 6, 2, 5 ] ) .vc.                                        &
       mf( [ 2, 1, 4, 2 ] ) .vc.                                        &
       mf( [ 1, 0, 0, 3 ] ) .vc.                                        &
       mf( [ 2, 1, 5, 1 ] ) .vc.                                        &
       mf( [ 2, 3, 3, 1 ] )
   call msDisplay(x,"x")
   call msDisplay( mfSortRows(x), "sortrows(x)" )
   call msDisplay( mfSortRows(x,[3]), "sortrows(x,[3])" )
   call msDisplay( mfSortRows(x,[1,-3]), "sortrows(x,[1,-3])" )

   ! msSortRows --------------------------------------------------------
   call print_separation("msSortRows")

   print "(/,A)", "*** test number 108 ***"
   x = mfMagic(3)
   call msDisplay(x,"x")
   call msSortRows( mfOut(a,i), x, 1 )
   call msDisplay(a,"a",i,"i")
   call msSortRows( mfOut(a,i), x, 2, "descending" )
   call msDisplay(a,"a",i,"i")
   call msSortRows( mfOut(a,i), x, 3 )
   call msDisplay(a,"a",i,"i")

   print "(/,A)", "*** test number 109 ***"
   x = mf( [ 3, 6, 2, 5 ] ) .vc.                                        &
       mf( [ 2, 1, 4, 2 ] ) .vc.                                        &
       mf( [ 1, 0, 0, 3 ] ) .vc.                                        &
       mf( [ 2, 1, 5, 1 ] ) .vc.                                        &
       mf( [ 2, 3, 3, 1 ] )
   call msDisplay(x,"x")
   call msSortRows( mfOut(a,i), x )
   call msDisplay(a,"a",i,"i")
   call msSortRows( mfOut(a,i), x, [3] )
   call msDisplay(a,"a",i,"i")
   call msSortRows( mfOut(a,i), x, [1,-3] )
   call msDisplay(a,"a",i,"i")

   ! mfDiff ------------------------------------------------------------
   call print_separation("mfDiff")

   print "(/,A)", "*** test number 110 ***"
   x = mf( [ 1, 2, 3, 2, 1 ] )
   call msDisplay(x,"x")
   call msDisplay(mfDiff(x),"mfDiff(x)")

   print "(/,A)", "*** test number 111 ***"
   x = .t. x
   call msDisplay(x,"x")
   call msDisplay(mfDiff(x),"mfDiff(x)")

   ! mfMean ------------------------------------------------------------
   call print_separation("mfMean")

   print "(/,A)", "*** test number 112 ***"
   x = mf( [ 1, 2, 3, 4, 5 ] )
   call msDisplay(x,"x")
   call msDisplay(mfMean(x),"mfMean(x)")

   print "(/,A)", "*** test number 113 ***"
   x = mfMagic(3)
   call msDisplay(x,"x")
   call msDisplay(mfMean(x),"mfMean(x)")
   print "(/,A)", "*** test number 114 ***"
   call msDisplay(mfMean(x,1),"mfMean(x,1)")
   print "(/,A)", "*** test number 115 ***"
   call msDisplay(mfMean(x,2),"mfMean(x,2)")

   ! mfVar -------------------------------------------------------------
   call print_separation("mfVar")

   print "(/,A)", "*** test number 116 ***"
   x = mf( [ 1, 2, 3, 4, 5 ] )
   call msDisplay(x,"x")
   call msDisplay(mfVar(x),"mfVar(x)")

   print "(/,A)", "*** test number 117 ***"
   x = mfMagic(3)
   call msDisplay(x,"x")
   call msDisplay(mfVar(x),"mfVar(x)")
   print "(/,A)", "*** test number 118 ***"
   call msDisplay(mfVar(x,1),"mfVar(x,1)")
   print "(/,A)", "*** test number 119 ***"
   call msDisplay(mfVar(x,2),"mfVar(x,2)")

   ! mfStd -------------------------------------------------------------
   call print_separation("mfStd")

   print "(/,A)", "*** test number 120 ***"
   x = mf( [ 1, 2, 3, 4, 5 ] )
   call msDisplay(x,"x")
   call msDisplay(mfStd(x),"mfStd(x)")

   print "(/,A)", "*** test number 121 ***"
   x = mfMagic(3)
   call msDisplay(x,"x")
   call msDisplay(mfStd(x),"mfStd(x)")
   print "(/,A)", "*** test number 122 ***"
   call msDisplay(mfStd(x,1),"mfStd(x,1)")
   print "(/,A)", "*** test number 123 ***"
   call msDisplay(mfStd(x,2),"mfStd(x,2)")

   ! mfRMS -------------------------------------------------------------
   call print_separation("mfRMS")

   print "(/,A)", "*** test number 124 ***"
   x = mfSin( mfLinSpace(0.0d0,10*MF_PI,200) )
   print *, "x = [ 0:10*pi ] with 200 values"
   call msDisplay(mfRMS(x),"mfRMS(x)")
   call msDisplay(mfStd(x),"mfStd(x)")

   print "(/,A)", "*** test number 125 ***"
   x = x + 1.0d0
   print *, "adding 1 to x"
   call msDisplay(mfRMS(x),"mfRMS(x)")
   call msDisplay(mfStd(x),"mfStd(x)")

   ! mfFourierCos, mfInvFourierCos -------------------------------------
   call print_separation("mfFourierCos, mfInvFourierCos")

   print "(/,A)", "*** test number 126 ***"
   x = .t. mfLinspace( 0.0d0, MF_PI, 11 )
   y = mfCos( 5.0d0*x )
   call msDisplay(y,"y")
   z = mfFourierCos(y)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFourierCos(y)")

   print "(/,A)", "*** test number 127 ***"
   call msDisplay( mfNorm(mfInvFourierCos(z)-y) < 20*MF_EPS,            &
                "| mfInvFourierCos(z) - y | < 20*MF_EPS")

   ! mfFourierSin, mfInvFourierSin -------------------------------------
   call print_separation("mfFourierSin, mfInvFourierSin")

   print "(/,A)", "*** test number 128 ***"
   x = .t. mfLinspace( 0.0d0, MF_PI, 11 )
   y = mfSin( 5.0d0*x )
   call msDisplay(y,"y")
   z = mfFourierSin(y)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFourierSin(y)")

   print "(/,A)", "*** test number 129 ***"
   call msDisplay( mfNorm(mfInvFourierSin(z)-y) < 20*MF_EPS,            &
                "| mfInvFourierSin(z) - y | < 20*MF_EPS")

   ! mfFourierLeg, mfInvFourierLeg -------------------------------------
   call print_separation("mfFourierLeg, mfInvFourierLeg")

   print "(/,A)", "*** test number 130 ***"
   theta = mfLinspace( -MF_PI, 0.0d0, 6 )
   x = mfCos(theta)
   c = [ 0.0, 1.0, 1.0, 0.0, 0.0, 0.0 ]
   y = mfZeros(1,6)
   do k = 1, 6
      y = y + mfGet(c,k)*mfLegendre(k-1,x)
   end do
   call msDisplay( y, "y" )
   z = mfFourierLeg(y)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay( z, "z = mfFourierLeg(y)" )

   print "(/,A)", "*** test number 131 ***"
   call msDisplay( mfNorm(mfInvFourierLeg(z)-y) < 20*MF_EPS,            &
                "| mfInvFourierLeg(z) - y | < 20*MF_EPS")

   ! msHist ------------------------------------------------------------
   call print_separation("msHist")

   ! discrete values
   print "(/,A)", "*** test number 132 ***"
   x = .t. mf( [ 12, 11, 8, 10, 6, 8, 9, 10, 12, 9, 10, 8, 6, 10 ] )
   call msHist( mfOut( y, z ), x, 6.0d0-0.5d0, 12.0d0+0.5d0, 7 )
   call msDisplay( y, "y (hist)", z, "z (x_bin)" )

   print "(/,A)", "*** test number 133 ***"
   x = mfLinspace( -MF_PI, MF_PI, 100 )
   x = .t. mfCos(x)
   call msHist( mfOut( y, z ), x, -1.0d0, 1.0d0, 10 )
   call msDisplay( y, "y (hist)", z, "z (x_bin)" )

   ! mfMedian ----------------------------------------------------------
   call print_separation("mfMedian")

   print "(/,A)", "*** test number 134 ***"
   x = [ 0, 3, 5, 2, 8, 4, 3, 5, 3, 2, 5, 3 ]
   call msDisplay(x,"x")
   call msDisplay(mfMedian(x),"mfMedian(x)")

   print "(/,A)", "*** test number 135 ***"
   x = [ 0, 3, 5, 2, 8, 4, 3, 5, 3, 2, 5 ]
   call msDisplay(x,"x")
   call msDisplay(mfMedian(x),"mfMedian(x)")

   print "(/,A)", "*** test number 136 ***"
   x = mf( [ 1, 2, 4, 4 ] ) .vc.                                        &
       mf( [ 3, 4, 6, 6 ] ) .vc.                                        &
       mf( [ 5, 6, 8, 8 ] )
   call msDisplay(x,"x")
   call msDisplay(mfMedian(x),"mfMedian(x)")
   print "(/,A)", "*** test number 137 ***"
   call msDisplay(mfMedian(x,2),"mfMedian(x,2)")

   ! mfMoments ---------------------------------------------------------
   call print_separation("mfMoments")

   print "(/,A)", "*** test number 138 ***"
   x = [ 0, 3, 5, 2, 8, 4, 3, 5, 3, 2, 5, 3, 4, 6, 4, 8, 9, 6, 4, 7,    &
         8, 5, 3, 7, 4, 9, 5, 2, 7, 4, 9, 0, 5, 7, 4, 7, 3, 4, 8, 4,    &
         6, 8, 4, 3, 9, 6, 4, 8, 6, 4, 8, 9, 3, 2, 8, 9, 5, 4, 8, 6 ]
   call msDisplay(x,"x")
   call msDisplay(mfMoments(x),"mfMoments(x)")

   ! mfQuantile ---------------------------------------------------------
   call print_separation("mfQuantile")

   print "(/,A)", "*** test number 139 ***"
   v = [ 5.04, 4.9, 4.83, 5.24, 4.79, 5.10, 4.96 ]
   call msDisplay(v,"v")
   call msDisplay(mfQuantile(v,0.1d0),"mfQuantile(v,0.1d0)")
   call msDisplay(mfQuantile(v,0.9d0),"mfQuantile(v,0.9d0)")

   ! mfExtrema ---------------------------------------------------------
   call print_separation("mfExtrema")

   print "(/,A)", "*** test number 140 ***"
   x = [ 5, 9, 3, 6, 0, 2 ]
   call msDisplay(x,"x")
   call msDisplay(mfExtrema(x),"mfExtrema(x)")

   ! mfGradient --------------------------------------------------------
   call print_separation("mfGradient")

   print "(/,A)", "*** test number 141 ***"
   x = mfLinSpace(0.0d0,1.0d0,4)
   call msDisplay(x,"x")
   z = x**2
   call msDisplay(z,"z")
   x = mfGradient( z, 1.0d0/3 )
   print *, " At nodes:"
   call msDisplay(x,"grad(z)")

   print "(/,A)", "*** test number 142 ***"
   x = mfLinSpace(0.0d0,1.0d0,4)
   x = x .vc. x
   call msDisplay(x,"x")
   z = x**2
   call msDisplay(z,"z")
   x = mfGradient( z, 1.0d0/3, dim=2 )
   print *, " At nodes:"
   call msDisplay(x,"grad(z) along 2nd dimension")

   print "(/,A)", "*** test number 143 ***"
   x = .t. mfLinSpace(0.0d0,1.0d0,4)
   call msDisplay(x,"x")
   z = x**2
   call msDisplay(z,"z")
   x = mfGradient( z, 1.0d0/3 )
   print *, " At nodes:"
   call msDisplay(x,"grad(z)")

   print "(/,A)", "*** test number 144 ***"
   x = .t. mfLinSpace(0.0d0,1.0d0,4)
   x = x .hc. x
   call msDisplay(x,"x")
   z = x**2
   call msDisplay(z,"z")
   x = mfGradient( z, 1.0d0/3, dim=1 )
   print *, " At nodes:"
   call msDisplay(x,"grad(z) along 1st dimension")

   print "(/,A)", "*** test number 145 ***"
   x = mfLinSpace(0.0d0,1.0d0,4)
   call msDisplay(x,"x")
   z = x**2
   call msDisplay(z,"z")
   x = mfGradient( z, 1.0d0/3, location="centered" )
   print *, " At cells' centers:"
   call msDisplay(x,"grad(z)")

   print "(/,A)", "*** test number 146 ***"
   x = mfLinSpace(0.0d0,1.0d0,4)
   x = x .vc. x
   call msDisplay(x,"x")
   z = x**2
   call msDisplay(z,"z")
   x = mfGradient( z, 1.0d0/3, dim=2, location="centered" )
   print *, " At cells' centers:"
   call msDisplay(x,"grad(z) along 2nd dimension")

   print "(/,A)", "*** test number 147 ***"
   x = .t. mfLinSpace(0.0d0,1.0d0,4)
   call msDisplay(x,"x")
   z = x**2
   call msDisplay(z,"z")
   x = mfGradient( z, 1.0d0/3, location="centered" )
   print *, " At cells' centers:"
   call msDisplay(x,"grad(z)")

   print "(/,A)", "*** test number 148 ***"
   x = .t. mfLinSpace(0.0d0,1.0d0,4)
   x = x .hc. x
   call msDisplay(x,"x")
   z = x**2
   call msDisplay(z,"z")
   x = mfGradient( z, 1.0d0/3, dim=1, location="centered" )
   print *, " At cells' centers:"
   call msDisplay(x,"grad(z) along 1st dimension")

   ! msGradient --------------------------------------------------------
   call print_separation("msGradient")

   print "(/,A)", "*** test number 149 ***"
   call msMeshGrid( mfOut(x,y), mfLinSpace(0.0d0,1.0d0,4),              &
                            .t. mfLinSpace(1.0d0,0.0d0,5) )
   call msDisplay(x,"x",y,"y")
   z = (x+y)**2
   call msDisplay(z,"z")
   call msGradient( mfOut(Fy,Fx), z, -1.0d0/4, 1.0d0/3 )
   print *, " At nodes:"
   call msDisplay(Fx,"grad_x(z)",Fy,"grad_y(z)")

   print "(/,A)", "*** test number 150 ***"
   call msMeshGrid( mfOut(x,y), mfLinSpace(0.0d0,1.0d0,4),              &
                            .t. mfLinSpace(1.0d0,0.0d0,5) )
   call msDisplay(x,"x",y,"y")
   z = (x+y)**2
   call msDisplay(z,"z")
   call msGradient( mfOut(Fy,Fx), z, -1.0d0/4, 1.0d0/3, location="centered" )
   print *, " At cells' centers:"
   call msDisplay(Fx,"grad_x(z)",Fy,"grad_y(z)")

   ! mfFFT, mfInvFFT ---------------------------------------------------
   call print_separation("mfFFT, mfInvFFT")

   ! real case -- row vector
   print "(/,A)", "*** test number 151 ***"
   x = mfLinspace( 0.0d0, 2.0d0*MF_PI, 13 )
   call msSet( MF_EMPTY, x, 13 )
   y = mfCos( 2.0d0*x )
   call msDisplay(y,"y")
   z = mfFFT(y)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFFT(y)")

   print "(/,A)", "*** test number 152 ***"
   call msDisplay( mfNorm(mfInvFFT(z)-y) < 20*MF_EPS,                   &
                "| mfInvFFT(z) - y | < 20*MF_EPS")

   ! real case -- column vector
   print "(/,A)", "*** test number 153 ***"
   x = .t. mfLinspace( 0.0d0, 2.0d0*MF_PI, 13 )
   call msSet( MF_EMPTY, x, 13 )
   y = mfSin( 4.0d0*x )
   call msDisplay(y,"y")
   z = mfFFT(y)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFFT(y)")

   print "(/,A)", "*** test number 154 ***"
   call msDisplay( mfNorm(mfInvFFT(z)-y) < 20*MF_EPS,                   &
                "| mfInvFFT(z) - y | < 20*MF_EPS")

   ! complex case -- row vector
   print "(/,A)", "*** test number 155 ***"
   x = mfLinspace( 0.0d0, 2.0d0*MF_PI, 13 )
   call msSet( MF_EMPTY, x, 13 )
   y = mfCos( 2.0d0*x ) + MF_I*mfSin( 4.0d0*x )
   call msDisplay(y,"y")
   z = mfFFT(y)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFFT(y)")

   print "(/,A)", "*** test number 156 ***"
   call msDisplay( mfNorm(mfInvFFT(z)-y) < 20*MF_EPS,                   &
                "| mfInvFFT(z) - y | < 20*MF_EPS")

   ! complex case -- column vector
   print "(/,A)", "*** test number 157 ***"
   x = .t. mfLinspace( 0.0d0, 2.0d0*MF_PI, 13 )
   call msSet( MF_EMPTY, x, 13 )
   y = mfCos( 2.0d0*x ) + MF_I*mfSin( 4.0d0*x )
   call msDisplay(y,"y")
   z = mfFFT(y)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFFT(y)")

   print "(/,A)", "*** test number 158 ***"
   call msDisplay( mfNorm(mfInvFFT(z)-y) < 20*MF_EPS,                   &
                "| mfInvFFT(z) - y | < 20*MF_EPS")

   ! real case -- matrix
   print "(/,A)", "*** test number 159 ***"
   x = .t. mfLinspace( 0.0d0, 2.0d0*MF_PI, 13 )
   call msSet( MF_EMPTY, x, 13 )
   y = MF_EMPTY
   do k = 1, 6
      y = y .hc. mfCos( dble(k)*x )
   end do
   call msDisplay(y,"y")
   z = mfFFT(y)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFFT(y)")

   print "(/,A)", "*** test number 160 ***"
   call msDisplay( mfNorm(mfInvFFT(z)-y) < 25*MF_EPS,                   &
                "| mfInvFFT(z) - y | < 25*MF_EPS")

   ! use of dim=1
   print "(/,A)", "*** test number 161 ***"
   x = .t. mfLinspace( 0.0d0, 2.0d0*MF_PI, 13 )
   call msSet( MF_EMPTY, x, 13 )
   y = MF_EMPTY
   do k = 1, 6
      y = y .hc. mfCos( dble(k)*x )
   end do
   call msDisplay(y,"y")
   z = mfFFT(y,dim=1)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFFT(y,dim=1)")

   print "(/,A)", "*** test number 162 ***"
   call msDisplay( mfNorm(mfInvFFT(z,dim=1)-y) < 25*MF_EPS,             &
                "| mfInvFFT(z,dim=1) - y | < 25*MF_EPS")

   ! use of dim=2
   print "(/,A)", "*** test number 163 ***"
   x = .t. mfLinspace( 0.0d0, 2.0d0*MF_PI, 13 )
   call msSet( MF_EMPTY, x, 13 )
   y = MF_EMPTY
   do k = 1, 6
      y = y .hc. mfCos( dble(k)*x )
   end do
   call msDisplay(y,"y")
   z = mfFFT(y,dim=2)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFFT(y,dim=2)")

   print "(/,A)", "*** test number 164 ***"
   call msDisplay( mfNorm(mfInvFFT(z,dim=2)-y) < 25*MF_EPS,             &
                "| mfInvFFT(z,dim=2) - y | < 25*MF_EPS")

   ! complex case -- matrix
   print "(/,A)", "*** test number 165 ***"
   x = .t. mfLinspace( 0.0d0, 2.0d0*MF_PI, 13 )
   call msSet( MF_EMPTY, x, 13 )
   y = MF_EMPTY
   do k = 1, 6
      y = y .hc. ( mfCos( dble(k)*x ) + MF_I*mfSin( dble(k)*x ) )
   end do
   call msDisplay(y,"y")
   z = mfFFT(y)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFFT(y)")

   print "(/,A)", "*** test number 166 ***"
   call msDisplay( mfNorm(mfInvFFT(z)-y) < 50*MF_EPS,                   &
                "| mfInvFFT(z) - y | < 50*MF_EPS")

   ! use of dim=1
   print "(/,A)", "*** test number 167 ***"
   x = .t. mfLinspace( 0.0d0, 2.0d0*MF_PI, 13 )
   call msSet( MF_EMPTY, x, 13 )
   y = MF_EMPTY
   do k = 1, 6
      y = y .hc. ( mfCos( dble(k)*x ) + MF_I*mfSin( dble(k)*x ) )
   end do
   call msDisplay(y,"y")
   z = mfFFT(y,dim=1)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFFT(y,dim=1)")

   print "(/,A)", "*** test number 168 ***"
   call msDisplay( mfNorm(mfInvFFT(z,dim=1)-y) < 50*MF_EPS,             &
                "| mfInvFFT(z,dim=1) - y | < 50*MF_EPS")

   ! use of dim=2
   print "(/,A)", "*** test number 169 ***"
   x = .t. mfLinspace( 0.0d0, 2.0d0*MF_PI, 13 )
   call msSet( MF_EMPTY, x, 13 )
   y = MF_EMPTY
   do k = 1, 6
      y = y .hc. ( mfCos( dble(k)*x ) + MF_I*mfSin( dble(k)*x ) )
   end do
   call msDisplay(y,"y")
   z = mfFFT(y,dim=2)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFFT(y,dim=2)")

   print "(/,A)", "*** test number 170 ***"
   call msDisplay( mfNorm(mfInvFFT(z,dim=2)-y) < 50*MF_EPS,             &
                "| mfInvFFT(z,dim=2) - y | < 50*MF_EPS")

   ! mfFFT2, mfInvFFT2 -------------------------------------------------
   call print_separation("mfFFT2, mfInvFFT2")

   ! real matrix
   print "(/,A)", "*** test number 171 ***"
   x = .t. mfLinspace( 0.0d0, 2.0d0*MF_PI, 13 )
   call msSet( MF_EMPTY, x, 13 )
   y = MF_EMPTY
   do k = 1, 6
      y = y .hc. mfCos( dble(k)*x )
   end do
   call msDisplay(y,"y")
   z = mfFFT2(y)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFFT2(y)")

   print "(/,A)", "*** test number 172 ***"
   call msDisplay( mfNorm(mfInvFFT2(z)-y) < 25*MF_EPS,                  &
                "| mfInvFFT2(z) - y | < 25*MF_EPS")

   ! complex matrix
   print "(/,A)", "*** test number 173 ***"
   x = .t. mfLinspace( 0.0d0, 2.0d0*MF_PI, 13 )
   call msSet( MF_EMPTY, x, 13 )
   y = MF_EMPTY
   do k = 1, 6
      y = y .hc. ( mfCos( dble(k)*x ) + MF_I*mfSin( dble(k)*x ) )
   end do
   call msDisplay(y,"y")
   z = mfFFT2(y)
   call zeroes_small_values( z, 1.0d-12 )
   call msDisplay(z,"z = mfFFT2(y)")

   print "(/,A)", "*** test number 174 ***"
   call msDisplay( mfNorm(mfInvFFT2(z)-y) < 50*MF_EPS,                  &
                "| mfInvFFT2(z) - y | < 50*MF_EPS")

   ! mfFunFit ----------------------------------------------------------
   call print_separation("mfFunFit")

   print "(/,A)", "*** test number 175 ***"
   print "()"
   print *, "Fitting of data points by a Boltzmann law"
   x = mfColon(1,19)
   x = ( x - 1.0 ) * 10.0
   x = .t. x

   y = [ 0., 0., 0.05, 0.08, 0.1, 0.15, 0.18, 0.2, 0.3,                 &
         0.4, 0.5, 0.6, 0.7, 0.8, 0.85, 0.89, 0.9, 0.95, 1. ]
   y = .t. y

   n = 2
   p = [ 80., 30. ]
   tol = 0.01
   z = mfFunFit( x, y, boltzmann, p, n, tol )
   call msDisplay( z, "parameters of fitting" )
   if( All(mfAbs(z-mf([99.3691,24.3905])) > 1.0d-3) ) then
      print *, "   *** mfFunFit failed! ***   <---------------"
   end if

   ! msFunFit ----------------------------------------------------------
   call print_separation("msFunFit")

   print "(/,A)", "*** test number 176 ***"
   print "()"
   print *, "Fitting of data points by a Boltzmann law"
   x = mfColon(1,19)
   x = ( x - 1.0 ) * 10.0
   x = .t. x

   y = [ 0., 0., 0.05, 0.08, 0.1, 0.15, 0.18, 0.2, 0.3,                 &
         0.4, 0.5, 0.6, 0.7, 0.8, 0.85, 0.89, 0.9, 0.95, 1. ]
   y = .t. y

   n = 2
   p = [ 80., 30. ]
   tol = 0.01
   call msFunFit( mfout(z,r2), x, y, boltzmann, p, n, tol )
   call msDisplay( z, "parameters of fitting" )
   call msDisplay( r2, "correlation coefficient" )
   if( All(mfAbs(z-mf([99.3691,24.3905])) > 1.0d-3) ) then
      print *, "   *** mfFunFit failed! ***   <---------------"
   end if

   ! mfSmooth ----------------------------------------------------------
   call print_separation("mfSmooth")

   print "(/,A)", "*** test number 177 ***"
   print "()"
   print *, "Smoothing a row vector of alternate values"
   v = [ ( (-1)**j, j = 1, 10 ) ]
   call msDisplay( v, "v" )
   ! default span
   call msDisplay( mfSmooth(v), "mfSmooth(v) [with default span of 5]" )
   ! other span values
   call msDisplay( mfSmooth(v,3), "mfSmooth(v,3)" )
   call msDisplay( mfSmooth(v,7), "mfSmooth(v,7)" )
   call msDisplay( mfSmooth(v,9), "mfSmooth(v,9)" )
   call msDisplay( mfSmooth(v,11), "mfSmooth(v,11)" )
   call msDisplay( mfSmooth(v,19), "mfSmooth(v,19)" )

   print "(/,A)", "*** test number 178 ***"
   print "()"
   print *, "Smoothing a col vector of alternate values"
   v = [ ( (-1)**j, j = 1, 10 ) ]
   v = .t. v
   call msDisplay( v, "v" )
   ! default span
   call msDisplay( mfSmooth(v), "mfSmooth(v) [with default span of 5]" )
   ! other span values
   call msDisplay( mfSmooth(v,3), "mfSmooth(v,3)" )
   call msDisplay( mfSmooth(v,7), "mfSmooth(v,7)" )
   call msDisplay( mfSmooth(v,9), "mfSmooth(v,9)" )
   call msDisplay( mfSmooth(v,11), "mfSmooth(v,11)" )
   call msDisplay( mfSmooth(v,19), "mfSmooth(v,19)" )

   print "()"

   ! mfXCorr -----------------------------------------------------------
   call print_separation("mfXCorr")

   print "(/,A)", "*** test number 179 ***"
   print "()"
   print *, "Autocorrelation of a vector of real values"
   v = [ 0.0d0, 1.0d0, 1.0d0, 1.0d0, 0.0d0, 0.0d0, 1.0d0, 1.0d0, 0.0d0, &
         0.0d0, 0.0d0, 1.0d0, 1.0d0, 1.0d0, 0.0d0, 0.0d0, 0.0d0, 0.0d0, &
         1.0d0, 1.0d0, 0.0d0, 0.0d0, 1.0d0, 1.0d0, 1.0d0, 0.0d0, 0.0d0 ]
   v = .t. v
   y = mfXCorr( v, scale="normalized" )
   call msDisplay( y, 'mfXCorr(v,scale="normalized")' )

   print "(/,A)", "*** test number 180 ***"
   print "()"
   print *, "Partial autocorrelation of a vector of real values"
   y = mfXCorr( v, maxlag=6, scale="normalized" )
   call msDisplay( y, 'mfXCorr(v,maxlag=6,scale="normalized")' )

   print "()"

   ! mfXCorr2 ----------------------------------------------------------
   call print_separation("mfXCorr2")

   print "(/,A)", "*** test number 181 ***"
   print "()"
   print *, "Autocorrelation of a matrix of real values"
   v = [ 0.0d0, 0.0d0, 1.0d0, 1.0d0, 0.0d0, 0.0d0, 1.0d0, 1.0d0 ]
   A = mfMul( .t.v, v )
   call msDisplay( A, "A" )
   C = mfXCorr2( A )
   call msDisplay( C, 'mfXCorr2(A)' )

   print "(/,A)", "*** test number 182 ***"
   print "()"
   print *, "Partial autocorrelation of a vector of real values"
   C = mfXCorr2( A, maxlag_r=4, maxlag_c=4 )
   call msDisplay( C, 'mfXCorr2(A,maxlag_r=4,maxlag_c=4")' )

   print "()"

   ! End of tests ------------------------------------------------------
   print "(A)", "                         ┏━━━━━━━━━━━━━━━━━━━━━━━━━┓"
   print "(A)", "                         ┃ End of test_mod_datafun ┃"
   print "(A)", "                         ┗━━━━━━━━━━━━━━━━━━━━━━━━━┛"

   !--------------------------------------------------------------------
99 continue

   print "()"
   call msSetMsgLevel(2)

   call msRelease( x, y, z, a, i, c, theta )
   call msRelease( res, x1, x2, p, r2, status, v )
   call msRelease( Fx, Fy )

end program
