module mod_polyfun ! Polynomial functions

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!-----------------------------------------------------------------------
!                             used modules
!-----------------------------------------------------------------------

   use mod_matfun

   use mod_fileio

#ifndef _DEVLP
   use mod_mfdebug ! required for the 2nd pass of the double compilation
   use mod_mfarray ! required for the 2nd pass of the double compilation
   use mod_core ! required for the 2nd pass of the double compilation
#endif

   use splines

   use bezier

   use minpack

   implicit none

#ifndef _DEVLP
   private
#endif

   interface mfTriSearch
      module procedure mfTriSearch_mfarray
      module procedure mfTriSearch_double
   end interface mfTriSearch
   !------ API end ------

   interface mfNodeSearch
      module procedure mfNodeSearch_mfarray
      module procedure mfNodeSearch_double
   end interface mfNodeSearch
   !------ API end ------

   interface mfTetraSearch
      module procedure mfTetraSearch_mfarray
      module procedure mfTetraSearch_double
   end interface mfTetraSearch
   !------ API end ------

   interface mfNodeSearch3D
      module procedure mfNodeSearch3D_mfarray
      module procedure mfNodeSearch3D_double
   end interface mfNodeSearch3D
   !------ API end ------

   interface msRelease
      module procedure msRelease_mfTriConnect
      module procedure msRelease_mfPLdomain
      module procedure msRelease_mfPLdomain3D
      module procedure msRelease_mfTetraConnect
      module procedure msRelease_mfVoronoiStruct
   end interface msRelease
   !------ API end ------

   interface assignment(=)
      module procedure msAssign_mfTriConnect_copy
      module procedure msAssign_mfVoronoiStruct_copy
   end interface assignment(=)
   !------ API end ------

   interface msAssign
      module procedure msAssign_mfTriConnect
!!      module procedure msAssign_mfVoronoiStruct
   end interface msAssign
   !------ API end ------

#ifdef _INTEL_IFC
   public :: msAssign
#endif

   type, public :: mfPLdomain ! Piecewise Linear domain (2D description)
      ! n_xy(nn,2): nodes by 2D-coordinates       (nn >= 3)
      real(kind=MF_DOUBLE), allocatable :: n_xy(:,:)
      ! edge_n(ne,2): edges by nodes              (ne >= 3)
      integer,              allocatable :: edge_n(:,:)
      ! holes_xy(nh,2): holes by 2D-coordinates   (nh >= 0)
      real(kind=MF_DOUBLE), allocatable :: holes_xy(:,:)
   end type mfPLdomain

   type, public :: mfPLdomain3D ! 3D Piecewise Linear domain (3D description)
      ! n_xyz(nn,3): nodes by 3D-coordinates      (nn >= 4)
      real(kind=MF_DOUBLE), allocatable :: n_xyz(:,:)
      ! edge_n(ne,2): edges by nodes              (ne >= 6)
      integer, allocatable :: edge_n(:,:)
      ! face_e(nf): faces by egdes.               (nf >= 4)
      ! It is a vector of list of edges. Each list contains the
      ! indices of k edges defining a face. Typically k=3 (triangles)
      ! or k=4 (quadrangles), but could be larger; this is why we use
      ! a varying size integer list.
      type(mf_Int_List), allocatable :: face_e(:)
      ! holes_xyz(nh,3): holes by 3D-coordinates  (nh >= 0)
      real(kind=MF_DOUBLE), allocatable :: holes_xyz(:,:)
   end type mfPLdomain3D

   ! type(mfTriConnect): declared in mod_fileio
   !                     (cf. file 'fml_polyfun/mfTriConnect.inc')
   public :: mfTriConnect

   type, public :: mfTetraConnect ! Tetrahedral Connectivity
#ifndef _DEVLP
      private
#endif
      logical :: init = .false.
      ! nn: nb of nodes
      ! n_xyz(nn,3): nodes by 3D-coordinates
      real(kind=MF_DOUBLE), pointer :: n_xyz(:,:) => null()
      ! nt: nb of tetrahedra
      ! tetra_n(nt,4): tetrahedra by nodes -- direct orientation
      !                ([1,2,3] is a direct triangle if 4 is above them)
      integer,              pointer :: tetra_n(:,:) => null()
      ! tetra_neighbors(nt,4): tetrahedra by tetra (neighbors)
      integer,              pointer :: tetra_neighbors(:,:) => null()
      ! nodes sorted: coord. 'x' (whatever coord. 'y' and 'z')
      integer,              pointer :: nodes_sorted_x(:) => null()
      ! nodes sorted: coord. 'y' (whatever coord. 'x' and 'z')
      integer,              pointer :: nodes_sorted_y(:) => null()
      ! nodes sorted: coord. 'z' (whatever coord. 'x' and 'y')
      integer,              pointer :: nodes_sorted_z(:) => null()
      ! nf : nb of faces
      ! face_n(nf,3): faces by nodes
      integer,              pointer :: face_n(:,:) => null()
      ! face_tetra(nf,2): face common to 2 tetra
      integer,              pointer :: face_tetra(:,:) => null()
      ! tetra_f(nt,4): tetrahedra by faces
      integer,              pointer :: tetra_f(:,:) => null()
      ! n_tetra(nn): one tetra which contains a given node
      integer,              pointer :: n_tetra(:) => null()
   end type mfTetraConnect

   type, public :: mfVoronoiStruct ! Voronoi structure (2D)
      integer :: init = 0 ! 0: nothing is defined
                          ! 1: only vertices are defined
                          ! 2: only neighbors are defined
                          ! 3: both are defined
      ! nn: nb of nodes
      ! n_xy(nn,2): nodes by 2D-coordinates
      real(kind=MF_DOUBLE), pointer :: n_xy(:,:) => null()
      ! nv: total nb of vertices
      ! v_xy(nn,2): vertices (of Voronoi cells) by 2D-coordinates
      real(kind=MF_DOUBLE), pointer :: v_xy(:,:) => null()
      ! vertices(nn): list of vertices by Voronoi cell
      type(mf_Int_List), pointer :: vertices(:) => null()
      ! neighbors(nn): list of neighbors by Voronoi cell
      type(mf_Int_List), pointer :: neighbors(:) => null()
      logical :: status_temporary = .false.
   end type mfVoronoiStruct

   type, public :: mfVoronoiStruct3D ! Voronoi structure (3D)
      integer :: init = 0 ! 0: nothing is defined
                          ! 1: only vertices are defined
                          ! 2: only neighbors are defined
                          ! 3: both are defined
      ! nn: nb of nodes
      ! n_xyz(nn,3): nodes by 3D-coordinates
      real(kind=MF_DOUBLE), pointer :: n_xyz(:,:) => null()
      ! nv: total nb of vertices
      ! v_xyz(nn,3): vertices (of Voronoi cells) by 3D-coordinates
      real(kind=MF_DOUBLE), pointer :: v_xyz(:,:) => null()
      ! vertices(nn): list of vertices by Voronoi cell
      type(mf_Int_List), pointer :: vertices(:) => null()
      ! neighbors(nn): list of neighbors by Voronoi cell
      type(mf_Int_List), pointer :: neighbors(:) => null()
      logical :: status_temporary = .false.
   end type mfVoronoiStruct3D

   interface msPrepProgress
      module procedure msPrepProgress_int
      module procedure msPrepProgress_dble
   end interface

   interface msPrintProgress
      module procedure msPrintProgress_int
      module procedure msPrintProgress_dble
   end interface

   interface mfInterp1
      module procedure mfInterp1_real
      module procedure mfInterp1_bool
   end interface

   interface mfInterp2
      module procedure mfInterp2_real
      module procedure mfInterp2_bool
   end interface

   interface mfPPVal
      module procedure mfPPVal_real
      module procedure mfPPVal_bool
   end interface

   interface mfPPDer
      module procedure mfPPDer_real
      module procedure mfPPDer_bool
   end interface

   interface msDelaunay
      module procedure msDelaunay_xy
      module procedure msDelaunay_PLdomain
   end interface

   interface mfDelaunay3D
      module procedure mfDelaunay3D_vec
      module procedure mfDelaunay3D_mat
   end interface

   interface msDelaunay3D
      module procedure msDelaunay3D_PLC3D
   end interface

   interface msBuildTetraConnect
      module procedure msBuildTetraConnect_vec
      module procedure msBuildTetraConnect_mat
   end interface

   public :: mfPolyVal, &
             mfPolyFit, &
             msPolyFit, &
             mfSpline, msSpline, &
             mfPPVal, &
             mfPPDer, &
             mfInterp1, &
             mfInterp2

   public :: mfDelaunay, msDelaunay, &
             msBuildTriConnect, &
             msUpdateTriConnect, &
             msExtractTriConnect, &
             msCheckDomainConvexity, &
             msTriNodeNeighbors, &
             mfTriSearch, &
             mfNodeSearch, &
             mfGridData, &
             mfGridData3D, &
             msPrintTriConnect, &
             msPrintPLdomain, &
             msCheckPLDomainValidity, &
             mfVoronoi, &
             msPrintVoronoi

   public :: mfDelaunay3D, msDelaunay3D, &
             msBuildTetraConnect, &
             msDel3DNodeNeighbors, &
             mfTetraSearch, &
             mfNodeSearch3D, &
             msEndDelaunay3D, &
             msPrintTetraConnect, &
             msPrintPLdomain3D, &
             msCheckPLDomain3DValidity

   public :: mfLegendre, &
             mfFourierLeg, &
             mfInvFourierLeg, &
             mfRoots, &
             mfPoly, &
             msRelease, &
             msPrepProgress, &
             msPrintProgress, &
             msPostProgress

   public :: mfFunFit, &
             msFunFit

   ! global variables for 'Progress'
   type(mfArray), save, private :: progress_x, progress_y,              &
                                   progress_linfit

   private :: binomials_n_2n, leg_in_cos

   ! delaunay 3D
   integer(kind=MF_ADDRESS) :: delaunay3d_addr

   ! global data for mf/msFunFit
   real(kind=MF_DOUBLE), save, allocatable :: x_data(:), y_data(:)

   private :: x_data, y_data,                                           &
              msRelease_mfTriConnect, msRelease_mfPLdomain,             &
              msRelease_mfTetraConnect, msAssign_mfTriConnect,          &
              msRelease_mfVoronoiStruct

   ! internal routine used in msStreamline
   public :: build_tri_from_rect

contains
!_______________________________________________________________________
!
#include "fml_core/Progress.inc"
!_______________________________________________________________________
!
#include "fml_datafun/fourier_leg.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/Delaunay.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/Delaunay3D.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/Voronoi.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/FunFit.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/GridData.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/GridData3D.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/Interp1.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/Interp2.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/Poly.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/PolyFit.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/PolyVal.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/PPDer.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/PPVal.inc"
!_______________________________________________________________________
!
#include "fml_polyfun/Spline.inc"
!_______________________________________________________________________
!
   function cubic_interp( x, y, xi ) result( yi )

      implicit real(kind=MF_DOUBLE) (t)

      real(kind=MF_DOUBLE), intent(in) :: x(4), y(4), xi
      real(kind=MF_DOUBLE) :: yi
      !------ API end ------

#ifdef _DEVLP
      ! Compute the unique cubic polynomial passing through [x(1:4),y(1:4)]
      ! and return the value at the abscissa xi.

      real(kind=MF_DOUBLE) :: xm, xp, xp2, A, B, C, D, new_x

      xp2 = x(4) - x(2)
      xp  = x(3) - x(2)
      xm  = x(1) - x(2)
      t1 = xm - xp
      t2 = -y(2) + y(1)
      t3 = y(2) - y(3)
      t4 = xp2 ** 2
      t5 = xp2 * t4
      t6 = xp ** 2
      t7 = xm ** 2
      t8 = t2 * t6
      t9 = t3 * t7
      t10 = t3 * xm
      t11 = t2 * xp
      t12 = xm * xp
      t13 = t12 * y(2)
      t14 = t12 * t1
      t15 = (-xp2 + xm + xp) * xp2 - t12
      t16 = -t6 + t7
      t2 = t2 * xp * t6
      t3 = t3 * xm * t7
      t17 = (t1 * xp2 + t6 - t7) * xp2 + t14
      t6 = t7 * t6
      t7 = 1.0d0 / xp2
      t17 = 1.0d0 / t17
      t15 = 1.0d0 / t15
      t18 = -1.0d0 / t1
      t19 = 1.0d0 / xm / xp
      A = -t19 * (t4 * (t10 + t11) + (-t8 - t9) * xp2 + t13 * t1        &
          - t14 * y(4)) * t18 * t15 * t7
      B = -t19 * (t5 * (-t10 - t11) + (t2 + t3) * xp2 + t12 * t16       &
          * y(4) - t13 * t16) * t7 * t17
      C = -t19 * (t4 * (-t2 - t3) + t5 * (t8 + t9) + t6 *               &
          y(2) * t1 - t6 * t1 * y(4)) * t7 * t17
      D = y(2)

      new_x = xi - x(2)

      yi = A*new_x**3 + B*new_x**2 + C*new_x + D

#endif
   end function cubic_interp
!_______________________________________________________________________
!
   subroutine msBuildTriConnect( x, y, tri, tri_connect,                &
                                 check_tri_orient,                      &
                                 tri_renum, equil_face_orient )

      type(mfArray)                           :: x, y
      type(mfArray)                           :: tri ! sometimes 'out'
      type(mfTriConnect)                      :: tri_connect
      logical,           intent(in), optional :: check_tri_orient
      logical,           intent(in), optional :: tri_renum,             &
                                                 equil_face_orient
      !------ API end ------

#ifdef _DEVLP
      !### Don't put intent(out) to tri_connect, we will not be able to
      !    release it!

      real(kind=MF_DOUBLE), pointer :: x_ptr_vec(:), y_ptr_vec(:)
      real(kind=MF_DOUBLE), pointer :: tri_ptr_mat(:,:)
      integer :: nb_pts, nb_tri, min_tri, max_tri
      integer :: status, nb_disconn_tri
      character(len=9) :: str
      logical :: tri_renumbering, equil_face_orientation, nodes_sort,   &
                 check_tri_orientation, tri_modified
      integer :: mf_message_level_save

      character(len=*), parameter :: ROUTINE_NAME = "msBuildTriConnect"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, y, tri )

      if( mfIsEmpty(x) .or. mfIsEmpty(y) .or. mfIsEmpty(tri) ) then
         call PrintMessage( trim(ROUTINE_NAME), "W",                    &
                            "'x', 'y' or 'tri' is empty!" )
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE .or. &
          tri%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "real arrays required!" )
         go to 99
      end if

      if( x%shape(1) /= 1 .and. x%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' must be a vector!" )
         go to 99
      end if

      if( y%shape(1) /= 1 .and. y%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'y' must be a vector!" )
         go to 99
      end if

      if( x%shape(1) /= y%shape(1) .or. x%shape(2) /= y%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' and 'y' must have the same shape!" )
         go to 99
      end if

      ! avoid memory leak next time we call the current routine
      ! using the same object
      call msRelease( tri_connect )

!### TODO 2: j'ai enlevé les trois blocs pour pouvoir pointer sur x, y et
!          tri, même s'ils sont tempo. C'est en interne à Muesli, donc je
!          maîtrise !
!          -> voir s'il y a d'autres routines où je pourrais également
!             faire cela
!!      if( x%status_temporary ) then
!!         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
!!                            "mfArray 'x' cannot be tempo!" )
!!      end if

!!      if( y%status_temporary ) then
!!         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
!!                            "mfArray 'y' cannot be tempo!" )
!!      end if

      mf_message_level_save = mf_message_level
      mf_message_level = 0
      call msPointer( x, x_ptr_vec, no_crc=.true. )
      call msPointer( y, y_ptr_vec, no_crc=.true. )
      mf_message_level = mf_message_level_save

      nb_pts = size(x_ptr_vec)

      if( nb_pts < 3 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' and 'y' must have at least 3 values!" )
         go to 99
      end if

!### TODO 2: idem ci-dessus
!!      if( tri%status_temporary ) then
!!         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
!!                            "mfArray 'tri' cannot be tempo!" )
!!      end if

      call msPointer( tri, tri_ptr_mat, no_crc=.true. )

      nb_tri = size(tri_ptr_mat,1)

      if( size(tri_ptr_mat,2) /= 3 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'tri' must have 3 columns!" )
         go to 99
      end if

      min_tri = minval( tri_ptr_mat(:,:) )
      max_tri = maxval( tri_ptr_mat(:,:) )

      if( min_tri < 1 .or. nb_pts < max_tri ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'tri' seems to be wrong!",         &
                            "(nb of nodes and triangles are not consistent)")
         go to 99
      end if

      allocate( tri_connect%n_xy(nb_pts,2) )

      tri_connect%n_xy(:,1) = x_ptr_vec(:)
      tri_connect%n_xy(:,2) = y_ptr_vec(:)

      allocate( tri_connect%tri_n(nb_tri,3) )

      tri_connect%tri_n(:,:) = tri_ptr_mat(:,:)

      if( present(check_tri_orient) ) then
         check_tri_orientation = check_tri_orient
      else
         check_tri_orientation = .true.
      end if
      if( check_tri_orientation ) then
         call PrintMessage( trim(ROUTINE_NAME), "I",                    &
                            "checking triangles' orientation can be time consuming!", &
                            "-> if you are sure that all triangles have a direct orientation,", &
                            "   you can use the optional argument 'check_tri_orient' with", &
                            "   the FALSE value..." )
      end if

      if( present(tri_renum) ) then
         tri_renumbering = tri_renum
      else
         tri_renumbering = .false.
      end if

      if( present(equil_face_orient) ) then
         equil_face_orientation = equil_face_orient
      else
         equil_face_orientation = .false.
      end if

      call build_tri_conn( tri_connect%n_xy, tri_connect%tri_n,         &
                           tri_connect%face_n, tri_connect%face_tri,    &
                           tri_connect%tri_f, tri_connect%n_tri,        &
                           check_tri_orientation, tri_modified,         &
                           nb_disconn_tri, tri_renumbering,             &
                           equil_face_orientation )

      if( .not. tri%status_temporary ) then
         if( tri_modified .or. tri_renumbering ) then
            tri%double(:,:) = tri_connect%tri_n(:,:)
         end if
      end if

      if( equil_face_orientation ) then
         write(str,"(I0)") nb_disconn_tri
         call PrintMessage( trim(ROUTINE_NAME), "I",                    &
                           "nb disconnected triangles = " // trim(str) )
      end if

      tri_connect%init = .true.
      if( tri_renumbering ) then
         tri_connect%tri_renumbering = .true.
      end if
      if( equil_face_orientation ) then
         tri_connect%face_oriented = .true.
      end if

      call msFreePointer( x, x_ptr_vec )
      call msFreePointer( y, y_ptr_vec )
      call msFreePointer( tri, tri_ptr_mat )

      if( mf_phys_units ) then
         ! verifying the physical dimension

         ! 'x' and 'y' must have the same physical units
         call verif_adim( x%units, y%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'y'",&
                               "are not consistent!" )
            go to 99
         end if

         call verif_adim( tri%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical unit of 'tri'",            &
                               "must be dimensionless!" )
            go to 99
         end if

      end if

 99   continue

      call msFreeArgs( x, y, tri )
      call msAutoRelease( x, y, tri )

#endif
   end subroutine msBuildTriConnect
!_______________________________________________________________________
!
   subroutine msUpdateTriConnect( x, y, tri_connect )

      type(mfArray)                      :: x, y
      type(mfTriConnect), intent(in out) :: tri_connect
      !------ API end ------

#ifdef _DEVLP
      ! Modifies the nodes coordinates in a mfTriConnect structure.
      ! After this update, the knowledge of the convexity is lost.
      ! A simple 'Info' message is emitted.

      integer :: nn, dim

      character(len=*), parameter :: ROUTINE_NAME = "msUpdateTriConnect"

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. tri_connect%init ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "tri_connect is not initialized!" )
         return
      end if

      call msInitArgs( x, y )

      if( mfIsEmpty(x) .or. mfIsEmpty(y) ) then
         call PrintMessage( trim(ROUTINE_NAME), "W",                    &
                            "'x' or 'y' is empty!" )
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "real arrays required for 'x' and 'y'!" )
         go to 99
      end if

      if( x%shape(1) /= 1 .and. x%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' must be a vector!" )
         go to 99
      end if

      if( y%shape(1) /= 1 .and. y%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'y' must be a vector!" )
         go to 99
      end if

      if( x%shape(1) /= y%shape(1) .or. x%shape(2) /= y%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' and 'y' must have the same shape!" )
         go to 99
      end if

      ! Get number of nodes from the mfTriConnect structure.
      nn = size( tri_connect%n_xy, 1 )

      if( x%shape(1) == nn ) then
         dim = 1
      else if( x%shape(2) == nn ) then
         dim = 2
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "non conforming shape for mfArray 'x'!" )
         go to 99
      end if

      if( dim == 1 ) then
         tri_connect%n_xy(:,1) = x%double(:,1)
         tri_connect%n_xy(:,2) = y%double(:,1)
      else ! dim = 2
         tri_connect%n_xy(:,1) = x%double(1,:)
         tri_connect%n_xy(:,2) = y%double(1,:)
      end if

 99   continue

      call msFreeArgs( x, y )
      call msAutoRelease( x, y )

#endif
   end subroutine msUpdateTriConnect
!_______________________________________________________________________
!
   subroutine msExtractTriConnect( tri_connect,                         &
                                   n_xy, tri_n, face_n, face_tri,       &
                                   tri_f, n_tri, convexity,             &
                                   boundary_nodes, boundary_faces )

      type(mfTriConnect)                       :: tri_connect

      real(kind=MF_DOUBLE), allocatable, optional :: n_xy(:,:)
      integer, allocatable,              optional :: tri_n(:,:),        &
                                                     face_n(:,:),       &
                                                     face_tri(:,:),     &
                                                     tri_f(:,:),        &
                                                     n_tri(:)
      integer,                           optional :: convexity
      type(mf_Int_List), allocatable,    optional :: boundary_nodes(:)
      type(mf_Int_List), allocatable,    optional :: boundary_faces(:)
      !------ API end ------

#ifdef _DEVLP
      integer :: nn, nt, nf, nb_boundaries, ib, jf, jj, jjf

      character(len=*), parameter :: ROUTINE_NAME = "msExtractTriConnect"

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. tri_connect%init ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "tri_connect is not initialized!" )
         return
      end if

      if( present(n_xy) ) then
         nn = size(tri_connect%n_xy,1)
         if( allocated(n_xy) ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "n_xy was already allocated!",           &
                               "-> going to overwrite it." )
            if( any(shape(n_xy) /= [nn,2]) ) then
               deallocate( n_xy )
               allocate( n_xy(nn,2) )
            end if
         else
            allocate( n_xy(nn,2) )
         end if
         n_xy(:,:) = tri_connect%n_xy(:,:)
      end if

      if( present(tri_n) ) then
         nt = size(tri_connect%tri_n,1)
         if( allocated(tri_n) ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "tri_n was already allocated!",          &
                               "-> going to overwrite it." )
            if( any(shape(tri_n) /= [nt,3]) ) then
               deallocate( tri_n )
               allocate( tri_n(nt,3) )
            end if
         else
            allocate( tri_n(nt,3) )
         end if
         tri_n(:,:) = tri_connect%tri_n(:,:)
      end if

      if( present(face_n) ) then
         nf = size(tri_connect%face_n,1)
         if( allocated(face_n) ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "face_n was already allocated!",         &
                               "-> going to overwrite it." )
            if( any(shape(face_n) /= [nf,2]) ) then
               deallocate( face_n )
               allocate( face_n(nf,2) )
            end if
         else
            allocate( face_n(nf,2) )
         end if
         face_n(:,:) = tri_connect%face_n(:,:)
      end if

      if( present(face_tri) ) then
         if( .not. present(convexity) ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "when the optional arg 'face_tri' is present, the other arg 'convexity'", &
                               "must be also present!",                 &
                               "(-> it will help you interpreting special negative indices in the second", &
                               "    column of 'face_tri')" )
!### Not coherent: if a Warning is set, face_tri should still be available!
!             if( allocated(face_tri) ) then
!                deallocate( face_tri )
!             end if
!             allocate( face_tri(0,0) )
!             return
         end if
         nf = size(tri_connect%face_tri,1)
         if( allocated(face_tri) ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "face_tri was already allocated!",       &
                               "-> going to overwrite it." )
            if( any(shape(face_tri) /= [nf,2]) ) then
               deallocate( face_tri )
               allocate( face_tri(nf,2) )
            end if
         else
            allocate( face_tri(nf,2) )
         end if
         face_tri(:,:) = tri_connect%face_tri(:,:)
      end if

      if( present(tri_f) ) then
         nt = size(tri_connect%tri_f,1)
         if( allocated(tri_f) ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "tri_f was already allocated!",          &
                               "-> going to overwrite it." )
            if( any(shape(tri_f) /= [nt,3]) ) then
               deallocate( tri_f )
               allocate( tri_f(nt,3) )
            end if
         else
            allocate( tri_f(nt,3) )
         end if
         tri_f(:,:) = tri_connect%tri_f(:,:)
      end if

      if( present(n_tri) ) then
         nn = size(tri_connect%n_tri)
         if( allocated(n_tri) ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "n_tri was already allocated!",          &
                               "-> going to overwrite it." )
            if( size(n_tri) /= nn ) then
               deallocate( n_tri )
               allocate( n_tri(nn) )
            end if
         else
            allocate( n_tri(nn) )
         end if
         n_tri(:) = tri_connect%n_tri(:)
      end if

      if( present(boundary_nodes) .or. present(boundary_faces) ) then
         if( tri_connect%convex_domain == -1 ) then
            ! This call allows the determination of boundary faces
            call msCheckDomainConvexity( tri_connect )
         end if
         if( present(boundary_faces) ) then
            if( allocated(boundary_faces) ) then
               call msRelease( boundary_faces )
            end if
            ! Get number of boundaries
            nb_boundaries = size( tri_connect%faces_boundary_ptr, 1 )
            allocate( boundary_faces(0:nb_boundaries-1) )
            do ib = 0, nb_boundaries-1
               nf = tri_connect%faces_boundary_ptr(ib+1,2) -            &
                    tri_connect%faces_boundary_ptr(ib+1,1) + 1
               allocate( boundary_faces(ib)%list(nf) )
               do jf = 1, nf
                  jj = tri_connect%faces_boundary_ptr(ib+1,1) + jf - 1
                  boundary_faces(ib)%list(jf) = tri_connect%faces_boundary(jj)
               end do
            end do
         end if
         if( present(boundary_nodes) ) then
            if( allocated(boundary_nodes) ) then
               call msRelease( boundary_nodes )
            end if
            ! Get number of boundaries
            nb_boundaries = size( tri_connect%faces_boundary_ptr, 1 )
            allocate( boundary_nodes(0:nb_boundaries-1) )
            do ib = 0, nb_boundaries-1
               nf = tri_connect%faces_boundary_ptr(ib+1,2) -            &
                    tri_connect%faces_boundary_ptr(ib+1,1) + 1
               allocate( boundary_nodes(ib)%list(nf) )
               do jf = 1, nf
                  jj = tri_connect%faces_boundary_ptr(ib+1,1) + jf - 1
                  ! Get the first node of this boundary face
                  jjf = tri_connect%faces_boundary(jj)
                  boundary_nodes(ib)%list(jf) = tri_connect%face_n(jjf,1)
               end do
            end do
         end if
      end if

      if( present(convexity) ) then
         convexity = tri_connect%convex_domain
      end if

#endif
   end subroutine msExtractTriConnect
!_______________________________________________________________________
!
   subroutine build_tri_from_rect( ni, nj, x, y, tc )

      integer,              intent(in) :: ni, nj
      real(kind=MF_DOUBLE), intent(in) :: x(:), y(:)
      type(mfTriConnect)               :: tc
      !------ API end ------

#ifdef _DEVLP
      ! For internal use only: build the connectivity for a simple,
      ! triangular mesh from a structured rectangle mesh.

      ! This routine concerns a structured mesh (i,j).
      !  'ni' is the number of horizontal lines (row lines)
      !  'nj' is the number of vertical lines (column lines)

      !        1            2            3            nj-1          nj
      !        o----- 1 ----o----- 2 ----o     ...     o--- nj-1 ---o
      !        |\           |\           |             |\           |
      !        |  \    2    |  \    4    |             |  \   2*nj  |
      !        |    \       |    \       |             |    \       |
      !        |      \     |      \     |             |      \     |
      !        |   1    \   |   3    \   |             | 2*nj-1 \   |
      !        |          \ |          \ |             |          \ |
      !   nj+1 o---- nj ----o--- nj+1 ---o     ...     o------------o
      !        |\           |\           |             |\           |
      !        |  \  2*nj+2 |  \   ...   |             |  \         |
      !        |    \       |    \       |             |    \       |
      !        |      \     |      \     |             |      \     |
      !        | 2*nj+1 \   |  ...   \   |             |        \   |
      !        |          \ |          \ |             |          \ |
      !        o------------o------------o     ...     o------------o
      !        .            .            .             .            .
      !        .            .            .             .            .
      !        .            .            .             .            .
      !        o------------o------------o     ...     o------------o
      !        |\           |\           |             |\           |
      !        |  \         |  \         |             |  \         |
      !        |    \       |    \       |             |    \       |
      !        |      \     |      \     |             |      \     |
      !        |        \   |        \   |             |        \   |
      !        |          \ |          \ |             |          \ |
      !        o------------o------------o     ...     o------------o ni*nj

      ! 1) in the input vectors 'x' and 'y' (both of size ni*nj),
      !    the nodes must be numbered first by column, then by row.
      ! 2) generated triangles are numbered as shown above.
      ! 3) horizontal faces are numbered first by column, then by row;
      !    vertical faces follow, numbered by column, then by row;
      !    diagonal faces follow, numbered by column, then by row.

      integer :: i, j, n1, n2, n3, n4, f1, f2, f3, f4, f5
      integer :: nn, n_tri, i_tri, nf, i_face, i_node

      integer :: tmp
      logical :: x_axis_same_j, y_axis_same_i, direct_orientation

   !------ end of declarations -- execution starts hereafter  ------

      ! from three nodes in the first square, determine the orientation
      ! of the axis

      x_axis_same_j = x(2) > x(1)
      y_axis_same_i = y(nj+1) < y(1)
      direct_orientation = x_axis_same_j .eqv. y_axis_same_i

      nn = ni*nj
      allocate( tc%n_xy(nn,2) )
      do i = 1, nn
         tc%n_xy(i,:) = [ x(i), y(i) ]
      end do

      ! face_n table: description of faces by nodes
      nf = ni*(nj-1) + nj*(ni-1) + (ni-1)*(nj-1)
      allocate( tc%face_n(nf,2) )
      i_face = 0
      do i = 1, ni
         do j = 1, nj - 1
            n1 = (i-1)*nj + j
            i_face = i_face + 1
            tc%face_n(i_face,1) = n1
            tc%face_n(i_face,2) = n1 + 1
!!print "(A,I0,A,I0,2X,I0)", "face ", i_face, " : ", n1, n1+1
         end do
      end do
      do i = 1, ni - 1
         do j = 1, nj
            n1 = (i-1)*nj + j
            i_face = i_face + 1
            tc%face_n(i_face,1) = n1
            tc%face_n(i_face,2) = n1 + nj
         end do
      end do
      do i = 1, ni - 1
         do j = 1, nj - 1
            n1 = (i-1)*nj + j
            i_face = i_face + 1
            tc%face_n(i_face,1) = n1
            tc%face_n(i_face,2) = n1 + nj + 1
         end do
      end do

      ! tri_n table: description of triangles by nodes
      n_tri = (ni-1)*(nj-1)*2
      allocate( tc%tri_n(n_tri,3) )
      i_tri = 0
      do i = 1, ni - 1
         do j = 1, nj - 1

            ! the four nodes describing the rectangle
            !        n1 o------------o n2
            !           |\           |
            !           |  \         |
            !           |    \       |
            !           |      \     |
            !           |        \   |
            !           |          \ |
            !        n3 o------------o n4

            n1 = (i-1)*nj + j
            n2 = n1 + 1
            n3 = n1 + nj
            n4 = n3 + 1

            i_tri = i_tri + 1
            if( direct_orientation ) then
               tc%tri_n(i_tri,1) = n1
               tc%tri_n(i_tri,2) = n3
               tc%tri_n(i_tri,3) = n4
            else
               tc%tri_n(i_tri,1) = n1
               tc%tri_n(i_tri,2) = n4
               tc%tri_n(i_tri,3) = n3
            end if

            i_tri = i_tri + 1
            if( direct_orientation ) then
               tc%tri_n(i_tri,1) = n1
               tc%tri_n(i_tri,2) = n4
               tc%tri_n(i_tri,3) = n2
            else
               tc%tri_n(i_tri,1) = n1
               tc%tri_n(i_tri,2) = n2
               tc%tri_n(i_tri,3) = n4
            end if

         end do
      end do

      ! n_tri table: one triangle including a given node
      allocate( tc%n_tri(nn) )
      i_node = 0
      i_tri = 1
      do i = 1, ni - 1
         do j = 1, nj - 1
            i_node = i_node + 1
            tc%n_tri(i_node) = i_tri
            i_tri = i_tri + 2
         end do
         ! j = nj
         i_tri = i_tri - 1
         i_node = i_node + 1
         tc%n_tri(i_node) = i_tri
         i_tri = i_tri + 1
      end do
      ! i = ni
      i_tri = i_tri - 2*(nj-1)
      do j = 1, nj - 1
         i_node = i_node + 1
         tc%n_tri(i_node) = i_tri
         i_tri = i_tri + 2
      end do
      i_tri = i_tri - 1
      i_node = i_node + 1
      tc%n_tri(i_node) = i_tri

      ! tri_f table: description of triangles by faces
      allocate( tc%tri_f(n_tri,3) )
      i_tri = 0
      do i = 1, ni - 1
         do j = 1, nj - 1

            ! the five faces in the rectangle
            !           o---- f1 ----o
            !           |\           |
            !           |  \         |
            !        f3 |    \f5     | f4
            !           |      \     |
            !           |        \   |
            !           |          \ |
            !           o---- f2 ----o
            !
            ! Rquires that (see in 'is_in_triangle') that, in the local
            ! numbering of nodes and faces, if the nodes describing the
            ! triangle are n1, n2, n3, then the faces must be stored
            ! n1-n2, n2-n3, n3-n1 (in this order).

            f1 = (i-1)*(nj-1) + j
            f2 = f1 + (nj-1)
            f3 = ni*(nj-1) + (i-1)*nj + j
            f4 = f3 + 1
            f5 = ni*(nj-1) + nj*(ni-1) + (i-1)*(nj-1) + j

            i_tri = i_tri + 1
            if( direct_orientation ) then
               tc%tri_f(i_tri,1) = f3
               tc%tri_f(i_tri,2) = f2
               tc%tri_f(i_tri,3) = f5
            else
               tc%tri_f(i_tri,1) = f5
               tc%tri_f(i_tri,2) = f2
               tc%tri_f(i_tri,3) = f3
            end if

            i_tri = i_tri + 1
            if( direct_orientation ) then
               tc%tri_f(i_tri,1) = f5
               tc%tri_f(i_tri,2) = f4
               tc%tri_f(i_tri,3) = f1
            else
               tc%tri_f(i_tri,1) = f1
               tc%tri_f(i_tri,2) = f4
               tc%tri_f(i_tri,3) = f5
            end if

         end do
      end do

      ! face_tri table: description of faces by triangles
      ! (taking care that a zero index must be located only in the 2nd position)
      allocate( tc%face_tri(nf,2) )
      i_face = 0
      do i = 1, ni
         do j = 1, nj - 1
            ! horizontal faces
            i_face = i_face + 1

            if( i == 1 ) then
               tc%face_tri(i_face,1) = 2*i_face
               tc%face_tri(i_face,2) = 0
            else if( i == ni ) then
               tc%face_tri(i_face,1) = 2*i_face - (2*(nj-1)+1)
               tc%face_tri(i_face,2) = 0
            else
               tc%face_tri(i_face,1) = 2*i_face
               tc%face_tri(i_face,2) = 2*i_face - (2*(nj-1)+1)
            end if
         end do
      end do
      i_tri = 1
      do i = 1, ni - 1
         do j = 1, nj
            ! vertical faces
            i_face = i_face + 1
            if( j == 1 ) then
               tc%face_tri(i_face,1) = i_tri
               tc%face_tri(i_face,2) = 0
            else if( j == nj ) then
               tc%face_tri(i_face,1) = i_tri - 1
               tc%face_tri(i_face,2) = 0
            else
               tc%face_tri(i_face,1) = i_tri
               tc%face_tri(i_face,2) = i_tri - 1
            end if
            if( j /= nj ) then
               i_tri = i_tri + 2
            end if
         end do
      end do
      i_tri = 1
      do i = 1, ni - 1
         do j = 1, nj - 1
            ! diagonal faces
            i_face = i_face + 1
            tc%face_tri(i_face,1) = i_tri
            tc%face_tri(i_face,2) = i_tri + 1
            i_tri = i_tri + 2
         end do
      end do

      tc%init = .true.

#endif
   end subroutine build_tri_from_rect
!_______________________________________________________________________
!
   subroutine msCheckDomainConvexity( tri_connect )

      type(mfTriConnect) :: tri_connect
      !------ API end ------

#ifdef _DEVLP
      ! After the use of this subroutine:
      !  1) the domain convexity is known, i.e. the component
      !     tri_connect%convex_domain is no longer equal to -1 (= UNKNOWN),
      !     but is equal either to 1 (= TRUE) or 0 (= FALSE);
      !  2) second indices of the 'face_tri' table give information about
      !     part of the boundary which are actually holes inside the domain.
      !  3) 'faces_boundary' and 'faces_boundary_ptr', which are components
      !     of then 'mfTriConnect' structure, contain the list of faces
      !     for each boundary. Exterior boundary is #0, while interior
      !     boundaries have positive numbering. Order of the internal
      !     boundaries follows the smallest node value of each internal
      !     boundary.

      character(len=*), parameter :: ROUTINE_NAME = "msCheckDomainConvexity"

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. tri_connect%init ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'tri_connect' doesn't seem to be a valid triangle connectivity!", &
                            "-> did you initialize it using msBuildTriConnect?" )
         return
      end if

      if( tri_connect%convex_domain == -1 ) then
         call check_tri_convexity( tri_connect%convex_domain,           &
                                   tri_connect%n_xy, tri_connect%tri_n, &
                                   tri_connect%face_tri, tri_connect%tri_f, &
                                   tri_connect%n_tri, tri_connect%face_n, &
                                   tri_connect%faces_boundary,          &
                                   tri_connect%faces_boundary_ptr )
      else
         call PrintMessage( trim(ROUTINE_NAME), "I",                    &
                            "the convexity is already known." )
      end if

#endif
   end subroutine msCheckDomainConvexity
!_______________________________________________________________________
!
   function mfTriSearch_mfarray( tri_connect, x, y, strict )            &
   result( num )

      type(mfTriConnect) :: tri_connect
      type(mfArray) :: x, y
      logical, intent(in), optional :: strict
      integer :: num
      !------ API end ------

#ifdef _DEVLP
      ! Search for a triangle enclosing the point (x,y);
      ! Returns the triangle number, or a negative value if not found.

      real(kind=MF_DOUBLE) :: xi, yi
      integer :: status

      character(len=*), parameter :: ROUTINE_NAME = "mfTriSearch"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, y )

      if( x%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x' must be a scalar mfArray, of type real!" )
         go to 99
      end if

      if( y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'y' must be a scalar mfArray, of type real!" )
         go to 99
      end if

      xi = mfDble( mfGet(x,1) )
      yi = mfDble( mfGet(y,1) )

      num = mfTriSearch_double( tri_connect, xi, yi, strict )

      if( mf_phys_units ) then
         ! verifying the physical dimension

         ! 'x' and 'y' must have the same physical units
         call verif_adim( x%units, y%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'y'",&
                               "are not consistent!" )
            go to 99
         end if

      end if

 99   continue

      call msFreeArgs( x, y )

      call msAutoRelease( x, y )

#endif
   end function mfTriSearch_mfarray
!_______________________________________________________________________
!
   function mfTriSearch_double( tri_connect, xi, yi, strict )           &
   result( num )

      type(mfTriConnect) :: tri_connect
      real(kind=MF_DOUBLE) :: xi, yi
      logical, intent(in), optional :: strict
      integer :: num
      !------ API end ------

#ifdef _DEVLP
      ! Search for a triangle enclosing the point (xi,yi);
      ! Returns the triangle number, or a negative value if not found.

      real(kind=MF_DOUBLE) :: dmin, dist, x_mid, y_mid
      integer :: i, k, hole, f, fmin
      logical :: inside_only

      character(len=*), parameter :: ROUTINE_NAME = "mfTriSearch"

   !------ end of declarations -- execution starts hereafter  ------

      num = 0

      if( present(strict) ) then
         inside_only = strict
      else
         inside_only = .true.
      end if

      if( .not. tri_connect%init ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "arg 'tri_connect' is not initialized!",    &
                            "(you must use 'msBuildTriConnect' routine before)" )
         return
      end if

      num = tsearch( tri_connect%convex_domain, tri_connect%n_xy,       &
                     tri_connect%tri_n, tri_connect%face_tri,           &
                     tri_connect%tri_f, tri_connect%n_tri,              &
                     tri_connect%face_n, tri_connect%faces_boundary,    &
                     tri_connect%faces_boundary_ptr,                    &
                     xi, yi )

      if( num <= 0 ) then

         if( inside_only ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "your point is outside the meshed domain!" )
            return
         else
            ! The targeted point is outside the meshed domain
            hole = 1 - num
            dmin = MF_REALMAX
            do i = tri_connect%faces_boundary_ptr(hole,1),              &
                   tri_connect%faces_boundary_ptr(hole,2)
               f = tri_connect%faces_boundary(i)
               x_mid = ( tri_connect%n_xy(tri_connect%face_n(f,1),1)    &
                       + tri_connect%n_xy(tri_connect%face_n(f,2),1) ) / 2.0d0
               y_mid = ( tri_connect%n_xy(tri_connect%face_n(f,1),2)    &
                       + tri_connect%n_xy(tri_connect%face_n(f,2),2) ) / 2.0d0
               dist = (xi-x_mid)**2 + (yi-y_mid)**2
               if( dist < dmin ) then
                  dmin = dist
                  fmin = f
               end if
            end do
            ! Retrieve the triangle number
            num = tri_connect%face_tri(fmin,1) ! first element is never zero
         end if

      end if

#endif
   end function mfTriSearch_double
!_______________________________________________________________________
!
   function mfNodeSearch_mfarray( tri_connect, x, y, strict )           &
   result( num )

      type(mfTriConnect) :: tri_connect
      type(mfArray) :: x, y
      logical, intent(in), optional :: strict
      integer :: num
      !------ API end ------

#ifdef _DEVLP
      ! returns a triangle number, or 0 (when the point is outside the
      ! convex hull)

      real(kind=MF_DOUBLE) :: xi, yi
      integer :: status

      character(len=*), parameter :: ROUTINE_NAME = "mfNodeSearch"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, y )

      if( x%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x' must be a scalar mfArray, of type real!" )
         go to 99
      end if

      if( y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'y' must be a scalar mfArray, of type real!" )
         go to 99
      end if

      xi = mfDble( mfGet(x,1) )
      yi = mfDble( mfGet(y,1) )

      num = mfNodeSearch_double( tri_connect, xi, yi, strict )

      if( mf_phys_units ) then
         ! verifying the physical dimension

         ! 'x' and 'y' must have the same physical units
         call verif_adim( x%units, y%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'y'",&
                               "are not consistent!" )
            go to 99
         end if

      end if

 99   continue

      call msFreeArgs( x, y )
      call msAutoRelease( x, y )

#endif
   end function mfNodeSearch_mfarray
!_______________________________________________________________________
!
   function mfNodeSearch_double( tri_connect, xi, yi, strict )          &
   result( num )

      type(mfTriConnect) :: tri_connect
      real(kind=MF_DOUBLE) :: xi, yi
      logical, intent(in), optional :: strict
      integer :: num
      !------ API end ------

#ifdef _DEVLP
      ! returns a triangle number, or 0 (when the point is outside the
      ! convex hull)

      real(kind=MF_DOUBLE) :: dmin, dist
      integer :: i, k, hole, f
      logical :: inside_only

      character(len=*), parameter :: ROUTINE_NAME = "mfNodeSearch"

   !------ end of declarations -- execution starts hereafter  ------

      num = 0

      if( present(strict) ) then
         inside_only = strict
      else
         inside_only = .true.
      end if

      if( .not. tri_connect%init ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "arg 'tri_connect' is not initialized!",    &
                            "(you must use 'msBuildTriConnect' routine before)" )
         return
      end if

      k = tsearch( tri_connect%convex_domain, tri_connect%n_xy,         &
                   tri_connect%tri_n, tri_connect%face_tri,             &
                   tri_connect%tri_f, tri_connect%n_tri,                &
                   tri_connect%face_n, tri_connect%faces_boundary,      &
                   tri_connect%faces_boundary_ptr,                      &
                   xi, yi )

      if( k <= 0 ) then

         if( inside_only ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "your point is outside the meshed domain!" )
            num = k
            return
         else
            ! The targeted point is outside the meshed domain
            hole = 1 - k
            dmin = MF_REALMAX
            do i = tri_connect%faces_boundary_ptr(hole,1),              &
                   tri_connect%faces_boundary_ptr(hole,2)
               f = tri_connect%faces_boundary(i)
               dist = (xi-tri_connect%n_xy(tri_connect%face_n(f,1),1))**2 &
                    + (yi-tri_connect%n_xy(tri_connect%face_n(f,1),2))**2
               if( dist < dmin ) then
                  dmin = dist
                  num = tri_connect%face_n(f,1)
               end if
               dist = (xi-tri_connect%n_xy(tri_connect%face_n(f,2),1))**2 &
                    + (yi-tri_connect%n_xy(tri_connect%face_n(f,2),2))**2
               if( dist < dmin ) then
                  dmin = dist
                  num = tri_connect%face_n(f,2)
               end if
            end do
         end if

      else

         ! among the three vertices of the enclosing triangle, find
         ! the nearest vertex
         dmin = MF_REALMAX
         do i = 1, 3
            dist = (xi-tri_connect%n_xy(tri_connect%tri_n(k,i),1))**2   &
                 + (yi-tri_connect%n_xy(tri_connect%tri_n(k,i),2))**2
            if( dist < dmin ) then
               dmin = dist
               num = tri_connect%tri_n(k,i)
            end if
         end do

      end if

#endif
   end function mfNodeSearch_double
!_______________________________________________________________________
!
   subroutine msPrintTriConnect( tri_connect, short_info )

      type(mfTriConnect) :: tri_connect
      logical, optional :: short_info
      !------ API end ------

#ifdef _DEVLP
      ! This routine is designed to check easily a triangles connectivity.
      ! It should be ok only for a moderate number of nodes, since the
      ! print format is everywhere 'I3' for all integers!
      !
      ! The optional argument 'short_info' allows the user to just print
      ! the numbers of different items: nodes, triangles and faces.
      ! It allows moreover to get infos for big structures.

      integer :: i, n, nn, nt, nf
      logical :: short_output

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. tri_connect%init ) then
         print *, "(Muesli:) msPrintTriConnect: argument is not initialized!"
         return
      end if

      nn = size(tri_connect%n_xy,1)
      nt = size(tri_connect%tri_n,1)
      nf = size(tri_connect%face_n,1)

      if( present(short_info) ) then
         short_output = short_info
      else
         short_output = .false.
      end if

      n = max( nn, nt, nf )
      if( n > 999 .and. .not. short_output ) then
         print *, "(Muesli:) msPrintTriConnect: too big mesh!"
         print *, "   -> you can only get the numbers of items by adding the"
         print *, "      optional argument 'short_info'."
         print *, "   (or recompile the Muesli library using an integer format"
         print *, "    greater than 'I3')"
         return
      end if

      print "(/,A)", "         ┏━━━━━━━━━━━━━━━━━━━━━━━━┓"
      print "(A)",   "         ┃ Triangles Connectivity ┃"
      print "(A,/)", "         ┗━━━━━━━━━━━━━━━━━━━━━━━━┛"
      print "(A,I0)","    nn: number of nodes     = ", nn
      print "(A,I0)","    nt: number of triangles = ", nt
      print "(A,I0)","    nf: number of faces     = ", nf
      print *
      select case( tri_connect%convex_domain )
         case( -1 )
      print "(A,I0)","    domain convexity: unknown"
         case(  0 )
      print "(A,I0)","    domain convexity: false"
         case(  1 )
      print "(A,I0)","    domain convexity: true"
      end select

      if( short_output ) then
         print *
         return
      end if

!!      print "(A,L0)","        face_oriented = ", tri_connect%face_oriented
      print "(/,A)", "  ┌──────────────────────────────────────┐"
      print "(A)",   "  │ n_xy(nn,2):  nodes by 2D-coordinates │"
      print "(A)",   "  ├──────────────────────────────────────┤"
      print "(A)",   "  │ node               coordinates       │"
      do i = 1, nn
         print "(2X,'│',1X,I3,7X,2(2X,ES11.4),' │')", i, tri_connect%n_xy(i,:)
      end do
      print "(A)",   "  └──────────────────────────────────────┘"
      print "(/,A)", "  ┌───────────────────────────────────┐"
      print "(A)",   "  │ tri_n(nt,3):  triangles by nodes  │"
      print "(A)",   "  │              (direct orientation) │"
      print "(A)",   "  ├───────────────────────────────────┤"
      print "(A)",   "  │ triangle            nodes         │"
      do i = 1, nt
         print "(2X,'│',1X,I3,10X,3(2X,I3),6X,'│')", i, tri_connect%tri_n(i,:)
      end do
      print "(A)",   "  └───────────────────────────────────┘"
      print "(/,A)", "  ┌───────────────────────────────┐"
      print "(A)",   "  │ face_n(nf,2):  faces by nodes │"
      print "(A)",   "  ├───────────────────────────────┤"
      print "(A)",   "  │ face            nodes         │"
      do i = 1, nf
         print "(2X,'│',1X,I3,9X,2(2X,I3),8X,'│')", i, tri_connect%face_n(i,:)
      end do
      print "(A)",   "  └───────────────────────────────┘"
      print "(/,A)", "  ┌─────────────────────────┐"
      print "(A)",   "  │ face_tri(nf,2):  face   │"
      print "(A)",   "  │   common to 2 triangles │"
      print "(A)",   "  ├─────────────────────────┤"
      print "(A)",   "  │ face        triangles   │"
      do i = 1, n
         print "(2X,'│',1X,I3,7X,2(2X,I3),4X,'│')", i, tri_connect%face_tri(i,:)
      end do
      print "(A)",   "  └─────────────────────────┘"
      print "(/,A)", "  ┌───────────────────────────────────┐"
      print "(A)",   "  │ tri_f(nt,3):  triangles by faces  │"
      print "(A)",   "  │              (direct orientation) │"
      print "(A)",   "  ├───────────────────────────────────┤"
      print "(A)",   "  │ triangle          faces           │"
      n = size(tri_connect%tri_f,1)
      do i = 1, n
         print "(2X,'│',1X,I3,8X,3(2X,I3),8X,'│')", i, tri_connect%tri_f(i,:)
      end do
      print "(A)",   "  └───────────────────────────────────┘"
      print "(/,A)", "  ┌───────────────────────────────┐"
      print "(A)",   "  │ n_tri(nn): one triangle which │"
      print "(A)",   "  │         contains a given node │"
      print "(A)",   "  ├───────────────────────────────┤"
      print "(A)",   "  │ node            triangle      │"
      n = size(tri_connect%n_tri)
      do i = 1, n
         print "(2X,'│',1X,I3,15X,I3,9X,'│')", i, tri_connect%n_tri(i)
      end do
      print "(A)",   "  └───────────────────────────────┘"

#endif
   end subroutine msPrintTriConnect
!_______________________________________________________________________
!
   subroutine msCheckPLDomainValidity( PL_domain, status )

      type(mfPLdomain), intent(in)  :: PL_domain
      integer,          intent(out) :: status

      ! Returns status=1 for a valid PSLG structure (0 on the contrary)

      integer :: nn, ne, nh, i, start, nb_contours
      character(len=3) :: curr_contour_str ! no more 999 contours!

      character(len=*), parameter :: ROUTINE_NAME = "msCheckPLDomainValidity"

   !------ end of declarations -- execution starts hereafter  ------

      ! Default: invalid until proven valid
      status = 0

      ! Check nodes array
      if( .not. allocated(PL_domain%n_xy) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "PL_domain%n_xy is not allocated." )
         return
      else
         if( size(PL_domain%n_xy,2) /= 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "PL_domain%n_xy must have exactly 2 columns." )
            return
         end if
         nn = size(PL_domain%n_xy,1)
         if( nn < 3 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "Insufficient number of lines in PL_domain%n_xy.", &
                               "(the minimum PL domain is a triangle)" )
            return
         end if
      end if

      ! Check edges array
      if( .not. allocated(PL_domain%edge_n) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "PL_domain%edge_n is not allocated." )
         return
      else
         if( size(PL_domain%edge_n,2) /= 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "PL_domain%edge_n must have exactly 2 columns." )
            return
         end if
         ne = size(PL_domain%edge_n,1)
         if( ne < 3 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "Insufficient number of lines in PL_domain%edge_n.", &
                               "(the minimum PL domain is a triangle)" )
            return
         end if
      end if

      ! Check that indices are valid
      do i = 1, ne
         if( PL_domain%edge_n(i,1) < 1 .or.                             &
             PL_domain%edge_n(i,1) > nn ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "at least on index in PL_domain%edge_n is out-of-range.", &
                               "(in the first column)" )
            return
         end if
         if( PL_domain%edge_n(i,2) < 1 .or.                             &
             PL_domain%edge_n(i,2) > nn ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "at least on index in PL_domain%edge_n is out-of-range.", &
                               "(in the second column)" )
            return
         end if
         if( PL_domain%edge_n(i,1) == PL_domain%edge_n(i,2) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "found a zero-length edge in PL_domain%edge_n." )
            return ! no zero-length edge
         end if
      end do

      ! Check consecutive ordering of edges
      ! Meaning: edge(i,2) == edge(i+1,1) for all i
      ! If forming a closed polygon: last node connects to first

      ! However, many closed contours may be present in the array...

      i = 1
      nb_contours = 0
      do while( i <= ne )
         start = i

         ! 1) go inside a cycle
         do while( i < ne )
            if( PL_domain%edge_n(i,2) == PL_domain%edge_n(i+1,1) ) then
               i = i + 1
            else
               exit
            end if
         end do

         ! 2) the cycle must be closed
         if( PL_domain%edge_n(i,2) /= PL_domain%edge_n(start,1) ) then
            write( curr_contour_str, "(I0)" ) nb_contours + 1
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "one contour is not closed in PL_domain%edge_n.", &
                               "(this is the contour #" //              &
                               trim(curr_contour_str) // ")" )
            return
         end if

         nb_contours = nb_contours + 1
         i = i + 1   ! next cycle
      end do

      ! Check holes (simple check)
      if( allocated(PL_domain%holes_xy) ) then
         if( size(PL_domain%holes_xy,2) /= 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "PL_domain%holes_xy must have exactly 2 columns." )
            return
         end if
         ! The number of holes is not verified.
      end if

      ! Everything OK
      status = 1

   end subroutine msCheckPLDomainValidity
!_______________________________________________________________________
!
   subroutine msPrintPLdomain( PL_domain, short_info )

      type(mfPLdomain) :: PL_domain
      logical, optional :: short_info
      !------ API end ------

#ifdef _DEVLP
      ! This routine is designed to check easily a PL domain definition.
      ! It should be ok only for a moderate number of nodes, since the
      ! print format is everywhere 'I3' for all integers!
      !
      ! The optional argument 'short_info' allows the user to just print
      ! the numbers of different items: nodes, triangles and faces.
      ! It allows then to get infos for big structures.

      integer :: i, n, nn, ne, nh
      logical :: short_output

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. allocated(PL_domain%n_xy) ) then
         nn = -1
      else
         nn = size(PL_domain%n_xy,1)
      end if

      if( .not. allocated(PL_domain%edge_n) ) then
         ne = -1
      else
         ne = size(PL_domain%edge_n,1)
      end if

      if( .not. allocated(PL_domain%holes_xy) ) then
         nh = -1
      else
         nh = size(PL_domain%holes_xy,1)
      end if

      if( present(short_info) ) then
         short_output = short_info
      else
         short_output = .false.
      end if

      n = max( nn, ne, nh )
      if( n > 999 .and. .not. short_output ) then
         print *, "(Muesli:) msPrintPLdomain: too big mesh!"
         print *, "   -> you can only get the numbers of items by adding the"
         print *, "      optional argument 'short_info'."
         print *, "   (or recompile the Muesli library using an integer format"
         print *, "    greater than 'I3')"
         return
      end if

      print "(/,A)", "         ┏━━━━━━━━━━━━━━━━━━━━━━━━━┓"
      print "(A)",   "         ┃ Piecewise Linear domain ┃"
      print "(A,/)", "         ┗━━━━━━━━━━━━━━━━━━━━━━━━━┛"
      if( n <= 0 ) then
         print "(A)", " Empty object."
         return
      end if
      ! Use 'max' to avoid printing -1 when arrays are not allocated.
      print "(A,I0)","    nn: number of nodes = ", max(nn,0)
      print "(A,I0)","    ne: number of edges = ", max(ne,0)
      print "(A,I0)","    nh: number of holes = ", max(nh,0)

      if( short_output ) then
         print *
         return
      end if

      print "(/,A)", "  ┌──────────────────────────────────────┐"
      print "(A)",   "  │ n_xy(nn,2):  nodes by 2D-coordinates │"
      print "(A)",   "  ├──────────────────────────────────────┤"
      print "(A)",   "  │ node               coordinates       │"
      do i = 1, nn
         print "(2X,'│',1X,I3,7X,2(2X,ES11.4),' │')", i, PL_domain%n_xy(i,:)
      end do
      print "(A)",   "  └──────────────────────────────────────┘"

      print "(/,A)", "  ┌───────────────────────────────┐"
      print "(A)",   "  │ edge_n(ne,2):  edges by nodes │"
      print "(A)",   "  ├───────────────────────────────┤"
      print "(A)",   "  │ edge            nodes         │"
      do i = 1, ne
         print "(2X,'│',1X,I3,9X,2(2X,I3),8X,'│')", i, PL_domain%edge_n(i,:)
      end do
      print "(A)",   "  └───────────────────────────────┘"

      print "(/,A)", "  ┌──────────────────────────────────────────┐"
      print "(A)",   "  │ holes_xy(nn,2):  holes by 2D-coordinates │"
      print "(A)",   "  ├──────────────────────────────────────────┤"
      print "(A)",   "  │ node                 coordinates         │"
      do i = 1, nh
         print "(2X,'│',1X,I3,9X,2(2X,ES11.4),'   │')", i, PL_domain%holes_xy(i,:)
      end do
      print "(A)",   "  └──────────────────────────────────────────┘"

#endif
   end subroutine msPrintPLdomain
!_______________________________________________________________________
!
   subroutine msTriNodeNeighbors( tri_connect,                          &
                                  connected_nodes, connected_faces,     &
                                  oriented )

      type(mfTriConnect), intent(in)           :: tri_connect
      type(mf_Int_List), allocatable, optional :: connected_nodes(:)
      type(mf_Int_List), allocatable, optional :: connected_faces(:)
      ! The following argument is not documented in the official doc.
      logical,            intent(in), optional :: oriented
      !------ API end ------

#ifdef _DEVLP
      integer :: nn, nt
      character(len=*), parameter :: ROUTINE_NAME = "msTriNodeNeighbors"
      logical :: oriented_0

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. tri_connect%init ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "arg 'tri_connect' is not initialized!",    &
                            "(you must use 'msBuildTriConnect' routine before)" )
         return
      end if

      if( present(oriented) ) then
         oriented_0 = oriented
      else
         oriented_0 = .false.
      end if

      if( oriented_0 ) then
         if( tri_connect%face_oriented ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                              "'tri_connect' cannot be used in this routine when faces are oriented!", &
                              "-> recall msBuildTriConnect with 'equil_face_orient'=.false." )
            return
         end if
      end if

      nn = size(tri_connect%n_tri,1)
      nt = size(tri_connect%tri_n,1)

      if( present(connected_nodes) ) then
         if( present(connected_faces) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                              "'connected_nodes' and 'connected_faces'", &
                              "cannot be both present!")
            return
         end if
         if( allocated(connected_nodes) ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                              "'connected_nodes' should be not allocated!", &
                              "(going to deallocate it before using it)")
            deallocate( connected_nodes )
         end if
         allocate( connected_nodes(nn) )
         call build_connected_nodes( tri_connect%face_n, tri_connect%face_tri, &
                                     tri_connect%tri_f, tri_connect%n_tri, &
                                     tri_connect%tri_n,                 &
                                     connected_nodes, oriented_0 )
      else ! .not. present(connected_nodes)
         if( .not. present(connected_faces) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                              "among 'connected_nodes' and 'connected_faces',", &
                              "at least one must be present!")
            return
         end if
         if( allocated(connected_faces) ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                              "'connected_faces' should be not allocated!", &
                              "(going to deallocate it before using it)")
            deallocate( connected_faces )
         end if
         allocate( connected_faces(nn) )
         call build_connected_faces( tri_connect%face_n, tri_connect%face_tri, &
                                     tri_connect%tri_f, tri_connect%n_tri, &
                                     tri_connect%tri_n,                 &
                                     connected_faces, oriented_0 )
      end if

#endif
   end subroutine msTriNodeNeighbors
!_______________________________________________________________________
!
   subroutine msRelease_mfTriConnect( tri_connect )

      type(mfTriConnect) :: tri_connect
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( tri_connect%init ) then

         deallocate( tri_connect%n_xy )

         deallocate( tri_connect%tri_n )

         deallocate( tri_connect%face_n )

         deallocate( tri_connect%face_tri )

         deallocate( tri_connect%tri_f )

         deallocate( tri_connect%n_tri )

         tri_connect%init            = .false.
         tri_connect%tri_renumbering = .false.
         tri_connect%face_oriented   = .false.

         if( tri_connect%convex_domain /= -1 ) then

            deallocate(tri_connect%faces_boundary)

            deallocate(tri_connect%faces_boundary_ptr)

            tri_connect%convex_domain = -1

         end if

      end if

#endif
   end subroutine msRelease_mfTriConnect
!_______________________________________________________________________
!
   subroutine msRelease_mfPLdomain( x )

      type(mfPLdomain) :: x
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( allocated(x%n_xy) ) then
         deallocate( x%n_xy )
      end if
      if( allocated(x%edge_n) ) then
         deallocate( x%edge_n )
      end if
      if( allocated(x%holes_xy) ) then
         deallocate( x%holes_xy )
      end if

#endif
   end subroutine msRelease_mfPLdomain
!_______________________________________________________________________
!
   subroutine msPrintPLdomain3D( PL_domain3D, short_info )

      type(mfPLdomain3D) :: PL_domain3D
      logical, optional :: short_info
      !------ API end ------

#ifdef _DEVLP
      ! This routine is designed to check easily a 3D PL domain definition.
      ! It should be ok only for a moderate number of nodes, since the
      ! print format is everywhere 'I3' for all integers!
      !
      ! The optional argument 'short_info' allows the user to just print
      ! the numbers of different items: nodes, triangles and faces.
      ! It allows then to get infos for big structures.

      integer :: i, n, nn, ne, nf, nh
      logical :: short_output

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. allocated(PL_domain3D%n_xyz) ) then
         nn = -1
      else
         nn = size(PL_domain3D%n_xyz,1)
      end if

      if( .not. allocated(PL_domain3D%edge_n) ) then
         ne = -1
      else
         ne = size(PL_domain3D%edge_n,1)
      end if

      if( .not. allocated(PL_domain3D%face_e) ) then
         nf = -1
      else
         nf = size(PL_domain3D%face_e,1)
      end if

      if( .not. allocated(PL_domain3D%holes_xyz) ) then
         nh = -1
      else
         nh = size(PL_domain3D%holes_xyz,1)
      end if

      if( present(short_info) ) then
         short_output = short_info
      else
         short_output = .false.
      end if

      n = max( nn, ne, nh )
      if( n > 999 .and. .not. short_output ) then
         print *, "(Muesli:) msPrintPLdomain3D: too big mesh!"
         print *, "   -> you can only get the numbers of items by adding the"
         print *, "      optional argument 'short_info'."
         print *, "   (or recompile the Muesli library using an integer format"
         print *, "    greater than 'I3')"
         return
      end if

      print "(/,A)", "         ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓"
      print "(A)",   "         ┃ 3D Piecewise Linear domain ┃"
      print "(A,/)", "         ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛"
      if( n <= 0 ) then
         print "(A)", " Empty object."
         return
      end if
      ! Use 'max' to avoid printing -1 when arrays are not allocated.
      print "(A,I0)","    nn: number of nodes = ", max(nn,0)
      print "(A,I0)","    ne: number of edges = ", max(ne,0)
      print "(A,I0)","    nf: number of faces = ", max(nf,0)
      print "(A,I0)","    nh: number of holes = ", max(nh,0)

      if( short_output ) then
         print *
         return
      end if

      print "(/,A)", "  ┌───────────────────────────────────────────────────┐"
      print "(A)",   "  │ n_xyz(nn,2):  nodes by 3D-coordinates             │"
      print "(A)",   "  ├───────────────────────────────────────────────────┤"
      print "(A)",   "  │ node                     coordinates              │"
      do i = 1, nn
         print "(2X,'│',1X,I3,7X,3(2X,ES11.4),' │')", i, PL_domain3D%n_xyz(i,:)
      end do
      print "(A)",   "  └───────────────────────────────────────────────────┘"

      print "(/,A)", "  ┌───────────────────────────────┐"
      print "(A)",   "  │ edge_n(ne,2):  edges by nodes │"
      print "(A)",   "  ├───────────────────────────────┤"
      print "(A)",   "  │ edge            nodes         │"
      do i = 1, ne
         print "(2X,'│',1X,I3,9X,2(2X,I3),8X,'│')", i, PL_domain3D%edge_n(i,:)
      end do
      print "(A)",   "  └───────────────────────────────┘"

      print "(/,A)", "  ┌───────────────────────────────┐"
      print "(A)",   "  │ face_e(nf):  faces by edges   │"
      print "(A)",   "  ├───────────────────────────────┤"
      print "(A)",   "  │ face            edges         │"
      do i = 1, nf
         print "(2X,'│',1X,I3,4X,4(2X,I3),3X,'│')", i, PL_domain3D%face_e(i)%list(:)
      end do
      print "(A)",   "  └───────────────────────────────┘"

      print "(/,A)", "  ┌───────────────────────────────────────────────────────┐"
      print "(A)",   "  │ holes_xyz(nn,3):  holes by 3D-coordinates             │"
      print "(A)",   "  ├───────────────────────────────────────────────────────┤"
      print "(A)",   "  │ node                       coordinates                │"
      do i = 1, nh
         print "(2X,'│',1X,I3,9X,3(2X,ES11.4),'   │')", i, PL_domain3D%holes_xyz(i,:)
      end do
      print "(A)",   "  └───────────────────────────────────────────────────────┘"

#endif
   end subroutine msPrintPLdomain3D
!_______________________________________________________________________
!
   subroutine msCheckPLDomain3DValidity( PL_domain_3D, status )

      type(mfPLdomain3D), target, intent(in)  :: PL_domain_3D
      integer,                    intent(out) :: status

      ! Returns status=1 for a valid PSLG structure (0 on the contrary)

      integer :: nn, ne, nf, nh, i, j, start, nb_contours
      integer :: n_strt, n_last, n_seg, n1, n2
      integer, pointer :: lst(:), edg(:,:)

      character(len=3) :: i_str ! no more 999 contours!

      character(len=*), parameter :: ROUTINE_NAME = "msCheckPLDomain3DValidity"

   !------ end of declarations -- execution starts hereafter  ------

      ! Default: invalid until proven valid
      status = 0

      ! Check nodes array
      if( .not. allocated(PL_domain_3D%n_xyz) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "PL_domain_3D%n_xyz is not allocated." )
         return
      else
         if( size(PL_domain_3D%n_xyz,2) /= 3 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "PL_domain_3D%n_xyz must have exactly 3 columns." )
            return
         end if
         nn = size(PL_domain_3D%n_xyz,1)
         if( nn < 4 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "Insufficient number of lines in PL_domain_3D%n_xyz.", &
                               "(the minimum 3D PL domain is a tetrahedron)" )
            return
         end if
      end if

      ! Check edges array
      if( .not. allocated(PL_domain_3D%edge_n) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "PL_domain_3D%edge_n is not allocated." )
         return
      else
         if( size(PL_domain_3D%edge_n,2) /= 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "PL_domain_3D%edge_n must have exactly 2 columns." )
            return
         end if
         ne = size(PL_domain_3D%edge_n,1)
         if( ne < 6 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "Insufficient number of lines in PL_domain_3D%edge_n.", &
                               "(the minimum 3D PL domain is a tetrahedron)" )
            return
         end if
      end if

      ! Check that indices are valid
      do i = 1, ne
         if( PL_domain_3D%edge_n(i,1) < 1 .or.                          &
             PL_domain_3D%edge_n(i,1) > nn ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "at least on index in PL_domain_3D%edge_n is out-of-range.", &
                               "(in the first column)" )
            return
         end if
         if( PL_domain_3D%edge_n(i,2) < 1 .or.                          &
             PL_domain_3D%edge_n(i,2) > nn ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "at least on index in PL_domain_3D%edge_n is out-of-range.", &
                               "(in the second column)" )
            return
         end if
         if( PL_domain_3D%edge_n(i,1) == PL_domain_3D%edge_n(i,2) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "found a zero-length edge in PL_domain_3D%edge_n." )
            return ! no zero-length edge
         end if
      end do

      ! Check faces array
      if( .not. allocated(PL_domain_3D%face_e) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "PL_domain_3D%face_e is not allocated." )
         return
      else
         nf = size(PL_domain_3D%face_e)
         if( nf < 4 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "Insufficient number of lines in PL_domain_3D%face_e.", &
                               "(the minimum 3D PL domain is a tetrahedron)" )
            return
         end if
      end if

      ! Check each face
      edg => PL_domain_3D%edge_n
      do i = 1, nf
         lst => PL_domain_3D%face_e(i)%list
         n_seg = size(lst)
         ! Let's suppose that the first edge doesn't need to be swapped...
         n_strt = edg( lst(1), 1 )
         n_last = edg( lst(1), 2 )
         n1 = edg( lst(2), 1 ) ; n2 = edg( lst(2), 2 )
         if( n_last /= n1 .and. n_last /= n2 ) then
            ! First edge must be swapped
            n_strt = edg( lst(1), 2 )
            n_last = edg( lst(1), 1 )
         endif
         do j = 2, n_seg ! loop over the remaining edges
            n1 = edg( lst(j), 1 ) ; n2 = edg( lst(j), 2 )
            if( n_last == n1 ) then
               n_last = n2
            else if( n_last == n2 ) then
               n_last = n1
            else
               write( i_str, "(I0)" ) i
               call PrintMessage( trim(ROUTINE_NAME), "E",              &
                                  "face #" // trim(i_str) // " is not valid!" )
               return
            end if
         end do
         if( n_last /= n_strt ) then
            write( i_str, "(I0)" ) i
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "face #" // trim(i_str) // " is not valid!" )
            return
         end if
      end do

      ! Check holes (simple check)
      if( allocated(PL_domain_3D%holes_xyz) ) then
         if( size(PL_domain_3D%holes_xyz,2) /= 3 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "PL_domain_3D%holes_xyz must have exactly 3 columns." )
            return
         end if
         ! The number of holes is not verified.
      end if

      ! Everything OK
      status = 1

   end subroutine msCheckPLDomain3DValidity
!_______________________________________________________________________
!
   subroutine msRelease_mfPLdomain3D( x )

      type(mfPLdomain3D) :: x
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( allocated(x%n_xyz) ) then
         deallocate( x%n_xyz )
      end if
      if( allocated(x%edge_n) ) then
         deallocate( x%edge_n )
      end if
      if( allocated(x%face_e) ) then
         call msRelease( x%face_e )
         deallocate( x%face_e )
      end if
      if( allocated(x%holes_xyz) ) then
         deallocate( x%holes_xyz )
      end if

#endif
   end subroutine msRelease_mfPLdomain3D
!_______________________________________________________________________
!
   subroutine msBuildTetraConnect_vec( x, y, z, tetra, tetra_connect )

      type(mfArray) :: x, y, z, tetra
      type(mfTetraConnect) :: tetra_connect
      !------ API end ------

#ifdef _DEVLP
      real(kind=MF_DOUBLE), pointer :: x_ptr_vec(:), y_ptr_vec(:), z_ptr_vec(:)
      real(kind=MF_DOUBLE), pointer :: tetra_ptr_mat(:,:)
      integer :: nb_pts, nb_tetra, min_tetra, max_tetra
      integer :: status

      character(len=*), parameter :: ROUTINE_NAME = "msBuildTetraConnect"

   !------ end of declarations -- execution starts hereafter  ------

      if( delaunay3d_addr == 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "Delaunay 3D internal tetrahedrization not reachable!", &
                            "(forget to call mfDelaunay3D?",            &
                            " msEndDelaunay3D called too early?)" )
         return
      end if

      call msInitArgs( x, y, z, tetra )

      if( mfIsEmpty(x) .or. mfIsEmpty(y) .or. mfIsEmpty(z) .or.         &
          mfIsEmpty(tetra) ) then
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE .or. &
          z%data_type /= MF_DT_DBLE .or. tetra%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "real arrays required!" )
         go to 99
      end if

      if( x%shape(1) /= 1 .and. x%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' must be a vector!" )
         go to 99
      end if

      if( y%shape(1) /= 1 .and. y%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'y' must be a vector!" )
         go to 99
      end if

      if( z%shape(1) /= 1 .and. z%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'z' must be a vector!" )
         go to 99
      end if

      if( x%shape(1) /= y%shape(1) .or. x%shape(2) /= y%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' and 'y' must have the same shape!" )
         go to 99
      end if

      if( x%shape(1) /= z%shape(1) .or. x%shape(2) /= z%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' and 'z' must have the same shape!" )
         go to 99
      end if

      if( x%status_temporary ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' cannot be tempo!" )
      end if

      if( y%status_temporary ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'y' cannot be tempo!" )
      end if

      if( z%status_temporary ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'z' cannot be tempo!" )
      end if

      call msPointer( x, x_ptr_vec, no_crc=.true. )
      call msPointer( y, y_ptr_vec, no_crc=.true. )
      call msPointer( z, z_ptr_vec, no_crc=.true. )

      nb_pts = size(x_ptr_vec)

      if( nb_pts < 4 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x', 'y' and 'z' must have at least 4 values!" )
         go to 99
      end if

      if( tetra%status_temporary ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'tetra' cannot be tempo!" )
      end if

      call msPointer( tetra, tetra_ptr_mat, no_crc=.true. )

      nb_tetra = size(tetra_ptr_mat,1)

      if( size(tetra_ptr_mat,2) /= 4 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'tetra' must have 4 columns!" )
         go to 99
      end if

      min_tetra = minval( tetra_ptr_mat(:,:) )
      max_tetra = maxval( tetra_ptr_mat(:,:) )

      if( min_tetra < 1 .or. nb_pts < max_tetra ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'tetra' seems to be wrong!" )
         go to 99
      end if

      allocate( tetra_connect%n_xyz(nb_pts,3) )
      tetra_connect%n_xyz(:,1) = x_ptr_vec(:)
      tetra_connect%n_xyz(:,2) = y_ptr_vec(:)
      tetra_connect%n_xyz(:,3) = z_ptr_vec(:)

      allocate( tetra_connect%tetra_n(nb_tetra,4) )
      tetra_connect%tetra_n(:,:) = tetra_ptr_mat(:,:)

      allocate( tetra_connect%tetra_neighbors(nb_tetra,4) )
      call delaunay_3d_get_neighbors( delaunay3d_addr, tetra_connect%tetra_neighbors )

      call build_tetra_conn( tetra_connect%n_xyz, tetra_connect%tetra_n, &
                             tetra_connect%tetra_neighbors,             &
                             tetra_connect%nodes_sorted_x, tetra_connect%nodes_sorted_y, tetra_connect%nodes_sorted_z, &
                             tetra_connect%face_n, tetra_connect%face_tetra, &
                             tetra_connect%tetra_f, tetra_connect%n_tetra )

      tetra_connect%init = .true.

      call msFreePointer( x, x_ptr_vec )
      call msFreePointer( y, y_ptr_vec )
      call msFreePointer( z, z_ptr_vec )
      call msFreePointer( tetra, tetra_ptr_mat )

      if( mf_phys_units ) then
         ! verifying the physical dimension

         ! 'x' and 'y' must have the same physical units
         call verif_adim( x%units, y%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'y'",&
                               "are not consistent!" )
            go to 99
         end if

         ! 'x' and 'z' must have the same physical units
         call verif_adim( x%units, z%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'z'",&
                               "are not consistent!" )
            go to 99
         end if

         call verif_adim( tetra%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical unit of 'tetra'",          &
                               "must be dimensionless!" )
            go to 99
         end if

      end if

 99   continue

      call msFreeArgs( x, y, z, tetra )
      call msAutoRelease( x, y, z, tetra )

#endif
   end subroutine msBuildTetraConnect_vec
!_______________________________________________________________________
!
   subroutine msBuildTetraConnect_mat( coords, tetra, tetra_connect )

      type(mfArray) :: coords, tetra
      type(mfTetraConnect) :: tetra_connect
      !------ API end ------

#ifdef _DEVLP
      real(kind=MF_DOUBLE), pointer :: tetra_ptr_mat(:,:)
      integer :: nb_pts, nb_tetra, min_tetra, max_tetra
      integer :: status

      character(len=*), parameter :: ROUTINE_NAME = "msBuildTetraConnect"

   !------ end of declarations -- execution starts hereafter  ------

      if( delaunay3d_addr == 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "It seems that your 'tetra' mesh has not been obtain by 'mfDelaunay3D'!", &
                            "(or msEndDelaunay3D has been called too early!)", &
                            " -> in any case, the connectivity cannot be build." )
         go to 99
      end if

      call msInitArgs( coords, tetra )

      if( mfIsEmpty(coords) ) then
         go to 99
      end if

      if( coords%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "real 'coords' required!" )
         go to 99
      end if

      nb_pts = coords%shape(1)

      if( nb_pts < 4 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'coords' must have at least 4 rows!" )
         go to 99
      end if

      if( coords%shape(2) /= 3 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'coords' must have 3 columns!" )
         go to 99
      end if

      call msPointer( tetra, tetra_ptr_mat, no_crc=.true. )

      nb_tetra = size(tetra_ptr_mat,1)

      if( size(tetra_ptr_mat,2) /= 4 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'tetra' must have 4 columns!" )
         go to 99
      end if

      min_tetra = minval( tetra_ptr_mat(:,:) )
      max_tetra = maxval( tetra_ptr_mat(:,:) )

      if( min_tetra < 1 .or. nb_pts < max_tetra ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'tetra' seems to be wrong!" )
         go to 99
      end if

      allocate( tetra_connect%n_xyz(nb_pts,3) )
      tetra_connect%n_xyz = coords

      allocate( tetra_connect%tetra_n(nb_tetra,4) )
      tetra_connect%tetra_n(:,:) = tetra_ptr_mat(:,:)

      allocate( tetra_connect%tetra_neighbors(nb_tetra,4) )
      call delaunay_3d_get_neighbors( delaunay3d_addr, tetra_connect%tetra_neighbors )

      call build_tetra_conn( tetra_connect%n_xyz, tetra_connect%tetra_n, &
                             tetra_connect%tetra_neighbors,             &
                             tetra_connect%nodes_sorted_x, tetra_connect%nodes_sorted_y, tetra_connect%nodes_sorted_z, &
                             tetra_connect%face_n, tetra_connect%face_tetra, &
                             tetra_connect%tetra_f, tetra_connect%n_tetra )

      tetra_connect%init = .true.

      call msFreePointer( tetra, tetra_ptr_mat )

      if( mf_phys_units ) then
         ! verifying the physical dimension

         call verif_adim( tetra%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical unit of 'tetra'",          &
                               "must be dimensionless!" )
            go to 99
         end if

      end if

 99   continue

      call msFreeArgs( coords, tetra )
      call msAutoRelease( coords, tetra )

#endif
   end subroutine msBuildTetraConnect_mat
!_______________________________________________________________________
!
   subroutine msExtractTetraConnect( tetra_connect, tetra_f, face_n,    &
                                     face_tetra )

      type(mfTetraConnect)           :: tetra_connect
      integer, allocatable, optional :: tetra_f(:,:),                   &
                                        face_tetra(:,:),                &
                                        face_n(:,:)
      !------ API end ------

#ifdef _DEVLP
      integer :: nt, nf

      character(len=*), parameter :: ROUTINE_NAME = "msExtractTetraConnect"

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. tetra_connect%init ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "tetra_connect is not initialized!" )
         return
      end if

      if( present(tetra_f) ) then
         nt = size(tetra_connect%tetra_f,1)
         if( allocated(tetra_f) ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "tetra_f was already allocated!",        &
                               "-> going to overwrite it." )
            if( any(shape(tetra_f) /= [nt,3]) ) then
               deallocate( tetra_f )
               allocate( tetra_f(nt,4) )
            end if
         else
            allocate( tetra_f(nt,4) )
         end if
         tetra_f(:,:) = tetra_connect%tetra_f(:,:)
      end if

      if( present(face_n) ) then
         nf = size(tetra_connect%face_n,1)
         if( allocated(face_n) ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "face_n was already allocated!",         &
                               "-> going to overwrite it." )
            if( any(shape(face_n) /= [nf,3]) ) then
               deallocate( face_n )
               allocate( face_n(nf,3) )
            end if
         else
            allocate( face_n(nf,3) )
         end if
         face_n(:,:) = tetra_connect%face_n(:,:)
      end if

      if( present(face_tetra) ) then
         nf = size(tetra_connect%face_tetra,1)
         if( allocated(face_tetra) ) then
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "face_tetra was already allocated!",     &
                               "-> going to overwrite it." )
            if( any(shape(face_tetra) /= [nf,2]) ) then
               deallocate( face_tetra )
               allocate( face_tetra(nf,2) )
            end if
         else
            allocate( face_tetra(nf,2) )
         end if
         face_tetra(:,:) = tetra_connect%face_tetra(:,:)
      end if

#endif
   end subroutine msExtractTetraConnect
!_______________________________________________________________________
!
   function mfTetraSearch_mfarray( tetra_connect, x, y, z )             &
            result( num )

      type(mfTetraConnect) :: tetra_connect
      type(mfArray) :: x, y, z
      integer :: num
      !------ API end ------

#ifdef _DEVLP
      ! returns a tetrahedron number, or 0

      real(kind=MF_DOUBLE) :: xi, yi, zi
      integer :: status

      character(len=*), parameter :: ROUTINE_NAME = "mfTetraSearch"

   !------ end of declarations -- execution starts hereafter  ------

      num = 0

      if( .not. tetra_connect%init ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "arg 'tetra_connect' is not initialized!",  &
                            "(you must use 'msBuildTetraConnect' routine before)" )
         return
      end if

      call msInitArgs( x, y, z )

      if( .not. mfIsScalar(x) .or. .not. mfIsScalar(y) .or.             &
          .not. mfIsScalar(z) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x', 'y' and 'z' must be both scalar!" )
         go to 99
      end if

      xi = mfDble( mfGet(x,1) )
      yi = mfDble( mfGet(y,1) )
      zi = mfDble( mfGet(z,1) )

      num = tsearch_3D( tetra_connect%n_xyz, tetra_connect%tetra_n,     &
                        tetra_connect%nodes_sorted_x,                   &
                        tetra_connect%nodes_sorted_y,                   &
                        tetra_connect%nodes_sorted_z,                   &
                        tetra_connect%face_tetra, tetra_connect%tetra_f, &
                        tetra_connect%n_tetra, xi, yi, zi )

      if( mf_phys_units ) then
         ! verifying the physical dimension

         ! 'x' and 'y' must have the same physical units
         call verif_adim( x%units, y%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'y'",&
                               "are not consistent!" )
            go to 99
         end if

         ! 'x' and 'z' must have the same physical units
         call verif_adim( x%units, z%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'z'",&
                               "are not consistent!" )
            go to 99
         end if

      end if

 99   continue

      call msFreeArgs( x, y, z )

      call msAutoRelease( x, y, z )

#endif
   end function mfTetraSearch_mfarray
!_______________________________________________________________________
!
   function mfTetraSearch_double( tetra_connect, xi, yi, zi )           &
            result( num )

      type(mfTetraConnect) :: tetra_connect
      real(kind=MF_DOUBLE) :: xi, yi, zi
      integer :: num
      !------ API end ------

#ifdef _DEVLP
      ! returns a tetrahedron number, or 0

      character(len=*), parameter :: ROUTINE_NAME = "mfTetraSearch"

   !------ end of declarations -- execution starts hereafter  ------

      num = 0

      if( .not. tetra_connect%init ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "arg 'tetra_connect' is not initialized!",  &
                            "(you must use 'msBuildTetraConnect' routine before)" )
         return
      end if

      num = tsearch_3D( tetra_connect%n_xyz, tetra_connect%tetra_n,     &
                        tetra_connect%nodes_sorted_x,                   &
                        tetra_connect%nodes_sorted_y,                   &
                        tetra_connect%nodes_sorted_z,                   &
                        tetra_connect%face_tetra, tetra_connect%tetra_f, &
                        tetra_connect%n_tetra, xi, yi, zi )

#endif
   end function mfTetraSearch_double
!_______________________________________________________________________
!
   function mfNodeSearch3D_mfarray( tetra_connect, x, y, z )            &
            result( num )

      type(mfTetraConnect) :: tetra_connect
      type(mfArray) :: x, y, z
      integer :: num
      !------ API end ------

#ifdef _DEVLP
      ! returns a triangle number, or 0 (when the point is outside the
      ! convex hull)

      real(kind=MF_DOUBLE) :: xi, yi, zi, dmin, dist
      integer :: i, k

      character(len=*), parameter :: ROUTINE_NAME = "mfNodeSearch"

   !------ end of declarations -- execution starts hereafter  ------

      num = 0

      if( .not. tetra_connect%init ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "arg 'tetra_connect' is not initialized!",  &
                            "(you must use 'msBuildTetraConnect' routine before)" )
         return
      end if

      xi = mfDble( mfGet(x,1) )
      yi = mfDble( mfGet(y,1) )
      zi = mfDble( mfGet(z,1) )

      k = tsearch_3D( tetra_connect%n_xyz, tetra_connect%tetra_n,       &
                      tetra_connect%nodes_sorted_x,                     &
                      tetra_connect%nodes_sorted_y,                     &
                      tetra_connect%nodes_sorted_z,                     &
                      tetra_connect%face_tetra, tetra_connect%tetra_f,  &
                      tetra_connect%n_tetra, xi, yi, zi )

      if( k == 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "W",                    &
                            "your point is outside the convex hull!",   &
                            "(please verify by using 'mfTetraSearch')" )
         return
      end if

      ! among the four vertices of the enclosing tetrahedron, find
      ! the nearest vertex
      dmin = MF_REALMAX
      do i = 1, 4
         dist = sqrt( (xi-tetra_connect%n_xyz(tetra_connect%tetra_n(k,i),1))**2 + &
                      (yi-tetra_connect%n_xyz(tetra_connect%tetra_n(k,i),2))**2 + &
                      (zi-tetra_connect%n_xyz(tetra_connect%tetra_n(k,i),3))**2 )
         if( dist < dmin ) then
            dmin = dist
            num = tetra_connect%tetra_n(k,i)
         end if
      end do

#endif
   end function mfNodeSearch3D_mfarray
!_______________________________________________________________________
!
   function mfNodeSearch3D_double( tetra_connect, xi, yi, zi )          &
            result( num )

      type(mfTetraConnect) :: tetra_connect
      real(kind=MF_DOUBLE) :: xi, yi, zi
      integer :: num
      !------ API end ------

#ifdef _DEVLP
      ! returns a triangle number, or 0 (when the point is outside the
      ! convex hull)

      real(kind=MF_DOUBLE) :: dmin, dist
      integer :: i, k

      character(len=*), parameter :: ROUTINE_NAME = "mfNodeSearch"

   !------ end of declarations -- execution starts hereafter  ------

      num = 0

      if( .not. tetra_connect%init ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "arg 'tetra_connect' is not initialized!",  &
                            "(you must use 'msBuildTetraConnect' routine before)" )
         return
      end if

      k = tsearch_3D( tetra_connect%n_xyz, tetra_connect%tetra_n,       &
                      tetra_connect%nodes_sorted_x,                     &
                      tetra_connect%nodes_sorted_y,                     &
                      tetra_connect%nodes_sorted_z,                     &
                      tetra_connect%face_tetra, tetra_connect%tetra_f,  &
                      tetra_connect%n_tetra, xi, yi, zi )

      if( k == 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "W",                    &
                            "your point is outside the convex hull!",   &
                            "(please verify by using 'mfTetraSearch')" )
         return
      end if

      ! among the four vertices of the enclosing tetrahedron, find
      ! the nearest vertex
      dmin = MF_REALMAX
      do i = 1, 4
         dist = sqrt( (xi-tetra_connect%n_xyz(tetra_connect%tetra_n(k,i),1))**2 + &
                      (yi-tetra_connect%n_xyz(tetra_connect%tetra_n(k,i),2))**2 + &
                      (zi-tetra_connect%n_xyz(tetra_connect%tetra_n(k,i),3))**2 )
         if( dist < dmin ) then
            dmin = dist
            num = tetra_connect%tetra_n(k,i)
         end if
      end do

#endif
   end function mfNodeSearch3D_double
!_______________________________________________________________________
!
   subroutine msPrintTetraConnect( tetra_connect, short_info )

      type(mfTetraConnect) :: tetra_connect
      logical, optional :: short_info
      !------ API end ------

#ifdef _DEVLP
      ! This routine is designed to check easily a tetrahedral connectivity.
      ! It should be ok only for a small number of nodes, since the print
      ! format is everywhere 'I3' for all integers!
      !
      ! The optional argument 'short_info' allows the user to just print
      ! the numbers of different items: nodes, tetrahedra and faces.
      ! It allows moreover to get infos for big structures.

      integer :: i, n, nn, nt, nf
      logical :: short_output

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. tetra_connect%init ) then
         print *, "(Muesli:) msPrintTetraConnect: argument is not initialized!"
         return
      end if

      nn = size(tetra_connect%n_xyz,1)
      nt = size(tetra_connect%tetra_n,1)
      nf = size(tetra_connect%face_n,1)

      if( present(short_info) ) then
         short_output = short_info
      else
         short_output = .false.
      end if

      n = max( nn, nt, nf )
      if( n > 999 .and. .not. short_output ) then
         print *, "(Muesli:) msPrintTetraConnect: too big mesh!"
         print *, "   -> you can only get the numbers of items by adding the"
         print *, "      optional argument 'short_info'."
         print *, "          (or recompile the Muesli library using an integer format"
         print *, "           greater than 'I3')"
         return
      end if

      print "(/,A)", "        ┏━━━━━━━━━━━━━━━━━━━━━━━━━━┓"
      print "(A)",   "        ┃ Tetrahedral Connectivity ┃"
      print "(A)",   "        ┗━━━━━━━━━━━━━━━━━━━━━━━━━━┛"
      print "(A,I0)","    nn: number of nodes      = ", nn
      print "(A,I0)","    nt: number of tetrahedra = ", nt
      print "(A,I0)","    nf: number of faces      = ", nf
      print *

      if( short_output ) then
         print *
         return
      end if

      print "(/,A)", "  ┌────────────────────────────────────────────────┐"
      print "(A)",   "  │ n_xyz(nn,3):  nodes by 3D-coordinates          │"
      print "(A)",   "  ├────────────────────────────────────────────────┤"
      print "(A)",   "  │ node                  coordinates              │"
      n = size(tetra_connect%n_xyz,1)
      do i = 1, n
         print "(2X,'│',2X,I3,3X,3(2X,ES11.4),' │')", i, tetra_connect%n_xyz(i,:)
      end do
      print "(A)",   "  └────────────────────────────────────────────────┘"
      print "(/,A)", "  ┌────────────────────────────────────────┐"
      print "(A)",   "  │ tetra_n(nt,4):  tetrahedra by nodes    │"
      print "(A)",   "  │                 (direct orientation)   │"
      print "(A)",   "  ├────────────────────────────────────────┤"
      print "(A)",   "  │ tetra                    nodes         │"
      n = size(tetra_connect%tetra_n,1)
      do i = 1, n
         print "(2X,'│',2X,I3,12X,4(2X,I3),3X,'│')", i, tetra_connect%tetra_n(i,:)
      end do
      print "(A)",   "  └────────────────────────────────────────┘"
      print "(/,A)", "  ┌────────────────────────────────────────────────┐"
      print "(A)",   "  │ tetra_neighbors(nt,4):  tetrahedra by tetra    │"
      print "(A)",   "  │                         (ordered from tetra_n) │"
      print "(A)",   "  ├────────────────────────────────────────────────┤"
      print "(A)",   "  │ tetra                     tetra                │"
      do i = 1, n
         print "(2X,'│',2X,I3,12X,4(2X,I3),11X,'│')", i, tetra_connect%tetra_neighbors(i,:)
      end do
      print "(A)",   "  └────────────────────────────────────────────────┘"
      print "(/,A)", "  ┌───────────────────────────────┐"
      print "(A)",   "  │ face_n(nf,3): faces by nodes  │"
      print "(A)",   "  ├───────────────────────────────┤"
      print "(A)",   "  │ face            nodes         │"
      n = size(tetra_connect%face_n,1)
      do i = 1, n
         print "(2X,'│',2X,I3,8X,3(2X,I3),3X,'│')", i, tetra_connect%face_n(i,:)
      end do
      print "(A)",   "  └───────────────────────────────┘"
      print "(/,A)", "  ┌─────────────────────────┐"
      print "(A)",   "  │ face_tetra(nf,2):  face │"
      print "(A)",   "  │       common to 2 tetra │"
      print "(A)",   "  ├─────────────────────────┤"
      print "(A)",   "  │ face          tetra     │"
      do i = 1, n
         print "(2X,'│',2X,I3,8X,2(2X,I3),2X,'│')", i, tetra_connect%face_tetra(i,:)
      end do
      print "(A)",   "  └─────────────────────────┘"
      print "(/,A)", "  ┌─────────────────────────────────────┐"
      print "(A)",   "  │ tetra_f(nt,4):  tetrahedra by faces │"
      print "(A)",   "  ├─────────────────────────────────────┤"
      print "(A)",   "  │ tetra               faces           │"
      n = size(tetra_connect%tetra_f,1)
      do i = 1, n
         print "(2X,'│',2X,I3,10X,4(2X,I3),2X,'│')", i, tetra_connect%tetra_f(i,:)
      end do
      print "(A)",   "  └─────────────────────────────────────┘"
      print "(/,A)", "  ┌───────────────────────────────┐"
      print "(A)",   "  │ n_tetra(nn):  one tetra which │"
      print "(A)",   "  │         contains a given node │"
      print "(A)",   "  ├───────────────────────────────┤"
      print "(A)",   "  │ node            tetra         │"
      n = size(tetra_connect%n_tetra)
      do i = 1, n
         print "(2X,'│',2X,I3,14X,I3,9X,'│')", i, tetra_connect%n_tetra(i)
      end do
      print "(A)",   "  └───────────────────────────────┘"

#endif
   end subroutine msPrintTetraConnect
!_______________________________________________________________________
!
   subroutine msDel3DNodeNeighbors( tetra, node_neighbors )

      type(mfArray) :: tetra
      type(mf_Int_List), allocatable :: node_neighbors(:)
      !------ API end ------

#ifdef _DEVLP
      integer :: nb_pts, nb_tetra
      character(len=*), parameter :: ROUTINE_NAME = "msDel3DNodeNeighbors"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( tetra )

      if( mfIsEmpty(tetra) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "empty tetra!" )
         go to 99
      end if

      if( allocated(node_neighbors) ) then
         call PrintMessage( trim(ROUTINE_NAME), "W",                    &
                            "'node_neighbors' should be not allocated!", &
                            "(going to deallocate it before using it)")
         deallocate( node_neighbors )
      end if

      nb_tetra = size(tetra,1)

      nb_pts = maxval(tetra%double)
      allocate( node_neighbors(nb_pts) )

      call build_3d_node_neighbors( tetra%double, node_neighbors )

99   continue

      call msFreeArgs( tetra )
      call msAutoRelease( tetra )

#endif
   end subroutine msDel3DNodeNeighbors
!_______________________________________________________________________
!
   subroutine msRelease_mfTetraConnect( tetra_connect )

      type(mfTetraConnect) :: tetra_connect
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( tetra_connect%init ) then

         deallocate( tetra_connect%n_xyz )

         deallocate( tetra_connect%tetra_n )

         deallocate( tetra_connect%tetra_neighbors )

         deallocate( tetra_connect%nodes_sorted_x )

         deallocate( tetra_connect%nodes_sorted_y )

         deallocate( tetra_connect%nodes_sorted_z )

         deallocate( tetra_connect%face_n )

         deallocate( tetra_connect%face_tetra )

         deallocate( tetra_connect%tetra_f )

         deallocate( tetra_connect%n_tetra )

         tetra_connect%init = .false.

      end if

#endif
   end subroutine msRelease_mfTetraConnect
!_______________________________________________________________________
!
   function mfLegendre( n, A ) result( out )

      integer, intent(in) :: n
      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! normalized Legendre polynomial of degree n

      ! low level 'lep' function (computes Legendre polyn.) is
      ! located in 'mf_core' module file, because it is also used in
      ! the 'mod_datafun' module.

      character(len=*), parameter :: ROUTINE_NAME = "mfLegendre"

   !------ end of declarations -- execution starts hereafter  ------

      integer :: i, j
      integer :: status

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray must be real!" )
         go to 99
      end if

      out%shape = a%shape
      out%data_type = MF_DT_DBLE
      allocate( out%double(a%shape(1),a%shape(2)) )

      out%double(:,:) = lep( n, a%double(:,:) )

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )

      call msAutoRelease( a )

      call mf_restore_fpe( )

#endif
   end function mfLegendre
!_______________________________________________________________________
!
   subroutine binomials_n_2n( n, out )

      integer, intent(in) :: n
      real(kind=MF_DOUBLE) :: out(:)
      !------ API end ------

#ifdef _DEVLP
      ! Computes (under a real form), the coefficients :
      !
      !               [ 2k ]
      !               [    ]
      !               [  k ]
      !
      ! from k=1 to k=n     n >= 1
      !
      ! so the vector 'out' must have its length at least equal to n
      !
      ! the sequence is: 1, 2, 6, 20, 70, ...

      real(kind=MF_DOUBLE), allocatable :: a1(:), a2(:)
      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      if( size(out) < n ) then
         write(STDERR,*)
         write(STDERR,*) "(MUESLI binomials_n_2n:) internal error:"
         write(STDERR,*) "                         Please report this bug to: Edouard.Canot@univ-rennes.fr"
         mf_message_displayed = .true.
         call muesli_trace( pause="yes" )
         stop
      end if

      allocate( a1(2*n), a2(2*n) )

      a1(1) = 1.0d0
      out(1) = a1(1)
      if( n == 1 ) return
      do i = 2, 2*n
         a2(1) = 1.0d0
         do j= 2, i-1
            a2(j) = a1(j) + a1(j-1)
            if( i+1 == 2*j ) then
               out(j) = a2(j)
            end if
         end do
         a2(i) = 1.0d0
         a1(1:i) = a2(1:i)
      end do

#endif
   end subroutine binomials_n_2n
!_______________________________________________________________________
!
   subroutine leg_in_cos( n, out )

      integer, intent(in) :: n
      real(kind=MF_DOUBLE) :: out(:,:)
      !------ API end ------

#ifdef _DEVLP
      ! Stores the coefficients of the expansion of
      !         Pi(cos(theta)),      i = 1, n
      ! in terms of
      !         cos(j*theta),      j = 1, n
      !
      ! the matrix 'out' must have its length at least equal to (n,n)

      real(kind=MF_DOUBLE), allocatable :: binomial(:)
      integer :: i, j, k, np

   !------ end of declarations -- execution starts hereafter  ------

      if( size(out,1) < n .or. size(out,2) < n ) then
         write(STDERR,*)
         write(STDERR,*) "(MUESLI leg_in_cos:) internal error:"
         write(STDERR,*) "                     Please report this bug to: Edouard.Canot@univ-rennes.fr"
         mf_message_displayed = .true.
         call muesli_trace( pause="yes" )
         stop
      end if

      out(1,1) = 1.0d0
      if( n == 1 ) return

      ! we are concerned only with the upper triangular part of out:
      do j = 1, n
         do i = j-1, 1, -2
            out(i,j) = 0.0d0
         end do
      end do

      allocate( binomial(n) )
      call binomials_n_2n( n, binomial )

      do np = 2, n, 2
         do i = 2, np, 2
            j = (np-i)/2
            k = np - 1 - j
            out(i,np) = -2.0d0/4.0d0**(np-1) * binomial(j+1)*binomial(k+1)
         end do
      end do

      do np = 3, n, 2
         out(1,np) = 1.0d0/4.0d0**(np-1) * binomial((np-1)/2+1)**2
         do i = 3, np, 2
            j = (np-i)/2
            k = np - 1 - j
            out(i,np) = 2.0d0/4.0d0**(np-1) * binomial(j+1)*binomial(k+1)
         end do
      end do

#endif
   end subroutine leg_in_cos
!_______________________________________________________________________
!
   function mfFourierLeg( A, dim ) result( out )

      type(mfArray)                 :: A
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------

#ifdef _DEVLP
      ! Legendre Expansion
      !
      ! Discrete Legendre transformation, also called
      ! Spherical Fourier transformation
      ! (equivalent to a Fourier-cosine transformation, but uses
      !  Legendre polynomials on [-1,1])
      !
      ! To be valid, there exist some constraints about the data
      ! vector of length n+1:
      !   data is supposed to be equally spaced on [-1,1]
      !   and symmetric at points 1 and n+1;

      integer :: i, j

      character(len=*), parameter :: ROUTINE_NAME = "mfFourierLeg"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "data must be real!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! works on columns (as in Matlab)
            do j = 1, A%shape(2)
               call fourier_leg( A%double(:,j), out%double(:,j) )
            end do
         else if( dim == 2 ) then
            ! works on rows
            do i = 1, A%shape(1)
               call fourier_leg( A%double(i,:), out%double(i,:) )
            end do
         else
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( A%shape(1) == 1 .or. A%shape(2) == 1 ) then
            ! cas d'un vecteur
            if( A%shape(1) == 1 ) then
               call fourier_leg( A%double(1,:), out%double(1,:) )
            else if( A%shape(2) == 1 ) then
               call fourier_leg( A%double(:,1), out%double(:,1) )
            end if
         else
            ! matrix case -> works on columns
            do j = 1, A%shape(2)
               call fourier_leg( A%double(:,j), out%double(:,j) )
            end do
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfFourierLeg
!_______________________________________________________________________
!
   function mfInvFourierLeg( A, dim ) result( out )

      type(mfArray)                 :: A
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------

#ifdef _DEVLP
      ! Legendre Expansion
      !
      ! Discrete Legendre transformation, also called
      ! Spherical Fourier transformation
      ! (equivalent to a Fourier-cosine transformation, but uses
      !  Legendre polynomials on [-1,1])
      !
      ! To be valid, there exist some constraints about the data
      ! vector of length n+1:
      !   data is supposed to be equally spaced on [-1,1]
      !   and symmetric at points 1 and n+1;

      integer :: i, j

      character(len=*), parameter :: ROUTINE_NAME = "mfInvFourierLeg"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "data must be real!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! works on columns (as in Matlab)
            do j = 1, A%shape(2)
               call inv_fourier_leg( A%double(:,j), out%double(:,j) )
            end do
         else if( dim == 2 ) then
            ! works on rows
            do i = 1, A%shape(1)
               call inv_fourier_leg( A%double(i,:), out%double(i,:) )
            end do
         else
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( A%shape(1) == 1 .or. A%shape(2) == 1 ) then
            ! cas d'un vecteur
            if( A%shape(1) == 1 ) then
               call inv_fourier_leg( A%double(1,:), out%double(1,:) )
            else if( A%shape(2) == 1 ) then
               call inv_fourier_leg( A%double(:,1), out%double(:,1) )
            end if
         else
            ! matrix case -> works on columns
            do j = 1, A%shape(2)
               call inv_fourier_leg( A%double(:,j), out%double(:,j) )
            end do
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfInvFourierLeg
!_______________________________________________________________________
!
   function mfRoots( v ) result( out )

      type(mfArray) :: v
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! Find roots of a polynomial

      ! After Matlab.

      type(mfArray) :: ind_nz, c, v_strip
      integer :: idim, n, j, nnz, last_ind, status
      integer :: mf_message_level_save

      character(len=*), parameter :: ROUTINE_NAME = "mfRoots"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( v )

      if( mfIsEmpty(v) ) then
         go to 99
      end if

      ! 'v' cannot be sparse
      if( mfIsSparse(v) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "first arg. cannot be sparse!" )
         go to 99
      end if

      ! 'v' must be real
      if( v%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "first arg. must be real!" )
         go to 99
      end if

      ! 'v' must be a vector
      if( v%shape(1) == 1 ) then
         idim = 2
      else if( v%shape(2) == 1 ) then
         idim = 1
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "first arg. must be a vector mfArray!" )
         go to 99
      end if
      n = size(v)
      call msAssign( ind_nz, mfFind(v) )
      nnz = size( ind_nz )

      ! no roots if the polynomial is a constant
      if( nnz == 0 .or.                                                 &
          (nnz == 1 .and. nint(ind_nz%double(1,1)) == n ) ) then
         call PrintMessage( trim(ROUTINE_NAME), "W",                    &
                            "no roots because your polynomial is constant!" )
         go to 99
      end if

      last_ind = nint(ind_nz%double(1,nnz))
      ! strip leading and trailing zeros
      call msAssign( v_strip,                                           &
                     mfGet( v, nint(ind_nz%double(1,1)).to.last_ind ) )
      if( n > last_ind ) then
         ! stores roots which are zeros
         call msAssign( out, mfZeros( n-last_ind, 1 )*MF_I )
      end if

      call msAssign( c, mfCompan(v_strip) )
      call msAssign( out, out .vc. mfEig(c) )

      call msSilentRelease( c, v_strip )

      ! transpose because the shape of the result must be the same
      ! than those of v.
      ! (mfRoots and mfPoly must be inverse one to the other)
      if( idim == 2 ) then
         mf_message_level_save = mf_message_level
         mf_message_level = 0
         call msAssign( out, .t. out )
         mf_message_level = mf_message_level_save
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( v%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( v )

      call msAutoRelease( v )
      call msSilentRelease( ind_nz )

#endif
   end function mfRoots
!_______________________________________________________________________
!
   subroutine fcn_lmdif_funfit( m, n, var, fvec, fun )

      integer,              intent(in) :: m, n
      real(kind=MF_DOUBLE), intent(in) :: var(n)
      real(kind=MF_DOUBLE)             :: fvec(m)

      interface
         function fun( x, p, n ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x
            real(kind=MF_DOUBLE), intent(in) :: p(n)
            integer,              intent(in) :: n
            real(kind=MF_DOUBLE) :: res
         end function fun
      end interface
      !------ API end ------

#ifdef _DEVLP
      integer :: i

   !------ end of declarations -- execution starts hereafter  ------

      do i = 1, m
         fvec(i) = y_data(i) - fun(x_data(i),var,n)
      end do

      ! check NaN values
      if( MF_NUMERICAL_CHECK ) then
         do i = 1, m
            if( isnan(fvec(i)) ) then
               print "(/,A)", "(MUESLI FunFit:) [fun] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied FUN function."
               print "(20X,A,I0)", "This occured for x = ", x_data(i)
               call muesli_trace( pause="yes" )
               return
            end if
         end do
      end if

#endif
   end subroutine fcn_lmdif_funfit
!_______________________________________________________________________
!
   subroutine msAssign_mfTriConnect_copy( left, right )

      type(mfTriConnect), intent(in out) :: left
      type(mfTriConnect), intent(in)     :: right
      !------ API end ------

#ifdef _DEVLP
      ! usual form of the assignment : left = right
      ! which always do a copy

      ! right is released only if it is tempo

      integer :: nn, nt, nf, nb_fb, nb_boundaries

   !------ end of declarations -- execution starts hereafter  ------

      call msRelease_mfTriConnect( left )

      ! Cannot write: left = right
      if( .not. right%init ) then
         return
      end if
      left%init            = right%init
      left%tri_renumbering = right%tri_renumbering
      left%face_oriented   = right%face_oriented
      left%convex_domain   = right%convex_domain
      nn = size( right%n_xy, 1 )
      nt = size( right%tri_n, 1 )
      nf = size( right%face_n, 1 )
      allocate( left%n_xy(nn,2) )
      left%n_xy = right%n_xy
      allocate( left%tri_n(nt,3) )
      left%tri_n = right%tri_n
      allocate( left%face_n(nf,2) )
      left%face_n = right%face_n
      allocate( left%face_tri(nf,2) )
      left%face_tri = right%face_tri
      allocate( left%tri_f(nt,3) )
      left%tri_f = right%tri_f
      allocate( left%n_tri(nn) )
      left%n_tri = right%n_tri
      nb_fb = size( right%faces_boundary )
      if( right%convex_domain /= -1 ) then
         allocate( left%faces_boundary(nb_fb) )
         left%faces_boundary = right%faces_boundary
         nb_boundaries = size( right%faces_boundary_ptr, 1 )
         allocate( left%faces_boundary_ptr(nb_boundaries,2) )
         left%faces_boundary_ptr = right%faces_boundary_ptr
      end if

      if( right%status_temporary ) then
         call msRelease_mfTriConnect( right )
      end if

#endif
   end subroutine msAssign_mfTriConnect_copy
!_______________________________________________________________________
!
   subroutine msAssign_mfTriConnect( left, right )

      type(mfTriConnect) :: left
      type(mfTriConnect) :: right
      !------ API end ------

#ifdef _DEVLP
      ! subroutine form of the assignment : call msAssign(left,right),
      ! which is a special case of: left = right.

      ! this assignment just copy the pointers for the appropriate components

      ! if right is tempo: its tempo flag is removed (why?)
      ! if right is not tempo: ordinary copy is done

   !------ end of declarations -- execution starts hereafter  ------

      call msRelease_mfTriConnect( left )

      if( .not. right%init ) then
         return
      end if

      if( right%status_temporary ) then

         left%init            = right%init
         left%tri_renumbering = right%tri_renumbering
         left%face_oriented   = right%face_oriented
         left%convex_domain   = right%convex_domain

         left%n_xy     => right%n_xy
         left%tri_n    => right%tri_n
         left%face_n   => right%face_n
         left%face_tri => right%face_tri
         left%tri_f    => right%tri_f
         left%n_tri    => right%n_tri
         if( right%convex_domain /= -1 ) then
            left%faces_boundary     => right%faces_boundary
            left%faces_boundary_ptr => right%faces_boundary_ptr
         end if

         right%status_temporary = .false.

      else

         call msAssign_mfTriConnect_copy( left, right )

      end if

#endif
   end subroutine msAssign_mfTriConnect
!_______________________________________________________________________
!
   subroutine msAssign_mfVoronoiStruct_copy( left, right )

      type(mfVoronoiStruct), intent(in out) :: left
      type(mfVoronoiStruct), intent(in)     :: right
      !------ API end ------

#ifdef _DEVLP
      ! usual form of the assignment : left = right
      ! which always do a copy

      ! right is released only if it is tempo

      integer :: nn, nv

   !------ end of declarations -- execution starts hereafter  ------

      call msRelease_mfVoronoiStruct( left )

      ! Cannot write: left = right
      if( right%init == 0 ) then
         return
      end if
      left%init = right%init
      nn = size( right%n_xy, 1 )
      allocate( left%n_xy(nn,2) )
      left%n_xy = right%n_xy
      if( right%init == 1 .or. right%init == 3 ) then
         nv = size( right%v_xy, 1 )
         allocate( left%v_xy(nv,2) )
         left%v_xy = right%v_xy
         allocate( left%vertices(nn) )
         left%vertices = right%vertices
      end if
      if( right%init == 2 .or. right%init == 3 ) then
         allocate( left%neighbors(nn) )
         left%neighbors = right%neighbors
      end if

      if( right%status_temporary ) then
         call msRelease_mfVoronoiStruct( right )
      end if

#endif
   end subroutine msAssign_mfVoronoiStruct_copy
!_______________________________________________________________________
!
end module mod_polyfun
