module mod_physunits ! Physical Units

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!-----------------------------------------------------------------------
!                             used modules
!-----------------------------------------------------------------------

   use mod_mfarray

   implicit none

#ifndef _DEVLP
   private
#endif

   ! num_base_units : defined in 'mod_mfarray', should be <= 7

   ! names of fundamental quantities (see below) -- keep def. order
   ! (len=2 because non-ascii character Θ takes two chars)
   character(len=2), parameter :: fundam_name(7) =                      &
                                  [ "M ", "L ", "T ", "Θ", "I ", "N ", "J " ]

   type :: mfUnit
#ifndef _DEVLP
      private
#endif
      real(kind=MF_DOUBLE) :: value = 1.0d0
      type(rational) :: units(num_base_units)
      character(len=12) :: abbrev = ""
   end type mfUnit

   public :: mfUnit

   !---------------------------- parameters ----------------------------

   !--- magic keyword
   type(mfUnit), public, parameter :: SI_unit =                         &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "S.I." )

   !--- basic units ---

   ! M : Mass
   type(mfUnit), public, parameter :: u_kilogram =                      &
        mfUnit( 1.0d0,                                                  &
                [ RAT_1, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "kg" )
   type(mfUnit), public, parameter :: u_kg = u_kilogram

   ! L : Length
   type(mfUnit), public, parameter :: u_meter =                         &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_1, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "m" )
   type(mfUnit), public, parameter :: u_m = u_meter

   ! T : Time
   type(mfUnit), public, parameter :: u_second =                        &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_1, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "s" )
   type(mfUnit), public, parameter :: u_s = u_second

   ! Θ : temperature
   type(mfUnit), public, parameter :: u_kelvin =                        &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_0, RAT_1, RAT_0, RAT_0, RAT_0 ],    &
                "K" )
   type(mfUnit), public, parameter :: u_K = u_kelvin

   ! I : electric current
   type(mfUnit), public, parameter :: u_Ampere =                        &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_0, RAT_0, RAT_1, RAT_0, RAT_0 ],    &
                "A" )
   type(mfUnit), public, parameter :: u_A = u_Ampere

   ! N : quantity of substance
   type(mfUnit), public, parameter :: u_mole =                          &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_1, RAT_0 ],    &
                "mol" )
   type(mfUnit), public, parameter :: u_mol = u_mole

   ! J : luminous intensity
   type(mfUnit), public, parameter :: u_candela =                       &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_1 ],    &
                "cd" )
   type(mfUnit), public, parameter :: u_cd = u_candela

   !--- 22 derived units having a special name and a particular symbol ---

   ! radian (plane angle)
   type(mfUnit), public, parameter :: u_rad =                           &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "rad" )

   ! steradian (solide angle)
   type(mfUnit), public, parameter :: u_steradian =                     &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "sr" )

   ! hertz (frequency)
   type(mfUnit), public, parameter :: u_hertz =                         &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_M1, RAT_0, RAT_0, RAT_0, RAT_0 ],   &
                "Hz" )

   ! newton (force)
   type(mfUnit), public, parameter :: u_newton =                        &
        mfUnit( 1.0d0,                                                  &
                [ RAT_1, RAT_1, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0 ],   &
                "N" )

   ! pascal (pressure)
   type(mfUnit), public, parameter :: u_pascal =                        &
        mfUnit( 1.0d0,                                                  &
                [ RAT_1, RAT_M1, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0 ],  &
                "Pa" )

   ! joule (energy)
   type(mfUnit), public, parameter :: u_joule =                         &
        mfUnit( 1.0d0,                                                  &
                [ RAT_1, RAT_2, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0 ],   &
                "J" )

   ! watt (power)
   type(mfUnit), public, parameter :: u_watt =                          &
        mfUnit( 1.0d0,                                                  &
                [ RAT_1, RAT_2, RAT_M3, RAT_0, RAT_0, RAT_0, RAT_0 ],   &
                "W" )

   ! coulomb
   type(mfUnit), public, parameter :: u_coulomb =                       &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_1, RAT_0, RAT_1, RAT_0, RAT_0 ],    &
                "C" )

   ! volt
   type(mfUnit), public, parameter :: u_volt =                          &
        mfUnit( 1.0d0,                                                  &
                [ RAT_1, RAT_2, RAT_M3, RAT_0, RAT_M1, RAT_0, RAT_0 ],  &
                "V" )

   ! farad
   type(mfUnit), public, parameter :: u_farad =                         &
        mfUnit( 1.0d0,                                                  &
                [ RAT_M1, RAT_M2, RAT_4, RAT_0, RAT_2, RAT_0, RAT_0 ],  &
                "F" )

   ! ohm (electric resistivity)
   type(mfUnit), public, parameter :: u_ohm =                           &
        mfUnit( 1.0d0,                                                  &
                [ RAT_1, RAT_2, RAT_M3, RAT_0, RAT_M2, RAT_0, RAT_0 ],  &
                "Ω" )

   ! siemens (electric conductance)
   type(mfUnit), public, parameter :: u_siemens =                       &
        mfUnit( 1.0d0,                                                  &
                [ RAT_M1, RAT_M2, RAT_3, RAT_0, RAT_2, RAT_0, RAT_0 ],  &
                "S" )

   ! weber
   type(mfUnit), public, parameter :: u_weber =                         &
        mfUnit( 1.0d0,                                                  &
                [ RAT_1, RAT_2, RAT_M2, RAT_0, RAT_M1, RAT_0, RAT_0 ],  &
                "Wb" )

   ! tesla
   type(mfUnit), public, parameter :: u_tesla =                         &
        mfUnit( 1.0d0,                                                  &
                [ RAT_1, RAT_0, RAT_M2, RAT_0, RAT_M1, RAT_0, RAT_0 ],  &
                "T" )

   ! henry
   type(mfUnit), public, parameter :: u_henry =                         &
        mfUnit( 1.0d0,                                                  &
                [ RAT_1, RAT_2, RAT_M2, RAT_0, RAT_M2, RAT_0, RAT_0 ],  &
                "H" )

   ! degree Celsius
   type(mfUnit), public, parameter :: u_degree_Celsius =                &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_0, RAT_1, RAT_0, RAT_0, RAT_0 ],    &
                "°C" )

   ! luminous flux
   type(mfUnit), public, parameter :: u_lumen =                         &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_1 ],    &
                "lm" )

   ! illuminance
   type(mfUnit), public, parameter :: u_lux =                           &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0, RAT_1 ],   &
                "lx" )

   ! radionucleid activity
   type(mfUnit), public, parameter :: u_becquerel =                     &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_M1, RAT_0, RAT_0, RAT_0, RAT_0 ],   &
                "Bq" )

   ! absorbed dose
   type(mfUnit), public, parameter :: u_gray =                          &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_2, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0 ],   &
                "Gy" )

   ! equivalent dose
   type(mfUnit), public, parameter :: u_sievert =                       &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_2, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0 ],   &
                "Sv" )

   ! catalytic activity
   type(mfUnit), public, parameter :: u_katal =                         &
        mfUnit( 1.0d0,                                                  &
                [ RAT_0, RAT_0, RAT_M1, RAT_0, RAT_0, RAT_1, RAT_0 ],   &
                "kat" )

   !--- other derived units ---

   ! milligram
   type(mfUnit), public, parameter :: u_mg =                            &
        mfUnit( 1.0d-6,                                                 &
                [ RAT_1, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "mg" )

   ! centigram
   type(mfUnit), public, parameter :: u_cg =                            &
        mfUnit( 1.0d-5,                                                 &
                [ RAT_1, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "cg" )

   ! gram
   type(mfUnit), public, parameter :: u_g =                             &
        mfUnit( 1.0d-3,                                                 &
                [ RAT_1, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "g" )

   ! Angström
   type(mfUnit), public, parameter :: u_angstrom =                      &
        mfUnit( 1.0d-10,                                                &
                [ RAT_0, RAT_1, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "Å" )

   ! micrometer
   type(mfUnit), public, parameter :: u_micron =                        &
        mfUnit( 1.0d-6,                                                 &
                [ RAT_0, RAT_1, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "μm" )

   ! millimeter
   type(mfUnit), public, parameter :: u_mm =                            &
        mfUnit( 1.0d-3,                                                 &
                [ RAT_0, RAT_1, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "mm" )

   ! centimeter
   type(mfUnit), public, parameter :: u_cm =                            &
        mfUnit( 1.0d-2,                                                 &
                [ RAT_0, RAT_1, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "cm" )

   ! kilometer
   type(mfUnit), public, parameter :: u_km =                            &
        mfUnit( 1.0d+3,                                                 &
                [ RAT_0, RAT_1, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "km" )

   ! Astronomical Unit
   type(mfUnit), public, parameter :: u_astronomical_unit =             &
        mfUnit( 1.49597870700d+11,                                      &
                [ RAT_0, RAT_1, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "au" )

   ! minute (ot time)
   type(mfUnit), public, parameter :: u_min =                           &
        mfUnit( 60.0d0,                                                 &
                [ RAT_0, RAT_0, RAT_1, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "min" )
   type(mfUnit), public, parameter :: u_minute = u_min

   ! hour
   type(mfUnit), public, parameter :: u_hr =                            &
        mfUnit( 3600.0d0,                                               &
                [ RAT_0, RAT_0, RAT_1, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "h" )
   type(mfUnit), public, parameter :: u_hour = u_hr

   ! day
   type(mfUnit), public, parameter :: u_day =                           &
        mfUnit( 86400.0d0,                                              &
                [ RAT_0, RAT_0, RAT_1, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "d" )

   ! liter (volume)
   type(mfUnit), public, parameter :: u_liter =                         &
        mfUnit( 1.0d-3,                                                 &
                [ RAT_0, RAT_3, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ],    &
                "L" )

   ! millimeter of Hg (pressure)
   type(mfUnit), public, parameter :: u_mm_Hg =                         &
        mfUnit( 1.333224d+2,                                            &
                [ RAT_1, RAT_M1, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0 ],  &
                "mm Hg" )

   ! bar (pressure)
   type(mfUnit), public, parameter :: u_bar =                           &
        mfUnit( 1.0d5,                                                  &
                [ RAT_1, RAT_M1, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0 ],  &
                "bar" )

   ! atmosphere (pressure)
   type(mfUnit), public, parameter :: u_atm =                           &
        mfUnit( 1.013d5,                                                &
                [ RAT_1, RAT_M1, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0 ],  &
                "atm" )

   ! pound per square inch (pressure)
   type(mfUnit), public, parameter :: u_psi =                           &
        mfUnit( 6895.0d0,                                               &
                [ RAT_1, RAT_M1, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0 ],  &
                "psi" )

   ! calory (energy)
   type(mfUnit), public, parameter :: u_calory =                        &
        mfUnit( 4.18d0,                                                 &
                [ RAT_1, RAT_2, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0 ],   &
                "cal" )

   ! british thermal unit (energy)
   type(mfUnit), public, parameter :: u_btu =                           &
        mfUnit( 1053.4d0,                                               &
                [ RAT_1, RAT_2, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0 ],   &
                "BTU" )

   ! poiseuille (viscosity)
   type(mfUnit), public, parameter :: u_poiseuille =                    &
        mfUnit( 1.0d0,                                                  &
                [ RAT_1, RAT_M1, RAT_M1, RAT_0, RAT_0, RAT_0, RAT_0 ],  &
                "Pl" )

   !---   ---

   ! Some usual non-dimensional multipliers
   real(kind=MF_DOUBLE), public, parameter :: m_yotta = 1.0e+24
   real(kind=MF_DOUBLE), public, parameter :: m_zetta = 1.0e+21
   real(kind=MF_DOUBLE), public, parameter :: m_exa   = 1.0e+18
   real(kind=MF_DOUBLE), public, parameter :: m_peta  = 1.0e+15
   real(kind=MF_DOUBLE), public, parameter :: m_tera  = 1.0e+12
   real(kind=MF_DOUBLE), public, parameter :: m_giga  = 1.0e+09
   real(kind=MF_DOUBLE), public, parameter :: m_mega  = 1.0e+06
   real(kind=MF_DOUBLE), public, parameter :: m_kilo  = 1.0e+03
   real(kind=MF_DOUBLE), public, parameter :: m_hecto = 1.0e+02
   real(kind=MF_DOUBLE), public, parameter :: m_deca  = 1.0e+01
   real(kind=MF_DOUBLE), public, parameter :: m_deci  = 1.0e-01
   real(kind=MF_DOUBLE), public, parameter :: m_centi = 1.0e-02
   real(kind=MF_DOUBLE), public, parameter :: m_milli = 1.0e-03
   real(kind=MF_DOUBLE), public, parameter :: m_micro = 1.0e-06
   real(kind=MF_DOUBLE), public, parameter :: m_nano  = 1.0e-09
   real(kind=MF_DOUBLE), public, parameter :: m_pico  = 1.0e-12
   real(kind=MF_DOUBLE), public, parameter :: m_femto = 1.0e-15
   real(kind=MF_DOUBLE), public, parameter :: m_atto  = 1.0e-18
   real(kind=MF_DOUBLE), public, parameter :: m_zepto = 1.0e-21
   real(kind=MF_DOUBLE), public, parameter :: m_yocto = 1.0e-24

   !------------------------------------------------------
   ! Definitions of common physical constants (from NIST - http://physics.nist.gov/cuu/index.html)

   type(mfUnit), public, parameter :: c_speed_of_light =                &
        mfUnit( 2.99792458d+8,                                          &
                [ RAT_0, RAT_1, RAT_M1, RAT_0, RAT_0, RAT_0, RAT_0 ],   &
                "m/s" )

   type(mfUnit), public, parameter :: c_Planck =                        &
        mfUnit( 6.62606876d-34,                                         &
                [ RAT_1, RAT_2, RAT_M1, RAT_0, RAT_0, RAT_0, RAT_0 ],   &
                "J*s" )

   type(mfUnit), public, parameter :: c_Avogadro =                      &
        mfUnit( 6.02214199d+23,                                         &
                [ RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_M1, RAT_0 ],   &
                "mol^-1" )

   type(mfUnit), public, parameter :: c_universal_gas =                 &
        mfUnit( 8.314472d0,                                             &
                [ RAT_1, RAT_2, RAT_M2, RAT_M1, RAT_0, RAT_M1, RAT_0 ], &
                "J" )

   type(mfUnit), public, parameter :: c_Boltzmann =                     &
        mfUnit( 1.3806503d-23,                                          &
                [ RAT_1, RAT_2, RAT_M2, RAT_M1, RAT_0, RAT_0, RAT_0 ],  &
                "S.I." )

   type(mfUnit), public, parameter :: c_electron_charge =               &
        mfUnit( 1.602176462d-19,                                        &
                [ RAT_0, RAT_0, RAT_1, RAT_0, RAT_1, RAT_0, RAT_0 ],    &
                "C" )

   type(mfUnit), public, parameter :: c_gravity =                       &
        mfUnit( 6.673d-11,                                              &
                [ RAT_M1, RAT_3, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0 ],  &
                "S.I." )

   type(mfUnit), public, parameter :: c_gravity_accel =                 &
        mfUnit( 9.81d0,                                                 &
                [ RAT_0, RAT_1, RAT_M2, RAT_0, RAT_0, RAT_0, RAT_0 ],   &
                "m/s^2" )

   !------------------------------------------------------
   ! Factor conversion

   type(mfUnit), public, parameter :: f_kelvin_conversion =             &
        mfUnit( 2.7315d+2,                                              &
                [ RAT_0, RAT_0, RAT_0, RAT_1, RAT_0, RAT_0, RAT_0 ],    &
                "K" )

   !------------------------------------------------------

   ! operators involving mfArray and mfUnit are defined in
   ! mod_ops.F90 and implemented in fml_ops/OP.inc

   interface operator(+)
      module procedure plus_mfUnit
      module procedure add_mfUnit_mfUnit
   end interface operator(+)
   !------ API end ------

   interface operator(-)
      module procedure minus_mfUnit
      module procedure sub_mfUnit_mfUnit
   end interface operator(-)
   !------ API end ------

   interface operator(*)
      module procedure mul_int_mfUnit
      module procedure mul_real4_mfUnit
      module procedure mul_real8_mfUnit
      module procedure mul_mfUnit_mfUnit
   end interface operator(*)
   !------ API end ------

   interface operator(/)
      module procedure div_mfUnit_mfUnit
   end interface operator(/)
   !------ API end ------

   interface operator(**)
      module procedure pow_mfUnit_int
      module procedure pow_mfUnit_real8
   end interface operator(**)
   !------ API end ------

   interface operator(==)
      module procedure mfUnit_equal_mfUnit
   end interface operator(==)
   !------ API end ------

   interface msSetPhysDim
      module procedure msSetPhysDim_all_dims
      module procedure msSetPhysDim_mfArray
   end interface msSetPhysDim
   !------ API end ------

   interface mfHaveSamePhysDim
      module procedure HaveSamePhysDim_mfarray_mfarray
      module procedure HaveSamePhysDim_mfarray_mfunit
      module procedure HaveSamePhysDim_mfunit_mfarray
      module procedure HaveSamePhysDim_mfunit_mfunit
   end interface mfHaveSamePhysDim
   !------ API end ------

   public :: msSetPhysDim, &
             mfHasNoPhysDim, &
             mfHaveSamePhysDim, &
             operator(+), &
             operator(-), &
             operator(*), &
             operator(/), &
             operator(**), &
             msSetPhysUnitAbbrev, &
             process_units, &
             verif_adim

   type(rational), parameter :: UNIT_ZERO_DIM(num_base_units) =         &
                   [ RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ]

   private :: plus_mfUnit, &
              add_mfUnit_mfUnit, &
              minus_mfUnit, &
              sub_mfUnit_mfUnit, &
              mul_int_mfUnit, &
              mul_real4_mfUnit, &
              mul_real8_mfUnit, &
              mul_mfUnit_mfUnit, &
              div_mfUnit_mfUnit, &
              pow_mfUnit_int, &
              pow_mfUnit_real8, &
              mfUnit_equal_mfUnit, &
              msSetPhysDim_all_dims, &
              msSetPhysDim_mfArray, &
              HaveSamePhysDim_mfarray_mfarray, &
              HaveSamePhysDim_mfarray_mfunit, &
              HaveSamePhysDim_mfunit_mfarray, &
              HaveSamePhysDim_mfunit_mfunit

contains
!_______________________________________________________________________
!
#include "fml_physunits/find_unit_name.inc"
!_______________________________________________________________________
!
   function mfUnit_equal_mfUnit( units_1, units_2 ) result( bool )

      type(rational), intent(in) :: units_1(num_base_units),            &
                                    units_2(num_base_units)
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! '==' operator overload

      integer :: i

   !------ end of declarations -- execution starts hereafter  ------

      bool = .false.

      do i = 1, num_base_units
         if( units_1(i) /= units_2(i) ) then
            return
         end if
      end do

      bool = .true.

#endif
   end function mfUnit_equal_mfUnit
!_______________________________________________________________________
!
   function plus_mfUnit( u ) result( res )

      type(mfUnit), intent(in) :: u
      type(mfUnit) :: res
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      res%value = u%value

      if( mf_phys_units ) then
         res%units(:) = u%units(:)
      else
         call PrintMessage( "operator(+)", "W",                         &
                            "this operation involves a type(mfUnit) object:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function plus_mfUnit
!_______________________________________________________________________
!
   function add_mfUnit_mfUnit( u1, u2 ) result( res )

      type(mfUnit), intent(in) :: u1, u2
      type(mfUnit) :: res
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      res%value = u1%value + u2%value

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( u1%units, u2%units, status )
         if( status /= 0 ) then
            call PrintMessage( "operator(+)", "E",                      &
                               "the physical dimensions of the two operands",&
                               "are not consistent!" )
            return
         end if
         res%units(:) = u1%units(:)
      else
         call PrintMessage( "operator(+)", "W",                         &
                            "this operation involves type(mfUnit) objects:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function add_mfUnit_mfUnit
!_______________________________________________________________________
!
   function minus_mfUnit( u ) result( res )

      type(mfUnit), intent(in) :: u
      type(mfUnit) :: res
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      res%value = - u%value

      if( mf_phys_units ) then
         res%units(:) = u%units(:)
      else
         call PrintMessage( "operator(-)", "W",                         &
                            "this operation involves a type(mfUnit) object:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function minus_mfUnit
!_______________________________________________________________________
!
   function sub_mfUnit_mfUnit( u1, u2 ) result( res )

      type(mfUnit), intent(in) :: u1, u2
      type(mfUnit) :: res
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      res%value = u1%value - u2%value

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( u1%units, u2%units, status )
         if( status /= 0 ) then
            call PrintMessage( "operator(-)", "E",                      &
                               "the physical dimensions of the two operands",&
                               "are not consistent!" )
            return
         end if
         res%units(:) = u1%units(:)
      else
         call PrintMessage( "operator(-)", "W",                         &
                            "this operation involves type(mfUnit) objects:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function sub_mfUnit_mfUnit
!_______________________________________________________________________
!
   function mul_int_mfunit( x, u ) result( res )

      integer, intent(in) :: x
      type(mfUnit), intent(in) :: u
      type(mfUnit) :: res
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      res%value = x * u%value

      if( mf_phys_units ) then
         res%units(:) = u%units(:)
      else
         call PrintMessage( "operator(*)", "W",                         &
                            "this operation involves a type(mfUnit) object:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function mul_int_mfunit
!_______________________________________________________________________
!
   function mul_real4_mfunit( x, u ) result( res )

      real, intent(in) :: x
      type(mfUnit), intent(in) :: u
      type(mfUnit) :: res
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      res%value = x * u%value

      if( mf_phys_units ) then
         res%units(:) = u%units(:)
      else
         call PrintMessage( "operator(*)", "W",                         &
                            "this operation involves a type(mfUnit) object:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function mul_real4_mfunit
!_______________________________________________________________________
!
   function mul_real8_mfunit( x, u ) result( res )

      real(kind=MF_DOUBLE), intent(in) :: x
      type(mfUnit), intent(in) :: u
      type(mfUnit) :: res
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      res%value = x * u%value

      if( mf_phys_units ) then
         res%units(:) = u%units(:)
      else
         call PrintMessage( "operator(*)", "W",                         &
                            "this operation involves a type(mfUnit) object:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function mul_real8_mfunit
!_______________________________________________________________________
!
   function mul_mfUnit_mfUnit( u1, u2 ) result( res )

      type(mfUnit), intent(in) :: u1, u2
      type(mfUnit) :: res
      !------ API end ------

#ifdef _DEVLP
      integer :: i, status

   !------ end of declarations -- execution starts hereafter  ------

      res%value = u1%value * u2%value

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_add( u1%units(i), u2%units(i),                &
                               res%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "operator(*)", "E",                   &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               return
            else if( status == -1 ) then
               call PrintMessage( "operator(*)", "E",                   &
                                  "in processing physical units:",      &
                                  "Please report this bug to: Edouard.Canot@univ-rennes.fr" )
               return
            end if
         end do
      else
         call PrintMessage( "operator(*)", "W",                         &
                            "this operation involves type(mfUnit) objects:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function mul_mfUnit_mfUnit
!_______________________________________________________________________
!
   function div_mfUnit_mfUnit( u1, u2 ) result( res )

      type(mfUnit), intent(in) :: u1, u2
      type(mfUnit) :: res
      !------ API end ------

#ifdef _DEVLP
      integer :: i, status

   !------ end of declarations -- execution starts hereafter  ------

      res%value = u1%value / u2%value

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_sub( u1%units(i), u2%units(i),                &
                               res%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "operator(/)", "E",                   &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               return
            else if( status == -1 ) then
               call PrintMessage( "operator(/)", "E",                   &
                                  "in processing physical units:",      &
                                  "Please report this bug to: Edouard.Canot@univ-rennes.fr" )
               return
            end if
         end do
      else
         call PrintMessage( "operator(/)", "W",                         &
                            "this operation involves type(mfUnit) objects:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function div_mfUnit_mfUnit
!_______________________________________________________________________
!
   function pow_mfUnit_int( u, n ) result( res )

      type(mfUnit), intent(in) :: u
      integer, intent(in) :: n
      type(mfUnit) :: res
      !------ API end ------

#ifdef _DEVLP
      integer :: i, status

   !------ end of declarations -- execution starts hereafter  ------

      res%value = u%value ** n

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_mul( u%units(i), n,                           &
                               res%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "operator(**)", "E",                  &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               return
            else if( status == -1 ) then
               call PrintMessage( "operator(**)", "E",                  &
                                  "in processing physical units:",      &
                                  "Please report this bug to: Edouard.Canot@univ-rennes.fr" )
               return
            end if
         end do
      else
         call PrintMessage( "operator(**)", "W",                        &
                            "this operation involves a type(mfUnit) object:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function pow_mfUnit_int
!_______________________________________________________________________
!
   function pow_mfUnit_real8( u, x ) result( res )

      type(mfUnit), intent(in) :: u
      real(kind=MF_DOUBLE), intent(in) :: x
      type(mfUnit) :: res
      !------ API end ------

#ifdef _DEVLP
      integer :: i, status
      type(rational) :: rat

   !------ end of declarations -- execution starts hereafter  ------

      res%value = u%value ** x

      if( mf_phys_units ) then
         rat = x
         do i = 1, num_base_units
            call rational_mul( u%units(i), rat,                         &
                               res%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "operator(**)", "E",                  &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               return
            else if( status == -1 ) then
               call PrintMessage( "operator(**)", "E",                  &
                                  "in processing physical units:",      &
                                  "Please report this bug to: Edouard.Canot@univ-rennes.fr" )
               return
            end if
         end do
      else
         call PrintMessage( "operator(**)", "W",                        &
                            "this operation involves a type(mfUnit) object:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function pow_mfUnit_real8
!_______________________________________________________________________
!
   subroutine msSetPhysDim_all_dims( x,                                 &
                                     Mass, Length, Time,                &
                                     Temp, Electr_Intens, Mole,         &
                                     Lumin_Intens,                      &
                                     no_dim )

      type(mfArray), intent(in out) :: x
      real(kind=MF_DOUBLE), intent(in), optional :: Mass,               &
                                                    Length,             &
                                                    Time,               &
                                                    Temp,               &
                                                    Electr_Intens,      &
                                                    Mole,               &
                                                    Lumin_Intens
      logical, intent(in), optional :: no_dim
      !------ API end ------

      logical :: no_dim_0

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( x%parameter ) then
         call PrintMessage( "msSetPhysDim", "E",                        &
                            "'x' cannot be a protected array (pseudo-parameter)." )
         return
      end if

      if( x%status_temporary ) then
         call PrintMessage( "msSetPhysDim", "E",                        &
                            "it makes no sense to work in a tempo!",    &
                            "(moreover, the tempo will not be free)" )
         return
      end if

      if( x%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "msSetPhysDim", "E",                        &
                            "mfArray 'x' is empty!",                    &
                            "(you must set the Phys. Units only after some initialization)" )
      end if

      if( present(no_dim) ) then
         no_dim_0 = no_dim
      else
         no_dim_0 = .false.
      end if

      if( mf_phys_units ) then

         if( present(Mass) ) then
            if( no_dim_0 ) then
               call PrintMessage( "msSetPhysDim", "E",                  &
                                 "it makes no sense to use 'no_dim' when another", &
                                 "optional argument is used!" )
               return
            end if
            x%units(1) = Mass
         end if

         if( present(Length) ) then
            if( no_dim_0 ) then
               call PrintMessage( "msSetPhysDim", "E",                  &
                                 "it makes no sense to use 'no_dim' when another", &
                                 "optional argument is used!" )
               return
            end if
            x%units(2) = Length
         end if

         if( present(Time) ) then
            if( no_dim_0 ) then
               call PrintMessage( "msSetPhysDim", "E",                  &
                                 "it makes no sense to use 'no_dim' when another", &
                                 "optional argument is used!" )
               return
            end if
            x%units(3) = Time
         end if

         if( present(Temp) ) then
            if( no_dim_0 ) then
               call PrintMessage( "msSetPhysDim", "E",                  &
                                 "it makes no sense to use 'no_dim' when another", &
                                 "optional argument is used!" )
               return
            end if
            x%units(4) = Temp
         end if

         if( present(Electr_Intens) ) then
            if( no_dim_0 ) then
               call PrintMessage( "msSetPhysDim", "E",                  &
                                 "it makes no sense to use 'no_dim' when another", &
                                 "optional argument is used!" )
               return
            end if
            x%units(5) = Electr_Intens
         end if

         if( present(Mole) ) then
            if( no_dim_0 ) then
               call PrintMessage( "msSetPhysDim", "E",                  &
                                 "it makes no sense to use 'no_dim' when another", &
                                 "optional argument is used!" )
               return
            end if
            x%units(6) = Mole
         end if

         if( present(Lumin_Intens) ) then
            if( no_dim_0 ) then
               call PrintMessage( "msSetPhysDim", "E",                  &
                                 "it makes no sense to use 'no_dim' when another", &
                                 "optional argument is used!" )
               return
            end if
            x%units(7) = Lumin_Intens
         end if

         if( no_dim_0 ) then
            x%units(:) = [ RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ]
         end if

      else
         call PrintMessage( "msSetPhysDim", "W",                        &
                            "this operation involves physical units:",  &
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end subroutine msSetPhysDim_all_dims
!_______________________________________________________________________
!
   subroutine msSetPhysDim_mfArray( x, y )

      type(mfArray), intent(in out) :: x
      type(mfArray), intent(in) :: y
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( x%parameter ) then
         call PrintMessage( "msSetPhysDim", "E",                        &
                            "'x' cannot be a protected array (pseudo-parameter)." )
         return
      end if

      if( x%status_temporary ) then
         call PrintMessage( "msSetPhysDim", "E",                        &
                            "it makes no sense to work in a tempo!",    &
                            "(moreover, the tempo will not be free)" )
         return
      end if

      if( x%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "msSetPhysDim", "E",                        &
                            "mfArray 'x' is empty!",                    &
                            "(you must set the Phys. Units only after some initialization)" )
      end if

      if( mf_phys_units ) then
         x%units(:) = y%units(:)
      else
         call PrintMessage( "msSetPhysDim", "W",                        &
                            "this operation involves physical units:",  &
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end subroutine msSetPhysDim_mfArray
!_______________________________________________________________________
!
   function mfHasNoPhysDim( x ) result( bool )

      type(mfArray), intent(in) :: x
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( x%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfHasNoPhysDim", "E",                      &
                            "mfArray 'x' is empty!",                    &
                            "(you can inquire Phys. Units only after some initialization)" )
      end if

      if( mf_phys_units ) then
         call verif_adim( x%units, status=status )
         if( status == 0 ) then
            bool = .true.
         else
            bool = .false.
         end if
      else
         bool = .true.
         call PrintMessage( "mfHasNoPhysDim", "W",                      &
                            "this operation involves physical units:",  &
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function mfHasNoPhysDim
!_______________________________________________________________________
!
   function HaveSamePhysDim_mfarray_mfarray( x, y ) result( bool )

      type(mfArray), intent(in) :: x, y
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( x%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfHaveSamePhysDim", "E",                   &
                            "mfArray 'x' is empty!",                    &
                            "(you can inquire Phys. Units only after some initialization)" )
      end if

      if( y%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfHaveSamePhysDim", "E",                   &
                            "mfArray 'y' is empty!",                    &
                            "(you can inquire Phys. Units only after some initialization)" )
      end if

      if( mf_phys_units ) then
         call verif_adim( x%units, y%units, status=status )
         if( status == 0 ) then
            bool = .true.
         else
            bool = .false.
         end if
      else
         bool = .false.
         call PrintMessage( "mfHaveSamePhysDim", "W",                   &
                            "this operation involves physical units:",  &
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function HaveSamePhysDim_mfarray_mfarray
!_______________________________________________________________________
!
   function HaveSamePhysDim_mfarray_mfunit( x, y ) result( bool )

      type(mfArray), intent(in) :: x
      type(mfUnit), intent(in) :: y
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( x%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfHaveSamePhysDim", "E",                   &
                            "mfArray 'x' is empty!",                    &
                            "(you can inquire Phys. Units only after some initialization)" )
      end if

      if( mf_phys_units ) then
         call verif_adim( x%units, y%units, status=status )
         if( status == 0 ) then
            bool = .true.
         else
            bool = .false.
         end if
      else
         bool = .false.
         call PrintMessage( "HaveSamePhysDim", "W",                     &
                            "this operation involves a type(mfUnit) object:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function HaveSamePhysDim_mfarray_mfunit
!_______________________________________________________________________
!
   function HaveSamePhysDim_mfunit_mfarray( x, y ) result( bool )

      type(mfUnit), intent(in) :: x
      type(mfArray), intent(in) :: y
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( y%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfHaveSamePhysDim", "E",                   &
                            "mfArray 'y' is empty!",                    &
                            "(you can inquire Phys. Units only after some initialization)" )
      end if

      if( mf_phys_units ) then
         call verif_adim( x%units, y%units, status=status )
         if( status == 0 ) then
            bool = .true.
         else
            bool = .false.
         end if
      else
         bool = .false.
         call PrintMessage( "HaveSamePhysDim", "W",                     &
                            "this operation involves a type(mfUnit) object:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function HaveSamePhysDim_mfunit_mfarray
!_______________________________________________________________________
!
   function HaveSamePhysDim_mfunit_mfunit( x, y ) result( bool )

      type(mfUnit), intent(in) :: x, y
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_phys_units ) then
         call verif_adim( x%units, y%units, status=status )
         if( status == 0 ) then
            bool = .true.
         else
            bool = .false.
         end if
      else
         bool = .false.
         call PrintMessage( "HaveSamePhysDim", "W",                     &
                            "this operation involves type(mfUnit) objects:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end function HaveSamePhysDim_mfunit_mfunit
!_______________________________________________________________________
!
   subroutine process_units( units, phys_dim_str, unit, phys_unit_str )

      type(rational) :: units(num_base_units)
      character(len=*) :: phys_dim_str
      type(mfUnit), optional :: unit
      character(len=*), optional :: phys_unit_str
      !------ API end ------

#ifdef _DEVLP
      integer :: i
      character(len=30) :: tmp_str

   !------ end of declarations -- execution starts hereafter  ------

      phys_dim_str = "["
      do i = 1, num_base_units
         if( units(i)%num /= 0_rat_num_prec ) then
            if( units(i)%den == 1_rat_num_prec ) then
               if( units(i)%num == 1_rat_num_prec ) then
                  tmp_str = " " // trim(fundam_name(i))
               else
                  write(tmp_str,"(A,I0)")                               &
                        " " // trim(fundam_name(i)) // "^",             &
                        units(i)%num
               end if
            else
               write(tmp_str,"(A,I0,A,I0,A)")                           &
                     " " // trim(fundam_name(i)) // "^(",               &
                     units(i)%num, "/", units(i)%den, ")"
            end if
            phys_dim_str = trim(phys_dim_str) // tmp_str
         end if
      end do
      if( len_trim(phys_dim_str)== 1 ) then
         phys_dim_str = trim(phys_dim_str) // " - ]"
      else
         phys_dim_str = trim(phys_dim_str) // " ]"
      end if

      if( present(unit) .and. present(phys_unit_str) ) then
         phys_unit_str = "(" // trim(unit%abbrev) // ")"
      end if

#endif
   end subroutine process_units
!_______________________________________________________________________
!
   subroutine verif_adim( units_1, units_2, status )

      type(rational), intent(in) :: units_1(num_base_units)
      type(rational), intent(in), optional :: units_2(num_base_units)
      integer :: status
      !------ API end ------

#ifdef _DEVLP
      ! verifying that a physical quantity is dimensionless
      ! (if only one argument: units)

      ! in the case of two input arguments, the two units are
      ! subtracted, considering the elementary dimensions one after the
      ! other : 0 must be found in all cases.
      ! So, we verify consistency between the physical dimension
      ! of a MUESLI variable and the unity used to be printed.

      ! status = 0 : ok
      ! status /= 0 : physical dim. mismatch !

      integer :: i, status2
      type(rational) :: res

   !------ end of declarations -- execution starts hereafter  ------

      status = 0

      do i = 1, num_base_units
         if( present(units_2) ) then
            call rational_sub( units_1(i), units_2(i), res, status2 )
         else
            res = units_1(i)
            status2 = 0
         end if
         if( status2 == 1 ) then
            call PrintMessage( "msDisplay", "E",                        &
                               "in processing physical units:",         &
                               "integer overflow!" )
            status = 1
         else if( status2 == -1 ) then
            write(STDERR,*) "(MUESLI msDisplay:) internal error:"
            write(STDERR,*) "                    in processing physical units."
            write(STDERR,*) "                    Please report this bug to: Edouard.Canot@univ-rennes.fr"
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            stop
         end if
         if( res%num /= 0_rat_num_prec ) then
            status = 1
         end if
      end do

#endif
   end subroutine verif_adim
!_______________________________________________________________________
!
   subroutine msSetPhysUnitAbbrev( user_unit, abbrev )

      type(mfUnit) :: user_unit
      character(len=*) :: abbrev
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( mf_phys_units ) then
         user_unit%abbrev = abbrev
      else
         call PrintMessage( "msSetPhysUnitAbbrev", "W",                 &
                            "this operation involves a type(mfUnit) object:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

#endif
   end subroutine msSetPhysUnitAbbrev
!_______________________________________________________________________
!
end module mod_physunits
