module mod_mfarray

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

   use rational_numbers

   use mod_prop

   use mod_mfaux

   implicit none

   ! kind for eight-byte integer
   integer, parameter :: kind_8 = selected_int_kind(18) ! 64 bits

   ! DT : data type
   integer(kind=kind_1), parameter :: MF_DT_EMPTY    = 0,               &
                                      MF_DT_DBLE     = 1,               &
                                      MF_DT_CMPLX    = 2,               &
                                      MF_DT_BOOL     = 3,               &
                                      MF_DT_SP_DBLE  = 4,               &
                                      MF_DT_SP_CMPLX = 5,               &
                                      MF_DT_SP_BOOL  = 6,               &
                                      MF_DT_PERM_VEC = 7

   !---------------------------------------------------

   integer, parameter :: num_base_units = 7
   logical, save :: mf_phys_units = .false.

   integer(kind=kind_1), parameter :: zero_kind_1 = 0, one_kind_1 = 1

   type, public :: mfArray

#ifndef _DEVLP
      private
#endif

      integer(kind=kind_1) :: data_type = MF_DT_EMPTY ! current data type
      integer :: shape(2) = [ 0, 0 ] ! matrix dimensions (row,col)

      ! dense matrix
      real(kind=MF_DOUBLE),    pointer :: double(:,:) => null()
      complex(kind=MF_DOUBLE), pointer :: cmplx(:,:)  => null()
      ! booleans are defined as 1 and 0 in the 'double' field

      ! sparse matrix in CSC format (Compressed Sparse Column)
      !  a : real value  [nnz]      or      z : cmplx value [nnz]
      !  i : row         [nnz]
      !  j : pointer     [ncol+1]
      ! (cf. dns2csc in mod_sparse.F90)
      ! actual size of 'a', 'z' and 'i' (nzmax) can be larger than 'nz'
      ! actual size of 'j' (ncolmax+1) can be larger than 'ncol+1'
      real(kind=MF_DOUBLE),    pointer :: a(:) => null()
      complex(kind=MF_DOUBLE), pointer :: z(:) => null()
      integer,                 pointer :: i(:) => null(),               &
                                          j(:) => null()
      ! in each compressed column, rows are not necessarily sorted
      ! value may be : UNKNOWN, FALSE, TRUE (or hidden values)
      integer(kind=kind_1) :: row_sorted = UNKNOWN
      ! ptr to numeric structure of UMFPACK
      integer(kind=MF_ADDRESS), pointer :: umf4_ptr_numeric => null()

      ! matrix pattern and properties
      type(properties) :: prop

      integer(kind=kind_1) :: level_locked = zero_kind_1 ! used by 'msPointer'
      logical :: crc_stored = .false.
      integer :: crc = 0 ! 32-bit checksum is stored to verify that data has
                         ! not been modified during the use of 'msPointer'
                         ! Warning: 'crc' is also used to store arithmetic
                         !          implying MF_END (see seq_def)

      ! for the value returned by a function.
      logical :: status_temporary = .false.

      integer(kind=kind_1) :: level_protected = zero_kind_1

      ! restricted mode (used by 'msEquiv')
      logical :: status_restricted = .false.

      ! parameter mode (data are protected, except matrix properties)
      logical :: parameter = .false.

      ! physical unit
      type(rational) :: units(num_base_units)

   end type mfArray

   type, public :: mfMatFactor

#ifndef _DEVLP
      private
#endif

      integer :: data_type = MF_DT_EMPTY

      ! dense factor

      type(mfArray), pointer :: mf_ptr_1 => null() ! L
      type(mfArray), pointer :: mf_ptr_2 => null() ! U
      type(mfArray), pointer :: mf_ptr_3 => null() ! P

      ! sparse factor

      integer :: order ! nb of rows of the matrix

      ! 1 (LU  UMFPACK): L and U factors    ptr_1
      ! 2 (LL' CHOLMOD): only L  factor     ptr_1, ptr_2
      ! 4                + other obj        ptr_1, ptr_2, ptr_3
      ! 3 (QR  SPQR):    only Q Householder ptr_1, ptr_2, ptr_3, ptr_4
      integer :: package = 0

      ! contains addresses of C structures
      integer(kind=MF_ADDRESS), pointer :: ptr_1 => null(), &
                                           ptr_2 => null(), &
                                           ptr_3 => null(), &
                                           ptr_4 => null()

      ! used only for package = 3
      integer :: shape(2) ! shape of Q

      ! physical unit
      type(rational) :: units(num_base_units)

   end type mfMatFactor

!-----------------------------------------------------------------------
!                              parameters
!-----------------------------------------------------------------------

#include "fml_core/parameters.inc"

! global variables are located inside 'mod_mfdebug.F90'

!-----------------------------------------------------------------------

   integer, parameter :: STDERR_MD = 0 ! exclusive for the MEM. DEBUG
   private :: STDERR_MD

   interface realloc_array
      module procedure realloc_vec_int
      module procedure realloc_vec_real8
      module procedure realloc_vec_cmplx
      module procedure realloc_vec_real8_to_cmplx
      module procedure realloc_array_real8
      module procedure realloc_array_cmplx
      module procedure realloc_array_real8_to_cmplx
   end interface realloc_array
   !------ API end ------

   private :: realloc_vec_int, &
              realloc_vec_real8, &
              realloc_vec_cmplx, &
              realloc_vec_real8_to_cmplx, &
              realloc_array_real8, &
              realloc_array_cmplx, &
              realloc_array_real8_to_cmplx

contains
!_______________________________________________________________________
!
#include "fml_core/realloc.inc"
!_______________________________________________________________________
!
#define COLOR_RED    achar(27)//"[31m"
#define COLOR_ORANGE achar(27)//"[38;5;202m"
#define COLOR_YELLOW achar(27)//"[33m"
#define NO_COLOR     achar(27)//"[0m"
!_______________________________________________________________________
!
   function MF_COMPILATION_CONFIG() result( string )

      character(len=5) :: string
      !------ API end ------

#ifdef _DEVLP
      ! keep MF_COMPILATION_CONFIG as a function (executable at runtime)
      ! instead of the old manner, which was a parameter string, defined at
      ! compilation and stored in a precompiled module (this later way
      ! prevents to obtain the compilation config when we make a Muesli
      ! installation in twin mode -- i.e. both Debug and Optim libraries,
      ! but only one set of precompiled module files).

   !------ end of declarations -- execution starts hereafter  ------

#ifdef _OPTIM
      string = "Optim"
#else
      string = "Debug"
#endif
#endif

   end function MF_COMPILATION_CONFIG
!_______________________________________________________________________
!
   subroutine PrintMessage( routine, tag, msg1,                         &
                            msg2, msg3, msg4, msg5, msg6, msg7, msg8,   &
                            no_pause )

      character(len=*), intent(in)           :: routine
      character(len=1), intent(in)           :: tag
      character(len=*), intent(in)           :: msg1
      character(len=*), intent(in), optional :: msg2, msg3, msg4, msg5, &
                                                msg6, msg7, msg8
      logical,          intent(in), optional :: no_pause
      !------ API end ------

#ifdef _DEVLP
      ! msg1 should be a short message
      !
      ! long explanation may be split between the remaining character
      ! variables.

      ! formatting of 'msg1' and 'msg2' is modified.

      ! no_pause has been added to allow the suppression of the pause
      ! required by the ERROR level.
      ! By default, no_pause is false.

      integer :: routine_len
      character(len=12) :: form

      logical :: no_pause_arg

   !------ end of declarations -- execution starts hereafter  ------

      ! The following variable is used to communicate with muesli_trace()
      ! (keep this statement at the beginning of the routine)
      mf_message_displayed = .false.

      form = "(3X,A)"

      if( present(no_pause) ) then
         no_pause_arg = no_pause
      else
         no_pause_arg = .false.
      end if

      select case( to_upper(tag) )
         case( "I" ) ! info message
            if( mf_message_level == 3 ) then
               write(STDERR,*)
               if( mf_colored_messages ) then
                  write(STDERR,*) COLOR_YELLOW
               end if
               write(STDERR,*) "(MUESLI " // trim(routine) //           &
                               ":) info: "
               write(STDERR,form) trim(msg1)
               if( present(msg2) ) then
                  write(STDERR,form) trim(msg2)
               end if
               if( present(msg3) ) then
                  write(STDERR,form) trim(msg3)
               end if
               if( present(msg4) ) then
                  write(STDERR,form) trim(msg4)
               end if
               if( present(msg5) ) then
                  write(STDERR,form) trim(msg5)
               end if
               if( present(msg6) ) then
                  write(STDERR,form) trim(msg6)
               end if
               if( present(msg7) ) then
                  write(STDERR,form) trim(msg7)
               end if
               if( present(msg8) ) then
                  write(STDERR,form) trim(msg8)
               end if
               if( mf_colored_messages ) then
                  write(STDERR,*) NO_COLOR
               else
                  write(STDERR,*)
               end if
               mf_message_displayed = .true.
               if( mf_traceback_level == 3 ) then
                  call muesli_trace( pause="no" )
               end if
            end if
         case( "W" ) ! Warning message
            if( mf_message_level >= 2 ) then
               write(STDERR,*)
               if( mf_colored_messages ) then
                  write(STDERR,*) COLOR_ORANGE
               end if
               write(STDERR,*) "(MUESLI " // trim(routine) //           &
                               ":) Warning: "
               write(STDERR,form) trim(msg1)
               if( present(msg2) ) then
                  write(STDERR,form) trim(msg2)
               end if
               if( present(msg3) ) then
                  write(STDERR,form) trim(msg3)
               end if
               if( present(msg4) ) then
                  write(STDERR,form) trim(msg4)
               end if
               if( present(msg5) ) then
                  write(STDERR,form) trim(msg5)
               end if
               if( present(msg6) ) then
                  write(STDERR,form) trim(msg6)
               end if
               if( present(msg7) ) then
                  write(STDERR,form) trim(msg7)
               end if
               if( present(msg8) ) then
                  write(STDERR,form) trim(msg8)
               end if
               if( mf_colored_messages ) then
                  write(STDERR,*) NO_COLOR
               else
                  write(STDERR,*)
               end if
               mf_message_displayed = .true.
               if( mf_traceback_level >= 2 ) then
                  call muesli_trace( pause="no" )
               end if
            end if
         case( "E" ) ! ERROR message
            write(STDERR,*)
            if( mf_colored_messages ) then
               write(STDERR,*) COLOR_RED
            end if
            write(STDERR,*) "(MUESLI " // trim(routine) //              &
                            ":) ERROR: "
            write(STDERR,form) trim(msg1)
            if( present(msg2) ) then
               write(STDERR,form) trim(msg2)
            end if
            if( present(msg3) ) then
               write(STDERR,form) trim(msg3)
            end if
            if( present(msg4) ) then
               write(STDERR,form) trim(msg4)
            end if
            if( present(msg5) ) then
               write(STDERR,form) trim(msg5)
            end if
            if( present(msg6) ) then
               write(STDERR,form) trim(msg6)
            end if
            if( present(msg7) ) then
               write(STDERR,form) trim(msg7)
            end if
            if( present(msg8) ) then
               write(STDERR,form) trim(msg8)
            end if
            if( mf_colored_messages ) then
               write(STDERR,*) NO_COLOR
            else
               write(STDERR,*)
            end if
            mf_message_displayed = .true.
            if( mf_message_level >= 1 ) then
!### TODO ?: to be documented
               if( no_pause_arg ) then
                  ! no pause is required by some routines (especially those
                  ! of the funfun module) because a returned status allow
                  ! the user to take a decision after a problem...
                  if( mf_traceback_level >= 1 ) then
                     call muesli_trace( pause="no" )
                  end if
               else
                  if( mf_traceback_level >= 1 ) then
                     call muesli_trace( pause="yes" )
                  end if
               end if
            else
               if( mf_colored_messages ) then
                  write(STDERR,*) COLOR_RED
                  write(STDERR,*) "[program stops because message level is 0]"//NO_COLOR
               else
                  write(STDERR,*)
                  write(STDERR,*) "[program stops because message level is 0]"
               end if
               if( mf_traceback_level >= 1 ) then
                  call muesli_trace( pause ="no" )
               end if
               stop
            end if
         case default
            write(STDERR,*)
            write(STDERR,*) "(MUESLI PrintMessage:) internal error:"
            write(STDERR,*) "                       unknown tag! (message will be ignored)"
            call muesli_trace( pause ="yes" )
            stop
      end select
#endif

   end subroutine PrintMessage
!_______________________________________________________________________
!
   function op2_pattern_prop( p1, p2 ) result( p )

      type(properties), intent(in) :: p1, p2
      type(properties) :: p
      !------ API end ------

#ifdef _DEVLP
      ! set the pattern flags ('tril' and 'triu') after a binary
      ! operation (element-wise)on two mfArrays: '+', '-', mfMul
      !
      ! DIAG (= tril & triu )   acts like a neutral element (case 1, 2)
      ! GENE (= !tril & !triu ) acts like an absorbing element (case 5)

   !------ end of declarations -- execution starts hereafter  ------

      if( p1%tril == TRUE .and. p1%triu == TRUE ) then ! p1 is DIAG
         p%tril = p2%tril
         p%triu = p2%triu
      else if( p2%tril == TRUE .and. p2%triu == TRUE ) then ! p2 is DIAG
         p%tril = p1%tril
         p%triu = p1%triu
      else if( p1%tril == TRUE .and. p2%tril == TRUE ) then
         p%tril = TRUE
      else if( p1%triu == TRUE .and. p2%triu == TRUE ) then
         p%triu = TRUE
      else if( p1%tril == FALSE .and. p1%triu == FALSE .or.             &
               p2%tril == FALSE .and. p2%triu == FALSE ) then
         p%tril = FALSE
         p%triu = FALSE
      end if
      ! for other cases, 'p' needs not to be changed.
#endif

   end function op2_pattern_prop
!_______________________________________________________________________
!
end module mod_mfarray
