module mod_mf_gsl

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

   ! based on GSL-1.7 -- oct. 2005 (translation to f90 by É. Canot)
   !   ok for GSL-1.10 -- sep. 2007
   ! (pathnames refer to the main GSL directory)
   ! 2 bugs fixed (cf. below the string "GSL bug")

   use mod_ieee, GSL_POSINF => MF_INF,                                  &
                    GSL_NAN => MF_NAN

   implicit none

!#ifndef _DEVLP
   private
!#endif

   integer, parameter :: GSL_DOUBLE = kind(1.0d0)

   ! gsl_machine.h
   real(kind=GSL_DOUBLE), parameter :: GSL_DBL_EPSILON =      2.2204460492503131d-16
   real(kind=GSL_DOUBLE), parameter :: GSL_SQRT_DBL_EPSILON = 1.4901161193847656d-08

   ! gsl_math.h
   real(kind=GSL_DOUBLE), parameter :: M_PI =                 3.1415926535897932d0
   real(kind=GSL_DOUBLE), parameter :: M_PI_2 =               1.5707963267948966d0
   real(kind=GSL_DOUBLE), parameter :: M_LN2 =                6.9314718055994531d-01

   public :: mf_gsl_complex_tan, &
             mf_gsl_complex_arccos_real, &
             mf_gsl_complex_arccos, &
             mf_gsl_complex_arcsin_real, &
             mf_gsl_complex_arcsin, &
             mf_gsl_complex_arctan, &
             mf_gsl_complex_cosh, &
             mf_gsl_complex_sinh, &
             mf_gsl_complex_tanh, &
             mf_gsl_complex_arccosh_real, &
             mf_gsl_complex_arccosh, &
             mf_gsl_complex_arcsinh, &
             mf_gsl_complex_arctanh_real, &
             mf_gsl_complex_arctanh

contains
!_______________________________________________________________________
!
   function mf_gsl_hypot( x, y ) result( res )

      real(kind=GSL_DOUBLE), intent(in) :: x, y
      real(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! sys/hypot.c

      real(kind=GSL_DOUBLE) :: xabs, yabs, min, max, u

   !------ end of declarations -- execution starts hereafter  ------

      xabs = abs(x)
      yabs = abs(y)

      if( xabs < yabs ) then
         min = xabs
         max = yabs
      else
         min = yabs
         max = xabs
      end if

      if( min == 0.0d0 ) then
         res = max
         return
      end if

      u = min / max
      res = max*sqrt(1.0d0+u**2)

   end function mf_gsl_hypot
!_______________________________________________________________________
!
   function mf_gsl_log1p( x ) result( res )

      real(kind=GSL_DOUBLE), intent(in) :: x
      real(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! sys/log1p.c

      real(kind=GSL_DOUBLE) :: y, z

   !------ end of declarations -- execution starts hereafter  ------

      y = 1.0d0 + x
      z = y - 1.0d0
      res = log(y) - (z-x)/y ! cancels errors with IEEE arithmetic

   end function mf_gsl_log1p
!_______________________________________________________________________
!
   function mf_gsl_acosh( x ) result( res )

      real(kind=GSL_DOUBLE), intent(in) :: x
      real(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! sys/invhyp.c

      real(kind=GSL_DOUBLE) :: t

   !------ end of declarations -- execution starts hereafter  ------

      if( x > 1.0d0 / GSL_SQRT_DBL_EPSILON ) then
         res = log(x) + M_LN2
      else if( x > 2.0d0 ) then
         res = log( 2.0d0*x - 1.0d0/(sqrt(x**2-1.0d0)+x) )
      else if( x > 1.0d0 ) then
         t = x - 1.0d0
         res = mf_gsl_log1p( t + sqrt(2.0d0*t + t**2) )
      else if( x == 1.0d0 ) then
         res = 0.0d0
      else
         res = GSL_NAN
      end if

   end function mf_gsl_acosh
!_______________________________________________________________________
!
   function mf_gsl_asinh( x ) result( res )

      real(kind=GSL_DOUBLE), intent(in) :: x
      real(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! sys/invhyp.c

      real(kind=GSL_DOUBLE) :: a, s, a2

   !------ end of declarations -- execution starts hereafter  ------

      a = abs(x)

      if( x < 0.0d0 ) then
         s = -1.0d0
      else
         s =  1.0d0
      end if

      if( a > 1.0d0 / GSL_SQRT_DBL_EPSILON ) then
         res = s*(log(a)+M_LN2)
      else if( a > 2.0d0 ) then
         res = s*log(2.0d0*a+1.0d0/(a+sqrt(a**2+1.0d0)))
      else if( a > GSL_SQRT_DBL_EPSILON ) then
         a2 = a**2
         res = s*mf_gsl_log1p(a+a2/(1.0d0+sqrt(1.0d0+a2)))
      else
         res = x
      end if

   end function mf_gsl_asinh
!_______________________________________________________________________
!
   function mf_gsl_atanh( x ) result( res )

      real(kind=GSL_DOUBLE), intent(in) :: x
      real(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! sys/invhyp.c

      real(kind=GSL_DOUBLE) :: a, s

   !------ end of declarations -- execution starts hereafter  ------

      a = abs(x)

      if( x < 0.0d0 ) then
         s = -1.0d0
      else
         s =  1.0d0
      end if

      if( a > 1.0d0 ) then
         res = GSL_NAN
      else if( a == 1.0d0 ) then
         if( x < 0.0d0 ) then
            res = -GSL_POSINF
         else
            res = GSL_POSINF
         end if
      else if( a >= 0.5d0 ) then
         res = s*0.5d0*mf_gsl_log1p(2.0d0*a/(1.0d0-a))
      else if( a > GSL_DBL_EPSILON ) then
         res = s*0.5d0*mf_gsl_log1p(2.0d0*a+2.0d0*a**2/(1.0d0-a))
      else
         res = x
      end if

   end function mf_gsl_atanh
!_______________________________________________________________________
!
   function mf_gsl_complex_arccos_real( x ) result( res )

      real(kind=GSL_DOUBLE), intent(in) :: x
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! complex/math.c

   !------ end of declarations -- execution starts hereafter  ------

      if( abs(x) <= 1.0d0 ) then
         res = cmplx( acos(x), 0.0d0, kind=GSL_DOUBLE )
      else
         if( x < 0.0d0 ) then
            res = cmplx( M_PI, -mf_gsl_acosh(-x), kind=GSL_DOUBLE )
         else
            res = cmplx( 0.0d0, mf_gsl_acosh(x), kind=GSL_DOUBLE )
         end if
      end if

   end function mf_gsl_complex_arccos_real
!_______________________________________________________________________
!
   function mf_gsl_complex_arcsin_real( x ) result( res )

      real(kind=GSL_DOUBLE), intent(in) :: x
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! complex/math.c

   !------ end of declarations -- execution starts hereafter  ------

      if( abs(x) <= 1.0d0 ) then
         res = cmplx( asin(x), 0.0d0, kind=GSL_DOUBLE )
      else
         if( x < 0.0d0 ) then
            res = cmplx( -M_PI_2, mf_gsl_acosh(-x), kind=GSL_DOUBLE )
         else
            res = cmplx( M_PI_2, -mf_gsl_acosh(x), kind=GSL_DOUBLE )
         end if
      end if

   end function mf_gsl_complex_arcsin_real
!_______________________________________________________________________
!
   function mf_gsl_complex_arccosh_real( x ) result( res )

      real(kind=GSL_DOUBLE), intent(in) :: x
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! complex/math.c

   !------ end of declarations -- execution starts hereafter  ------

      if( x >= 1.0d0 ) then
         res = cmplx( mf_gsl_acosh(x), 0.0d0, kind=GSL_DOUBLE )
      else
         if( x >= -1.0d0 ) then
            res = cmplx( 0.0d0, acos(x), kind=GSL_DOUBLE )
         else
            res = cmplx( mf_gsl_acosh(-x), M_PI, kind=GSL_DOUBLE )
         end if
      end if

   end function mf_gsl_complex_arccosh_real
!_______________________________________________________________________
!
   function mf_gsl_complex_arctanh_real( x ) result( res )

      real(kind=GSL_DOUBLE), intent(in) :: x
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! complex/math.c

   !------ end of declarations -- execution starts hereafter  ------

      if( -1.0d0 < x .and. x < 1.0d0 ) then
         res = cmplx( mf_gsl_atanh(x), 0.0d0, kind=GSL_DOUBLE )
      else
         if( x < 0.0d0 ) then
            res = cmplx( mf_gsl_atanh(1.0d0/x), M_PI_2, kind=GSL_DOUBLE )
         else
            res = cmplx( mf_gsl_atanh(1.0d0/x), -M_PI_2, kind=GSL_DOUBLE )
         end if
      end if

   end function mf_gsl_complex_arctanh_real
!_______________________________________________________________________
!
   function mf_gsl_complex_tan( z ) result( res )

      complex(kind=GSL_DOUBLE), intent(in) :: z
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! complex/math.c

      real(kind=GSL_DOUBLE) :: r, i, d, u, c, s, t

   !------ end of declarations -- execution starts hereafter  ------

      r = real(z)
      i = aimag(z)

      if( abs(i) < 1.0d0 ) then

         d = cos(r)**2 + sinh(i)**2

         res = cmplx( 0.5d0*sin(2.0d0*r)/d, 0.5d0*sinh(2.0d0*i)/d, kind=GSL_DOUBLE )

      else

         u = exp(-i)
         c = 2.0d0*u/(1.0d0-u**2)
         d = 1.0d0 + cos(r)**2 * c**2

         s = c**2
         t = 1.0d0/tanh(i)

         res = cmplx( 0.5d0*sin(2.0d0*r)*s/d, t/d, kind=GSL_DOUBLE )

      end if

   end function mf_gsl_complex_tan
!_______________________________________________________________________
!
   function mf_gsl_complex_arccos( z ) result( res )

      complex(kind=GSL_DOUBLE), intent(in) :: z
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! complex/math.c

   !------ end of declarations -- execution starts hereafter  ------

      real(kind=GSL_DOUBLE) :: r, i, x, y, u, v, a, b, y2, d, apx, am1
      real(kind=GSL_DOUBLE) :: re, im
      real(kind=GSL_DOUBLE), parameter :: a_crossover = 1.5d0,          &
                                          b_crossover = 0.6417d0

      r = real(z)
      i = aimag(z)

      if( i == 0.0d0 ) then
         res = mf_gsl_complex_arccos_real(r)
      else
         x = abs(r)
         y = abs(i)
         u = mf_gsl_hypot(x+1.0d0,y)
         v = mf_gsl_hypot(x-1.0d0,y)
         a = 0.5d0*(u+v)
         b = x / a
         y2 = y**2
         if( b <= b_crossover ) then
            re = acos(b)
         else
            if( x <= 1.0d0 ) then
               d = 0.5d0*(a+x)*(y2/(u+x+1.0d0)+(v+(1.0d0-x)))
               re = atan(sqrt(d)/x)
            else
               apx = a + x
               d = 0.5d0*(apx/(u+x+1.0d0)+apx/(v+(x-1.0d0)))
               re = atan((y*sqrt(d))/x)
            end if
         end if
         if( a <= a_crossover ) then
            if( x < 1.0d0 ) then
               am1 = 0.5d0*(y2/(u+(x+1.0d0))+y2/(v+(1.0d0-x)))
            else
               am1 = 0.5d0*(y2/(u+(x+1.0d0))+(v+(x-1.0d0)))
            end if
            im = mf_gsl_log1p(am1+sqrt(am1*(a+1.0d0)))
         else
            im = log(a+sqrt(a**2-1.0d0))
         end if
         if( r >= 0.0d0 ) then
            if( i >= 0.0d0 ) then
               res = cmplx(re,-im, kind=GSL_DOUBLE)
            else
               res = cmplx(re,im, kind=GSL_DOUBLE)
            end if
         else
            if( i >= 0.0d0 ) then
               res = cmplx(M_PI-re,-im, kind=GSL_DOUBLE)
            else
               res = cmplx(M_PI-re,im, kind=GSL_DOUBLE)
            end if
         end if
      end if

   end function mf_gsl_complex_arccos
!_______________________________________________________________________
!
   function mf_gsl_complex_arcsin( z ) result( res )

      complex(kind=GSL_DOUBLE), intent(in) :: z
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      real(kind=GSL_DOUBLE) :: r, i, x, y, u, v, a, b, y2, d, apx, am1
      real(kind=GSL_DOUBLE) :: re, im
      real(kind=GSL_DOUBLE), parameter :: a_crossover = 1.5d0,          &
                                          b_crossover = 0.6417d0

      ! complex/math.c

   !------ end of declarations -- execution starts hereafter  ------

      r = real(z)
      i = aimag(z)

      if( i == 0.0d0 ) then
         res = mf_gsl_complex_arcsin_real(r)
      else
         x = abs(r)
         y = abs(i)
         u = mf_gsl_hypot(x+1.0d0,y)
         v = mf_gsl_hypot(x-1.0d0,y)
         a = 0.5d0*(u+v)
         b = x / a
         y2 = y**2
         if( b <= b_crossover ) then
            re = asin(b)
         else
            if( x <= 1.0d0 ) then
               d = 0.5d0*(a+x)*(y2/(u+x+1.0d0)+(v+(1.0d0-x)))
               re = atan(x/sqrt(d))
            else
               apx = a + x
               d = 0.5d0*(apx/(u+x+1.0d0)+apx/(v+(x-1.0d0)))
               re = atan(x/(y*sqrt(d)))
            end if
         end if
         if( a <= a_crossover ) then
            if( x < 1.0d0 ) then
               am1 = 0.5d0*(y2/(u+(x+1.0d0))+y2/(v+(1.0d0-x)))
            else
               am1 = 0.5d0*(y2/(u+(x+1.0d0))+(v+(x-1.0d0)))
            end if
            im = mf_gsl_log1p(am1+sqrt(am1*(a+1.0d0)))
         else
            im = log(a+sqrt(a**2-1.0d0))
         end if
         if( r >= 0.0d0 ) then
            if( i >= 0.0d0 ) then
               res = cmplx(re,im, kind=GSL_DOUBLE)
            else
               res = cmplx(re,-im, kind=GSL_DOUBLE)
            end if
         else
            if( i >= 0.0d0 ) then
               res = cmplx(-re,im, kind=GSL_DOUBLE)
            else
               res = cmplx(-re,-im, kind=GSL_DOUBLE)
            end if
         end if
      end if

   end function mf_gsl_complex_arcsin
!_______________________________________________________________________
!
   function mf_gsl_complex_arctan( z ) result( res )

      complex(kind=GSL_DOUBLE), intent(in) :: z
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! complex/math.c

      real(kind=GSL_DOUBLE) :: r, i, rsmall, u, a, b
      real(kind=GSL_DOUBLE) :: im

   !------ end of declarations -- execution starts hereafter  ------

      r = real(z)
      i = aimag(z)

      if( i == 0.0d0 ) then
         res = cmplx( atan(r), 0.0d0, kind=GSL_DOUBLE )
      else
         ! FIXME: This is a naive implementation which does not fully
         ! take into account cancellation errors, overflow, underflow
         ! etc.  It would benefit from the Hull et al treatment.
         rsmall = mf_gsl_hypot(r,i)
         u = 2.0d0*i/(1.0d0+rsmall**2)

         ! FIXME: the following cross-over should be optimized but 0.1
         ! seems to work ok
         if( abs(u) < 0.1d0 ) then
            im = 0.25d0*(mf_gsl_log1p(u)-mf_gsl_log1p(-u))
         else
            a = mf_gsl_hypot(r,i+1.0d0)
            b = mf_gsl_hypot(r,i-1.0d0)
! multiple tests added by É. Canot
            if( a == 0.0d0 ) then
               im = -GSL_POSINF
            else if( b == 0.0d0 ) then
               im = GSL_POSINF
            else
               im = 0.5d0*log(a/b)
            end if
         end if
         if( r == 0.0d0 ) then
            if( i > 1.0d0 ) then
               res = cmplx( M_PI_2, im, kind=GSL_DOUBLE )
            else if( i < -1.0d0 ) then
               res = cmplx( -M_PI_2, im, kind=GSL_DOUBLE )
            else
! GSL bug : should be (NaN,Inf) (contrary as in MATLAB and MUESLI)
!!               res = cmplx( 0.0d0, im, kind=GSL_DOUBLE )
               ! in this case, im is already assigned to Inf
               res = cmplx(GSL_NAN, im, kind=GSL_DOUBLE )
            end if
         else
            res = cmplx(0.5d0*atan2(2*r,((1.0d0+rsmall)*(1.0d0-rsmall))),im, kind=GSL_DOUBLE)
         end if
      end if
   end function mf_gsl_complex_arctan
!_______________________________________________________________________
!
   function mf_gsl_complex_cosh( z ) result( res )

      complex(kind=GSL_DOUBLE), intent(in) :: z
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! complex/math.c

      real(kind=GSL_DOUBLE) :: r, i

   !------ end of declarations -- execution starts hereafter  ------

      r = real(z)
      i = aimag(z)

      res = cmplx( cosh(r)*cos(i), sinh(r)*sin(i), kind=GSL_DOUBLE )

   end function mf_gsl_complex_cosh
!_______________________________________________________________________
!
   function mf_gsl_complex_sinh( z ) result( res )

      complex(kind=GSL_DOUBLE), intent(in) :: z
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! complex/math.c

      real(kind=GSL_DOUBLE) :: r, i

   !------ end of declarations -- execution starts hereafter  ------

      r = real(z)
      i = aimag(z)

      res = cmplx( sinh(r)*cos(i), cosh(r)*sin(i), kind=GSL_DOUBLE )

   end function mf_gsl_complex_sinh
!_______________________________________________________________________
!
   function mf_gsl_complex_tanh( z ) result( res )

      complex(kind=GSL_DOUBLE), intent(in) :: z
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! complex/math.c

      real(kind=GSL_DOUBLE) :: r, i, d, f
      real(kind=GSL_DOUBLE) :: im

   !------ end of declarations -- execution starts hereafter  ------

      r = real(z)
      i = aimag(z)

      d = cos(i)**2 + sinh(r)**2
      im = 0.5d0*sin(2.0d0*i)/d
      if( abs(r) < 1.0d0 ) then
         res = cmplx( sinh(r)*cosh(r)/d, im, kind=GSL_DOUBLE )
      else
         f = 1.0d0 + (cos(i)/sinh(r))**2
         res = cmplx( 1.0d0/(tanh(r)*f), im, kind=GSL_DOUBLE )
      end if

   end function mf_gsl_complex_tanh
!_______________________________________________________________________
!
   function mf_gsl_complex_arccosh( z ) result( res )

      complex(kind=GSL_DOUBLE), intent(in) :: z
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! complex/math.c

      complex(kind=GSL_DOUBLE) :: a

   !------ end of declarations -- execution starts hereafter  ------

      if( aimag(z) == 0.0d0 ) then
         res = mf_gsl_complex_arccosh_real( real(z) )
      else
         a = mf_gsl_complex_arccos(z)
         if( aimag(a) > 0.0d0 ) then
            res = a * (0.0d0,-1.0d0)
         else
            res = a * (0.0d0,1.0d0)
         end if
      end if

   end function mf_gsl_complex_arccosh
!_______________________________________________________________________
!
   function mf_gsl_complex_arcsinh( z ) result( res )

      complex(kind=GSL_DOUBLE), intent(in) :: z
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! complex/math.c

      real(kind=GSL_DOUBLE) :: r, i
      complex(kind=GSL_DOUBLE) :: a

   !------ end of declarations -- execution starts hereafter  ------

      if( aimag(z) == 0.0d0 ) then
         res = mf_gsl_asinh( real(z) )
      else

         a = z * (0.0d0,1.0d0)
         a = mf_gsl_complex_arcsin(a)

         if( real(z)*aimag(z) > 0.0d0 ) then
! GSL bug : this case has been forgotten !
            r = real(a)
            i = aimag(a)
            res = cmplx(-i,-r, kind=GSL_DOUBLE)
         else
            res = a * (0.0d0,-1.0d0)
         end if

      end if

   end function mf_gsl_complex_arcsinh
!_______________________________________________________________________
!
   function mf_gsl_complex_arctanh( z ) result( res )

      complex(kind=GSL_DOUBLE), intent(in) :: z
      complex(kind=GSL_DOUBLE) :: res
      !------ API end ------

      ! complex/math.c

      complex(kind=GSL_DOUBLE) :: a

   !------ end of declarations -- execution starts hereafter  ------

      if( aimag(z) == 0.0d0 ) then
         res = mf_gsl_complex_arctanh_real( real(z) )
      else
         a = z * (0.0d0,1.0d0)
         a = mf_gsl_complex_arctan(a)
         res = a * (0.0d0,-1.0d0)
      end if

   end function mf_gsl_complex_arctanh
!_______________________________________________________________________
!
end module
