module mod_ieee ! IEEE special values checking

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!
! This source file must be compiled without optimization
!
!-----------------------------------------------------------------------
!                             used modules
!-----------------------------------------------------------------------

   use mod_mfdebug

   implicit none

#ifndef _DEVLP
   private
#endif

! Inf and NaN values : we must choose "Quiet NaN" which propagate
! silently through floating-point operations.
! NaNs are mainly used to masked some values in arrays. For example,
! these values are not drawn on figures when using mfPlot().

#if defined _INTEL_IFC
   real, parameter :: MF_INF_SINGLE = transfer( Z"7F800000", 1.0)

   real, parameter :: MF_NAN_SINGLE = transfer( Z"FFC00000", 1.0)

   real(kind=MF_DOUBLE), parameter :: MF_INF =                          &
                                 transfer( Z"7FF0000000000000", 1.0d0)
   real(kind=MF_DOUBLE), parameter :: MF_NAN =                          &
                                 transfer( Z"FFF8000000000000", 1.0d0)
#elif defined _GNU_GFC
   ! From GCC-10, a BOZ literal constant cannot be an argument to transfer

   real, parameter :: MF_INF_SINGLE = transfer( 2139095040, 1.0)

   real, parameter :: MF_NAN_SINGLE = transfer(   -4194304, 1.0)

   integer, parameter :: int8 = selected_int_kind(10) ! 8 bytes integer

   real(kind=MF_DOUBLE), parameter :: MF_INF =                          &
                                 transfer(9218868437227405312_int8,1.0d0)
   real(kind=MF_DOUBLE), parameter :: MF_NAN =                          &
                                 transfer(  -2251799813685248_int8,1.0d0)
#else
- '(MUESLI mod_ieee:) compiler not defined!'
#endif

   public :: MF_INF_SINGLE, MF_NAN_SINGLE ! for FGL
   public :: MF_INF, MF_NAN

   interface mf_isinf
      module procedure mf_isinf_real4_scal
      module procedure mf_isinf_real8_scal
      module procedure mf_isinf_cmplx8_scal
   end interface
   !------ API end ------

   interface mf_isnan
      module procedure mf_isnan_real4_scal
      module procedure mf_isnan_real8_scal
      module procedure mf_isnan_real8_mat
      module procedure mf_isnan_cmplx8_scal
   end interface
   !------ API end ------

   interface mf_isfinite
      module procedure mf_isfinite_real4_scal
      module procedure mf_isfinite_real8_scal
      module procedure mf_isfinite_real8_vec
      module procedure mf_isfinite_real8_mat
      module procedure mf_isfinite_cmplx8_scal
   end interface
   !------ API end ------

   ! define public access of these previous functions, because they
   ! are used in FGL
   public :: mf_isinf, mf_isnan, mf_isfinite

   ! mfIsInf, mfIsNaN, mfIsFinite are defined in mod_elmat

   private :: mf_isinf_real4_scal, &
              mf_isinf_real8_scal, &
              mf_isinf_cmplx8_scal, &
              mf_isnan_real4_scal, &
              mf_isnan_real8_scal, &
              mf_isnan_cmplx8_scal, &
              mf_isfinite_real4_scal, &
              mf_isfinite_real8_scal, &
              mf_isfinite_real8_vec, &
              mf_isfinite_real8_mat, &
              mf_isfinite_cmplx8_scal

   ! add some simple functions for double precision, if not already
   ! defined byt the compiler
   public :: isinf

#if defined _INTEL_IFC
#else
   public :: isnan
#endif
   public :: isfinite

contains
!_______________________________________________________________________
!
! isinf, isnan, isfinite implementations (as in Matlab-7.4):
!
!   for a real number x, isinf(x) returns TRUE if x is +Inf or -Inf
!
!   for a complex number z, isinf(z) returns TRUE if either
!   the real or imaginary part of z is infinite
!
!------
!
!   for a real number x, isnan(x) returns TRUE if x is NaN
!
!   for a complex number z, isnan(z) returns TRUE if either
!   the real or imaginary part of z is NaN
!
!------
!
!   for a real number x, isfinite(x) returns TRUE if x is finite
!
!   for a complex number z, isfinite(z) returns TRUE if both
!   the real or imaginary part of z is finite
!
!------
!
!   for any number x, exactly one of the three quantities isfinite(x),
!   isinf(x), and isnan(x) is TRUE
!_______________________________________________________________________
!
   pure function isinf( x ) result( bool )

      double precision, intent(in) :: x
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( abs(x) == MF_INF ) then
         bool = .true.
      else
         bool = .false.
      end if

#endif
   end function isinf
!_______________________________________________________________________
!
   pure function mf_isinf_real4_scal( x ) result( bool )

      real, intent(in) :: x
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( abs(x) == MF_INF_SINGLE ) then
         bool = .true.
      else
         bool = .false.
      end if

#endif
   end function mf_isinf_real4_scal
!_______________________________________________________________________
!
   pure function mf_isinf_real8_scal( x ) result( bool )

      real(kind=MF_DOUBLE), intent(in) :: x
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( abs(x) == MF_INF ) then
         bool = .true.
      else
         bool = .false.
      end if

#endif
   end function mf_isinf_real8_scal
!_______________________________________________________________________
!
   pure function mf_isinf_cmplx8_scal( z ) result( bool )

      complex(kind=MF_DOUBLE), intent(in) :: z
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( abs(real(z))  == MF_INF .or.                                   &
          abs(aimag(z)) == MF_INF ) then
         bool = .true.
      else
         bool = .false.
      end if

#endif
   end function mf_isinf_cmplx8_scal
!_______________________________________________________________________
!
#if defined _INTEL_IFC
#else
   pure function isnan( x ) result( bool )

      double precision, intent(in) :: x
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( x /= x ) then
         bool = .true.
      else
         bool = .false.
      end if

#endif
   end function isnan
#endif
!_______________________________________________________________________
!
   pure function mf_isnan_real4_scal( x ) result( bool )

      real, intent(in) :: x
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

#if defined _INTEL_IFC
      bool = isnan(x)
#else
      if( x /= x ) then
         bool = .true.
      else
         bool = .false.
      end if
#endif

#endif
   end function mf_isnan_real4_scal
!_______________________________________________________________________
!
   pure function mf_isnan_real8_scal( x ) result( bool )

      real(kind=MF_DOUBLE), intent(in) :: x
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

#if defined _INTEL_IFC
      bool = isnan(x)
#else
      if( x /= x ) then
         bool = .true.
      else
         bool = .false.
      end if
#endif

#endif
   end function mf_isnan_real8_scal
!_______________________________________________________________________
!
   pure function mf_isnan_real8_mat( x ) result( bool )

      real(kind=MF_DOUBLE), intent(in) :: x(:,:)
      logical :: bool( size(x,1), size(x,2) )
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      integer :: m, n, i, j

      m = size(x,1)
      n = size(x,2)

      do j = 1, n
         do i = 1, m
            bool(i,j) = mf_isnan_real8_scal(x(i,j))
         end do
      end do

#endif
   end function mf_isnan_real8_mat
!_______________________________________________________________________
!
   pure function mf_isnan_cmplx8_scal( z ) result( bool )

      complex(kind=MF_DOUBLE), intent(in) :: z
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

#if defined _INTEL_IFC
      bool = isnan(real(z)) .or. isnan(aimag(z))
#else
      if( real(z)  /= real(z) .or.                                      &
          aimag(z) /= aimag(z) ) then
         bool = .true.
      else
         bool = .false.
      end if
#endif

#endif
   end function mf_isnan_cmplx8_scal
!_______________________________________________________________________
!
   pure function isfinite( x ) result( bool )

      double precision, intent(in) :: x
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      bool = ( .not. isinf(x) ) .and. ( .not. isnan(x) )

#endif
   end function isfinite
!_______________________________________________________________________
!
   pure function mf_isfinite_real4_scal( x ) result( bool )

      real, intent(in) :: x
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      bool = ( .not. mf_isinf(x) ) .and. ( .not. mf_isnan(x) )

#endif
   end function mf_isfinite_real4_scal
!_______________________________________________________________________
!
   pure function mf_isfinite_real8_scal( x ) result( bool )

      real(kind=MF_DOUBLE), intent(in) :: x
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      bool = ( .not. mf_isinf(x) ) .and. ( .not. mf_isnan(x) )

#endif
   end function mf_isfinite_real8_scal
!_______________________________________________________________________
!
   pure function mf_isfinite_real8_vec( x ) result( bool )

      real(kind=MF_DOUBLE), intent(in) :: x(:)
      logical :: bool( size(x) )
      !------ API end ------

#ifdef _DEVLP
      integer :: m, i

   !------ end of declarations -- execution starts hereafter  ------

      m = size(x)

      do i = 1, m
         bool(i) = ( .not. mf_isinf(x(i)) ) .and.                       &
                   ( .not. mf_isnan(x(i)) )
      end do

#endif
   end function mf_isfinite_real8_vec
!_______________________________________________________________________
!
   pure function mf_isfinite_real8_mat( x ) result( bool )

      real(kind=MF_DOUBLE), intent(in) :: x(:,:)
      logical :: bool( size(x,1), size(x,2) )
      !------ API end ------

#ifdef _DEVLP
      integer :: m, n, i, j

   !------ end of declarations -- execution starts hereafter  ------

      m = size(x,1)
      n = size(x,2)

      do j = 1, n
         do i = 1, m
            bool(i,j) = ( .not. mf_isinf(x(i,j)) ) .and.                &
                        ( .not. mf_isnan(x(i,j)) )
         end do
      end do

#endif
   end function mf_isfinite_real8_mat
!_______________________________________________________________________
!
   pure function mf_isfinite_cmplx8_scal( z ) result( bool )

      complex(kind=MF_DOUBLE), intent(in) :: z
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      bool = ( .not. mf_isinf(z) ) .and. ( .not. mf_isnan(z) )

#endif
   end function mf_isfinite_cmplx8_scal
!_______________________________________________________________________
!
end module mod_ieee
