module mod_datafun ! Data Analysis Functions

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!-----------------------------------------------------------------------
!                             used modules
!-----------------------------------------------------------------------

   use mod_elmat

#ifndef _DEVLP
   use mod_mfarray ! required for the 2nd pass of the double compilation
   use mod_core ! required for the 2nd pass of the double compilation
#endif

   use gft

   implicit none

#ifndef _DEVLP
   private
#endif

   interface mfMax
      module procedure mfMax_mfArray
      module procedure mfMax_mfArray_mfArray
      module procedure mfMax_mfArray_real8
   end interface mfMax
   !------ API end ------

   interface mfMin
      module procedure mfMin_mfArray
      module procedure mfMin_mfArray_mfArray
      module procedure mfMin_mfArray_real8
   end interface mfMin
   !------ API end ------

   interface mfSort
      module procedure mfSort_vec
      module procedure mfSort_matrix
   end interface mfSort
   !------ API end ------

   interface mfIsSorted
      module procedure mfIsSorted_vec
   end interface mfIsSorted
   !------ API end ------

   interface msSort
      module procedure msSort_vec
      module procedure msSort_vec_out
      module procedure msSort_matrix
      module procedure msSort_matrix_out
   end interface msSort
   !------ API end ------

   interface mfSortRows
      module procedure mfSortRows_mode
      module procedure mfSortRows_cols
   end interface mfSortRows
   !------ API end ------

   interface msSortRows
      module procedure msSortRows_mode
      module procedure msSortRows_mode_out
      module procedure msSortRows_cols
      module procedure msSortRows_cols_out
   end interface msSortRows
   !------ API end ------

   interface mfGradient
      module procedure mfGradient_real8
      module procedure mfGradient_mfArray
   end interface mfGradient
   !------ API end ------

   public :: mfMax, msMax, &
             mfMin, msMin, &
             mfExtrema, &
             mfProd, &
             mfSum, &
             mfSort, msSort, &
             mfIsSorted, &
             mfSortRows, msSortRows, &
             mfDiff, &
             mfMean, &
             mfMedian, &
             mfVar, &
             mfStd, &
             msHist, &
             mfMoments, &
             mfRMS, &
             mfFourierCos, &
             mfInvFourierCos, &
             mfFourierSin, &
             mfInvFourierSin, &
             mfGradient, msGradient, &
             mfIsMember, &
             mfIntersect, &
             mfUnion, &
             mfSmooth, &
             mfFFT, &
             mfInvFFT, &
             mfFFT2, &
             mfInvFFT2, &
             mfXCorr, &
             mfXCorr2, &
             mfQuantile

   private :: median, &
              localize, &
              moments

   private :: mfSortRows_mode, mfSortRows_cols,                         &
              msSortRows_mode, msSortRows_cols,                         &
              msSortRows_mode_out, msSortRows_cols_out

contains
!_______________________________________________________________________
!
#include "fml_datafun/Max.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/maxab.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/maxab1.inc"
!_______________________________________________________________________
!
#include "fml_datafun/Min.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/minab.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/minab1.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/sumcols.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/sumrows.inc"
!_______________________________________________________________________
!
#include "fml_datafun/fourier_cos.inc"
!_______________________________________________________________________
!
#include "fml_datafun/fourier_sin.inc"
!_______________________________________________________________________
!
#include "fml_datafun/fft.inc"
!_______________________________________________________________________
!
#include "fml_datafun/ifft.inc"
!_______________________________________________________________________
!
#include "fml_datafun/fft2.inc"
!_______________________________________________________________________
!
#include "fml_datafun/ifft2.inc"
!_______________________________________________________________________
!
#include "fml_datafun/gradient.inc"
!_______________________________________________________________________
!
#include "fml_datafun/Sort.inc"
!_______________________________________________________________________
!
#include "fml_datafun/Stats.inc"
!_______________________________________________________________________
!
#include "fml_datafun/moments.inc"
!_______________________________________________________________________
!
   function mfIsSorted_vec( A, mode ) result( bool )

      type(mfArray)                          :: A
      character(len=*), intent(in), optional :: mode
      logical                                :: bool
      !------ API end ------

#ifdef _DEVLP
      ! checks that the vector A is sorted, according the mode :
      ! "ascend", "descend" or "either"; only the three first
      ! characters of the optional 'mode' are tested.

      ! this routine accepts a complex vector : sorting concerns first
      ! the module, then the phase-angle.

      ! also accepts NaNs : these special IEEE values should be located
      ! at the end of the sorted vector (as in Matlab).

      real(kind=MF_DOUBLE), pointer :: tab(:)
      complex(kind=MF_DOUBLE), pointer :: tab_cmplx(:)
      character(len=3) :: mode_sort
      logical :: sorted
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( .not. mfIsNumeric(A) ) then
         call PrintMessage( "mfIsSorted", "E",                          &
                            "numeric array required!" )
         go to 99
      end if

      if( A%shape(1) /= 1 .and. A%shape(2) /= 1 ) then
         call PrintMessage( "mfIsSorted", "E",                          &
                            "mfArray 'a' should be a vector!" )
         go to 99
      end if

      if( present(mode) ) then
         if( to_lower(mode(1:3)) == "asc" ) then
            mode_sort = "asc"
         else if( to_lower(mode(1:3)) == "des" ) then
            mode_sort = "des"
         else if( to_lower(mode(1:3)) == "eit" ) then
            mode_sort = "eit"
         else
            call PrintMessage( "mfIsSorted", "E",                       &
                               "bad arg value for 'mode' (must be 'ascend', 'descend' or 'either'!" )
            go to 99
         end if
      else
         mode_sort = "asc"
      end if

      if( A%data_type == MF_DT_DBLE ) then

         if( A%shape(1) == 1 ) then
            tab => A%double(1,:)
         else ! A%shape(2) == 1
            tab => A%double(:,1)
         end if

         if( mode_sort == "asc" ) then
            bool = is_sorted_ascend_real8( tab ) == 0
         else if( mode_sort == "des" ) then
            bool = is_sorted_descend_real8( tab ) == 0
         else ! "either"
            status = is_sorted_ascend_real8( tab )
            if( status == 0 ) then
               bool = .true.
               go to 99
            else if( status == 2 ) then
               bool = .false.
               go to 99
            end if
            status = is_sorted_descend_real8( tab )
            if( status == 0 ) then
               bool = .true.
            else
               bool = .false.
            end if
         end if

      else ! MF_DT_CMPLX

         if( A%shape(1) == 1 ) then
            tab_cmplx => A%cmplx(1,:)
         else ! A%shape(2) == 1
            tab_cmplx => A%cmplx(:,1)
         end if

         if( mode_sort == "asc" ) then
            bool = is_sorted_ascend_cmplx8( tab_cmplx ) == 0
         else if( mode_sort == "des" ) then
            bool = is_sorted_descend_cmplx8( tab_cmplx ) == 0
         else ! "either"
            status = is_sorted_ascend_cmplx8( tab_cmplx )
            if( status == 0 ) then
               bool = .true.
               go to 99
            else if( status == 2 ) then
               bool = .false.
               go to 99
            end if
            status = is_sorted_descend_cmplx8( tab_cmplx )
            if( status == 0 ) then
               bool = .true.
            else
               bool = .false.
            end if
         end if

      end if

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfIsSorted_vec
!_______________________________________________________________________
!
   function mfProd( a, dim ) result( out )

      type(mfArray)                 :: a
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j

      call msInitArgs( a )

   !------ end of declarations -- execution starts hereafter  ------

      if( mfIsEmpty(a) ) then
         go to 99
      end if

      if( mfIsSparse(a) ) then
         call PrintMessage( "mfProd", "E",                              &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfProd", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfProd", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%data_type = a%data_type
      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! product by column (as in Matlab)
            out%shape = [ 1, a%shape(2) ]
            if( a%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do j = 1, a%shape(2)
                  out%double(1,j) = product(a%double(:,j))
               end do
            else if( a%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do j = 1, a%shape(2)
                  out%cmplx(1,j) = product(a%cmplx(:,j))
               end do
            end if
         else if( dim == 2 ) then
            ! product by row
            out%shape = [ a%shape(1), 1 ]
            if( a%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do i = 1, a%shape(1)
                  out%double(i,1) = product(a%double(i,:))
               end do
            else if( a%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do i = 1, a%shape(1)
                  out%cmplx(i,1) = product(a%cmplx(i,:))
               end do
            end if
         else
            call PrintMessage( "mfProd", "E",                           &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( a%shape(1) == 1 .or. a%shape(2) == 1 ) then
            ! vector
            out%shape = [ 1, 1 ]
            if( a%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               if( a%shape(1) == 1 ) then
                  out%double(1,1) = product(a%double(1,:))
               else if( a%shape(2) == 1 ) then
                  out%double(1,1) = product(a%double(:,1))
               end if
            else if( a%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               if( a%shape(1) == 1 ) then
                  out%cmplx(1,1) = product(a%cmplx(1,:))
               else if( a%shape(2) == 1 ) then
                  out%cmplx(1,1) = product(a%cmplx(:,1))
               end if
            end if
         else
            ! matrix -> product by column
            out%shape = [ 1, a%shape(2) ]
            if( a%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do j = 1, a%shape(2)
                  out%double(1,j) = product(a%double(:,j))
               end do
            else if( a%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do j = 1, a%shape(2)
                  out%cmplx(1,j) = product(a%cmplx(:,j))
               end do
            end if
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = a%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )
      call msAutoRelease( a )

#endif
   end function mfProd
!_______________________________________________________________________
!
   function mfSum( a, dim ) result( out )

      type(mfArray)                 :: a
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j, nrow, ncol

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a )

      if( mfIsEmpty(a) ) then
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfSum", "E",                               &
                            "cannot be applied to boolean!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfSum", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      nrow = a%shape(1)
      ncol = a%shape(2)

      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! sum by column (as in Matlab)
            out%shape = [ 1, ncol ]
            if( a%data_type == MF_DT_DBLE ) then
               out%data_type = a%data_type
               allocate( out%double(out%shape(1),out%shape(2)) )

               do j = 1, ncol
                  out%double(1,j) = sum(a%double(:,j))
               end do
            else if( a%data_type == MF_DT_CMPLX ) then
               out%data_type = a%data_type
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do j = 1, ncol
                  out%cmplx(1,j) = sum(a%cmplx(:,j))
               end do
            else if( a%data_type == MF_DT_SP_DBLE ) then
               out%data_type = MF_DT_DBLE
               allocate( out%double(out%shape(1),out%shape(2)) )

               call sumcols( ncol, a%a, a%i, a%j, out%double(1,:) )
            else if( a%data_type == MF_DT_SP_CMPLX ) then
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               call sumcols_cmplx( ncol, a%z, a%i, a%j, out%cmplx(1,:) )
            end if
         else if( dim == 2 ) then
            ! sum by row
            out%shape = [ nrow, 1 ]
            if( a%data_type == MF_DT_DBLE ) then
               out%data_type = a%data_type
               allocate( out%double(out%shape(1),out%shape(2)) )

               do i = 1, nrow
                  out%double(i,1) = sum(a%double(i,:))
               end do
            else if( a%data_type == MF_DT_CMPLX ) then
               out%data_type = a%data_type
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do i = 1, nrow
                  out%cmplx(i,1) = sum(a%cmplx(i,:))
               end do
            else if( a%data_type == MF_DT_SP_DBLE ) then
               out%data_type = MF_DT_DBLE
               allocate( out%double(out%shape(1),out%shape(2)) )

               call sumrows( nrow, ncol, a%a, a%i, a%j, out%double(:,1) )
            else if( a%data_type == MF_DT_SP_CMPLX ) then
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               call sumrows_cmplx( nrow, ncol, a%z, a%i, a%j, out%cmplx(:,1) )
            end if
         else
            call PrintMessage( "mfSum", "E",                            &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( nrow == 1 .or. ncol == 1 ) then
            ! vector
            out%shape = [ 1, 1 ]
            if( a%data_type == MF_DT_DBLE ) then
               out%data_type = a%data_type
               allocate( out%double(out%shape(1),out%shape(2)) )

               if( nrow == 1 ) then
                  out%double(1,1) = sum(a%double(1,:))
               else if( ncol == 1 ) then
                  out%double(1,1) = sum(a%double(:,1))
               end if
            else if( a%data_type == MF_DT_CMPLX ) then
               out%data_type = a%data_type
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               if( nrow == 1 ) then
                  out%cmplx(1,1) = sum(a%cmplx(1,:))
               else if( ncol == 1 ) then
                  out%cmplx(1,1) = sum(a%cmplx(:,1))
               end if
            end if
         else
            ! matrix -> sum by column
            out%shape = [ 1, ncol ]
            if( a%data_type == MF_DT_DBLE ) then
               out%data_type = a%data_type
               allocate( out%double(out%shape(1),out%shape(2)) )

               do j = 1, ncol
                  out%double(1,j) = sum(a%double(:,j))
               end do
            else if( a%data_type == MF_DT_CMPLX ) then
               out%data_type = a%data_type
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do j = 1, ncol
                  out%cmplx(1,j) = sum(a%cmplx(:,j))
               end do
            else if( a%data_type == MF_DT_SP_DBLE ) then
               out%data_type = MF_DT_DBLE
               allocate( out%double(out%shape(1),out%shape(2)) )

               call sumcols( ncol, a%a, a%i, a%j, out%double(1,:) )
            else if( a%data_type == MF_DT_SP_CMPLX ) then
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               call sumcols_cmplx( ncol, a%z, a%i, a%j, out%cmplx(1,:) )
            end if
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = a%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )
      call msAutoRelease( a )

#endif
   end function mfSum
!_______________________________________________________________________
!
   function mfDiff( A, dim ) result( out )

      type(mfArray)                 :: A
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(a) ) then
         call PrintMessage( "mfDiff", "E",                              &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( mfIsScalar(a) ) then
         call PrintMessage( "mfDiff", "E",                              &
                            "'A' cannot be a scalar!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfDiff", "E",                              &
                            "cannot be applied to boolean matrices!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfDiff", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%data_type = A%data_type

      if( present(dim) ) then
         ! A is always considered as a matrix
         if( A%shape(1) == 1 .or. A%shape(2) == 1 ) then ! not a matrix
            call PrintMessage( "mfDiff", "E",                           &
                               "when using 'dim' arg, 'A' must be a matrix!" )
            go to 99
         end if
         if( dim == 1 ) then
            out%shape = [ A%shape(1)-1, A%shape(2) ]
            if( A%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do i = 1, out%shape(1)
                  out%double(i,:) = A%double(i+1,:) - A%double(i,:)
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do i = 1, out%shape(1)
                  out%cmplx(i,:) = A%cmplx(i+1,:) - A%cmplx(i,:)
               end do
            end if
         else ! dim == 2
            out%shape = [ A%shape(1), A%shape(2)-1 ]
            if( A%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do i = 1, out%shape(2)
                  out%double(:,i) = A%double(:,i+1) - A%double(:,i)
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do i = 1, out%shape(2)
                  out%cmplx(:,i) = A%cmplx(:,i+1) - A%cmplx(:,i)
               end do
            end if
         end if
      else
         if( A%shape(2) == 1 ) then ! column vector
            out%shape = [ A%shape(1)-1, 1 ]
            if( A%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do i = 1, out%shape(1)
                  out%double(i,1) = A%double(i+1,1) - A%double(i,1)
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do i = 1, out%shape(1)
                  out%cmplx(i,1) = A%cmplx(i+1,1) - A%cmplx(i,1)
               end do
            end if
         else if( A%shape(1) == 1 ) then ! row vector
            out%shape = [ 1, A%shape(2)-1 ]
            if( A%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do i = 1, out%shape(2)
                  out%double(1,i) = A%double(1,i+1) - A%double(1,i)
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do i = 1, out%shape(2)
                  out%cmplx(1,i) = A%cmplx(1,i+1) - A%cmplx(1,i)
               end do
            end if
         else ! matrix
            out%shape = [ A%shape(1)-1, A%shape(2) ]
            if( A%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do i = 1, out%shape(1)
                  out%double(i,:) = A%double(i+1,:) - A%double(i,:)
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do i = 1, out%shape(1)
                  out%cmplx(i,:) = A%cmplx(i+1,:) - A%cmplx(i,:)
               end do
            end if
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfDiff
!_______________________________________________________________________
!
   subroutine msHist( out, x, x_min, x_max, n_bin )

      type(mfArray), intent(in) :: x
      real(kind=MF_DOUBLE), intent(in) :: x_min, x_max
      integer, intent(in) :: n_bin
      type(mf_Out) :: out
      !------ API end ------

#ifdef _DEVLP
      type(mfArray), pointer :: num, x_bin
      integer :: dim, n, i, k
      real(kind=MF_DOUBLE), allocatable :: xr_bin(:)
      real(kind=MF_DOUBLE) :: fact

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x )

      ! 1 or 2 output args must be specified
      if( out%n /= 1 .and. out%n /= 2 ) then
         call PrintMessage( "msHist", "E",                              &
                            "one or two output args required!",         &
                            "syntax is : call msHist ( mfOut(num), x, x_min, x_max, n_bin )", &
                            "(or)  call msHist ( mfOut(num,x_bin), x, x_min, x_max, n_bin )" )
         go to 99
      end if

      ! internal check for all mfOut args
      if( .not. args_mfout_ok( out, x ) ) then
         call PrintMessage( "msHist", "E",                              &
                            "output arguments cannot be tempo, or cannot share", &
                            "same memory as another input argument." )
         go to 99
      end if

      num => out%ptr1
      call msSilentRelease( num )
      if( out%n == 2 ) then
         x_bin => out%ptr2
         call msSilentRelease( x_bin )
      end if

      ! checking that x is allocated
      if( mfIsEmpty(x) ) then
         call PrintMessage( "msHist", "E",                              &
                            "'x' empty!" )
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "msHist", "E",                              &
                            "'x' must be real!" )
         go to 99
      end if

      if( x%shape(1)==0 .or. x%shape(2)==0 ) then
         call PrintMessage( "msHist", "W",                              &
                            "nothing to do" )
         go to 99
      end if

      ! checking that 'x' is a 1-D array
      if( x%shape(1)==1 ) then
         dim = 2
      else if( x%shape(2)==1 ) then
         dim = 1
      else
         call PrintMessage( "msHist", "E",                              &
                            "cannot yet build an histogram for a matrix" )
         go to 99
      end if

      ! some additional tests
      if( x_max<=x_min .or. n_bin<1 ) then
         call PrintMessage( "msHist", "E",                              &
                            "invalid arguments!" )
         go to 99
      end if

      n = size(x%double)
      allocate( xr_bin(n_bin+1) )
      xr_bin(1) = x_min
      fact = (x_max-x_min)/n_bin
      do i = 2, n_bin+1
         xr_bin(i) = x_min + (i-1.0d0)*fact
      end do
      if( out%n == 2 ) then
         x_bin = xr_bin(:)
      end if
      if( dim==1 ) then
         call msAssign( num, mfZeros(n_bin,1) )
         if( out%n == 2 ) then
            call msAssign( x_bin, .t. x_bin )
         end if
         do i = 1, n
            k = localize( x%double(i,1), xr_bin, n_bin )
            if( 1 <= k .and. k <= n_bin ) then
               num%double(k,1) = num%double(k,1) + 1
            end if
         end do
      else if( dim==2 ) then
         call msAssign( num, mfZeros(1,n_bin) )
         do i = 1, n
            k = localize( x%double(1,i), xr_bin, n_bin )
            if( 1 <= k .and. k <= n_bin ) then
               num%double(1,k) = num%double(1,k) + 1
            end if
         end do
      end if

 99   continue

      call msFreeArgs( x )
      call msAutoRelease( x )

#endif
   end subroutine msHist
!_______________________________________________________________________
!
   function localize( val, x, n ) result( ind )

      real(kind=MF_DOUBLE), intent(in) :: val, x(:)
      integer, intent(in) :: n
      integer :: ind
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      ind = 0
      if( val < x(1) ) then
         return
      end if
      do ind = 1, n
         if( val < x(ind+1) ) then
            return
         end if
      end do
      ind = n + 1

#endif
   end function localize
!_______________________________________________________________________
!
   function mfExtrema( A ) result( out )

      type(mfArray) :: a
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfExtrema", "E",                           &
                            "arg must be a real mfArray!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, 2 ]
      allocate( out%double(1,2) )

      if( A%data_type == MF_DT_DBLE ) then
         out%double(1,1) = minval( A%double )
         out%double(1,2) = maxval( A%double )
      else if( A%data_type == MF_DT_SP_DBLE ) then
         out%double(1,1) = minval( A%a )
         out%double(1,2) = maxval( A%a )
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfExtrema
!_______________________________________________________________________
!
   function mfIsMember( a, set, tol ) result( out )

      type(mfArray) :: a, set
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! same as in Matlab
      !
      ! returns a boolean mfArray which shows which elements of 'A'
      ! are member of 'set'

      type(mfArray) :: set_copy
      integer :: i, j, k, m, n, n_set
      real(kind=MF_DOUBLE) :: elt
      real(kind=MF_DOUBLE), pointer :: set_sorted(:) ! long column
      integer :: k1, k2, km
      logical :: freepointer_used
      real(kind=MF_DOUBLE) :: tol_0

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a, set )

      freepointer_used = .false.

      if( mfIsEmpty(a) ) then
         go to 99
      end if

      if( mfIsEmpty(set) ) then
         call msAssign( out, a /= a )
         out%status_temporary = .true.
         go to 99
      end if

      if( a%data_type /= MF_DT_DBLE .or. set%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfIsMember", "E",                           &
                            "args must be real, dense mfArrays!" )
         go to 99
      end if

      if( present(tol) ) then
         tol_0 = tol
         if( tol_0 < 0.0d0 ) then
            call PrintMessage( "mfIsMember", "E",                       &
                               "'tol' must be a positive real number!" )
            go to 99
         end if
      else
         tol_0 = 0.0d0
      end if

      ! out is a boolean mfArray
      out%data_type = MF_DT_BOOL
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      n_set = size(set)

      ! sorting 'set' (if it is tempo, no copy)
      if( set%status_temporary ) then
         !call msPointer( set, set_sorted )  forbidden in MUESLI
         set_sorted => rank_2_to_1_real8( set%double, n_set )
      else
         call msAssign( set_copy, set )
         call msPointer( set_copy, set_sorted, no_crc=.true. )
         freepointer_used = .true.
      end if
      call quick_sort( "asc", set_sorted )

      m = a%shape(1)
      n = a%shape(2)
      do j = 1, n
         do i = 1, m
            out%double(i,j) = 0.0d0 ! FALSE
            elt = a%double(i,j)
            if( elt < set_sorted(1)-tol_0 .or.                          &
                set_sorted(n_set)+tol_0 < elt ) then
               cycle
            end if
            if( n_set > 4 ) then
               ! binary search for elt in set_sorted(:)
               k1 = 1
               k2 = n_set
               do while( k2-k1 > 1 )
                  km = (k1+k2)/2
                  if( elt < set_sorted(km)-tol_0 ) then
                     k2 = km
                  else
                     k1 = km
                  end if
               end do
               if( abs(elt-set_sorted(k1)) <= tol_0 .or.                &
                   abs(elt-set_sorted(k2)) <= tol_0 ) then
                  out%double(i,j) = 1.0d0 ! TRUE
               end if
            else
               do k = 1, n_set
                  if( abs(elt-set_sorted(k)) <= tol_0 ) then
                     out%double(i,j) = 1.0d0 ! TRUE
                     exit
                  else if( elt < set_sorted(k) ) then
                     exit
                  end if
               end do
            end if
         end do
      end do

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a, set )
      call msAutoRelease( a, set )

      if( freepointer_used ) then
         call msFreePointer( set_copy, set_sorted )
      end if
      call msSilentRelease( set_copy )

#endif
   end function mfIsMember
!_______________________________________________________________________
!
   function mfIntersect( A, B, tol ) result( out )

      type(mfArray) :: A, B
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! same as in Matlab
      !
      ! returns common elements of 'A' and 'B'.
      !
      ! 'A' and 'B' may have any shape; however, the result is a row vector.

      type(mfArray) :: A_copy, B_copy

      real(kind=MF_DOUBLE), pointer :: A_ptr(:), B_ptr(:)

      real(kind=MF_DOUBLE), allocatable :: common_elements(:)
      integer :: iA, iB, ic, n, nA, nB, status
      real(kind=MF_DOUBLE) :: tol_0

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, B )

      if( mfIsEmpty(A) .or. mfIsEmpty(B) ) then
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE .or. B%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfIntersect", "E",                         &
                            "args must be real, dense mfArrays!" )
         go to 99
      end if

      if( present(tol) ) then
         tol_0 = tol
         if( tol_0 < 0.0d0 ) then
            call PrintMessage( "mfIntersect", "E",                      &
                               "'tol' must be a positive real number!" )
            go to 99
         end if
      else
         tol_0 = 0.0d0
      end if

      ! By using mfUnique without options, the result is sorted
      call msAssign( A_copy, mfUnique( mfReshape(A,1,MF_NO_ARG), tol=tol_0 ) )
      call msAssign( B_copy, mfUnique( mfReshape(B,1,MF_NO_ARG), tol=tol_0 ) )

      ! Quick access via pointer
      call msPointer( A_copy, A_ptr, no_crc=.true. )
      call msPointer( B_copy, B_ptr, no_crc=.true. )

      nA = size(A_copy)
      nB = size(B_copy)
      n = min( nA, nB )
      allocate( common_elements(n) )

      ! Travel both lists in parallel, using two indices iA and iB
      iA = 1
      iB = 1
      ic = 0
      do
         if( abs(A_ptr(iA)-B_ptr(iB)) <= tol_0 ) then
            ic = ic + 1
            common_elements(ic) = A_ptr(iA)
            if( iA == nA ) exit
            iA = iA + 1
            if( iB == nB ) exit
            iB = iB + 1
         else if( A_ptr(iA) < B_ptr(iB) ) then
            if( iA == nA ) exit
            iA = iA + 1
         else ! A_ptr(iA) > B_ptr(iB)
            if( iB == nB ) exit
            iB = iB + 1
         end if
      end do

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, ic ]
      allocate( out%double(out%shape(1),out%shape(2)) )
      out%double(1,:) = common_elements(1:ic)

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, b%units, status )
         if( status /= 0 ) then
            call PrintMessage( "mfIntersect", "E",                      &
                               "the physical dimensions of the two mfArray's",&
                               "are not consistent!" )
            go to 99
         end if
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, B )
      call msAutoRelease( A, B )

      call msFreePointer( A_copy, A_ptr )
      call msFreePointer( B_copy, B_ptr )
      call msRelease( A_copy, B_copy )

#endif
   end function mfIntersect
!_______________________________________________________________________
!
   function mfUnion( A, B, tol ) result( out )

      type(mfArray) :: A, B
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! same as in Matlab
      !
      ! returns all elements of 'A' and 'B', in only one instance.
      !
      ! 'A' and 'B' may have any shape; however, the result is a row vector.

      integer :: status
      real(kind=MF_DOUBLE) :: tol_0

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, B )

      if( mfIsEmpty(A) ) then
         out = B
         go to 98
      end if

      if( mfIsEmpty(B) ) then
         out = A
         go to 98
      end if

      if( A%data_type /= MF_DT_DBLE .or. B%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfUnion", "E",                             &
                            "args must be real, dense mfArrays!" )
         go to 99
      end if

      if( present(tol) ) then
         tol_0 = tol
         if( tol_0 < 0.0d0 ) then
            call PrintMessage( "mfUnion", "E",                          &
                               "'tol' must be a positive real number!" )
            go to 99
         end if
      else
         tol_0 = 0.0d0
      end if

      ! By using mfUnique without options, the result is sorted
      out = mfUnique( mfReshape(A,1,MF_NO_ARG) .hc.                     &
                      mfReshape(B,1,MF_NO_ARG), tol=tol_0 )

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, b%units, status )
         if( status /= 0 ) then
            call PrintMessage( "mfUnion", "E",                          &
                               "the physical dimensions of the two mfArray's",&
                               "are not consistent!" )
            go to 99
         end if
         out%units(:) = A%units(:)
      end if

 98   continue

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, B )
      call msAutoRelease( A, B )

#endif
   end function mfUnion
!_______________________________________________________________________
!
   function mfSmooth( v, span ) result( out )

      type(mfArray)                      :: v
      integer,      intent(in), optional :: span
      type(mfArray)                      :: out
      !------ API end ------

#ifdef _DEVLP
      ! same as in Matlab
      !
      ! Smooth the vector 'v' by using an moving average method.
      ! The 'span' integer' specify the total (odd) number of points
      ! used; by default, it is 5.

      integer :: dim, p, n, i, j, k

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( v )

      if( mfIsEmpty(v) ) then
         go to 99
      end if

      if( .not. mfIsReal(v) ) then
         call PrintMessage( "mfSmooth", "E",                            &
                            "'v' must be a real vector!" )
         go to 99
      end if

      if( v%shape(1) == 1 ) then
         dim = 2
      else if( v%shape(2) == 1 ) then
         dim = 1
      else
         call PrintMessage( "mfSmooth", "E",                            &
                            "'v' must be a real vector!" )
         go to 99
      end if
      n = size(v%double,dim)

      if( present(span) ) then
         if( span == 0 ) then
            ! nothing to do
            go to 99
         else if( span < 0 ) then
            call PrintMessage( "mfSmooth", "E",                         &
                               "'span' cannot be negative!" )
            go to 99
         end if
         p = (span - 1)/2
         if( 2*p+1 /= span ) then
            call PrintMessage( "mfSmooth", "E",                         &
                               "'span' must be odd!" )
            go to 99
         end if
      else
         p = 2 ! so that span = 2 + 1 + 2 = 5
      end if

      if( 2*p+1 > n ) then
         call PrintMessage( "mfSmooth", "W",                            &
                            "'span' is greater than the size of the vector!", &
                            "Numerical values are going to be smoothed, but not", &
                            "with the expected strength." )
         ! we have to decrease the value of 'p', else some indices below
         ! will become out-of-range!
         if( mod(n,2) == 0 ) then
            p = (n-2)/2
         else
            p = (n-1)/2
         end if
      end if

      ! out is a real mfArray
      out%data_type = MF_DT_DBLE
      out%shape = v%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( dim == 1 ) then
         if( p == 1 ) then
            out%double(1,1) = v%double(1,1)
            do i = 2, n-1
               out%double(i,1) = ( v%double(i-1,1) + v%double(i,1) + v%double(i+1,1) ) / 3.0d0
            end do
            out%double(n,1) = v%double(n,1)
         else
            do i = 1, p
              out%double(i,1) = smooth( v%double(:,1), i, i-1 );
            end do
            do i = 1+p, N-p
              out%double(i,1) = smooth( v%double(:,1), i, p );
            end do
            do i = 1, p
              out%double(N+1-i,1) = smooth( v%double(:,1), N+1-i, i-1 );
            end do
         end if
      else ! dim == 2
         if( p == 1 ) then
            out%double(1,1) = v%double(1,1)
            do i = 2, n-1
               out%double(1,i) = ( v%double(1,i-1) + v%double(1,i) + v%double(1,i+1) ) / 3.0d0
            end do
            out%double(1,n) = v%double(1,n)
         else
            do i = 1, p
              out%double(1,i) = smooth( v%double(1,:), i, i-1 );
            end do
            do i = 1+p, N-p
              out%double(1,i) = smooth( v%double(1,:), i, p );
            end do
            do i = 1, p
              out%double(1,N+1-i) = smooth( v%double(1,:), N+1-i, i-1 );
            end do
         end if
      end if

      if( mf_phys_units ) then
         out%units(:) = v%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( v )
      call msAutoRelease( v )

   contains

      function smooth( y, i, N ) result( out )
         real(kind=MF_DOUBLE), intent(in) :: y(:)
         integer,              intent(in) :: i, N
         real(kind=MF_DOUBLE) :: out
         out = sum( y(i-N:i+N) ) / (2*N+1)
      end

#endif
   end function mfSmooth
!_______________________________________________________________________
!
   function mfXCorr( v, maxlag, scale ) result( out )

      type(mfArray)                          :: v
      integer,          intent(in), optional :: maxlag
      character(len=*), intent(in), optional :: scale
      type(mfArray)                 :: out
      !------ API end ------

      ! Computes the autocorrelation of the (real) vector v.
      !
      ! Output is a vector of same shape, with the maximum value
      ! located at its center (it has an odd number of elements, and
      ! is symmetric w.r.t. its center).

#ifdef _DEVLP
      integer :: i, k, dim, N, NN, i2
      logical :: normalized
      real(kind=MF_DOUBLE) :: max_val

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( v )

      if( mfIsEmpty(v) ) then
         go to 99
      end if

      if( mfIsScalar(v) ) then
         call PrintMessage( "mfXCorr", "E",                             &
                            "'v' cannot be a scalar!" )
         go to 99
      end if

      if( v%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfXCorr", "E",                             &
                            "'v' must have real values!" )
         go to 99
      end if

      out%data_type = v%data_type

      if( v%shape(1) == 1 ) then
         dim = 2
      else if( v%shape(2) == 1 ) then
         dim = 1
      else
         call PrintMessage( "mfXCorr", "E",                             &
                            "'v' must be a vector!" )
         go to 99
      end if

      N = v%shape(dim)

      if( present(maxlag) ) then
         if( maxlag < 0 ) then
            call PrintMessage( "mfXCorr", "W",                          &
                               "'maxlag' must be >= 0!",                &
                               "(it has been set to N-1)" )
            i2 = 0
         else if( maxlag > N - 1 ) then
            call PrintMessage( "mfXCorr", "W",                          &
                               "'maxlag' must be <= N-1!",              &
                               "(it has been set to N-1)" )
            i2 = N - 1
         else
            i2 = maxlag
         end if
      else
         i2 = N - 1
      end if

      NN = 2*i2 + 1

      if( present(scale) ) then
         if( to_lower(scale) == "normalized" ) then
            normalized = .true.
         else
         end if
      else
         normalized = .false.
      end if

      if( normalized ) then
         ! compute the max value
         max_val = sum(v%double**2)
      else
         max_val = 1.0d0
      end if

      if( dim == 1 ) then
         out%shape(1) = NN
         out%shape(2) = 1
         allocate( out%double(NN,1) )
         do i = 1, NN
            out%double(i,1) = 0.0d0
            do k = max( 1, 2 - i + i2 ),                                &
                   min( N, 1+N - i + i2 )
               out%double(i,1) = out%double(i,1)                        &
                               + v%double(k,1)*v%double(k+i-i2-1,1)/max_val
            end do
         end do
      else ! dim = 2
         out%shape(1) = 1
         out%shape(2) = NN
         allocate( out%double(1,NN) )
         do i = 1, NN
            out%double(1,i) = 0.0d0
            do k = max( 1, 2 - i + i2 ),                                &
                   min( N, 1+N - i + i2 )
               out%double(1,i) = out%double(1,i)                        &
                               + v%double(1,k)*v%double(1,k+i-i2-1)/max_val
            end do
         end do
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = v%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( v )
      call msAutoRelease( v )

#endif
   end function mfXCorr
!_______________________________________________________________________
!
   function mfXCorr2( A, maxlag_r, maxlag_c, scale ) result( out )

      type(mfArray)                          :: A
      integer,          intent(in), optional :: maxlag_r, maxlag_c
      character(len=*), intent(in), optional :: scale
      type(mfArray)                 :: out
      !------ API end ------

      ! Computes the autocorrelation of the (real) matrix A.
      !
      ! Output is a matrix with the maximum value located nearly its
      ! center.

#ifdef _DEVLP
      integer :: i, j, k, l, dim, M, N, MM, NN, i2, j2
      logical :: normalized
      real(kind=MF_DOUBLE) :: max_val

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsScalar(A) ) then
         call PrintMessage( "mfXCorr2", "E",                            &
                            "'A' cannot be a scalar!" )
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfXCorr2", "E",                            &
                            "'A' must have real values!" )
         go to 99
      end if

      out%data_type = A%data_type

      if( A%shape(1) == 1 .or. A%shape(2) == 1 ) then
         call PrintMessage( "mfXCorr2", "E",                            &
                            "'A' cannot be a vector!",                  &
                            "(please use mfXCorr instead)" )
         go to 99
      end if

      M = A%shape(1)
      N = A%shape(2)

      if( present(maxlag_r) ) then
         if( maxlag_r < 0 ) then
            call PrintMessage( "mfXCorr2", "W",                         &
                               "'maxlag_r' must be >= 0!",              &
                               "(it has been set to M-1)" )
            i2 = 0
         else if( maxlag_r > M - 1 ) then
            call PrintMessage( "mfXCorr2", "W",                         &
                               "'maxlag_r' must be <= M-1!",            &
                               "(it has been set to M-1)" )
            i2 = M - 1
         else
            i2 = maxlag_r
         end if
      else
         i2 = M - 1
      end if

      MM = 2*i2 + 1

      if( present(maxlag_c) ) then
         if( maxlag_c < 0 ) then
            call PrintMessage( "mfXCorr2", "W",                         &
                               "'maxlag_c' must be >= 0!",              &
                               "(it has been set to N-1)" )
            j2 = 0
         else if( maxlag_c > N - 1 ) then
            call PrintMessage( "mfXCorr2", "W",                         &
                               "'maxlag_c' must be <= N-1!",            &
                               "(it has been set to N-1)" )
            j2 = N - 1
         else
            j2 = maxlag_c
         end if
      else
         j2 = N - 1
      end if

      NN = 2*j2 + 1

      if( present(scale) ) then
         if( to_lower(scale) == "normalized" ) then
            normalized = .true.
         else
         end if
      else
         normalized = .false.
      end if

      if( normalized ) then
         ! compute the max value
         max_val = sum(A%double**2)
      else
         max_val = 1.0d0
      end if

      out%shape(1) = MM
      out%shape(2) = NN
      allocate( out%double(MM,NN) )
      do i = 1, MM
      do j = 1, NN
         out%double(i,j) = 0.0d0
         do k = max( 1, 2 - i + i2 ),                                   &
                min( M, 1+M - i + i2 )
         do l = max( 1, 2 - j + j2 ),                                   &
                min( N, 1+N - j + j2 )
            out%double(i,j) = out%double(i,j)                           &
                  + A%double(k,l)*A%double(k+i-i2-1,l+j-j2-1)/max_val
         end do
         end do
      end do
      end do

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfXCorr2
!_______________________________________________________________________
!
end module mod_datafun
