/* This wrapper has been created by É. Canot, to call the functions
 * of JC_VORONOI from Fortran.
 * É. Canot -- IPR/CNRS -- 2024-10-06
 */

#include <stdio.h>
#include <stdlib.h>

// use 'double' real-type

#define JC_VORONOI_IMPLEMENTATION
#define JCV_REAL_TYPE double
#define JCV_REAL_TYPE_EPSILON 2.23e-16
#define JCV_ATAN2 atan2
#define JCV_SQRT sqrt
#define JCV_PI 3.1415926535897932
#define JCV_FLT_MAX 1.7976931348623157e+308

#define MAX_INT 2147483647

#include "jc_voronoi_EC.h"

/* global variables because, when called from Fortran, we must have
   two successive calls */
jcv_diagram diagram;

int find_vert_ind( int site, int neighbor_1, int neighbor_2 );

/* Fortran interface */
#ifdef UNDERSCORE
#define JCV_VORONOI_PREP jcv_voronoi_prep_
#else
#define JCV_VORONOI_PREP jcv_voronoi_prep
#endif
void JCV_VORONOI_PREP( double* x_min, double* x_max,
                       double* y_min, double* y_max,
                       int* nb_pts, double* x_pts, double* y_pts,
                       int* nb_edges, int* nb_neighbors,
                       int* nb_vertices, int* perm_sites ) {

  // Just for counting the number of edges, neighbors and unique vertices

  int i, site, last_site;
  jcv_rect bounding_box = { { *x_min, *y_min }, { *x_max, *y_max } };
  jcv_point points[*nb_pts];
  jcv_site* sites;
  jcv_graphedge *graph_edge, *last_edge;

  memset(&diagram, 0, sizeof(jcv_diagram));

  for( i=0; i<*nb_pts; i++ ) {
    points[i].x = x_pts[i];
    points[i].y = y_pts[i];
  }

  jcv_diagram_generate( *nb_pts, (const jcv_point *)points, &bounding_box,
                        0, &diagram );

  sites = jcv_diagram_get_sites(&diagram);

  /* Retrieve the permutation sites (input points are travelled from
   * bottom to top). */
  perm_sites[0] = sites[0].index+1;
  for( i=0; i<diagram.numsites; i++ ) {
    perm_sites[i] = sites[i].index + 1; // Fortran indices begin from 1
  }

  // define the internal num for each site
  for( i=0; i<diagram.numsites; i++ ) {
    sites[i].EC_num = i;
  }

  // define the last neighbor site in each site (EC_last_neig)
//printf("\n--- begin of test #1 ---\n");
  for( i=0; i<diagram.numsites; i++ ) {
//printf("i: %d\n",i);
    graph_edge = sites[i].edges;
    while (graph_edge) {
      last_edge = graph_edge;
//if( last_edge->neighbor ) {
//  printf("neighbor: %d\n",last_edge->neighbor->EC_num);
//    } else {
//  printf("no neighbor\n");
//}
      graph_edge = graph_edge->next;
    }
    if( last_edge->neighbor ) {
      last_site = last_edge->neighbor->EC_num;
    } else {
      last_site = MAX_INT;
    }
    sites[i].EC_last_neig = last_site;
  }
//printf("--- end of test #1 ---\n\n");

  *nb_edges = 0;
  *nb_neighbors = 0;
  *nb_vertices = 0;
//printf("\n--- begin of test #2 ---\n");
  for( i=0; i<diagram.numsites; i++ ) {
//printf("i: %d\n",i);
    graph_edge = sites[i].edges;
    last_site = sites[i].EC_last_neig;
//printf("lastsite: %d\n",last_site);
    while (graph_edge) {
      *nb_edges = *nb_edges + 1;
      if( graph_edge->neighbor ) {
        // the current edge has a neighbor
        *nb_neighbors = *nb_neighbors + 1;
        site = graph_edge->neighbor->EC_num;
      } else {
        site = MAX_INT;
      }
//printf("lastsite: %d, site: %d\n",last_site, site);
      if( (last_site > i) & (site > i) ) {
        *nb_vertices = *nb_vertices + 1;
      }
      graph_edge = graph_edge->next;
      last_site = site;
    }
  }
//printf("--- end of test #2 ---\n\n");

}

/* Fortran interface */
#ifdef UNDERSCORE
#define JCV_VORONOI_GET_EDGES jcv_voronoi_get_edges_
#else
#define JCV_VORONOI_GET_EDGES jcv_voronoi_get_edges
#endif
/* Returns 0 if all is correct.
 * Returns -1 in case of internal error (see 'find_vert_ind') */
int JCV_VORONOI_GET_EDGES( int* ptr_edges, int* list_edges,
                           double* x_vert, double* y_vert ) {

  int i, j, k, site, last_site, vert_ind;
  jcv_site* sites;
  jcv_graphedge *graph_edge;

  sites = jcv_diagram_get_sites(&diagram);

  j = 0; k = 0;
  ptr_edges[0] = 1; // Fortran indices begin from 1
//printf("\n--- begin of test #3 ---\n");
  for( i=0; i<diagram.numsites; i++ ) {
//printf("i: %d\n",i);
    graph_edge = sites[i].edges;
    last_site = sites[i].EC_last_neig;
//printf("lastsite: %d\n",last_site);
    while( graph_edge ) {
      if( graph_edge->neighbor ) {
        // the current edge has a neighbor
        site = graph_edge->neighbor->EC_num;
      } else {
        site = MAX_INT;
      }
// if( i == 2 ) {
//   printf("lastsite: %d, site: %d\n",last_site, site);
// }
//printf("lastsite: %d, site: %d\n",last_site, site);
      if( site < i ) {
        /* Loop over edges of 'site' to find 'vert_ind'
        * (vertex index common to sites 'last_site' and 'i')
        */
        vert_ind = find_vert_ind( site, i, last_site );
        if( vert_ind < 0 ) return -1;
// if( i == 2 ) {
//   printf("found: vert_ind = %d\n",vert_ind);
// }
        list_edges[k] = vert_ind + 1; // Fortran indices begin from 1
        graph_edge->EC_vert_ind = vert_ind;
        k++;
      } else if( last_site < i ) {
        /* Loop over edges of 'last_site' to find 'vert_ind'
        * (vertex index common to sites 'site' and 'i')
        */
        vert_ind = find_vert_ind( last_site, site, i );
        if( vert_ind < 0 ) return -1;
        list_edges[k] = vert_ind + 1; // Fortran indices begin from 1
        graph_edge->EC_vert_ind = vert_ind;
        k++;
      } else {
        x_vert[j] = graph_edge->pos[0].x;
        y_vert[j] = graph_edge->pos[0].y;
//printf("  vertex %d recorded (unique)\n",j);
        list_edges[k] = j + 1; // Fortran indices begin from 1
        graph_edge->EC_vert_ind = j;
        j++; k++;
      }
      graph_edge = graph_edge->next;
      last_site = site;
    }

    if( i+1 < diagram.numsites ) {
      ptr_edges[i+1] = k + 1; // Fortran indices begin from 1
    }

  }
//printf("--- end of test #3 ---\n\n");

  return 0;
}

/* Loop over edges of 'site' to find 'vert_ind'
 * (vertex index common to sites 'neighbor_1' and 'neighbor_2')
 * Returns -1 if not found (internal error)
 */
int find_vert_ind( int site, int neighbor_1, int neighbor_2 ) {

  int last_site, s;
  jcv_site* sites;
  jcv_graphedge *graph_edge, *last_edge;

  sites = jcv_diagram_get_sites(&diagram);
//printf("    find_vert_ind: loop over edge of site %d\n",site);
//printf("    in order to find neighbors %d and %d\n",neighbor_1,neighbor_2);

// if( (site==1) & (neighbor_1==2) & (neighbor_2==0) ) {
//   printf("    find_vert_ind: loop over edge of site %d\n",site);
//   printf("    in order to find neighbors %d and %d ...\n",neighbor_1,neighbor_2);
// }
  graph_edge = sites[site].edges;
  last_site = sites[site].EC_last_neig;

  // find just last_edge
  while (graph_edge) {
    last_edge = graph_edge;
    graph_edge = graph_edge->next;
  }

  graph_edge = sites[site].edges;
  while( graph_edge ) {
    if( graph_edge->neighbor ) {
      s = graph_edge->neighbor->EC_num;
    } else {
      s = MAX_INT;
    }
// if( (site==1) & (neighbor_1==2) & (neighbor_2==0) ) {
//   printf("      examining %d and %d\n",last_site,s);
// }
    if( ((last_site == neighbor_1) & (s == neighbor_2)) ) {
      return graph_edge->EC_vert_ind;
    }
    last_site = s;
    last_edge = graph_edge;
    graph_edge = graph_edge->next;
  }

  // INTERNAL ERROR -- vertex index not found!
  return -1;
}

/* Fortran interface */
#ifdef UNDERSCORE
#define JCV_VORONOI_GET_NEIGHBORS jcv_voronoi_get_neighbors_
#else
#define JCV_VORONOI_GET_NEIGHBORS jcv_voronoi_get_neighbors
#endif
void JCV_VORONOI_GET_NEIGHBORS( int* ptr_neighbors, int* neighbors ) {

  int i, j;
  const jcv_site* sites;
  jcv_graphedge* graph_edge;

  sites = jcv_diagram_get_sites(&diagram);

  j = 0;
  ptr_neighbors[0] = 1; // Fortran indices begin from 1
  for( i=0; i<diagram.numsites; i++ ) {
    graph_edge = sites[i].edges;
//printf("site: %d, original index: %d\n", i+1, sites[i].index+1);
    while( graph_edge ) {
      if( graph_edge->neighbor ) {
        neighbors[j] = graph_edge->neighbor->index + 1; // Fortran indices begin from 1
        j++;
      }
      graph_edge = graph_edge->next;
    }
    if( i+1 < diagram.numsites ) {
      ptr_neighbors[i+1] = j+1; // Fortran indices begin from 1
    }
  }

}

/* Fortran interface */
#ifdef UNDERSCORE
#define JCV_VORONOI_FREE jcv_voronoi_free_
#else
#define JCV_VORONOI_FREE jcv_voronoi_free
#endif
void JCV_VORONOI_FREE( ) {

  jcv_diagram_free(&diagram);

}
