/*
 * UMFPACK: Fortran 90 wrapper
 *
 * modified by É. Canot -- IPR/CNRS
 * (adapted from umf4_f77wrapper and umf4_f77zwrapper)
 *
 * Copyright É. Canot 2003-2025 -- IPR/CNRS
 *
 * March 23, 2017
 */

#include "SuiteSparse_config.h"

/* ========================================================================== */
/* === umf4_f90wrapper (adapted from umf4_f77wrapper and umf4_f77zwrapper) == */
/* ========================================================================== */

/* -------------------------------------------------------------------------- */
/* UMFPACK Version 5.5, Copyright (c) 1995-2012 by Timothy A. Davis.  CISE,   */
/* Univ. of Florida.  All Rights Reserved.  See ../Doc/License for License.   */
/* web: http://www.cise.ufl.edu/research/sparse/umfpack                       */
/* -------------------------------------------------------------------------- */

#include "umfpack.h"
#include <ctype.h>
#include <stdio.h>
#ifdef NULL
#undef NULL
#endif
#define NULL 0

/* -------------------------------------------------------------------------- */
/* integer type: SuiteSparse_long                                                      */
/* -------------------------------------------------------------------------- */

/*
 * Using SuiteSparse_long is the only way to have the same definition both
 * for Windows and Linux.
 * (that is: 32bit integer on 32bit OS, and 64bit integer on 64bit OS)
 */

/* ========================================================================== */
/* === added "_d" for DOUBLE case and "_z" for DOUBLE COMPLEX =============== */
/* ========================================================================== */

/* ========================================================================== */
/*                       R E A L     V A L U E S                              */
/* ========================================================================== */

/* -------------------------------------------------------------------------- */
/* umf4def: set default control parameters */
/* -------------------------------------------------------------------------- */

/* call umf4def_d (control) */

#ifdef UNDERSCORE
#define umf4def_d umf4def_d_
#endif
void umf4def_d (double Control [UMFPACK_CONTROL])
{
   umfpack_dl_defaults (Control);
}

/* -------------------------------------------------------------------------- */
/* umf4sym: pre-ordering and symbolic factorization                           */
/* -------------------------------------------------------------------------- */

/* call umf4sym_d (m, n, Ap, Ai, Ax, symbolic, control, info) */

#ifdef UNDERSCORE
#define umf4sym_d umf4sym_d_
#endif
void umf4sym_d (int *m, int *n, int Ap [ ], int Ai [ ],
                double Ax [ ], void **Symbolic,
                double Control [UMFPACK_CONTROL], double Info [UMFPACK_INFO])
{
   SuiteSparse_long i, nnz, *AAp, *AAi;

   AAp = malloc( (*n+1)*sizeof(SuiteSparse_long) );
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i]; /* shift done in the calling Fortran unit */
   }

   nnz = Ap[*n];
   AAi = malloc( (nnz)*sizeof(SuiteSparse_long) );
   for( i=0; i<nnz; i++ ){
      AAi[i] = Ai[i]; /* shift done in the calling Fortran unit */
   }

   (void) umfpack_dl_symbolic ( (SuiteSparse_long)*m, (SuiteSparse_long)*n, AAp, AAi, Ax,
                                Symbolic, Control, Info );

   free(AAp);
   free(AAi);
}

/* -------------------------------------------------------------------------- */
/* umf4qsym: symbolic factorization with given ordering                       */
/*           Qinit is a given permutation vector of length n                  */
/* -------------------------------------------------------------------------- */

/* call umf4qsym_d (m, n, Ap, Ai, Ax, Qinit, symbolic, control, info) */

#ifdef UNDERSCORE
#define umf4qsym_d umf4qsym_d_
#endif
void umf4qsym_d (int *m, int *n, int Ap [ ], int Ai [ ],
                 double Ax [ ], int Qinit [ ], void **Symbolic,
                 double Control [UMFPACK_CONTROL], double Info [UMFPACK_INFO])
{
   SuiteSparse_long i, nnz, *AAp, *AAi, *QQ;

   AAp = malloc( (*n+1)*sizeof(SuiteSparse_long) );
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i]; /* shift done in the calling Fortran unit */
   }

   nnz = Ap[*n];
   AAi = malloc( (nnz)*sizeof(SuiteSparse_long) );
   for( i=0; i<nnz; i++ ){
      AAi[i] = Ai[i]; /* shift done in the calling Fortran unit */
   }

   QQ = malloc( (*n)*sizeof(SuiteSparse_long) );
   for( i=0; i<*n; i++ ){
      QQ[i] = Qinit[i]; /* shift done in the calling Fortran unit */
   }

   (void) umfpack_dl_qsymbolic ( (SuiteSparse_long)*m, (SuiteSparse_long)*n, AAp, AAi, Ax, QQ,
                                 Symbolic, Control, Info );

   free(AAp);
   free(AAi);
   free(QQ);
}

/* -------------------------------------------------------------------------- */
/* umf4num: numeric factorization                                             */
/* -------------------------------------------------------------------------- */

/* call umf4num_d (n, Ap, Ai, Ax, symbolic, numeric, control, info) */

#ifdef UNDERSCORE
#define umf4num_d umf4num_d_
#endif
void umf4num_d (int *n, int Ap [ ], int Ai [ ], double Ax [ ],
                void **Symbolic, void **Numeric,
                double Control [UMFPACK_CONTROL], double Info [UMFPACK_INFO])
{
   SuiteSparse_long i, nnz, *AAp, *AAi;

   AAp = malloc( (*n+1)*sizeof(SuiteSparse_long) );
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i]; /* shift done in the calling Fortran unit */
   }

   nnz = Ap[*n];
   AAi = malloc( (nnz)*sizeof(SuiteSparse_long) );
   for( i=0; i<nnz; i++ ){
      AAi[i] = Ai[i]; /* shift done in the calling Fortran unit */
   }

   (void) umfpack_dl_numeric (AAp, AAi, Ax, *Symbolic, Numeric, Control,
                              Info);

   free(AAp);
   free(AAi);
}

/* -------------------------------------------------------------------------- */
/* umf4sol: solve a linear system without iterative refinement                */
/*          many variants with first arg: cf. umfpack.h                       */
/* -------------------------------------------------------------------------- */

/* call umf4sol_d (sys, x, b, numeric, control, info) */

#ifdef UNDERSCORE
#define umf4sol_d umf4sol_d_
#endif
void umf4sol_d (int *sys, double x [ ], double b [ ], void **Numeric,
                double Control [UMFPACK_CONTROL], double Info [UMFPACK_INFO])
{
   Control [UMFPACK_IRSTEP] = 0;

   (void) umfpack_dl_solve ( (SuiteSparse_long)*sys, (SuiteSparse_long *) NULL, (SuiteSparse_long *) NULL, (double *) NULL,
                             x, b, *Numeric, Control, Info );
}

/* -------------------------------------------------------------------------- */
/* umf4wsol: solve a linear system without iterative refinement               */
/*           workspace must be allocated before, and so allow                 */
/*             to multiple solve (via many successice calls)                  */
/*           many variants with first arg: cf. umfpack.h                      */
/* -------------------------------------------------------------------------- */

/* call umf4wsol_d (sys, x, b, numeric, control, info, Wi, W) */

#ifdef UNDERSCORE
#define umf4wsol_d umf4wsol_d_
#endif
void umf4wsol_d (int *sys, double x [ ], double b [ ], void **Numeric,
                 double Control [UMFPACK_CONTROL], double Info [UMFPACK_INFO],
                 SuiteSparse_long Wi [ ], double W [ ])
{
   Control [UMFPACK_IRSTEP] = 0;

   (void) umfpack_dl_wsolve ( (SuiteSparse_long)*sys, (SuiteSparse_long *) NULL, (SuiteSparse_long *) NULL, (double *) NULL,
                              x, b, *Numeric, Control, Info, Wi, W);
}

/* -------------------------------------------------------------------------- */
/* umf4solr: solve a linear system with iterative refinement                  */
/*           many variants with first arg: cf. umfpack.h                      */
/* -------------------------------------------------------------------------- */

/* call umf4solr_d (sys, Ap, Ai, Ax, x, b, numeric, control, info) */

#ifdef UNDERSCORE
#define umf4solr_d umf4solr_d_
#endif
void umf4solr_d (int *sys, int Ap [ ], int Ai [ ], double Ax [ ],
                 double x [ ], double b [ ], void **Numeric,
                 double Control [UMFPACK_CONTROL], double Info [UMFPACK_INFO])
{
   SuiteSparse_long i, n, nnz, *AAp, *AAi;

   n = Info[UMFPACK_NCOL];
   AAp = malloc( (n+1)*sizeof(SuiteSparse_long) );
   for( i=0; i<n+1; i++ ){
      AAp[i] = Ap[i]; /* shift done in the calling Fortran unit */
   }

   nnz = Ap[n];
   AAi = malloc( (nnz)*sizeof(SuiteSparse_long) );
   for( i=0; i<nnz; i++ ){
      AAi[i] = Ai[i]; /* shift done in the calling Fortran unit */
   }

   (void) umfpack_dl_solve ( (SuiteSparse_long)*sys, AAp, AAi, Ax, x, b, *Numeric, Control,
                             Info);

   free(AAp);
   free(AAi);
}

/* -------------------------------------------------------------------------- */
/* umf4fsym: free the Symbolic object                                         */
/* -------------------------------------------------------------------------- */

/* call umf4fsym_d (symbolic) */

#ifdef UNDERSCORE
#define umf4fsym_d umf4fsym_d_
#endif
void umf4fsym_d (void **Symbolic)
{
   umfpack_dl_free_symbolic (Symbolic);
}

/* -------------------------------------------------------------------------- */
/* umf4fnum: free the Numeric object                                          */
/* -------------------------------------------------------------------------- */

/* call umf4fnum_d (numeric) */

#ifdef UNDERSCORE
#define umf4fnum_d umf4fnum_d_
#endif
void umf4fnum_d (void **Numeric)
{
   umfpack_dl_free_numeric (Numeric);
}

/* ========================================================================== */
/* === other interfaces, added by É. Canot ================================== */
/* ========================================================================== */

/* -------------------------------------------------------------------------- */
/* umf4nzudiag: get the NNZ of U_diag                                         */
/* -------------------------------------------------------------------------- */

/* call umf4nzudiag_d (nz_udiag, numeric, status) */

#ifdef UNDERSCORE
#define umf4nzudiag_d umf4nzudiag_d_
#endif
void umf4nzudiag_d (int *nz_udiag, void **Numeric, int *status)
{
   SuiteSparse_long lnz, unz, n_row, n_col, nnz_udiag;

   *status = (int)umfpack_dl_get_lunz ( &lnz, &unz, &n_row, &n_col,
                                        &nnz_udiag, *Numeric );
   *nz_udiag = (int)nnz_udiag;
}

/* -------------------------------------------------------------------------- */
/* umf4getudiag: get the diagonal of U                                        */
/* -------------------------------------------------------------------------- */

/* call umf4getudiag_d (udiag, numeric, status) */

#ifdef UNDERSCORE
#define umf4getudiag_d umf4getudiag_d_
#endif
void umf4getudiag_d (double udiag [ ], void **Numeric, int *status)
{
   *status = (int)umfpack_dl_get_numeric (
                      (SuiteSparse_long *) NULL, (SuiteSparse_long *) NULL, (double *) NULL,
                      (SuiteSparse_long *) NULL, (SuiteSparse_long *) NULL, (double *) NULL,
                      (SuiteSparse_long *) NULL, (SuiteSparse_long *) NULL,
                      udiag,
                      (SuiteSparse_long *) NULL, (double *) NULL,
                      *Numeric );
}

/* -------------------------------------------------------------------------- */
/* umf4nzlu: get the NNZ of L and U factors                                   */
/* -------------------------------------------------------------------------- */

/* call umf4nzlu_d (lnz, unz, numeric, status) */

#ifdef UNDERSCORE
#define umf4nzlu_d umf4nzlu_d_
#endif
void umf4nzlu_d (int *lnz, int *unz, void **Numeric, int *status)
{
   SuiteSparse_long n_row, n_col, nz_udiag, lnnz, unnz;

   *status = (int)umfpack_dl_get_lunz ( &lnnz, &unnz, &n_row, &n_col,
                                        &nz_udiag, *Numeric );
   *lnz = (int)lnnz;
   *unz = (int)unnz;
}

/* -------------------------------------------------------------------------- */
/* umf4getlu: get the L, U factors                                            */
/*            (CAUTION: L is returned in the CSR format                       */
/*              whereas U is returned in the CSC format)                      */
/* -------------------------------------------------------------------------- */

/* call umf4getlu_d ( m, n, lnz, unz,
 *                    lp, lj, lx, up, ui, ux, numeric, status)                */

#ifdef UNDERSCORE
#define umf4getlu_d umf4getlu_d_
#endif
void umf4getlu_d (int *m, int *n, int *lnz, int *unz,
                  int Lp[ ], int Lj[ ], double Lx[ ],
                  int Up[ ], int Ui[ ], double Ux[ ],
                  void **Numeric, int *status)
{
   SuiteSparse_long i, *LLp, *LLj, *UUp, *UUi;

   LLp = malloc( (*m+1)*sizeof(SuiteSparse_long) );
   LLj = malloc( (*lnz)*sizeof(SuiteSparse_long) );

   UUp = malloc( (*n+1)*sizeof(SuiteSparse_long) );
   UUi = malloc( (*unz)*sizeof(SuiteSparse_long) );

   *status = (int)umfpack_dl_get_numeric (
                      LLp, LLj, Lx,
                      UUp, UUi, Ux,
                      (SuiteSparse_long *) NULL, (SuiteSparse_long *) NULL,
                      (double *) NULL,
                      (SuiteSparse_long *) NULL, (double *) NULL,
                      *Numeric );

   for( i=0; i<*m+1; i++ ){
      Lp[i] = LLp[i]; /* shift has to be done in the calling Fortran unit */
   }
   for( i=0; i<*lnz; i++ ){
      Lj[i] = LLj[i]; /* shift has to be done in the calling Fortran unit */
   }

   for( i=0; i<*n+1; i++ ){
      Up[i] = UUp[i]; /* shift has to be done in the calling Fortran unit */
   }
   for( i=0; i<*unz; i++ ){
      Ui[i] = UUi[i]; /* shift has to be done in the calling Fortran unit */
   }


   free(LLp);
   free(LLj);
   free(UUp);
   free(UUi);
}

/* -------------------------------------------------------------------------- */
/* umf4getlupqr: get the L, U, P, Q, R factors                                */
/*            (CAUTION: L is returned in the CSR format                       */
/*              whereas U is returned in the CSC format)                      */
/* -------------------------------------------------------------------------- */

/* call umf4getlupqr_d ( m, n, lnz, unz,
 *                       lp, lj, lx, up, ui, ux, p, q, rs, numeric, status)   */

#ifdef UNDERSCORE
#define umf4getlupqr_d umf4getlupqr_d_
#endif
void umf4getlupqr_d (int *m, int *n, int *lnz, int *unz,
                     int Lp[ ], int Lj[ ], double Lx[ ],
                     int Up[ ], int Ui[ ], double Ux[ ],
                     int P[ ], int Q[ ], double Rs[ ],
                     void **Numeric, int *status)
{
   SuiteSparse_long i, *LLp, *LLj, *UUp, *UUi, *PP, *QQ;

   LLp = malloc( (*m+1)*sizeof(SuiteSparse_long) );
   LLj = malloc( (*lnz)*sizeof(SuiteSparse_long) );

   UUp = malloc( (*n+1)*sizeof(SuiteSparse_long) );
   UUi = malloc( (*unz)*sizeof(SuiteSparse_long) );

   PP  = malloc( (*m)*sizeof(SuiteSparse_long) );
   QQ  = malloc( (*n)*sizeof(SuiteSparse_long) );

   *status = (int)umfpack_dl_get_numeric (
                      LLp, LLj, Lx,
                      UUp, UUi, Ux,
                      PP, QQ,
                      (double *) NULL,
                      (SuiteSparse_long *) NULL, Rs,
                      *Numeric );

   for( i=0; i<*m+1; i++ ){
      Lp[i] = LLp[i]; /* shift has to be done in the calling Fortran unit */
   }
   for( i=0; i<*lnz; i++ ){
      Lj[i] = LLj[i]; /* shift has to be done in the calling Fortran unit */
   }

   for( i=0; i<*n+1; i++ ){
      Up[i] = UUp[i]; /* shift has to be done in the calling Fortran unit */
   }
   for( i=0; i<*unz; i++ ){
      Ui[i] = UUi[i]; /* shift has to be done in the calling Fortran unit */
   }

   for( i=0; i<*m; i++ ){
      P[i] = PP[i]; /* shift has to be done in the calling Fortran unit */
   }
   for( i=0; i<*n; i++ ){
      Q[i] = QQ[i]; /* shift has to be done in the calling Fortran unit */
   }

   free(LLp);
   free(LLj);
   free(UUp);
   free(UUi);
   free(PP);
   free(QQ);
}

/* ========================================================================== */
/*                    C O M P L E X     V A L U E S                           */
/* ========================================================================== */

/* -------------------------------------------------------------------------- */
/* umf4def: set default control parameters (cmplx)                            */
/* -------------------------------------------------------------------------- */

/* call umf4def_z (control) */

#ifdef UNDERSCORE
#define umf4def_z umf4def_z_
#endif
void umf4def_z (double Control [UMFPACK_CONTROL])
{
   umfpack_zl_defaults (Control);
}

/* -------------------------------------------------------------------------- */
/* umf4sym: pre-ordering and symbolic factorization (cmplx)                   */
/* -------------------------------------------------------------------------- */

/* call umf4sym_z (m, n, Ap, Ai, Ax, Az, symbolic, control, info) */

#ifdef UNDERSCORE
#define umf4sym_z umf4sym_z_
#endif
void umf4sym_z (int *m, int *n, int Ap [ ], int Ai [ ],
                double Ax [ ], double Az [ ], void **Symbolic,
                double Control [UMFPACK_CONTROL], double Info [UMFPACK_INFO])
{
   SuiteSparse_long i, nnz, *AAp, *AAi;

   AAp = malloc( (*n+1)*sizeof(SuiteSparse_long) );
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i]; /* shift done in the calling Fortran unit */
   }

   nnz = Ap[*n];
   AAi = malloc( (nnz)*sizeof(SuiteSparse_long) );
   for( i=0; i<nnz; i++ ){
      AAi[i] = Ai[i]; /* shift done in the calling Fortran unit */
   }

   (void) umfpack_zl_symbolic ( (SuiteSparse_long)*m, (SuiteSparse_long)*n, AAp, AAi, Ax, Az,
                                Symbolic, Control, Info);

   free(AAp);
   free(AAi);
}

/* -------------------------------------------------------------------------- */
/* umf4num: numeric factorization (cmplx)                                     */
/* -------------------------------------------------------------------------- */

/* call umf4num_z (n, Ap, Ai, Ax, Az, symbolic, numeric, control, info) */

#ifdef UNDERSCORE
#define umf4num_z umf4num_z_
#endif
void umf4num_z (int *n, int Ap [ ], int Ai [ ], double Ax [ ], double Az [ ],
                void **Symbolic, void **Numeric,
                double Control [UMFPACK_CONTROL], double Info [UMFPACK_INFO])
{
   SuiteSparse_long i, nnz, *AAp, *AAi;

   AAp = malloc( (*n+1)*sizeof(SuiteSparse_long) );
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i]; /* shift done in the calling Fortran unit */
   }

   nnz = Ap[*n];
   AAi = malloc( (nnz)*sizeof(SuiteSparse_long) );
   for( i=0; i<nnz; i++ ){
      AAi[i] = Ai[i]; /* shift done in the calling Fortran unit */
   }

   (void) umfpack_zl_numeric (AAp, AAi, Ax, Az, *Symbolic, Numeric,
                              Control, Info);

   free(AAp);
   free(AAi);
}

/* -------------------------------------------------------------------------- */
/* umf4sol: solve a linear system without iterative refinement (cmplx)        */
/*            many variants with first arg: cf. umfpack.h                     */
/* -------------------------------------------------------------------------- */

/* call umf4sol_z (sys, Xx, Xz, Bx, Bz, numeric, control, info) */

#ifdef UNDERSCORE
#define umf4sol_z umf4sol_z_
#endif
void umf4sol_z (int *sys, double Xx [ ], double Xz [ ],
                double Bx [ ], double Bz [ ], void **Numeric,
                double Control [UMFPACK_CONTROL], double Info [UMFPACK_INFO])
{
   Control [UMFPACK_IRSTEP] = 0;

   (void) umfpack_zl_solve ( (SuiteSparse_long)*sys, (SuiteSparse_long *) NULL, (SuiteSparse_long *) NULL,
                             (double *) NULL, (double *) NULL,
                             Xx, Xz, Bx, Bz, *Numeric, Control, Info);
}

/* -------------------------------------------------------------------------- */
/* umf4wsol: solve a linear system without iterative refinement (cmplx)       */
/*           workspace must be allocated before, and so allow                 */
/*             to multiple solve                                              */
/*           many variants with first arg: cf. umfpack.h                      */
/* -------------------------------------------------------------------------- */

/* call umf4wsol_z (sys, Xx, Xz, Bx, Bz, numeric, control, info, Wi, W) */

#ifdef UNDERSCORE
#define umf4wsol_z umf4wsol_z_
#endif
void umf4wsol_z (int *sys, double Xx [ ], double Xz [ ],
                 double Bx [ ], double Bz [ ], void **Numeric,
                 double Control [UMFPACK_CONTROL], double Info [UMFPACK_INFO],
                 SuiteSparse_long Wi [ ], double W [ ])
{
   Control [UMFPACK_IRSTEP] = 0;

   (void) umfpack_zl_wsolve ( (SuiteSparse_long)*sys, (SuiteSparse_long *) NULL, (SuiteSparse_long *) NULL,
                              (double *) NULL, (double *) NULL,
                              Xx, Xz, Bx, Bz, *Numeric, Control, Info, Wi, W);
}

/* -------------------------------------------------------------------------- */
/* umf4nzudiag: get the NNZ of U_diag (cmplx)                                 */
/* -------------------------------------------------------------------------- */

/* call umf4nzudiag_z (nz_udiag, numeric, status) */

#ifdef UNDERSCORE
#define umf4nzudiag_z umf4nzudiag_z_
#endif
void umf4nzudiag_z (int *nz_udiag, void **Numeric, int *status)
{
   SuiteSparse_long lnz, unz, n_row, n_col, nnz_udiag;

   *status = (int)umfpack_zl_get_lunz (&lnz, &unz, &n_row, &n_col,
                                       &nnz_udiag, *Numeric);
   *nz_udiag = (int)nnz_udiag;
}

/* -------------------------------------------------------------------------- */
/* umf4getudiag: get the diagonal of U (cmplx)                                */
/* -------------------------------------------------------------------------- */

/* call umf4getudiag_z (udiag, numeric, status) */

#ifdef UNDERSCORE
#define umf4getudiag_z umf4getudiag_z_
#endif
void umf4getudiag_z (double udiagr [ ], double udiagi [ ],
                     void **Numeric, int *status)
{
   *status = (int)umfpack_zl_get_numeric (
                (SuiteSparse_long *) NULL, (SuiteSparse_long *) NULL, (double *) NULL, (double *) NULL,
                (SuiteSparse_long *) NULL, (SuiteSparse_long *) NULL, (double *) NULL, (double *) NULL,
                (SuiteSparse_long *) NULL, (SuiteSparse_long *) NULL,
                udiagr, udiagi,
                (SuiteSparse_long *) NULL, (double *) NULL,
                *Numeric);
}


/* -------------------------------------------------------------------------- */
/* umf4nzlu: get the NNZ of L and U factors (cmplx)                           */
/* -------------------------------------------------------------------------- */

/* call umf4nzlu_z (lnz, unz, numeric, status) */

#ifdef UNDERSCORE
#define umf4nzlu_z umf4nzlu_z_
#endif
void umf4nzlu_z (int *lnz, int *unz, void **Numeric, int *status)
{
   SuiteSparse_long n_row, n_col, nz_udiag, lnnz, unnz;

   *status = (int)umfpack_zl_get_lunz ( &lnnz, &unnz, &n_row, &n_col,
                                        &nz_udiag, *Numeric);
   *lnz = (int)lnnz;
   *unz = (int)unnz;
}

/* -------------------------------------------------------------------------- */
/* umf4getlu: get the L, U factors (cmplx)                                    */
/*            (CAUTION: L is returned in the CSR format                       */
/*              whereas U is returned in the CSC format)                      */
/* -------------------------------------------------------------------------- */

/* call umf4getlu_z ( m, n, lnz, unz,
 *                    lp, lj, lr, lz, up, ui, ur, uz, numeric, status)        */

#ifdef UNDERSCORE
#define umf4getlu_z umf4getlu_z_
#endif
void umf4getlu_z (int *m, int *n, int *lnz, int *unz,
                  int Lp[ ], int Lj[ ], double Lr[ ], double Lz[ ],
                  int Up[ ], int Ui[ ], double Ur[ ], double Uz[ ],
                  void **Numeric, int *status)
{
   SuiteSparse_long i, *LLp, *LLj, *UUp, *UUi;

   LLp = malloc( (*m+1)*sizeof(SuiteSparse_long) );
   LLj = malloc( (*lnz)*sizeof(SuiteSparse_long) );

   UUp = malloc( (*n+1)*sizeof(SuiteSparse_long) );
   UUi = malloc( (*unz)*sizeof(SuiteSparse_long) );

   *status = (int)umfpack_zl_get_numeric (
                      LLp, LLj, Lr, Lz,
                      UUp, UUi, Ur, Uz,
                      (SuiteSparse_long *) NULL, (SuiteSparse_long *) NULL,
                      (double *) NULL, (double *) NULL,
                      (SuiteSparse_long *) NULL, (double *) NULL,
                      *Numeric);

   for( i=0; i<*m+1; i++ ){
      Lp[i] = LLp[i]; /* shift has to be done in the calling Fortran unit */
   }
   for( i=0; i<*lnz; i++ ){
      Lj[i] = LLj[i]; /* shift has to be done in the calling Fortran unit */
   }

   for( i=0; i<*n+1; i++ ){
      Up[i] = UUp[i]; /* shift has to be done in the calling Fortran unit */
   }
   for( i=0; i<*unz; i++ ){
      Ui[i] = UUi[i]; /* shift has to be done in the calling Fortran unit */
   }

   free(LLp);
   free(LLj);
   free(UUp);
   free(UUi);
}

/* -------------------------------------------------------------------------- */
/* umf4getlupqr: get the L, U, P, Q, R factors (cmplx)                        */
/*            (CAUTION: L is returned in the CSR format                       */
/*              whereas U is returned in the CSC format)                      */
/* -------------------------------------------------------------------------- */

/* call umf4getlupqr_z ( m, n, lnz, unz,
 *                       lp, lj, lr, lz, up, ui, ur, uz,
                         p, q, rs, numeric, status)                           */

#ifdef UNDERSCORE
#define umf4getlupqr_z umf4getlupqr_z_
#endif
void umf4getlupqr_z (int *m, int *n, int *lnz, int *unz,
                     int Lp[ ], int Lj[ ], double Lr[ ], double Lz[ ],
                     int Up[ ], int Ui[ ], double Ur[ ], double Uz[ ],
                     int P[ ], int Q[ ], double Rs[ ],
                     void **Numeric, int *status)
{
   SuiteSparse_long i, *LLp, *LLj, *UUp, *UUi, *PP, *QQ;

   LLp = malloc( (*m+1)*sizeof(SuiteSparse_long) );
   LLj = malloc( (*lnz)*sizeof(SuiteSparse_long) );

   UUp = malloc( (*n+1)*sizeof(SuiteSparse_long) );
   UUi = malloc( (*unz)*sizeof(SuiteSparse_long) );

   PP  = malloc( (*m)*sizeof(SuiteSparse_long) );
   QQ  = malloc( (*n)*sizeof(SuiteSparse_long) );

   *status = (int)umfpack_zl_get_numeric (
                LLp, LLj, Lr, Lz,
                UUp, UUi, Ur, Uz,
                PP, QQ,
                (double *) NULL, (double *) NULL,
                (SuiteSparse_long *) NULL, Rs,
                *Numeric);

   for( i=0; i<*m+1; i++ ){
      Lp[i] = LLp[i]; /* shift has to be done in the calling Fortran unit */
   }
   for( i=0; i<*lnz; i++ ){
      Lj[i] = LLj[i]; /* shift has to be done in the calling Fortran unit */
   }

   for( i=0; i<*n+1; i++ ){
      Up[i] = UUp[i]; /* shift has to be done in the calling Fortran unit */
   }
   for( i=0; i<*unz; i++ ){
      Ui[i] = UUi[i]; /* shift has to be done in the calling Fortran unit */
   }

   for( i=0; i<*m; i++ ){
      P[i] = PP[i]; /* shift has to be done in the calling Fortran unit */
   }
   for( i=0; i<*n; i++ ){
      Q[i] = QQ[i]; /* shift has to be done in the calling Fortran unit */
   }

   free(LLp);
   free(LLj);
   free(UUp);
   free(UUi);
   free(PP);
   free(QQ);
}

/* -------------------------------------------------------------------------- */
/* umf4fnum: free the Numeric object (cmplx)                                  */
/* -------------------------------------------------------------------------- */

/* call umf4fnum_z (numeric) */

#ifdef UNDERSCORE
#define umf4fnum_z umf4fnum_z_
#endif
void umf4fnum_z (void **Numeric)
{
   umfpack_zl_free_numeric (Numeric);
}

/* -------------------------------------------------------------------------- */
/* umf4fsym: free the Symbolic object (cmplx)                                 */
/* -------------------------------------------------------------------------- */

/* call umf4fsym_z (symbolic) */

#ifdef UNDERSCORE
#define umf4fsym_z umf4fsym_z_
#endif
void umf4fsym_z (void **Symbolic)
{
   umfpack_zl_free_symbolic (Symbolic);
}

/* -------------------------------------------------------------------------- */
/* get_suitesparse_version: get the full version numbers                      */
/* -------------------------------------------------------------------------- */

/* call get_suitesparse_version ( version )   */

#ifdef UNDERSCORE
#define get_suitesparse_version get_suitesparse_version_
#endif
void get_suitesparse_version (int *version )
{
   SuiteSparse_version( version );
}

