/*
 * SPQR: Fortran 90 wrapper
 *
 * Copyright É. Canot 2008-2016 -- IPR/CNRS
 *
 * December 17, 2015
 *
 *   addresses are stored in 32bit or 64bit int, according to the OS used
 *
 *   SPQR uses the 'long' version of CHOLMOD objects.
 *   CAUTION: use only the 'SuiteSparse_long' definition,
 *            which automatically changes according the OS.
 */

#include "spqr.hpp"

#define TRUE 1
#define FALSE 0

/* Address type: ADDRESS */
#ifdef _64_BITS
#define ADDRESS long long int
#else
#define ADDRESS int
#endif

extern "C" {

/*----------------------------------------------------------------------
 *
 * Solve A x = b        [least-square pb]  over-determined system (m>n)
 * Returns a status 0 if ok.
 *
 *  input: m, n, nnz, Ap, Ai, Ax,
 *         b
 *
 *  output: out, status
 *
 * Fortran call:
 *  call spqr_solve_lsq( m, n, nnz, Ap, Ai, Ax, b
 *                       out, status )
 */
#ifdef UNDERSCORE
#define SPQR_SOLVE_LSQ spqr_solve_lsq_
#else
#define SPQR_SOLVE_LSQ spqr_solve_lsq
#endif
void SPQR_SOLVE_LSQ(
     int* m, int* n, int* nnz, int Ap[], int Ai[], double Ax[],
          double b[],
     double out[], int* status
)
{
   SuiteSparse_long i;
   cholmod_sparse *AA;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   int ordering;
   double tol = 1.e-15;
   cholmod_dense *BB, *X;
   double *BBx, *Xx;
   cholmod_common *c;

   c = (cholmod_common*) malloc( sizeof(cholmod_common) );
   cholmod_l_start ( c );
   /* 2nd line: sorted, packed, unsymmetric, type */
   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*m, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 0, CHOLMOD_REAL,
                                   c );

   AAp = (SuiteSparse_long *)AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = (SuiteSparse_long *)AA->i;
   AAx = (double *)AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   BB = cholmod_l_allocate_dense( (SuiteSparse_long)*m, (SuiteSparse_long)1, (SuiteSparse_long)*m,
                                  CHOLMOD_REAL, c );
   BBx = (double *)BB->x;
   for( i=0; i<*m; i++ ){
      BBx[i] = b[i];
   }

   ordering = 0;

   X = SuiteSparseQR <double> (ordering, tol, AA, BB, c);

   cholmod_l_free_work( c );
   cholmod_l_free_sparse( &AA, c );
   cholmod_l_free_dense( &BB, c );

   if (X) {
      *status = 0; /* ok */

      Xx = (double *)X->x;
      for( i=0; i<*n; i++ ){
         out[i] = Xx[i];
      }

      cholmod_l_free_dense( &X, c );

   } else {
      *status = -1;
   }

   cholmod_l_finish( c );
   free( c );
}

/*----------------------------------------------------------------------
 *
 * Solve A' x = b       [least-square pb]  over-determined system (n>m)
 * Returns a status 0 if ok.
 *
 * Here, A is passed as arg. (1-based indexes). So it must be transpose
 * in this routine. Didn't find (yet ?) a routine which directly works
 * with the transpose of A.
 *
 *  input: m, n, nnz, Ap, Ai, Ax,
 *         b
 *
 *  output: out, status
 *
 * Fortran call:
 *  call spqr_tsolve_lsq( m, n, nnz, Ap, Ai, Ax, b
 *                        out, status )
 */
#ifdef UNDERSCORE
#define SPQR_TSOLVE_LSQ spqr_tsolve_lsq_
#else
#define SPQR_TSOLVE_LSQ spqr_tsolve_lsq
#endif
void SPQR_TSOLVE_LSQ(
     int* m, int* n, int* nnz, int Ap[], int Ai[], double Ax[],
          double b[],
     double out[], int* status
)
{
   SuiteSparse_long i;
   cholmod_sparse *AA, *tAA;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   int ordering;
   double tol = 1.e-15;
   cholmod_dense *BB, *X;
   double *BBx, *Xx;
   cholmod_common *c;

   c = (cholmod_common*) malloc( sizeof(cholmod_common) );
   cholmod_l_start ( c );
   /* 2nd line: sorted, packed, unsymmetric, type */
   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*m, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 0, CHOLMOD_REAL,
                                   c );

   AAp = (SuiteSparse_long *)AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = (SuiteSparse_long *)AA->i;
   AAx = (double *)AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   BB = cholmod_l_allocate_dense( (SuiteSparse_long)*n, (SuiteSparse_long)1, (SuiteSparse_long)*n,
                                  CHOLMOD_REAL, c );
   BBx = (double *)BB->x;
   for( i=0; i<*n; i++ ){
      BBx[i] = b[i];
   }

   ordering = 0;
   tAA = cholmod_l_transpose(AA,1,c);

   X = SuiteSparseQR <double> (ordering, tol, tAA, BB, c);

   cholmod_l_free_sparse( &tAA, c );
   cholmod_l_free_work( c );
   cholmod_l_free_sparse( &AA, c );
   cholmod_l_free_dense( &BB, c );

   if (X) {
      *status = 0; /* ok */

      Xx = (double *)X->x;
      for( i=0; i<*m; i++ ){
         out[i] = Xx[i];
      }

      cholmod_l_free_dense( &X, c );

   } else {
      *status = -1;
   }

   cholmod_l_finish( c );
   free( c );
}

/*----------------------------------------------------------------------
 *
 * Solve A x = b       [min. 2-norm sol]  under-determined system (m<n)
 * Returns a status 0 if ok.
 *
 *  input: m, n, nnz, Ap, Ai, Ax,
 *         b
 *
 *  output: out, status
 *
 * Fortran call:
 *  call spqr_solve_min2norm( m, n, nnz, Ap, Ai, Ax, b
 *                            out, status )
 */
#ifdef UNDERSCORE
#define SPQR_SOLVE_MIN2NORM spqr_solve_min2norm_
#else
#define SPQR_SOLVE_MIN2NORM spqr_solve_min2norm
#endif
void SPQR_SOLVE_MIN2NORM(
     int* m, int* n, int* nnz, int Ap[], int Ai[], double Ax[],
          double b[],
     double out[], int* status
)
{
   SuiteSparse_long i;
   cholmod_sparse *AA;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   int ordering;
   double tol = 1.e-15;
   cholmod_dense *BB, *X;
   double *BBx, *Xx;
   cholmod_common *c;

   c = (cholmod_common*) malloc( sizeof(cholmod_common) );
   cholmod_l_start ( c );
   /* 2nd line: sorted, packed, unsymmetric, type */
   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*m, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 0, CHOLMOD_REAL,
                                   c );

   AAp = (SuiteSparse_long *)AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = (SuiteSparse_long *)AA->i;
   AAx = (double *)AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   BB = cholmod_l_allocate_dense( (SuiteSparse_long)*m, (SuiteSparse_long)1, (SuiteSparse_long)*m,
                                  CHOLMOD_REAL, c );
   BBx = (double *)BB->x;
   for( i=0; i<*m; i++ ){
      BBx[i] = b[i];
   }

   ordering = 0;

   X = SuiteSparseQR_min2norm <double> (ordering, tol, AA, BB, c);

   cholmod_l_free_work( c );
   cholmod_l_free_sparse( &AA, c );
   cholmod_l_free_dense( &BB, c );

   if (X) {
      *status = 0; /* ok */

      Xx = (double *)X->x;
      for( i=0; i<*n; i++ ){
         out[i] = Xx[i];
      }

      cholmod_l_free_dense( &X, c );

   } else {
      *status = -1;
   }

   cholmod_l_finish( c );
   free( c );
}

/*----------------------------------------------------------------------
 *
 * Solve A' x = b      [min. 2-norm sol]  under-determined system (n<m)
 * Returns a status 0 if ok.
 *
 * Here, A is passed as arg. (1-based indexes). So it must be transpose
 * in this routine. Didn't find (yet ?) a routine which directly works
 * with the transpose of A.
 *
 *  input: m, n, nnz, Ap, Ai, Ax,
 *         b
 *
 *  output: out, status
 *
 * Fortran call:
 *  call spqr_tsolve_min2norm( m, n, nnz, Ap, Ai, Ax, b
 *                             out, status )
 */
#ifdef UNDERSCORE
#define SPQR_TSOLVE_MIN2NORM spqr_tsolve_min2norm_
#else
#define SPQR_TSOLVE_MIN2NORM spqr_tsolve_min2norm
#endif
void SPQR_TSOLVE_MIN2NORM(
     int* m, int* n, int* nnz, int Ap[], int Ai[], double Ax[],
          double b[],
     double out[], int* status
)
{
   SuiteSparse_long i;
   cholmod_sparse *AA, *tAA;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   int ordering;
   double tol = 1.e-15;
   cholmod_dense *BB, *X;
   double *BBx, *Xx;
   cholmod_common *c;

   c = (cholmod_common*) malloc( sizeof(cholmod_common) );
   cholmod_l_start ( c );
   /* 2nd line: sorted, packed, unsymmetric, type */
   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*m, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 0, CHOLMOD_REAL,
                                   c );

   AAp = (SuiteSparse_long *)AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = (SuiteSparse_long *)AA->i;
   AAx = (double *)AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   BB = cholmod_l_allocate_dense( (SuiteSparse_long)*n, (SuiteSparse_long)1, (SuiteSparse_long)*n,
                                  CHOLMOD_REAL, c );
   BBx = (double *)BB->x;
   for( i=0; i<*n; i++ ){
      BBx[i] = b[i];
   }

   ordering = 0;
   tAA = cholmod_l_transpose(AA,1,c);

   X = SuiteSparseQR_min2norm <double> (ordering, tol, tAA, BB, c);

   cholmod_l_free_sparse( &tAA, c );
   cholmod_l_free_work( c );
   cholmod_l_free_sparse( &AA, c );
   cholmod_l_free_dense( &BB, c );

   if (X) {
      *status = 0; /* ok */

      Xx = (double *)X->x;
      for( i=0; i<*m; i++ ){
         out[i] = Xx[i];
      }

      cholmod_l_free_dense( &X, c );

   } else {
      *status = -1;
   }

   cholmod_l_finish( c );
   free( c );
}

/*----------------------------------------------------------------------
 *
 * Do a Q-less factorization of A. Returns a status 0 if ok.
 * A must be sorted.
 *
 *  input: m, n, nnz, Ap, Ai, Ax
 *
 *  output: c_addr, R_addr, rnz, status
 *          (R size will be n-by-n)
 *
 * Fortran call:
 *   call spqr_q_less_prep( m, n, nnz, Ap, Ai, Ax,
 *                          c_addr, R_addr, rnz, status )
 */
#ifdef UNDERSCORE
#define SPQR_Q_LESS_PREP spqr_q_less_prep_
#else
#define SPQR_Q_LESS_PREP spqr_q_less_prep
#endif
void SPQR_Q_LESS_PREP(
     int* m, int* n, int* nnz, int Ap[], int Ai[], double Ax[],
     ADDRESS* c_addr, ADDRESS* R_addr, int* Rnz, int* status
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *AA, *R;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   int ordering;
/* ### TODO ?: what is this tolerance? for which use? */
   double tol = 1.e-15;
   SuiteSparse_long econ;
   SuiteSparse_long rank;

   cholmod_common *c;

   c = (cholmod_common*) malloc( sizeof(cholmod_common) );
   *c_addr = (ADDRESS)c;
   cholmod_l_start ( c );

   /* 2nd line: sorted, packed, unsymmetric, type */
   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*m, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 0, CHOLMOD_REAL,
                                   c );

   AAp = (SuiteSparse_long *)AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = (SuiteSparse_long *)AA->i;
   AAx = (double *)AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   ordering = 0;
   /* R size will be e-by-n */
   econ = *n;

   /* [Q,R,E] = qr(A), discarding Q (SuiteSparseQR.hpp) */
   rank = SuiteSparseQR <double> (ordering, tol, econ, AA, &R, NULL, c);

   cholmod_l_free_work( c );
   cholmod_l_free_sparse( &AA, c );

   if (rank==-1) {
      /* not useful to return c->status; in SPQR it is set either to
         CHOLMOD_OK or to CHOLMOD_OUT_OF_MEMORY */
      *status = -1;
   } else {
      *status = 0; /* ok */

      *R_addr = (ADDRESS)R;
      *Rnz = cholmod_l_nnz( R, c );

   }

}

/*----------------------------------------------------------------------
 *
 * Get the entries of the R factor. Free R and c.
 *
 *  input: c_addr, R_addr, nn, rnz
 *
 *  output: Rp, Ri, Rx
 *
 * Fortran call:
 *   call spqr_get_R( c_addr, R_addr, nn, rnz
 *                    Rp, Ri, Rx )
 */
#ifdef UNDERSCORE
#define SPQR_GET_R spqr_get_r_
#else
#define SPQR_GET_R spqr_get_r
#endif
void SPQR_GET_R(
     ADDRESS* c_addr, ADDRESS* R_addr, int* nn, int* rnz,
     int Rp[], int Ri[], double Rx[]
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *RR;
   SuiteSparse_long *RRp, *RRi;
   double *RRx;

   cholmod_common *c;

   c = (cholmod_common *)*c_addr;
   RR = (cholmod_sparse *)*R_addr;

   RRp = (SuiteSparse_long *)RR->p;
   for( i=0; i<*nn+1; i++ ){
      Rp[i] = RRp[i] + 1; /* shift */
   }

   RRi = (SuiteSparse_long *)RR->i;
   RRx = (double *)RR->x;
   for( i=0; i<*rnz; i++ ){
      Ri[i] = RRi[i] + 1; /* shift */
      Rx[i] = RRx[i];
   }

   cholmod_l_free_sparse( &RR, c );

   cholmod_l_finish( c );
   free( c );
}

/*----------------------------------------------------------------------
 *
 * Do a QR factorization of A. Returns a status 0 if ok.
 * A must be sorted. Q is sparse
 *
 *  input: m, n, nnz, Ap, Ai, Ax, ordering, tol, econ
 *
 *  output: c_addr, Q_addr, qnz, R_addr, rnz, P_addr, rank, status
 *
 * Fortran call:
 *   call spqr_qr_prep( m, n, nnz, Ap, Ai, Ax, ordering, tol, econ
 *                      c_addr, Q_addr, qnz, R_addr, rnz, P_addr,
 *                      rank, status )
 */
#ifdef UNDERSCORE
#define SPQR_QR_PREP spqr_qr_prep_
#else
#define SPQR_QR_PREP spqr_qr_prep
#endif
void SPQR_QR_PREP(
     int* m, int* n, int* nnz, int Ap[], int Ai[], double Ax[],
             int* ordering, double* tol, int* econ1,
     ADDRESS* c_addr, ADDRESS* Q_addr, int* Qnz, ADDRESS* R_addr,
             int* Rnz, ADDRESS* P_addr, int* rank1, int* status
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i, rank;
   SuiteSparse_long econ = *econ1;

   cholmod_sparse *AA, *Q, *R;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   SuiteSparse_long *P;

   cholmod_common *c;

   c = (cholmod_common*) malloc( sizeof(cholmod_common) );
   *c_addr = (ADDRESS)c;
   cholmod_l_start ( c );

   /* 2nd line: sorted, packed, unsymmetric, type */
   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*m, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 0, CHOLMOD_REAL,
                                   c );

   AAp = (SuiteSparse_long *)AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = (SuiteSparse_long *)AA->i;
   AAx = (double *)AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   /* [Q,R,E] = qr(A), returning Q as a sparse matrix */
   rank = SuiteSparseQR <double> (*ordering, *tol, econ, AA,
                                   &Q, &R, &P, c);
   *rank1 = rank;

   cholmod_l_free_work( c );
   cholmod_l_free_sparse( &AA, c );

   if (rank==-1) {
      /* not useful to return c->status; in SPQR it is set either to
         CHOLMOD_OK or to CHOLMOD_OUT_OF_MEMORY */
      *status = -1;
   } else {
      *status = 0; /* ok */

      *Q_addr = (ADDRESS)Q;
      *Qnz = cholmod_l_nnz( Q, c );

      *R_addr = (ADDRESS)R;
      *Rnz = cholmod_l_nnz( R, c );

      *P_addr = (ADDRESS)P;

   }

}

/*----------------------------------------------------------------------
 *
 * Get the entries of the factors Q and R. Free Q, R and c.
 *
 *  input: c_addr, Q_addr, qn, qnz, R_addr, rn, rnz
 *
 *  output: Qp, Qi, Qx, Rp, Ri, Rx
 *
 * Fortran call:
 *   call spqr_get_Q_R( c_addr, Q_addr, qn, qnz, R_addr, rn, rnz,
 *                      Qp, Qi, Qx, Rp, Ri, Rx )
 */
#ifdef UNDERSCORE
#define SPQR_GET_Q_R spqr_get_q_r_
#else
#define SPQR_GET_Q_R spqr_get_q_r
#endif
void SPQR_GET_Q_R(
     ADDRESS* c_addr, ADDRESS* Q_addr, int* qn, int* qnz,
                   ADDRESS* R_addr, int* rn, int* rnz,
     int Qp[], int Qi[], double Qx[], int Rp[], int Ri[], double Rx[]
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *QQ;
   SuiteSparse_long *QQp, *QQi;
   double *QQx;

   cholmod_sparse *RR;
   SuiteSparse_long *RRp, *RRi;
   double *RRx;

   cholmod_common *c;

   c = (cholmod_common *)*c_addr;
   QQ = (cholmod_sparse *)*Q_addr;
   RR = (cholmod_sparse *)*R_addr;

   QQp = (SuiteSparse_long *)QQ->p;
   for( i=0; i<*qn+1; i++ ){
      Qp[i] = QQp[i] + 1; /* shift */
   }

   QQi = (SuiteSparse_long *)QQ->i;
   QQx = (double *)QQ->x;
   for( i=0; i<*qnz; i++ ){
      Qi[i] = QQi[i] + 1; /* shift */
      Qx[i] = QQx[i];
   }

   RRp = (SuiteSparse_long *)RR->p;
   for( i=0; i<*rn+1; i++ ){
      Rp[i] = RRp[i] + 1; /* shift */
   }

   RRi = (SuiteSparse_long *)RR->i;
   RRx = (double *)RR->x;
   for( i=0; i<*rnz; i++ ){
      Ri[i] = RRi[i] + 1; /* shift */
      Rx[i] = RRx[i];
   }

   cholmod_l_free_sparse( &QQ, c );
   cholmod_l_free_sparse( &RR, c );

   cholmod_l_finish( c );
   free( c );
}

/*----------------------------------------------------------------------
 *
 * Get the entries of the factors Q, R and the vector permutation p.
 * Free Q, R, P and c.
 *
 *  input: c_addr, Q_addr, qn, qnz, R_addr, rn, rnz, p_addr, pn
 *
 *  output: Qp, Qi, Qx, Rp, Ri, Rx, p
 *
 * Fortran call:
 *   call spqr_get_Q_R_P( c_addr, Q_addr, qn, qnz, R_addr, rn, rnz,
 *                                p_addr, pn,
 *                        Qp, Qi, Qx, Rp, Ri, Rx, p )
 */
#ifdef UNDERSCORE
#define SPQR_GET_Q_R_P spqr_get_q_r_p_
#else
#define SPQR_GET_Q_R_P spqr_get_q_r_p
#endif
void SPQR_GET_Q_R_P(
     ADDRESS* c_addr, ADDRESS* Q_addr, int* qn, int* qnz,
                   ADDRESS* R_addr, int* rn, int* rnz,
                   ADDRESS* p_addr, int* pn,
     int Qp[], int Qi[], double Qx[], int Rp[], int Ri[], double Rx[],
               int p[]
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *QQ;
   SuiteSparse_long *QQp, *QQi;
   double *QQx;

   cholmod_sparse *RR;
   SuiteSparse_long *RRp, *RRi;
   double *RRx;

   SuiteSparse_long *PP;

   cholmod_common *c;

   c = (cholmod_common *)*c_addr;
   QQ = (cholmod_sparse *)*Q_addr;
   RR = (cholmod_sparse *)*R_addr;

   QQp = (SuiteSparse_long *)QQ->p;
   for( i=0; i<*qn+1; i++ ){
      Qp[i] = QQp[i] + 1; /* shift */
   }

   QQi = (SuiteSparse_long *)QQ->i;
   QQx = (double *)QQ->x;
   for( i=0; i<*qnz; i++ ){
      Qi[i] = QQi[i] + 1; /* shift */
      Qx[i] = QQx[i];
   }

   RRp = (SuiteSparse_long *)RR->p;
   for( i=0; i<*rn+1; i++ ){
      Rp[i] = RRp[i] + 1; /* shift */
   }

   RRi = (SuiteSparse_long *)RR->i;
   RRx = (double *)RR->x;
   for( i=0; i<*rnz; i++ ){
      Ri[i] = RRi[i] + 1; /* shift */
      Rx[i] = RRx[i];
   }

   PP = (SuiteSparse_long *)*p_addr;
   for( i=0; i<*pn; i++ ){
      p[i] = PP[i] + 1; /* shift */
   }

   cholmod_l_free_sparse( &QQ, c );
   cholmod_l_free_sparse( &RR, c );
   free( PP );

   cholmod_l_finish( c );
   free( c );
}

/*----------------------------------------------------------------------
 *
 * Do a QR factorization of A. Returns a status 0 if ok.
 * A must be sorted. Q is returned in Householder form
 *
 *  input: m, n, nnz, Ap, Ai, Ax, ordering, tol, econ
 *
 *  output: c_addr, H_addr, HTau_addr, HPinv_addr
                    R_addr, rnz, P_addr, rank, status
 *
 * Fortran call:
 *   call spqr_qhr_prep( m, n, nnz, Ap, Ai, Ax, ordering, tol, econ
 *                       c_addr, H_addr, HTau_addr, HPinv_addr,
 *                               R_addr, rnz, P_addr, rank, status )
 */
#ifdef UNDERSCORE
#define SPQHR_QR_PREP spqhr_qr_prep_
#else
#define SPQHR_QR_PREP spqhr_qr_prep
#endif
void SPQHR_QR_PREP(
     int* m, int* n, int* nnz, int Ap[], int Ai[], double Ax[],
             int* ordering, double* tol, int* econ1,
     ADDRESS* c_addr, ADDRESS* H_addr, ADDRESS* HTau_addr,
             ADDRESS* HPinv_addr, ADDRESS* R_addr, int* Rnz,
             ADDRESS* P_addr, int* rank1, int* status
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i, rank;
   SuiteSparse_long econ = *econ1;

   cholmod_sparse *AA, *H, *R;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;
   cholmod_dense *HTau;

   SuiteSparse_long *P, *HPinv;

   cholmod_common *c;

   c = (cholmod_common*) malloc( sizeof(cholmod_common) );
   *c_addr = (ADDRESS)c;
   cholmod_l_start ( c );

   /* 2nd line: sorted, packed, unsymmetric, type */
   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*m, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 0, CHOLMOD_REAL,
                                   c );

   AAp = (SuiteSparse_long *)AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = (SuiteSparse_long *)AA->i;
   AAx = (double *)AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   /* [Q,R,E] = qr(A) where Q is returned in Householder form */
   rank = SuiteSparseQR <double> (*ordering, *tol, econ, AA,
                                   &R, &P, &H, &HPinv, &HTau, c);
   *rank1 = rank;

   cholmod_l_free_work( c );
   cholmod_l_free_sparse( &AA, c );

   if (rank==-1) {
      *status = -1;
   } else {
      *status = 0; /* ok */

      *H_addr = (ADDRESS)H;
      *HTau_addr = (ADDRESS)HTau;
      *HPinv_addr = (ADDRESS)HPinv;

      *R_addr = (ADDRESS)R;
      *Rnz = cholmod_l_nnz( R, c );

      *P_addr = (ADDRESS)P;

   }

}

/*----------------------------------------------------------------------
 *
 * Get the entries of a cholmod_sparse object. Free this object.
 *
 *  input: c_addr, A_addr, ncol_A, nz_A
 *
 *  output: Ap, Ai, Ax
 *
 * Fortran call:
 *   call spqr_get_sparse_entries( c_addr, A_addr, ncol_A, nz_A
 *                                 Ap, Ai, Ax )
 */
#ifdef UNDERSCORE
#define SPQR_GET_SPARSE_ENTRIES spqr_get_sparse_entries_
#else
#define SPQR_GET_SPARSE_ENTRIES spqr_get_sparse_entries
#endif
void SPQR_GET_SPARSE_ENTRIES(
     ADDRESS* c_addr, ADDRESS* A_addr, int* ncol_A, int* nz_A,
     int Ap[], int Ai[], double Ax[]
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *AA;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   cholmod_common *c;

   c = (cholmod_common *)*c_addr;
   AA = (cholmod_sparse *)*A_addr;

   AAp = (SuiteSparse_long *)AA->p;
   for( i=0; i<*ncol_A+1; i++ ){
      Ap[i] = AAp[i] + 1; /* shift */
   }

   AAi = (SuiteSparse_long *)AA->i;
   AAx = (double *)AA->x;
   for( i=0; i<*nz_A; i++ ){
      Ai[i] = AAi[i] + 1; /* shift */
      Ax[i] = AAx[i];
   }

   cholmod_l_free_sparse( &AA, c );

}

/*----------------------------------------------------------------------
 *
 * Get the entries of the factor R and permutation p.
 * Free R, P and c.
 *
 *  input: c_addr, R_addr, rn, rnz, p_addr, pn
 *
 *  output: Rp, Ri, Rx, p
 *
 * Fortran call:
 *   call spqhr_get_R_P( c_addr, R_addr, rn, rnz, p_addr, pn,
 *                       Rp, Ri, Rx, p )
 */
#ifdef UNDERSCORE
#define SPQHR_GET_R_P spqhr_get_r_p_
#else
#define SPQHR_GET_R_P spqhr_get_r_p
#endif
void SPQHR_GET_R_P(
     ADDRESS* c_addr, ADDRESS* R_addr, int* rn, int* rnz,
                   ADDRESS* p_addr, int* pn,
     int Rp[], int Ri[], double Rx[], int p[]
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *RR;
   SuiteSparse_long *RRp, *RRi;
   double *RRx;

   SuiteSparse_long *PP;

   cholmod_common *c;

   c = (cholmod_common *)*c_addr;
   RR = (cholmod_sparse *)*R_addr;

   RRp = (SuiteSparse_long *)RR->p;
   for( i=0; i<*rn+1; i++ ){
      Rp[i] = RRp[i] + 1; /* shift */
   }

   RRi = (SuiteSparse_long *)RR->i;
   RRx = (double *)RR->x;
   for( i=0; i<*rnz; i++ ){
      Ri[i] = RRi[i] + 1; /* shift */
      Rx[i] = RRx[i];
   }

   PP = (SuiteSparse_long *)*p_addr;
   for( i=0; i<*pn; i++ ){
      p[i] = PP[i] + 1; /* shift */
   }

   cholmod_l_free_sparse( &RR, c );
   free( PP );

}

/*----------------------------------------------------------------------
 *
 * Apply Householder vector on the left, i.e. Q'*A
 * A is dense
 *
 *  input: c_addr, H_addr, HTau_addr, HPinv_addr,
           A, nrow_A, ncol_A, ld_A
 *
 *  output: out, status
 *
 * Fortran call:
 *   call spqhr_mleft_dense( c_addr, H_addr, HTau_addr, HPinv_addr,
 *                                   A, nrow_A, ncol_A, ld_A
 *                           out, status )
 */
#ifdef UNDERSCORE
#define SPQHR_MLEFT_DENSE spqhr_mleft_dense_
#else
#define SPQHR_MLEFT_DENSE spqhr_mleft_dense
#endif
void SPQHR_MLEFT_DENSE(
     ADDRESS* c_addr, ADDRESS* H_addr, ADDRESS* HTau_addr,
     ADDRESS* HPinv_addr, double* A, int* nrow_A, int* ncol_A, int* ld_A,
     double* out, int* status
)
{
   SuiteSparse_long i, j, k, nrow_out, ncol_out, ld_out;

   cholmod_sparse *H;
   cholmod_dense *HTau, *Adense, *outdense;
   SuiteSparse_long *HPinv;
   cholmod_common *c;

   c = (cholmod_common *)*c_addr;
   H = (cholmod_sparse *)*H_addr;
   HTau = (cholmod_dense *)*HTau_addr;
   HPinv = (SuiteSparse_long *)*HPinv_addr;
   double *AAp, *outp;

   Adense = cholmod_l_allocate_dense( (SuiteSparse_long)*nrow_A, (SuiteSparse_long)*ncol_A, (SuiteSparse_long)*ld_A,
                                      CHOLMOD_REAL, c );
   AAp = (double *)Adense->x;
   for( j=0; j<*ncol_A; j++ ){
      for( i=0; i<*nrow_A; i++ ){
         k = i + j*(*ld_A);
         AAp[k] = A[k];
      }
   }

   outdense = SuiteSparseQR_qmult <double> ( 0, H, HTau, HPinv, Adense, c );

   cholmod_l_free_work( c );
   cholmod_l_free_dense( &Adense, c );

   if (outdense) {
      *status = 0; /* ok */

      nrow_out = outdense->nrow;
      ncol_out = outdense->ncol;
      ld_out = outdense->d;

      outp = (double *)outdense->x;
      k = 0;
      for( j=0; j<ncol_out; j++ ){
         for( i=0; i<nrow_out; i++ ){
            out[k++] = outp[i + j*ld_out];
         }
      }

      cholmod_l_free_dense( &outdense, c );

   } else {
      *status = -1;
   }
}

/*----------------------------------------------------------------------
 *
 * Apply Householder vector on the right, i.e. A*Q
 * A is dense
 *
 *  input: c_addr, H_addr, HTau_addr, HPinv_addr,
           A, nrow_A, ncol_A, ld_A
 *
 *  output: out, status
 *
 * Fortran call:
 *   call spqhr_mright_dense( c_addr, H_addr, HTau_addr, HPinv_addr,
 *                                    A, nrow_A, ncol_A, ld_A
 *                            out, status )
 */
#ifdef UNDERSCORE
#define SPQHR_MRIGHT_DENSE spqhr_mright_dense_
#else
#define SPQHR_MRIGHT_DENSE spqhr_mright_dense
#endif
void SPQHR_MRIGHT_DENSE(
     ADDRESS* c_addr, ADDRESS* H_addr, ADDRESS* HTau_addr,
     ADDRESS* HPinv_addr, double* A, int* nrow_A, int* ncol_A, int* ld_A,
     double* out, int* status
)
{
   SuiteSparse_long i, j, k, nrow_out, ncol_out, ld_out;

   cholmod_sparse *H;
   cholmod_dense *HTau, *Adense, *outdense;
   SuiteSparse_long *HPinv;
   cholmod_common *c;

   c = (cholmod_common *)*c_addr;
   H = (cholmod_sparse *)*H_addr;
   HTau = (cholmod_dense *)*HTau_addr;
   HPinv = (SuiteSparse_long *)*HPinv_addr;
   double *AAp, *outp;

   Adense = cholmod_l_allocate_dense( *nrow_A, *ncol_A, *ld_A,
                                      CHOLMOD_REAL, c );
   AAp = (double *)Adense->x;
   for( j=0; j<*ncol_A; j++ ){
      for( i=0; i<*nrow_A; i++ ){
         k = i + j*(*ld_A);
         AAp[k] = A[k];
      }
   }

   outdense = SuiteSparseQR_qmult <double> ( 3, H, HTau, HPinv, Adense, c );

   cholmod_l_free_work( c );
   cholmod_l_free_dense( &Adense, c );

   if (outdense) {
      *status = 0; /* ok */

      nrow_out = outdense->nrow;
      ncol_out = outdense->ncol;
      ld_out = outdense->d;

      outp = (double *)outdense->x;
      k = 0;
      for( j=0; j<ncol_out; j++ ){
         for( i=0; i<nrow_out; i++ ){
            out[k++] = outp[i + j*ld_out];
         }
      }

      cholmod_l_free_dense( &outdense, c );

   } else {
      *status = -1;
   }
}

/*----------------------------------------------------------------------
 *
 * Apply Householder vector on the left, i.e. Q'*A
 * A is sparse. Returns a status 0 if ok.
 *
 *  input: c_addr, H_addr, HTau_addr, HPinv_addr,
           nrow_A, ncol_A, nnz_A, Ap, Ai, Ax
 *
 *  output: out_addr, out_nz, status
 *
 * Fortran call:
 *   call spqhr_mleft_sparse_prep(
 *         c_addr, H_addr, HTau_addr, HPinv_addr,
 *         nrow_A, ncol_A, nnz_A, Ap, Ai, Ax,
 *         out_addr, out_nz, status )
 */
#ifdef UNDERSCORE
#define SPQHR_MLEFT_SPARSE_PREP spqhr_mleft_sparse_prep_
#else
#define SPQHR_MLEFT_SPARSE_PREP spqhr_mleft_sparse_prep
#endif
void SPQHR_MLEFT_SPARSE_PREP(
     ADDRESS* c_addr, ADDRESS* H_addr, ADDRESS* HTau_addr,
     ADDRESS* HPinv_addr, int* nrow_A, int* ncol_A, int* nnz_A,
     int Ap[], int Ai[], double Ax[],
     ADDRESS* out_addr, int* out_nz, int* status
)
{
   SuiteSparse_long i;

   cholmod_sparse *H, *Asparse, *outsparse;
   cholmod_dense *HTau;
   SuiteSparse_long *HPinv;
   cholmod_common *c;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   c = (cholmod_common *)*c_addr;
   H = (cholmod_sparse *)*H_addr;
   HTau = (cholmod_dense *)*HTau_addr;
   HPinv = (SuiteSparse_long *)*HPinv_addr;

   /* 2nd line: sorted, packed, unsymmetric, type */
   Asparse = cholmod_l_allocate_sparse( (SuiteSparse_long)*nrow_A, (SuiteSparse_long)*ncol_A, (SuiteSparse_long)*nnz_A,
                                        TRUE, TRUE, 0, CHOLMOD_REAL,
                                        c );
   AAp = (SuiteSparse_long *)Asparse->p;
   for( i=0; i<*ncol_A+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = (SuiteSparse_long *)Asparse->i;
   AAx = (double *)Asparse->x;
   for( i=0; i<*nnz_A; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   outsparse = SuiteSparseQR_qmult <double> ( 0, H, HTau, HPinv, Asparse, c );

   cholmod_l_free_work( c );
   cholmod_l_free_sparse( &Asparse, c );

   if (outsparse) {
      *status = 0; /* ok */

      *out_addr = (ADDRESS)outsparse;
      *out_nz = cholmod_l_nnz( outsparse, c );

   } else {
      *status = -1;
   }
}

/*----------------------------------------------------------------------
 *
 * Apply Householder vector on the right, i.e. A*Q
 * A is sparse. Returns a status 0 if ok.
 *
 *  input: c_addr, H_addr, HTau_addr, HPinv_addr,
           nrow_A, ncol_A, nnz_A, Ap, Ai, Ax
 *
 *  output: out_addr, out_nz, status
 *
 * Fortran call:
 *   call spqhr_mright_sparse_prep(
 *         c_addr, H_addr, HTau_addr, HPinv_addr,
 *         nrow_A, ncol_A, nnz_A, Ap, Ai, Ax,
 *         out_addr, out_nz, status )
 */
#ifdef UNDERSCORE
#define SPQHR_MRIGHT_SPARSE_PREP spqhr_mright_sparse_prep_
#else
#define SPQHR_MRIGHT_SPARSE_PREP spqhr_mright_sparse_prep
#endif
void SPQHR_MRIGHT_SPARSE_PREP(
     ADDRESS* c_addr, ADDRESS* H_addr, ADDRESS* HTau_addr,
     ADDRESS* HPinv_addr, int* nrow_A, int* ncol_A, int* nnz_A,
     int Ap[], int Ai[], double Ax[],
     ADDRESS* out_addr, int* out_nz, int* status
)
{
   SuiteSparse_long i;

   cholmod_sparse *H, *Asparse, *outsparse;
   cholmod_dense *HTau;
   SuiteSparse_long *HPinv;
   cholmod_common *c;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   c = (cholmod_common *)*c_addr;
   H = (cholmod_sparse *)*H_addr;
   HTau = (cholmod_dense *)*HTau_addr;
   HPinv = (SuiteSparse_long *)*HPinv_addr;

   /* 2nd line: sorted, packed, unsymmetric, type */
   Asparse = cholmod_l_allocate_sparse( (SuiteSparse_long)*nrow_A, (SuiteSparse_long)*ncol_A, (SuiteSparse_long)*nnz_A,
                                        TRUE, TRUE, 0, CHOLMOD_REAL,
                                        c );
   AAp = (SuiteSparse_long *)Asparse->p;
   for( i=0; i<*ncol_A+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = (SuiteSparse_long *)Asparse->i;
   AAx = (double *)Asparse->x;
   for( i=0; i<*nnz_A; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   outsparse = SuiteSparseQR_qmult <double> ( 3, H, HTau, HPinv, Asparse, c );

   cholmod_l_free_work( c );
   cholmod_l_free_sparse( &Asparse, c );

   if (outsparse) {
      *status = 0; /* ok */

      *out_addr = (ADDRESS)outsparse;
      *out_nz = cholmod_l_nnz( outsparse, c );

   } else {
      *status = -1;
   }
}

/*----------------------------------------------------------------------
 *
 * Free a (long int) vector.
 *
 *  input: A_addr
 *
 * Fortran call:
 *   call spqr_free_long_vec( A_addr )
 */
#ifdef UNDERSCORE
#define SPQR_FREE_LONG_VEC spqr_free_long_vec_
#else
#define SPQR_FREE_LONG_VEC spqr_free_long_vec
#endif
void SPQR_FREE_LONG_VEC(
     ADDRESS* A_addr
)
{

   SuiteSparse_long *A;

   A = (SuiteSparse_long *)*A_addr;

   free( A );

}

/*----------------------------------------------------------------------
 *
 * Free a cholmod_common object.
 *
 *  input: c_addr
 *
 * Fortran call:
 *   call spqr_free_common( c_addr )
 */
#ifdef UNDERSCORE
#define SPQR_FREE_COMMON spqr_free_common_
#else
#define SPQR_FREE_COMMON spqr_free_common
#endif
void SPQR_FREE_COMMON(
     ADDRESS* c_addr
)
{

   cholmod_common *c;

   c = (cholmod_common *)*c_addr;

   cholmod_l_finish( c );
   free( c );

}

/*----------------------------------------------------------------------
 *
 * Free a cholmod_dense object.
 *
 *  input: c_addr, A_addr
 *
 * Fortran call:
 *   call spqr_free_dense( c_addr, A_addr )
 */
#ifdef UNDERSCORE
#define SPQR_FREE_DENSE spqr_free_dense_
#else
#define SPQR_FREE_DENSE spqr_free_dense
#endif
void SPQR_FREE_DENSE(
     ADDRESS* c_addr, ADDRESS* A_addr
)
{

   cholmod_dense *AA;
   cholmod_common *c;

   c = (cholmod_common *)*c_addr;
   AA = (cholmod_dense *)*A_addr;

   cholmod_l_free_dense( &AA, c );

}

/*----------------------------------------------------------------------
 *
 * Free a cholmod_sparse object.
 *
 *  input: c_addr, A_addr
 *
 * Fortran call:
 *   call spqr_free_sparse( c_addr, A_addr )
 */
#ifdef UNDERSCORE
#define SPQR_FREE_SPARSE spqr_free_sparse_
#else
#define SPQR_FREE_SPARSE spqr_free_sparse
#endif
void SPQR_FREE_SPARSE(
     ADDRESS* c_addr, ADDRESS* A_addr
)
{

   cholmod_sparse *AA;
   cholmod_common *c;

   c = (cholmod_common *)*c_addr;
   AA = (cholmod_sparse *)*A_addr;

   cholmod_l_free_sparse( &AA, c );

}

/*--------------------------------------------------------------------*/

}
