module splines

   ! version 0.7.4 -- 26 May 2018
   ! = 0.7.3 -- modify stop messages to write "internal error".
   ! = 0.7.2 -- program doesn't stop in int_spl for out-of-range input
   !            abscissa if out_of_range (new opt. arg.) is present.
   ! = 0.7.1 -- preprocessing done for precision-dependent items.
   ! = 0.7   -- in only one module (double precision only).
   !
   ! (C) Édouard CANOT -- Edouard.Canot@univ-rennes.fr -- CNRS

   implicit none

   integer, parameter :: PREC = kind(1.0d0)
   private

   !-----------
   interface cub_spl
      module procedure cub_spl_driver_double
   end interface
   !-----------
   interface int_spl
      module procedure int_spl_double
   end interface
   !-----------
   interface som_spl
      module procedure som_spl_driver_double
   end interface
   !-----------

   public :: cub_spl, int_spl, som_spl

   interface diag3
      module procedure diag3_sym_double
   end interface

   interface diag5
      module procedure diag5_sym_double
   end interface

   interface cub_spl_std
      module procedure cub_spl_std_double
   end interface

   interface cub_spl_fit
      module procedure cub_spl_fit_double
   end interface

   interface cyclic3
      module procedure cyclic3_sym_double
   end interface

   interface cyclic5
      module procedure cyclic5_sym_double
   end interface

   interface cub_spl_per
      module procedure cub_spl_per_double
   end interface

   interface cub_spl_per_fit
      module procedure cub_spl_per_fit_double
   end interface

contains
!_______________________________________________________________________
!
subroutine diag3_sym_double( a, b, r, u )

   real(kind=PREC), dimension(:), intent(in)  :: a, b, r
   real(kind=PREC), dimension(:), intent(out) :: u
   !------ API end ------

   ! Système à 3 diagonales symétrique :
   !
   !    a(1:n-1) : sous-diagonale
   !    b(1:n)   :      diagonale principale
   !    a(1:n-1) :  sur-diagonale
   !
   !    r(1:n)   : vecteur second membre (right)
   !    u(1:n)   : vecteur solution      (unknown)
   !
   ! rem. 1 : les vecteurs : a, b, r  ne sont pas modifiés
   !
   ! rem. 2 : on peut calculer u directement dans r, l'appel est alors
   !          call PROGRAM_NAME( a, b, r, r )

   real(kind=PREC), dimension(size(a)) :: gam
   integer                             :: n, i
   real(kind=PREC)                     :: bet

   !------ end of declarations -- execution starts hereafter  ------

   n = size(u)

   if( size(a)+1 /= n  .or.                                             &
       size(b)   /= n  .or.                                             &
       size(r)   /= n       ) then
      print *, "internal error: in splines[diag3]:"
      print *, "                wrong array dimensions"
      stop
   end if

   bet = b(1)
   if( bet == 0.0_PREC ) then
      print *, "internal error: in splines[diag3]:"
      print *, "                rewrite equations"
      stop
   end if
   u(1) = r(1)/bet
   do i = 2, n
      gam(i-1) = a(i-1)/bet
      bet = b(i) - a(i-1)*gam(i-1)
      if( bet == 0.0_PREC ) then
         print *, "internal error: in splines[diag3]:"
         print *, "                failed"
         stop
      end if
      u(i) = (r(i)-a(i-1)*u(i-1))/bet
   end do
   do i = n-1, 1, -1
      u(i) = u(i) - gam(i)*u(i+1)
   end do

end subroutine diag3_sym_double
!_______________________________________________________________________
!
subroutine diag5_sym_double( a, b, c, r, u )

   real(kind=PREC), dimension(:), intent(in)  :: a, b, c, r
   real(kind=PREC), dimension(:), intent(out) :: u
   !------ API end ------

   ! Système à 5 diagonales symétrique :
   !
   !    a(1:n-2) : sous(2)-diagonale
   !    b(1:n-1) :    sous-diagonale
   !    c(1:n)   :         diagonale principale
   !    b(1:n-1) :     sur-diagonale
   !    a(1:n-2) :  sur(2)-diagonale
   !
   !    r(1:n)   : vecteur second membre (right)
   !    u(1:n)   : vecteur solution      (unknown)
   !
   ! rem. 1 : les vecteurs : a, b, c, r  ne sont pas modifiés
   !
   ! rem. 2 : on peut calculer u directement dans r, l'appel est alors
   !          call PROGRAM_NAME( a, b, c, r, r )

   real(kind=PREC), dimension(size(b)) :: del
   real(kind=PREC), dimension(size(a)) :: eta
   integer                             :: n, i
   real(kind=PREC)                     :: bet, gam

   !------ end of declarations -- execution starts hereafter  ------

   n = size(u)

   if( n < 3 ) then
      print *, "internal error: in splines[diag5]:"
      print *, "                n must be >= 3"
      stop
   end if

   if( size(a)+2 /= n  .or.                                             &
       size(b)+1 /= n  .or.                                             &
       size(c)   /= n  .or.                                             &
       size(r)   /= n       ) then
      print *, "internal error: in splines[diag5]:"
      print *, "                wrong array dimensions"
      stop
   end if

   ! forward-substitution

   gam = c(1)
   if( gam == 0.0_PREC ) then
      print *, "internal error: in splines[diag5]:"
      print *, "                rewrite equations"
      stop
   end if
   del(1) = b(1)/gam
   u(1) = r(1)/gam

   eta(1) = a(1)/gam
   gam = c(2) - b(1)*del(1)
   del(2) = ( b(2)-b(1)*eta(1) )/gam
   u(2) = ( r(2)-b(1)*u(1) )/gam

   do i = 3, n-1
      eta(i-1) = a(i-1)/gam
      bet = b(i-1) - a(i-2)*del(i-2)
      gam = c(i) - a(i-2)*eta(i-2) - bet*del(i-1)
      if( gam == 0.0_PREC ) then
         print *, "internal error: in splines[diag5]:"
         print *, "                failed"
         stop
      end if
      del(i) = ( b(i)-bet*eta(i-1) )/gam
      u(i) = ( r(i)-a(i-2)*u(i-2)-bet*u(i-1) )/gam
   end do

   bet = b(n-1) - a(n-2)*del(n-2)
   gam = c(n) - a(n-2)*eta(n-2) - bet*del(n-1)
   u(n) = ( r(n)-a(n-2)*u(n-2)-bet*u(n-1) )/gam

   ! back-substitution

   u(n-1) = u(n-1) - del(n-1)*u(n)

   do i = n-2, 1, -1
      u(i) = u(i) - del(i)*u(i+1) - eta(i)*u(i+2)
   end do

end subroutine diag5_sym_double
!_______________________________________________________________________
!
subroutine cub_spl_std_double( x, y, y_sec, kbeg, BCbeg, kend, BCend )

   real(kind=PREC), dimension(:), intent(in)  :: x, y
   real(kind=PREC), dimension(:), intent(out) :: y_sec
   integer,                       intent(in)  :: kbeg, kend
   real(kind=PREC),               intent(in)  :: BCbeg, BCend
   !------ API end ------

   real(kind=PREC), dimension(size(x))   ::  e     ! diagonale principale
   real(kind=PREC), dimension(size(x)-1) ::  f     ! sous-diagonale

   real(kind=PREC), dimension(size(x))   ::  h     ! intervalles
   integer :: i, n

   !------ end of declarations -- execution starts hereafter  ------

   n = size(x)

   if( n < 2 ) then
      print *, "internal error: in splines[cub_spl]:"
      print *, "                n must be >= 2"
      stop
   end if

   ! cas trivial

   if( n == 2 ) then
      call trivial( x, y, y_sec, kbeg, BCbeg, kend, BCend )
      return
   end if

   ! cas général

   h(2:n) = x(2:n) - x(1:n-1)

   ! remplissage des tableaux

   do i = 2, n-1
      e(i) = (h(i+1)+h(i))/3.0_PREC
   end do

   do i = 2, n-2
      f(i) = h(i+1)/6.0_PREC
   end do

   do i = 2, n-1
      y_sec(i) = (y(i+1)-y(i))/h(i+1) - (y(i)-y(i-1))/h(i)
   end do

   if( kbeg == 1 ) then
      e(2) = e(2) - h(2)/12.0_PREC
      y_sec(2) = y_sec(2) + BCbeg/2.0_PREC - (y(2)-y(1))/2.0_PREC/h(2)
   else if( kbeg == 3 ) then
      e(2) = e(2) + h(2)/6.0_PREC
   else
      y_sec(2) = y_sec(2) - BCbeg*h(2)/6.0_PREC
   end if

   if( kend == 1 ) then
      e(n-1) = e(n-1) - h(n)/12.0_PREC
      y_sec(n-1) = y_sec(n-1) - BCend/2.0_PREC                          &
                   + (y(n)-y(n-1))/2.0_PREC/h(n)
   else if( kend == 3 ) then
      e(n-1) = e(n-1) + h(n)/6.0_PREC
   else
      y_sec(n-1) = y_sec(n-1) - BCend*h(n)/6.0_PREC
   end if

   ! appel de la routine diag3

   call diag3( f(2:n-2), e(2:n-1), y_sec(2:n-1), y_sec(2:n-1) )

   if( kbeg == 1 ) then
      y_sec(1) = -y_sec(2)/2.0_PREC + 3.0_PREC*(y(2)-y(1))/h(2)**2      &
                 - 3.0_PREC*BCbeg/h(2)
   else if( kbeg == 3 ) then
      y_sec(1) = y_sec(2)
   else
      y_sec(1) = BCbeg
   end if

   if( kend == 1 ) then
      y_sec(n) = -y_sec(n-1)/2.0_PREC - 3.0_PREC*(y(n)-y(n-1))/h(n)**2  &
                 + 3.0_PREC*BCend/h(n)
   else if( kend == 3 ) then
      y_sec(n) = y_sec(n-1)
   else
      y_sec(n) = BCend
   end if

contains

   subroutine trivial( x, y, y_sec, kbeg, BCbeg, kend, BCend )

      real(kind=PREC), dimension(2), intent(in)  :: x, y
      real(kind=PREC), dimension(2), intent(out) :: y_sec
      integer,                       intent(in)  :: kbeg, kend
      real(kind=PREC),               intent(in)  :: BCbeg, BCend

      real(kind=PREC) :: h

      h = x(2) - x(1)

      if( kbeg == 1 ) then
         if( kend == 1 ) then
            y_sec(1) = 2.0_PREC*(-2.0_PREC*BCbeg*h-h*BCend-3.0_PREC*y(1)   &
                                 +3.0_PREC*y(2))/h**2
            y_sec(2) = -2.0_PREC*(-BCbeg*h-2.0_PREC*h*BCend-3.0_PREC*y(1)  &
                                  +3.0_PREC*y(2))/h**2
         else
            y_sec(1) = (-h**2*BCend-6.0_PREC*BCbeg*h-6.0_PREC*y(1)+        &
                        6.0_PREC*y(2))/h**2/2.0_PREC
            y_sec(2) = BCend
         end if
      else
         y_sec(1) = BCbeg
         if( kend == 1 ) then
            y_sec(2) = -(BCbeg*h**2-6.0_PREC*h*BCend-6.0_PREC*y(1)+        &
                         6.0_PREC*y(2))/h**2/2.0_PREC
         else
            y_sec(2) = BCend
         end if
      end if

   end subroutine trivial

end subroutine cub_spl_std_double
!_______________________________________________________________________
!
subroutine cub_spl_fit_double( x, y, y_sec, kbeg, BCbeg, kend, BCend,   &
                               p, y_fit )

   real(kind=PREC), dimension(:), intent(in)  :: x, y
   real(kind=PREC), dimension(:), intent(out) :: y_sec
   integer,                       intent(in)  :: kbeg, kend
   real(kind=PREC),               intent(in)  :: BCbeg, BCend
   real(kind=PREC), dimension(:), intent(in)  :: p
   real(kind=PREC), dimension(:), intent(out) :: y_fit
   !------ API end ------

   real(kind=PREC), dimension(size(x))   ::  e     ! diag. principale
   real(kind=PREC), dimension(size(x)-1) ::  f     ! sous-diagonale
   real(kind=PREC), dimension(size(x)-2) ::  g     ! sous(2)-diagonale

   real(kind=PREC), dimension(size(x))   ::  h     ! intervalles
   integer :: i, n

   !------ end of declarations -- execution starts hereafter  ------

   n = size(x)

   if( n < 2 ) then
      print *, "internal error: in splines[cub_spl]:"
      print *, "                n must be >= 2"
      stop
   end if

   ! cas trivial

   if( n == 2 ) then
      call trivial( x, y, y_sec, kbeg, BCbeg, kend, BCend, p, y_fit )
      return
   end if

   ! cas général

   h(2:n) = x(2:n) - x(1:n-1)

   ! remplissage des tableaux -- partie interpolation

   do i = 2, n-1
      e(i) = (h(i)+h(i+1))/3.0_PREC
   end do

   do i = 1, n-1
      f(i) = h(i+1)/6.0_PREC
   end do

   y_sec(1) = (y(2)-y(1))/h(2)
   do i = 2, n-1
      y_sec(i) = (y(i+1)-y(i))/h(i+1) - (y(i)-y(i-1))/h(i)
   end do

   ! remplissage des tableaux -- partie approximation

   e(1) = 2.0_PREC*f(1) + (1.0_PREC/p(1)+1.0_PREC/p(2))/h(2)**2
   f(1) = f(1) - 1.0_PREC/h(2)**2/p(1)                                  &
              - (1.0_PREC/h(2)+1.0_PREC/h(3))/h(2)/p(2)

   do i = 2, n-1
      e(i) = e(i) + 1.0_PREC/h(i)**2/p(i-1)                             &
                  + (1.0_PREC/h(i)+1.0_PREC/h(i+1))**2/p(i)             &
                  + 1.0_PREC/h(i+1)**2/p(i+1)
   end do

   do i = 2, n-2
      f(i) = f(i) - (1.0_PREC/h(i)+1.0_PREC/h(i+1))/h(i+1)/p(i)         &
                  - (1.0_PREC/h(i+1)+1.0_PREC/h(i+2))/h(i+1)/p(i+1)
   end do

   do i = 1, n-2
      g(i) = 1.0_PREC/h(i+1)/h(i+2)/p(i+1)
   end do

   e(n) = 2.0_PREC*f(n-1) + (1.0_PREC/p(n-1)+1.0_PREC/p(n))/h(n)**2
   f(n-1) = f(n-1) - (1.0_PREC/h(n-1)+1.0_PREC/h(n))/h(n)/p(n-1)        &
                   - 1.0_PREC/h(n)**2/p(n)
   y_sec(n) = -(y(n)-y(n-1))/h(n)

   if( kbeg == 1 ) then
      y_sec(1) = y_sec(1) - BCbeg
      if( kend == 1 ) then
         y_sec(n) = y_sec(n) + BCend

         ! BC : y' et y'

         call diag5( g(1:n-2), f(1:n-1), e(1:n), y_sec(1:n),            &
                     y_sec(1:n) )
      else
         y_sec(n) = BCend
         y_sec(n-1) = y_sec(n-1) - f(n-1)*BCend
         y_sec(n-2) = y_sec(n-2) - g(n-2)*BCend

         ! BC : y' et y''

         if( n == 3 ) then
            call diag3( f(1:n-2), e(1:n-1), y_sec(1:n-1), y_sec(1:n-1) )
         else
            call diag5( g(1:n-3), f(1:n-2), e(1:n-1), y_sec(1:n-1),     &
                        y_sec(1:n-1) )
         end if
      end if
   else
      y_sec(1) = BCbeg
      y_sec(2) = y_sec(2) - f(1)*BCbeg
      y_sec(3) = y_sec(3) - g(1)*BCbeg
      if( kend == 1 ) then
         y_sec(n) = y_sec(n) + BCend

         ! BC : y'' et y'

         if( n == 3 ) then
            call diag3( f(2:n-1), e(2:n), y_sec(2:n), y_sec(2:n) )
         else
            call diag5( g(2:n-2), f(2:n-1), e(2:n), y_sec(2:n),         &
                        y_sec(2:n) )
         end if
      else
         y_sec(n) = BCend
         y_sec(n-1) = y_sec(n-1) - f(n-1)*BCend
         y_sec(n-2) = y_sec(n-2) - g(n-2)*BCend

         ! BC : y'' et y''

         if( n == 3 ) then
            y_sec(2) = y_sec(2) / e(2)
         else if( n == 4 ) then
            call diag3( f(2:n-2), e(2:n-1), y_sec(2:n-1), y_sec(2:n-1) )
         else
            call diag5( g(2:n-3), f(2:n-2), e(2:n-1), y_sec(2:n-1),     &
                        y_sec(2:n-1) )
         end if
      end if
   end if

   ! calcul des valeurs lissées

   y_fit(1) = y(1) + (y_sec(1)-y_sec(2))/h(2)/p(1)
   do i = 2, n-1
      y_fit(i) = y(i) - ( (y_sec(i-1)-y_sec(i))/h(i)                    &
                        + (y_sec(i+1)-y_sec(i))/h(i+1) )/p(i)
   end do
   y_fit(n) = y(n) + (y_sec(n)-y_sec(n-1))/h(n)/p(n)

contains

   subroutine trivial( x, y, y_sec, kbeg, BCbeg, kend, BCend, p, y_fit )

      real(kind=PREC), dimension(2), intent(in)  :: x, y
      real(kind=PREC), dimension(2), intent(out) :: y_sec
      integer,                       intent(in)  :: kbeg, kend
      real(kind=PREC),               intent(in)  :: BCbeg, BCend
      real(kind=PREC), dimension(2), intent(in)  :: p
      real(kind=PREC), dimension(2), intent(out) :: y_fit

      real(kind=PREC) :: h

      h = x(2) - x(1)

      if( kbeg == 1 ) then
         if( kend == 1 ) then
            y_sec(1) = 2.0_PREC/h*(-6.0_PREC*BCbeg*p(2)-6.0_PREC*p(1)*     &
                       BCbeg-2.0_PREC*BCbeg*p(1)*h**3*p(2)+6.0_PREC*BCend* &
                       p(2)+6.0_PREC*p(1)*BCend-BCend*p(1)*h**3*p(2)-      &
                       3.0_PREC*p(2)*h**2*p(1)*y(1)+3.0_PREC*p(1)*h**2*    &
                       p(2)*y(2))/(12.0_PREC*p(2)+12.0_PREC*p(1)+p(1)*h**3 &
                       *p(2))
            y_sec(2) = -2.0_PREC/h*(-BCbeg*p(1)*h**3*p(2)-2.0_PREC*BCend*  &
                       p(1)*h**3*p(2)-3.0_PREC*p(2)*h**2*p(1)*y(1)+        &
                       3.0_PREC*p(1)*h**2*p(2)*y(2)+6.0_PREC*BCbeg*p(2)+   &
                       6.0_PREC*p(1)*BCbeg-6.0_PREC*BCend*p(2)-6.0_PREC*   &
                       p(1)*BCend)/(12.0_PREC*p(2)+12.0_PREC*p(1)+p(1)*    &
                       h**3*p(2))
            y_fit(1) = (-6.0_PREC*BCbeg*h*p(2)-6.0_PREC*h*BCend*p(2)+      &
                       12.0_PREC*p(2)*y(2)+12.0_PREC*p(1)*y(1)+p(2)*h**3*  &
                       p(1)*y(1))/(12.0_PREC*p(2)+12.0_PREC*p(1)+p(1)*h**3 &
                       *p(2))
            y_fit(2) = (12.0_PREC*p(2)*y(2)+6.0_PREC*p(1)*h*BCbeg+6.0_PREC &
                       *p(1)*h*BCend+p(1)*h**3*p(2)*y(2)+12.0_PREC*p(1)*   &
                       y(1))/(12.0_PREC*p(2)+12.0_PREC*p(1)+p(1)*h**3*p(2))


         else
            y_sec(1) = -(3.0_PREC*BCend*p(2)+3.0_PREC*BCend*p(1)+h**3*     &
                       BCend*p(1)*p(2)+6.0_PREC*BCbeg*p(1)*h**2*p(2)+      &
                       6.0_PREC*p(2)*h*p(1)*y(1)-6.0_PREC*p(1)*h*p(2)*     &
                       y(2))/(3.0_PREC*p(2)+3.0_PREC*p(1)+p(1)*h**3*p(2))/2
            y_sec(2) = BCend
            y_fit(1) = (-3.0_PREC*BCbeg*h*p(2)+3.0_PREC*p(2)*y(2)+3.0_PREC &
                       *p(1)*y(1)+p(2)*h**3*p(1)*y(1))/(3.0_PREC*p(2)+     &
                       3.0_PREC*p(1)+p(1)*h**3*p(2))
            y_fit(2) = (3.0_PREC*p(2)*y(2)+3.0_PREC*p(1)*h*BCbeg+p(1)*h**3 &
                       *p(2)*y(2)+3.0_PREC*p(1)*y(1))/(3.0_PREC*p(2)+      &
                       3.0_PREC*p(1)+p(1)*h**3*p(2))
         end if
      else
         y_sec(1) = BCbeg
         if( kend == 1 ) then
            y_sec(2) = -(3.0_PREC*BCbeg*p(2)+3.0_PREC*BCbeg*p(1)+BCbeg*    &
                       h**3*p(1)*p(2)-6.0_PREC*BCend*p(1)*h**2*p(2)-       &
                       6.0_PREC*p(2)*h*p(1)*y(1)+6.0_PREC*p(1)*h*p(2)*     &
                       y(2))/(3.0_PREC*p(2)+3.0_PREC*p(1)+p(1)*h**3*p(2))  &
                       /2.0_PREC
            y_fit(1) = (-3.0_PREC*h*BCend*p(2)+3.0_PREC*p(2)*y(2)+3.0_PREC &
                       *p(1)*y(1)+p(2)*h**3*p(1)*y(1))/(3.0_PREC*p(2)+     &
                       3.0_PREC*p(1)+p(1)*h**3*p(2))
            y_fit(2) = (3.0_PREC*p(2)*y(2)+3.0_PREC*p(1)*h*BCend+p(1)*h**3 &
                       *p(2)*y(2)+3.0_PREC*p(1)*y(1))/(3.0_PREC*p(2)+      &
                       3.0_PREC*p(1)+p(1)*h**3*p(2))
         else
            y_sec(2) = BCend
            y_fit(1) = y(1)
            y_fit(2) = y(2)
         end if
      end if

   end subroutine trivial

end subroutine cub_spl_fit_double
!_______________________________________________________________________
!
subroutine cyclic3_sym_double( a, b, r, x )

   real(kind=PREC), dimension(:), intent(in) :: a, b, r
   real(kind=PREC), dimension(:), intent(out):: x
   !------ API end ------

   ! Système cyclique à 3 diagonales symétrique :
   !
   !    a(1:n)   : sous-diagonale
   !    b(1:n)   :      diagonale principale
   !    a(1:n)   :  sur-diagonale
   !
   !    r(1:n)   : vecteur second membre (right)
   !    x(1:n)   : vecteur solution      (unknown)
   !
   ! rem. 1 : les vecteurs : a, b, r  ne sont pas modifiés
   !
   ! rem. 2 : on peut calculer u directement dans r, l'appel est alors
   !          call PROGRAM_NAME( a, b, r, r )

   integer                                   :: n
   real(kind=PREC)                           :: alpha, fact, gamma
   real(kind=PREC), dimension(size(x))       :: bb, u, z

   !------ end of declarations -- execution starts hereafter  ------

   n = size(x)

   if( size(a)   /= n  .or.                                             &
       size(b)   /= n  .or.                                             &
       size(r)   /= n       ) then
      print *, "internal error: in splines[cyclic3]:"
      print *, "                wrong array dimensions"
      stop
   end if

   ! alpha : terme des coins inf. gauche et sup. droit

   alpha = a(n)

   gamma = -b(1)

   bb(1) = b(1) - gamma
   bb(2:n-1) = b(2:n-1)
   bb(n) = b(n) - alpha*alpha/gamma

   call diag3( a(1:n-1), bb, r, x )

   u(1) = gamma
   u(2:n-1) = 0.0_PREC
   u(n) = alpha

   call diag3( a(1:n-1), bb, u, z )
   fact = (x(1)+alpha*x(n)/gamma)/(1.0_PREC+z(1)+alpha*z(n)/gamma)
   x = x - fact*z

   end subroutine cyclic3_sym_double
!_______________________________________________________________________
!
subroutine cyclic5_sym_double( a, b, c, r, x )

   real(kind=PREC), dimension(:), intent(in) :: a, b, c, r
   real(kind=PREC), dimension(:), intent(out):: x
   !------ API end ------

   ! Système cyclique à 5 diagonales symétrique :
   !
   !    a(1:n) : sous(2)-diagonale
   !    b(1:n) :    sous-diagonale
   !    c(1:n) :         diagonale principale
   !    b(1:n) :     sur-diagonale
   !    a(1:n) :  sur(2)-diagonale
   !
   !    r(1:n)   : vecteur second membre (right)
   !    x(1:n)   : vecteur solution      (unknown)
   !
   ! rem. 1 : les vecteurs : a, b, c, r  ne sont pas modifiés
   !
   ! rem. 2 : on peut calculer u directement dans r, l'appel est alors
   !          call PROGRAM_NAME( a, b, c, r, r )

   integer                                   :: n, i
   real(kind=PREC)                           :: det
   real(kind=PREC), dimension(size(x))       :: cc, u, z1, z2
   real(kind=PREC), dimension(size(x)-1)     :: bb
   real(kind=PREC) :: alpha, gamma, epsilon
   real(kind=PREC), dimension(2,2) :: m, h
   real(kind=PREC), dimension(2) :: xi, theta

   !------ end of declarations -- execution starts hereafter  ------

   n = size(x)

   if( size(a) /= n  .or.                                               &
       size(b) /= n  .or.                                               &
       size(c) /= n  .or.                                               &
       size(r) /= n       ) then
      print *, "internal error: in splines[cyclic5]:"
      print *, "                wrong array dimensions"
      stop
   end if

   !-- alpha, gamma, ... : cf. convention dans ./maple/Woodbury3.mws

   alpha = b(n)
   gamma = a(n-1)
   epsilon = a(n)

   !-- correction des 3 diagonales

   bb(1) = b(1) - epsilon
   bb(2:n-2) = b(2:n-2)
   bb(n-1) = b(n-1) - gamma

   cc(1) = c(1) - alpha
   cc(2:n-1) = c(2:n-1)
   cc(n) = c(n) - alpha

   !-- résolution du problème de base

   call diag5( a(1:n-2), bb, cc, r, x )

   !-- 1er pb auxiliaire

   u(1) = alpha
   u(2) = epsilon
   u(3:n-2) = 0.0_PREC
   u(n-1) = gamma
   u(n) = alpha

   call diag5( a(1:n-2), bb, cc, u, z1 )

   !-- 2nd pb auxiliaire

   u(1) = 1.0_PREC
   u(2:n-1) = 0.0_PREC
   u(n) = 1.0_PREC

   call diag5( a(1:n-2), bb, cc, u, z2 )

   !-- construction de la matrice : M = I + V^t * Z

   m(1,1) = 1.0_PREC + z1(1) + z1(n)
   m(1,2) =            z2(1) + z2(n)
   m(2,1) =            epsilon*z1(2) + gamma*z1(n-1)
   m(2,2) = 1.0_PREC + epsilon*z2(2) + gamma*z2(n-1)

   !-- déterminant

   det = m(1,1)*m(2,2) - m(1,2)*m(2,1)
   if( det == 0.0_PREC ) then
     print *, "internal error: in splines[cyclic5:]:"
     print *, "                det = 0."
     stop
   end if

   !-- inversion directe H = M^(-1)

   h(1,1) =  m(2,2)/det
   h(1,2) = -m(1,2)/det
   h(2,1) = -m(2,1)/det
   h(2,2) =  m(1,1)/det

   xi(1) = x(1) + x(n)
   xi(2) = epsilon*x(2) + gamma*x(n-1)

   theta(1) = h(1,1)*xi(1) + h(1,2)*xi(2)
   theta(2) = h(2,1)*xi(1) + h(2,2)*xi(2)

   do i = 1, n
      x(i) = x(i) - z1(i)*theta(1) - z2(i)*theta(2)
   end do

end subroutine cyclic5_sym_double
!_______________________________________________________________________
!
subroutine cub_spl_per_double( x, y, y_sec )

   real(kind=PREC), dimension(:), intent(in)  :: x, y
   real(kind=PREC), dimension(:), intent(out) :: y_sec
   !------ API end ------

   real(kind=PREC), dimension(size(x)-1) ::  b     ! diag. principale
   real(kind=PREC), dimension(size(x)-1) ::  a     ! sous-diagonale

   real(kind=PREC), dimension(size(x))   ::  h     ! intervalles
   integer :: i, n

   !------ end of declarations -- execution starts hereafter  ------

   n = size(x)

   if( n < 3 ) then
      print *, "internal error: in splines[cub_spl]:"
      print *, "                n must be >= 3"
      stop
   end if

   h(2:n) = x(2:n) - x(1:n-1)

   ! remplissage des tableaux

   a(1) = h(2)/6.0_PREC
   b(1) = h(n)/3.0_PREC + h(2)/3.0_PREC
   y_sec(1) = (y(2)-y(1))/h(2) - (y(1)-y(n-1))/h(n)

   do i = 2, n-1
      a(i) = h(i+1)/6.0_PREC
      b(i) = h(i)/3.0_PREC + h(i+1)/3.0_PREC
      y_sec(i) = (y(i+1)-y(i))/h(i+1) - (y(i)-y(i-1))/h(i)
   end do

   call cyclic3( a, b, y_sec(1:n-1), y_sec(1:n-1) )

   y_sec(n) = y_sec(1)

end subroutine cub_spl_per_double
!_______________________________________________________________________
!
subroutine cub_spl_per_fit_double( x, y, y_sec, p, y_fit )

   real(kind=PREC), dimension(:), intent(in)  :: x, y
   real(kind=PREC), dimension(:), intent(out) :: y_sec
   real(kind=PREC), dimension(:), intent(in)  :: p
   real(kind=PREC), dimension(:), intent(out) :: y_fit
   !------ API end ------

   real(kind=PREC), dimension(size(x)-1) ::  e     ! diag. principale
   real(kind=PREC), dimension(size(x)-1) ::  f     ! sous-diagonale
   real(kind=PREC), dimension(size(x)-1) ::  g     ! sous(2)-diagonale

   real(kind=PREC), dimension(size(x))   ::  h     ! intervalles
   integer :: i, n

   !------ end of declarations -- execution starts hereafter  ------

   n = size(x)

   if( n < 3 ) then
      print *, "internal error: in splines[cub_spl]:"
      print *, "                n must be >= 3"
      stop
   end if

   h(2:n) = x(2:n) - x(1:n-1)

   ! remplissage des tableaux -- partie interpolation

   e(1) = (h(n)+h(2))/3.0_PREC
   do i = 2, n-1
      e(i) = (h(i)+h(i+1))/3.0_PREC
   end do

   f(1) = h(2)/6.0_PREC
   do i = 2, n-1
      f(i) = h(i+1)/6.0_PREC
   end do

   y_sec(1) = (y(2)-y(1))/h(2) - (y(1)-y(n-1))/h(n)
   do i = 2, n-1
      y_sec(i) = (y(i+1)-y(i))/h(i+1) - (y(i)-y(i-1))/h(i)
   end do

   ! remplissage des tableaux -- partie approximation

   e(1) = e(1) + 1.0_PREC/h(n)**2/p(n-1)                                &
               + (1.0_PREC/h(n)+1.0_PREC/h(2))**2/p(1)                  &
               + 1.0_PREC/h(2)**2/p(2)
   f(1) = f(1) - (1.0_PREC/h(n)+1.0_PREC/h(2))/h(2)/p(1)                &
               - (1.0_PREC/h(2)+1.0_PREC/h(3))/h(2)/p(2)
   g(1) = 1.0_PREC/h(2)/h(3)/p(2)

   do i = 2, n-2
      e(i) = e(i) + 1.0_PREC/h(i)**2/p(i-1)                             &
                  + (1.0_PREC/h(i)+1.0_PREC/h(i+1))**2/p(i)             &
                  + 1.0_PREC/h(i+1)**2/p(i+1)
      f(i) = f(i) - (1.0_PREC/h(i)+1.0_PREC/h(i+1))/h(i+1)/p(i)         &
                  - (1.0_PREC/h(i+1)+1.0_PREC/h(i+2))/h(i+1)/p(i+1)
      g(i) = 1.0_PREC/h(i+1)/h(i+2)/p(i+1)
   end do

   e(n-1) = e(n-1) + 1.0_PREC/h(n-1)**2/p(n-2)                          &
                   + (1.0_PREC/h(n-1)+1.0_PREC/h(n))**2/p(n-1)          &
                   + 1.0_PREC/h(n)**2/p(1)
   f(n-1) = f(n-1) - (1.0_PREC/h(n-1)+1.0_PREC/h(n))/h(n)/p(n-1)        &
                   - (1.0_PREC/h(n)+1.0_PREC/h(2))/h(n)/p(1)
   g(n-1) = 1.0_PREC/h(n)/h(2)/p(1)

   ! appel de la routine cyclic5

   if( n == 3 ) then
      y_sec(1) = -6.0_PREC*p(1)*h(2)*h(3)*p(2)*(-y(2)+y(1))/(12.0_PREC  &
                 *p(1)*h(2)+12.0_PREC*p(1)*h(3)+p(1)*h(2)**2*h(3)**2*   &
                 p(2)+12.0_PREC*p(2)*h(3)+12.0_PREC*p(2)*h(2))
      y_sec(2) = 6.0_PREC*p(1)*h(2)*h(3)*p(2)*(-y(2)+y(1))/(12.0_PREC*  &
                 p(1)*h(2)+12.0_PREC*p(1)*h(3)+p(1)*h(2)**2*h(3)**2*    &
                 p(2)+12.0_PREC*p(2)*h(3)+12.0_PREC*p(2)*h(2))
   else
      call cyclic5( g(1:n-1), f(1:n-1), e(1:n-1), y_sec(1:n-1), y_sec(1:n-1) )
   end if

   y_sec(n) = y_sec(1)

   ! calcul des valeurs lissées

   y_fit(1) = y(1) - (1.0_PREC/h(n)*(y_sec(n-1)-y_sec(1))               &
                   + 1.0_PREC/h(2)*(y_sec(2)-y_sec(1)))/p(1)
   do i = 2, n-1
     y_fit(i) = y(i) - (1.0_PREC/h(i)*(y_sec(i-1)-y_sec(i))             &
                     + 1.0_PREC/h(i+1)*(y_sec(i+1)-y_sec(i)))/p(i)
   end do
   y_fit(n) = y_fit(1)

end subroutine cub_spl_per_fit_double
!_______________________________________________________________________
!
function som_spl_double( x, y, y_sec,                                   &
                         x_beg, x_end )

   real(kind=PREC), dimension(:), intent(in) :: x, y, y_sec
   real(kind=PREC),               intent(in) :: x_beg, x_end
   real(kind=PREC) :: som_spl_double
   !------ API end ------

   real(kind=PREC) :: a, b, sign, tmp
   real(kind=PREC) :: int, inta, intb, hi, hj, hk, ax2, axi2, bx2, bxj2
   integer :: ibeg, iend, i, j, k

   !------ end of declarations -- execution starts hereafter  ------

   ibeg = 1
   iend = size(x)
   a = x_beg
   b = x_end

   !-- test du sens d'intégration

   sign = 1.0_PREC
   if( b < a ) then
      tmp = a
      a = b
      b = tmp
      sign = -sign
   else if(a == b) then
      som_spl_double = 0.0_PREC
      return
   end if

   !-- recherche des intervalles contenant a et b

   if( a < x(ibeg+1) ) then
      i = ibeg + 1
   else if( x(iend) <= a ) then
      i = iend
   else
      do k = ibeg+1, iend
         if( x(k-1) <= a .and. a < x(k) ) i = k
      end do
   end if

   if( b < x(ibeg+1) ) then
      j = ibeg + 1
   else if( x(iend) <= b ) then
      j = iend
   else
      do k = ibeg+1, iend
         if( x(k-1) <= b .and. b < x(k) ) j = k
      end do
   end if

   !-- intégrale de a à x(i)

   hi   = x(i)-x(i-1)
   ax2  = (a-x(i-1))*(a-x(i-1))
   axi2 = (x(i)-a)*(x(i)-a)
   inta = y_sec(i)*(hi**4-ax2**2)/(hi*24.0_PREC)                        &
        + (y(i)-y_sec(i)*hi**2/6.0_PREC)*(hi**2-ax2)/(hi*2.0_PREC)      &
        + y_sec(i-1)*axi2**2/(hi*24.0_PREC)                             &
        + (y(i-1)-y_sec(i-1)*hi**2/6.0_PREC)*axi2/(hi*2.0_PREC)

   !-- intégrale de x(j-1) à b

   hj   = x(j)-x(j-1)
   bx2  = (b-x(j-1))*(b-x(j-1))
   bxj2 = (x(j)-b)*(x(j)-b)
   intb = y_sec(j-1)*(hj**4-bxj2**2)/(hj*24.0_PREC)                     &
        + (y(j-1)-y_sec(j-1)*hj**2/6.0_PREC)*(hj**2-bxj2)/(hj*2.0_PREC) &
        + y_sec(j)*bx2**2/(hj*24.0_PREC)                                &
        + (y(j)-y_sec(j)*hj**2/6.0_PREC)*bx2/(hj*2.0_PREC)

   !-- intégrale complète. trois cas :
   !      (i)   a et b sont dans le même intervalle (i=j)
   !      (ii)  a et b sont dans des intervalles voisins (i=j-1)
   !      (iii) a et b sont dans des intervalles disjoints (i<j-1)

   if(i == j) then
      hi = x(i)-x(i-1)
      int = inta + intb - ( (y(i)+y(i-1))*hi/2.0_PREC                   &
                           -(y_sec(i)+y_sec(i-1))*hi**3/24.0_PREC )
   else
      int = inta + intb
      if(i+1 <= j-1) then
         do k=i+1,j-1
            hk = x(k)-x(k-1)
            int = int + (y(k)+y(k-1))*hk/2.0_PREC                       &
                 - (y_sec(k)+y_sec(k-1))*(hk**3)/24.0_PREC
         end do
      end if
   end if

   som_spl_double = sign * int

end function som_spl_double
!_______________________________________________________________________
!
subroutine cub_spl_driver_double(x, y, y_sec,                           &
                                 k_beg, BC_beg, k_end, BC_end,          &
                                 weight, y_fit, period, err)

   real(kind=PREC), dimension(:), intent(in)            :: x, y
   real(kind=PREC), dimension(:), intent(out)           :: y_sec
   integer,                       intent(in),  optional :: k_beg, k_end
   real(kind=PREC),               intent(in),  optional :: BC_beg, BC_end
   real(kind=PREC), dimension(:), intent(in),  optional :: weight
   real(kind=PREC), dimension(:), intent(out), optional :: y_fit
   logical,                       intent(in),  optional :: period
   integer,                       intent(out), optional :: err
   !------ API end ------

   real(kind=PREC) :: BCbeg, BCend
   integer         :: ibeg, iend, kbeg, kend
   logical         :: periodic

   !------ end of declarations -- execution starts hereafter  ------

   !-- tests des arguments optionnels...

   if( present(k_beg) ) then
      kbeg = k_beg
   else
      kbeg = 2
   end if

   if( present(k_end) ) then
      kend = k_end
   else
      kend = 2
   end if

   if( present(BC_beg) ) then
      BCbeg = BC_beg
   else
      BCbeg = 0.0_PREC
   end if

   if( present(BC_end) ) then
      BCend = BC_end
   else
      BCend = 0.0_PREC
   end if

   if( present(period) ) then
      periodic = period
   else
      periodic = .false.
   end if

   if( present(err) ) then
      err = 0
   end if

   !-- tests sur la dimension des tableaux (args. obligatoires)...

   if( size(y) /= size(x) ) then
      if( present(err) ) then
         err = -1
      else
         print *, "internal error: in splines[cub_spl]:"
         print *, "                dimension ERROR between 'x' and 'y'"
         stop
      end if
   end if
   if( size(y_sec) /= size(x) ) then
      if( present(err) ) then
         err = -2
      else
         print *, "internal error: in splines[cub_spl]:"
         print *, "                dimension ERROR between 'x' and 'y_sec'"
         stop
      end if
   end if
   if( kbeg < 1 .or. 3 < kbeg ) then
      if( present(err) ) then
         err = -3
      else
         print *, "internal error: in splines[cub_spl]:"
         print *, "                bad argument value for 'k_beg'"
         stop
      end if
   end if
   if( kend < 1 .or. 3 < kend ) then
      if( present(err) ) then
         err = -4
      else
         print *, "internal error: in splines[cub_spl]:"
         print *, "                bad argument value for 'k_end'"
         stop
      end if
   end if

   ibeg = 1
   iend = size(x)

   !--------------------------------------------------------------------
   if( present(weight) ) then
      if( size(weight) /= size(x) ) then
         if( present(err) ) then
            err = -5
         else
            print *, "internal error: in splines[cub_spl]:"
            print *, "                dimension ERROR between 'x' and 'weight'"
            stop
         end if
      end if
      if( .not. present(y_fit) ) then
         if( present(err) ) then
            err = -6
         else
            print *, "internal error: in splines[cub_spl]:"
            print *, "                'y_fit' and 'weight' must be together present."
            stop
         end if
      end if
      if( size(y_fit) /= size(x) ) then
         if( present(err) ) then
            err = -7
         else
            print *, "internal error: in splines[cub_spl]:"
            print *, "                dimension ERROR between 'x' and 'y_fit'"
            stop
         end if
      end if
      if( periodic ) then
         if( present(k_beg)  .or. present(k_end) .or.                   &
              present(BC_beg) .or. present(BC_end)     ) then
            if( present(err) ) then
               err = 1
            else
               print *, "Warning: in splines[cub_spl]:"
               print *, "         B.C. arguments cannot be present,"
               print *, "         in case of periodic spline. (B.C. ignored)"
            end if
         end if
         call cub_spl_per_fit(x, y, y_sec, weight, y_fit)
      else
         call cub_spl_fit(x, y, y_sec, kbeg, BCbeg, kend, BCend,        &
                          weight, y_fit)
      end if
   else if( periodic ) then
      if( present(k_beg)  .or. present(k_end) .or.                     &
           present(BC_beg) .or. present(BC_end)     ) then
         if( present(err) ) then
            err = 2
         else
            print *, "Warning: in splines[cub_spl]:"
            print *, "         B.C. arguments cannot be present,"
            print *, "         in case of periodic spline. (B.C. ignored)"
         end if
      end if
      call cub_spl_per(x, y, y_sec)
   else
      if( kbeg == 3 .and. BCbeg /= 0.0_PREC ) then
         print *, "internal error: in splines[cub_spl]:"
         print *, "                bad argument value for 'BC_beg'"
         stop
      end if
      if( kend == 3 .and. BCend /= 0.0_PREC ) then
         print *, "internal error: in splines[cub_spl]:"
         print *, "                bad argument value for 'BC_end'"
         stop
      end if
      call cub_spl_std(x, y, y_sec, kbeg, BCbeg, kend, BCend)
   end if

end subroutine cub_spl_driver_double
!_______________________________________________________________________
!
subroutine int_spl_double( x, y, y_sec, t,                              &
                           y_0, y_1, y_2, any_t, period,                &
                           bin_search, backward, err,                   &
                           out_of_range )

   real(kind=PREC), intent(in)            :: x(:), y(:), y_sec(:), t
   real(kind=PREC), intent(out), optional :: y_0, y_1, y_2
   logical,         intent(in),  optional :: any_t, period,             &
                                             bin_search, backward
   integer,         intent(out), optional :: err
   real(kind=PREC), intent(in),  optional :: out_of_range
   !------ API end ------

   integer, save   :: i_save
   integer         :: i_old, i, ibeg, iend, i_inf, i_sup
   real(kind=PREC) :: hi, tau1, tau2, tt
   logical         :: anyt, periodic, binary_search, backward_direction

   !------ end of declarations -- execution starts hereafter  ------

   !-- tests des arguments optionnels...

   if( present(any_t) ) then
      anyt = any_t
   else
      anyt = .false.
   end if

   if( present(period) ) then
      periodic = period
   else
      periodic = .false.
   end if

   if( present(err) ) then
      err = 0
   end if

   if( present(bin_search) ) then
      binary_search = bin_search
   else
      binary_search = .false.
   end if

   if( present(backward) ) then
      backward_direction = backward
   else
      backward_direction = .false.
   end if

   !-- check array size...

   if( size(y) /= size(x) ) then
      if( present(err) ) then
         err = -1
      else
         print *, "internal error: in splines[int_spl] : "
         print *, "                dimension error between 'x' and 'y'"
         stop
      end if
   end if

   if( size(y_sec) /= size(x) ) then
      if( present(err) ) then
         err = -2
      else
         print *, "internal error: in splines[int_spl] : "
         print *, "                dimension error between 'x' and 'y_sec'"
         stop
      end if
   end if

   ibeg = 1
   iend = size(x)

   !-- for the periodic case, reduction of the argument t

   tt = t
   if( anyt .and. periodic ) then
      if( tt < x(ibeg) ) then
         tt = tt + int((x(iend)-tt)/(x(iend)-x(ibeg)))*(x(iend)-x(ibeg))
      else if( tt > x(iend) ) then
         tt = tt - int((tt-x(ibeg))/(x(iend)-x(ibeg)))*(x(iend)-x(ibeg))
      end if
   end if

   !-- searching for t s.t. it is in [x(i-1),x(i)]

   if( tt < x(ibeg) ) then
      if( .not. anyt ) then
         if( present(out_of_range) ) then
            if( present(y_0) ) then
               y_0 = out_of_range
            end if
            if( present(y_1) ) then
               y_1 = out_of_range
            end if
            if( present(y_2) ) then
               y_2 = out_of_range
            end if
         else
            if( present(err) ) then
               err = -3
            else
               print *, "internal error: in splines[int_spl] : "
               print *, "                argument 't' is < x(ibeg)"
               stop
            end if
         end if
      end if
      i = ibeg + 1
   else if( tt == x(ibeg) ) then
      i = ibeg + 1
   else if( tt > x(iend) ) then
      if( .not. anyt ) then
         if( present(out_of_range) ) then
            if( present(y_0) ) then
               y_0 = out_of_range
            end if
            if( present(y_1) ) then
               y_1 = out_of_range
            end if
            if( present(y_2) ) then
               y_2 = out_of_range
            end if
         else
            if( present(err) ) then
               err = -4
            else
               print *, "internal error: in splines[int_spl] : "
               print *, "                argument 't' is > x(iend)"
               stop
            end if
         end if
      end if
      i = iend
   else if( tt == x(iend) ) then
      i = iend
   else if( binary_search ) then

      i_inf = ibeg
      i_sup = iend
      do
         if( i_inf + 1 == i_sup ) then
            i = i_sup
            go to 1
         end if
         i = ( i_inf + i_sup ) / 2
         if( tt > x(i) ) then
            i_inf = i
         else
            i_sup = i
         end if
      end do

   else if( backward_direction ) then

      !-- case of backward incremental search

      i_old = max(ibeg + 1,i_save)
      i_old = min(i_old,iend)

      if( tt > x(i_old)) then
         i_old = iend
      end if

      do i = i_old, ibeg + 1, -1
         if( tt > x(i-1) ) go to 1
      end do

   else

      !-- case of forward incremental search (default)

      i_old = max(ibeg + 1,i_save)
      i_old = min(i_old,iend)

      if( tt < x(i_old-1)) then
         i_old = ibeg + 1
      end if

      do i = i_old, iend
         if( tt < x(i) ) go to 1
      end do

   end if

1  i_save = i

   hi = x(i) - x(i-1)
   tau1 = (tt-x(i-1)) / hi
   tau2 = 1.0_PREC - tau1

   if( present(y_0) ) then
      y_0 =   y_sec(i-1) * hi**2 * tau2**3 / 6.0_PREC                   &
            + y_sec(i)   * hi**2 * tau1**3 / 6.0_PREC                   &
            + (y(i-1) - y_sec(i-1) * hi**2 / 6.0_PREC ) * tau2          &
            + (y(i)   - y_sec(i)   * hi**2 / 6.0_PREC ) * tau1
   end if

   if( present(y_1) ) then
      y_1 = - y_sec(i-1)*hi/2.0_PREC*tau2**2                            &
            + y_sec(i)*hi/2.0_PREC*tau1**2                              &
            + (y(i)-y(i-1))/hi - (y_sec(i)-y_sec(i-1))*hi/6.0_PREC
   end if

   if( present(y_2) ) then
      y_2 =   y_sec(i-1)*tau2 + y_sec(i)*tau1
   end if

   if( present(y_0) .or. present(y_1) .or. present(y_2) ) then
      return
   else
      if( present(err) ) then
         err = 1
      else
         print *, "warning: in splines[int_spl] : "
         print *, "         nothing calculated!"
      end if
   end if

end subroutine int_spl_double
!_______________________________________________________________________
!
function som_spl_driver_double( x, y, y_sec,                            &
                                x_beg, x_end, any_x, period, err )

   real(kind=PREC), dimension(:), intent(in)            :: x, y, y_sec
   real(kind=PREC),               intent(in), optional  :: x_beg, x_end
   logical,                       intent(in), optional  :: any_x, period
   integer,                       intent(out), optional :: err
   real(kind=PREC) :: som_spl_driver_double
   !------ API end ------

   real(kind=PREC) :: a, b, period_som = 0.0_PREC
   integer :: ibeg, iend, m_a = 0, m_b = 0
   logical :: anyx, periodic, period_som_calc = .false.

   !------ end of declarations -- execution starts hereafter  ------

   ibeg = 1
   iend = size(x)

   !-- tests des arguments optionnels...

   if( present(x_beg) ) then
      a = x_beg
   else
      a = x(ibeg)
   end if

   if( present(x_end) ) then
      b = x_end
   else
      b = x(iend)
   end if

   if( present(any_x) ) then
      anyx = any_x
   else
      anyx = .false.
   end if

   if( present(period) ) then
      periodic = period
   else
      periodic = .false.
   end if

   if( present(err) ) then
      err = 0
   end if

   !-- tests sur la dimension des tableaux...

   if( size(y) /= size(x) ) then
      if( present(err) ) then
         err = -1
      else
         print *, "internal error: in splines[som_spl] : "
         print *, "                bad dimension between 'x' and 'y'"
         stop
      end if
   end if

   if( size(y_sec) /= size(x) ) then
      if( present(err) ) then
         err = -2
      else
         print *, "internal error: in splines[som_spl] : "
         print *, "                bad dimension between 'x' and 'y_sec'"
         stop
      end if
   end if

   !-- tests des limites de l'intégrale...

   if( a < x(ibeg) ) then
      if( .not. anyx ) then
         if( present(err) ) then
            err = -3
         else
            print *, "internal error: in splines[som_spl] : "
            print *, "                argument 'x_beg' is < x(ibeg)"
            stop
         end if
      else if( periodic ) then

         if( .not. period_som_calc ) then
            period_som = som_spl_double( x, y, y_sec, x(ibeg), x(iend) )
            period_som_calc = .true.
         end if

         !-- réduction de l'argument a

         m_a = int((x(iend)-a)/(x(iend)-x(ibeg)))
         a = a + m_a*(x(iend)-x(ibeg))

      end if
   else if( x(iend) < a ) then
      if( .not. anyx ) then
         if( present(err) ) then
            err = -4
         else
            print *, "internal error: in splines[som_spl] : "
            print *, "                argument 'x_beg' is > x(iend)"
            stop
         end if
      else if( periodic ) then

         if( .not. period_som_calc ) then
            period_som = som_spl_double( x, y, y_sec, x(ibeg), x(iend) )
            period_som_calc = .true.
         end if

         !-- réduction de l'argument a

         m_a = int((a-x(ibeg))/(x(iend)-x(ibeg)))
         a = a - m_a*(x(iend)-x(ibeg))

      end if
   end if

   if( b < x(ibeg) ) then
      if( .not. anyx ) then
         if( present(err) ) then
            err = -5
         else
            print *, "internal error: in splines[som_spl] : "
            print *, "                argument 'x_end' is < x(ibeg)"
            stop
         end if
      else if( periodic ) then

         if( .not. period_som_calc ) then
            period_som = som_spl_double( x, y, y_sec, x(ibeg), x(iend) )
            period_som_calc = .true.
         end if

         !-- réduction de l'argument b

         m_b = int((x(iend)-b)/(x(iend)-x(ibeg)))
         b = b + m_b*(x(iend)-x(ibeg))

      end if
   else if( x(iend) < b ) then
      if( .not. anyx ) then
         if( present(err) ) then
            err = -6
         else
            print *, "internal error: in splines[som_spl] : "
            print *, "                argument 'x_end' is > x(iend)"
            stop
         end if
      else if( periodic ) then

         if( .not. period_som_calc ) then
            period_som = som_spl_double( x, y, y_sec, x(ibeg), x(iend) )
            period_som_calc = .true.
         end if

         !-- réduction de l'argument b

         m_b = int((b-x(ibeg))/(x(iend)-x(ibeg)))
         b = b - m_b*(x(iend)-x(ibeg))

      end if
   end if

   som_spl_driver_double = som_spl_double( x, y, y_sec, a, b ) +        &
                           (m_a+m_b)*period_som

end function som_spl_driver_double
!_______________________________________________________________________
!
end module splines
