module mod_ddebd2

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

   use mod_core, only: MF_DOUBLE,                                       &
                       NAMED_EQN_PRESENCE, NAMED_EQN_PTR,               &
                       NAMED_VAR_PRESENCE, NAMED_VAR_PTR,               &
                       search_index_in_eqn_groups, search_index_in_var_groups

   use mod_mfaux, only: crc_32_bits

   use mod_mfdebug, only: mf_message_displayed, muesli_trace,           &
                          MF_ADDRESS, MF_UF_LONG

   implicit none

!-----------------------------------------------------------------------
!                           global variables
!-----------------------------------------------------------------------

#include "umf4_f90wrapper.inc"

   ! contains data for communication between the following routines
   ! of different DE solvers, in the case where the Jacobian is a
   ! sparse matrix. Allow also communication for dense data, for
   ! debugging purpose.

   integer, save :: nnz
   ! these arrays describes the sparse Jacobian matrix (square)
   ! (allocated by 'DLSOD', deallocated by 'ddebd2_free')
   real(kind=MF_DOUBLE), save, allocatable :: pd(:)
   integer, save, allocatable :: ipd(:), jpd(:)

   ! A 32-bit checksum in order to verify that structure of the
   ! jacobian has not been changed between calls.
   ! (used to economize symbolic analysis in 'sp_lu_fact')
   ! The checksum for ipd(:) only should be sufficient because this
   ! last vector being sorted, it is impossible to get another matrix
   ! having a different jpd(:)!
   integer, save :: jac_ipd_crc = 0
   logical, save :: spjac_const_struct,                                 &
                    first_call_to_sp_lu = .true.

   ! following indexes are 0-based for C-routines
   integer, save, allocatable :: ipd_C(:), jpd_C(:)

   ! declarations for UMFPACK
   integer(kind=MF_ADDRESS), save :: numeric = 0, symbolic = 0 ! LU handle
   real(kind=MF_DOUBLE), save :: control(20), infos(90)

   ! use to retrieve the initial condition computed by DDASSL (YPRIME)
   ! at a time close to (but not exactly equal to) the initial time.
   real(kind=MF_DOUBLE), save, allocatable :: yp_init_ddassl(:)

   ! use to retrieve all the times of stepping
   integer, save :: n_times
   logical, save :: save_times = .false.
   real(kind=MF_DOUBLE), save, allocatable :: times_solve(:)

   ! use to retrieve the orders at each time step
   integer, save :: n_orders
   logical, save :: save_orders = .false.
   real(kind=MF_DOUBLE), save, allocatable :: orders_solve(:)

   ! Sizes (real and integer parts) of common DDEBD1 (see DDEBDF family)
   integer, parameter :: len_dbl_ddebd1 = 218, len_int_ddebd1 = 36

   ! Used in ODE and DAE integrators
   real(kind=MF_DOUBLE), allocatable, save :: yy(:), yyp(:), rwork(:)
   integer,              allocatable, save :: iwork(:)

   ! for debugging purpose, when the dense Jacobian is singular
   ! (see ddajac.F).
   logical, save :: ddajac_failed = .false., ddajac_must_quit = .false.
   real(kind=MF_DOUBLE), save, allocatable :: transp_Jac(:,:), dgeco_z(:)
   real(kind=MF_DOUBLE), allocatable :: ddajac_FJAC(:,:)
   real, allocatable :: ddajac_err(:,:)

   ! for debugging purpose, keep the current time of integration
   real(kind=MF_DOUBLE), save :: current_time = -1.0d0,                 &
                                 current_percent = -1.0d0

   integer :: unit_monitor_y = -1, unit_monitor_yp = -1
   integer, allocatable :: monitoryind(:), monitorypind(:)
   character(len=80) :: format_monitor_y = "", format_monitor_yp = ""
   logical :: monitor_pause

   ! Jacobian matrix is symmetric and positive definite.
   ! Thus flag is set in ODE/DAE driver routines.
   logical, save :: jac_symm_pos_def

   ! declarations for CHOLMOD
   integer(kind=MF_ADDRESS), save :: c_addr = 0, LL_addr = 0, AA_addr = 0

contains
!_______________________________________________________________________
!
   subroutine sp_lu_fact( n, rcond )

      integer, intent(in) :: n
      real(kind=MF_DOUBLE), optional :: rcond

      ! Do a sparse LU factorization of the (pd,ipd,jpd) matrix
      ! (CSC storage)

      ! Matrix must be square (nrow = ncol = n) and row-sorted!

      ! (comes from: 'msLU_mfMatFactor')

      integer :: crc, nz_udiag, status
      logical :: new_sparse_struct
      real(kind=MF_DOUBLE), allocatable :: udiag(:)

      !------------------------------------

      if( spjac_const_struct ) then
         ! The user claims that the structure of the sparse matrix is the same
         ! along all the time integration: no need to compute the checksum!
         new_sparse_struct = .false.
      else
         ! computing CRC (32-bit) of the structure of the sparse matrix
         crc = crc_32_bits( ipd(1:nnz) )
         new_sparse_struct = crc /= jac_ipd_crc
      end if

      if( first_call_to_sp_lu .or. new_sparse_struct ) then

         ! set UMFPACK-4 default parameters
         call umf4def_d(control)
         ! print control parameters.
         ! Set control(1) to 1 to print error messages only
         control(1) = 1

         if( symbolic /= 0 ) then
            ! free the symbolic analysis
            call umf4fsym_d(symbolic)
         end if

         ! convert indexes from 1-based (Fortran) to 0-based (C)
         ipd_C(1:nnz) = ipd(1:nnz) - 1
         jpd_C(1:n+1) = jpd(1:n+1) - 1

         ! pre-order and symbolic analysis
         call umf4sym_d( n, n, jpd_C(1), ipd_C(1), pd(1), symbolic,     &
                         control, infos )

         ! check umf4sym error condition
         if( nint(infos(1)) < 0 ) then
            write(0,*) "(MUESLI OdeSolve/DaeSolve:) [BDF/sp_lu_fact:] internal error"
            write(0,"(A,I0)") "   error occurred in umf4sym_d: infos(1) = ", nint(infos(1))
            write(0,*) "  Please report this bug to: Edouard.Canot@univ-rennes.fr"
            if( nint(infos(1)) == -8 ) then ! UMFPACK-version dependent !
               ! UMFPACK_ERROR_invalid_matrix (-8)
               ! (the following description has been adapted to fortran-like 1-based
               !  indexes !)
               write(0,*) "  Number of entries in the matrix is negative, Ap(1) is not 1,"
               write(0,*) "  a column has a negative number of entries, a row index is out of"
               write(0,*) "  bounds, or the columns of input matrix were jumbled (unsorted"
               write(0,*) "  or duplicated entries)."
            end if
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            stop
         end if

         jac_ipd_crc = crc
         first_call_to_sp_lu = .false.

      end if

      if( numeric /= 0 ) then
         ! free the numeric factorization (from a previous call)
         call umf4fnum_d(numeric)
      end if

      ! numeric factorization
      call umf4num_d( n, jpd_C(1), ipd_C(1), pd(1), symbolic, numeric,  &
                      control, infos )

      if( nint(infos(1)) < 0 ) then
         ! check umf4num error condition
         if( nint(infos(1)) == -4 ) then ! UMFPACK-version dependent !
            write(0,*) "(MUESLI OdeSolve/DaeSolve:) [BDF/sp_lu_fact:] internal error"
            write(0,*) "  [UMFPACK_ERROR_invalid_Symbolic_object]"
            write(0,*) "  Please report this bug to: Edouard.Canot@univ-rennes.fr"
         else
            write(0,"(A,I0)") "   error occurred in umf4num_d: infos(1) = ", nint(infos(1))
         end if
         mf_message_displayed = .true.
         call muesli_trace( pause="yes" )
         stop
      else if( nint(infos(1)) > 0 ) then
         ! these are warning
         if( nint(infos(1)) == 1 ) then ! UMFPACK-version dependent !
            write(0,*) "(MUESLI OdeSolve/DaeSolve:) [BDF/sp_lu_fact:] Warning"
            write(0,*) "  [UMFPACK_WARNING_singular_matrix]"
            write(0,*) "  Numeric factorization was successful, but the matrix is"
            write(0,*) "  singular. UMFPACK computed a valid numeric factorization,"
            write(0,*) "  but you will get a divide by zero if you try to solve the"
            write(0,*) "  corresponding linear system."
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
         end if
      end if

      if( present(rcond) ) then

         ! get the NNZ of U_diag
         call umf4nzudiag_d( nz_udiag, numeric, status )

         ! check umf4nzudiag error condition
         if( status /= 0 ) then
            write(0,*) "(MUESLI OdeSolve/DaeSolve:) [BDF/sp_lu_fact:] ERROR"
            write(0,*) "  Cannot get nnz of U_diag in order to estimate reciprocal condition number!"
            write(0,*) "  [in umf4nzudiag_d]."
            write(0,*) "  -> rcond set to 1.0d0"
            rcond = 1.0d0 ! value such that nothing is done in ddjac
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            return
         end if

         if( nz_udiag < n ) then
            rcond = 0.0d0
         else
            allocate( udiag(nz_udiag) )

            call umf4getudiag_d( udiag(1), numeric, status )

            ! check umf4getudiag error condition
            if( status /= 0 ) then
               write(0,*) "(MUESLI OdeSolve/DaeSolve:) [BDF/sp_lu_fact:] ERROR"
               write(0,*) "  Cannot get U_diag in order to estimate reciprocal condition number!"
               write(0,*) "  [in umf4getudiag_d]."
               write(0,*) "  -> rcond set to 1.0d0"
               rcond = 1.0d0 ! value such that nothing is done in ddjac
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               return
            end if

            rcond = minval( abs(udiag) ) / maxval( abs(udiag) )
         end if

      end if

   end subroutine sp_lu_fact
!_______________________________________________________________________
!
   subroutine sp_lu_solv( n, y )

      integer,              intent(in)     :: n
      real(kind=MF_DOUBLE), intent(in out) :: y(*)

      ! Solves the system: PD*x = y
      ! solution is returned in vector y
      !
      ! where (pd,ipd,jpd) is the CSC sparse representation of the
      ! matrix PD

      ! (comes from 'mfLDiv_mfMatFactor')

      ! Declarations for UMFPACK
      integer, save :: sys = 0

      !------------------------------------

      ! solve system, without iterative refinement
      call umf4sol_d( sys, y(1), y(1), numeric,                         &
                      control, infos )

      if( nint(infos(1)) < 0 ) then
         write(0,*)
         write(0,*) "(MUESLI OdeSolve/DaeSolve:) [BDF/sp_lu_solv:] internal error"
         write(0,"(A,I0)") "   error occurred in umf4sol_d: ", nint(infos(1))
         write(0,*) "  Please report this bug to: Edouard.Canot@univ-rennes.fr"
         mf_message_displayed = .true.
         call muesli_trace( pause="yes" )
         stop
      end if

   end subroutine sp_lu_solv
!_______________________________________________________________________
!
   subroutine ddebd2_free()

      ! in case where integration duration is zero, the arrays 'times_solve'
      ! and 'orders_solve' have not been allocated...
      if( allocated(pd) ) then
         deallocate( pd, ipd, jpd, ipd_C, jpd_C )
      end if

      if( symbolic /= 0 ) then
         ! free the symbolic analysis
         call umf4fsym_d(symbolic)
      end if

      if( numeric /= 0 ) then
         ! free the numeric factorization
         call umf4fnum_d(numeric)
      end if

      jac_ipd_crc = 0
      first_call_to_sp_lu = .true.

   end subroutine ddebd2_free
!_______________________________________________________________________
!
   subroutine sp_chol_fact( n )

      integer, intent(in) :: n

      ! Do a sparse Cholesky factorization of the (pd,ipd,jpd) matrix
      ! (CSC storage).
      ! Of course, the matrix must be symmetric and positive definite!

      ! Matrix must be square (nrow = ncol = n) and row-sorted!

      ! (comes from: 'msCholSpSymb')

      integer :: crc, status
      logical :: new_sparse_struct

      !------------------------------------

      if( spjac_const_struct ) then
         ! The user claims that the structure of the sparse matrix is the same
         ! along all the time integration: no need to compute the checksum!
         new_sparse_struct = .false.
      else
         ! computing CRC (32-bit) of the structure of the sparse matrix
         crc = crc_32_bits( ipd(1:nnz) )
         new_sparse_struct = crc /= jac_ipd_crc
      end if

      if( first_call_to_sp_lu .or. new_sparse_struct ) then

         if( c_addr /= 0 ) then
            ! free the symbolic analysis
            call cholmod_free_factor_2( c_addr, LL_addr, AA_addr )
         end if

         ! symbolic analysis
         call cholmod_llt_symb( n, nnz, jpd(1), ipd(1),                 &
                                c_addr, LL_addr, AA_addr )

         jac_ipd_crc = crc
         first_call_to_sp_lu = .false.

      end if

      ! numeric factorization
      call cholmod_llt_num( n, nnz, pd(1), c_addr, LL_addr, AA_addr,    &
                            status )

      ! check whether factorization is ok
      if( status /= 0 ) then
         write(0,*) "(MUESLI OdeSolve/DaeSolve:) [BDF/sp_chol_fact:] ERROR"
         write(0,*) "  in Cholesky-factorization of A by CHOLMOD:"
         write(0,*) "  A is not positive definite!"
         mf_message_displayed = .true.
         call muesli_trace( pause="yes" )
         return
      end if

   end subroutine sp_chol_fact
!_______________________________________________________________________
!
   subroutine sp_chol_solv( n, y )

      integer,              intent(in)     :: n
      real(kind=MF_DOUBLE), intent(in out) :: y(*)

      ! Solves the system: PD*x = y
      ! solution is returned in vector y
      !
      ! (comes from 'mfLDiv_mfMatFactor')

      !------------------------------------

      ! solve system, without iterative refinement
      call cholmod_solve_factor( n, c_addr, LL_addr, y(1), y(1) )

   end subroutine sp_chol_solv
!_______________________________________________________________________
!
   subroutine ddebd2_free2()

      if( allocated(ddajac_FJAC) ) then
         deallocate( ddajac_FJAC )
      end if

      if( allocated(ddajac_err) ) then
         deallocate( ddajac_err )
      end if

   end subroutine ddebd2_free2
!_______________________________________________________________________
!
end module
