!DECK DSTEPS
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 28 jun 2023
!
   SUBROUTINE DSTEPS( DF, NEQN, Y, X, H, EPS, WT, START, HOLD, K,       &
          KOLD, CRASH, PHI, P, YP, PSI, ALPHA, BETA, SIG, V, W, G,      &
          PHASE1, NS, NORND, KSTEPS, TWOU, FOURU, XOLD, KPREV, IVC, IV, &
          KGI, GI )
!***BEGIN PROLOGUE  DSTEPS
!***PURPOSE  Integrate a system of first order ordinary differential
!            equations one step.
!***LIBRARY   SLATEC (DEPAC)
!***CATEGORY  I1A1B
!***TYPE      DOUBLE PRECISION (STEPS-S, DSTEPS-D)
!***KEYWORDS  ADAMS METHOD, DEPAC, INITIAL VALUE PROBLEMS, ODE,
!             ORDINARY DIFFERENTIAL EQUATIONS, PREDICTOR-CORRECTOR
!***AUTHOR  Shampine, L. F., (SNLA)
!           Gordon, M. K., (SNLA)
!             MODIFIED BY H.A. WATTS
!***DESCRIPTION
!
!   Written by L. F. Shampine and M. K. Gordon
!
!   Abstract
!
!   Subroutine  DSTEPS  is normally used indirectly through subroutine
!   DDEABM.  Because  DDEABM  suffices for most problems and is much
!   easier to use, using it should be considered before using  DSTEPS
!   alone.
!
!   Subroutine DSTEPS integrates a system of  NEQN  first order ordinary
!   differential equations one step, normally from X to X+H, using a
!   modified divided difference form of the Adams Pece formulas.  Local
!   extrapolation is used to improve absolute stability and accuracy.
!   The code adjusts its order and step size to control the local error
!   per unit step in a generalized sense.  Special devices are included
!   to control roundoff error and to detect when the user is requesting
!   too much accuracy.
!
!   This code is completely explained and documented in the text,
!   Computer Solution of Ordinary Differential Equations, The Initial
!   Value Problem  by L. F. Shampine and M. K. Gordon.
!   Further details on use of this code are available in "Solving
!   Ordinary Differential Equations with ODE, STEP, and INTRP",
!   by L. F. Shampine and M. K. Gordon, SLA-73-1060.
!
!
!   The parameters represent --
!      DF -- subroutine to evaluate derivatives
!      NEQN -- number of equations to be integrated
!      Y(*) -- solution vector at X
!      X -- independent variable
!      H -- appropriate step size for next step.  Normally determined by
!           code
!      EPS -- local error tolerance
!      WT(*) -- vector of weights for error criterion
!      START -- logical variable set .TRUE. for first step,  .FALSE.
!           otherwise
!      HOLD -- step size used for last successful step
!      K -- appropriate order for next step (determined by code)
!      KOLD -- order used for last successful step
!      CRASH -- logical variable set .TRUE. when no step can be taken,
!           .FALSE. otherwise.
!      YP(*) -- derivative of solution vector at  X  after successful
!           step
!      KSTEPS -- counter on attempted steps
!      TWOU -- 2.*U where U is machine unit roundoff quantity
!      FOURU -- 4.*U where U is machine unit roundoff quantity
!
!   The variables X,XOLD,KOLD,KGI and IVC and the arrays Y,PHI,ALPHA,G,
!   W,P,IV and GI are required for the interpolation subroutine SINTRP.
!   The remaining variables and arrays are included in the call list
!   only to eliminate local retention of variables between calls.
!
!   Input to DSTEPS
!
!      First call --
!
!   The user must provide storage in his calling program for all arrays
!   in the call list, namely
!
!     DIMENSION Y(NEQN),WT(NEQN),PHI(NEQN,16),P(NEQN),YP(NEQN),PSI(12),
!    1  ALPHA(12),BETA(12),SIG(13),V(12),W(12),G(13),GI(11),IV(10)
!
!    **Note**
!
!   The user must also declare  START,  CRASH,  PHASE1  and  NORND
!   logical variables and  DF  an EXTERNAL subroutine, supply the
!   subroutine  DF(X,Y,YP)  to evaluate
!      DY(I)/DX = YP(I) = DF(X,Y(1),Y(2),...,Y(NEQN))
!   and initialize only the following parameters.
!      NEQN -- number of equations to be integrated
!      Y(*) -- vector of initial values of dependent variables
!      X -- initial value of the independent variable
!      H -- nominal step size indicating direction of integration
!           and maximum size of step.  Must be variable
!      EPS -- local error tolerance per step.  Must be variable
!      WT(*) -- vector of non-zero weights for error criterion
!      START -- .TRUE.
!      YP(*) -- vector of initial derivative values
!      KSTEPS -- set KSTEPS to zero
!      TWOU -- 2.*U where U is machine unit roundoff quantity
!      FOURU -- 4.*U where U is machine unit roundoff quantity
!   Define U to be the machine unit roundoff quantity by calling
!   the function routine  D1MACH,  U = D1MACH(4), or by
!   computing U so that U is the smallest positive number such
!   that 1.0+U .GT. 1.0.
!
!   DSTEPS  requires that the L2 norm of the vector with components
!   LOCAL ERROR(L)/WT(L)  be less than  EPS  for a successful step.  The
!   array  WT  allows the user to specify an error test appropriate
!   for his problem.  For example,
!      WT(L) = 1.0  specifies absolute error,
!            = ABS(Y(L))  error relative to the most recent value of the
!                 L-th component of the solution,
!            = ABS(YP(L))  error relative to the most recent value of
!                 the L-th component of the derivative,
!            = MAX(WT(L),ABS(Y(L)))  error relative to the largest
!                 magnitude of L-th component obtained so far,
!            = ABS(Y(L))*RELERR/EPS + ABSERR/EPS  specifies a mixed
!                 relative-absolute test where  RELERR  is relative
!                 error,  ABSERR  is absolute error and  EPS =
!                 MAX(RELERR,ABSERR) .
!
!      Subsequent calls --
!
!   Subroutine  DSTEPS  is designed so that all information needed to
!   continue the integration, including the step size  H  and the order
!   K, is returned with each step.  With the exception of the step
!   size, the error tolerance, and the weights, none of the parameters
!   should be altered.  The array  WT  must be updated after each step
!   to maintain relative error tests like those above.  Normally the
!   integration is continued just beyond the desired endpoint and the
!   solution interpolated there with subroutine  SINTRP .  If it is
!   impossible to integrate beyond the endpoint, the step size may be
!   reduced to hit the endpoint since the code will not take a step
!   larger than the  H  input.  Changing the direction of integration,
!   i.e., the sign of  H, requires the user set  START = .TRUE. before
!   calling  DSTEPS  again.  This is the only situation in which  START
!   should be altered.
!
!   Output from DSTEPS
!
!      Successful Step --
!
!   The subroutine returns after each successful step with  START  and
!   CRASH  set .FALSE. .  X  represents the independent variable
!   advanced one step of length  HOLD  from its value on input and  Y
!   the solution vector at the new value of  X .  All other parameters
!   represent information corresponding to the new  X  needed to
!   continue the integration.
!
!      Unsuccessful Step --
!
!   When the error tolerance is too small for the machine precision,
!   the subroutine returns without taking a step and  CRASH = .TRUE. .
!   An appropriate step size and error tolerance for continuing are
!   estimated and all other information is restored as upon input
!   before returning.  To continue with the larger tolerance, the user
!   just calls the code again.  A restart is neither required nor
!   desirable.
!
!***REFERENCES  L. F. Shampine and M. K. Gordon, Solving ordinary
!                 differential equations with ODE, STEP, and INTRP,
!                 Report SLA-73-1060, Sandia Laboratories, 1973.
!***ROUTINES CALLED  D1MACH, DHSTRT
!***REVISION HISTORY  (YYMMDD)
!   740101  DATE WRITTEN
!   920501  Last modification. For details see original file.
!***END PROLOGUE  DSTEPS
!
      use mod_mfdebug, only: MF_NUMERICAL_CHECK, muesli_trace
      use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe

      IMPLICIT NONE

      INTEGER KPREV, IVC, KGI, JV, IV

      INTEGER I, IFAIL, IM1, IP1, IQ, J, K, KM1, KM2, KNEW,             &
            KOLD, KP1, KP2, KSTEPS, L, LIMIT1, LIMIT2, NEQN, NS, NSM2,  &
            NSP1, NSP2
      DOUBLE PRECISION ABSH, ALPHA, BETA, BIG,                          &
            EPS, ERK, ERKM1, ERKM2, ERKP1, ERR,                         &
            FOURU, G, GI, GSTR, H, HNEW, HOLD, P, P5EPS, PHI, PSI, R,   &
            REALI, REALNS, RHO, ROUND, SIG, TAU, TEMP1,                 &
            TEMP2, TEMP3, TEMP4, TEMP5, TEMP6, TWO, TWOU, U, V, W, WT,  &
            X, XOLD, Y, YP
      LOGICAL START,CRASH,PHASE1,NORND
      DIMENSION Y(*),WT(*),PHI(NEQN,16),P(*),YP(*),PSI(12),             &
        ALPHA(12),BETA(12),SIG(13),V(12),W(12),G(13),GI(11),IV(10)
      DIMENSION TWO(13),GSTR(13)
      EXTERNAL DF
      SAVE TWO, GSTR
!
      DATA TWO(1),TWO(2),TWO(3),TWO(4),TWO(5),TWO(6),TWO(7),TWO(8),     &
           TWO(9),TWO(10),TWO(11),TWO(12),TWO(13)                       &
           /2.0D0,4.0D0,8.0D0,16.0D0,32.0D0,64.0D0,128.0D0,256.0D0,     &
            512.0D0,1024.0D0,2048.0D0,4096.0D0,8192.0D0/
      DATA GSTR(1),GSTR(2),GSTR(3),GSTR(4),GSTR(5),GSTR(6),GSTR(7),     &
           GSTR(8),GSTR(9),GSTR(10),GSTR(11),GSTR(12),GSTR(13)          &
           /0.5D0,0.0833D0,0.0417D0,0.0264D0,0.0188D0,0.0143D0,0.0114D0,&
            0.00936D0,0.00789D0,0.00679D0,0.00592D0,0.00524D0,0.00468D0/
!
!     communication with slatec/drkfs.f, slatec/dfehl.f, slatec/dhstrt.f
!                        slatec/ddes.f, slatec/dsteps.f
!                        slatec/dlsod.f, slatec/dstod.f
!                        [fml_funfun]/msOdeSolve_JacDF
      integer :: nb_step, nb_deriv
      double precision :: dt_min, dt_max
      common /slatec_odesolve_1/ dt_min, dt_max, nb_step, nb_deriv

      integer :: iflag, istat, jj

      integer :: i_eqn_group, i_group_eqn

!-----------------------------------------------------------------------

!       ***     BEGIN BLOCK 0     ***
!   Check if step size or error tolerance is too small for machine
!   precision.  If first step, initialize PHI array and estimate a
!   starting step size.
!                   ***
!
!   If step size is too small, determine an acceptable one
!
!***First executable statement  DSTEPS
      CRASH = .TRUE.
      IF( ABS(H) >= FOURU*ABS(X) ) GO TO 5
      H = SIGN(FOURU*ABS(X),H)
      RETURN
    5 P5EPS = 0.5D0*EPS

!   If error tolerance is too small, increase it to an acceptable value
      ROUND = 0.0D0
      DO L = 1, NEQN
        ROUND = ROUND + (Y(L)/WT(L))**2
      END DO
      ROUND = TWOU*SQRT(ROUND)
      IF( P5EPS >= ROUND ) GO TO 15
      EPS = 2.0D0*ROUND*(1.0D0 + FOURU)
      RETURN
   15 CRASH = .FALSE.
      G(1) = 1.0D0
      G(2) = 0.5D0
      SIG(1) = 1.0D0
      IF( .NOT. START ) GO TO 99

!   Initialize.  compute appropriate step size for first step
      DO L = 1, NEQN
        PHI(L,1) = YP(L)
        PHI(L,2) = 0.0D0
      END DO

      U = D1MACH(4)
      BIG = SQRT(D1MACH(2))
      CALL DHSTRT(DF,NEQN,X,X+H,Y,YP,WT,1,U,BIG,                        &
                  PHI(1,3),PHI(1,4),PHI(1,5),PHI(1,6),H,istat)

      HOLD = 0.0D0
      K = 1
      KOLD = 0
      KPREV = 0
      START = .FALSE.
      PHASE1 = .TRUE.
      NORND = .TRUE.
      IF( P5EPS > 100.0D0*ROUND ) GO TO 99
      NORND = .FALSE.
      DO L = 1, NEQN
        PHI(L,15) = 0.0D0
      END DO
   99 IFAIL = 0
!       ***     END BLOCK 0     ***
!
!       ***     BEGIN BLOCK 1     ***
!   Compute coefficients of formulas for this step.  Avoid computing
!   those quantities again when step size is not changed.
!                   ***
  100 KP1 = K+1
      KP2 = K+2
      KM1 = K-1
      KM2 = K-2

!   NS is the number of DSTEPS taken with size H, including the current
!   one.  When K < NS, no coefficients change
      IF( H /= HOLD ) NS = 0
      IF( NS <= KOLD ) NS = NS+1
      NSP1 = NS+1
      IF( K < NS ) GO TO 199

!   Compute those components of ALPHA(*), BETA(*), PSI(*), SIG(*) WHICH
!   are changed
      BETA(NS) = 1.0D0
      REALNS = NS
      ALPHA(NS) = 1.0D0/REALNS
      TEMP1 = H*REALNS
      SIG(NSP1) = 1.0D0
      IF( K < NSP1 ) GO TO 110
      DO I = NSP1, K
        IM1 = I-1
        TEMP2 = PSI(IM1)
        PSI(IM1) = TEMP1
        BETA(I) = BETA(IM1)*PSI(IM1)/TEMP2
        TEMP1 = TEMP2 + H
        ALPHA(I) = H/TEMP1
        REALI = I
        SIG(I+1) = REALI*ALPHA(I)*SIG(I)
      END DO
  110 PSI(K) = TEMP1

!   Compute coefficients G(*)
!
!   Initialize V(*) and set W(*).
      IF( NS > 1 ) GO TO 120
      DO IQ = 1, K
        TEMP3 = IQ*(IQ+1)
        V(IQ) = 1.0D0/TEMP3
        W(IQ) = V(IQ)
      END DO
      IVC = 0
      KGI = 0
      IF( K == 1 ) GO TO 140
      KGI = 1
      GI(1) = W(2)
      GO TO 140

!   If order was raised, update diagonal part of V(*)
  120 IF( K <= KPREV ) GO TO 130
      IF( IVC == 0 ) GO TO 122
      JV = KP1 - IV(IVC)
      IVC = IVC - 1
      GO TO 123
  122 JV = 1
      TEMP4 = K*KP1
      V(K) = 1.0D0/TEMP4
      W(K) = V(K)
      IF( K /= 2 ) GO TO 123
      KGI = 1
      GI(1) = W(2)
  123 NSM2 = NS-2
      IF( NSM2 < JV ) GO TO 130
      DO J = JV, NSM2
        I = K-J
        V(I) = V(I) - ALPHA(J+1)*V(I+1)
        W(I) = V(I)
      END DO
      IF( I /= 2 ) GO TO 130
      KGI = NS - 1
      GI(KGI) = W(2)

!   Update V(*) and set W(*)
  130 LIMIT1 = KP1 - NS
      TEMP5 = ALPHA(NS)
      DO IQ = 1, LIMIT1
        V(IQ) = V(IQ) - TEMP5*V(IQ+1)
        W(IQ) = V(IQ)
      END DO
      G(NSP1) = W(1)
      IF( LIMIT1 == 1 ) GO TO 137
      KGI = NS
      GI(KGI) = W(2)
  137 W(LIMIT1+1) = V(LIMIT1+1)
      IF( K >= KOLD ) GO TO 140
      IVC = IVC + 1
      IV(IVC) = LIMIT1 + 2

!   Compute the G(*) in the work vector W(*)
  140 NSP2 = NS + 2
      KPREV = K
      IF( KP1 < NSP2 ) GO TO 199
      DO I = NSP2, KP1
        LIMIT2 = KP2 - I
        TEMP6 = ALPHA(I-1)
        DO IQ = 1, LIMIT2
          W(IQ) = W(IQ) - TEMP6*W(IQ+1)
        END DO
        G(I) = W(1)
      END DO
  199 CONTINUE
!       ***     END BLOCK 1     ***
!
!       ***     BEGIN BLOCK 2     ***
!   Predict a solution P(*), evaluate derivatives using predicted
!   solution, estimate local error at order K and errors at orders K,
!   K-1, K-2 as if constant step size were used.
!                   ***
!
!   Increment counter on attempted DSTEPS
      KSTEPS = KSTEPS + 1

!   Change PHI to PHI STAR
      IF( K < NSP1 ) GO TO 215
      DO I = NSP1, K
        TEMP1 = BETA(I)
        DO L = 1, NEQN
          PHI(L,I) = TEMP1*PHI(L,I)
        END DO
      END DO

!   Predict solution and differences
  215 DO L = 1, NEQN
        PHI(L,KP2) = PHI(L,KP1)
        PHI(L,KP1) = 0.0D0
        P(L) = 0.0D0
      END DO
      DO J = 1, K
        I = KP1 - J
        IP1 = I+1
        TEMP2 = G(I)
        DO L = 1, NEQN
          P(L) = P(L) + TEMP2*PHI(L,I)
          PHI(L,I) = PHI(L,I) + PHI(L,IP1)
        END DO
      END DO
      IF( NORND ) GO TO 240
      DO L = 1, NEQN
        TAU = H*P(L) - PHI(L,15)
        P(L) = Y(L) + TAU
        PHI(L,16) = (P(L) - Y(L)) - TAU
      END DO
      GO TO 250
  240 DO L = 1, NEQN
        P(L) = Y(L) + H*P(L)
      END DO
  250 XOLD = X
      X = X + H
      ABSH = ABS(H)
      nb_deriv = nb_deriv + 1
      iflag = 0
      call mf_restore_fpe( )
      CALL DF( X, P, YP, iflag )
      call mf_save_and_disable_fpe( )
      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
         do jj = 1, NEQN
            if( isnan(YP(jj)) ) then
               print "(/,A)", "(MUESLI OdeSolve:) [dsteps] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied DERIV routine."
               print "(20X,A,I0)", "This occured in yprime(k) for k = ", jj
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( jj, i_eqn_group,     &
                                                       i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               return
            end if
         end do
      end if

!   Estimate errors at orders K, K-1, K-2
      ERKM2 = 0.0D0
      ERKM1 = 0.0D0
      ERK = 0.0D0
      DO L = 1, NEQN
        TEMP3 = 1.0D0/WT(L)
        TEMP4 = YP(L) - PHI(L,1)
        IF( KM2 ) 265, 260, 255
  255   ERKM2 = ERKM2 + ((PHI(L,KM1)+TEMP4)*TEMP3)**2
  260   ERKM1 = ERKM1 + ((PHI(L,K)+TEMP4)*TEMP3)**2
  265   ERK = ERK + (TEMP4*TEMP3)**2
      END DO
      IF( KM2 ) 280, 275, 270
  270 ERKM2 = ABSH*SIG(KM1)*GSTR(KM2)*SQRT(ERKM2)
  275 ERKM1 = ABSH*SIG(K)*GSTR(KM1)*SQRT(ERKM1)
  280 TEMP5 = ABSH*SQRT(ERK)
      ERR = TEMP5*(G(K)-G(KP1))
      ERK = TEMP5*SIG(KP1)*GSTR(K)
      KNEW = K

!   Test if order should be lowered
      IF( KM2 ) 299, 290, 285
  285 IF( MAX(ERKM1,ERKM2) <= ERK ) KNEW = KM1
      GO TO 299
  290 IF( ERKM1 <= 0.5D0*ERK ) KNEW = KM1

!   Test if step successful
  299 IF( ERR <= EPS ) GO TO 400
!       ***     END BLOCK 2     ***
!
!       ***     BEGIN BLOCK 3     ***
!   The step is unsuccessful.  Restore  X, PHI(*,*), PSI(*).
!   If third consecutive failure, set order to 1.  If step fails more
!   than three times, consider an optimal step size.  Double error
!   tolerance and return if estimated step size is too small for machine
!   precision.
!                   ***
!
!   Restore X, PHI(*,*) and PSI(*)
      PHASE1 = .FALSE.
      X = XOLD
      DO I = 1, K
        TEMP1 = 1.0D0/BETA(I)
        IP1 = I+1
        DO L = 1, NEQN
          PHI(L,I) = TEMP1*(PHI(L,I) - PHI(L,IP1))
        END DO
      END DO
      IF( K < 2 ) GO TO 320
      DO I = 2, K
        PSI(I-1) = PSI(I) - H
      END DO

!   On third failure, set order to 1.  Thereafter, use optimal step
!   size
  320 IFAIL = IFAIL + 1
      TEMP2 = 0.5D0
      IF( IFAIL - 3 ) 335, 330, 325
  325 IF( P5EPS < 0.25D0*ERK ) TEMP2 = SQRT(P5EPS/ERK)
  330 KNEW = 1
  335 H = TEMP2*H

      K = KNEW
      NS = 0
      IF( ABS(H) >= FOURU*ABS(X) ) GO TO 340
      CRASH = .TRUE.
      H = SIGN(FOURU*ABS(X),H)
      EPS = EPS + EPS
      RETURN
  340 GO TO 100
!       ***     END BLOCK 3     ***
!
!       ***     BEGIN BLOCK 4     ***
!   The step is successful.  Correct the predicted solution, evaluate
!   the derivatives using the corrected solution and update the
!   differences.  Determine best order and step size for next step.
!                   ***
  400 KOLD = K
      HOLD = H

!   Correct and evaluate
      TEMP1 = H*G(KP1)
      IF( NORND ) GO TO 410
      DO L = 1, NEQN
        TEMP3 = Y(L)
        RHO = TEMP1*(YP(L) - PHI(L,1)) - PHI(L,16)
        Y(L) = P(L) + RHO
        PHI(L,15) = (Y(L) - P(L)) - RHO
        P(L) = TEMP3
      END DO
      GO TO 420
  410 DO L = 1, NEQN
        TEMP3 = Y(L)
        Y(L) = P(L) + TEMP1*(YP(L) - PHI(L,1))
        P(L) = TEMP3
      END DO
  420 continue
      nb_deriv = nb_deriv + 1
      iflag = 0
      call mf_restore_fpe( )
      CALL DF( X, Y, YP, iflag )
      call mf_save_and_disable_fpe( )
      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
         do jj = 1, NEQN
            if( isnan(YP(jj)) ) then
               print "(/,A)", "(MUESLI OdeSolve:) [dsteps] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied DERIV routine."
               print "(20X,A,I0)", "This occured in yprime(k) for k = ", jj
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( jj, i_eqn_group,     &
                                                       i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               return
            end if
         end do
      end if

!   Update differences for next step
      DO L = 1, NEQN
        PHI(L,KP1) = YP(L) - PHI(L,1)
        PHI(L,KP2) = PHI(L,KP1) - PHI(L,KP2)
      END DO
      DO I = 1, K
        DO L = 1, NEQN
          PHI(L,I) = PHI(L,I) + PHI(L,KP1)
        END DO
      END DO

!   Estimate error at order K+1 unless:
!     In first phase when always raise order, already decided to lower order,
!     step size not constant so estimate unreliable
      ERKP1 = 0.0D0
      IF( KNEW == KM1 .OR. K == 12 ) PHASE1 = .FALSE.
      IF( PHASE1 ) GO TO 450
      IF( KNEW == KM1 ) GO TO 455
      IF( KP1 > NS ) GO TO 460
      DO L = 1, NEQN
        ERKP1 = ERKP1 + (PHI(L,KP2)/WT(L))**2
      END DO
      ERKP1 = ABSH*GSTR(KP1)*SQRT(ERKP1)

!   Using estimated error at order K+1, determine appropriate order
!   for next step
      IF( K > 1 ) GO TO 445
      IF( ERKP1 >= 0.5D0*ERK ) GO TO 460
      GO TO 450
  445 IF( ERKM1 <= MIN(ERK,ERKP1) ) GO TO 455
      IF( ERKP1 >= ERK .OR. K == 12 ) GO TO 460

!   Here ERKP1 < ERK < MAX(ERKM1,ERKM2) else order would have
!   been lowered in block 2.  Thus order is to be raised
!
!   Raise order
  450 K = KP1
      ERK = ERKP1
      GO TO 460

!   Lower order
  455 K = KM1
      ERK = ERKM1

!   With new order determine appropriate step size for next step
  460 HNEW = H + H
      IF( PHASE1 ) GO TO 465
      IF( P5EPS >= ERK*TWO(K+1) ) GO TO 465
      HNEW = H
      IF( P5EPS >= ERK ) GO TO 465
      TEMP2 = K+1
      R = (P5EPS/ERK)**(1.0D0/TEMP2)
      HNEW = ABSH*MAX(0.5D0,MIN(0.9D0,R))
      HNEW = SIGN(MAX(HNEW,FOURU*ABS(X)),H)
  465 H = HNEW

      if( h < dt_min ) dt_min = h
      if( h > dt_max ) dt_max = h

   END
