!DECK DSLVS
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 1 Mar 2019
!
subroutine DSLVS( WM, IWM, X, TEM )

!***BEGIN PROLOGUE  DSLVS
!***SUBSIDIARY
!***PURPOSE  Subsidiary to DDEBDF
!***LIBRARY   SLATEC
!***TYPE      DOUBLE PRECISION (SLVS-S, DSLVS-D)
!***AUTHOR  Watts, H. A., (SNLA)
!***DESCRIPTION
!
!   DSLVS solves the linear system in the iteration scheme for the
!   integrator package DDEBDF.
!
!***SEE ALSO  DDEBDF
!***ROUTINES CALLED  DGBSL, DGESL
!***COMMON BLOCKS    DDEBD1
!***REVISION HISTORY  (YYMMDD)
!   820301  DATE WRITTEN
!   890531  Changed all specific intrinsics to generic.  (WRB)
!   891214  Prologue converted to Version 4.0 format.  (BAB)
!   900328  Added TYPE section.  (WRB)
!   910722  Updated AUTHOR section.  (ALS)
!   920422  Changed DIMENSION statement.  (WRB)
!***END PROLOGUE  DSLVS
!
    implicit none

    INTEGER :: I, IER, IOWND, IOWNS, IWM(*), JSTART, KFLAG, L, MAXORD,  &
               MEBAND, METH, MITER, ML, MU, N, NFE, NJE, NQ, NQU, NST,  &
               ISPARSE, IUSERCALL, JAC_OUTDATED, info
    DOUBLE PRECISION :: DI, EL0, H, HL0, HMIN, HMXI, HU, PHL0,          &
                        R, ROWND, ROWNS, TEM(*), TN, UROUND, WM(*), X(*)

    COMMON /DDEBD1/ ROWND, ROWNS(210), EL0, H, HMIN, HMXI, HU, TN, UROUND, &
                    IOWND(14), IOWNS(6), IER, JSTART, KFLAG, L, METH,   &
                    MITER, MAXORD, N, NQ, NST, NFE, NJE, NQU, ISPARSE,  &
                    IUSERCALL, JAC_OUTDATED

!   ----------------------------------------------------------------------
!   This routine manages the solution of the linear system arising from
!   a chord iteration.  It is called by DSTOD if MITER /= 0.
!   If MITER is 1 or 2, it calls DGESL to accomplish this.
!   If MITER = 3 it updates the coefficient H*EL0 in the diagonal matrix,
!     and then computes the solution.
!   If MITER is 4 or 5, it calls DGBSL.
!   If MITER equals 6, it calls sp_lu_solv (sparse LU) or sp_chol_solv
!     (sparse Choklesky); in this latter case, WM and IWM are not
!     referenced, because the sparse matrix is saved into the data module
!     'mod_ddebd2'.
!   Communication with DSLVS uses the following variables:
!   WM  = double precision work space containing the inverse diagonal
!         matrix if MITER is 3 and the LU decomposition of the matrix
!         otherwise. Storage of matrix elements starts at WM(3).
!         WM also contains the following matrix-related data:
!           WM(1) = SQRT(UROUND) (Not used here),
!           WM(2) = HL0, the previous value of H*EL0, used if MITER = 3.
!   IWM = integer work space containing pivot information, starting at
!         IWM(21), if MITER is 1, 2, 4 or 5.  IWM also contains the band
!         parameters ML = IWM(1) and MU = IWM(2) if MITER is 4 or 5.
!   X   = the right-hand side vector on input, and the solution vector
!         on output, of length N.
!   TEM = vector of work space of length N, not used in this version.
!   IER = output flag (in COMMON).  IER = 0 if no trouble occurred.
!         IER = -1 if a singular matrix arose with MITER = 3.
!
!   This routine also uses the common variables EL0, H, MITER, and N.
!==============================================================================

   IER = 0

   SELECT CASE( MITER )

   CASE( 1, 2 ) ! Dense matrix
      if( jac_symm_pos_def ) then
         call dpotrs( 'U', N, 1, WM(3), N, X, N, info )
         if( info < 0 ) then
            print *, "(MUESLI:) dslvs: internal error"
            print *, "          illegal parameter in calling dpotrs"
            print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            stop
         end if
      else
         call dgetrs( 'N', N, 1, WM(3), N, IWM(21), X, N, info )
         if( info < 0 ) then
            print *, "(MUESLI:) dslvs: internal error"
            print *, "          illegal parameter in calling dgetrs"
            print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            stop
         end if
      end if

   CASE( 3 ) ! Diagonal approximation
      PHL0 = WM(2)
      HL0 = H*EL0
      WM(2) = HL0
      if( HL0 /=  PHL0 ) then
         R = HL0/PHL0
         do I = 1, N
            DI = 1.0D0 - R*(1.0D0 - 1.0D0/WM(I+2))
            if( ABS(DI) == 0.0D0 ) then
               IER = -1
               return
            end if
            WM(I+2) = 1.0D0/DI
         end do
      end if
      do I = 1, N
         X(I) = WM(I+2)*X(I)
      end do

   CASE( 4, 5 ) ! Banded matrix
      if( jac_symm_pos_def ) then
         MU = IWM(2)
         MEBAND = MU + 1
         CALL dpbtrs( 'U', N, MU, 1, WM(3), MEBAND, X, N, info )
         if( info < 0 ) then
            print *, "(MUESLI:) dslvs: internal error"
            print *, "          illegal parameter in calling dpbtrs"
            print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            stop
         end if
      else
         ML = IWM(1)
         MU = IWM(2)
         MEBAND = 2*ML + MU + 1
         CALL dgbtrs( 'N', N, ML, MU, 1, WM(3), MEBAND, IWM(21), X, N, info )
         if( info < 0 ) then
            print *, "(MUESLI:) dslvs: internal error"
            print *, "          illegal parameter in calling dgbtrs"
            print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            stop
         end if
      end if

   CASE( 6 ) ! Sparse matrix
      if( jac_symm_pos_def ) then
         ! SPARSE SOLVE via CHOLMOD
         CALL sp_chol_solv( N, X )
      else
         ! SPARSE SOLVE via UMFPACK
         CALL sp_lu_solv( N, X )
      end if

   END SELECT

end
