!DECK DRKFS
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 8 nov 2017
!
   SUBROUTINE DRKFS( DF, NEQ, T, Y, TOUT, INFO, RTOL, ATOL, IDID, H,    &
         TOLFAC, YP, F1, F2, F3, F4, F5, YS, TOLD, DTSIGN, U26, RER,    &
         INIT, KSTEPS, KOP, IQUIT, STIFF, NONSTF, NTSTEP, NSTIFS )

      use mod_mfdebug, only: MF_NUMERICAL_CHECK, muesli_trace
      use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe
      use mod_core, only: MF_DOUBLE, STDOUT,                            &
          sec_2_hms, go_home_on_term, msFlush, msPause

      IMPLICIT NONE

      EXTERNAL :: DF

      INTEGER, intent(in) :: NEQ

      INTEGER :: INFO(15), IDID, INIT, KSTEPS, KOP, IQUIT,              &
            NTSTEP, NSTIFS

!     TOUT may be modified by the routine (End Condition)
      DOUBLE PRECISION :: TOUT

      DOUBLE PRECISION :: T, Y(*), RTOL(*), ATOL(*), H, TOLFAC,         &
            YP(*), F1(*), F2(*), F3(*), F4(*), F5(*), YS(*), TOLD,      &
            DTSIGN, U26, RER

      LOGICAL :: STIFF, NONSTF

!***BEGIN PROLOGUE  DRKFS
!***SUBSIDIARY
!***PURPOSE  Subsidiary to DDERKF
!***LIBRARY   SLATEC
!***TYPE      DOUBLE PRECISION (DERKFS-S, DRKFS-D)
!***AUTHOR  Watts, H. A., (SNLA)
!***DESCRIPTION
!
!     Fehlberg Fourth-Fifth Order Runge-Kutta Method
! **********************************************************************
!
!     DRKFS integrates a system of first order ordinary differential
!     equations as described in the comments for DDERKF.
!
!     The arrays YP, F1, F2, F3, F4, F5 and YS (of length at least NEQ)
!     appear in the call list for variable dimensioning purposes.
!
!     The variables H, TOLFAC, TOLD, DTSIGN, U26, RER, INIT, KSTEPS, KOP
!     IQUIT, STIFF, NONSTF, NTSTEP and NSTIFS are used internally by the
!     code and appear in the call list to eliminate local retention of
!     variables between calls. Accordingly, these variables and the arra
!     YP should not be altered.
!     Items of possible interest are
!         H  - An appropriate step size to be used for the next step
!         TOLFAC - Factor of change in the tolerances
!         YP - Derivative of solution vector at T
!         KSTEPS - Counter on the number of steps attempted
!
! **********************************************************************
!
!***SEE ALSO  DDERKF
!***ROUTINES CALLED  D1MACH, DFEHL, DHSTRT, DHVNRM, XERMSG
!***REVISION HISTORY  (YYMMDD)
!   820301  DATE WRITTEN
!   910722  Last modification. For details see original file.
!***END PROLOGUE  DRKFS
!
      INTEGER :: K, KTOL, MXKOP, MXSTEP, NATOLP, NRTOLP
      DOUBLE PRECISION :: A, BIG, DT, DY, EE, EEOET,    &
            ES, ESTIFF, ESTTOL, ET, HMIN, REMIN, S, TOL, U, UTE, YAVG

      LOGICAL :: HFAILD, OUTPUT
      CHARACTER(len=8) :: XERN1
      CHARACTER(len=16) :: XERN3, XERN4

!     ..................................................................

!       A fifth order method will generally not be capable of delivering
!       accuracies near limiting precision on computers with long
!       wordlengths. To protect against limiting precision difficulties
!       arising from unreasonable accuracy requests, an appropriate
!       tolerance threshold remin is assigned for this method. This
!       value should not be changed across different machines.

      SAVE REMIN, MXSTEP, MXKOP
      DATA REMIN /1.0D-12/

!     ..................................................................

!       The expense of solving the problem is monitored by counting the
!       number of  steps attempted. when this exceeds  mxstep, the
!       counter is reset to zero and the user is informed about possible
!       excessive work.

      DATA MXSTEP /500/

!     ..................................................................

!       Inefficiency caused by too frequent output is monitored by
!       counting the number of step sizes which are severely shortened
!       due solely to the choice of output points. When the number of
!       abuses exceed mxkop, the counter is reset to zero and the user
!       is informed about possible misuse of the code.

      DATA MXKOP /100/

      integer :: times_current_length
      double precision, allocatable :: vector_tmp(:)

!     communication with slatec/drkfs.f, slatec/dfehl.f, slatec/dhstrt.f
!                        slatec/ddes.f, slatec/dsteps.f
!                        slatec/dlsod.f, slatec/dstod.f
!                        [fml_funfun]/msOdeSolve_JacDF
      integer :: nb_step, nb_deriv
      double precision :: dt_min, dt_max
      common /slatec_odesolve_1/ dt_min, dt_max, nb_step, nb_deriv

      logical :: print_progress, disp_times
      character(len=80) :: string
      real(kind=MF_DOUBLE) :: t_0, t_end, pc_fact, percent
      real(kind=MF_DOUBLE) :: total_time, left_time, rem_time
      integer :: clock_rate, clock_init, clock, kk
      integer :: hrs_1, min_1, sec_1, hrs_2, min_2, sec_2
      common /slatec_odesolve_progress/ print_progress, disp_times,     &
             t_0, t_end, pc_fact, clock_rate, clock_init

      integer :: iflag
      integer :: istat

      integer, save :: COUNT_ILL_COND
      integer, parameter :: COUNT_ILL_COND_MAX = 10 ! original value is 10

      logical :: end_condition_found
      double precision :: tend

      integer :: i_eqn_group, i_group_eqn

!     ..................................................................

      end_condition_found = .false.

      IF( INFO(1) == 0 ) THEN
         nb_step = 0
         dt_min =  8.8888D+88
         dt_max = -8.8888D+88
         nb_deriv = 0
         COUNT_ILL_COND = 0

         if( save_times ) then
           allocate( times_solve(500) )
           n_times = 1
           times_solve(1) = T
         end if

! On the first call, perform initialization --
!        Define the machine unit roundoff quantity  U  by calling the
!        function routine  D1MACH. The user must make sure that the
!        values set in D1MACH are relevant to the computer being used.

         U = D1MACH(4)
!        -- Set associated machine dependent parameters
         U26 = 26.0D0*U
         RER = 2.0D0*U + REMIN
!        -- Set termination flag
         IQUIT = 0
!        -- Set initialization indicator
         INIT = 0
!        -- Set counter for impact of output points
         KOP = 0
!        -- Set counter for attempted steps
         KSTEPS = 0
!        -- Set indicators for stiffness detection
         STIFF = .FALSE.
         NONSTF = .FALSE.
!        -- Set step counters for stiffness detection
         NTSTEP = 0
         NSTIFS = 0
!        -- Reset INFO(1) for subsequent calls
         INFO(1) = 1
      ENDIF

!.......................................................................

!        Check validity of input parameters on each entry

      IF (INFO(1) /= 0 .AND. INFO(1) /= 1) THEN
         WRITE (XERN1, '(I8)') INFO(1)
         CALL XERMSG ('SLATEC', 'DRKFS',                                &
            'IN DDERKF, INFO(1) MUST BE SET TO 0 ' //                   &
            'FOR THE START OF A NEW PROBLEM, AND MUST BE SET TO 1 ' //  &
            'FOLLOWING AN INTERRUPTED TASK.  YOU ARE ATTEMPTING TO ' // &
            'CONTINUE THE INTEGRATION ILLEGALLY BY CALLING THE CODE ' //&
            'WITH  INFO(1) = ' // XERN1, 3, 1)
         IDID = -33
      ENDIF

      IF (INFO(2) /= 0 .AND. INFO(2) /= 1) THEN
         WRITE (XERN1, '(I8)') INFO(2)
         CALL XERMSG ('SLATEC', 'DRKFS',                                &
            'IN DDERKF, INFO(2) MUST BE 0 OR 1 ' //                     &
            'INDICATING SCALAR AND VECTOR ERROR TOLERANCES, ' //        &
            'RESPECTIVELY.  YOU HAVE CALLED THE CODE WITH INFO(2) = ' //&
            XERN1, 4, 1)
         IDID = -33
      ENDIF

      IF (INFO(3) /= 0 .AND. INFO(3) /= 1) THEN
         WRITE (XERN1, '(I8)') INFO(3)
         CALL XERMSG ('SLATEC', 'DRKFS',                                &
            'IN DDERKF, INFO(3) MUST BE 0 OR 1 ' //                     &
            'INDICATING THE INTERVAL OR INTERMEDIATE-OUTPUT MODE OF ' //&
            'INTEGRATION, RESPECTIVELY.  YOU HAVE CALLED THE CODE ' //  &
            'WITH  INFO(3) = ' // XERN1, 5, 1)
         IDID = -33
      ENDIF

      IF (NEQ < 1) THEN
         WRITE (XERN1, '(I8)') NEQ
         CALL XERMSG ('SLATEC', 'DRKFS',                                &
            'IN DDERKF, THE NUMBER OF EQUATIONS ' //                    &
            'NEQ MUST BE A POSITIVE INTEGER.  YOU HAVE CALLED THE ' //  &
            'CODE WITH NEQ = ' // XERN1, 6, 1)
         IDID = -33
      ENDIF

      NRTOLP = 0
      NATOLP = 0
      DO K = 1, NEQ
         IF (NRTOLP == 0 .AND. RTOL(K) < 0.D0) THEN
            WRITE (XERN1, '(I8)') K
            WRITE (XERN3, '(1PE15.6)') RTOL(K)
            CALL XERMSG ('SLATEC', 'DRKFS',                             &
               'IN DDERKF, THE RELATIVE ERROR ' //                      &
               'TOLERANCES RTOL MUST BE NON-NEGATIVE.  YOU HAVE ' //    &
               'CALLED THE CODE WITH  RTOL(' // XERN1 // ') = ' //      &
               XERN3 // '.  IN THE CASE OF VECTOR ERROR TOLERANCES, ' //&
               'NO FURTHER CHECKING OF RTOL COMPONENTS IS DONE.', 7, 1)
            IDID = -33
            NRTOLP = 1
         ENDIF

         IF (NATOLP == 0 .AND. ATOL(K) < 0.D0) THEN
            WRITE (XERN1, '(I8)') K
            WRITE (XERN3, '(1PE15.6)') ATOL(K)
            CALL XERMSG ('SLATEC', 'DRKFS',                             &
               'IN DDERKF, THE ABSOLUTE ERROR ' //                      &
               'TOLERANCES ATOL MUST BE NON-NEGATIVE.  YOU HAVE ' //    &
               'CALLED THE CODE WITH  ATOL(' // XERN1 // ') = ' //      &
               XERN3 // '.  IN THE CASE OF VECTOR ERROR TOLERANCES, ' //&
               'NO FURTHER CHECKING OF ATOL COMPONENTS IS DONE.', 8, 1)
            IDID = -33
            NATOLP = 1
         ENDIF

         IF( INFO(2) == 0 ) GO TO 20
         IF( NATOLP > 0 .AND. NRTOLP > 0 ) GO TO 20
      END DO

!     Check some continuation possibilities

   20 IF( INIT /= 0 ) THEN
         IF( T == TOUT ) THEN
            WRITE (XERN3, '(1PE15.6)') T
            CALL XERMSG ('SLATEC', 'DRKFS',                             &
               'IN DDERKF, YOU HAVE CALLED THE ' //                     &
               'CODE WITH  T = TOUT = ' // XERN3 // '$$THIS IS NOT ' // &
               'ALLOWED ON CONTINUATION CALLS.', 9, 1)
            IDID=-33
         ENDIF
!
         IF( T /= TOLD ) THEN
            WRITE (XERN3, '(1PE15.6)') TOLD
            WRITE (XERN4, '(1PE15.6)') T
            CALL XERMSG ('SLATEC', 'DRKFS',                             &
               'IN DDERKF, YOU HAVE CHANGED THE ' //                    &
               'VALUE OF T FROM ' // XERN3 // ' TO ' // XERN4 //        &
               '$$THIS IS NOT ALLOWED ON CONTINUATION CALLS.', 10, 1)
            IDID=-33
         ENDIF
!
         IF( INIT /= 1 ) THEN
            IF( DTSIGN*(TOUT-T) < 0.D0 ) THEN
               WRITE (XERN3, '(1PE15.6)') TOUT
               CALL XERMSG ('SLATEC', 'DRKFS',                          &
                  'IN DDERKF, BY CALLING THE CODE WITH TOUT = ' //      &
                  XERN3 // ' YOU ARE ATTEMPTING TO CHANGE THE ' //      &
                  'DIRECTION OF INTEGRATION.$$THIS IS NOT ALLOWED ' //  &
                  'WITHOUT RESTARTING.', 11, 1)
               IDID=-33
            ENDIF
         ENDIF
      ENDIF

!     Invalid input detected

      IF( IDID == -33 ) THEN
         IF( IQUIT /= -33 ) THEN
            IQUIT = -33
            GOTO 540
         ELSE
            CALL XERMSG ('SLATEC', 'DRKFS',                             &
               'IN DDERKF, INVALID INPUT WAS ' //                       &
               'DETECTED ON SUCCESSIVE ENTRIES.  IT IS IMPOSSIBLE ' //  &
               'TO PROCEED BECAUSE YOU HAVE NOT CORRECTED THE ' //      &
               'PROBLEM, SO EXECUTION IS BEING TERMINATED.', 12, 2)
            RETURN
         ENDIF
      ENDIF

!           ............................................................
!
!                RTOL = ATOL = 0. Is allowed as valid input and
!                interpreted as asking for the most accurate solution
!                possible. In this case, the relative error tolerance
!                RTOL is reset to the smallest value RER which is likely
!                to be reasonable for this method and machine.

            DO K = 1, NEQ
               IF( RTOL(K) + ATOL(K) > 0.0D0 ) GO TO 180
                  RTOL(K) = RER
                  IDID = -2
  180          CONTINUE
!           ...Exit
               IF( INFO(2) == 0 ) GO TO 200
            END DO
  200       CONTINUE

            IF( IDID /= -2 ) GO TO 210

!              RTOL=ATOL=0 on input, so RTOL was changed to a
!                                       small positive value
               TOLFAC = 1.0D0
            GO TO 530
  210       CONTINUE

!                       Branch on status of initialization indicator
!                              INIT=0 means initial derivatives and
!                                  starting step size not yet computed
!                              INIT=1 means starting step size not yet
!                                  computed
!                              INIT=2 means no further initialization
!                                  required

                        IF( INIT == 0 ) GO TO 220
!                    ......Exit
                           IF( INIT == 1 ) GO TO 240
!                 .........Exit
                           GO TO 260
  220                   CONTINUE

!                       ................................................
!
!                            more initialization -- evaluate initial
!                                                -- derivatives

                        INIT = 1
                        A = T
                        nb_deriv = nb_deriv + 1
                        iflag = 0
                        call mf_restore_fpe( )
                        CALL DF( A, Y, YP, iflag )
                        call mf_save_and_disable_fpe( )
      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
         do k = 1, neq
            if( isnan(YP(k)) ) then
               print "(/,A)", "(MUESLI OdeSolve:) [drkfs] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied DERIV routine."
               print "(20X,A,I0)", "This occured in yprime(k) for k = ", k
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( k, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               IDID = -14
               return
            end if
         end do
      end if

                        if( iflag < 0 ) then
                           IDID = -100
                           return
                        end if
!                       if iflag is positive, continue integration...
                        IF( T /= TOUT ) GO TO 230

!                          Interval mode
                           IDID = 2
                           T = TOUT
                           TOLD = T
!     .....................Exit
                           return
  230                   CONTINUE
  240                CONTINUE

!                    -- Set sign of integration direction  and
!                    -- estimate starting step size

                     INIT = 2
                     DTSIGN = SIGN(1.0D0,TOUT-T)
                     U = D1MACH(4)
                     BIG = SQRT(D1MACH(2))
                     UTE = U**0.375D0
                     DY = UTE*DHVNRM(Y,NEQ)
                     IF( DY == 0.0D0 ) DY = UTE
                     KTOL = 1
                     DO K = 1, NEQ
                        IF( INFO(2) == 1 ) KTOL = K
                        TOL = RTOL(KTOL)*ABS(Y(K)) + ATOL(KTOL)
                        IF( TOL == 0.0D0 ) TOL = DY*RTOL(KTOL)
                        F1(K) = TOL
                     END DO

                     CALL DHSTRT(DF,NEQ,T,TOUT,Y,YP,F1,4,U,BIG,F2,F3,F4,&
                                 F5,H,ISTAT)
                     if( istat /= 0 ) then
                        IDID = -100
                        return
                     end if
  260             CONTINUE

!  Set step size for integration in the direction
!  from T to TOUT and set output point indicator

                  DT = TOUT - T
                  H = SIGN(H,DT)
                  OUTPUT = .FALSE.

!  Test to see if dderkf is being severely impacted by too many output
!  points

                  IF( ABS(H) >= 2.0D0*ABS(DT) ) KOP = KOP + 1
                  IF( KOP <= MXKOP ) GO TO 270

!  Unnecessary frequency of output is restricting the step size choice
                     IDID = -5
                     KOP = 0
                  GO TO 510
  270             CONTINUE

                     IF (ABS(DT) > U26*ABS(T)) GO TO 290

!  If too close to output point, extrapolate and return

                        DO K = 1, NEQ
                           Y(K) = Y(K) + DT*YP(K)
                        END DO
                        A = TOUT
                        nb_deriv = nb_deriv + 1
                        iflag = 0
                        call mf_restore_fpe( )
                        CALL DF( A, Y, YP, iflag )
                        call mf_save_and_disable_fpe( )
      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
         do k = 1, neq
            if( isnan(YP(k)) ) then
               print "(/,A)", "(MUESLI OdeSolve:) [drkfs] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied DERIV routine."
               print "(20X,A,I0)", "This occured in yprime(k) for k = ", k
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( k, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               IDID = -14
               return
            end if
         end do
      end if
                        if( iflag < 0 ) then
                           IDID = -100
                           return
                        end if
!                       nothing special for iflag > 0 ?
                        KSTEPS = KSTEPS + 1
                     GO TO 500
  290                CONTINUE
!                       Begin block permitting ...exits to 490

! *********************************************
! *********************************************
!        Step by step integration

  300                      CONTINUE
!                             Begin block permitting ...exits to 480
                                 HFAILD = .FALSE.

!  To protect against impossible accuracy requests, compute a tolerance
!  factor based on the requested error tolerance and a level of
!  accuracy achievable at limiting precision

                                 TOLFAC = 0.0D0
                                 KTOL = 1
                                 DO K = 1, NEQ
                                    IF( INFO(2) == 1 ) KTOL = K
                                    ET = RTOL(KTOL)*ABS(Y(K))           &
                                         + ATOL(KTOL)
                                    IF( ET > 0.0D0 ) GO TO 310
                                       TOLFAC = MAX(TOLFAC,             &
                                                      RER/RTOL(KTOL))
                                    GO TO 320
  310                               CONTINUE
                                       TOLFAC = MAX(TOLFAC,             &
                                                      ABS(Y(K))         &
                                                      *(RER/ET))
  320                               CONTINUE
                                 END DO
                                 IF( TOLFAC <= 1.0D0 ) GO TO 340

!  Requested error unattainable due to limited precision available
                                    TOLFAC = 2.0D0*TOLFAC
                                    IDID = -2
!              .....................Exit
                                    GO TO 520
  340                            CONTINUE

!  Set smallest allowable step size

                                 HMIN = U26*ABS(T)

!  Adjust step size if necessary to hit the output point --
!  Look ahead two steps to avoid drastic changes in the step size and
!  thus lessen the impact of output points on the code. Stretch the
!  step size by, at most, an amount equal to the safety factor of 9/10.

                                 DT = TOUT - T
                                 IF( ABS(DT) >= 2.0D0*ABS(H) )          &
                                    GO TO 370
                                    IF(ABS(DT) > ABS(H)/0.9D0 )         &
                                       GO TO 350

!  The next step, if successful, will complete the integration to
!  the output point
                                       OUTPUT = .TRUE.
                                       H = DT
                                    GO TO 360
  350                               CONTINUE

                                       H = 0.5D0*DT
  360                               CONTINUE
  370                            CONTINUE

!  ****************************************
!  Core integrator for taking a single step
!  ****************************************
!  To avoid problems with zero crossings, relative error is
!  measured using the average of the magnitudes of the solution at
!  the beginning and end of a step.
!  The error estimate formula has been grouped to control loss of
!  significance.
!  Local error estimates for a first order method using the same
!  step size as the Fehlberg method are calculated as part of the
!  test for stiffness.
!  To distinguish the various arguments, H is not permitted
!  to become smaller than 26 units of roundoff in T. Practical
!  limits on the change in the step size are enforced to smooth
!  the step size selection process and to avoid excessive
!  chattering on problems having discontinuities. To prevent
!  unnecessary failures, the code uses 9/10 the step size
!  it estimates will succeed.
!  After a step failure, the step size is not allowed to increase
!  for the next attempted step. This makes the code more efficient
!  on problems having discontinuities and more effective in general
!  since local extrapolation is being used and extra caution seems
!  warranted.

!  Monitor number of steps attempted

  380                            CONTINUE
                                    IF( KSTEPS <= MXSTEP ) GO TO 390

!  A significant amount of work has been expended
                                       IDID = -1
                                       KSTEPS = 0
!              ........................Exit
                                       IF( .NOT.STIFF ) GO TO 520

!  Problem appears to be stiff
                                       IDID = -4
                                       STIFF = .FALSE.
                                       NONSTF = .FALSE.
                                       NTSTEP = 0
                                       NSTIFS = 0
!              ........................Exit
                                       GO TO 520

  390                               CONTINUE

!  Advance an approximate solution over one step of length H

!C      print *, "calling DFEHL with: T = ", T, ", Y = ", Y(1), ", YP =
!C               YP(1), "; H = ", H, "; T+H = ", T+H

            CALL DFEHL( DF, NEQ, T, Y, H, YP, F1, F2, F3, F4, F5, YS,   &
                        istat, tend )
            if( istat /= 0 ) then
!C      print *, "(MUESLI drkfs:) line 563: ",
!C               "DFEHL reported an exception in DF!"
               if( istat == -1 ) then
!                 Illegal condition... trying to continue integration
!                 with a smaller timestep
                  COUNT_ILL_COND = COUNT_ILL_COND + 1
                  if( COUNT_ILL_COND > COUNT_ILL_COND_MAX ) then
                     IDID = -10
!!print *, "DBG [slatec]/DRKFS: IDID = -10:", &
!!         "more than ", COUNT_ILL_COND_MAX, " step refinements!"
                     return
                  else
!C      print *, "drkfs line 568: *** Illegal Cond. ***    T = ", T,
!C               "; T+H = ", T+H
                     H = 0.25d0*H
                     OUTPUT = .FALSE.
!C      print *, "new H = ", H, "   (and go to 390)"
                     go to 390
                  end if
               else if( istat == -2 ) then
!                 Emergency exit... we must return the current values (T,Y,YP)
                  IDID = -11
                  return
               else if( istat == 3 .and.                                &
                        .not. end_condition_found ) then
!### TODO ?: il faut sans doute différer ces actions à un pas obtenu
!            avec SUCCÈS ! (mettre un flag spécial et le lire plus bas)
!            End condition... we must return the values (Y,YP) at tend
!            (returned value from DFEHL)
                  end_condition_found = .true.
!C      print *, "drkfs line 582: *** End Condition ***    tend = ", ten
                  H = tend - T
                  TOUT = T + H
                  go to 390
               end if
            end if
                                    KSTEPS = KSTEPS + 1

!  Compute and test allowable tolerances versus local error estimates.
!  Note that relative error is measured with respect to the average of
!  the magnitudes of the solution at the beginning and end of the step.
!  Local error estimates for a special first order method are
!  calculated only when the stiffness detection is turned on.

                                    EEOET = 0.0D0
                                    ESTIFF = 0.0D0
                                    KTOL = 1
                                    DO K = 1, NEQ
                                       YAVG = 0.5D0*( ABS(Y(K))         &
                                                    + ABS(YS(K)) )
                                       IF( INFO(2) == 1 ) KTOL = K
                                       ET = RTOL(KTOL)*YAVG + ATOL(KTOL)
                                       IF( ET > 0.0D0 ) GO TO 400

!  Pure relative error inappropriate when solution vanishes

                                          IDID = -3
!              ...........................Exit
                                          GO TO 520
  400                                  CONTINUE
!
                                       EE = ABS((-2090.0D0*YP(K)        &
                                                  +(21970.0D0*F3(K)     &
                                                    -15048.0D0*F4(K)))  &
                                                 +(22528.0D0*F2(K)      &
                                                   -27360.0D0*F5(K)))
                                       IF( STIFF .OR. NONSTF ) GO TO 410
                                          ES = ABS(H                    &
                                                    *(0.055455D0*YP(K)  &
                                                      -0.035493D0*F1(K) &
                                                      -0.036571D0*F2(K) &
                                                      +0.023107D0*F3(K) &
                                                      -0.009515D0*F4(K) &
                                                      +0.003017D0*F5(K))&
                                                      )
                                          ESTIFF = MAX(ESTIFF,ES/ET)
  410                                  CONTINUE
                                       EEOET = MAX(EEOET,EE/ET)
                                    END DO

                                    ESTTOL = ABS(H)*EEOET/752400.0D0

!                                ...Exit
                                    IF( ESTTOL <= 1.0D0 ) GO TO 440

!  Unsuccessful step

                                    IF( ABS(H) > HMIN ) GO TO 430

!  Requested error unattainable at smallest allowable step size
                                       TOLFAC = 1.69D0*ESTTOL
                                       IDID = -2
!              ........................Exit
                                       GO TO 520
  430                               CONTINUE

!  Reduce the step size, try again.
!  The decrease is limited to a factor of 1/10

                                    HFAILD = .TRUE.
                                    OUTPUT = .FALSE.
                                    S = 0.1D0
                                    IF( ESTTOL < 59049.0D0 )            &
                                       S = 0.9D0/ESTTOL**0.2D0
                                    H = SIGN(MAX(S*ABS(H),HMIN),H)
                                 GO TO 380
  440                            CONTINUE

!  Successful step
!  Store solution at T+H and evaluate derivatives there

      nb_step = nb_step + 1

      if( h < dt_min ) dt_min = h
      if( h > dt_max ) dt_max = h
                                 T = T + H

      if( save_times ) then
        times_current_length = size(times_solve)
        n_times = n_times + 1
        if( n_times > times_current_length ) then
            allocate( vector_tmp(times_current_length) )
            vector_tmp(:) = times_solve(:)
            deallocate( times_solve )
            allocate( times_solve(2*times_current_length) )
            times_solve(1:times_current_length) = vector_tmp(:)
            deallocate( vector_tmp )
        end if
        times_solve(n_times) = T
!       saving the current time (for debugging purpose)
        current_time = T
      end if

      if( print_progress ) then
        percent = (T-t_0)*pc_fact
        if( percent > 100.0d0 ) percent = 100.0d0
        if( disp_times ) then
          call system_clock( count=clock )
          left_time = dble(clock-clock_init)/clock_rate
!         Global Linear Estimator
          total_time = left_time/(T-t_0)*(t_end-t_0)
          call sec_2_hms( left_time, hrs_1, min_1, sec_1 )
          rem_time = nint(max(0.0d0, total_time-left_time))
          call sec_2_hms( rem_time, hrs_2, min_2, sec_2 )
          write(string,'(F5.1,A,I3,A,2(I2,A),A,I3,A,2(I2,A))')          &
                percent, ' % (time left = ',                            &
                hrs_1, 'h ', min_1, 'm ', sec_1, 's ',                  &
                '-- estim. remain. time = ',                            &
                hrs_2, 'h ', min_2, 'm ', sec_2, 's)'
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
          call put_string_back_on_term_no_adv( trim(string) // char(0) )
#else
          call go_home_on_term()
          write(STDOUT,'(A)',advance='no') trim(string)
          call msFlush(STDOUT)
#endif
        else
          write(string,'(F5.1,A)') percent, ' %'
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
          call put_string_back_on_term_no_adv( trim(string) // char(0) )
#else
          call go_home_on_term()
          write(STDOUT,'(A)',advance='no') trim(string)
          call msFlush(STDOUT)
#endif
        end if
      end if

                                 DO K = 1, NEQ
                                    Y(K) = YS(K)
                                 END DO
                                 A = T
      nb_deriv = nb_deriv + 1
      iflag = 0
!     -- evaluation of the derivative after a successful step
                                 call mf_restore_fpe( )
                                 CALL DF( A, Y, YP, iflag )
                                 call mf_save_and_disable_fpe( )
      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
         do k = 1, neq
            if( isnan(YP(k)) ) then
               print "(/,A)", "(MUESLI OdeSolve:) [drkfs] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied DERIV routine."
               print "(20X,A,I0)", "This occured in yp(k) for k = ", k
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( k, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               IDID = -14
               return
            end if
         end do
      end if
      if( iflag < 0 ) then
!        since the step was successful, it looks strange that we
!        got an exception at the end of the step (already called)
               print *, "(MUESLI:) internal error."
               print *, "          (SLATEC/drkfs: line 730:",           &
                        " case not supported)"
               print *, "          For your information, istat = ",     &
                        istat, " after calling DFEHL."
               print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
               print *
               print *, "  pause for debugging purpose"
               print *, "  [RETURN] to resume..."
               read *
               stop "*** job aborted ***"
      end if
!     En cas de pas infructueux, H a certainement été réduit, il ne faut
!     quitter que si le nouveau TOUT est atteint !
      if( end_condition_found .and. T >= TOUT ) then
         idid = 12
         return
      end if

! storing y(:) and yp(:) in files (monitoring)
      if( unit_monitor_y /= -1 ) then
         write( unit_monitor_y, format_monitor_y )                      &
                T, Y(monitoryind)
         call flush( unit_monitor_y )
      end if
      if( unit_monitor_yp /= -1 ) then
         write( unit_monitor_yp, format_monitor_yp )                    &
                T, YP(monitorypind)
         call flush( unit_monitor_yp )
      end if
      if( monitor_pause ) then
         print *
         call msPause( "[MUESLI] ODE solver: monitoring pause" )
      end if

!  Choose next step size. The increase is limited to a factor of 5.
!  If step failure has just occurred, next step size is not allowed
!  to increase

                                 S = 5.0D0
                                 IF( ESTTOL > 1.889568D-4 )             &
                                    S = 0.9D0/ESTTOL**0.2D0
                                 IF( HFAILD ) S = MIN(S,1.0D0)
!  Increase the step size only if an Illegal Condition has not been
!  encountered.
      if( COUNT_ILL_COND == 0 ) then
                                 H = SIGN(MAX(S*ABS(H),HMIN),H)
!C      print *, "drkfs: line 759: next step H = ", H
      end if

!  Check for stiffness (if not already detected)

!  In a sequence of 50 successful steps by the Fehlberg method, 25
!  successful steps by the first order method indicates stiffness
!  and turns the test off. If 26 failures by the first order method
!  occur, the test is turned off until this sequence of 50 steps by
!  the Fehlberg method is completed.

!                             ...Exit
                                 IF( STIFF ) GO TO 480
                                 NTSTEP = MOD(NTSTEP+1,50)
                                 IF( NTSTEP == 1 ) NONSTF = .FALSE.
!                             ...Exit
                                 IF( NONSTF ) GO TO 480
                                 IF( ESTIFF > 1.0D0 ) GO TO 460

!  Successful step with first order method
                                    NSTIFS = NSTIFS + 1
!  Turn test on after 25 indications of stiffness
                                    IF( NSTIFS == 25 ) STIFF = .TRUE.
                                 GO TO 470
  460                            CONTINUE

!  Unsuccessful step with first order method
                                 IF( NTSTEP - NSTIFS <= 25 ) GO TO 470
!  Turn stiffness detection off for this block of fifty steps
                                    NONSTF = .TRUE.
!  Reset stiff step counter
                                    NSTIFS = 0
  470                            CONTINUE
  480                         CONTINUE

! ******************************************
!         End of core integrator
! ******************************************

!  Should we take another step
!                       ......Exit
                              IF( OUTPUT ) GO TO 490
                           IF( INFO(3) == 0 ) GO TO 300

! ******************************************
! ******************************************

!  Integration successfully completed

!           One-step mode
                           IDID = 1
                           TOLD = T
!     .....................Exit
                           return
  490                   CONTINUE
  500                CONTINUE
!
!  Interval mode
                     IDID = 2
                     T = TOUT
                     TOLD = T
!     ...............Exit
                     return
  510             CONTINUE
  520          CONTINUE
  530       CONTINUE
  540    CONTINUE

!  Integration task interrupted

         INFO(1) = -1
         TOLD = T
!     ...Exit
         IF( IDID /= -2 ) return

!  The error tolerances are increased to values which are appropriate
!  for continuing
         RTOL(1) = TOLFAC*RTOL(1)
         ATOL(1) = TOLFAC*ATOL(1)
!     ...Exit
         IF( INFO(2) == 0 ) return
         DO K = 2, NEQ
            RTOL(K) = TOLFAC*RTOL(K)
            ATOL(K) = TOLFAC*ATOL(K)
         END DO

   END
