! Part of MUESLI Numerical Library
! Copyright É. Canot 2003-2025 -- IPR/CNRS

!DECK DQK41 modified for function of 2 variables
   SUBROUTINE DQK41_3( F, X, A, B, RESULT, ABSERR, RESABS, RESASC )
!***BEGIN PROLOGUE  DQK41
!***PURPOSE  To compute I = Integral of F over (A,B), with error
!                           estimate
!                       J = Integral of ABS(F) over (A,B)
!***LIBRARY   SLATEC (QUADPACK)
!***CATEGORY  H2A1A2
!***TYPE      DOUBLE PRECISION (QK41-S, DQK41-D)
!***KEYWORDS  41-POINT GAUSS-KRONROD RULES, QUADPACK, QUADRATURE
!***AUTHOR  Piessens, Robert
!             Applied Mathematics and Programming Division
!             K. U. Leuven
!           de Doncker, Elise
!             Applied Mathematics and Programming Division
!             K. U. Leuven
!***DESCRIPTION
!
!           Integration rules
!           Standard fortran subroutine
!           Double precision version
!
!           PARAMETERS
!            ON ENTRY
!              F      - Double precision
!                       Function subprogram defining the integrand
!                       FUNCTION F(X). The actual name for F needs to be
!                       declared E X T E R N A L in the calling program.
!
!              A      - Double precision
!                       Lower limit of integration
!
!              B      - Double precision
!                       Upper limit of integration
!
!            ON RETURN
!              RESULT - Double precision
!                       Approximation to the integral I
!                       RESULT is computed by applying the 41-POINT
!                       GAUSS-KRONROD RULE (RESK) obtained by optimal
!                       addition of abscissae to the 20-POINT GAUSS
!                       RULE (RESG).
!
!              ABSERR - Double precision
!                       Estimate of the modulus of the absolute error,
!                       which should not exceed ABS(I-RESULT)
!
!              RESABS - Double precision
!                       Approximation to the integral J
!
!              RESASC - Double precision
!                       Approximation to the integral of ABS(F-I/(B-A))
!                       over (A,B)
!
!***REFERENCES  (NONE)
!***ROUTINES CALLED  D1MACH
!***REVISION HISTORY  (YYMMDD)
!   800101  DATE WRITTEN
!   890531  Changed all specific intrinsics to generic.  (WRB)
!   890531  REVISION DATE from Version 3.2
!   891214  Prologue converted to Version 4.0 format.  (BAB)
!***END PROLOGUE  DQK41
!
      DOUBLE PRECISION X
      DOUBLE PRECISION A,ABSC,ABSERR,B,CENTR,DHLGTH,                    &
        EPMACH,F,FC,FSUM,FVAL1,FVAL2,FV1,FV2,HLGTH,RESABS,RESASC,&
        RESG,RESK,RESKH,RESULT,UFLOW,WG,WGK,XGK
      INTEGER J,JTW,JTWM1
      EXTERNAL F
!
      DIMENSION FV1(20),FV2(20),XGK(21),WGK(21),WG(10)
!
!           THE ABSCISSAE AND WEIGHTS ARE GIVEN FOR THE INTERVAL (-1,1).
!           BECAUSE OF SYMMETRY ONLY THE POSITIVE ABSCISSAE AND THEIR
!           CORRESPONDING WEIGHTS ARE GIVEN.
!
!           XGK    - ABSCISSAE OF THE 41-POINT GAUSS-KRONROD RULE
!                    XGK(2), XGK(4), ...  ABSCISSAE OF THE 20-POINT
!                    GAUSS RULE
!                    XGK(1), XGK(3), ...  ABSCISSAE WHICH ARE OPTIMALLY
!                    ADDED TO THE 20-POINT GAUSS RULE
!
!           WGK    - WEIGHTS OF THE 41-POINT GAUSS-KRONROD RULE
!
!           WG     - WEIGHTS OF THE 20-POINT GAUSS RULE
!
!
! GAUSS QUADRATURE WEIGHTS AND KRONROD QUADRATURE ABSCISSAE AND WEIGHTS
! AS EVALUATED WITH 80 DECIMAL DIGIT ARITHMETIC BY L. W. FULLERTON,
! BELL LABS, NOV. 1981.
!
      SAVE WG, XGK, WGK
      DATA WG  (  1) / 0.017614007139152118311861962351853D0 /
      DATA WG  (  2) / 0.040601429800386941331039952274932D0 /
      DATA WG  (  3) / 0.062672048334109063569506535187042D0 /
      DATA WG  (  4) / 0.083276741576704748724758143222046D0 /
      DATA WG  (  5) / 0.101930119817240435036750135480350D0 /
      DATA WG  (  6) / 0.118194531961518417312377377711382D0 /
      DATA WG  (  7) / 0.131688638449176626898494499748163D0 /
      DATA WG  (  8) / 0.142096109318382051329298325067165D0 /
      DATA WG  (  9) / 0.149172986472603746787828737001969D0 /
      DATA WG  ( 10) / 0.152753387130725850698084331955098D0 /
!
      DATA XGK (  1) / 0.998859031588277663838315576545863D0 /
      DATA XGK (  2) / 0.993128599185094924786122388471320D0 /
      DATA XGK (  3) / 0.981507877450250259193342994720217D0 /
      DATA XGK (  4) / 0.963971927277913791267666131197277D0 /
      DATA XGK (  5) / 0.940822633831754753519982722212443D0 /
      DATA XGK (  6) / 0.912234428251325905867752441203298D0 /
      DATA XGK (  7) / 0.878276811252281976077442995113078D0 /
      DATA XGK (  8) / 0.839116971822218823394529061701521D0 /
      DATA XGK (  9) / 0.795041428837551198350638833272788D0 /
      DATA XGK ( 10) / 0.746331906460150792614305070355642D0 /
      DATA XGK ( 11) / 0.693237656334751384805490711845932D0 /
      DATA XGK ( 12) / 0.636053680726515025452836696226286D0 /
      DATA XGK ( 13) / 0.575140446819710315342946036586425D0 /
      DATA XGK ( 14) / 0.510867001950827098004364050955251D0 /
      DATA XGK ( 15) / 0.443593175238725103199992213492640D0 /
      DATA XGK ( 16) / 0.373706088715419560672548177024927D0 /
      DATA XGK ( 17) / 0.301627868114913004320555356858592D0 /
      DATA XGK ( 18) / 0.227785851141645078080496195368575D0 /
      DATA XGK ( 19) / 0.152605465240922675505220241022678D0 /
      DATA XGK ( 20) / 0.076526521133497333754640409398838D0 /
      DATA XGK ( 21) / 0.000000000000000000000000000000000D0 /
!
      DATA WGK (  1) / 0.003073583718520531501218293246031D0 /
      DATA WGK (  2) / 0.008600269855642942198661787950102D0 /
      DATA WGK (  3) / 0.014626169256971252983787960308868D0 /
      DATA WGK (  4) / 0.020388373461266523598010231432755D0 /
      DATA WGK (  5) / 0.025882133604951158834505067096153D0 /
      DATA WGK (  6) / 0.031287306777032798958543119323801D0 /
      DATA WGK (  7) / 0.036600169758200798030557240707211D0 /
      DATA WGK (  8) / 0.041668873327973686263788305936895D0 /
      DATA WGK (  9) / 0.046434821867497674720231880926108D0 /
      DATA WGK ( 10) / 0.050944573923728691932707670050345D0 /
      DATA WGK ( 11) / 0.055195105348285994744832372419777D0 /
      DATA WGK ( 12) / 0.059111400880639572374967220648594D0 /
      DATA WGK ( 13) / 0.062653237554781168025870122174255D0 /
      DATA WGK ( 14) / 0.065834597133618422111563556969398D0 /
      DATA WGK ( 15) / 0.068648672928521619345623411885368D0 /
      DATA WGK ( 16) / 0.071054423553444068305790361723210D0 /
      DATA WGK ( 17) / 0.073030690332786667495189417658913D0 /
      DATA WGK ( 18) / 0.074582875400499188986581418362488D0 /
      DATA WGK ( 19) / 0.075704497684556674659542775376617D0 /
      DATA WGK ( 20) / 0.076377867672080736705502835038061D0 /
      DATA WGK ( 21) / 0.076600711917999656445049901530102D0 /
!
!
!           LIST OF MAJOR VARIABLES
!           -----------------------
!
!           CENTR  - MID POINT OF THE INTERVAL
!           HLGTH  - HALF-LENGTH OF THE INTERVAL
!           ABSC   - ABSCISSA
!           FVAL*  - FUNCTION VALUE
!           RESG   - RESULT OF THE 20-POINT GAUSS FORMULA
!           RESK   - RESULT OF THE 41-POINT KRONROD FORMULA
!           RESKH  - APPROXIMATION TO MEAN VALUE OF F OVER (A,B), I.E.
!                    TO I/(B-A)
!
!           MACHINE DEPENDENT CONSTANTS
!           ---------------------------
!
!           EPMACH IS THE LARGEST RELATIVE SPACING.
!           UFLOW IS THE SMALLEST POSITIVE MAGNITUDE.
!
!***FIRST EXECUTABLE STATEMENT  DQK41
      EPMACH = D1MACH(4)
      UFLOW = D1MACH(1)
!
      CENTR = 0.5D+00*(A+B)
      HLGTH = 0.5D+00*(B-A)
      DHLGTH = ABS(HLGTH)
!
!           COMPUTE THE 41-POINT GAUSS-KRONROD APPROXIMATION TO
!           THE INTEGRAL, AND ESTIMATE THE ABSOLUTE ERROR.
!
      RESG = 0.0D+00
      FC = F(X,CENTR)
      RESK = WGK(21)*FC
      RESABS = ABS(RESK)
      DO 10 J=1,10
        JTW = J*2
        ABSC = HLGTH*XGK(JTW)
        FVAL1 = F(X,CENTR-ABSC)
        FVAL2 = F(X,CENTR+ABSC)
        FV1(JTW) = FVAL1
        FV2(JTW) = FVAL2
        FSUM = FVAL1+FVAL2
        RESG = RESG+WG(J)*FSUM
        RESK = RESK+WGK(JTW)*FSUM
        RESABS = RESABS+WGK(JTW)*(ABS(FVAL1)+ABS(FVAL2))
   10 END DO
      DO 15 J = 1,10
        JTWM1 = J*2-1
        ABSC = HLGTH*XGK(JTWM1)
        FVAL1 = F(X,CENTR-ABSC)
        FVAL2 = F(X,CENTR+ABSC)
        FV1(JTWM1) = FVAL1
        FV2(JTWM1) = FVAL2
        FSUM = FVAL1+FVAL2
        RESK = RESK+WGK(JTWM1)*FSUM
        RESABS = RESABS+WGK(JTWM1)*(ABS(FVAL1)+ABS(FVAL2))
   15 END DO
      RESKH = RESK*0.5D+00
      RESASC = WGK(21)*ABS(FC-RESKH)
      DO 20 J=1,20
        RESASC = RESASC+WGK(J)*(ABS(FV1(J)-RESKH)+ABS(FV2(J)-RESKH))
   20 END DO
      RESULT = RESK*HLGTH
      RESABS = RESABS*DHLGTH
      RESASC = RESASC*DHLGTH
      ABSERR = ABS((RESK-RESG)*HLGTH)
      IF(RESASC.NE.0.0D+00.AND.ABSERR.NE.0.D+00)                        &
        ABSERR = RESASC*MIN(0.1D+01,(0.2D+03*ABSERR/RESASC)**1.5D+00)
      IF(RESABS.GT.UFLOW/(0.5D+02*EPMACH)) ABSERR = MAX                 &
        ((EPMACH*0.5D+02)*RESABS,ABSERR)
   END
