! Part of MUESLI Numerical Library
! Copyright É. Canot 2003-2025 -- IPR/CNRS

!DECK DQK31 modified for function of 2 variables
   SUBROUTINE DQK31_2( F, Y, A, B, RESULT, ABSERR, RESABS, RESASC )
!***BEGIN PROLOGUE  DQK31
!***PURPOSE  To compute I = Integral of F over (A,B) with error
!                           estimate
!                       J = Integral of ABS(F) over (A,B)
!***LIBRARY   SLATEC (QUADPACK)
!***CATEGORY  H2A1A2
!***TYPE      DOUBLE PRECISION (QK31-S, DQK31-D)
!***KEYWORDS  31-POINT GAUSS-KRONROD RULES, QUADPACK, QUADRATURE
!***AUTHOR  Piessens, Robert
!             Applied Mathematics and Programming Division
!             K. U. Leuven
!           de Doncker, Elise
!             Applied Mathematics and Programming Division
!             K. U. Leuven
!***DESCRIPTION
!
!           Integration rules
!           Standard fortran subroutine
!           Double precision version
!
!           PARAMETERS
!            ON ENTRY
!              F      - Double precision
!                       Function subprogram defining the integrand
!                       FUNCTION F(X). The actual name for F needs to be
!                       Declared E X T E R N A L in the calling program.
!
!              A      - Double precision
!                       Lower limit of integration
!
!              B      - Double precision
!                       Upper limit of integration
!
!            ON RETURN
!              RESULT - Double precision
!                       Approximation to the integral I
!                       RESULT is computed by applying the 31-POINT
!                       GAUSS-KRONROD RULE (RESK), obtained by optimal
!                       addition of abscissae to the 15-POINT GAUSS
!                       RULE (RESG).
!
!              ABSERR - Double precision
!                       Estimate of the modulus of the modulus,
!                       which should not exceed ABS(I-RESULT)
!
!              RESABS - Double precision
!                       Approximation to the integral J
!
!              RESASC - Double precision
!                       Approximation to the integral of ABS(F-I/(B-A))
!                       over (A,B)
!
!***REFERENCES  (NONE)
!***ROUTINES CALLED  D1MACH
!***REVISION HISTORY  (YYMMDD)
!   800101  DATE WRITTEN
!   890531  Changed all specific intrinsics to generic.  (WRB)
!   890531  REVISION DATE from Version 3.2
!   891214  Prologue converted to Version 4.0 format.  (BAB)
!***END PROLOGUE  DQK31
      DOUBLE PRECISION Y
      DOUBLE PRECISION A,ABSC,ABSERR,B,CENTR,DHLGTH,                    &
        EPMACH,F,FC,FSUM,FVAL1,FVAL2,FV1,FV2,HLGTH,RESABS,RESASC,&
        RESG,RESK,RESKH,RESULT,UFLOW,WG,WGK,XGK
      INTEGER J,JTW,JTWM1
      EXTERNAL F
!
      DIMENSION FV1(15),FV2(15),XGK(16),WGK(16),WG(8)
!
!           THE ABSCISSAE AND WEIGHTS ARE GIVEN FOR THE INTERVAL (-1,1).
!           BECAUSE OF SYMMETRY ONLY THE POSITIVE ABSCISSAE AND THEIR
!           CORRESPONDING WEIGHTS ARE GIVEN.
!
!           XGK    - ABSCISSAE OF THE 31-POINT KRONROD RULE
!                    XGK(2), XGK(4), ...  ABSCISSAE OF THE 15-POINT
!                    GAUSS RULE
!                    XGK(1), XGK(3), ...  ABSCISSAE WHICH ARE OPTIMALLY
!                    ADDED TO THE 15-POINT GAUSS RULE
!
!           WGK    - WEIGHTS OF THE 31-POINT KRONROD RULE
!
!           WG     - WEIGHTS OF THE 15-POINT GAUSS RULE
!
!
! GAUSS QUADRATURE WEIGHTS AND KRONROD QUADRATURE ABSCISSAE AND WEIGHTS
! AS EVALUATED WITH 80 DECIMAL DIGIT ARITHMETIC BY L. W. FULLERTON,
! BELL LABS, NOV. 1981.
!
      SAVE WG, XGK, WGK
      DATA WG  (  1) / 0.030753241996117268354628393577204D0 /
      DATA WG  (  2) / 0.070366047488108124709267416450667D0 /
      DATA WG  (  3) / 0.107159220467171935011869546685869D0 /
      DATA WG  (  4) / 0.139570677926154314447804794511028D0 /
      DATA WG  (  5) / 0.166269205816993933553200860481209D0 /
      DATA WG  (  6) / 0.186161000015562211026800561866423D0 /
      DATA WG  (  7) / 0.198431485327111576456118326443839D0 /
      DATA WG  (  8) / 0.202578241925561272880620199967519D0 /
!
      DATA XGK (  1) / 0.998002298693397060285172840152271D0 /
      DATA XGK (  2) / 0.987992518020485428489565718586613D0 /
      DATA XGK (  3) / 0.967739075679139134257347978784337D0 /
      DATA XGK (  4) / 0.937273392400705904307758947710209D0 /
      DATA XGK (  5) / 0.897264532344081900882509656454496D0 /
      DATA XGK (  6) / 0.848206583410427216200648320774217D0 /
      DATA XGK (  7) / 0.790418501442465932967649294817947D0 /
      DATA XGK (  8) / 0.724417731360170047416186054613938D0 /
      DATA XGK (  9) / 0.650996741297416970533735895313275D0 /
      DATA XGK ( 10) / 0.570972172608538847537226737253911D0 /
      DATA XGK ( 11) / 0.485081863640239680693655740232351D0 /
      DATA XGK ( 12) / 0.394151347077563369897207370981045D0 /
      DATA XGK ( 13) / 0.299180007153168812166780024266389D0 /
      DATA XGK ( 14) / 0.201194093997434522300628303394596D0 /
      DATA XGK ( 15) / 0.101142066918717499027074231447392D0 /
      DATA XGK ( 16) / 0.000000000000000000000000000000000D0 /
!
      DATA WGK (  1) / 0.005377479872923348987792051430128D0 /
      DATA WGK (  2) / 0.015007947329316122538374763075807D0 /
      DATA WGK (  3) / 0.025460847326715320186874001019653D0 /
      DATA WGK (  4) / 0.035346360791375846222037948478360D0 /
      DATA WGK (  5) / 0.044589751324764876608227299373280D0 /
      DATA WGK (  6) / 0.053481524690928087265343147239430D0 /
      DATA WGK (  7) / 0.062009567800670640285139230960803D0 /
      DATA WGK (  8) / 0.069854121318728258709520077099147D0 /
      DATA WGK (  9) / 0.076849680757720378894432777482659D0 /
      DATA WGK ( 10) / 0.083080502823133021038289247286104D0 /
      DATA WGK ( 11) / 0.088564443056211770647275443693774D0 /
      DATA WGK ( 12) / 0.093126598170825321225486872747346D0 /
      DATA WGK ( 13) / 0.096642726983623678505179907627589D0 /
      DATA WGK ( 14) / 0.099173598721791959332393173484603D0 /
      DATA WGK ( 15) / 0.100769845523875595044946662617570D0 /
      DATA WGK ( 16) / 0.101330007014791549017374792767493D0 /
!
!
!           LIST OF MAJOR VARIABLES
!           -----------------------
!           CENTR  - MID POINT OF THE INTERVAL
!           HLGTH  - HALF-LENGTH OF THE INTERVAL
!           ABSC   - ABSCISSA
!           FVAL*  - FUNCTION VALUE
!           RESG   - RESULT OF THE 15-POINT GAUSS FORMULA
!           RESK   - RESULT OF THE 31-POINT KRONROD FORMULA
!           RESKH  - APPROXIMATION TO THE MEAN VALUE OF F OVER (A,B),
!                    I.E. TO I/(B-A)
!
!           MACHINE DEPENDENT CONSTANTS
!           ---------------------------
!           EPMACH IS THE LARGEST RELATIVE SPACING.
!           UFLOW IS THE SMALLEST POSITIVE MAGNITUDE.
!***FIRST EXECUTABLE STATEMENT  DQK31
      EPMACH = D1MACH(4)
      UFLOW = D1MACH(1)
!
      CENTR = 0.5D+00*(A+B)
      HLGTH = 0.5D+00*(B-A)
      DHLGTH = ABS(HLGTH)
!
!           COMPUTE THE 31-POINT KRONROD APPROXIMATION TO
!           THE INTEGRAL, AND ESTIMATE THE ABSOLUTE ERROR.
!
      FC = F(CENTR,Y)
      RESG = WG(8)*FC
      RESK = WGK(16)*FC
      RESABS = ABS(RESK)
      DO 10 J=1,7
        JTW = J*2
        ABSC = HLGTH*XGK(JTW)
        FVAL1 = F(CENTR-ABSC,Y)
        FVAL2 = F(CENTR+ABSC,Y)
        FV1(JTW) = FVAL1
        FV2(JTW) = FVAL2
        FSUM = FVAL1+FVAL2
        RESG = RESG+WG(J)*FSUM
        RESK = RESK+WGK(JTW)*FSUM
        RESABS = RESABS+WGK(JTW)*(ABS(FVAL1)+ABS(FVAL2))
   10 END DO
      DO 15 J = 1,8
        JTWM1 = J*2-1
        ABSC = HLGTH*XGK(JTWM1)
        FVAL1 = F(CENTR-ABSC,Y)
        FVAL2 = F(CENTR+ABSC,Y)
        FV1(JTWM1) = FVAL1
        FV2(JTWM1) = FVAL2
        FSUM = FVAL1+FVAL2
        RESK = RESK+WGK(JTWM1)*FSUM
        RESABS = RESABS+WGK(JTWM1)*(ABS(FVAL1)+ABS(FVAL2))
   15 END DO
      RESKH = RESK*0.5D+00
      RESASC = WGK(16)*ABS(FC-RESKH)
      DO 20 J=1,15
        RESASC = RESASC+WGK(J)*(ABS(FV1(J)-RESKH)+ABS(FV2(J)-RESKH))
   20 END DO
      RESULT = RESK*HLGTH
      RESABS = RESABS*DHLGTH
      RESASC = RESASC*DHLGTH
      ABSERR = ABS((RESK-RESG)*HLGTH)
      IF(RESASC.NE.0.0D+00.AND.ABSERR.NE.0.0D+00)                       &
        ABSERR = RESASC*MIN(0.1D+01,(0.2D+03*ABSERR/RESASC)**1.5D+00)
      IF(RESABS.GT.UFLOW/(0.5D+02*EPMACH)) ABSERR = MAX                 &
        ((EPMACH*0.5D+02)*RESABS,ABSERR)
   END
