! Part of MUESLI Numerical Library
! Copyright É. Canot 2003-2025 -- IPR/CNRS

!DECK DQAGE modified for function of 2 variables
   SUBROUTINE DQAGE_3( F, X, A, B, EPSABS, EPSREL, KEY, LIMIT, RESULT,  &
                       ABSERR, NEVAL, IER, ALIST, BLIST, RLIST, ELIST,  &
                       IORD, LAST )
!***BEGIN PROLOGUE  DQAGE
!***PURPOSE  The routine calculates an approximation result to a given
!            definite integral   I = Integral of F over (A,B),
!            hopefully satisfying following claim for accuracy
!            ABS(I-RESLT).LE.MAX(EPSABS,EPSREL*ABS(I)).
!***LIBRARY   SLATEC (QUADPACK)
!***CATEGORY  H2A1A1
!***TYPE      DOUBLE PRECISION (QAGE-S, DQAGE-D)
!***KEYWORDS  AUTOMATIC INTEGRATOR, GAUSS-KRONROD RULES,
!             GENERAL-PURPOSE, GLOBALLY ADAPTIVE, INTEGRAND EXAMINATOR,
!             QUADPACK, QUADRATURE
!***AUTHOR  Piessens, Robert
!             Applied Mathematics and Programming Division
!             K. U. Leuven
!           de Doncker, Elise
!             Applied Mathematics and Programming Division
!             K. U. Leuven
!***DESCRIPTION
!
!        Computation of a definite integral
!        Standard fortran subroutine
!        Double precision version
!
!        PARAMETERS
!         ON ENTRY
!            F      - Double precision
!                     Function subprogram defining the integrand
!                     function F(X). The actual name for F needs to be
!                     declared E X T E R N A L in the driver program.
!
!            A      - Double precision
!                     Lower limit of integration
!
!            B      - Double precision
!                     Upper limit of integration
!
!            EPSABS - Double precision
!                     Absolute accuracy requested
!            EPSREL - Double precision
!                     Relative accuracy requested
!                     If  EPSABS.LE.0
!                     and EPSREL.LT.MAX(50*REL.MACH.ACC.,0.5D-28),
!                     the routine will end with IER = 6.
!
!            KEY    - Integer
!                     Key for choice of local integration rule
!                     A Gauss-Kronrod pair is used with
!                          7 - 15 points if KEY.LT.2,
!                         10 - 21 points if KEY = 2,
!                         15 - 31 points if KEY = 3,
!                         20 - 41 points if KEY = 4,
!                         25 - 51 points if KEY = 5,
!                         30 - 61 points if KEY.GT.5.
!
!            LIMIT  - Integer
!                     Gives an upper bound on the number of subintervals
!                     in the partition of (A,B), LIMIT.GE.1.
!
!         ON RETURN
!            RESULT - Double precision
!                     Approximation to the integral
!
!            ABSERR - Double precision
!                     Estimate of the modulus of the absolute error,
!                     which should equal or exceed ABS(I-RESULT)
!
!            NEVAL  - Integer
!                     Number of integrand evaluations
!
!            IER    - Integer
!                     IER = 0 Normal and reliable termination of the
!                             routine. It is assumed that the requested
!                             accuracy has been achieved.
!                     IER.GT.0 Abnormal termination of the routine
!                             The estimates for result and error are
!                             less reliable. It is assumed that the
!                             requested accuracy has not been achieved.
!            ERROR MESSAGES
!                     IER = 1 Maximum number of subdivisions allowed
!                             has been achieved. One can allow more
!                             subdivisions by increasing the value
!                             of LIMIT.
!                             However, if this yields no improvement it
!                             is rather advised to analyze the integrand
!                             in order to determine the integration
!                             difficulties. If the position of a local
!                             difficulty can be determined(e.g.
!                             SINGULARITY, DISCONTINUITY within the
!                             interval) one will probably gain from
!                             splitting up the interval at this point
!                             and calling the integrator on the
!                             subranges. If possible, an appropriate
!                             special-purpose integrator should be used
!                             which is designed for handling the type of
!                             difficulty involved.
!                         = 2 The occurrence of roundoff error is
!                             detected, which prevents the requested
!                             tolerance from being achieved.
!                         = 3 Extremely bad integrand behaviour occurs
!                             at some points of the integration
!                             interval.
!                         = 6 The input is invalid, because
!                             (EPSABS.LE.0 and
!                              EPSREL.LT.MAX(50*REL.MACH.ACC.,0.5D-28),
!                             RESULT, ABSERR, NEVAL, LAST, RLIST(1),
!                             ELIST(1) and IORD(1) are set to zero.
!                             ALIST(1) and BLIST(1) are set to A and B
!                             respectively.
!
!            ALIST   - Double precision
!                      Vector of dimension at least LIMIT, the first
!                       LAST  elements of which are the left
!                      end points of the subintervals in the partition
!                      of the given integration range (A,B)
!
!            BLIST   - Double precision
!                      Vector of dimension at least LIMIT, the first
!                       LAST  elements of which are the right
!                      end points of the subintervals in the partition
!                      of the given integration range (A,B)
!
!            RLIST   - Double precision
!                      Vector of dimension at least LIMIT, the first
!                       LAST  elements of which are the
!                      integral approximations on the subintervals
!
!            ELIST   - Double precision
!                      Vector of dimension at least LIMIT, the first
!                       LAST  elements of which are the moduli of the
!                      absolute error estimates on the subintervals
!
!            IORD    - Integer
!                      Vector of dimension at least LIMIT, the first K
!                      elements of which are pointers to the
!                      error estimates over the subintervals,
!                      such that ELIST(IORD(1)), ...,
!                      ELIST(IORD(K)) form a decreasing sequence,
!                      with K = LAST if LAST.LE.(LIMIT/2+2), and
!                      K = LIMIT+1-LAST otherwise
!
!            LAST    - Integer
!                      Number of subintervals actually produced in the
!                      subdivision process
!
!***REFERENCES  (NONE)
!***ROUTINES CALLED  D1MACH, DQK15, DQK21, DQK31, DQK41, DQK51, DQK61,
!                    DQPSRT
!***REVISION HISTORY  (YYMMDD)
!   800101  DATE WRITTEN
!   890531  Changed all specific intrinsics to generic.  (WRB)
!   890831  Modified array declarations.  (WRB)
!   890831  REVISION DATE from Version 3.2
!   891214  Prologue converted to Version 4.0 format.  (BAB)
!***END PROLOGUE  DQAGE
!
      DOUBLE PRECISION X
      DOUBLE PRECISION A,ABSERR,ALIST,AREA,AREA1,AREA12,AREA2,A1,A2,B,  &
        BLIST,B1,B2,DEFABS,DEFAB1,DEFAB2,ELIST,EPMACH,           &
        EPSABS,EPSREL,ERRBND,ERRMAX,ERROR1,ERROR2,ERRO12,ERRSUM,F,      &
        RESABS,RESULT,RLIST,UFLOW
      INTEGER IER,IORD,IROFF1,IROFF2,K,KEY,KEYF,LAST,LIMIT,MAXERR,NEVAL,&
        NRMAX
!
      DIMENSION ALIST(*),BLIST(*),ELIST(*),IORD(*),                     &
        RLIST(*)
!
      EXTERNAL F
!
!            LIST OF MAJOR VARIABLES
!            -----------------------
!
!           ALIST     - LIST OF LEFT END POINTS OF ALL SUBINTERVALS
!                       CONSIDERED UP TO NOW
!           BLIST     - LIST OF RIGHT END POINTS OF ALL SUBINTERVALS
!                       CONSIDERED UP TO NOW
!           RLIST(I)  - APPROXIMATION TO THE INTEGRAL OVER
!                      (ALIST(I),BLIST(I))
!           ELIST(I)  - ERROR ESTIMATE APPLYING TO RLIST(I)
!           MAXERR    - POINTER TO THE INTERVAL WITH LARGEST
!                       ERROR ESTIMATE
!           ERRMAX    - ELIST(MAXERR)
!           AREA      - SUM OF THE INTEGRALS OVER THE SUBINTERVALS
!           ERRSUM    - SUM OF THE ERRORS OVER THE SUBINTERVALS
!           ERRBND    - REQUESTED ACCURACY MAX(EPSABS,EPSREL*
!                       ABS(RESULT))
!           *****1    - VARIABLE FOR THE LEFT SUBINTERVAL
!           *****2    - VARIABLE FOR THE RIGHT SUBINTERVAL
!           LAST      - INDEX FOR SUBDIVISION
!
!
!           MACHINE DEPENDENT CONSTANTS
!           ---------------------------
!
!           EPMACH  IS THE LARGEST RELATIVE SPACING.
!           UFLOW  IS THE SMALLEST POSITIVE MAGNITUDE.
!
!***FIRST EXECUTABLE STATEMENT  DQAGE
      EPMACH = D1MACH(4)
      UFLOW = D1MACH(1)
!
!           TEST ON VALIDITY OF PARAMETERS
!           ------------------------------
!
      IER = 0
      NEVAL = 0
      LAST = 0
      RESULT = 0.0D+00
      ABSERR = 0.0D+00
      ALIST(1) = A
      BLIST(1) = B
      RLIST(1) = 0.0D+00
      ELIST(1) = 0.0D+00
      IORD(1) = 0
      IF( EPSABS.LE.0.0D+00.AND.                                        &
          EPSREL.LT.MAX(0.5D+02*EPMACH,0.5D-28) ) IER = 6
      IF( IER.EQ.6 ) RETURN
!
!           FIRST APPROXIMATION TO THE INTEGRAL
!           -----------------------------------
!
      KEYF = KEY
      IF( KEY.LE.0 ) KEYF = 1
      IF( KEY.GE.7 ) KEYF = 6
      NEVAL = 0
      IF( KEYF.EQ.1 ) CALL DQK15_3(F,X,A,B,RESULT,ABSERR,DEFABS,RESABS)
      IF( KEYF.EQ.2 ) CALL DQK21_3(F,X,A,B,RESULT,ABSERR,DEFABS,RESABS)
      IF( KEYF.EQ.3 ) CALL DQK31_3(F,X,A,B,RESULT,ABSERR,DEFABS,RESABS)
      IF( KEYF.EQ.4 ) CALL DQK41_3(F,X,A,B,RESULT,ABSERR,DEFABS,RESABS)
      IF( KEYF.EQ.5 ) CALL DQK51_3(F,X,A,B,RESULT,ABSERR,DEFABS,RESABS)
      IF( KEYF.EQ.6 ) CALL DQK61_3(F,X,A,B,RESULT,ABSERR,DEFABS,RESABS)
      LAST = 1
      RLIST(1) = RESULT
      ELIST(1) = ABSERR
      IORD(1) = 1
!
!           TEST ON ACCURACY.
!
      ERRBND = MAX(EPSABS,EPSREL*ABS(RESULT))
      IF( ABSERR<=0.5D+02*EPMACH*DEFABS.AND.ABSERR>ERRBND ) IER = 2
      IF( LIMIT.EQ.1 ) IER = 1
      IF( IER.NE.0.OR.(ABSERR.LE.ERRBND.AND.ABSERR.NE.RESABS)           &
          .OR.ABSERR.EQ.0.0D+00 ) GO TO 60
!
!           INITIALIZATION
!           --------------
!
!
      ERRMAX = ABSERR
      MAXERR = 1
      AREA = RESULT
      ERRSUM = ABSERR
      NRMAX = 1
      IROFF1 = 0
      IROFF2 = 0
!
!           MAIN DO-LOOP
!           ------------
!
      DO 30 LAST = 2, LIMIT
!
!           BISECT THE SUBINTERVAL WITH THE LARGEST ERROR ESTIMATE.
!
        A1 = ALIST(MAXERR)
        B1 = 0.5D+00*(ALIST(MAXERR)+BLIST(MAXERR))
        A2 = B1
        B2 = BLIST(MAXERR)
        IF( KEYF.EQ.1 ) CALL DQK15_3(F,X,A1,B1,AREA1,ERROR1,RESABS,DEFAB1)
        IF( KEYF.EQ.2 ) CALL DQK21_3(F,X,A1,B1,AREA1,ERROR1,RESABS,DEFAB1)
        IF( KEYF.EQ.3 ) CALL DQK31_3(F,X,A1,B1,AREA1,ERROR1,RESABS,DEFAB1)
        IF( KEYF.EQ.4 ) CALL DQK41_3(F,X,A1,B1,AREA1,ERROR1,RESABS,DEFAB1)
        IF( KEYF.EQ.5 ) CALL DQK51_3(F,X,A1,B1,AREA1,ERROR1,RESABS,DEFAB1)
        IF( KEYF.EQ.6 ) CALL DQK61_3(F,X,A1,B1,AREA1,ERROR1,RESABS,DEFAB1)
        IF( KEYF.EQ.1 ) CALL DQK15_3(F,X,A2,B2,AREA2,ERROR2,RESABS,DEFAB2)
        IF( KEYF.EQ.2 ) CALL DQK21_3(F,X,A2,B2,AREA2,ERROR2,RESABS,DEFAB2)
        IF( KEYF.EQ.3 ) CALL DQK31_3(F,X,A2,B2,AREA2,ERROR2,RESABS,DEFAB2)
        IF( KEYF.EQ.4 ) CALL DQK41_3(F,X,A2,B2,AREA2,ERROR2,RESABS,DEFAB2)
        IF( KEYF.EQ.5 ) CALL DQK51_3(F,X,A2,B2,AREA2,ERROR2,RESABS,DEFAB2)
        IF( KEYF.EQ.6 ) CALL DQK61_3(F,X,A2,B2,AREA2,ERROR2,RESABS,DEFAB2)
!
!           IMPROVE PREVIOUS APPROXIMATIONS TO INTEGRAL
!           AND ERROR AND TEST FOR ACCURACY.
!
        NEVAL = NEVAL+1
        AREA12 = AREA1+AREA2
        ERRO12 = ERROR1+ERROR2
        ERRSUM = ERRSUM+ERRO12-ERRMAX
        AREA = AREA+AREA12-RLIST(MAXERR)
        IF(DEFAB1.EQ.ERROR1.OR.DEFAB2.EQ.ERROR2) GO TO 5
        IF( ABS(RLIST(MAXERR)-AREA12).LE.0.1D-04*ABS(AREA12)            &
            .AND.ERRO12.GE.0.99D+00*ERRMAX ) IROFF1 = IROFF1+1
        IF( LAST.GT.10.AND.ERRO12.GT.ERRMAX ) IROFF2 = IROFF2+1
    5   RLIST(MAXERR) = AREA1
        RLIST(LAST) = AREA2
        ERRBND = MAX(EPSABS,EPSREL*ABS(AREA))
        IF( ERRSUM <= ERRBND ) GO TO 8
!
!           TEST FOR ROUNDOFF ERROR AND EVENTUALLY SET ERROR FLAG.
!
        IF( IROFF1.GE.6.OR.IROFF2.GE.20 ) IER = 2
!
!           SET ERROR FLAG IN THE CASE THAT THE NUMBER OF SUBINTERVALS
!           EQUALS LIMIT.
!
        IF( LAST.EQ.LIMIT ) IER = 1
!
!           SET ERROR FLAG IN THE CASE OF BAD INTEGRAND BEHAVIOUR
!           AT A POINT OF THE INTEGRATION RANGE.
!
        IF( MAX(ABS(A1),ABS(B2)).LE.(0.1D+01+0.1D+03*                   &
            EPMACH)*(ABS(A2)+0.1D+04*UFLOW) ) IER = 3
!
!           APPEND THE NEWLY-CREATED INTERVALS TO THE LIST.
!
    8   IF( ERROR2.GT.ERROR1 ) GO TO 10
        ALIST(LAST) = A2
        BLIST(MAXERR) = B1
        BLIST(LAST) = B2
        ELIST(MAXERR) = ERROR1
        ELIST(LAST) = ERROR2
        GO TO 20
   10   ALIST(MAXERR) = A2
        ALIST(LAST) = A1
        BLIST(LAST) = B1
        RLIST(MAXERR) = AREA2
        RLIST(LAST) = AREA1
        ELIST(MAXERR) = ERROR2
        ELIST(LAST) = ERROR1
!
!           CALL SUBROUTINE DQPSRT TO MAINTAIN THE DESCENDING ORDERING
!           IN THE LIST OF ERROR ESTIMATES AND SELECT THE SUBINTERVAL
!           WITH THE LARGEST ERROR ESTIMATE (TO BE BISECTED NEXT).
!
   20   CALL DQPSRT(LIMIT,LAST,MAXERR,ERRMAX,ELIST,IORD,NRMAX)
! ***JUMP OUT OF DO-LOOP
        IF( IER.NE.0.OR.ERRSUM.LE.ERRBND ) GO TO 40
   30 END DO
!
!           COMPUTE FINAL RESULT.
!           ---------------------
!
   40 RESULT = 0.0D+00
      DO K = 1, LAST
        RESULT = RESULT+RLIST(K)
      END DO
      ABSERR = ERRSUM
   60 IF( KEYF /= 1 ) NEVAL = (10*KEYF+1)*(2*NEVAL+1)
      IF( KEYF == 1 ) NEVAL = 30*NEVAL+15

   END
