!DECK DPJAC
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 1 Mar 2019
!
subroutine DPJAC( NEQ, Y, YH, NYH, EWT, FTEM, SAVF, WM, IWM,            &
                  DF, DF_FLAG, DJAC, SPDJAC,                            &
                  CHECK_JAC, PRINT_CHECK_JAC )

!***BEGIN PROLOGUE  DPJAC
!***SUBSIDIARY
!***PURPOSE  Subsidiary to DDEBDF
!***LIBRARY   SLATEC
!***TYPE      DOUBLE PRECISION (PJAC-S, DPJAC-D)
!***AUTHOR  Watts, H. A., (SNLA)
!***DESCRIPTION
!
!   DPJAC sets up the iteration matrix (involving the Jacobian) for the
!   integration package DDEBDF.
!
!***SEE ALSO  DDEBDF
!***ROUTINES CALLED  DGBFA, DGEFA, DVNRMS
!***COMMON BLOCKS    DDEBD1
!***REVISION HISTORY  (YYMMDD)
!   820301  DATE WRITTEN
!   890531  Changed all specific intrinsics to generic.  (WRB)
!   890911  Removed unnecessary intrinsics.  (WRB)
!   891214  Prologue converted to Version 4.0 format.  (BAB)
!   900328  Added TYPE section.  (WRB)
!   910722  Updated AUTHOR section.  (ALS)
!   920422  Changed DIMENSION statement.  (WRB)
!***END PROLOGUE  DPJAC
!
!   ----------------------------------------------------------------------
!   DPJAC is called by DSTOD to compute and process the matrix
!   P = I - H*EL(1)*J, where J is an approximation to the jacobian. Here J
!   is computed by the user-supplied routine DJAC if MITER = 1 or 4, SPDJAC
!   if MITER = 6, or by finite differencing if MITER = 2, 3 or 5.
!   If MITER = 3, a diagonal approximation to J is used. J is stored in
!   WM and replaced by P.
!   If MITER /= 3, P is then subjected to LU or Cholesky decomposition in
!   preparation for later solution of linear systems with P as coefficient
!   matrix. This is done by DGEFA (dgetrf) if MITER = 1 or 2, by DGBFA if
!   MITER = 4 or 5, and by sparse appropriate routines if MITER = 6.
!
!   In addition to variables described previously, communication with
!   DPJAC uses the following:
!     Y    = array containing predicted values on entry.
!     FTEM = work array of length N (ACOR in DSTOD).
!     SAVF = array containing DF evaluated at predicted Y.
!     WM   = double precision work space for matrices. On output it
!            contains the inverse diagonal matrix if MITER = 3 and the
!            LU decomposition of P if MITER is 1, 2, 4 or 5.
!            Storage of matrix elements starts at WM(3).
!            WM also contains the following matrix-related data:
!              WM(1) = SQRT(UROUND), used in numerical jacobian increments.
!              WM(2) = H*EL0, saved for later use if MITER = 3.
!     IWM  = integer work space containing pivot information, starting
!            at IWM(21), if MITER is 1, 2, 4 or 5.  IWM also contains
!            the band parameters ML = IWM(1) and MU = IWM(2) if MITER
!            is 4 or 5.
!     EL0  = EL(1) (input).
!     IER  = Output error flag,  = 0 if no trouble, /= 0 if P matrix found to
!            be singular (general case) or not positive definite (SPD case).
!
!   This routine also uses the common variables EL0, H, TN, UROUND,
!   MITER, N, NFE, AND NJE.
!==============================================================================

   use mod_ddebd2, err => ddajac_err

   use mod_ieee, only: MF_NAN
   use mod_mfdebug, only: MF_NUMERICAL_CHECK
   use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe,        &
                        full_chk_der_ode
   use mod_mfarray, only: PrintMessage
   use mod_fileio, only: post_chk_der, post_full_chk_der
   use minpack

   implicit none

   INTEGER :: I, I1, I2, IER, II, IOWND, IOWNS, IWM(*), J, J1, JJ,      &
              JSTART, KFLAG, L, LENP, MAXORD, MBA, MBAND, MEB1, MEBAND, &
              METH, MITER, ML, MU, N, NEQ, NFE, NJE, NQ, NQU, NST, NYH, &
              ISPARSE, K, IUSERCALL, JAC_OUTDATED
   DOUBLE PRECISION :: CON, DI, EL0, EWT(*), FAC, FTEM(*), H, HL0, HMIN, &
                       HMXI, HU, R, R0, ROWND, ROWNS, SAVF(*), SRUR, TN, &
                       UROUND, WM(*), Y(*), YH(NYH,*), YI, YJ, YJJ
   EXTERNAL DF, DJAC, SPDJAC
   INTEGER :: CHECK_JAC, PRINT_CHECK_JAC, DF_FLAG

   COMMON /DDEBD1/ ROWND, ROWNS(210), EL0, H, HMIN, HMXI, HU, TN, UROUND, &
                   IOWND(14), IOWNS(6), IER, JSTART, KFLAG, L, METH,    &
                   MITER, MAXORD, N, NQ, NST, NFE, NJE, NQU, ISPARSE,   &
                   IUSERCALL, JAC_OUTDATED

   ! communication with slatec/drkfs.f, slatec/dfehl.f, slatec/dhstrt.f
   !                    slatec/ddes.f, slatec/dsteps.f
   !                    slatec/dlsod.f, slatec/dstod.f
   !                    [fml_funfun]/msOdeSolve_JacDF
   !                    [fml_funfun]/msOdeSolve_JacUser
   !                    [fml_funfun]/msOdeSolve_JacUserSP
   integer :: nb_step, nb_deriv
   double precision :: dt_min, dt_max
   common /slatec_odesolve_1/ dt_min, dt_max, nb_step, nb_deriv
   integer :: nb_jac, nb_solv
   common /slatec_odesolve_2/ nb_jac, nb_solv

   integer :: iflag
   double precision :: Ynew(n), DY(n), FTEMP(n), quality(n)

   ! the big array err(NEQ,NEQ) has been moved to the mod_ddebd2 module.

   logical :: box_constrained = .false.
   double precision :: ubounds(1)

   integer :: i_eqn_group, i_group_eqn, i_var_group, i_group_var
   integer :: ind, irow, imax, ind_symm
   logical :: quick_check_ok

!=======================================================================

   NJE = NJE + 1
   HL0 = H*EL0

   SELECT CASE( MITER )

   CASE( 1 )
      !-----------------------------------------------------------------
      ! Dense user-supplied matrix: call DJAC and multiply by scalar.
      !-----------------------------------------------------------------
      LENP = N**2
      ! set all elements of the jacobian to zero
      do I = 1, LENP
         WM(2+I) = 0.0d0
      end do

      call mf_restore_fpe( )
      call DJAC( TN, Y, WM(3), N )
      call mf_save_and_disable_fpe( )

      ! very quick check about storage if jacobian is tagged as SPD: two
      ! subdiagonal elements must be zero (see the complete test below...)
      if( jac_symm_pos_def ) then
         quick_check_ok = .true.
         J = N/2
         ind = (J-1)*N + J + 1
         if( WM(2+ind) /= 0.0d0 ) then
            quick_check_ok = .false.
         end if
         J = N - 1
         ind = (J-1)*N + J + 1
         if( WM(2+ind) /= 0.0d0 ) then
            quick_check_ok = .false.
         end if
         if( .not. quick_check_ok ) then
            print "(/,A)", "(MUESLI OdeSolve:) [BDF/dpjac] ERROR"
            print "(20X,A)", "You have tagged the (dense) jacobian matrix as SPD, so"
            print "(20X,A)", "it should contain only the upper part!"
            print "(20X,A)", "Here a quick check about two subdiagonal elements failed."
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            return
         end if
      end if

      if( MF_NUMERICAL_CHECK ) then

         ! if jacobian is tagged as SPD, check that only the upper part
         ! is provided
         if( jac_symm_pos_def ) then
            J1 = 2
            do J = 1, N
               ! travelling the lower part of the jacobian...
               do I = J+1, N
                  if( WM(I+J1) /= 0.0d0 ) then
                     print "(/,A)", "(MUESLI OdeSolve:) [BDF/dpjac] ERROR"
                     print "(20X,A)", "You have tagged the (dense) jacobian matrix as SPD, so"
                     print "(20X,A)", "it should contain only the upper part!"
                     print "(20X,A,I0,A,I0)", "Here the jacobian element for (i,j) = ", i, " ", j
                     print "(20X,A)", "is not zero."
                     mf_message_displayed = .true.
                     call muesli_trace( pause="yes" )
                     return
                  end if
               end do
               J1 = J1 + N
            end do
         end if

         ! check NaN values
         J1 = 2
         do J = 1, N
            if( jac_symm_pos_def ) then
               ! for travelling the upper part of the jacobian...
               IMAX = J
            else
               IMAX = N
            end if
            do I = 1, IMAX
               if( isnan(WM(I+J1)) ) then
                  print "(/,A)", "(MUESLI OdeSolve:) [BDF/dpjac] ERROR"
                  print "(20X,A)", "A NaN value has been found after calling the", &
                                   "user-supplied JAC routine."
                  print "(20X,A,I0,A,I0)", "This occured in matrix for (i,j) = ", i, " ", j
                  if( NAMED_EQN_PRESENCE ) then
                     call search_index_in_eqn_groups( i, i_eqn_group,      &
                                                         i_group_eqn )
                     print "(/,20X,A,A)", "Named equation is: ",           &
                                          trim(NAMED_EQN_PTR(i_eqn_group)%name)
                     print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                  end if
                  mf_message_displayed = .true.
                  call muesli_trace( pause="yes" )
                  return
               end if
            end do
            J1 = J1 + N
         end do

         if( check_jac == 1 ) then
            ! quick check of the jacobian

            if( jac_symm_pos_def ) then
               ! As the jacobian is provided only by its upper part, we have
               ! to complete the lower part for the following check.
               call do_full_jacobian()
            end if

            ! compute first FTEM(1:N)
            iflag = 0
            call mf_restore_fpe( )
            call DF( TN, Y, FTEM, iflag )
            call mf_save_and_disable_fpe( )
            ! check NaN values
            if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
               do k = 1, neq
                  if( isnan(FTEM(k)) ) then
                     print "(/,A)", "(MUESLI OdeSolve:) [BDF/dpjac] ERROR"
                     print "(20X,A)", "A NaN value has been found after calling the", &
                                      "user-supplied DERIV routine."
                     print "(20X,A,I0)", "This occured in yprime(k) for k = ", k
                     if( NAMED_EQN_PRESENCE ) then
                        call search_index_in_eqn_groups( k, i_eqn_group, &
                                                            i_group_eqn )
                        print "(/,20X,A,A)", "Named equation is: ",     &
                                             trim(NAMED_EQN_PTR(i_eqn_group)%name)
                        print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                     end if
                     mf_message_displayed = .true.
                     call muesli_trace( pause="yes" )
                     return
                  end if
               end do
            end if

            if( iflag /= 0 ) then
               DF_FLAG = iflag
               return
            end if

            call chkder( N, N, Y, FTEM, WM(3), Ynew, DY, FTEM, 1,       &
                         quality, box_constrained, ubounds )

            iflag = 0
            call mf_restore_fpe( )
            call DF( TN, Ynew, FTEMP, iflag )
            call mf_save_and_disable_fpe( )
            ! check NaN values
            if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
               do k = 1, neq
                  if( isnan(FTEMP(k)) ) then
                     print "(/,A)", "(MUESLI OdeSolve:) [drkfs] ERROR"
                     print "(20X,A)", "A NaN value has been found after calling the", &
                                      "user-supplied DERIV routine."
                     print "(20X,A,I0)", "This occured in yprime(k) for k = ", k
                     if( NAMED_EQN_PRESENCE ) then
                        call search_index_in_eqn_groups( k, i_eqn_group, &
                                                            i_group_eqn )
                        print "(/,20X,A,A)", "Named equation is: ",     &
                                             trim(NAMED_EQN_PTR(i_eqn_group)%name)
                        print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                     end if
                     mf_message_displayed = .true.
                     call muesli_trace( pause="yes" )
                     return
                  end if
               end do
            end if

            if( iflag /= 0 ) then
               DF_FLAG = iflag
               return
            end if

            call chkder( N, N, Y, FTEM, WM(3), Ynew, DY, FTEMP, 2,      &
                         quality, box_constrained, ubounds )

            call post_chk_der( "OdeSolve", n, n, quality,               &
                               print_check_jac, time=TN )

         else if( check_jac == 2 ) then
            ! full check of the jacobian

            if( jac_symm_pos_def ) then
               ! As the jacobian is provided only by its upper part, we have
               ! to complete the lower part for the following check.
               call do_full_jacobian()
            end if

            if( .not. allocated(err) ) then
               allocate( err(NEQ,NEQ) )
            end if

            ! compute first FTEM(1:N)
            iflag = 0
            call mf_restore_fpe( )
            call DF( TN, Y, FTEM, iflag )
            call mf_save_and_disable_fpe( )
            ! check NaN values
            if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
               do k = 1, neq
                  if( isnan(FTEM(k)) ) then
                     print "(/,A)", "(MUESLI OdeSolve:) [BDF/dpjac] ERROR"
                     print "(20X,A)", "A NaN value has been found after calling the", &
                                      "user-supplied DERIV routine."
                     print "(20X,A,I0)", "This occured in yprime(k) for k = ", k
                     if( NAMED_EQN_PRESENCE ) then
                        call search_index_in_eqn_groups( k, i_eqn_group, &
                                                            i_group_eqn )
                        print "(/,20X,A,A)", "Named equation is: ",     &
                                             trim(NAMED_EQN_PTR(i_eqn_group)%name)
                        print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                     end if
                     mf_message_displayed = .true.
                     call muesli_trace( pause="yes" )
                     return
                  end if
               end do
            end if

            if( iflag /= 0 ) then
               DF_FLAG = iflag
               return
            end if

            call full_chk_der_ode( N, TN, Y, DF, FTEM, WM(3), err, iflag )

            if( iflag /= 0 ) then
               DF_FLAG = iflag
               return
            end if

            call post_full_chk_der( "OdeSolve", n, n, err,              &
                                    print_check_jac, time=TN )

         end if

      end if ! end of Numerical Check

      CON = -HL0
      if( jac_symm_pos_def ) then
         ! working only on the upper part of the jacobian
         J1 = 2
         do J = 1, N
            do I = 1, J
               WM(I+J1) = WM(I+J1)*CON
            end do
            J1 = J1 + N
         end do
      else
         do I = 1, LENP
            WM(I+2) = WM(I+2)*CON
         end do
      end if

      call do_dense_factorization()

   CASE( 2 )
      !-----------------------------------------------------------------
      ! Dense finite-difference-generated matrix: make N calls to DF to
      ! approximate J.
      !-----------------------------------------------------------------
      FAC = DVNRMS(N,SAVF,EWT)
      R0 = 1000.0D0*ABS(H)*UROUND*N*FAC
      if( R0 == 0.0D0 ) R0 = 1.0D0
      SRUR = WM(1)
      J1 = 2
      iflag = 0
      do J = 1, N
         YJ = Y(J)
         R = MAX(SRUR*ABS(YJ),R0*EWT(J))
         Y(J) = Y(J) + R
         FAC = -HL0/R
         nb_deriv = nb_deriv + 1
         if( MF_NUMERICAL_CHECK ) then
            ! to be sure that, in case of use of an End Condition, all the
            ! coefficients of the vector RESID are effectively computed...
            FTEM(1:N) = MF_NAN
         endif

         call mf_restore_fpe( )
         call DF( TN, Y, FTEM, iflag )
         call mf_save_and_disable_fpe( )

         ! check NaN values
         ! Warning: in case of emergency of DERIV (flag iflag = -1 or -2)
         ! we must avoid the following check
         if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
            do k = 1, N
               if( isnan(FTEM(k)) ) then
                  print "(/,A)", "(MUESLI OdeSolve:) [BDF/dpjac] ERROR"
                  print "(20X,A)", "A NaN value has been found after",  &
                                   " calling the user-supplied DERIV routine."
                  print "(20X,A,I0)", "This occured in yprime(i) for i = ", k
                  if( NAMED_EQN_PRESENCE ) then
                     call search_index_in_eqn_groups( k, i_eqn_group,   &
                                                         i_group_eqn )
                     print "(/,20X,A,A)", "Named equation is: ",        &
                                          trim(NAMED_EQN_PTR(i_eqn_group)%name)
                     print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                  end if
                  return
               end if
            end do
         endif

         if( iflag /= 0 ) then
            DF_FLAG = iflag
            return
         end if

         do I = 1, N
            WM(I+J1) = (FTEM(I) - SAVF(I))*FAC
         end do
         Y(J) = YJ
         J1 = J1 + N
      end do
      NFE = NFE + N

      call do_dense_factorization()

   CASE( 3 )
      !-----------------------------------------------------------------
      ! Construct a diagonal approximation to J AND P.
      !-----------------------------------------------------------------
      WM(2) = HL0
      IER = 0
      R = EL0*0.1D0
      do I = 1, N
         Y(I) = Y(I) + R*(H*SAVF(I) - YH(I,2))
      end do
      nb_deriv = nb_deriv + 1
      iflag = 0

      call mf_restore_fpe( )
      call DF( TN, Y, WM(3), iflag )
      call mf_save_and_disable_fpe( )

      if( iflag /= 0 ) then
         DF_FLAG = iflag
         return
      end if

      NFE = NFE + 1
      do I = 1, N
         R0 = H*SAVF(I) - YH(I,2)
         DI = 0.1D0*R0 - H*(WM(I+2) - SAVF(I))
         WM(I+2) = 1.0D0
         if( ABS(R0) < UROUND*EWT(I) ) cycle
         if( ABS(DI) == 0.0D0 ) then
            IER = -1
            return
         end if
         WM(I+2) = 0.1D0*R0/DI
      end do

   CASE( 4 )
      !-----------------------------------------------------------------
      ! Banded user-supplied matrix: call DJAC and multiply by scalar.
      !-----------------------------------------------------------------
      ML = IWM(1)
      MU = IWM(2)
      if( jac_symm_pos_def ) then
         MBAND = MU + 1
         MEBAND = MBAND
      else
         MBAND = ML + MU + 1
         MEBAND = MBAND + ML
      end if
      LENP = MEBAND*N
      ! Set jacobian to zero before calling the user-supplied routine
      do I = 1, LENP
         WM(2+I) = 0.0d0
      end do

      call mf_restore_fpe( )
      call DJAC( TN, Y, WM(3), MEBAND )
      call mf_save_and_disable_fpe( )

      CON = -HL0
      do I = 1, LENP
         WM(I+2) = WM(I+2)*CON
      end do

      call do_banded_factorization()

   CASE( 5 )
      !-----------------------------------------------------------------
      ! Banded finite-difference-generated matrix:
      ! make MBAND calls to DF to approximate J.
      !-----------------------------------------------------------------
      ML = IWM(1)
      MU = IWM(2)
      MBAND = ML + MU + 1
      MBA = MIN(MBAND,N)
      MEBAND = MBAND + ML
      MEB1 = MEBAND - 1
      SRUR = WM(1)
      FAC = DVNRMS(N,SAVF,EWT)
      R0 = 1000.0D0*ABS(H)*UROUND*N*FAC
      if( R0 == 0.0D0 ) R0 = 1.0D0
      do J = 1, MBA
         do I = J, N, MBAND
            YI = Y(I)
            R = MAX(SRUR*ABS(YI),R0*EWT(I))
            Y(I) = Y(I) + R
         end do
         iflag = 0
         nb_deriv = nb_deriv + 1

         call mf_restore_fpe( )
         call DF( TN, Y, FTEM, iflag )
         call mf_save_and_disable_fpe( )

         if( iflag /= 0 ) then
            DF_FLAG = iflag
            return
         end if

         do JJ = J, N, MBAND
            Y(JJ) = YH(JJ,1)
            YJJ = Y(JJ)
            R = MAX(SRUR*ABS(YJJ),R0*EWT(JJ))
            FAC = -HL0/R
            I1 = MAX(JJ-MU,1)
            I2 = MIN(JJ+ML,N)
            II = JJ*MEB1 - ML + 2
            do I = I1, I2
               WM(II+I) = (FTEM(I) - SAVF(I))*FAC
            end do
         end do
      end do
      NFE = NFE + MBA

      call do_banded_factorization()

   CASE( 6 )
      !-----------------------------------------------------------
      ! Sparse matrix PD
      !-----------------------------------------------------------
      call mf_restore_fpe( )
      call SPDJAC( TN, Y, N, 1, PD(1), IPD(1), JPD(1), NNZ )
      call mf_save_and_disable_fpe( )

      if( MF_NUMERICAL_CHECK ) then

         ! if jacobian is tagged as SPD, check that only the upper part
         ! is provided
         if( jac_symm_pos_def ) then
            ! travelling the lower part of the jacobian...
            do j = 1, N
               do k = JPD(j), JPD(j+1) - 1
                  if( IPD(k) > j ) then
                     print "(/,A)", "(MUESLI OdeSolve:) [BDF/dpjac] ERROR"
                     print "(20X,A)", "You have tagged the (sparse) jacobian matrix as SPD, so"
                     print "(20X,A)", "it should contain only the upper part!"
                     print "(20X,A,I0,A,I0)", "Here the jacobian element for (i,j) = ", IPD(k), " ", j
                     print "(20X,A)", "is not zero."
                     mf_message_displayed = .true.
                     call muesli_trace( pause="yes" )
                     return
                  end if
               end do
            end do
         end if

         ! check NaN values
         do k = 1, nnz
            if( isnan(PD(k)) ) then
               print "(/,A)", "(MUESLI OdeSolve:) [BDF/dpjac] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied SPJAC routine."
               i = IPD(k)
               ! find column
               do j = 1, neq-1
                  if( k < JPD(j+1) ) exit
               end do
               print "(20X,A,I0,A,I0)", "This occured in matrix for (i,j) = ", i, " ", j
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( i, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                  if( NAMED_VAR_PRESENCE ) then
                     call search_index_in_var_groups( k, i_var_group,   &
                                                         i_group_var )
                     print "(/,20X,A,A)", "Named variable is: ",        &
                                          trim(NAMED_EQN_PTR(i_var_group)%name)
                     print "(20X,A,I0)", "Variable number is: ", i_group_var
                  end if
               end if
               return
            end if
         end do

      end if

      ! Multiply by scalar (so PD(:) is overwritten)
      CON = -HL0
      PD(:) = PD(:)*CON
      ! Adding (in place) identity matrix.
      ! Very fast code because the diagonal already exists in PD(:)
      ! (eventually with zero value); this is ok because a test has
      ! already been added in DLSOD.F
      do J = 1, N
         do K = JPD(J), JPD(J+1)-1
            if( IPD(K) == J ) THEN
               PD(K) = PD(K) + 1.0D0
            end if
         end do
      end do

      call do_sparse_factorization()

   END SELECT

contains
!_______________________________________________________________________
!
   subroutine do_full_jacobian()

      do J = 1, N
         ! travelling the lower part of the jacobian...
         do I = J+1, N
            ind      = (J-1)*N + I
            ind_symm = (I-1)*N + J
            WM(2+ind) = WM(2+ind_symm)
         end do
      end do

   end subroutine do_full_jacobian
!_______________________________________________________________________
!
   subroutine do_dense_factorization()

      ! Add identity matrix.
      J = 3 ! (working array WM begins at 3)
      do I = 1, N
         WM(J) = WM(J) + 1.0D0
         J = J + (N + 1)
      end do

      if( jac_symm_pos_def ) then
         ! Do Cholesky decomposition of P.
         call dpotrf( 'U', N, WM(3), N, IER )
         if( IER < 0 ) then
            print *, "(MUESLI:) dpjac: internal error"
            print *, "          illegal parameter in calling dpotrf"
            print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            stop
         else if( IER > 0 ) then
            print *, "(MUESLI OdeSolve:) [BDF/dpjac] ERROR"
            print *, "                   The jacobian matrix should be positive definite!"
            print *, "                   'DPOTRF' routine failed, so it is not SPD."
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            return
         end if
      else
         ! Do LU decomposition of P.
         call dgetrf( N, N, WM(3), N, IWM(21), IER )
         if( IER < 0 ) then
            print *, "(MUESLI:) dpjac: internal error"
            print *, "          illegal parameter in calling dgetrf"
            print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            stop
         else if( IER > 0 ) then
            print "()"
            print *, "(MUESLI OdeSolve:) [BDF/dpjac] Warning"
            print *, "          The factorization using DGETRF has been completed,"
            print *, "          but the factor U is exactly singular, and division"
            print *, "          by zero will occur..."
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            return
         end if
      end if

   end subroutine do_dense_factorization
!_______________________________________________________________________
!
   subroutine do_banded_factorization()

      ! Add identity matrix.
      II = MBAND + 2
      do I = 1, N
         WM(II) = WM(II) + 1.0D0
         II = II + MEBAND
      end do

      if( jac_symm_pos_def ) then
         ! Do Cholesky decomposition of P.
         call dpbtrf( 'U', N, MU, WM(3), MEBAND, IER )
         if( IER < 0 ) then
            print *, "(MUESLI:) dpjac: internal error"
            print *, "          illegal parameter in calling dpbtrf"
            print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            stop
         else if( IER > 0 ) then
            print *, "(MUESLI OdeSolve:) [BDF/dpjac] ERROR"
            print *, "                   The jacobian matrix should be positive definite!"
            print *, "                   'DPOTRF' routine failed, so it is not SPD."
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            return
         end if
      else
         ! Do LU decomposition of P.
         call dgbtrf ( N, N, ML, MU, WM(3), MEBAND, IWM(21), IER )
         if( IER < 0 ) then
            print *, "(MUESLI:) dpjac: internal error"
            print *, "          illegal parameter in calling dgbtrf"
            print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            stop
         else if( IER > 0 ) then
            print "()"
            print *, "(MUESLI OdeSolve:) [BDF/dpjac] Warning"
            print *, "          The factorization using DGBTRF has been completed,"
            print *, "          but the factor U is exactly singular, and division"
            print *, "          by zero will occur..."
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            return
         end if
      end if

   end subroutine do_banded_factorization
!_______________________________________________________________________
!
   subroutine do_sparse_factorization()

      if( jac_symm_pos_def ) then
         ! Do sparse Cholesky decomposition of P.
         ! Factors are stored as module data, saved in 'mod_ddebd2.F90'
         call sp_chol_fact( N )
      else
         ! Do sparse LU decomposition of P.
         ! LU factors are stored as module data, saved in 'mod_ddebd2.F90'
         call sp_lu_fact( N )
      end if

   end subroutine do_sparse_factorization
!_______________________________________________________________________
!
end
