!DECK DHSTRT
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 17 oct 2017
!
   SUBROUTINE DHSTRT( DF, NEQ, A, B, Y, YPRIME, ETOL, MORDER, SMALL,    &
                      BIG, SPY, PV, YP, SF, H, istat )

      use mod_mfdebug, only: MF_NUMERICAL_CHECK, muesli_trace
      use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe

      IMPLICIT NONE

      EXTERNAL DF

      INTEGER, intent(in) :: NEQ

      INTEGER :: MORDER, ISTAT

      DOUBLE PRECISION, intent(in) :: A, B, Y(*), YPRIME(*), ETOL(*),   &
            SMALL, BIG

      DOUBLE PRECISION :: SPY(*), PV(*), YP(*), SF(*), H

!***BEGIN PROLOGUE  DHSTRT
!***SUBSIDIARY
!***PURPOSE  Subsidiary to DDEABM, DDEBDF and DDERKF
!***LIBRARY   SLATEC
!***TYPE      DOUBLE PRECISION (HSTART-S, DHSTRT-D)
!***AUTHOR  Watts, H. A., (SNLA)
!***DESCRIPTION
!
!   DHSTRT computes a starting step size to be used in solving initial
!   value problems in ordinary differential equations.
!
! **********************************************************************
!  ABSTRACT
!
!     Subroutine DHSTRT computes a starting step size to be used by an
!     initial value method in solving ordinary differential equations.
!     It is based on an estimate of the local Lipschitz constant for the
!     differential equation (lower bound on a norm of the Jacobian),
!     a bound on the differential equation (first derivative), and
!     a bound on the partial derivative of the equation with respect to
!     the independent variable.
!     (all approximated near the initial point A)
!
!     Subroutine DHSTRT uses a function subprogram DHVNRM for computing
!     a vector norm. The maximum norm is presently utilized though it
!     can easily be replaced by any other vector norm. It is presumed
!     that any replacement norm routine would be carefully coded to
!     prevent unnecessary underflows or overflows from occurring, and
!     also, would not alter the vector or number of components.
!
! **********************************************************************
!  On input you must provide the following
!
!      DF -- This is a subroutine of the form
!                               DF( X, U, UPRIME, IFLAG )
!             which defines the system of first order differential
!             equations to be solved. For the given values of X and the
!             vector  U(*)=(U(1),U(2),...,U(NEQ)), the subroutine must
!             evaluate the NEQ components of the system of differential
!             equations  DU/DX=DF(X,U)  and store the derivatives in the
!             array UPRIME(*), that is,  UPRIME(I) = * DU(I)/DX *  for
!             equations I=1,...,NEQ.
!
!             Subroutine DF must not alter X or U(*). You must declare
!             the name DF in an external statement in your program that
!             calls DHSTRT. You must dimension U and UPRIME in DF.
!
!             IFLAG is always set to zero before the call. The routine
!             DF may change the value of IFLAG to non-zero if it
!             encounters some special conditions (see DRKFS). The
!             routine quit only if IFLAG is negative.
!
!      NEQ -- This is the number of (first order) differential equations
!             to be integrated.
!
!      A -- This is the initial point of integration.
!
!      B -- This is a value of the independent variable used to define
!             the direction of integration. A reasonable choice is to
!             set  B  to the first point at which a solution is desired.
!             You can also use  B, if necessary, to restrict the length
!             of the first integration step because the algorithm will
!             not compute a starting step length which is bigger than
!             ABS(B-A), unless  B  has been chosen too close to  A.
!             (it is presumed that DHSTRT has been called with  B
!             different from  A  on the machine being used. Also see the
!             discussion about the parameter  SMALL.)
!
!      Y(*) -- This is the vector of initial values of the NEQ solution
!             components at the initial point  A.
!
!      YPRIME(*) -- This is the vector of derivatives of the NEQ
!             solution components at the initial point  A.
!             (defined by the differential equations in subroutine DF)
!
!      ETOL -- This is the vector of error tolerances corresponding to
!             the NEQ solution components. It is assumed that all
!             elements are positive. Following the first integration
!             step, the tolerances are expected to be used by the
!             integrator in an error test which roughly requires that
!                        ABS(LOCAL ERROR) <= ETOL
!             for each vector component.
!
!      MORDER -- This is the order of the formula which will be used by
!             the initial value method for taking the first integration
!             step.
!
!      SMALL -- This is a small positive machine dependent constant
!             which is used for protecting against computations with
!             numbers which are too small relative to the precision of
!             floating point arithmetic.  SMALL  should be set to
!             (approximately) the smallest positive DOUBLE PRECISION
!             number such that  (1.+SMALL) > 1.  on the machine being
!             used. The quantity  SMALL**(3/8)  is used in computing
!             increments of variables for approximating derivatives by
!             differences.  Also the algorithm will not compute a
!             starting step length which is smaller than
!             100*SMALL*ABS(A).
!
!      BIG -- This is a large positive machine dependent constant which
!             is used for preventing machine overflows. A reasonable
!             choice is to set big to (approximately) the square root of
!             the largest DOUBLE PRECISION number which can be held in
!             the machine.
!
!      SPY(*), PV(*), YP(*), SF(*) -- These are DOUBLE PRECISION work
!             arrays of length NEQ which provide the routine with needed
!             storage space.
!
! **********************************************************************
!  On Output  (after the return from DHSTRT),
!
!      H -- is an appropriate starting step size to be attempted by the
!             differential equation method.
!
!           All parameters in the call list remain unchanged except for
!           the working arrays SPY(*),PV(*),YP(*), and SF(*).
!
!      ISTAT -- is the return status. It is zero if H is computed.
!           It is non-zero if the routine fails to compute H, for
!           example, if IFLAG is set to a non-zero value in DF.
!
! **********************************************************************
!
!***SEE ALSO  DDEABM, DDEBDF, DDERKF
!***ROUTINES CALLED  DHVNRM
!***REVISION HISTORY  (YYMMDD)
!   820301  DATE WRITTEN
!   910722  Last modification. For details see original file.
!***END PROLOGUE  DHSTRT
!

      INTEGER :: I, J, K, LK
      DOUBLE PRECISION :: ABSDX, DA, DELF, DELY, DFDUB, DFDXB, DX, DY, &
                 FBND, RELPER, SRYDPB, TOLEXP, TOLMIN, TOLP, TOLSUM, YDPB

      ! communication with slatec/drkfs.f, slatec/dfehl.f, slatec/dhstrt.f
      !                    slatec/ddes.f, slatec/dsteps.f
      !                    slatec/dlsod.f, slatec/dstod.f
      !                    [fml_funfun]/msOdeSolve_JacDF
      !                    [fml_funfun]/msOdeSolve_JacUser
      !                    [fml_funfun]/msOdeSolve_JacUserSP
      integer :: nb_step, nb_deriv
      double precision :: dt_min, dt_max
      common /slatec_odesolve_1/ dt_min, dt_max, nb_step, nb_deriv
      integer :: nb_jac, nb_solv
      common /slatec_odesolve_2/ nb_jac, nb_solv

      integer :: iflag

      integer :: i_eqn_group, i_group_eqn

      !.................................................................
      ISTAT = 0

      ! Begin block permitting ...exits to 160

      DX = B - A
      ABSDX = ABS(DX)
      RELPER = SMALL**0.375D0

      ! ................................................................

      ! Compute an approximate bound (DFDXB) on the partial derivative of
      ! the equation with respect to the independent variable. Protect
      ! against an overflow. Also compute a bound (FBND) on the first
      ! derivative locally.

      DA = SIGN( MAX(MIN(RELPER*ABS(A),ABSDX),100.0D0*SMALL*ABS(A)), DX )
      IF( DA == 0.0D0 ) DA = RELPER*DX
      nb_deriv = nb_deriv + 1
      iflag = 0

      call mf_restore_fpe( )
      CALL DF( A+DA, Y, SF, iflag )
      call mf_save_and_disable_fpe( )
      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
         do k = 1, neq
            if( isnan(SF(k)) ) then
               print "(/,A)", "(MUESLI OdeSolve:) [dhstrt] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied DERIV routine."
               print "(20X,A,I0)", "This occured in yprime(k) for k = ", k
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( k, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               return
            end if
         end do
      end if

      if( iflag < 0 ) then
         ISTAT = iflag
         return
      end if
      DO J = 1, NEQ
         YP(J) = SF(J) - YPRIME(J)
      END DO
      DELF = DHVNRM(YP,NEQ)
      DFDXB = BIG
      IF (DELF < BIG*ABS(DA)) DFDXB = DELF/ABS(DA)
      FBND = DHVNRM(SF,NEQ)

      !.................................................................

      ! Compute an estimate (DFDUB) of the local Lipschitz constant for
      ! the system of differential equations. This also represents an
      ! estimate of the norm of the jacobian locally.  Three iterations
      ! (two when NEQ=1) are used to estimate the Lipschitz constant by
      ! numerical differences. The first perturbation vector is based on
      ! the initial derivatives and direction of integration. The second
      ! perturbation vector is formed using another evaluation of the
      ! differential equation.  The third perturbation vector is formed
      ! using perturbations based only on the initial values. Components
      ! that are zero are always changed to non-zero values (except on
      ! the first iteration). When information is available, care is
      ! taken to ensure that components of the perturbation vector have
      ! signs which are consistent with the slopes of local solution
      ! curves. Also choose the largest bound (FBND) for the first
      ! derivative.

      ! Perturbation vector size is held constant for all iterations.
      ! Compute this change from the size of the vector of initial values.
      DELY = RELPER*DHVNRM(Y,NEQ)
      IF( DELY == 0.0D0 ) DELY = RELPER
      DELY = SIGN(DELY,DX)
      DELF = DHVNRM(YPRIME,NEQ)
      FBND = MAX(FBND,DELF)
      if( DELF /= 0.0D0 ) then
      ! Use initial derivatives for first perturbation
         DO J = 1, NEQ
            SPY(J) = YPRIME(J)
            YP(J) = YPRIME(J)
         END DO
      else
         ! Cannot have a null perturbation vector
         DO J = 1, NEQ
            SPY(J) = 0.0D0
            YP(J) = 1.0D0
         END DO
         DELF = DHVNRM(YP,NEQ)
      end if

      DFDUB = 0.0D0
      LK = MIN(NEQ+1,3)
      DO K = 1, LK
         ! Define perturbed vector of initial values
         DO J = 1, NEQ
            PV(J) = Y(J) + DELY*(YP(J)/DELF)
         END DO
         if( K /= 2 ) then
            ! Evaluate derivatives associated with perturbed
            ! vector  and  compute corresponding differences
            nb_deriv = nb_deriv + 1
            iflag = 0

            call mf_restore_fpe( )
            CALL DF( A, PV, YP, iflag )
            call mf_save_and_disable_fpe( )
            ! check NaN values
            if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
               do i = 1, neq
                  if( isnan(YP(i)) ) then
                     print "(/,A)", "(MUESLI OdeSolve:) [dhstrt] ERROR"
                     print "(20X,A)", "A NaN value has been found after calling the", &
                                      "user-supplied DERIV routine."
                     print "(20X,A,I0)", "This occured in yprime(i) for i = ", i
                     if( NAMED_EQN_PRESENCE ) then
                        call search_index_in_eqn_groups( i, i_eqn_group, &
                                                            i_group_eqn )
                        print "(/,20X,A,A)", "Named equation is: ",     &
                                             trim(NAMED_EQN_PTR(i_eqn_group)%name)
                        print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                     end if
                     mf_message_displayed = .true.
                     call muesli_trace( pause="yes" )
                     return
                  end if
               end do
            end if

            if( iflag < 0 ) then
               ISTAT = iflag
               return
            end if
            DO J = 1, NEQ
               PV(J) = YP(J) - YPRIME(J)
            END DO
         else
            ! Use a shifted value of the independent variable
            !                       in computing one estimate
            nb_deriv = nb_deriv + 1
            iflag = 0

            call mf_restore_fpe( )
            CALL DF( A+DA, PV, YP, iflag )
            call mf_save_and_disable_fpe( )
            ! check NaN values
            if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
               do i = 1, neq
                  if( isnan(YP(i)) ) then
                     print "(/,A)", "(MUESLI OdeSolve:) [dhstrt] ERROR"
                     print "(20X,A)", "A NaN value has been found after calling the", &
                                      "user-supplied DERIV routine."
                     print "(20X,A,I0)", "This occured in yprime(i) for i = ", i
                     if( NAMED_EQN_PRESENCE ) then
                        call search_index_in_eqn_groups( i, i_eqn_group, &
                                                            i_group_eqn )
                        print "(/,20X,A,A)", "Named equation is: ",     &
                                             trim(NAMED_EQN_PTR(i_eqn_group)%name)
                        print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                     end if
                     mf_message_displayed = .true.
                     call muesli_trace( pause="yes" )
                     return
                  end if
               end do
            end if

            if( iflag < 0 ) then
               ISTAT = iflag
               return
            end if
            DO J = 1, NEQ
               PV(J) = YP(J) - SF(J)
            END DO
         end if
         ! Choose largest bounds on the first derivative
         !                and a local lipschitz constant
         FBND = MAX(FBND,DHVNRM(YP,NEQ))
         DELF = DHVNRM(PV,NEQ)
         !...Exit
         IF( DELF >= BIG*ABS(DELY) ) GO TO 150
         DFDUB = MAX(DFDUB,DELF/ABS(DELY))
         !......Exit
         IF( K == LK ) GO TO 160
         ! Choose next perturbation vector
         IF (DELF == 0.0D0) DELF = 1.0D0
         DO J = 1, NEQ
            if( K /= 2 ) then
               DY = ABS(PV(J))
               IF( DY == 0.0D0 ) DY = DELF
            else
               DY = Y(J)
               IF( DY == 0.0D0 ) DY = DELY/RELPER
            end if
            IF( SPY(J) == 0.0D0 ) SPY(J) = YP(J)
            IF( SPY(J) /= 0.0D0 ) DY = SIGN(DY,SPY(J))
            YP(J) = DY
         END DO
         DELF = DHVNRM(YP,NEQ)
      END DO
  150 CONTINUE

      ! Protect against an overflow
      DFDUB = BIG
  160 CONTINUE

      ! ................................................................

      ! Compute a bound (YDPB) on the norm of the second derivative

      YDPB = DFDXB + DFDUB*FBND

      !..................................................................

      ! Define the tolerance parameter upon which the starting step
      ! size is to be based.  A value in the middle of the error
      ! tolerance range is selected.

      TOLMIN = BIG
      TOLSUM = 0.0D0
      DO K = 1, NEQ
         TOLEXP = LOG10(ETOL(K))
         TOLMIN = MIN(TOLMIN,TOLEXP)
         TOLSUM = TOLSUM + TOLEXP
      END DO
      TOLP = 10.0D0**(0.5D0*(TOLSUM/NEQ + TOLMIN)/(MORDER+1))

      !.................................................................

      ! Compute a starting step size based on the above first and
      ! second derivative information

      ! Restrict the step length to be not bigger than ABS(B-A).
      ! (unless  B  is too close to  A)
      H = ABSDX

      if( YDPB == 0.0D0 .and. FBND == 0.0D0 ) then

         ! Both first derivative term (FBND) and second derivative term
         ! (YDPB) are zero
         IF( TOLP < 1.0D0 ) H = ABSDX*TOLP
      else

         if( YDPB == 0.0D0 ) then
            ! Only second derivative term (YDPB) is zero
            IF( TOLP < FBND*ABSDX ) H = TOLP/FBND
         else
            ! Second derivative term (YDPB) is non-zero
            SRYDPB = SQRT(0.5D0*YDPB)
            IF( TOLP < SRYDPB*ABSDX ) H = TOLP/SRYDPB
         end if
      end if

      ! Further restrict the step length to be not bigger than  1/DFDUB
      IF( H*DFDUB > 1.0D0 ) H = 1.0D0/DFDUB

      ! Finally, restrict the step length to be not smaller than
      ! 100*SMALL*ABS(A). However, if A=0 and the computed H underflowed
      ! to zero, the algorithm returns SMALL*ABS(B) for the step length.
      H = MAX(H,100.0D0*SMALL*ABS(A))
      IF( H == 0.0D0 ) H = SMALL*ABS(B)

      ! Now set direction of integration
      H = SIGN(H,DX)

   END
