!DECK DFEHL
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 17 oct 2017
!
   SUBROUTINE DFEHL( DF, NEQ, T, Y, H, YP, F1, F2, F3, F4, F5, YS,      &
                     istat, tend )

      use mod_mfdebug, only: MF_NUMERICAL_CHECK, muesli_trace
      use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe

      IMPLICIT NONE

      EXTERNAL :: DF

      INTEGER, intent(in) :: NEQ

      DOUBLE PRECISION, intent(in) :: T, Y(*), H, YP(*)

      DOUBLE PRECISION :: F1(*), F2(*), F3(*), F4(*), F5(*), YS(*)

      INTEGER :: istat

      double precision :: tend

!***BEGIN PROLOGUE  DFEHL
!***SUBSIDIARY
!***PURPOSE  Subsidiary to DDERKF
!***LIBRARY   SLATEC
!***TYPE      DOUBLE PRECISION (DEFEHL-S, DFEHL-D)
!***AUTHOR  Watts, H. A., (SNLA)
!***DESCRIPTION
!
!     Fehlberg Fourth-Fifth Order Runge-Kutta Method
! **********************************************************************
!
!    DFEHL integrates a system of NEQ first order
!    ordinary differential equations of the form
!               DU/DX = DF(X,U)
!    over one step when the vector Y(*) of initial values for U(*) and
!    the vector YP(*) of initial derivatives, satisfying  YP = DF(T,Y),
!    are given at the starting point X=T.
!
!    DFEHL advances the solution over the fixed step H and returns
!    the fifth order (sixth order accurate locally) solution
!    approximation at T+H in the array YS(*).
!    F1, ..., F5 are arrays of dimension NEQ which are needed
!    for internal storage.
!    The formulas have been grouped to control loss of significance.
!    DFEHL should be called with an H not smaller than 13 units of
!    roundoff in T so that the various independent arguments can be
!    distinguished.
!
!    ISTAT is an ouput integer which tell the calling routine that
!    the computation failed (mainly because DF returned a non-zero
!    flag).
!
!    TEND is the value of time for which the integrator must stop
!    (iflag was equal to 3 in the DF subroutine).
!
! **********************************************************************
!
!***SEE ALSO  DDERKF
!***ROUTINES CALLED  (NONE)
!***REVISION HISTORY  (YYMMDD)
!   820301  DATE WRITTEN
!   910722  Last modification. For details see original file.
!***END PROLOGUE  DFEHL
!

!     communication with slatec/drkfs.f, slatec/dfehl.f, slatec/dhstrt.f
!                        slatec/ddes.f, slatec/dsteps.f
!                        slatec/dlsod.f, slatec/dstod.f
!                        [fml_funfun]/msOdeSolve_JacDF
      integer :: nb_step, nb_deriv
      double precision :: dt_min, dt_max
      common /slatec_odesolve_1/ dt_min, dt_max, nb_step, nb_deriv

      INTEGER :: K
      DOUBLE PRECISION :: CH
      DOUBLE PRECISION :: T1, T2, T3, T4, T5

      integer :: iflag

      integer :: i_eqn_group, i_group_eqn

!-----------------------------------------------------------------------

      istat = 0

      CH = H/4.0d0
      DO K = 1, NEQ
         YS(K) = Y(K) + CH*YP(K)
      END DO
      nb_deriv = nb_deriv + 1
      iflag = 0
      T1 = T + CH

      call mf_restore_fpe( )
      CALL DF( T1, YS, F1, iflag )
      call mf_save_and_disable_fpe( )
      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
         do k = 1, neq
            if( isnan(F1(k)) ) then
               print "(/,A)", "(MUESLI OdeSolve:) [drkfs] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied DERIV routine."
               print "(20X,A,I0)", "This occured in yprime(k) for k = ", k
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( k, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               return
            end if
         end do
      end if

      if( iflag /= 0 ) then
!        user exception encountered
         istat = iflag
         tend = T1
         return
      end if

      CH = (3.0d0/32.0d0)*H
      DO K = 1, NEQ
         YS(K) = Y(K) + CH*(YP(K) + 3.0d0*F1(K))
      END DO
      nb_deriv = nb_deriv + 1
      iflag = 0
      T2 = T + (3.0d0/8.0d0)*H

      call mf_restore_fpe( )
      CALL DF( T2, YS, F2, iflag )
      call mf_save_and_disable_fpe( )
      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
         do k = 1, neq
            if( isnan(F2(k)) ) then
               print "(/,A)", "(MUESLI OdeSolve:) [drkfs] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied DERIV routine."
               print "(20X,A,I0)", "This occured in yprime(k) for k = ", k
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( k, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               return
            end if
         end do
      end if

      if( iflag /= 0 ) then
!        user exception encountered
         istat = iflag
         tend = T2
         return
      end if

      CH = H/2197.0d0
      DO K = 1, NEQ
         YS(K) = Y(K)                                                   &
                 + CH                                                   &
                   *(1932.0d0*YP(K) + (7296.0d0*F2(K) - 7200.0d0*F1(K)))
      END DO
      nb_deriv = nb_deriv + 1
      iflag = 0
      T3 = T + (12.0d0/13.0d0)*H

      call mf_restore_fpe( )
      CALL DF( T3, YS, F3, iflag )
      call mf_save_and_disable_fpe( )
      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
         do k = 1, neq
            if( isnan(F3(k)) ) then
               print "(/,A)", "(MUESLI OdeSolve:) [drkfs] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied DERIV routine."
               print "(20X,A,I0)", "This occured in yprime(k) for k = ", k
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( k, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               return
            end if
         end do
      end if

      if( iflag /= 0 ) then
!        user exception encountered
         istat = iflag
         tend = T3
         return
      end if

      CH = H/4104.0d0
      DO K = 1, NEQ
         YS(K) = Y(K)                                                   &
                 + CH                                                   &
                   *((8341.0d0*YP(K) - 845.0d0*F3(K))                   &
                     + (29440.0d0*F2(K) - 32832.0d0*F1(K)))
      END DO
      nb_deriv = nb_deriv + 1
      iflag = 0
      T4 = T + H

      call mf_restore_fpe( )
      CALL DF( T4, YS, F4, iflag )
      call mf_save_and_disable_fpe( )
      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
         do k = 1, neq
            if( isnan(F4(k)) ) then
               print "(/,A)", "(MUESLI OdeSolve:) [drkfs] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied DERIV routine."
               print "(20X,A,I0)", "This occured in yprime(k) for k = ", k
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( k, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               return
            end if
         end do
      end if

      if( iflag /= 0 ) then
!        user exception encountered
         istat = iflag
         tend = T4
         return
      end if

      CH = H/20520.0d0
      DO K = 1, NEQ
         YS(K) = Y(K)                                                   &
                 + CH                                                   &
                   *((-6080.0d0*YP(K)                                   &
                      + (9295.0d0*F3(K) - 5643.0d0*F4(K)))              &
                     + (41040.0d0*F1(K) - 28352.0d0*F2(K)))
      END DO
      nb_deriv = nb_deriv + 1
      iflag = 0
      T5 = T + 0.5d0*H

      call mf_restore_fpe( )
      CALL DF( T5, YS, F5, iflag )
      call mf_save_and_disable_fpe( )
      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. iflag >= 0 ) then
         do k = 1, neq
            if( isnan(F5(k)) ) then
               print "(/,A)", "(MUESLI OdeSolve:) [drkfs] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied DERIV routine."
               print "(20X,A,I0)", "This occured in yprime(k) for k = ", k
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( k, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               return
            end if
         end do
      end if

      if( iflag /= 0 ) then
!        user exception encountered
         istat = iflag
         tend = T5
         return
      end if

!  Compute approximate solution at T+H

      CH = H/7618050.0d0
      DO K = 1, NEQ
         YS(K) = Y(K)                                                   &
                 + CH                                                   &
                   *((902880.0d0*YP(K)                                  &
                      + (3855735.0d0*F3(K) - 1371249.0d0*F4(K)))        &
                     + (3953664.0d0*F2(K) + 277020.0d0*F5(K)))
      END DO

   END
