!DECK DDEBDF
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 8 feb 2022
!
   SUBROUTINE DDEBDF( DF, NEQ, T, Y, TOUT, INFO, RTOL, ATOL, IDID,      &
                      RWORK, LRW, IWORK, LIW, DJAC, SPDJAC,             &
                      CHECK_JAC, PRINT_CHECK_JAC )

      implicit none

      integer :: NEQ, INFO(15), IDID, LRW, IWORK(*), LIW

      double precision :: T, Y(*), TOUT, RTOL(*), ATOL(*), RWORK(*)

      external :: DF, DJAC, SPDJAC

      integer :: CHECK_JAC, PRINT_CHECK_JAC

!***BEGIN PROLOGUE  DDEBDF
!***PURPOSE  Solve an initial value problem in ordinary differential
!            equations using backward differentiation formulas.  It is
!            intended primarily for stiff problems.
!***LIBRARY   SLATEC (DEPAC)
!***CATEGORY  I1A2
!***TYPE      DOUBLE PRECISION (DEBDF-S, DDEBDF-D)
!***KEYWORDS  BACKWARD DIFFERENTIATION FORMULAS, DEPAC,
!             INITIAL VALUE PROBLEMS, ODE,
!             ORDINARY DIFFERENTIAL EQUATIONS, STIFF
!***AUTHOR  Shampine, L. F., (SNLA)
!           Watts, H. A., (SNLA)
!***DESCRIPTION
!
!   This is the backward differentiation code in the package of
!   differential equation solvers DEPAC, consisting of the codes
!   DDERKF, DDEABM, and DDEBDF.  Design of the package was by
!   L. F. Shampine and H. A. Watts.  It is documented in
!        SAND-79-2374, DEPAC - Design of a User Oriented Package of ODE
!                              Solvers.
!   DDEBDF is a driver for a modification of the code LSODE written by
!             A. C. Hindmarsh
!             Lawrence Livermore Laboratory
!             Livermore, California 94550
!
! **********************************************************************
! **             DEPAC PACKAGE OVERVIEW                               **
! **********************************************************************
!
!   You have a choice of three differential equation solvers from DEPAC.
!   The following brief descriptions are meant to aid you in choosing the
!   most appropriate code for your problem.
!
!   DDERKF is a fifth order Runge-Kutta code. It is the simplest of the
!   three choices, both algorithmically and in the use of the code.
!   DDERKF is primarily designed to solve non-stiff and mildly stiff
!   differential equations when derivative evaluations are not expensive.
!   It should generally not be used to get high accuracy results nor
!   answers at a great many specific points. Because DDERKF has very low
!   overhead costs, it will usually result in the least expensive
!   integration when solving problems requiring a modest amount of
!   accuracy and having equations that are not costly to evaluate.
!   DDERKF attempts to discover when it is not suitable for the task posed.
!
!   DDEABM is a variable order (one through twelve) Adams code. Its
!   complexity lies somewhere between that of DDERKF and DDEBDF. DDEABM
!   is primarily designed to solve non-stiff and mildly stiff differential
!   equations when derivative evaluations are expensive, high accuracy
!   results are needed or answers at many specific points are required.
!   DDEABM attempts to discover when it is not suitable for the task posed.
!
!   DDEBDF is a variable order (one through five) backward differentiation
!   formula code.  It is the most complicated of the three choices. DDEBDF
!   is primarily designed to solve stiff differential equations at crude
!   to moderate tolerances. If the problem is very stiff at all, DDERKF
!   and DDEABM will be quite inefficient compared to DDEBDF. However,
!   DDEBDF will be inefficient compared to DDERKF and DDEABM on non-stiff
!   problems because it uses much more storage, has a much larger overhead,
!   and the low order formulas will not give high accuracies efficiently.
!
!   The concept of stiffness cannot be described in a few words. If you do
!   not know the problem to be stiff, try either DDERKF or DDEABM. Both of
!   these codes will inform you of stiffness when the cost of solving such
!   problems becomes important.
!
! **********************************************************************
! ** ABSTRACT                                                         **
! **********************************************************************
!
!   Subroutine DDEBDF uses the backward differentiation formulas of orders
!   one through five to integrate a system of NEQ first order ordinary
!   differential equations of the form
!                         DU/DX = DF(X,U)
!   when the vector Y(*) of initial values for U(*) at X=T is given.
!   The subroutine integrates from T to TOUT. It is easy to continue the
!   integration to get results at additional TOUT. This is the interval
!   mode of operation. It is also easy for the routine to return with
!   the solution at each intermediate step on the way to TOUT. This is
!   the intermediate-output mode of operation.
!
! **********************************************************************
! ** Description of the arguments to DDEBDF (An overview)             **
! **********************************************************************
!
!   The Parameters are:
!
!      DF -- This is the name of a subroutine which you provide to
!             define the differential equations.
!
!      NEQ -- This is the number of (first order) differential equations
!             to be integrated.
!
!      T -- This is a DOUBLE PRECISION value of the independent variable.
!
!      Y(*) -- This DOUBLE PRECISION array contains the solution
!             components at T.
!
!      TOUT -- This is a DOUBLE PRECISION point at which a solution is
!             desired.
!
!      INFO(*) -- The basic task of the code is to integrate the
!             differential equations from T to TOUT and return an answer
!             at TOUT.  INFO(*) is an INTEGER array which is used to
!             communicate exactly how you want this task to be carried out.
!
!      RTOL, ATOL -- These DOUBLE PRECISION quantities represent relative
!             and absolute error tolerances which you provide to indicate
!             how accurately you wish the solution to be computed. You may
!             choose them to be both scalars or else both vectors.
!
!      IDID -- This scalar quantity is an indicator reporting what the
!             code did.  You must monitor this INTEGER variable to decide
!             what action to take next.
!
!      RWORK(*), LRW -- RWORK(*) is a DOUBLE PRECISION work array of length
!             LRW which provides the code with needed storage space.
!
!      IWORK(*), LIW -- IWORK(*) is an INTEGER work array of length LIW
!             which provides the code with needed storage space and an
!             across call flag.
!
!      DJAC -- This is the name of a subroutine which you may choose to
!             provide for defining the Jacobian matrix of partial
!             derivatives DF/DU.
!
!  Quantities which are used as input items are
!     NEQ, T, Y(*), TOUT, INFO(*), RTOL, ATOL, RWORK(1), LRW, IWORK(1),
!     IWORK(2), and LIW.
!
!  Quantities which may be altered by the code are
!     T, Y(*), INFO(1), INFO(8), RTOL, ATOL, IDID, RWORK(*) and IWORK(*).
!
! **********************************************************************
! ** INPUT -- What to do on the first call to DDEBDF                  **
! **********************************************************************
!
!   The first call of the code is defined to be the start of each new
!   problem.  Read through the descriptions of all the following items,
!   provide sufficient storage space for designated arrays, set
!   appropriate variables for the initialization of the problem, and
!   give information about how you want the problem to be solved.
!
!
!      DF -- Provide a subroutine of the form
!                               DF( X, U, UPRIME, IFLAG )
!             to define the system of first order differential equations
!             which is to be solved. For the given values of X and the
!             vector  U(*)=(U(1),U(2),...,U(NEQ)), the subroutine must
!             evaluate the NEQ components of the system of differential
!             equations  DU/DX=DF(X,U)  and store the derivatives in the
!             array UPRIME(*), that is,  UPRIME(I) = * DU(I)/DX *  for
!             equations I=1,...,NEQ.
!
!             Subroutine DF must not alter X or U(*). You must declare
!             the name DF in an external statement in your program that
!             calls DDEBDF. You must dimension U and UPRIME in DF.
!
!             IFLAG is an integer flag which is always equal to zero on
!             input. Subroutine DF should alter IFLAG only if it encounters
!             an illegal value of Y or a stop condition.
!               * set IFLAG = -1 if an input value is illegal, and DDEBDF
!                 will try to solve the problem without getting IFLAG = -1.
!                 If IFLAG = -1 cannot be avoid, DDEBDF will return control
!                 to the calling program with IDID = -10.
!               * if IFLAG = -2, DDEBDF will return control to the calling
!                 program with IDID = -11.
!
!      NEQ -- Set it to the number of differential equations. (NEQ >= 1)
!
!      T -- Set it to the initial point of the integration.
!             You must use a program variable for T because the code
!             changes its value.
!
!      Y(*) -- Set this vector to the initial values of the NEQ solution
!             components at the initial point. You must dimension Y at
!             least NEQ in your calling program.
!
!      TOUT -- Set it to the first point at which a solution is desired.
!             You can take TOUT = T, in which case the code will evaluate
!             the derivative of the solution at T and return. Integration
!             either forward in T (TOUT > T) or backward in T (TOUT < T)
!             is permitted.
!
!             The code advances the solution from T to TOUT using step
!             sizes which are automatically selected so as to achieve the
!             desired accuracy. If you wish, the code will return with the
!             solution and its derivative following each intermediate step
!             (intermediate-output mode) so that you can monitor them, but
!             you still must provide TOUT in accord with the basic aim of
!             the code.
!
!             The first step taken by the code is a critical one because
!             it must reflect how fast the solution changes near the
!             initial point. The code automatically selects an initial
!             step size which is practically always suitable for the
!             problem. By using the fact that the code will not step past
!             TOUT in the first step, you could, if necessary, restrict
!             the length of the initial step size. HOW???
!
!             For some problems it may not be permissible to integrate
!             past a point TSTOP because a discontinuity occurs there
!             or the solution or its derivative is not defined beyond
!             TSTOP.  When you have declared a TSTOP point (see INFO(4)
!             and RWORK(1)), you have told the code not to integrate
!             past TSTOP.  In this case any TOUT beyond TSTOP is invalid
!             input.
!
!      INFO(*) -- Use the INFO array to give the code more details about
!             how you want your problem solved.  This array should be
!             dimensioned of length 15 to accommodate other members of
!             DEPAC or possible future extensions, though DDEBDF uses
!             only the first eight entries.  You must respond to all of
!             the following items which are arranged as questions.  The
!             simplest use of the code corresponds to answering all
!             questions as YES, i.e. setting all entries of INFO to 0.
!
!        INFO(1) -- This parameter enables the code to initialize
!               itself.  You must set it to indicate the start of every
!               new problem.
!
!            **** Is this the first call for this problem?
!                  YES -- Set INFO(1) = 0
!                   NO -- Not applicable here.
!                         See below for continuation calls. ****
!
!        INFO(2) -- How much accuracy you want of your solution
!               is specified by the error tolerances RTOL and ATOL.
!               The simplest use is to take them both to be scalars.
!               To obtain more flexibility, they can both be vectors.
!               The code must be told your choice.
!
!            **** Are both error tolerances RTOL, ATOL scalars?
!                  YES -- Set INFO(2) = 0
!                         and input scalars for both RTOL and ATOL
!                   NO -- Set INFO(2) = 1
!                         and input arrays for both RTOL and ATOL ****
!
!        INFO(3) -- The code integrates from T in the direction
!               of TOUT by steps.  If you wish, it will return the
!               computed solution and derivative at the next
!               intermediate step (the intermediate-output mode) or
!               TOUT, whichever comes first.  This is a good way to
!               proceed if you want to see the behavior of the solution.
!               If you must have solutions at a great number of specific
!               TOUT points, this way will compute them efficiently.
!
!            **** Do you want the solution only at TOUT (and NOT at the
!                 next intermediate step)?
!                  YES -- Set INFO(3) = 0
!                   NO -- Set INFO(3) = 1 ****
!
!        INFO(4) -- To handle solutions at a great number of specific
!               values TOUT efficiently, this code may integrate past
!               TOUT and interpolate to obtain the result at TOUT.
!               Sometimes it is not possible to integrate beyond some
!               point TSTOP because the equation changes there or it is
!               not defined past TSTOP.  Then you must tell the code
!               not to go past.
!
!            **** Can the integration be carried out without any
!                 restrictions on the independent variable T?
!                  YES -- Set INFO(4)=0
!                   NO -- Set INFO(4)=1
!                         and define the stopping point TSTOP by
!                         setting RWORK(1)=TSTOP ****
!
!        INFO(5) -- To solve stiff problems it is necessary to use the
!               Jacobian matrix of partial derivatives of the system
!               of differential equations.  If you do not provide a
!               subroutine to evaluate it analytically (see the
!               description of the item DJAC in the call list), it will
!               be approximated by numerical differencing in this code.
!               Although it is less trouble for you to have the code
!               compute partial derivatives by numerical differencing,
!               the solution will be more reliable if you provide the
!               derivatives via DJAC.  Sometimes numerical differencing
!               is cheaper than evaluating derivatives in DJAC and
!               sometimes it is not - this depends on your problem.
!
!               If your problem is linear, i.e. has the form
!               DU/DX = DF(X,U) = J(X)*U + G(X)   for some matrix J(X)
!               and vector G(X), the Jacobian matrix  DF/DU = J(X).
!               Since you must provide a subroutine to evaluate DF(X,U)
!               analytically, it is little extra trouble to provide
!               subroutine DJAC for evaluating J(X) analytically.
!               Furthermore, in such cases, numerical differencing is
!               much more expensive than analytic evaluation.
!
!            **** Do you want the code to evaluate the partial
!                 derivatives automatically by numerical differences?
!                  YES -- Set INFO(5)=0
!                   NO -- Set INFO(5)=1
!                         and provide subroutine DJAC for evaluating the
!                         Jacobian matrix ****
!
!        INFO(6) -- DDEBDF will perform much better if the Jacobian
!               matrix is banded and the code is told this.  In this
!               case, the storage needed will be greatly reduced,
!               numerical differencing will be performed more cheaply,
!               and a number of important algorithms will execute much
!               faster.  The differential equation is said to have
!               half-bandwidths ML (lower) and MU (upper) if equation I
!               involves only unknowns Y(J) with
!                              I-ML <= J <= I+MU
!               for all I=1,2,...,NEQ.  Thus, ML and MU are the widths
!               of the lower and upper parts of the band, respectively,
!               with the main diagonal being excluded.  If you do not
!               indicate that the equation has a banded Jacobian,
!               the code works with a full matrix of NEQ**2 elements
!               (stored in the conventional way).  Computations with
!               banded matrices cost less time and storage than with
!               full matrices if  2*ML+MU < NEQ.  If you tell the
!               code that the Jacobian matrix has a banded structure and
!               you want to provide subroutine DJAC to compute the
!               partial derivatives, then you must be careful to store
!               the elements of the Jacobian matrix in the special form
!               indicated in the description of DJAC.
!
!            **** Do you want to solve the problem using a full
!                 (dense) Jacobian matrix (and not a special banded
!                 structure)?
!                  YES -- Set INFO(6)=0
!                   NO -- Set INFO(6)=1
!                         and provide the lower (ML) and upper (MU)
!                         bandwidths by setting
!                         IWORK(1)=ML
!                         IWORK(2)=MU ****
!
!        INFO(7) -- The Jacobian matrix may have a sparse structure
!
!            **** Do you want to solve the problem using a dense
!                 (either full or banded) Jacobian matrix (and not a
!                  sparse structure)?
!                  YES -- Set INFO(7)=0
!                   NO -- Set INFO(7)=1 (sparse) ****
!
!        INFO(8) -- The DDEDBF call corresponds to a new run
!
!            **** Is it the first call in a group of calls (interval mode)?
!                 (This corresponds to a new call from the user side)
!                  YES -- Set INFO(8)=0
!                   NO -- Set INFO(8)=1 ****
!
!      RTOL, ATOL -- You must assign relative (RTOL) and absolute (ATOL)
!             error tolerances to tell the code how accurately you want
!             the solution to be computed.  They must be defined as
!             program variables because the code may change them.  You
!             have two choices --
!                  Both RTOL and ATOL are scalars. (INFO(2)=0)
!                  Both RTOL and ATOL are vectors. (INFO(2)=1)
!             In either case all components must be non-negative.
!
!             The tolerances are used by the code in a local error test
!             at each step which requires roughly that
!                     ABS(LOCAL ERROR) <= RTOL*ABS(Y)+ATOL
!             for each vector component.
!             (More specifically, a root-mean-square norm is used to
!             measure the size of vectors, and the error test uses the
!             magnitude of the solution at the beginning of the step.)
!
!             The true (global) error is the difference between the true
!             solution of the initial value problem and the computed
!             approximation. Practically all present day codes, including
!             this one, control the local error at each step and do not
!             even attempt to control the global error directly. Roughly
!             speaking, they produce a solution Y(T) which satisfies the
!             differential equations with a residual R(T),
!                     DY(T)/DT = DF(T,Y(T)) + R(T),
!             and, almost always, R(T) is bounded by the error tolerances.
!             Usually, but not always, the true accuracy of the computed
!             Y is comparable to the error tolerances. This code will
!             usually, but not always, deliver a more accurate solution
!             if you reduce the tolerances and integrate again. By
!             comparing two such solutions you can get a fairly reliable
!             idea of the true error in the solution at the bigger
!             tolerances.
!
!             Setting ATOL=0. results in a pure relative error test on
!             that component. Setting RTOL=0. results in a pure absolute
!             error test on that component.  A mixed test with non-zero
!             RTOL and ATOL corresponds roughly to a relative error test
!             when the solution component is much bigger than ATOL and to
!             an absolute error test when the solution component is smaller
!             than the threshold ATOL.
!
!             Proper selection of the absolute error control parameters
!             ATOL  requires you to have some idea of the scale of the
!             solution components.  To acquire this information may mean
!             that you will have to solve the problem more than once. In
!             the absence of scale information, you should ask for some
!             relative accuracy in all the components (by setting  RTOL
!             values non-zero) and perhaps impose extremely small
!             absolute error tolerances to protect against the danger of
!             a solution component becoming zero.
!
!             The code will not attempt to compute a solution at an
!             accuracy unreasonable for the machine being used.  It will
!             advise you if you ask for too much accuracy and inform
!             you as to the maximum accuracy it believes possible.
!
!      RWORK(*) -- Dimension this DOUBLE PRECISION work array of length
!             LRW in your calling program.
!
!      RWORK(1) -- If you have set INFO(4)=0, you can ignore this
!             optional input parameter.  Otherwise you must define a
!             stopping point TSTOP by setting   RWORK(1) = TSTOP.
!             (For some problems it may not be permissible to integrate
!             past a point TSTOP because a discontinuity occurs there
!             or the solution or its derivative is not defined beyond
!             TSTOP.)
!
!      LRW -- Set it to the declared length of the RWORK array.
!             You must have
!                  LRW >= 24 + 10*NEQ + NEQ**2
!             for the full (dense) Jacobian case (when INFO(6)=0),
!                  LRW >= 24 + 10*NEQ + (2*ML+MU+1)*NEQ
!             for the banded Jacobian case (when INFO(6)=1), or
!                  LRW >= 24 + 10*NEQ
!             for the sparse Jacobian matrix (when INFO(7)=1).
!
!      IWORK(*) -- Dimension this INTEGER work array of length LIW in
!             your calling program.
!
!      IWORK(1), IWORK(2) -- If you have set INFO(6)=0, you can ignore
!             these optional input parameters. Otherwise you must define
!             the half-bandwidths ML (lower) and MU (upper) of the
!             Jacobian matrix by setting    IWORK(1) = ML   and
!             IWORK(2) = MU.  (The code will work with a full matrix
!             of NEQ**2 elements unless it is told that the problem has
!             a banded Jacobian, in which case the code will work with
!             a matrix containing at most  (2*ML+MU+1)*NEQ  elements.)
!
!      LIW -- Set it to the declared length of the IWORK array.
!             You must have LIW >= 56 + NEQ.
!
!     DJAC -- If you have set INFO(5)=0, you can ignore this parameter
!             by treating it as a dummy argument. (For some compilers
!             you may have to write a dummy subroutine named  DJAC  in
!             order to avoid problems associated with missing external
!             routine names.)  Otherwise, you must provide a subroutine
!             of the form
!                    SUBROUTINE DJAC(X,U,PD,NROWPD)
!             to define the Jacobian matrix of partial derivatives DF/DU
!             of the system of differential equations   DU/DX = DF(X,U).
!             For the given values of X and the vector
!             U(*)=(U(1),U(2),...,U(NEQ)), the subroutine must evaluate
!             the non-zero partial derivatives  DF(I)/DU(J)  for each
!             differential equation I=1,...,NEQ and each solution
!             component J=1, ..., NEQ, and store these values in the
!             matrix PD.  The elements of PD are set to zero before each
!             call to DJAC so only non-zero elements need to be defined.
!
!             Subroutine DJAC must not alter X, U(*), or NROWPD. You
!             must declare the name DJAC in an external statement in
!             your program that calls DDEBDF. NROWPD is the row
!             dimension of the PD matrix and is assigned by the code.
!             Therefore you must dimension PD in DJAC according to
!                    DIMENSION PD(NROWPD,1)
!             You must also dimension U in DJAC.
!
!             The way you must store the elements into the PD matrix
!             depends on the structure of the Jacobian which you
!             indicated by INFO(6).
!             *** INFO(6)=0 -- Full (Dense) Jacobian ***
!                 When you evaluate the (non-zero) partial derivative
!                 of equation I with respect to variable J, you must
!                 store it in PD according to
!                                PD(I,J) = * DF(I)/DU(J) *
!             *** INFO(6)=1 -- Banded Jacobian with ML Lower and MU
!                 Upper Diagonal Bands (refer to INFO(6) description of
!                 ML and MU) ***
!                 When you evaluate the (non-zero) partial derivative
!                 of equation I with respect to variable J, you must
!                 store it in PD according to
!                                IROW = I - J + ML + MU + 1
!                                PD(IROW,J) = * DF(I)/DU(J) *
!
!   SPDJAC -- If INFO(7)=1 (sparse Jacobian), you must provide a
!             subroutine of the form
!                    SUBROUTINE SPDJAC(X,U,NROWPD,JOB,PD,IPD,JPD,NNZ)
!             The Jacobian matrix is defined by the three vectors:
!                    PD, IPD, JPD
!             which correspond to the CSC format (Compact Sparse Column)
!             WARNING: on the first call, this routine must return
!                      only the value of NNZ (arrays PD, IPD, JPD
!                      will not referenced)
!
! **********************************************************************
! ** OUTPUT -- After any return from DDEBDF                           **
! **********************************************************************
!
!   The principal aim of the code is to return a computed solution at
!   TOUT, although it is also possible to obtain intermediate results
!   along the way.  To find out whether the code achieved its goal or
!   if the integration process was interrupted before the task was
!   completed, you must check the IDID parameter.
!
!
!      T -- The solution was successfully advanced to the
!             output value of T.
!
!      Y(*) -- Contains the computed solution approximation at T.
!             The approximate derivative of the solution at T is contained
!             in RWORK(20+1), ..., RWORK(20+NEQ).
!
!      IDID -- Reports what the code did
!
!                         *** Task Completed ***
!                   Reported by positive values of IDID
!
!             IDID = 1 -- (intermediate-output mode only) The code has
!                       successfully reached a requested intermediate
!                       time, but has not yet reached TOUT.
!
!             IDID = 2 -- The integration to TOUT was successfully
!                       completed (T=TOUT) by stepping exactly to TOUT.
!
!             IDID = 3 -- The integration to TOUT was successfully
!                       completed (T=TOUT) by stepping past TOUT.
!                       Y(*) was obtained by interpolation.
!
!             IDID = 12 -- IRES equal to 3 was encountered
!                     and the program tried to finish at the current
!                     time step.
!
!                         *** Task Interrupted ***
!                   Reported by negative values of IDID
!
!             IDID = -1 -- A large amount of work has been expended.
!                       (500 steps attempted)
!
!             IDID = -2 -- The error tolerances are too stringent.
!
!             IDID = -3 -- The local error test cannot be satisfied
!                       because you specified a zero component in ATOL
!                       and the corresponding computed solution
!                       component is zero.  Thus, a pure relative error
!                       test is impossible for this component.
!
!             IDID = -4, -5 -- Not applicable for this code but used
!                       by other members of DEPAC.
!
!             IDID = -6 -- DDEBDF had repeated convergence test failures
!                       on the last attempted step.
!
!             IDID = -7 -- DDEBDF had repeated error test failures on
!                       the last attempted step.
!
!             IDID = -10 -- IFLAG = -1 was encountered in DF and control
!                     is being returned to the calling program, despite
!                     attempts to eliminate this error by decreasing the
!                     time step.
!
!             IDID = -11 -- IFLAG = -2 was encountered in DF and control
!                     is being returned to the calling program.
!
!             IDID = other > -33  -- Not applicable for this code but
!                       used by other members of DEPAC or possible
!                       future extensions.
!
!                         *** Task Terminated ***
!                   Reported by the value of IDID=-33
!
!             IDID = -33 -- The code has encountered trouble from which
!                       it cannot recover.  A message is printed
!                       explaining the trouble and control is returned
!                       to the calling program.  For example, this
!                       occurs when invalid input is detected.
!
!      RTOL, ATOL -- These quantities remain unchanged except when
!             IDID = -2.  In this case, the error tolerances have been
!             increased by the code to values which are estimated to be
!             appropriate for continuing the integration.  However, the
!             reported solution at T was obtained using the input values
!             of RTOL and ATOL.
!
!      RWORK, IWORK -- Contain information which is usually of no
!             interest to the user but necessary for subsequent calls.
!             However, you may find use for
!
!             RWORK(11)--which contains the step size H to be
!                        attempted on the next step.
!
!             RWORK(12)--If the tolerances have been increased by the
!                        code (IDID = -2), they were multiplied by the
!                        value in RWORK(12).
!
!             RWORK(13)--which contains the current value of the
!                        independent variable, i.e. the farthest point
!                        integration has reached.  This will be
!                        different from T only when interpolation has
!                        been performed (IDID=3).
!
!             RWORK(20+I)--which contains the approximate derivative
!                        of the solution component Y(I).  In DDEBDF, it
!                        is never obtained by calling subroutine DF to
!                        evaluate the differential equation using T and
!                        Y(*), except at the initial point of
!                        integration.
!
! **********************************************************************
! ** INPUT -- What to do to continue the integration                  **
! **             (calls after the first)                              **
! **********************************************************************
!
!   This code is organized so that subsequent calls to continue the
!   integration involve little (if any) additional effort on your part.
!   You must monitor the IDID parameter in order to determine what to do
!   next.
!
!   Recalling that the principal task of the code is to integrate from
!   T to TOUT (the interval mode), usually all you will need to do is
!   specify a new TOUT upon reaching the current TOUT.
!
!   Do not alter any quantity not specifically permitted below, in
!   particular do not alter NEQ, T, Y(*), RWORK(*), IWORK(*) or the
!   differential equation in subroutine DF. Any such alteration
!   constitutes a new problem and must be treated as such, i.e. you
!   must start afresh.
!
!   You cannot change from vector to scalar error control or vice versa
!   (INFO(2)) but you can change the size of the entries of RTOL, ATOL.
!   Increasing a tolerance makes the equation easier to integrate.
!   Decreasing a tolerance will make the equation harder to integrate
!   and should generally be avoided.
!
!   You can switch from the intermediate-output mode to the interval
!   mode (INFO(3)) or vice versa at any time.
!
!   If it has been necessary to prevent the integration from going past
!   a point TSTOP (INFO(4), RWORK(1)), keep in mind that the code will
!   not integrate to any TOUT beyond the currently specified TSTOP.
!   Once TSTOP has been reached you must change the value of TSTOP or
!   set INFO(4)=0.  You may change INFO(4) or TSTOP at any time but you
!   must supply the value of TSTOP in RWORK(1) whenever you set INFO(4)=1.
!
!   Do not change INFO(5), INFO(6), IWORK(1), IWORK(2) or INFO(7),
!   unless you are going to restart the code.
!
!   The parameter INFO(1) is used by the code to indicate the beginning
!   of a new problem and to indicate whether integration is to be
!   continued. You must input the value INFO(1)=0 when starting a new
!   problem. You must input the value INFO(1)=1 if you wish to continue
!   after an interrupted task. Do not set INFO(1)=0 on a continuation
!   call unless you want the code to restart at the current T.
!
!                    *** Following a Completed Task ***
!    If
!        IDID = 1, call the code again to continue the integration
!                another step in the direction of TOUT.
!
!        IDID = 2 or 3, define a new TOUT and call the code again.
!                TOUT must be different from T.  You cannot change
!                the direction of integration without restarting.
!
!                    *** Following an Interrupted Task ***
!                To show the code that you realize the task was
!                interrupted and that you want to continue, you
!                must take appropriate action and reset INFO(1) = 1
!    If
!        IDID = -1, the code has attempted 500 steps. If you want to
!                continue, set INFO(1) = 1 and call the code again.
!                An additional 500 steps will be allowed.
!
!        IDID = -2, the error tolerances RTOL, ATOL have been increased
!                to values the code estimates appropriate for
!                continuing. You may want to change them yourself.
!                If you are sure you want to continue with relaxed
!                error tolerances, set INFO(1)=1 and call the code again.
!
!        IDID = -3, a solution component is zero and you set the
!                corresponding component of ATOL to zero. If you are
!                sure you want to continue, you must first alter the
!                error criterion to use positive values for those
!                components of ATOL corresponding to zero solution
!                components, then set INFO(1)=1 and call the code again.
!
!        IDID = -4, -5 -- cannot occur with this code but used by other
!                members of DEPAC.
!
!        IDID = -6, repeated convergence test failures occurred on the
!                last attempted step in DDEBDF.  An inaccurate Jacobian
!                may be the problem. If you are absolutely certain you
!                want to continue, restart the integration at the
!                current T by setting INFO(1)=0 and call the code again.
!
!        IDID = -7, repeated error test failures occurred on the last
!                attempted step in DDEBDF. A singularity in the solution
!                may be present.  You should re-examine the problem
!                being solved.  If you are absolutely certain you want
!                to continue, restart the integration at the current T
!                by setting INFO(1)=0 and call the code again.
!
!        IDID = -10 -- IFLAG=-1 was encountered and control is being
!                returned to the calling program, despite attempts to
!                eliminate this error by decreasing the time step.
!
!        IDID = -11 -- IFLAG=-2 was encountered, and control is being
!                returned to the calling program.
!
!        IDID = other > -33 -- cannot occur with this code but used by
!                other members of DDEPAC or possible future extensions.
!
!                    *** Following a Terminated Task ***
!    If
!        IDID = -33, you cannot continue the solution of this problem.
!                An attempt to do so will result in your run being
!                terminated.
!
! **********************************************************************
!
!      ***** Warning *****
!
!   If DDEBDF is to be used in an overlay situation, you must save and
!   restore certain items used internally by DDEBDF  (values in the
!   common block DDEBD1).  This can be accomplished as follows.
!
!   To save the necessary values upon return from DDEBDF:
!      call DSVCO( RWORK(22+NEQ), IWORK(21+NEQ) )
!
!   To restore the necessary values before the next call to DDEBDF:
!      call DRSCO( RWORK(22+NEQ), IWORK(21+NEQ) )
!
!   Fix: It appears that we have actually to save all the common DDEBD1,
!        plus the whole arrays RWORK and IWORK...
!
!***REFERENCES  L. F. Shampine and H. A. Watts, DEPAC - design of a user
!                 oriented package of ODE solvers, Report SAND79-2374,
!                 Sandia Laboratories, 1979.
!***ROUTINES CALLED  DLSOD, XERMSG
!***COMMON BLOCKS    DDEBD1
!***REVISION HISTORY  (YYMMDD)
!   820301  DATE WRITTEN
!   920501  Last modification. For details see original file.
!***END PROLOGUE  DDEBDF

      INTEGER :: IACOR, IBAND, IBEGIN, IDELSN, IER, IEWT,               &
                 IINOUT, IINTEG, IJAC, ILRW, INIT, IOWNS, IQUIT, ISAVF, &
                 ISPARSE, ITOL, ITSTAR, ITSTOP, IUSERCALL, IWM, IYH,    &
                 IYPOUT, JSTART, KFLAG, KSTEPS, L, MAXORD, METH, MITER, &
                 ML, MU, N, NFE, NJE, NQ, NQU, NST, JAC_OUTDATED
      DOUBLE PRECISION :: EL0, H, HMIN, HMXI, HU, ROWNS, TN, TOLD,      &
                          UROUND
      LOGICAL :: INTOUT
      CHARACTER*8 :: XERN1, XERN2
      CHARACTER*16 :: XERN3

      COMMON /DDEBD1/ TOLD, ROWNS(210), EL0, H, HMIN, HMXI, HU, TN, UROUND, &
                      IQUIT, INIT, IYH, IEWT, IACOR, ISAVF, IWM, KSTEPS, &
                      IBEGIN, ITOL, IINTEG, ITSTOP, IJAC, IBAND, IOWNS(6), &
                      IER, JSTART, KFLAG, L, METH, MITER, MAXORD, N, NQ, &
                      NST, NFE, NJE, NQU, ISPARSE, IUSERCALL, JAC_OUTDATED

      !------ end of declarations -- execution starts hereafter  ------

      IDID = 0

      if( INFO(1) == 0 ) IWORK(LIW) = 0

      IYPOUT = 21                 ! for len and cumul, see at the end
      ITSTAR = IYPOUT + NEQ       ! of the current routine...

      ! Check for an apparent infinite loop
      if( IWORK(LIW) >= 5 ) then
         if( T == RWORK(ITSTAR) ) then
            WRITE (XERN3, '(ES13.6)') T
            CALL XERMSG ('SLATEC', 'DDEBDF',                            &
               'AN APPARENT INFINITE LOOP HAS BEEN DETECTED.$$' //      &
               'YOU HAVE MADE REPEATED CALLS AT T = ' // XERN3 //       &
               ' AND THE INTEGRATION HAS NOT ADVANCED.  CHECK THE ' //  &
               'WAY YOU HAVE SET PARAMETERS FOR THE CALL TO THE ' //    &
               'CODE, PARTICULARLY INFO(1).', 13, 2)
            RETURN
         end if
      end if

      ! Check validity of info parameters
      if( INFO(1) /= 0 .and. INFO(1) /= 1 ) then
         WRITE (XERN1, '(I0)') INFO(1)
         CALL XERMSG ('SLATEC', 'DDEBDF', 'INFO(1) MUST BE SET TO 0 ' //&
            'FOR THE  START OF A NEW PROBLEM, AND MUST BE SET TO 1 ' // &
            'FOLLOWING AN INTERRUPTED TASK.  YOU ARE ATTEMPTING TO ' // &
            'CONTINUE THE INTEGRATION ILLEGALLY BY CALLING THE ' //     &
            'CODE WITH  INFO(1) = ' // XERN1, 3, 1)
         IDID = -33
      end if

      if( INFO(2) /= 0 .and. INFO(2) /= 1 ) then
         WRITE (XERN1, '(I0)') INFO(2)
         CALL XERMSG ('SLATEC', 'DDEBDF', 'INFO(2) MUST BE 0 OR 1 ' //  &
            'INDICATING SCALAR AND VECTOR ERROR TOLERANCES, ' //        &
            'RESPECTIVELY.  YOU HAVE CALLED THE CODE WITH INFO(2) = ' //&
            XERN1, 4, 1)
         IDID = -33
      end if

      if( INFO(3) /= 0 .and. INFO(3) /= 1 ) then
         WRITE (XERN1, '(I0)') INFO(3)
         CALL XERMSG ('SLATEC', 'DDEBDF', 'INFO(3) MUST BE 0 OR 1 ' //  &
            'INDICATING THE INTERVAL OR INTERMEDIATE-OUTPUT MODE OF ' //&
            'INTEGRATION, RESPECTIVELY.  YOU HAVE CALLED THE CODE ' //  &
            'WITH  INFO(3) = ' // XERN1, 5, 1)
         IDID = -33
      end if

      if( INFO(4) /= 0 .and. INFO(4) /= 1 ) then
         WRITE (XERN1, '(I0)') INFO(4)
         CALL XERMSG ('SLATEC', 'DDEBDF', 'INFO(4) MUST BE 0 OR 1 ' //  &
            'INDICATING WHETHER OR NOT THE INTEGRATION INTERVAL IS ' // &
            'TO BE RESTRICTED BY A POINT TSTOP.  YOU HAVE CALLED ' //   &
            'THE CODE WITH INFO(4) = ' // XERN1, 14, 1)
         IDID = -33
      end if

      if( INFO(5) /= 0 .and. INFO(5) /= 1 ) then
         WRITE (XERN1, '(I0)') INFO(5)
         CALL XERMSG ('SLATEC',  'DDEBDF', 'INFO(5) MUST BE 0 OR 1 ' // &
            'INDICATING WHETHER THE CODE IS TOLD TO FORM THE ' //       &
            'JACOBIAN MATRIX BY NUMERICAL DIFFERENCING OR YOU ' //      &
            'PROVIDE A SUBROUTINE TO EVALUATE IT ANALYTICALLY.  ' //    &
            'YOU HAVE CALLED THE CODE WITH INFO(5) = ' // XERN1, 15, 1)
         IDID = -33
      end if

      if( INFO(6) /= 0 .and. INFO(6) /= 1 ) then
         WRITE (XERN1, '(I0)') INFO(6)
         CALL XERMSG ('SLATEC', 'DDEBDF', 'INFO(6) MUST BE 0 OR 1 ' //  &
            'INDICATING WHETHER THE CODE IS TOLD TO TREAT THE ' //      &
            'JACOBIAN AS A FULL (DENSE) MATRIX OR AS HAVING A ' //      &
            'SPECIAL BANDED STRUCTURE.  YOU HAVE CALLED THE CODE ' //   &
            'WITH INFO(6) = ' // XERN1, 16, 1)
         IDID = -33
      end if

      if( INFO(7) /= 0 .and. INFO(7) /= 1 ) then
         WRITE (XERN1, '(I0)') INFO(7)
         CALL XERMSG ('SLATEC', 'DDEBDF', 'INFO(7) MUST BE 0 OR 1 ' //  &
            'INDICATING WHETHER THE CODE IS TOLD TO TREAT THE ' //      &
            'JACOBIAN AS A FULL (DENSE) MATRIX OR AS HAVING A ' //      &
            'SPECIAL SPARSE STRUCTURE.  YOU HAVE CALLED THE CODE ' //   &
            'WITH INFO(7) = ' // XERN1, 16, 1)
         IDID = -33
      end if

      ILRW = NEQ
      if( INFO(6) /= 0 ) then

         ! Check bandwidth parameters
         ML = IWORK(1)
         MU = IWORK(2)
         ILRW = 2*ML + MU + 1

         if( ML < 0 .or. NEQ <= ML .or. MU < 0 .or. NEQ <= MU ) then
            WRITE (XERN1, '(I0)') ML
            WRITE (XERN2, '(I0)') MU
            CALL XERMSG ('SLATEC', 'DDEBDF', 'YOU HAVE SET INFO(6) ' // &
               '= 1, TELLING THE CODE THAT THE JACOBIAN MATRIX HAS ' // &
               'A SPECIAL BANDED STRUCTURE.  HOWEVER, THE LOWER ' //    &
               '(UPPER) BANDWIDTHS  ML (MU) VIOLATE THE CONSTRAINTS ' //&
               'ML,MU >= 0 AND  ML,MU < NEQ.  YOU HAVE CALLED ' // &
               'THE CODE WITH ML = ' // XERN1 // ' AND MU = ' // XERN2, &
               17, 1)
            IDID = -33
         end if
      end if

      ! Sparse jacobian matrix
      if( INFO(7) /= 0 ) then
         ILRW = 0
      end if

      ! Check LRW and LIW for sufficient storage allocation
      if( LRW < 24 + (10 + ILRW)*NEQ ) then
         WRITE (XERN1, '(I0)') LRW
         if( INFO(7) == 0 ) then
            if( INFO(6) == 0 ) then
               CALL XERMSG ('SLATEC', 'DDEBDF', 'LENGTH OF ARRAY RWORK ' //&
                  'MUST BE AT LEAST 24 + 10*NEQ + NEQ*NEQ.$$' //          &
                  'YOU HAVE CALLED THE CODE WITH  LRW = ' // XERN1, 1, 1)
            ELSE
               CALL XERMSG ('SLATEC', 'DDEBDF', 'LENGTH OF ARRAY RWORK ' //&
                  'MUST BE AT LEAST 24 + 10*NEQ + (2*ML+MU+1)*NEQ.$$' //  &
                  'YOU HAVE CALLED THE CODE WITH  LRW = ' // XERN1, 18, 1)
            end if
         ELSE
            CALL XERMSG ('SLATEC', 'DDEBDF', 'LENGTH OF ARRAY RWORK ' //&
               'MUST BE AT LEAST 24 + 10*NEQ.$$' //                    &
               'YOU HAVE CALLED THE CODE WITH  LRW = ' // XERN1, 18, 1)
         end if
         IDID = -33
      end if

      if( LIW < 56 + NEQ ) then
         WRITE (XERN1, '(I0)') LIW
         CALL XERMSG ('SLATEC', 'DDEBDF', 'LENGTH OF ARRAY IWORK ' //   &
            'BE AT LEAST  56 + NEQ.  YOU HAVE CALLED THE CODE WITH ' // &
            'LIW = ' // XERN1, 2, 1)
         IDID = -33
      end if

      ! Indexes for the arrays stored in the IWORK work array
      IINOUT = 21 + NEQ

      if( INFO(1) /= 0 ) INTOUT = IWORK(IINOUT) /= -1

      ! Indexes for the arrays stored in the RWORK work array
     !IYPOUT = 21                 ! len = NEQ         ind = 21
     !ITSTAR = IYPOUT + NEQ       ! len = 1           ind = 21 + NEQ
      IYH = ITSTAR + 1            ! len = 6*NEQ       ind = 22 + NEQ
      IEWT = IYH + 6*NEQ          ! len = NEQ         ind = 22 + 7*NEQ
      ISAVF = IEWT + NEQ          ! len = NEQ         ind = 22 + 8*NEQ
      IACOR = ISAVF + NEQ         ! len = NEQ         ind = 22 + 9*NEQ
      IWM = IACOR + NEQ           ! len = 2+ILRW*NEQ  ind = 22 + 10*NEQ
      IDELSN = IWM + 2 + ILRW*NEQ ! len = 1           ind = 24 + (10+ILRW)*NEQ

      IBEGIN = INFO(1)
      ITOL   = INFO(2)
      IINTEG = INFO(3)
      ITSTOP = INFO(4)
      IJAC   = INFO(5)
      IBAND  = INFO(6)
      RWORK(ITSTAR) = T

      ISPARSE = INFO(7)
      if( IBAND == 1 .and. ISPARSE == 1 ) then
         call XERMSG ('SLATEC', 'DDEBDF', 'INVALID INPUT PARAMETERS:' //&
             ' JACOBIAN MATRIX CANNOT BE BOTH BANDED AND SPARSE', 18, 1)
         IDID = -34
      end if

      IUSERCALL = INFO(8)

      call DLSOD( DF, NEQ, T, Y, TOUT, RTOL, ATOL, IDID, RWORK(IYPOUT), &
                  RWORK(IYH), RWORK(IYH), RWORK(IEWT), RWORK(ISAVF),    &
                  RWORK(IACOR), RWORK(IWM), IWORK(1), DJAC, INTOUT,     &
                  RWORK(1), RWORK(12), RWORK(IDELSN), SPDJAC,           &
                  CHECK_JAC, PRINT_CHECK_JAC )

      if( INTOUT ) then
         IWORK(IINOUT) = 1
      else
         IWORK(IINOUT) = -1
      end if

      if( IDID == 12 ) IDID = -12

      if( IDID /= -2 ) IWORK(LIW) = IWORK(LIW) + 1
      if( T /= RWORK(ITSTAR) ) IWORK(LIW) = 0
      RWORK(11) = H
      RWORK(13) = TN

      ! After a successful initialization in DLSOD, IBEGIN is set to 1.
      INFO(1) = IBEGIN

!### why?
      INFO(8) = IUSERCALL

      ! Additional copies in IWORK (easy access during save/restore)
      !   storing jac_struct (1, 2 or 3) in IWORK(3)
      !   storing NEQ                    in IWORK(4)

      if( INFO(6) == 0 .and. INFO(7) == 0 ) then
         ! Jacobian structure is dense (computed either by a user-routine,
         ! or by the code itself via Finite Differences
         IWORK(3) = 1
      else if( INFO(6) == 1 .and. INFO(7) == 0 ) then
         ! Jacobian structure is banded (computed by a user-routine)
         IWORK(3) = 2
      else ! INFO(6) == 0 .and. INFO(7) == 1
         ! Jacobian structure is sparse (computed by a user-routine)
         IWORK(3) = 3
      end if

      IWORK(4) = NEQ

   END
