!DECK DDASTP
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 31 Jan 2022
!
   SUBROUTINE DDASTP( X, Y, YPRIME, NEQ, RES, JAC, SPJAC, H, WT,        &
         JSTART, IDID, PHI, DELTA, E, WM, IWM, ALPHA, BETA, GAMMA,      &
         PSI, SIGMA, CJ, CJOLD, HOLD, S, HMIN, UROUND, IPHASE, JCALC, K,&
         KOLD, NS, NONNEG, NN_IND, NTEMP,                               &
         CHECK_JAC, PRINT_CHECK_JAC )

!***BEGIN PROLOGUE  DDASTP
!***SUBSIDIARY
!***PURPOSE  Perform one step of the DDASSL integration.
!***LIBRARY   SLATEC (DASSL)
!***TYPE      DOUBLE PRECISION (SDASTP-S, DDASTP-D)
!***AUTHOR  Petzold, Linda R., (LLNL)
!***DESCRIPTION
!-----------------------------------------------------------------------
!     DDASTP solves a system of differential/algebraic equations
!     of the form
!                           G(X,Y,YPRIME) = 0,
!     for one step (normally from X to X+H).
!
!     The methods used are modified divided difference, fixed leading
!     coefficient forms of backward differentiation formulas. The code
!     adjusts the stepsize and order to control the local error per
!     step.
!
!     The parameters represent:
!     X         -- Independent variable.
!     Y         -- Solution vector at X.
!     YPRIME    -- Derivative of solution vector after successful step.
!     NEQ       -- Number of equations to be integrated.
!     RES       -- External user-supplied subroutine to evaluate the
!                  residual. The call is:
!                         CALL RES(X,Y,YPRIME,DELTA,IRES)
!                  X,Y,YPRIME are input.  DELTA is output.
!                  On input, IRES = 0.  RES should alter IRES only
!                  if it encounters an illegal value of Y or a
!                  stop condition.  Set IRES = -1 if an input value
!                  of Y is illegal, and DDASTP will try to solve
!                  the problem without getting IRES = -1.  If
!                  IRES = -2, DDASTP returns control to the calling
!                  program. If IRES = 3, DDASTP will try to finish
!                  its time step exactly at the current time step
!                  which triggered IRES=3.
!     JAC       -- External user-supplied routine to evaluate the
!                  iteration matrix (this is optional). The call is:
!                         CALL JAC(X,Y,YPRIME,PD,CJ,LDJAC)
!                  PD is the matrix of partial derivatives,
!                  PD = DG/DY + CJ*DG/DYPRIME
!     SPJAC     -- External user-supplied routine to evaluate
!                  the iteration matrix (this is optional) under
!                  sparse storage. The call is of the form
!                    CALL SPJAC(X,Y,YPRIME,CJ,NROW,JOB,PD,IPD,JPD,NNZ)
!                  PD is the sparse matrix of partial derivatives,
!                  PD = DG/DY + CJ*DG/DYPRIME
!     H         -- Appropriate step size for next step.
!                  Normally determined by the code
!     WT        -- Vector of weights for error criterion.
!     JSTART    -- Integer variable: set 0 for first step, 1 otherwise.
!     IDID      -- Completion code with the following meanings:
!                    IDID =   1 -- The step was completed successfully
!                    IDID =  12 -- IRES equal to 3 was encountered,
!                                  and the step has been processed
!                                  with an appropriate stepsize.
!                    IDID =  -6 -- The error test failed repeatedly
!                    IDID =  -7 -- The corrector could not converge
!                    IDID =  -8 -- The iteration matrix is singular
!                    IDID =  -9 -- The corrector could not converge:
!                                  there were repeated error test
!                                  failures on this step.
!                    IDID = -10 -- The corrector could not converge
!                                  because IRES was equal to -1
!                    IDID = -11 -- IRES equal to -2 was encountered,
!                                  and control is being returned to
!                                  the calling program.
!                    IDID = -14 -- A NaN value has been found in DELTA,
!                                  after calling RES().
!     PHI       -- Array of divided differences used by DDASTP.
!                  The length is NEQ*(K+1), where K is the maximum
!                  order.
!     DELTA, E  -- Work vectors for DDASTP of length NEQ
!     WM, IWM   -- Real and integer arrays storing matrix information
!                  such as the matrix of partial derivatives,
!                  permutation vector, and various other information.
!
!     NONNEG    -- Number of non-negativity constraints to be applied
!                  to some equations. Index of these equations are
!                  described in the NN_IND(*) integer array.
!     NN_IND(*) -- Integer array, of size at least NONNEG, which
!                  contains indexes of equations which must verified
!                  a non-negative constraint.
!
!     The other parameters are information which is needed internally
!     by DDASTP to continue from step to step.
!
!-----------------------------------------------------------------------
!***ROUTINES CALLED  DDAJAC, DDANRM, DDASLV, DDATRP
!***REVISION HISTORY  (YYMMDD)
!   830315  DATE WRITTEN
!   000711  Last modification. For details see original file.
!***END PROLOGUE  DDASTP

      use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe
      use mod_core, only: msPause
      use mod_mfdebug, only: mf_message_displayed, muesli_trace,        &
                             MF_NUMERICAL_CHECK
      use mod_ieee, only: MF_NAN

      IMPLICIT NONE

      INTEGER :: NEQ, JSTART, IDID, IWM(*), IPHASE, JCALC, K,           &
                 KOLD, NS, NONNEG, NN_IND(*), NTEMP
      DOUBLE PRECISION :: X, Y(*), YPRIME(*), H, WT(*), PHI(NEQ,*),     &
                 DELTA(*), E(*), WM(*), ALPHA(*), BETA(*), GAMMA(*),    &
                 PSI(*), SIGMA(*), CJ, CJOLD, HOLD, S, HMIN, UROUND
      EXTERNAL :: RES, JAC, SPJAC
      INTEGER :: CHECK_JAC, PRINT_CHECK_JAC

      INTEGER :: I, IER, IRES, J, J1, KDIFF, KM1, KNEW, KP1, KP2, M,    &
                 NCF, NEF, NSF, NSP1, II, MTYPE
      DOUBLE PRECISION :: ALPHA0, ALPHAS, CJLAST, CK, DELNRM, ENORM,    &
                 ERK, ERKM1, ERKM2, ERKP1, ERR, EST, HNEW, OLDNRM,      &
                 PNORM, R, RATE, TEMP1, TEMP2, TERK, TERKM1, TERKM2,    &
                 TERKP1, XOLD
      LOGICAL :: CONVGD

      integer, parameter :: LMXORD = 3, LNST = 11, LNRE = 12, LNJE = 13, &
                            LETF = 14, LCTF = 15, LMTYPE=4

      integer, save :: MAXIT = 4
      double precision, save :: XRATE = 0.25D0

      double precision, parameter :: H_FACTOR = 2.0d0

      double precision :: dt_min, dt_max
      integer :: nb_step, nb_resid
      common /slatec_daesolve_1/ dt_min, dt_max, nb_step, nb_resid

      double precision :: cpu_time_0
      integer :: inside_init, nb_resid_0, nb_jac_0, nb_solv_0
      common /slatec_daesolve_3/ cpu_time_0, inside_init, nb_resid_0,   &
                                 nb_jac_0, nb_solv_0

      ! Variables added for stopping conditions
      logical :: end_condition_found

      integer :: i_eqn_group, i_group_eqn

      !-----------------------------------------------------------------------
      ! BLOCK 1.
      ! Initialize. on the first call, set the order to 1 and initialize
      ! other variables.
      !-----------------------------------------------------------------------

      ! Initializations for all calls
      !***First executable statement  DDASTP
      IDID = 1
      XOLD = X
      NCF = 0
      NSF = 0
      NEF = 0
      end_condition_found = .false.
      if( JSTART == 0 ) then

         ! If this is the first step, perform other initializations
         IWM(LETF) = 0
         IWM(LCTF) = 0
         K = 1
         KOLD = 0
         HOLD = 0.0D0
         JSTART = 1
         PSI(1) = H
         CJOLD = 1.0D0/H
         CJ = CJOLD
         S = 100.D0
         JCALC = -1
         DELNRM = 1.0D0
         IPHASE = 0
         NS = 0

      end if

      !-----------------------------------------------------------------------
      ! BLOCK 2
      ! Compute coefficients of formulas for this step.
      !-----------------------------------------------------------------------
  200 CONTINUE
      KP1 = K + 1
      KP2 = K + 2
      KM1 = K - 1
      XOLD = X
      IF( H /= HOLD .OR. K /= KOLD ) NS = 0
      NS = MIN(NS+1,KOLD+2)
      NSP1 = NS + 1
      IF( KP1 < NS ) GO TO 230

      BETA(1) = 1.0D0
      ALPHA(1) = 1.0D0
      TEMP1 = H
      GAMMA(1) = 0.0D0
      SIGMA(1) = 1.0D0
      DO I = 2, KP1
         TEMP2 = PSI(I-1)
         PSI(I-1) = TEMP1
         BETA(I) = BETA(I-1)*PSI(I-1)/TEMP2
         TEMP1 = TEMP2+H
         ALPHA(I) = H/TEMP1
         SIGMA(I) = (I-1)*SIGMA(I-1)*ALPHA(I)
         GAMMA(I) = GAMMA(I-1) + ALPHA(I-1)/H
      ENDDO
      PSI(KP1) = TEMP1
  230 CONTINUE

      ! Compute ALPHAS, ALPHA0
      ALPHAS = 0.0D0
      ALPHA0 = 0.0D0
      DO I = 1, K
        ALPHAS = ALPHAS - 1.0D0/I
        ALPHA0 = ALPHA0 - ALPHA(I)
        CONTINUE
      ENDDO

      ! Compute leading coefficient CJ
      CJLAST = CJ
      CJ = -ALPHAS/H

      ! Compute variable stepsize error coefficient CK
      CK = ABS(ALPHA(KP1) + ALPHAS - ALPHA0)
      CK = MAX(CK,ALPHA(KP1))

      ! Decide whether new jacobian is needed
      TEMP1 = (1.0D0 - XRATE)/(1.0D0 + XRATE)
      TEMP2 = 1.0D0/TEMP1
      IF( CJ/CJOLD < TEMP1 .OR. CJ/CJOLD > TEMP2 ) JCALC = -1
      IF( CJ /= CJLAST ) S = 100.D0

      ! Change PHI to PHI-star
      IF( KP1 < NSP1 ) GO TO 280
      DO J = NSP1, KP1
         DO I = 1, NEQ
            PHI(I,J) = BETA(J)*PHI(I,J)
         ENDDO
      ENDDO
  280 CONTINUE

      ! Update time
      X = X + H

      !-----------------------------------------------------------------------
      ! BLOCK 3
      ! Predict the solution and derivative, and solve the corrector
      ! equation
      !-----------------------------------------------------------------------

      ! First, predict the solution and derivative
  300 CONTINUE
      DO I = 1, NEQ
         Y(I) = PHI(I,1)
         YPRIME(I) = 0.0D0
      ENDDO
      DO J = 2, KP1
         DO I = 1, NEQ
            Y(I) = Y(I) + PHI(I,J)
            YPRIME(I) = YPRIME(I) + GAMMA(J)*PHI(I,J)
         ENDDO
      ENDDO
      PNORM = DDANRM( NEQ, Y, WT )

      ! Solve the corrector equation using a modified Newton scheme.
      CONVGD = .TRUE.
      M = 0
      IWM(LNRE) = IWM(LNRE) + 1
      IRES = 0
      nb_resid = nb_resid + 1
      if( inside_init == 1 ) nb_resid_0 = nb_resid_0 + 1
      if( MF_NUMERICAL_CHECK ) then
         ! To be sure that, in case of use of an End Condition, all the
         ! coefficients of the vector RESID are effectively computed...
         DELTA(1:NEQ) = MF_NAN
      endif

      call mf_restore_fpe( )
      CALL RES( X, Y, YPRIME, DELTA, IRES )
      call mf_save_and_disable_fpe( )

      if( MF_NUMERICAL_CHECK .and. IRES >= 0 ) then
         do i = 1, neq
            if( isnan(delta(i)) ) then
               print "(/,A)", "(MUESLI DaeSolve:) [BDF/ddastp] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied RESID routine."
               print "(20X,A,I0)", "This occured in delta(i) for i = ", i
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( i, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               IDID = -14
               RETURN
            end if
         end do
      endif
      IF( IRES < 0 ) GO TO 380
      if( IRES == 3 ) then
         end_condition_found = .true.
      end if

      ! If indicated, reevaluate the iteration matrix
      ! PD = DG/DY + CJ*DG/DYPRIME (where G(X,Y,YPRIME) = 0).
      ! Set JCALC to 0 as an indicator that this has been done.
      IF( JCALC /= -1 ) GO TO 340
      IWM(LNJE) = IWM(LNJE) + 1
      JCALC = 0
      CALL DDAJAC( NEQ, X, Y, YPRIME, DELTA, CJ, H, IER, WT, E,         &
                   WM, IWM, RES, IRES, UROUND, JAC, SPJAC, NTEMP,       &
                   CHECK_JAC, PRINT_CHECK_JAC )
      if( ddajac_must_quit ) then
         ! Singular Jacobian (case handled by dgefa_info)
         IDID = -8
         RETURN
      end if
      CJOLD = CJ
      S = 100.D0
      IF( IRES < 0 ) GO TO 380
      IF( IER /= 0 ) GO TO 380
      NSF = 0

      ! Initialize the error accumulation vector E.
  340 CONTINUE
      DO I = 1, NEQ
         E(I) = 0.0D0
      ENDDO

      ! Corrector loop.
  350 CONTINUE

      ! Multiply residual by TEMP1 to accelerate convergence
      TEMP1 = 2.0D0/(1.0D0 + CJ/CJOLD)
      DO I = 1, NEQ
         DELTA(I) = DELTA(I) * TEMP1
      ENDDO

      MTYPE = IWM(LMTYPE)
      if( MTYPE == 3 ) then
         ! Sparse jacobian
         if( numeric == 0 .and. symbolic == 0 ) then
            ! This means that the jacobian is not ready (due to
            ! continuation?) => set up again of the jacobian
            CALL DDAJAC( NEQ, X, Y, YPRIME, DELTA, CJ, H, IER, WT, E,   &
                         WM, IWM, RES, IRES, UROUND, JAC, SPJAC, NTEMP, &
                         CHECK_JAC, PRINT_CHECK_JAC )
         end if
      end if

      ! Compute a new iterate (back-substitution).
      ! Store the correction in DELTA.
      CALL DDASLV( NEQ, DELTA, WM, IWM )

      ! Update Y, E and YPRIME
      DO I = 1, NEQ
         Y(I) = Y(I) - DELTA(I)
         E(I) = E(I) - DELTA(I)
         YPRIME(I) = YPRIME(I) - CJ*DELTA(I)
      ENDDO

      ! Test for convergence of the iteration
      DELNRM = DDANRM( NEQ, DELTA, WT )
      IF( DELNRM <= 100.D0*UROUND*PNORM ) GO TO 375
      IF( M > 0 ) GO TO 365
         OLDNRM = DELNRM
         GO TO 367
  365 RATE = (DELNRM/OLDNRM)**(1.0D0/M)
      IF( RATE > 0.90D0 ) GO TO 370
      S = RATE/(1.0D0 - RATE)
  367 IF( S*DELNRM <= 0.33D0 ) GO TO 375

      ! The corrector has not yet converged.  Update M and test
      ! whether the maximum number of iterations have been tried.
      M = M + 1
      IF( M >= MAXIT ) GO TO 370

      ! Evaluate the residual and go back to do another iteration
      IWM(LNRE) = IWM(LNRE) + 1
      IRES = 0
      nb_resid = nb_resid + 1
      if( inside_init == 1 ) nb_resid_0 = nb_resid_0 + 1

      call mf_restore_fpe( )
      CALL RES( X, Y, YPRIME, DELTA, IRES )
      call mf_save_and_disable_fpe( )

      IF( IRES < 0 ) GO TO 380
      if( ires == 3 ) then
         end_condition_found = .true.
      end if
      GO TO 350

      ! The corrector failed to converge in MAXIT iterations. If the
      ! iteration matrix is not current, re-do the step with a new
      ! iteration matrix.
  370 CONTINUE
      IF( JCALC == 0 ) GO TO 380
      JCALC = -1
      GO TO 300

      ! The iteration has converged.  If nonnegativity of solution is
      ! required, set the solution nonnegative, if the perturbation
      ! to do it is small enough.  If the change is too large, then
      ! consider the corrector iteration to have failed.
      ! (modified by E.C. : generalized non-negativity constraints to be
      !  applied only to a subset of the NEQ equations)
  375 IF( NONNEG == 0 ) GO TO 390
      DO II = 1, NONNEG
         I = NN_IND(II)
         DELTA(I) = MIN(Y(I),0.0D0)
      END DO
      DELNRM = DDANRM(NEQ,DELTA,WT)
      IF( DELNRM > 0.33D0 ) GO TO 380
      DO II = 1, NONNEG
         I = NN_IND(II)
         E(I) = E(I) - DELTA(I)
      END DO
      GO TO 390

      ! Exits from block 3
      ! No convergence with current iteration matrix, or singular
      ! iteration matrix (or error flag set in the RES subroutine).
  380 CONVGD = .FALSE.
  390 JCALC = 1
      IF( .NOT. CONVGD ) GO TO 600

      !-----------------------------------------------------------------------
      ! BLOCK 4
      ! Estimate the errors at orders K, K-1, K-2 as if constant stepsize
      ! was used. Estimate the local error at order K and test whether
      ! the current step is successful.
      !-----------------------------------------------------------------------

      ! Estimate errors at orders K, K-1, K-2
      ENORM = DDANRM(NEQ,E,WT)
      ERK = SIGMA(K+1)*ENORM
      TERK = (K+1)*ERK
      EST = ERK
      KNEW = K
      IF( K == 1 ) GO TO 430
      DO I = 1, NEQ
         DELTA(I) = PHI(I,KP1) + E(I)
      ENDDO
      ERKM1 = SIGMA(K)*DDANRM(NEQ,DELTA,WT)
      TERKM1 = K*ERKM1
      IF( K > 2 ) GO TO 410
      IF( TERKM1 <= 0.5D0*TERK ) GO TO 420
      GO TO 430
  410 CONTINUE
      DO I = 1, NEQ
         DELTA(I) = PHI(I,K) + DELTA(I)
      ENDDO
      ERKM2 = SIGMA(K-1)*DDANRM(NEQ,DELTA,WT)
      TERKM2 = (K-1)*ERKM2
      IF( MAX(TERKM1,TERKM2) > TERK ) GO TO 430
      ! Lower the order
  420 CONTINUE
      KNEW = K - 1
      EST = ERKM1

      ! Calculate the local error for the current step to see if the
      ! step was successful
  430 CONTINUE
      ERR = CK * ENORM
      IF( ERR > 1.0D0 ) GO TO 600

      !-----------------------------------------------------------------------
      ! BLOCK 5
      ! The step is successful. Determine the best order and stepsize for
      ! the next step. Update the differences for the next step.
      !-----------------------------------------------------------------------
      ! already set at the beginning of the routine
!!      IDID = 1
      ! store in files (monitoring)
      if( unit_monitor_y /= -1 ) then
         write( unit_monitor_y, format_monitor_y )                      &
                X, Y(monitoryind)
         call flush( unit_monitor_y )
      end if
      if( unit_monitor_yp /= -1 ) then
         write( unit_monitor_yp, format_monitor_yp )                    &
                X, YPRIME(monitorypind)
         call flush( unit_monitor_yp )
      end if
      if( monitor_pause ) then
         print *
         call msPause( "[MUESLI] DAE solver: monitoring pause" )
      end if

      IWM(LNST) = IWM(LNST) + 1
      KDIFF = K - KOLD
      KOLD = K
      HOLD = H

      ! Estimate the error at order K+1 unless:
      !    - already decided to lower order, or
      !    - already using maximum order, or
      !    - stepsize not constant, or
      !    - order raised in previous step.

      IF( KNEW == KM1 .OR. K == IWM(LMXORD) ) IPHASE = 1
      IF( IPHASE == 0 ) GO TO 545
      IF( KNEW == KM1 ) GO TO 540
      IF( K == IWM(LMXORD) ) GO TO 550
      IF( KP1 >= NS .OR. KDIFF == 1 ) GO TO 550
      DO I = 1, NEQ
         DELTA(I) = E(I)-PHI(I,KP2)
      ENDDO
      ERKP1 = (1.0D0/(K+2))*DDANRM(NEQ,DELTA,WT)
      TERKP1 = (K+2)*ERKP1
      IF( K > 1 ) GO TO 520
      IF( TERKP1 >= 0.5D0*TERK ) GO TO 550
      GO TO 530
  520 IF( TERKM1 <= MIN(TERK,TERKP1) ) GO TO 540
      IF( TERKP1 >= TERK .OR. K == IWM(LMXORD) ) GO TO 550

      ! Raise order
  530 K = KP1
      EST = ERKP1
      GO TO 550

      ! Lower order
  540 K = KM1
      EST = ERKM1
      GO TO 550

      ! If IPHASE = 0, increase order by one and multiply stepsize by
      ! factor H_FACTOR
  545 K = KP1
      HNEW = H*H_FACTOR
      H = HNEW
      GO TO 575

      ! Determine the appropriate stepsize for the next step.
  550 CONTINUE
      HNEW = H
      TEMP2 = K + 1
      R = (2.0D0*EST+0.0001D0)**(-1.0D0/TEMP2)
      IF( R < 2.0D0 ) GO TO 555
      HNEW = H*H_FACTOR
      GO TO 560
  555 IF( R > 1.0D0 ) GO TO 560
      R = MAX(0.5D0,MIN(0.9D0,R))
      HNEW = H*R
  560 H = HNEW

      ! Update differences for next step
  575 CONTINUE
      IF( KOLD == IWM(LMXORD) ) GO TO 585
      DO I = 1, NEQ
         PHI(I,KP2) = E(I)
      ENDDO
  585 CONTINUE
      DO I = 1, NEQ
         PHI(I,KP1) = PHI(I,KP1) + E(I)
      ENDDO
      DO J1 = 2, KP1
         J = KP1 - J1 + 1
         DO I = 1, NEQ
            PHI(I,J) = PHI(I,J) + PHI(I,J+1)
         ENDDO
      ENDDO

      if( end_condition_found ) idid = 12

      RETURN

      !-----------------------------------------------------------------------
      ! BLOCK 6
      ! The step is unsuccessful or not adapted. Restore X, PSI, PHI.
      ! Determine appropriate stepsize for continuing the integration,
      ! or exit with an error flag if there have been many failures.
      !-----------------------------------------------------------------------
  600 IPHASE = 1

      ! Restore X, PHI, PSI.
      X = XOLD
      IF( KP1 < NSP1 ) GO TO 630
      DO J = NSP1, KP1
         TEMP1 = 1.0D0/BETA(J)
         DO I = 1, NEQ
            PHI(I,J) = TEMP1*PHI(I,J)
         ENDDO
      ENDDO
  630 CONTINUE
      DO I = 2, KP1
         PSI(I-1) = PSI(I) - H
      ENDDO

      ! Test whether failure is due to corrector iteration or error test.
      IF( CONVGD ) GO TO 660
      IWM(LCTF) = IWM(LCTF) + 1

      ! The Newton iteration failed to converge with the current iteration
      ! matrix.  Determine the cause of the failure and take appropriate
      ! action.
      ! -- added by E.C. because there is a bug for IRES = -1 or -2
      !    IRES = -1   Illegal condition
      !    IRES = -2   Emergency exit
      !    IRES =  3   End condition

      if( ires == -1 .or. ires == -2 ) go to 650

      IF( IER == 0 ) GO TO 650

      ! The iteration matrix is singular. Reduce the stepsize by a factor
      ! of 4.  If this happens three times in a row on the same step,
      ! return with an error flag.
      NSF = NSF + 1
      H = 0.25D0*H
      if( NSF < 3 .AND. ABS(H) >= HMIN ) then
         GO TO 690
      end if
      IDID = -8
      GO TO 675

      ! The Newton iteration failed to converge for a reason other than
      ! a singular iteration matrix.  If IRES = -2, then return.
      ! Otherwise, reduce the stepsize and try again, unless too many
      ! failures have occurred.
  650 CONTINUE
      if( IRES == -2 ) then
         IDID = -11
         GO TO 675
      end if

      NCF = NCF + 1
      H = 0.25D0*H
      if( NCF < 10 .AND. ABS(H) >= HMIN ) then
         GO TO 690
      end if
      IDID = -7
      IF( IRES == -1 ) IDID = -10
      IF( NEF >= 3) IDID = -9
      GO TO 675

      ! The Newton scheme converged, and the cause of the failure was
      ! the error estimate exceeding the tolerance.
  660 NEF = NEF + 1
      IWM(LETF) = IWM(LETF) + 1
      IF( NEF > 1 ) GO TO 665

      ! On first error test failure, keep current order or lower order
      ! by one. Compute new stepsize based on differences of the solution.
      K = KNEW
      TEMP2 = K + 1
      R = 0.90D0*(2.0D0*EST+0.0001D0)**(-1.0D0/TEMP2)
      R = MAX(0.25D0,MIN(0.9D0,R))
      H = H*R
      if( ABS(H) >= HMIN ) then
         GO TO 690
      end if
      IDID = -6
      GO TO 675

      ! On second error test failure, use the current order or decrease
      ! order by one. Reduce the stepsize by a factor of four.
  665 IF( NEF > 2 ) GO TO 670
      K = KNEW
      H = 0.25D0*H
      if( ABS(H) >= HMIN ) then
         GO TO 690
      end if
      IDID = -6
      GO TO 675

      ! On third and subsequent error test failures, set the order to
      ! one and reduce the stepsize by a factor of four.
  670 K = 1
      H = 0.25D0*H
      if( ABS(H) >= HMIN ) then
         GO TO 690
      end if
      IDID = -6
      GO TO 675

      ! For all crashes, restore Y to its last value, interpolate to
      ! find YPRIME at last X, and return
  675 CONTINUE
      CALL DDATRP( X, X, Y, YPRIME, NEQ, K, PHI, PSI )
      RETURN

      ! Go back and try this step again.
      ! If this is the first step, reset PSI(1) and rescale PHI(*,2).
  690 if( KOLD == 0 ) then
         PSI(1) = H
         do I = 1, NEQ
            PHI(I,2) = R*PHI(I,2)
         end do
      end if

      GO TO 200

   END
