!DECK DDASSL
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 24 jun 2025
!
   SUBROUTINE DDASSL( RES, NEQ, T, Y, YPRIME, TOUT, INFO, RTOL, ATOL,   &
                      IDID, RWORK, LRW, IWORK, LIW, JAC, SPJAC, NN_IND, &
                      CHECK_JAC, PRINT_CHECK_JAC )

      use mod_core, only: STDOUT, MF_NAN, sec_2_hms, msFlush, go_home_on_term

      implicit none

      integer :: NEQ, INFO(15), LRW, IWORK(*), LIW, NN_IND(*)
      integer, intent(out) :: IDID

      double precision :: T, Y(*), YPRIME(*), TOUT, RTOL(*), ATOL(*),   &
                          RWORK(*)

      external :: RES, JAC, SPJAC

      integer :: CHECK_JAC, PRINT_CHECK_JAC

!***BEGIN PROLOGUE  DDASSL
!***PURPOSE  This code solves a system of differential/algebraic
!            equations of the form G(T,Y,YPRIME) = 0.
!***LIBRARY   SLATEC (DASSL)
!***CATEGORY  I1A2
!***TYPE      DOUBLE PRECISION (SDASSL-S, DDASSL-D)
!***KEYWORDS  BACKWARD DIFFERENTIATION FORMULAS, DASSL,
!             DIFFERENTIAL/ALGEBRAIC, IMPLICIT DIFFERENTIAL SYSTEMS
!***AUTHOR  Petzold, Linda R., (LLNL)
!             Computing and Mathematics Research Division
!             Lawrence Livermore National Laboratory
!             L - 316, P.O. Box 808,
!             Livermore, CA.    94550
!***DESCRIPTION
!
! *Arguments:
!  (In the following, all real arrays should be type DOUBLE PRECISION.)
!
!  RES:EXT     This is a subroutine which you provide to define the
!              differential/algebraic system.
!
!  NEQ:IN      This is the number of equations to be solved.
!
!  T:INOUT     This is the current value of the independent variable.
!
!  Y(*):INOUT  This array contains the solution components at T.
!
!  YPRIME(*):INOUT  This array contains the derivatives of the solution
!              components at T.
!
!  TOUT:IN     This is a point at which a solution is desired.
!
!  INFO(N):IN  The basic task of the code is to solve the system from T
!              to TOUT and return an answer at TOUT.  INFO is an integer
!              array which is used to communicate exactly how you want
!              this task to be carried out.  (See below for details.)
!              N must be greater than or equal to 15.
!
!  RTOL,ATOL:INOUT  These quantities represent relative and absolute
!              error tolerances which you provide to indicate how
!              accurately you wish the solution to be computed.  You
!              may choose them to be both scalars or else both vectors.
!              Caution:  In Fortran 77, a scalar is not the same as an
!                        array of length 1.  Some compilers may object
!                        to using scalars for RTOL, ATOL.
!
!  IDID:OUT    This scalar quantity is an indicator reporting what the
!              code did.  You must monitor this integer variable to
!              decide what action to take next.
!
!  RWORK:WORK  A real work array of length LRW which provides the
!              code with needed storage space.
!
!  LRW:IN      The length of RWORK.  (See below for required length.)
!
!  IWORK:WORK  An integer work array of length LIW which provides the
!              code with needed storage space.
!
!  LIW:IN      The length of IWORK.  (See below for required length.)
!
!  JAC:EXT     This is the name of a subroutine which you may choose
!              to provide for defining a matrix of partial derivatives
!              described below.
!
!  SPJAC:EXT   This is the name of a subroutine which you may choose
!              to provide for defining a sparse matrix of partial
!              derivatives described below.
!
!  NN_IND(*):IN  Indexes of equations to which must applied a non-
!              negativity constraint. The size of this vector must be
!              equal at least to the number of constraints, which is
!              stored in INFO(10)
!
!  Quantities which may be altered by DDASSL are:
!     T, Y(*), YPRIME(*), INFO(1), RTOL, ATOL, IDID, RWORK(*) and IWORK(*)
!
! *Description
!
!  Subroutine DDASSL uses the backward differentiation formulas of
!  orders one through five to solve a system of fully implicit differential
!  equations for Y and YPRIME.
!  Values for Y and YPRIME at the initial time must be given as input.
!  These values must be consistent, (that is, if T,Y,YPRIME are the given
!  initial values, they must satisfy G(T,Y,YPRIME) = 0.); if you cannot
!  compute easily YPRIME at the initial time, the routine may try to
!  compute it itself (see below).
!  The subroutine solves the system from T to TOUT. It is easy to continue
!  the solution to get results at additional TOUT. This is the interval
!  mode of operation. Intermediate results can also be obtained easily by
!  using the intermediate-output capability.
!
!  The following detailed description is divided into subsections:
!    1. Input required for the first call to DDASSL.
!    2. Output after any return from DDASSL.
!    3. What to do to continue the integration.
!    4. Error messages.
!
!
! **********************************************************************
! ** INPUT -- What to do on the first call to DDASSL                  **
! **********************************************************************
!
!  The first call of the code is defined to be the start of each new
!  problem. Read through the descriptions of all the following items,
!  provide sufficient storage space for designated arrays, set
!  appropriate variables for the initialization of the problem, and
!  give information about how you want the problem to be solved.
!
!
!  RES -- Provide a subroutine of the form
!             SUBROUTINE RES( T, Y, YPRIME, DELTA, IRES )
!         to define the system of differential/algebraic equations which
!         is to be solved. For the given values of T, Y and YPRIME, the
!         subroutine should return the residual of the differential/algebraic
!         system
!             DELTA = G(T,Y,YPRIME)
!         (DELTA(*) is a vector of length NEQ which is output for RES.)
!
!         Subroutine RES must not alter T, Y or YPRIME.
!         You must declare the name RES in an external statement in your
!         program that calls DDASSL. You must dimension Y, YPRIME and
!         DELTA in RES.
!
!         IRES is an integer flag which is always equal to zero on
!         input. Subroutine RES should alter IRES only if it encounters
!         an illegal value of Y or a stop condition.
!           * set IRES = -1 if an input value is illegal, and DDASSL
!             will try to solve the problem without getting IRES = -1.
!             If IRES = -1 cannot be avoid, DDASSL will return control
!             to the calling program with IDID = -10.
!           * if IRES = -2, DDASSL will return control to the calling
!             program with IDID = -11.
!           * if IRES = -3, DDASSL will try to continue to integrate
!             by stopping exactly at this point. DDASSL will return
!             control to the calling program with IDID = -12.
!
!  NEQ -- Set it to the number of differential equations.
!         (NEQ >= 1)
!
!  T -- Set it to the initial point of the integration.
!         T must be defined as a variable.
!
!  Y(*) -- Set this vector to the initial values of the NEQ solution
!         components at the initial point. You must dimension Y of
!         length at least NEQ in your calling program.
!
!  YPRIME(*) -- Set this vector to the initial values of the NEQ
!         first derivatives of the solution components at the initial
!         point.  You must dimension YPRIME at least NEQ in your
!         calling program. If you do not know initial values of some
!         of the solution components, see the explanation of INFO(11).
!
!  TOUT -- Set it to the first point at which a solution
!         is desired. You can not take TOUT = T.
!         Integration either forward in T (TOUT > T) or backward in T
!         (TOUT < T) is permitted.
!
!         The code advances the solution from T to TOUT using
!         step sizes which are automatically selected so as to
!         achieve the desired accuracy. If you wish, the code will
!         return with the solution and its derivative at
!         intermediate steps (intermediate-output mode) so that
!         you can monitor them, but you still must provide TOUT.
!
!         The first step taken by the code is a critical one
!         because it must reflect how fast the solution changes near
!         the initial point. The code automatically selects an
!         initial step size which is practically always suitable for
!         the problem. To be sure that the code will not step
!         past TOUT in the first step, you could, if necessary,
!         restrict the length of the initial step size (See INFO(8)).
!
!         For some problems it may not be permissible to integrate
!         past a point TSTOP because a discontinuity occurs there
!         or the solution or its derivative is not defined beyond
!         TSTOP. Declaring a TSTOP point (See INFO(4) and RWORK(1))
!         allows the code not to integrate past TSTOP; in this case
!         any TOUT beyond TSTOP is invalid input. However, it may be
!         more practical to manage this kind of constraint by selecting
!         an appropriate value of IRES in your RES subroutine.
!
!  INFO(*) -- Use the INFO array to give the code more details about
!         how you want your problem solved.  This array should be
!         dimensioned of length 15, though DDASSL uses only the first
!         twelve entries.  You must respond to all of the following
!         items, which are arranged as questions.  The simplest use
!         of the code corresponds to answering all questions as yes,
!         i.e. setting all entries of INFO to 0.
!
!       INFO(1) - This parameter enables the code to initialize
!              itself. You must set it to indicate the start of every
!              new problem.
!
!          **** Is this the first call for this problem?
!                Yes - Set INFO(1) = 0
!                 No - Not applicable here.
!                      See below for continuation calls.  ****
!
!       INFO(2) - How much accuracy you want of your solution
!              is specified by the error tolerances RTOL and ATOL.
!              The simplest use is to take them both to be scalars.
!              To obtain more flexibility, they can both be vectors.
!              The code must be told your choice.
!
!          **** Are both error tolerances RTOL, ATOL scalars?
!                Yes - Set INFO(2) = 0
!                      and input scalars for both RTOL and ATOL
!                 No - Set INFO(2) = 1
!                      and input arrays for both RTOL and ATOL ****
!
!       INFO(3) - The code integrates from T in the direction
!              of TOUT by steps. If you wish, it will return the
!              computed solution and derivative at the next
!              intermediate step (the intermediate-output mode) or
!              TOUT, whichever comes first. This is a good way to
!              proceed if you want to see the behavior of the solution.
!              If you must have solutions at a great many specific
!              TOUT points, this code will compute them efficiently.
!
!          **** Do you want the solution only at
!                TOUT (and not at the next intermediate step)?
!                 Yes - Set INFO(3) = 0
!                  No - Set INFO(3) = 1 ****
!
!       INFO(4) - To handle solutions at a great many specific
!              values TOUT efficiently, this code may integrate past
!              TOUT and interpolate to obtain the result at TOUT.
!              Sometimes it is not possible to integrate beyond some
!              point TSTOP because the equation changes there or it is
!              not defined past TSTOP. Then you must tell the code
!              not to go past.
!
!           **** Can the integration be carried out without any
!                restrictions on the independent variable T?
!                 Yes - Set INFO(4)=0
!                  No - Set INFO(4)=1
!                       and define the stopping point TSTOP by
!                       setting RWORK(1)=TSTOP ****
!
!       INFO(5) - To solve differential/algebraic problems it is
!              necessary to use a matrix of partial derivatives of the
!              system of differential equations. If you do not
!              provide a subroutine to evaluate it analytically (see
!              description of the item JAC in the call list), it will
!              be approximated by numerical differencing in this code.
!              Although it is less trouble for you to have the code
!              compute partial derivatives by numerical differencing,
!              the solution will be more reliable if you provide the
!              derivatives via JAC. Sometimes numerical differencing
!              is cheaper than evaluating derivatives in JAC and
!              sometimes it is not - this depends on your problem.
!
!           **** Do you want the code to evaluate the partial
!                derivatives automatically by numerical differences?
!                   Yes - Set INFO(5)=0
!                    No - Set INFO(5)=1
!                  and provide subroutine JAC for evaluating the
!                  matrix of partial derivatives, either in a dense
!                  structure or in a banded one ****
!
!       INFO(6) - DDASSL will perform much better if the matrix of
!              partial derivatives, DG/DY + CJ*DG/DYPRIME,
!              (here CJ is a scalar determined by DDASSL)
!              is banded and the code is told this. In this
!              case, the storage needed will be greatly reduced,
!              numerical differencing will be performed much cheaper,
!              and a number of important algorithms will execute much
!              faster. The differential equation is said to have
!              half-bandwidths ML (lower) and MU (upper) if equation I
!              involves only unknowns Y(J) with
!                             I-ML <= J <= I+MU
!              for all I=1,2,...,NEQ. Thus, ML and MU are the widths
!              of the lower and upper parts of the band, respectively,
!              with the main diagonal being excluded. If you do not
!              indicate that the equation has a banded matrix of partial
!              derivatives, the code works with a full matrix of NEQ**2
!              elements (stored in the conventional way). Computations
!              with banded matrices cost less time and storage than with
!              full matrices if 2*ML+MU < NEQ. If you tell the
!              code that the matrix of partial derivatives has a banded
!              structure and you want to provide subroutine JAC to
!              compute the partial derivatives, then you must be careful
!              to store the elements of the matrix in the special form
!              indicated in the description of JAC.
!
!          **** Do you want to solve the problem using a full (dense)
!               matrix (and not a special banded structure)?
!                Yes - Set INFO(6)=0
!                 No - Set INFO(6)=1
!                       and provide the lower (ML) and upper (MU)
!                       bandwidths by setting
!                       IWORK(1)=ML
!                       IWORK(2)=MU ****
!
!
!        INFO(7) -- You can specify a maximum (absolute value of)
!              stepsize, so that the code will avoid passing over
!              very large regions.
!
!          ****  Do you want the code to decide on its own maximum
!                stepsize?
!                Yes - Set INFO(7)=0
!                 No - Set INFO(7)=1
!                      and define HMAX by setting
!                      RWORK(2)=HMAX ****
!
!        INFO(8) -- Differential/algebraic problems may occasionally
!              suffer from severe scaling difficulties on the first step.
!              If you know a great deal about the scaling of your problem,
!              you can help to alleviate this problem by specifying an
!              initial stepsize H0.
!
!          ****  Do you want the code to define its own initial stepsize?
!                Yes - Set INFO(8)=0
!                 No - Set INFO(8)=1
!                      and define H0 by setting
!                      RWORK(3)=H0 ****
!
!        INFO(9) -- If storage is a severe problem, you can save memory
!              by restricting the maximum order MAXORD.  The default value
!              is 5. For each order decrease below 5, the code requires NEQ
!              fewer locations, however it is likely to be slower. In any
!              case, you must have 1 <= MAXORD <= 5
!          ****  Do you want the maximum order to default to 5?
!                Yes - Set INFO(9)=0
!                 No - Set INFO(9)=1
!                      and define MAXORD by setting
!                      IWORK(3)=MAXORD ****
!
!        INFO(10) --If you know that the variables Y must always be
!               nonnegative, it may help to set this parameter. However,
!               it is probably best to try the code without using this
!               option first, and only to use this option if that doesn't
!               work very well.
!           ****  Do you want the code to solve the problem without
!                 invoking any special nonnegativity constraints?
!                  Yes - Set INFO(10)=0
!                   No - Set INFO(10) to the number of equations to which
!               a non negativity constraint will be applied. This number
!               must be comprise between 1 and NEQ ****
!
!        INFO(11) --DDASSL normally requires the initial T, Y, and
!               YPRIME to be consistent. That is, you must have
!               G(T,Y,YPRIME) = 0 at the initial time. If you do not
!               know the initial derivative precisely, you can let
!               DDASSL try to compute it.
!          ****   Are the initial T, Y, YPRIME consistent?
!                 Yes - Set INFO(11) = 0
!                  No - Set INFO(11) = 1, and set YPRIME to an initial
!               approximation to YPRIME.  (If you have no idea what YPRIME
!               should be, set it to zero. Note that the initial Y should
!               be such that there must exist a YPRIME so that
!               G(T,Y,YPRIME) = 0.) ****
!
!        INFO(12) -- The Jacobian matrix may have a sparse structure.
!               In such a case, a very efficient way to solve it is
!               to use a sparse linear solver (UMFPack, actually).
!
!            **** Do you want to solve the problem using a dense
!                 (either full or banded) Jacobian matrix (and not a
!                  sparse structure)?
!                  YES -- Set INFO(12)=0
!                   NO -- Set INFO(12)=1 (sparse) ****
!
!        INFO(13) -- The DDEDBF call corresponds to a new run
!
!            **** Is it the first call in a group of calls (interval mode)?
!                 (This corresponds to a new call from the user side)
!                  YES -- Set INFO(13)=0
!                   NO -- Set INFO(13)=1 ****
!
!  RTOL, ATOL -- You must assign relative (RTOL) and absolute (ATOL
!         error tolerances to tell the code how accurately you
!         want the solution to be computed.  They must be defined
!         as variables because the code may change them.  You
!         have two choices --
!               Both RTOL and ATOL are scalars. (INFO(2)=0)
!               Both RTOL and ATOL are vectors. (INFO(2)=1)
!         in either case all components must be non-negative.
!
!         The tolerances are used by the code in a local error
!         test at each step which requires roughly that
!               ABS(LOCAL ERROR) <= RTOL*ABS(Y)+ATOL
!         for each vector component.
!         (More specifically, a root-mean-square norm is used to
!         measure the size of vectors, and the error test uses the
!         magnitude of the solution at the beginning of the step.)
!
!         The true (global) error is the difference between the
!         true solution of the initial value problem and the
!         computed approximation.  Practically all present day
!         codes, including this one, control the local error at
!         each step and do not even attempt to control the global
!         error directly.
!         Usually, but not always, the true accuracy of the
!         computed Y is comparable to the error tolerances. This
!         code will usually, but not always, deliver a more
!         accurate solution if you reduce the tolerances and
!         integrate again.  By comparing two such solutions you
!         can get a fairly reliable idea of the true error in the
!         solution at the bigger tolerances.
!
!         Setting ATOL=0 results in a pure relative error test on
!         that component.  Setting RTOL=0 results in a pure
!         absolute error test on that component.  A mixed test
!         with non-zero RTOL and ATOL corresponds roughly to a
!         relative error test when the solution component is much
!         bigger than ATOL and to an absolute error test when the
!         solution component is smaller than the threshold ATOL.
!
!         The code will not attempt to compute a solution at an
!         accuracy unreasonable for the machine being used.  It will
!         advise you if you ask for too much accuracy and inform
!         you as to the maximum accuracy it believes possible.
!
!  RWORK(*) --  Dimension this real work array of length LRW in your
!         calling program.
!
!  LRW -- Set it to the declared length of the RWORK array.
!         You must have
!              LRW >= 40 + (MAXORD+4)*NEQ + NEQ**2
!         for the full (dense) JACOBIAN case (when INFO(6)=0), or
!              LRW >= 40 + (MAXORD+4)*NEQ + (2*ML+MU+1)*NEQ
!         for the banded user-defined JACOBIAN case
!         (when INFO(5)=1 and INFO(6)=1), or
!              LRW >= 40 + (MAXORD+4)*NEQ + (2*ML+MU+1)*NEQ
!                        + 2*(NEQ/(ML+MU+1)+1)
!         for the banded finite-difference-generated JACOBIAN case
!         (when INFO(5)=0 and INFO(6)=1), or
!              LRW >= 40 + (MAXORD+4)*NEQ
!         for the sparse Jacobian matrix (when INFO(12)=1).
!
!  IWORK(*) --  Dimension this integer work array of length LIW in
!         your calling program.
!
!  LIW -- Set it to the declared length of the IWORK array.
!               You must have LIW >= 20+NEQ
!
!  JAC -- If you have set INFO(5)=0, you can ignore this parameter
!         by treating it as a dummy argument.  Otherwise, you must
!         provide a subroutine of the form
!             SUBROUTINE JAC(T,Y,YPRIME,PD,CJ,NROWPD)
!         to define the matrix of partial derivatives
!             PD = DG/DY + CJ*DG/DYPRIME
!         CJ is a scalar which is input to JAC.
!         For the given values of T, Y, YPRIME, the subroutine must
!         evaluate the non-zero partial derivatives for each equation
!         and each solution component, and store these values in the
!         matrix PD.  The elements of PD are set to zero before each
!         call to JAC so only non-zero elements need to be defined.
!
!         Subroutine JAC must not alter T, Y(*), YPRIME(*) or CJ.
!         You must declare the name JAC in an EXTERNAL statement in
!         your program that calls DDASSL.  You must dimension Y,
!         YPRIME and PD in JAC.
!
!         The way you must store the elements into the PD matrix
!         depends on the structure of the matrix which you
!         indicated by INFO(6).
!         *** INFO(6)=0 -- Full (dense) matrix ***
!             Give PD a first dimension of NEQ.
!             When you evaluate the (non-zero) partial derivative
!             of equation I with respect to variable J, you must
!             store it in PD according to
!             PD(I,J) = "DG(I)/DY(J)+CJ*DG(I)/DYPRIME(J)"
!         *** INFO(6)=1 -- Banded JACOBIAN with ML lower and MU
!             upper diagonal bands (refer to INFO(6) description
!             of ML and MU) ***
!             Give PD a first dimension of 2*ML+MU+1.
!             when you evaluate the (non-zero) partial derivative
!             of equation I with respect to variable J, you must
!             store it in PD according to
!             IROW = I - J + ML + MU + 1
!             PD(IROW,J) = "DG(I)/DY(J)+CJ*DG(I)/DYPRIME(J)"
!
! SPJAC -- If INFO(12)=1 (sparse Jacobian), you must provide a
!         subroutine of the form
!             SUBROUTINE SPJAC(T,Y,YPRIME,CJ,NROWPD,JOB,PD,IPD,JPD,NNZ)
!         The Jacobian matrix is defined by the three vectors:
!             PD, IPD, JPD
!         which correspond to the CSC format (Compact Sparse Column)
!         WARNING: - on the first call, this routine must return
!                    only the value of NNZ (arrays PD, IPD, JPD
!                    will not referenced)
!                  - moreover, the sparse matrix must be 'row sorted'
!                    (in each column)
!
!  OPTIONALLY REPLACEABLE NORM ROUTINE:
!
!     DDASSL uses a weighted norm DDANRM to measure the size
!     of vectors such as the estimated error in each step.
!     A FUNCTION subprogram
!       DOUBLE PRECISION FUNCTION DDANRM(NEQ,V,WT)
!       DIMENSION V(NEQ),WT(NEQ)
!     is used to define this norm. Here, V is the vector
!     whose norm is to be computed, and WT is a vector of
!     weights.  A DDANRM routine has been included with DDASSL
!     which computes the weighted root-mean-square norm
!     given by
!       DDANRM=SQRT((1/NEQ)*SUM(V(I)/WT(I))**2)
!     this norm is suitable for most problems. In some
!     special cases, it may be more convenient and/or
!     efficient to define your own norm by writing a function
!     subprogram to be called instead of DDANRM. This should,
!     however, be attempted only after careful thought and
!     consideration.
!
! **********************************************************************
! ** OUTPUT -- After any return from DDASSL                           **
! **********************************************************************
!
!  The principal aim of the code is to return a computed solution at
!  TOUT, although it is also possible to obtain intermediate results
!  along the way. To find out whether the code achieved its goal
!  or if the integration process was interrupted before the task was
!  completed, you must check the IDID parameter.
!
!
!  T -- The solution was successfully advanced to the output value of T.
!
!  Y(*) -- Contains the computed solution approximation at T.
!
!  YPRIME(*) -- Contains the computed derivative approximation at T.
!
!  IDID -- Reports what the code did.
!
!                  *** Task completed ***
!             Reported by positive values of IDID
!
!     IDID = 1 -- A step was successfully taken in the intermediate-output
!            mode. The code has not yet reached TOUT.
!
!     IDID = 2 -- The integration to TSTOP was successfully completed
!            (T=TSTOP) by stepping exactly to TSTOP.
!
!     IDID = 3 -- The integration to TOUT was successfully completed
!            (T=TOUT) by stepping past TOUT.  Y(*) and YPRIME(*) are
!            obtained by interpolation.
!
!     IDID = 12 -- IRES equal to 3 was encountered and the program tried
!            to finish at the current time step.
!
!                  *** Task interrupted ***
!             Reported by negative values of IDID
!
!     IDID = -1 -- A large amount of work has been expended.
!             (About 500 steps)
!
!     IDID = -2 -- The error tolerances are too stringent.
!
!     IDID = -3 -- The local error test cannot be satisfied because you
!            specified a zero component in ATOL and the corresponding
!            computed solution component is zero. Thus, a pure relative
!            error test is impossible for this component.
!
!     IDID = -6 -- DDASSL had repeated error test failures on the last
!            attempted step.
!
!     IDID = -7 -- The corrector could not converge.
!
!     IDID = -8 -- The matrix of partial derivatives is singular.
!
!     IDID = -9 -- The corrector could not converge. There were repeated
!            error test failures in this step.
!
!     IDID = -10 -- The corrector could not converge because IRES was equal
!            to -1.
!
!     IDID = -11 -- IRES equal to -2 was encountered and control is being
!            returned to the calling program.
!
!     IDID = -102 -- DDASSL failed to compute the initial YPRIME.
!
!                  *** Task terminated ***
!             Reported by the value of IDID = -33
!
!     IDID = -33 -- The code has encountered trouble from which it cannot
!            recover. A message is printed explaining the trouble and
!            control is returned to the calling program. For example, this
!            occurs when invalid input is detected.
!
!  RTOL, ATOL -- These quantities remain unchanged except when IDID = -2.
!            In this case, the error tolerances have been increased by
!            the code to values which are estimated to be appropriate for
!            continuing the integration. However, the reported solution
!            at T was obtained using the input values of RTOL and ATOL.
!
!  RWORK, IWORK -- Contain information which is usually of no interest to
!            the user but necessary for subsequent calls. However, you may
!            find use for
!
!     RWORK(3) -- Which contains the step size H to be attempted on the
!            next step.
!
!     RWORK(4) -- Which contains the current value of the independent
!            variable, i.e., the farthest point integration has reached.
!            This will be different from T only when interpolation has been
!            performed (IDID=3).
!
!     RWORK(7) -- Which contains the stepsize used on the last successful
!            step.
!
!     IWORK(7) -- Which contains the order of the method to be attempted
!            on the next step.
!
!     IWORK(8) -- Which contains the order of the method used on the last
!            step.
!
!     IWORK(11) -- Which contains the number of steps taken so far.
!
!     IWORK(12) -- Which contains the number of calls to RES so far.
!
!     IWORK(13) -- Which contains the number of evaluations of the matrix
!             of partial derivatives needed so far.
!
!     IWORK(14) -- Which contains the total number of error test failures
!             so far.
!
!     IWORK(15) -- Which contains the total number of convergence test
!             failures so far. (includes singular iteration matrix failures.)
!
! **********************************************************************
! ** INPUT -- What to do to continue the integration                  **
! **             (calls after the first)                              **
! **********************************************************************
!
!  This code is organized so that subsequent calls to continue the
!  integration involve little (if any) additional effort on your
!  part. You must monitor the IDID parameter in order to determine
!  what to do next.
!
!  Recalling that the principal task of the code is to integrate
!  from T to TOUT (the interval mode), usually all you will need
!  to do is specify a new TOUT upon reaching the current TOUT.
!
!  Do not alter any quantity not specifically permitted below, in
!  particular do not alter NEQ, T, Y(*), YPRIME(*), RWORK(*), IWORK(*)
!  or the differential equation in subroutine RES. Any such
!  alteration constitutes a new problem and must be treated as such,
!  i.e., you must start afresh.
!
!  You cannot change from vector to scalar error control or vice
!  versa (INFO(2)), but you can change the value of the entries of
!  RTOL, ATOL. Increasing a tolerance makes the equation easier
!  to integrate. Decreasing a tolerance will make the equation
!  harder to integrate and should generally be avoided.
!
!  You can switch from the intermediate-output mode to the
!  interval mode (INFO(3)) or vice versa at any time.
!
!  If it has been necessary to prevent the integration from going
!  past a point TSTOP (INFO(4), RWORK(1)), keep in mind that the
!  code will not integrate to any TOUT beyond the currently
!  specified TSTOP. Once TSTOP has been reached you must change
!  the value of TSTOP or set INFO(4)=0. You may change INFO(4)
!  or TSTOP at any time but you must supply the value of TSTOP in
!  RWORK(1) whenever you set INFO(4)=1.
!
!  Do not change INFO(5), INFO(6), IWORK(1), or IWORK(2)
!  unless you are going to restart the code.
!
!                 *** Following a completed task ***
!  If
!     IDID = 1, call the code again to continue the integration
!                  another step in the direction of TOUT.
!
!     IDID = 2 or 3, a new TOUT can be defined and the code can be
!                  called again ; for the case IDID = 2, the
!                  possibility for continue the integration depends
!                  on the origine of TSTOP (if TSTOP has been
!                  introduced due to the nature of the problem, it is
!                  evident that the integration cannot be continued
!                  passed this critical time).
!                  TOUT must be different from T. You cannot change
!                  the direction of integration without restarting.
!
!                 *** Following an interrupted task ***
!               To tell the code that you realize the task was
!               interrupted and that you want to continue, you
!               must take appropriate action and set INFO(1) = 1
!  If
!    IDID = -1, the code has attempted 500 steps. If you want to
!               continue, set INFO(1) = 1 and call the code again.
!               An additional 500 steps will be allowed.
!
!    IDID = -2, The error tolerances RTOL, ATOL have been increased to
!               values the code estimates appropriate for continuing.
!               You may want to change them yourself. If you are sure
!               you want to continue with relaxed error tolerances,
!               set INFO(1)=1 and call the code again.
!
!    IDID = -3, A solution component is zero and you set the corresponding
!               component of ATOL to zero. If you are sure you want to
!               continue, you must first alter the error criterion to
!               use positive values for those components of ATOL
!               corresponding to zero solution components, then set
!               INFO(1)=1 and call the code again.
!
!    IDID = -4,-5  --- Cannot occur with this code.
!
!    IDID = -6, Repeated error test failures occurred on the last
!               attempted step in DDASSL. A singularity in the solution
!               may be present. If you are absolutely certain you want
!               to continue, you should restart the integration. (Provide
!               initial values of Y and YPRIME which are consistent)
!
!    IDID = -7, Repeated convergence test failures occurred on the last
!               attempted step in DDASSL. An inaccurate or ill-conditioned
!               JACOBIAN may be the problem. If you are absolutely certain
!               you want to continue, you should restart the integration.
!
!    IDID = -8, The matrix of partial derivatives is singular. Some of
!               your equations may be redundant. DDASSL cannot solve the
!               problem as stated. It is possible that the redundant
!               equations could be removed, and then DDASSL could solve
!               the problem. It is also possible that a solution to your
!               problem either does not exist or is not unique.
!
!    IDID = -9, DDASSL had multiple convergence test failures, preceded
!               by multiple error test failures, on the last attempte
!               step. It is possible that your problem is ill-posed,
!               and cannot be solved using this code. Or, there may b
!               a discontinuity or a singularity in the solution. If
!               you are absolutely certain you want to continue, you
!               should restart the integration.
!
!    IDID = -10, DDASSL had multiple convergence test failures because
!               IRES was equal to -1. If you are absolutely certain
!               you want to continue, you should restart the integration.
!
!    IDID = -11, IRES=-2 was encountered, and control is being returned
!               to the calling program.
!
!    IDID = -12, IRES=-3 was encountered, and program tried to finish at
!               the current time step.
!
!    IDID = -13, DDASSL error: some indices for the non-negativity
!               constraints stored in INFO(10) are not in the range [1,NEQ]
!
!    IDID = -14, DDASSL error: the user-supplied RESID routine returned
!               NaN values.
!
!    IDID = -15, DDASSL error: the user-supplied JAC routine (either
!               dense or sparse) returned NaN values.
!
!    IDID = -102, DDASSL failed to compute the initial YPRIME. This could
!               happen because the initial approximation to YPRIME was
!               not very good, or if a YPRIME consistent with the initial
!               Y does not exist. The problem could also be caused by an
!               inaccurate or singular iteration matrix.
!
!                 *** Following a terminated task ***
!
!  If IDID= -33, you cannot continue the solution of this problem. An
!               attempt to do so will result in your run being terminated.
!
!
!  -------- ERROR MESSAGES ---------------------------------------------
!
!      The SLATEC error print routine XERMSG is called in the event of
!   unsuccessful completion of a task.  Most of these are treated as
!   "recoverable errors", which means that (unless the user has directed
!   otherwise) control will be returned to the calling program for
!   possible action after the message has been printed.
!
!   In the event of a negative value of IDID other than -33, an appro-
!   priate message is printed and the "error number" printed by XERMSG
!   is the value of IDID.  There are quite a number of illegal input
!   errors that can lead to a returned value IDID=-33.  The conditions
!   and their printed "error numbers" are as follows:
!
!   Error number       Condition
!
!        1       Some element of INFO vector is not zero or one.
!        2       NEQ <= 0
!        3       MAXORD not in range.
!        4       LRW is less than the required length for RWORK.
!        5       LIW is less than the required length for IWORK.
!        6       Some element of RTOL is < 0
!        7       Some element of ATOL is < 0
!        8       All elements of RTOL and ATOL are zero.
!        9       INFO(4)=1 and TSTOP is behind TOUT.
!       10       HMAX < 0.0
!       11       TOUT is behind T.
!       12       INFO(8)=1 and H0=0.0
!       13       Some element of WT is <= 0.0
!       14       TOUT is too close to T to start integration.
!       15       INFO(4)=1 and TSTOP is behind T.
!       16       --( Not used in this version )--
!       17       ML illegal.  Either < 0 or > NEQ
!       18       MU illegal.  Either < 0 or > NEQ
!       19       TOUT = T.
!
!   If DDASSL is called again without any action taken to remove the
!   cause of an unsuccessful return, XERMSG will be called with a fatal
!   error flag, which will cause unconditional termination of the
!   program.  There are two such fatal errors:
!
!   Error number -998:  The last step was terminated with a negative
!       value of IDID other than -33, and no appropriate action was
!       taken.
!
!   Error number -999:  The previous call was terminated because of
!       illegal input (IDID=-33) and there is illegal input in the
!       present call, as well.  (Suspect infinite loop.)
!
!  ---------------------------------------------------------------------
!
!***REFERENCES  A DESCRIPTION OF DASSL: A DIFFERENTIAL/ALGEBRAIC
!                 SYSTEM SOLVER, L. R. PETZOLD, SAND82-8637,
!                 SANDIA NATIONAL LABORATORIES, SEPTEMBER 1982.
!***ROUTINES CALLED  D1MACH, DDAINI, DDANRM, DDASTP, DDATRP, DDAWTS,
!                    XERMSG
!***REVISION HISTORY  (YYMMDD)
!   830315  DATE WRITTEN
!   000711  Last modification. For details see original file.
!***END PROLOGUE  DDASSL
!
!**End

      ! Declare local variables.
      INTEGER :: I, ITEMP, LENIW, LENPD, LENRW, LE, LPD, LPHI, LWM,     &
         LWT, MBAND, MSAVE, MXORD, NTEMP, NZFLG, NDIAG, J, K
      DOUBLE PRECISION :: ATOLI, H, HMAX, HMIN, H0, R, RH, RTOLI, TDIST,&
         TN, TNEXT, TSTOP, UROUND, YPNORM, CJ
      LOGICAL :: DONE

      ! Auxiliary variables for conversion of values to be included in
      ! error messages.
      CHARACTER(len=8)  :: XERN1, XERN2
      CHARACTER(len=16) :: XERN3, XERN4

      ! Set pointers into IWORK
      integer, parameter :: LML=1, LMU=2, LMXORD=3, LMTYPE=4, LJCALC=5, &
         LPHASE=6, LK=7, LKOLD=8, LNS=9, LNSTL=10, LNST=11, LNRE=12,    &
         LNJE=13, LETF=14, LCTF=15, LNPD=16, LIPVT=21, LIWM=1

      ! Set relative offset into RWORK
      integer, parameter :: NPD=1

      ! Set pointers into RWORK
      integer, parameter :: LTSTOP=1, LHMAX=2, LH=3, LTN=4, LCJ=5,      &
         LCJOLD=6, LHOLD=7, LS=8, LROUND=9, LALPHA=11, LBETA=17,        &
         LGAMMA=23, LPSI=29, LSIGMA=35, LDELTA=41

      integer :: job

      double precision :: dt_min, dt_max
      integer :: nb_step, nb_resid
      common /slatec_daesolve_1/ dt_min, dt_max, nb_step, nb_resid

      integer :: nb_jac, nb_solv
      common /slatec_daesolve_2/ nb_jac, nb_solv

      double precision :: cpu_time_0
      integer :: inside_init, nb_resid_0, nb_jac_0, nb_solv_0
      common /slatec_daesolve_3/ cpu_time_0, inside_init, nb_resid_0,   &
                                 nb_jac_0, nb_solv_0

      integer :: times_current_length, orders_current_length
      double precision, allocatable :: vector_tmp(:)

      logical :: print_progress, disp_times
      character(len=80) :: string
      double precision :: cpu_time_init
      double precision :: t_0, t_end, pc_fact, percent
      double precision :: total_time, left_time, rem_time
      integer :: clock_rate, clock_init, clock, kk
      integer :: hrs_1, min_1, sec_1, hrs_2, min_2, sec_2
      common /slatec_daesolve_progress/ print_progress, disp_times,     &
             t_0, t_end, pc_fact, clock_rate, clock_init

      integer*8 :: total_size, n1
      character(len=20) :: n1sq_str
      character(len=11) :: nnz_str

      !------ end of declarations -- execution starts hereafter  ------

      IDID = 0

      if( INFO(13) == 0 ) then
         ! new run from the user
         nb_step = 0
         dt_min =  8.8888D+88
         dt_max = -8.8888D+88
         nb_resid = 0
         nb_jac = 0
         nb_solv = 0
         nb_resid_0 = 0
         nb_jac_0 = 0
         nb_solv_0 = 0
         inside_init = 0
         cpu_time_0 = 0.0d0
         if( save_times ) then
            allocate( times_solve(500) )
            n_times = 1
            times_solve(1) = T
         end if
         if( save_orders ) then
            allocate( orders_solve(500) )
            n_orders = 1
            orders_solve(1) = MF_NAN
         end if
         ! -- Reset INFO(13) for subsequent user-calls
         INFO(13) = 1
      end if

      IF( INFO(1) /= 0 ) GO TO 100

      !-----------------------------------------------------------------
      ! This block is executed for the initial call only.
      ! It contains checking of inputs and initializations.
      !----------------------------------------------------------------

      ! First check INFO array to make sure all elements of info
      ! are either zero or one.
      ! (added by E.C.: exception for INFO(10))
      DO I = 2, 9
         IF( INFO(I) /= 0 .AND. INFO(I) /= 1 ) GO TO 701
      END DO
      DO I = 11, 12
         IF( INFO(I) /= 0 .AND. INFO(I) /= 1 ) GO TO 701
      END DO

      IF( NEQ <= 0 ) GO TO 702

      ! Check and compute maximum order
      MXORD = 5
      IF( INFO(9) == 0 ) GO TO 20
         MXORD = IWORK(LMXORD)
         IF( MXORD < 1 .OR. MXORD > 5 ) GO TO 703
   20    IWORK(LMXORD) = MXORD

      ! Compute MTYPE, LENPD, LENRW. Check ML and MU.
      IF( INFO(12) == 0 ) GO TO 25
         ! sparse matrix
         LENPD = 0
         LENRW = 40 + (IWORK(LMXORD)+4)*NEQ + LENPD
         IWORK(LMTYPE) = 3
         if( .not. allocated(jpd) ) then

            ! Deallocation must be made in the calling program, after any
            ! call to DDASSL. Cf. routine 'ddebd2_free' in 'mod_ddebd2.F90'

            allocate( jpd(neq+1), jpd_c(neq+1) )
            ! the first call only initializes nnz (job=0)
            job = 0
            call spjac(t,y,yprime,cj,neq,job,cj,i,jpd(1),nnz)
            n1 = neq
            total_size = n1*n1
            if( nnz < 0 .or. total_size < nnz ) then
               write(n1sq_str,'(I0)') total_size
               write(nnz_str,'(I0)') nnz
               CALL XERMSG ('SLATEC', 'DDASSL',                         &
                 'INVALID IMPLEMENTATION OF THE USER-DEFINED SPARSE' // &
                 ' JACOBIAN:$$FIRST CALL MUST RETURN A VALID VALUE ' // &
                 'FOR NNZ! Got: NNZ = ' // trim(adjustl(nnz_str))    // &
                 ' WHICH IS OUT OF THE RANGE [0,'                    // &
                 trim(adjustl(n1sq_str)) // ']',                        &
                 18, 1)
               IDID = -34
               GO TO 690
            end if
            allocate( pd(nnz), ipd(nnz), ipd_c(nnz) )
         end if
         GO TO 60
   25 continue

      ! Added by E.C.
      if( INFO(10) /= 0 ) then
         ! non-negativity contraints to be handled
         if( info(10) < 1 .or. NEQ < info(10) ) then
            idid = -13
            goto 600
         end if
      end if

      IF( INFO(6) /= 0 ) GO TO 40
         ! dense matrix
         LENPD = NEQ**2
         LENRW = 40 + (IWORK(LMXORD)+4)*NEQ + LENPD
         IF( INFO(5) /= 0 ) GO TO 30
            IWORK(LMTYPE) = 2
            GO TO 60
   30       IWORK(LMTYPE) = 1
            GO TO 60
   40 continue
      ! band matrix
      IF( IWORK(LML) < 0 .OR. IWORK(LML) >= NEQ ) GO TO 717
      IF( IWORK(LMU) < 0 .OR. IWORK(LMU) >= NEQ ) GO TO 718
      LENPD = (2*IWORK(LML)+IWORK(LMU)+1)*NEQ
      IF( INFO(5) /= 0 ) GO TO 50
         IWORK(LMTYPE) = 5
         MBAND = IWORK(LML)+IWORK(LMU)+1
         MSAVE = (NEQ/MBAND)+1
         LENRW = 40 + (IWORK(LMXORD)+4)*NEQ + LENPD + 2*MSAVE
         GO TO 60
   50    IWORK(LMTYPE) = 4
         LENRW = 40 + (IWORK(LMXORD)+4)*NEQ + LENPD

      ! Check lengths of RWORK and IWORK
   60 LENIW = 20+NEQ
      IWORK(LNPD) = LENPD
      IF( LRW < LENRW ) GO TO 704
      IF( LIW < LENIW ) GO TO 705

      ! Check to see that TOUT is different from T
      IF( TOUT == T ) GO TO 719

      ! Check HMAX
      IF( INFO(7) == 0 ) GO TO 70
         HMAX = RWORK(LHMAX)
         IF( HMAX <= 0.0D0 ) GO TO 710
   70 CONTINUE

      ! Initialize counters
      IWORK(LNST) = 0
      IWORK(LNRE) = 0
      IWORK(LNJE) = 0

      IWORK(LNSTL) = 0

      GO TO 200

      !-----------------------------------------------------------------
      ! This block is for continuation calls only.
      ! Here we check INFO(1), and if the last step was interrupted
      ! we check whether appropriate action was taken.
      !-----------------------------------------------------------------

  100 CONTINUE
      ! Don't forget that INFO(1) may be equal to -1 (see after label 690,
      ! or at the end of this routine).
      IF( INFO(1) == 1 ) GO TO 110

      IF( INFO(1) /= -1 ) GO TO 701 ! Unexpected value (must be 0, 1, -1)

      ! Here INFO(1)=-1. The last step was interrupted by an error condition
      ! from DDASTP, and appropriate action was not taken. This is a fatal
      ! error.
      WRITE(XERN1, '(I0)') IDID
      CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'THE LAST STEP TERMINATED WITH A NEGATIVE VALUE OF IDID = ' // &
         XERN1 // ' AND NO APPROPRIATE ACTION WAS TAKEN.  ' //          &
         'RUN TERMINATED', -998, 2)
      RETURN

  110 CONTINUE
      IWORK(LNSTL) = IWORK(LNST)

      !-----------------------------------------------------------------
      ! This block is executed on all calls.
      ! The error tolerance parameters are checked, and the work array
      ! pointers are set.
      !-----------------------------------------------------------------

  200 CONTINUE
      ! Check RTOL, ATOL
      NZFLG = 0
      RTOLI = RTOL(1)
      ATOLI = ATOL(1)
      DO I = 1, NEQ
         IF( INFO(2) == 1 ) RTOLI = RTOL(I)
         IF( INFO(2) == 1 ) ATOLI = ATOL(I)
         IF( RTOLI > 0.0D0 .OR. ATOLI > 0.0D0 ) NZFLG=1
         IF( RTOLI < 0.0D0 ) GO TO 706
         IF( ATOLI < 0.0D0 ) GO TO 707
      END DO
      IF( NZFLG == 0 ) GO TO 708

      ! Set up RWORK storage. IWORK storage is fixed in data statement.
      LE = LDELTA + NEQ
      LWT = LE + NEQ
      LPHI = LWT + NEQ
      LPD = LPHI + (IWORK(LMXORD)+1)*NEQ
      LWM = LPD
      NTEMP = NPD + IWORK(LNPD)

      IF( INFO(1) == 1 ) GO TO 400

      !-----------------------------------------------------------------
      ! This block is executed on the initial call only.
      ! Set the initial step size, the error weight vector and PHI.
      ! Compute initial YPRIME, if necessary.
      !-----------------------------------------------------------------

      TN = T

      ! Set error weight vector WT
      CALL DDAWTS(NEQ,INFO(2),RTOL,ATOL,Y,RWORK(LWT))
      DO I = 1, NEQ
         IF( RWORK(LWT+I-1) <= 0.0D0 ) GO TO 713
      END DO

      ! Compute unit roundoff and HMIN
      UROUND = D1MACH(4)
      RWORK(LROUND) = UROUND
      HMIN = 4.0D0*UROUND*MAX(ABS(T),ABS(TOUT))

      ! Check initial interval to see that it is long enough
      TDIST = ABS(TOUT - T)
      IF( TDIST < HMIN ) GO TO 714

      ! Check H0, if this was input
      IF( INFO(8) == 0 ) GO TO 310
         H0 = RWORK(LH)
         IF( (TOUT - T)*H0 < 0.0D0 ) GO TO 711
         IF( H0 == 0.0D0 ) GO TO 712
         GO TO 320
  310 CONTINUE

      ! Compute initial stepsize, to be used by either DDASTP or DDAINI,
      ! depending on INFO(11)
      H0 = 0.001D0*TDIST
      YPNORM = DDANRM(NEQ,YPRIME,RWORK(LWT))
      IF( YPNORM > 0.5D0/H0 ) H0 = 0.5D0/YPNORM
      H0 = SIGN(H0,TOUT-T)
      ! Adjust H0 if necessary to meet HMAX bound
  320 CONTINUE
      IF( INFO(7) == 0 ) GO TO 330
         RH = ABS(H0)/RWORK(LHMAX)
         IF( RH > 1.0D0 ) H0 = H0/RH
         ! Compute TSTOP, if applicable
  330 CONTINUE
      IF( INFO(4) == 0 ) GO TO 340
         TSTOP = RWORK(LTSTOP)
         IF( (TSTOP - T)*H0 < 0.0D0 ) GO TO 715
         IF( (T + H0 - TSTOP)*H0 > 0.0D0 ) H0 = TSTOP - T
         IF( (TSTOP - TOUT)*H0 < 0.0D0 ) GO TO 709

      ! Compute initial derivative, updating TN and Y, if applicable
  340 CONTINUE
      IF( INFO(11) == 0 ) GO TO 350

      ! setting a flag to tell other routines that we are inside
      ! the initialization part (for statistics only).
      inside_init = 1
      call cpu_time(cpu_time_init)
      CALL DDAINI( TN, Y, YPRIME, NEQ, RES, JAC, SPJAC, H0, RWORK(LWT), &
                   IDID, RWORK(LPHI), RWORK(LDELTA), RWORK(LE),         &
                   RWORK(LWM), IWORK(LIWM), HMIN, RWORK(LROUND),        &
                   INFO(10), NN_IND, NTEMP, CHECK_JAC, PRINT_CHECK_JAC )
      inside_init = 0

!### modif É. C.  2025-06-24
H = H0 ! otherwise valgrind detects that H is uninitialized! (if IDID<0)
!###

      IF( IDID < 0 ) GO TO 390
      call cpu_time(cpu_time_0)
      cpu_time_0 = cpu_time_0 - cpu_time_init
      !--  added by E.C. (initial condition computed by DDAINI are saved
      !    in order to be restored in routines of DaeSolve)
      allocate( yp_init_ddassl(1:NEQ) )
      yp_init_ddassl(1:NEQ) = yprime(1:NEQ)
      !    deallocation will take place in DaeSolve
      !----

      nb_step = nb_step + 1
      ! Load H with H0.  Store H in RWORK(LH)
  350 H = H0
      RWORK(LH) = H

      ! Load Y and H*YPRIME into PHI(*,1) and PHI(*,2)
      ITEMP = LPHI + NEQ
      DO I = 1, NEQ
         RWORK(LPHI + I - 1) = Y(I)
         RWORK(ITEMP + I - 1) = H*YPRIME(I)
      END DO

  390 GO TO 500

      !-----------------------------------------------------------------
      ! This block is for continuation calls only.
      ! Its purpose is to check stop conditions before taking a step.
      ! Adjust H if necessary to meet HMAX bound.
      !-----------------------------------------------------------------

  400 CONTINUE
      UROUND = RWORK(LROUND)
      DONE = .FALSE.
      TN = RWORK(LTN)
      H = RWORK(LH)
      if( INFO(7) /= 0 ) then
         RH = ABS(H)/RWORK(LHMAX)
         if( RH > 1.0D0 ) H = H/RH
      end if
      IF( T == TOUT ) GO TO 719
      IF( (T-TOUT)*H > 0.0D0 ) GO TO 711

      if( INFO(4) == 0 ) then
         if( INFO(3) == 0 ) then
            IF( (TN-TOUT)*H < 0.0D0 ) GO TO 490
            CALL DDATRP( TN, TOUT, Y, YPRIME, NEQ,                      &
                         IWORK(LKOLD), RWORK(LPHI), RWORK(LPSI) )
            T = TOUT
            IDID = 3
            DONE = .TRUE.
            GO TO 490
         end if
         IF( (TN-T)*H <= 0.0D0 ) GO TO 490
         if( (TN-TOUT)*H < 0.0D0 ) then
            CALL DDATRP( TN, TN, Y, YPRIME, NEQ,                        &
                         IWORK(LKOLD), RWORK(LPHI), RWORK(LPSI) )
            T = TN
            IDID = 1
            DONE = .TRUE.
            GO TO 490
         end if
         CALL DDATRP( TN, TOUT, Y, YPRIME, NEQ,                         &
                      IWORK(LKOLD), RWORK(LPHI), RWORK(LPSI) )
         T = TOUT
         IDID = 3
         DONE = .TRUE.
         GO TO 490
      end if
      if( INFO(3) == 0 ) then
         TSTOP = RWORK(LTSTOP)
         IF( (TN-TSTOP)*H > 0.0D0 ) GO TO 715
         IF( (TSTOP-TOUT)*H < 0.0D0 ) GO TO 709
         IF( (TN-TOUT)*H < 0.0D0 ) GO TO 450
         CALL DDATRP( TN, TOUT, Y, YPRIME, NEQ,                         &
                      IWORK(LKOLD), RWORK(LPHI), RWORK(LPSI) )
         T = TOUT
         IDID = 3
         DONE = .TRUE.
         GO TO 490
      end if
      TSTOP = RWORK(LTSTOP)
      IF( (TN-TSTOP)*H > 0.0D0 ) GO TO 715
      IF( (TSTOP-TOUT)*H < 0.0D0 ) GO TO 709
      if( (TN-T)*H > 0.0D0 ) then
         if( (TN - TOUT)*H < 0.0D0 ) then
            CALL DDATRP( TN, TN, Y, YPRIME, NEQ,                        &
                         IWORK(LKOLD), RWORK(LPHI), RWORK(LPSI) )
            T = TN
            IDID = 1
            DONE = .TRUE.
            GO TO 490
         end if
         CALL DDATRP( TN, TOUT, Y, YPRIME, NEQ,                         &
                      IWORK(LKOLD), RWORK(LPHI), RWORK(LPSI) )
         T = TOUT
         IDID = 3
         DONE = .TRUE.
         GO TO 490
      end if
  450 CONTINUE
      ! Check whether we are within roundoff of TSTOP
      if( ABS(TN-TSTOP) <= 100.0d0*UROUND*(ABS(TN)+ABS(H)) ) then
         CALL DDATRP( TN, TSTOP, Y, YPRIME, NEQ,                        &
                      IWORK(LKOLD), RWORK(LPHI), RWORK(LPSI) )
         IDID = 2
         T = TSTOP
         DONE = .TRUE.
         GO TO 490
      end if
      TNEXT = TN+H
      if( (TNEXT-TSTOP)*H > 0.0D0 ) then
         H = TSTOP-TN
         RWORK(LH) = H
      end if

  490 CONTINUE
      IF( DONE ) GO TO 580

      !-----------------------------------------------------------------
      ! This block contains the call to the one-step integrator DDASTP.
      ! This is a looping point for the integration steps.
      ! Check for too many steps. Update WT.
      ! Check for too much accuracy requested. Compute minimum stepsize.
      !-----------------------------------------------------------------

  500 CONTINUE
      ! Check for failure to compute initial YPRIME
      IF( IDID == -12 ) GO TO 527

      ! Check for too many steps

      ! Update WT
      CALL DDAWTS( NEQ, INFO(2), RTOL, ATOL, RWORK(LPHI), RWORK(LWT) )
      DO I = 1, NEQ
         IF( RWORK(I+LWT-1) <= 0.0D0 ) THEN
            IDID = -3
            GO TO 527
         END IF
      END DO

      ! Test for too much accuracy requested.
      R = DDANRM(NEQ,RWORK(LPHI),RWORK(LWT))*100.0D0*UROUND
      IF( R <= 1.0D0 ) GO TO 525
      ! Multiply RTOL and ATOL by R and return
      IF( INFO(2) == 1 ) GO TO 523
         RTOL(1) = R*RTOL(1)
         ATOL(1) = R*ATOL(1)
         IDID = -2
         GO TO 527
  523 DO I = 1, NEQ
         RTOL(I) = R*RTOL(I)
         ATOL(I) = R*ATOL(I)
      END DO
      IDID = -2
      GO TO 527
  525 CONTINUE

      ! Compute minimum stepsize
      HMIN = 4.0D0*UROUND*MAX(ABS(TN),ABS(TOUT))

      ! Test H vs. HMAX
      IF( INFO(7) /= 0 ) THEN
         RH = ABS(H)/RWORK(LHMAX)
         IF( RH > 1.0D0 ) H = H/RH
      ENDIF

      if( H < dt_min ) dt_min = H
      if( H > dt_max ) dt_max = H

      CALL DDASTP( TN, Y, YPRIME, NEQ, RES, JAC, SPJAC, H, RWORK(LWT),  &
                   INFO(1), IDID, RWORK(LPHI), RWORK(LDELTA), RWORK(LE), &
                   RWORK(LWM), IWORK(LIWM), RWORK(LALPHA), RWORK(LBETA), &
                   RWORK(LGAMMA), RWORK(LPSI), RWORK(LSIGMA), RWORK(LCJ), &
                   RWORK(LCJOLD), RWORK(LHOLD), RWORK(LS), HMIN,        &
                   RWORK(LROUND), IWORK(LPHASE), IWORK(LJCALC),         &
                   IWORK(LK), IWORK(LKOLD), IWORK(LNS), INFO(10), NN_IND, &
                   NTEMP, CHECK_JAC, PRINT_CHECK_JAC )

  527 IF( IDID < 0 ) GO TO 600

      !-----------------------------------------------------------------
      ! This block handles the case of a successful return from DDASTP
      ! (IDID = 1 or 12).  Test for stop conditions.
      !-----------------------------------------------------------------

      if( save_times ) then
         times_current_length = size(times_solve)
         n_times = n_times + 1
         if( n_times > times_current_length ) then
            allocate( vector_tmp(times_current_length) )
            vector_tmp(:) = times_solve(:)
            deallocate( times_solve )
            allocate( times_solve(2*times_current_length) )
            times_solve(1:times_current_length) = vector_tmp(:)
            deallocate( vector_tmp )
         end if
         times_solve(n_times) = TN
         ! saving the current time (for debugging purpose)
         current_time = TN
      end if

      if( save_orders ) then
         orders_current_length = size(orders_solve)
         n_orders = n_orders + 1
         if( n_orders > orders_current_length ) then
            allocate( vector_tmp(orders_current_length) )
            vector_tmp(:) = orders_solve(:)
            deallocate( orders_solve )
            allocate( orders_solve(2*orders_current_length) )
            orders_solve(1:orders_current_length) = vector_tmp(:)
            deallocate( vector_tmp )
         end if
         orders_solve(n_orders) = IWORK(LKOLD)
      end if

      nb_step = nb_step + 1

      if( print_progress ) then
         percent = (TN-t_0)*pc_fact
         ! saving the current percentage (for debugging purpose)
         current_percent = percent
         if( percent > 100.0d0 ) percent = 100.0d0
         if( disp_times ) then
            call system_clock( count=clock )
            left_time = dble(clock-clock_init)/clock_rate
            ! Global Linear Estimator
            total_time = left_time/(TN-t_0)*(t_end-t_0)
            call sec_2_hms( left_time, hrs_1, min_1, sec_1 )
            rem_time = nint(max(0.0d0, total_time-left_time))
            call sec_2_hms( rem_time, hrs_2, min_2, sec_2 )
            write(string,'(F5.1,A,I3,A,2(I2,A),A,I3,A,2(I2,A))')        &
                  percent, ' % (time left = ',                          &
                  hrs_1, 'h ', min_1, 'm ', sec_1, 's ',                &
                  '-- estim. remain. time = ',                          &
                  hrs_2, 'h ', min_2, 'm ', sec_2, 's)'
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
            call put_string_back_on_term_no_adv( trim(string) // char(0) )
#else
            call go_home_on_term()
            write(STDOUT,'(A)',advance='no') trim(string)
            call msFlush(STDOUT)
#endif
         else
            write(string,'(F5.1,A)') percent, ' %'
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
            call put_string_back_on_term_no_adv( trim(string) // char(0) )
#else
            call go_home_on_term()
            write(STDOUT,'(A)',advance='no') trim(string)
            call msFlush(STDOUT)
#endif
         end if
      end if

      if( idid == 12 ) then
         if( TN <= TOUT ) then
            T = TN
            idid = -12
            go to 580
         end if
      end if

      if( INFO(4) == 0 ) then
         if( INFO(3) == 0 ) then
            IF( (TN-TOUT)*H < 0.0D0 ) GO TO 500
            CALL DDATRP( TN, TOUT, Y, YPRIME, NEQ,                      &
                         IWORK(LKOLD), RWORK(LPHI), RWORK(LPSI) )
            T = TOUT
            IDID = 3
            GO TO 580
         end if
         if( (TN-TOUT)*H < 0.0D0 ) then
             T = TN
             IDID = 1
             GO TO 580
         end if
         CALL DDATRP( TN, TOUT, Y, YPRIME, NEQ,                         &
                      IWORK(LKOLD), RWORK(LPHI), RWORK(LPSI) )
         T = TOUT
         IDID = 3
         GO TO 580
      end if
      if( INFO(3) == 0 ) then
         if( (TN-TOUT)*H >= 0.0D0 ) then
            CALL DDATRP( TN, TOUT, Y, YPRIME, NEQ,                      &
                         IWORK(LKOLD), RWORK(LPHI), RWORK(LPSI) )
            T = TOUT
            IDID = 3
            GO TO 580
         end if
         if( ABS(TN-TSTOP) > 100.0D0*UROUND*(ABS(TN)+ABS(H)) ) then
            TNEXT = TN+H
            IF( (TNEXT-TSTOP)*H <= 0.0D0 ) GO TO 500
            H = TSTOP-TN
            GO TO 500
         end if
         CALL DDATRP( TN, TSTOP, Y, YPRIME, NEQ,                        &
                      IWORK(LKOLD), RWORK(LPHI), RWORK(LPSI) )
         IDID = 2
         T = TSTOP
         GO TO 580
      end if
      IF( (TN-TOUT)*H < 0.0D0 ) then
         IF( ABS(TN-TSTOP) > 100.0D0*UROUND*(ABS(TN)+ABS(H)) ) then
            T = TN
            IDID = 1
            GO TO 580
         end if
         CALL DDATRP( TN, TSTOP, Y, YPRIME, NEQ,                        &
                      IWORK(LKOLD), RWORK(LPHI), RWORK(LPSI) )
         IDID = 2
         T = TSTOP
         GO TO 580
      end if
      CALL DDATRP( TN, TOUT, Y, YPRIME, NEQ,                            &
                   IWORK(LKOLD), RWORK(LPHI), RWORK(LPSI) )
      T = TOUT
      IDID = 3
      GO TO 580

      !-----------------------------------------------------------------
      ! All successful returns from DDASSL are made from this block.
      !-----------------------------------------------------------------

  580 CONTINUE

      RWORK(LTN) = TN
      RWORK(LH) = H

      ! Store additional information for save/restore the internal data
      ! in case of continuation.
      ! Jacobian Structure
      if( INFO(6) == 1 ) then
         ! Banded structure
         if( INFO(5) == 1 ) then
            ! Banded structure provided by a user-defined routine
            IWORK(17) = 2
         else ! INFO(5) == 0
            ! Banded structure generated by Finite-Difference
            IWORK(17) = 3
         end if
      else if( INFO(12) == 1 ) then
         ! Sparse structure
         IWORK(17) = 4
      else
         ! Dense structure
         IWORK(17) = 1
      end if
      ! Number of equations
      IWORK(18) = NEQ

      ! Added by E.C. to keep the value of T (= TOUT) for the next interval...
      RWORK(10) = T

      RETURN

      !-----------------------------------------------------------------
      ! This block handles all unsuccessful returns other than
      ! for illegal input.
      !-----------------------------------------------------------------

  600 CONTINUE

      ! can handle only -1 to -15 (other are ignored)
      select case( IDID )
         case( -1, -4, -5 )
            ! Do nothing (original error message removed by E. C.)
         case( -2 )
            ! Too much accuracy for machine precision
            WRITE (XERN3, '(ES13.6)') TN
            CALL XERMSG ('SLATEC', 'DDASSL',                                  &
               'AT T = ' // XERN3 // ' TOO MUCH ACCURACY REQUESTED FOR ' //   &
               'PRECISION OF MACHINE. RTOL AND ATOL WERE INCREASED TO ' //    &
               'APPROPRIATE VALUES', IDID, 1)
         case( -3 )
            ! WT(I) <= 0.0 for some I (not at start of problem)
            WRITE (XERN3, '(ES13.6)') TN
            CALL XERMSG ('SLATEC', 'DDASSL',                                  &
               'AT T = ' // XERN3 // ' SOME ELEMENT OF WT HAS BECOME <= ' //  &
               '0.0', IDID, 1)
         case( -6 )
            ! Error test failed repeatedly or with H=HMIN
            WRITE (XERN3, '(ES13.6)') TN
            WRITE (XERN4, '(ES13.6)') H
            CALL XERMSG ('SLATEC', 'DDASSL',                                  &
               'AT T = ' // XERN3 // ' AND STEPSIZE H = ' // XERN4 //         &
               ' THE ERROR TEST FAILED REPEATEDLY OR WITH ABS(H)=HMIN',       &
               IDID, 1)
         case( -7 )
            ! Corrector convergence failed repeatedly or with H=HMIN
            WRITE (XERN3, '(ES13.6)') TN
            WRITE (XERN4, '(ES13.6)') H
            CALL XERMSG ('SLATEC', 'DDASSL',                                  &
               'AT T = ' // XERN3 // ' AND STEPSIZE H = ' // XERN4 //         &
               ' THE CORRECTOR FAILED TO CONVERGE REPEATEDLY OR WITH ' //     &
               'ABS(H)=HMIN', IDID, 1)
         case( -8 )
            ! The iteration matrix is singular
            ! (error message processed in DaeSolve routines)
         case( -9 )
            ! Corrector failure preceded by error test failures.
            WRITE (XERN3, '(ES13.6)') TN
            WRITE (XERN4, '(ES13.6)') H
            CALL XERMSG ('SLATEC', 'DDASSL',                                  &
               'AT T = ' // XERN3 // ' AND STEPSIZE H = ' // XERN4 //         &
               ' THE CORRECTOR COULD NOT CONVERGE.  ALSO, THE ERROR TEST ' // &
               'FAILED REPEATEDLY.', IDID, 1)
         case( -10 )
            ! Corrector failure because IRES = -1
            ! (error message processed in DaeSolve routines)
         case( -11 )
            ! Failure because IRES = -2
            ! (error message processed in DaeSolve routines)
         case( -12 )
            ! Failed to compute initial YPRIME
            WRITE (XERN3, '(ES13.6)') TN
            WRITE (XERN4, '(ES13.6)') H0
            CALL XERMSG ('SLATEC', 'DDASSL',                                  &
               'AT T = ' // XERN3 // ' AND STEPSIZE H = ' // XERN4 //         &
               ' THE INITIAL YPRIME COULD NOT BE COMPUTED', IDID, 1)
         case( -13 )
            ! Bad number of nonnegativity constraints
            CALL XERMSG ('SLATEC', 'DDASSL',                                  &
               'THE NUMBER OF NONNEGATIVITY CONSTRAINTS MUST BE IN THE ' //   &
               'RANGE [1,NEQ]', IDID, 1)
         case( -14 )
            ! Illegal value (NaN) found in user-supplied RESID routine
            CALL XERMSG ('SLATEC', 'DDASSL',                                  &
               'NaN value found in the vector of residue', IDID, 1)
         case( -15 )
            ! Illegal value (NaN) found in user-supplied JAC routine
            CALL XERMSG ('SLATEC', 'DDASSL',                                  &
               'NaN value found in the jacobian matrix', IDID, 1)
         case default
            print "(A,I0,A)", "DDASSL: line 1577: For your information: IDID = ", &
                              IDID, " has not been processed."
      end select

  690 CONTINUE

      INFO(1) = -1
      T = TN
      RWORK(LTN) = TN
      RWORK(LH) = H

      RETURN

      !-----------------------------------------------------------------
      ! This block handles all error returns due to illegal input, as
      ! detected before calling DDASTP. First the error message routine
      ! is called. If this happens twice in succession, execution is
      ! terminated.
      !-----------------------------------------------------------------

  701 CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'UNEXPECTED VALUE IN INFO VECTOR', 1, 1)
      GO TO 750

  702 WRITE (XERN1, '(I0)') NEQ
      CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'NEQ = ' // XERN1 // ' <= 0', 2, 1)
      GO TO 750

  703 WRITE (XERN1, '(I0)') MXORD
      CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'MAXORD = ' // XERN1 // ' NOT IN RANGE', 3, 1)
      GO TO 750

  704 WRITE (XERN1, '(I0)') LENRW
      WRITE (XERN2, '(I0)') LRW
      CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'RWORK LENGTH NEEDED, LENRW = ' // XERN1 //                    &
         ', EXCEEDS LRW = ' // XERN2, 4, 1)
      GO TO 750

  705 WRITE (XERN1, '(I0)') LENIW
      WRITE (XERN2, '(I0)') LIW
      CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'IWORK LENGTH NEEDED, LENIW = ' // XERN1 //                    &
         ', EXCEEDS LIW = ' // XERN2, 5, 1)
      GO TO 750

  706 CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'SOME ELEMENT OF RTOL IS < 0', 6, 1)
      GO TO 750

  707 CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'SOME ELEMENT OF ATOL IS < 0', 7, 1)
      GO TO 750

  708 CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'ALL ELEMENTS OF RTOL AND ATOL ARE ZERO', 8, 1)
      GO TO 750

  709 WRITE (XERN3, '(ES13.6)') TSTOP
      WRITE (XERN4, '(ES13.6)') TOUT
      CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'INFO(4) = 1 AND TSTOP = ' // XERN3 // ' BEHIND TOUT = ' //    &
         XERN4, 9, 1)
      GO TO 750

  710 WRITE (XERN3, '(ES13.6)') HMAX
      CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'HMAX = ' // XERN3 // ' < 0.0', 10, 1)
      GO TO 750

  711 WRITE (XERN3, '(ES13.6)') TOUT
      WRITE (XERN4, '(ES13.6)') T
      CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'TOUT = ' // XERN3 // ' BEHIND T = ' // XERN4, 11, 1)
      GO TO 750

  712 CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'INFO(8)=1 AND H0=0.0', 12, 1)
      GO TO 750

  713 CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'SOME ELEMENT OF WT IS <= 0.0', 13, 1)
      GO TO 750

  714 WRITE (XERN3, '(ES13.6)') TOUT
      WRITE (XERN4, '(ES13.6)') T
      CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'TOUT = ' // XERN3 // ' TOO CLOSE TO T = ' // XERN4 //         &
         ' TO START INTEGRATION', 14, 1)
      GO TO 750

  715 WRITE (XERN3, '(ES13.6)') TSTOP
      WRITE (XERN4, '(ES13.6)') T
      CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'INFO(4)=1 AND TSTOP = ' // XERN3 // ' BEHIND T = ' // XERN4,  &
         15, 1)
      GO TO 750

  717 WRITE (XERN1, '(I0)') IWORK(LML)
      CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'ML = ' // XERN1 // ' ILLEGAL.  EITHER < 0 OR > NEQ',          &
         17, 1)
      GO TO 750

  718 WRITE (XERN1, '(I0)') IWORK(LMU)
      CALL XERMSG ('SLATEC', 'DDASSL',                                  &
         'MU = ' // XERN1 // ' ILLEGAL.  EITHER < 0 OR > NEQ',          &
         18, 1)
      GO TO 750

  719 WRITE (XERN3, '(ES13.6)') TOUT
      CALL XERMSG ('SLATEC', 'DDASSL',                                  &
        'TOUT = T = ' // XERN3, 19, 1)
      GO TO 750

  750 IDID = -33
      IF( INFO(1) == -1 ) THEN
         CALL XERMSG ('SLATEC', 'DDASSL',                               &
            'REPEATED OCCURRENCES OF ILLEGAL INPUT$$' //                &
            'RUN TERMINATED. APPARENT INFINITE LOOP', -999, 2)
      ENDIF

      INFO(1) = -1

   END
