!DECK DDASLV
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 1 Mar 2019
!
subroutine DDASLV( NEQ, DELTA, WM, IWM )

!***BEGIN PROLOGUE  DDASLV
!***SUBSIDIARY
!***PURPOSE  Linear system solver for DDASSL.
!***LIBRARY   SLATEC (DASSL)
!***TYPE      DOUBLE PRECISION (SDASLV-S, DDASLV-D)
!***AUTHOR  Petzold, Linda R., (LLNL)
!***DESCRIPTION
!-----------------------------------------------------------------------
!     THIS ROUTINE MANAGES THE SOLUTION OF THE LINEAR SYSTEM ARISING
!     IN THE NEWTON ITERATION.
!     MATRICES AND REAL TEMPORARY STORAGE AND REAL INFORMATION ARE
!     STORED IN THE ARRAY WM.
!     INTEGER MATRIX INFORMATION IS STORED IN THE ARRAY IWM.
!     FOR A DENSE MATRIX, THE LINPACK ROUTINE DGESL IS CALLED.
!     FOR A BANDED MATRIX,THE LINPACK ROUTINE DGBSL IS CALLED.
!     FOR A SPARSE MATRIX, IT CALLS sp_lu_solv (SPARSE LU)
!        (in this latter case, WM and IWM are not referenced, because
!         the sparse matrix is saved into the data module 'MOD_DDEBD2')
!-----------------------------------------------------------------------
!***ROUTINES CALLED  DGBSL, DGESL
!***REVISION HISTORY  (YYMMDD)
!   830315  DATE WRITTEN
!   901009  Finished conversion to SLATEC 4.0 format (F.N.Fritsch)
!   901019  Merged changes made by C. Ulrich with SLATEC 4.0 format.
!   901026  Added explicit declarations for all variables and minor
!           cosmetic changes to prologue.  (FNF)
!***END PROLOGUE  DDASLV

   use mod_ieee, only: mf_isfinite

   IMPLICIT NONE

   INTEGER  NEQ, IWM(*)
   DOUBLE PRECISION  DELTA(*), WM(*)

   INTEGER  MEBAND, MTYPE, info
   integer, parameter :: NPD=1, LML=1, LMU=2, LMTYPE=4, LIPVT=21

   integer :: nb_jac, nb_solv
   common /slatec_daesolve_2/ nb_jac, nb_solv

   double precision :: cpu_time_0
   integer :: inside_init, nb_resid_0, nb_jac_0, nb_solv_0
   common /slatec_daesolve_3/ cpu_time_0, inside_init, nb_resid_0,      &
                              nb_jac_0, nb_solv_0

   double precision :: ddajac_rcond_min
   logical :: save_sing_jac, ddajac_investig
   common /slatec_ddajac_status/ ddajac_rcond_min, save_sing_jac,       &
                                 ddajac_investig

!=======================================================================
   nb_solv = nb_solv + 1
   if( inside_init == 1 ) nb_solv_0 = nb_solv_0 + 1

   MTYPE = IWM(LMTYPE)

   SELECT CASE( MTYPE )

   CASE( 1, 2 ) ! Dense matrix
      if( .not. ddajac_investig ) then
         if( jac_symm_pos_def ) then
            call dpotrs( 'U', NEQ, 1, WM(NPD), NEQ, DELTA, NEQ, info )
            if( info < 0 ) then
               print *, "(MUESLI:) ddaslv: internal error"
               print *, "          illegal parameter in calling dpotrs"
               print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
               stop
            end if
         else
            call dgetrs( 'N', NEQ, 1, WM(NPD), NEQ, IWM(LIPVT), DELTA, NEQ, info )
            if( .not. all(mf_isfinite(DELTA(1:NEQ))) ) then
               print *, "(MUESLI DaeSolve:) ddaslv: warning"
               print *, "          solution after solving the Newton iteration for the DAE solver"
               print *, "          contains Inf or NaN value(s)..."
            end if
            if( info < 0 ) then
               print *, "(MUESLI DaeSolve:) ddaslv: internal error"
               print *, "          illegal parameter in calling dgetrs"
               print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
               stop
            end if
         end if
      else ! ddajac_investig
         ! (Using Linpack, as in the original DDASLV version)
         if( jac_symm_pos_def ) then
            CALL DPOSL( WM(NPD), NEQ, NEQ, DELTA )
         else
            CALL DGESL( WM(NPD), NEQ, NEQ, IWM(LIPVT), DELTA, 0 )
         end if
      end if

   CASE( 3 ) ! Sparse matrix
      if( jac_symm_pos_def ) then
         ! SPARSE SOLVE via CHOLMOD
         CALL sp_chol_solv( NEQ, DELTA )
      else
         ! SPARSE SOLVE via UMFPACK
         CALL sp_lu_solv( NEQ, DELTA )
      end if

   CASE( 4, 5 ) ! Banded matrix
      if( jac_symm_pos_def ) then
         MEBAND = IWM(LMU) + 1
         CALL dpbtrs( 'U', NEQ, IWM(LMU), 1, WM(NPD), MEBAND, DELTA, NEQ, info )
         if( info < 0 ) then
            print *, "(MUESLI:) ddaslv: internal error"
            print *, "          illegal parameter in calling dpbtrs"
            print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            stop
         end if
      else
         MEBAND = 2*IWM(LML) + IWM(LMU) + 1
         CALL dgbtrs( 'N', NEQ, IWM(LML), IWM(LMU), 1, WM(NPD), MEBAND, &
                      IWM(LIPVT), DELTA, NEQ, info )
         if( info < 0 ) then
            print *, "(MUESLI:) ddaslv: internal error"
            print *, "          illegal parameter in calling dgbtrs"
            print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            stop
         end if
      end if

   END SELECT

end
