!DECK DDAJAC
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 11 Mar 2019
!
subroutine DDAJAC( NEQ, X, Y, YPRIME, DELTA, CJ, H, IER, WT, E, WM,     &
                   IWM, RES, IRES, UROUND, JAC, SPJAC, NTEMP,           &
                   CHECK_JAC, PRINT_CHECK_JAC )

!***BEGIN PROLOGUE  DDAJAC
!***SUBSIDIARY
!***PURPOSE  Compute the iteration matrix for DDASSL and form the
!            LU-decomposition.
!***LIBRARY   SLATEC (DASSL)
!***TYPE      DOUBLE PRECISION (SDAJAC-S, DDAJAC-D)
!***AUTHOR  Petzold, Linda R., (LLNL)
!***DESCRIPTION
!-----------------------------------------------------------------------
!     This routine computes the iteration matrix
!     PD = DG/DY + CJ*DG/DYPRIME   (where G(X,Y,YPRIME)=0).
!     Here PD is computed by the user-supplied
!     routine JAC if MTYPE is 1 or 4;
!     it is computed by numerical finite differencing
!        if MTYPE is 2 or 5;
!     it is computed via SPJAC (sparse matrix)
!        if MTYPE is 3.
!
!     LU-decomposition is applied to the iteration matrix PD
!
!  THE PARAMETERS HAVE THE FOLLOWING MEANINGS:
!     Y        = array containing predicted values.
!     YPRIME   = array containing predicted derivatives.
!     DELTA    = residual evaluated at (X,Y,YPRIME)
!                (used only if MTYPE=2 OR 5).
!     CJ       = scalar parameter defining iteration matrix.
!     H        = current stepsize in integration.
!     IER      = variable which is /= 0 if iteration matrix is
!                singular, and 0 otherwise.
!     WT       = vector of weights for computing norms.
!     E        = work space (temporary) of length NEQ.
!     WM       = real work space for matrices. On output it contains
!                the LU decomposition of the iteration matrix.
!     IWM      = integer work space containing matrix information.
!     RES      = name of the external user-supplied routine to
!                evaluate the residual function G(X,Y,YPRIME)
!     IRES     = flag which is equal to zero if no illegal values in
!                RES, and less than zero otherwise (if IRES is less
!                than zero, the matrix was not completed). In this
!                case (if IRES < 0), then IER = 0.
!     UROUND   = the unit roundoff error of the machine being used.
!     JAC      = name of the external user-supplied routine to
!                evaluate the iteration matrix (this routine is only
!                used if MTYPE is 1 or 4).
!     SPJAC    = name of the external user-supplied routine to
!                evaluate the iteration sparse matrix (this routine
!                is only used if MTYPE is 3).
!-----------------------------------------------------------------------
!***ROUTINES CALLED  DGBFA, DGEFA
!***REVISION HISTORY  (YYMMDD)
!   830315  DATE WRITTEN
!   901101  Last modification. For details see original file.
!***END PROLOGUE  DDAJAC
!
!-----------------------------------------------------------------------
!  Modif. É. Canot
!     2019 02 27: last argument of user-supplied subroutine JAC has been
!                 changed, to be consistant with the ODE solver usage.
!                 It is the leading dimension of the matrix (LDJAC) instead
!                 of the number of equations.
!                 This does'nt change anything for the dense case, but it
!                 does change for the banded case.
!-----------------------------------------------------------------------
   use mod_ddebd2, FJAC => ddajac_FJAC, err => ddajac_err

   use mod_ieee, only: MF_NAN
   use mod_mfdebug, only: MF_NUMERICAL_CHECK
   use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe,        &
                        full_chk_der_dae
   use mod_mfarray, only: PrintMessage
   use mod_fileio, only: post_chk_der, post_full_chk_der
   use minpack

   IMPLICIT NONE

   INTEGER :: NEQ, IER, IWM(*), IRES, NTEMP
   DOUBLE PRECISION :: X, Y(*), YPRIME(*), DELTA(*), CJ, H, WT(*),      &
                       E(*), WM(*), UROUND

   EXTERNAL :: RES, JAC, SPJAC
   INTEGER :: CHECK_JAC, PRINT_CHECK_JAC

   DOUBLE PRECISION :: FTEM(NEQ), FTEMP(NEQ),                           &
                       YNEW(NEQ), DY(NEQ), QUALITY(NEQ)

   ! the big arrays FJAC(NEQ,NEQ) and err(NEQ,NEQ) have been moved to
   ! the mod_ddebd2 module.
   !------------------------------------------------

   INTEGER :: I, I1, I2, II, IPSAVE, ISAVE, J, J1, K, L, LENPD,         &
              MBA, MBAND, MEB1, MEBAND, MSAVE, MTYPE, N, NPDM1, NROW
   DOUBLE PRECISION :: DEL, SQUR, YPSAVE, YSAVE, rcond

   integer, parameter :: NPD = 1, LML = 1, LMU = 2, LMTYPE = 4,         &
                         LIPVT = 21

   double precision :: dt_min, dt_max
   integer :: nb_step, nb_resid
   common /slatec_daesolve_1/ dt_min, dt_max, nb_step, nb_resid

   integer :: nb_jac, nb_solv
   common /slatec_daesolve_2/ nb_jac, nb_solv

   double precision :: cpu_time_0
   integer :: inside_init, nb_resid_0, nb_jac_0, nb_solv_0
   common /slatec_daesolve_3/ cpu_time_0, inside_init, nb_resid_0,      &
                              nb_jac_0, nb_solv_0

   double precision :: ddajac_rcond_min
   logical :: save_sing_jac, ddajac_investig
   common /slatec_ddajac_status/ ddajac_rcond_min, save_sing_jac,       &
                                 ddajac_investig

   logical :: box_constrained = .false.
   double precision :: ubounds(1)

   integer :: i_eqn_group, i_group_eqn, i_var_group, i_group_var
   integer :: ind, irow, imax, ind_symm, ML, MU
   logical :: quick_check_ok

!=======================================================================

   nb_jac = nb_jac + 1
   if( inside_init == 1 ) nb_jac_0 = nb_jac_0 + 1

   IER = 0
   NPDM1 = NPD - 1
   MTYPE = IWM(LMTYPE)

   SELECT CASE( MTYPE )

   CASE( 1 )
      !-----------------------------------------------------------------
      ! Dense user-supplied matrix
      !-----------------------------------------------------------------
      LENPD = NEQ**2
      ! Set jacobian to zero before calling the user-supplied routine
      DO I = 1, LENPD
         WM(NPDM1+I) = 0.0D0
      END DO

      call mf_restore_fpe( )
      CALL JAC( X, Y, YPRIME, WM(NPD), CJ, NEQ )
      call mf_save_and_disable_fpe( )

      ! very quick check about storage if jacobian is tagged as SPD: two
      ! subdiagonal elements must be zero (see the complete test below...)
      if( jac_symm_pos_def ) then
         quick_check_ok = .true.
         J = NEQ/2
         ind = (J-1)*NEQ + J + 1
         if( WM(NPDM1+ind) /= 0.0d0 ) then
            quick_check_ok = .false.
         end if
         J = NEQ - 1
         ind = (J-1)*NEQ + J + 1
         if( WM(NPDM1+ind) /= 0.0d0 ) then
            quick_check_ok = .false.
         end if
         if( .not. quick_check_ok ) then
            print "(/,A)", "(MUESLI DaeSolve:) [BDF/ddajac] ERROR"
            print "(20X,A)", "You have tagged the (dense) jacobian matrix as SPD, so"
            print "(20X,A)", "it should contain only the upper part!"
            print "(20X,A)", "Here a quick check about two subdiagonal elements failed."
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            return
         end if
      end if

      if( MF_NUMERICAL_CHECK ) then

         ! if jacobian is tagged as SPD, check that only the upper part
         ! is provided
         if( jac_symm_pos_def ) then
            J1 = NPDM1
            do J = 1, NEQ
               ! travelling the lower part of the jacobian...
               do I = J+1, NEQ
                  if( WM(I+J1) /= 0.0d0 ) then
                     print "(/,A)", "(MUESLI DaeSolve:) [BDF/ddajac] ERROR"
                     print "(20X,A)", "You have tagged the (dense) jacobian matrix as SPD, so"
                     print "(20X,A)", "it should contain only the upper part!"
                     print "(20X,A,I0,A,I0)", "Here the jacobian element for (i,j) = ", i, " ", j
                     print "(20X,A)", "is not zero."
                     mf_message_displayed = .true.
                     call muesli_trace( pause="yes" )
                     return
                  end if
               end do
               J1 = J1 + NEQ
            end do
         end if

         if( .not. allocated(FJAC) ) then
            allocate( FJAC(NEQ,NEQ) )
         end if

         ! the Jacobian must be called again with Cj = 0
         ! Set jacobian to zero before call the user-supplied routine
         FJAC(:,:) = 0.0d0

         call mf_restore_fpe( )
         CALL JAC( X, Y, YPRIME, FJAC, 0.0d0, NEQ )
         call mf_save_and_disable_fpe( )

         ! check NaN values
         do j = 1, neq
            if( jac_symm_pos_def ) then
               ! for travelling the upper part of the jacobian...
               IMAX = J
            else
               IMAX = NEQ
            end if
            do i = 1, IMAX
               if( isnan(FJAC(i,j)) ) then
                  print "(/,A)", "(MUESLI DaeSolve:) [BDF/ddajac] ERROR"
                  print "(20X,A)", "A NaN value has been found after calling the", &
                                 "user-supplied JAC routine."
                  print "(20X,A,I0,A,I0)", "This occured in matrix for (i,j) = ", i, " ", j
                  if( NAMED_EQN_PRESENCE ) then
                     call search_index_in_eqn_groups( i, i_eqn_group,   &
                                                         i_group_eqn )
                     print "(/,20X,A,A)", "Named equation is: ",        &
                                          trim(NAMED_EQN_PTR(i_eqn_group)%name)
                     print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                     if( NAMED_VAR_PRESENCE ) then
                        call search_index_in_var_groups( j, i_var_group, &
                                                            i_group_var )
                        print "(/,20X,A,A)", "Named variable is: ",     &
                                             trim(NAMED_EQN_PTR(i_var_group)%name)
                        print "(20X,A,I0)", "Variable number is: ", i_group_var
                     end if
                  end if
                  return
               end if
            end do
         end do

         if( check_jac == 1 ) then
            ! quick check of the jacobian

            ! compute first FTEM(1:NEQ)
            IRES = 0
            call mf_restore_fpe( )
            CALL RES( X, Y, YPRIME, FTEM, IRES )
            call mf_save_and_disable_fpe( )

            if( IRES < 0 ) return

            call chkder( NEQ, NEQ, Y, FTEM, FJAC, Ynew, DY, FTEM, 1,    &
                         quality, box_constrained, ubounds )

            IRES = 0
            call mf_restore_fpe( )
            CALL RES( X, Ynew, YPRIME, FTEMP, IRES )
            call mf_save_and_disable_fpe( )

            if( IRES < 0 ) return

            call chkder( NEQ, NEQ, Y, FTEM, FJAC, Ynew, DY, FTEMP, 2,   &
                         quality, box_constrained, ubounds )

            call post_chk_der( "DaeSolve", neq, neq, quality,           &
                               print_check_jac, time=X )

         else if( check_jac == 2 ) then
            ! full check of the jacobian

            if( .not. allocated(err) ) then
               allocate( err(NEQ,NEQ) )
            end if

            ! compute first FTEM(1:NEQ)
            IRES = 0
            call mf_restore_fpe( )
            CALL RES( X, Y, YPRIME, FTEM, IRES )
            call mf_save_and_disable_fpe( )

            if( IRES < 0 ) return

            call full_chk_der_dae( NEQ, X, Y, YPRIME, RES, FTEM, FJAC,  &
                                   err, IRES )

            if( IRES < 0 ) return

            call post_full_chk_der( "DaeSolve", neq, neq, err,          &
                                    print_check_jac, time=X )

         end if

      endif

      call do_dense_factorization()

   CASE( 2 )
      !-----------------------------------------------------------------
      ! Dense finite-difference-generated matrix
      ! Calling NEQ times the 'RES' user-routine, in order to build
      ! the Jacobian matrix.
      !-----------------------------------------------------------------
      IRES = 0
      NROW = NPDM1
      SQUR = SQRT(UROUND)
      DO I = 1, NEQ
         DEL = SQUR*MAX(ABS(Y(I)),ABS(H*YPRIME(I)),ABS(WT(I)))
         DEL = SIGN(DEL,H*YPRIME(I))
         DEL = (Y(I)+DEL) - Y(I)
         YSAVE = Y(I)
         YPSAVE = YPRIME(I)
         Y(I) = Y(I) + DEL
         YPRIME(I) = YPRIME(I) + CJ*DEL
         nb_resid = nb_resid + 1
         if( inside_init == 1 ) nb_resid_0 = nb_resid_0 + 1
         if( MF_NUMERICAL_CHECK ) then
            ! to be sure that, in case of use of an End Condition, all the
            ! coefficients of the vector RESID are effectively computed...
            E(1:NEQ) = MF_NAN
         endif

         call mf_restore_fpe( )
         CALL RES( X, Y, YPRIME, E, IRES )
         call mf_save_and_disable_fpe( )

         ! check NaN values
         ! Warning: in case of emergency of RESID (flag IRES = -1 or -2)
         ! we must avoid the following check
         if( MF_NUMERICAL_CHECK .and. IRES >= 0 ) then
            do k = 1, neq
               if( isnan(E(k)) ) then
                  print "(/,A)", "(MUESLI DaeSolve:) [BDF/ddajac] ERROR"
                  print "(20X,A)", "A NaN value has been found after",  &
                                   " calling the user-supplied RESID routine."
                  print "(20X,A,I0)", "This occured in delta(i) for i = ", k
                  if( NAMED_EQN_PRESENCE ) then
                     call search_index_in_eqn_groups( k, i_eqn_group,   &
                                                         i_group_eqn )
                     print "(/,20X,A,A)", "Named equation is: ",        &
                                          trim(NAMED_EQN_PTR(i_eqn_group)%name)
                     print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                  end if
                  return
               end if
            end do
         endif

         if( IRES < 0 ) return
         DO L = 1, NEQ
            WM(NROW+L) = (E(L)-DELTA(L))/DEL
         END DO

         NROW = NROW + NEQ
         Y(I) = YSAVE
         YPRIME(I) = YPSAVE
      END DO

      call do_dense_factorization()

   CASE( 3 )
      !-----------------------------------------------------------------
      ! Sparse matrix PD
      !-----------------------------------------------------------------
      ! calling SPJAC to compute the sparse matrix PD

      call mf_restore_fpe( )
      CALL SPJAC( X, Y, YPRIME, CJ, NEQ, 1, PD, IPD, JPD, NNZ )
      call mf_save_and_disable_fpe( )

      if( MF_NUMERICAL_CHECK ) then

         ! if jacobian is tagged as SPD, check that only the upper part
         ! is provided
         if( jac_symm_pos_def ) then
            ! travelling the lower part of the jacobian...
            do j = 1, NEQ
               do k = JPD(j), JPD(j+1) - 1
                  if( IPD(k) > j ) then
                     print "(/,A)", "(MUESLI DaeSolve:) [BDF/ddajac] ERROR"
                     print "(20X,A)", "You have tagged the (sparse) jacobian matrix as SPD, so"
                     print "(20X,A)", "it should contain only the upper part!"
                     print "(20X,A,I0,A,I0)", "Here the jacobian element for (i,j) = ", IPD(k), " ", j
                     print "(20X,A)", "is not zero."
                     mf_message_displayed = .true.
                     call muesli_trace( pause="yes" )
                     return
                  end if
               end do
            end do
         end if

         ! check NaN values
         do k = 1, nnz
            if( isnan(PD(k)) ) then
               print "(/,A)", "(MUESLI DaeSolve:) [BDF/ddajac] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied SPJAC routine."
               i = IPD(k)
               ! find column
               do j = 1, neq-1
                  if( k < JPD(j+1) ) exit
               end do
               print "(20X,A,I0,A,I0)", "This occured in matrix for (i,j) = ", i, " ", j
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( i, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                  if( NAMED_VAR_PRESENCE ) then
                     call search_index_in_var_groups( j, i_var_group,   &
                                                         i_group_var )
                     print "(/,20X,A,A)", "Named variable is: ",        &
                                          trim(NAMED_EQN_PTR(i_var_group)%name)
                     print "(20X,A,I0)", "Variable number is: ", i_group_var
                  end if
               end if
               return
            end if
         end do

      end if

      ! Do sparse factorization of PD.
      ! Factors are saved as module data in 'mod_ddebd2.f90'.
      if( .not. ddajac_investig ) then
         ! In OPTIM mode, the check for a singular jacobian matrix is
         ! deactivated
         call do_sparse_factorization()
      else ! ddajac_investig
         ddajac_failed = .false.
         ddajac_must_quit = .false.
         if( ddajac_rcond_min /= 0.0d0 ) then
            call sp_lu_fact(neq,rcond)
            if( rcond <= ddajac_rcond_min ) then
               ier = -1
               ddajac_failed = .true.
               ! now, we must quit the solver
               ddajac_must_quit = .true.
            end if
         else
            call do_sparse_factorization()
         end if
      end if

   CASE( 4 )
      !-----------------------------------------------------------------
      ! Banded user-supplied matrix
      !-----------------------------------------------------------------
      ML = IWM(LML)
      MU = IWM(LMU)
      if( jac_symm_pos_def ) then
         MBAND = MU + 1
         MEBAND = MBAND
      else
         MBAND = ML + MU + 1
         MEBAND = MBAND + ML
      end if
      LENPD = MEBAND*NEQ
      ! Set jacobian to zero before calling the user-supplied routine
      DO I = 1, LENPD
         WM(NPDM1+I) = 0.0D0
      END DO

      call mf_restore_fpe( )
      CALL JAC( X, Y, YPRIME, WM(NPD), CJ, MEBAND )
      call mf_save_and_disable_fpe( )

      call do_banded_factorization()

   CASE( 5 )
      !-----------------------------------------------------------------
      ! Banded finite-difference-generated matrix
      ! Calling NEQ times the 'RES' user-routine, in order to build
      ! the Jacobian matrix.
      !-----------------------------------------------------------------
      MBAND = IWM(LML) + IWM(LMU) + 1
      MBA = MIN(MBAND,NEQ)
      MEBAND = MBAND + IWM(LML)
      MEB1 = MEBAND - 1
      MSAVE = (NEQ/MBAND) + 1
      ISAVE = NTEMP - 1
      IPSAVE = ISAVE + MSAVE
      IRES = 0
      SQUR = SQRT(UROUND)
      DO J = 1, MBA
         DO N = J, NEQ, MBAND
            K = (N-J)/MBAND + 1
            WM(ISAVE+K) = Y(N)
            WM(IPSAVE+K) = YPRIME(N)
            DEL = SQUR*MAX(ABS(Y(N)),ABS(H*YPRIME(N)),ABS(WT(N)))
            DEL = SIGN(DEL,H*YPRIME(N))
            DEL = (Y(N)+DEL) - Y(N)
            Y(N) = Y(N) + DEL
            YPRIME(N) = YPRIME(N) + CJ*DEL
         END DO
         nb_resid = nb_resid + 1
         if( inside_init == 1 ) nb_resid_0 = nb_resid_0 + 1

         call mf_restore_fpe( )
         CALL RES( X, Y, YPRIME, E, IRES)
         call mf_save_and_disable_fpe( )

         IF( IRES < 0 ) RETURN
         DO N = J, NEQ, MBAND
            K = (N-J)/MBAND + 1
            Y(N) = WM(ISAVE+K)
            YPRIME(N) = WM(IPSAVE+K)
            DEL = SQUR*MAX(ABS(Y(N)),ABS(H*YPRIME(N)),ABS(WT(N)))
            DEL = SIGN(DEL,H*YPRIME(N))
            DEL = (Y(N)+DEL) - Y(N)
            I1 = MAX(1,(N-IWM(LMU)))
            I2 = MIN(NEQ,(N+IWM(LML)))
            II = N*MEB1 - IWM(LML) + NPDM1
            DO I = I1, I2
               WM(II+I) = (E(I)-DELTA(I))/DEL
            END DO
         END DO
      END DO

      call do_banded_factorization()

   END SELECT

contains
!_______________________________________________________________________
!
   subroutine do_dense_factorization()

      integer :: info

      if( ddajac_investig ) then
         ! debugging purpose, in case where the Jacobian matrix is singular;
         ! (cannot copy it after the call, because DGEFA or DGECO destroy it)
         ! -> we have to store the transpose of the jacobian matrix !
         ddajac_must_quit = .false.
         if( ddajac_failed ) then
            if( .not. allocated(transp_Jac) ) then
               allocate( transp_Jac(NEQ,NEQ) )
            end if
            k = 0
            if( jac_symm_pos_def ) then
               ! filling only the upper part
               do j = 1, NEQ
                  do i = 1, j
                     transp_Jac(i,j) = WM(NPD+k)
                     k = k + 1
                  end do
               end do
               ! copying the lower part
               do j = 1, NEQ
                  do i = j+1, NEQ
                     transp_Jac(i,j) = transp_Jac(j,i)
                  end do
               end do
            else
               do j = 1, NEQ
                  do i = 1, NEQ
                     transp_Jac(i,j) = WM(NPD+k)
                     k = k + 1
                  end do
               end do
            end if
            ! now, we must quit the solver
            ddajac_must_quit = .true.
         end if
      end if

      if( .not. ddajac_investig ) then
         if( jac_symm_pos_def ) then
            ! Do Cholesky decomposition of P.
            call dpotrf( 'U', NEQ, WM(NPD), NEQ, IER )
            if( IER < 0 ) then
               print *, "(MUESLI:) dpjac: internal error"
               print *, "          illegal parameter in calling dpotrf"
               print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
               stop
            else if( IER > 0 ) then
               print *, "(MUESLI DaeSolve:) [BDF/ddajac] ERROR"
               print *, "                   The jacobian matrix should be positive definite!"
               print *, "                   'DPOTRF' routine failed, so it is not SPD."
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               return
            end if
         else
            ! Do LU decomposition of P.
            call dgetrf( NEQ, NEQ, WM(NPD), NEQ, IWM(LIPVT), IER )
            if( IER < 0 ) then
               print "()"
               print *, "(MUESLI:) ddajac: internal error"
               print *, "          illegal parameter in calling DGETRF"
               print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
               stop
            else if( IER > 0 ) then
               print "()"
               print *, "(MUESLI DaeSolve:) [BDF/ddajac] Warning"
               print *, "          The factorization using DGETRF has been completed,"
               print *, "          but the factor U is exactly singular, and division"
               print *, "          by zero will occur..."
            end if
         end if
      else ! ddajac_investig
         ! (Using Linpack, as in the original DDAJAC version)
         ! DGECO is slightly slower than DGEFA, but it is more robust and
         ! it gives more reliable information about the set of equations.
         if( .not. allocated(dgeco_z) ) then
            allocate( dgeco_z(NEQ) )
         end if
         IER = 0
         if( jac_symm_pos_def ) then
            call DPOCO( WM(NPD), NEQ, NEQ, rcond, dgeco_z, info )
            if( info > 0 ) then
               print *, "(MUESLI DaeSolve:) [BDF/ddajac] ERROR"
               print *, "                   The jacobian matrix should be positive definite!"
               print *, "                   'DPOCO' routine failed, so it is not SPD."
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               return
            end if
         else
            call DGECO( WM(NPD), NEQ, NEQ, IWM(LIPVT), rcond, dgeco_z )
         end if
         ddajac_failed = .false.
         if( rcond <= ddajac_rcond_min ) then
            IER = -1
            ddajac_failed = .true.
         end if
      end if

   end subroutine do_dense_factorization
!_______________________________________________________________________
!
   subroutine do_banded_factorization()

      if( jac_symm_pos_def ) then
         ! Do Cholesky decomposition of P.
         call dpbtrf( 'U', NEQ, IWM(LMU), WM(NPD), MEBAND, IER )
         if( IER < 0 ) then
            print *, "(MUESLI:) dpjac: internal error"
            print *, "          illegal parameter in calling dpbtrf"
            print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            stop
         else if( IER > 0 ) then
            print *, "(MUESLI OdeSolve:) [BDF/dpjac] ERROR"
            print *, "                   The jacobian matrix should be positive definite!"
            print *, "                   'DPOTRF' routine failed, so it is not SPD."
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            return
         end if
      else
         ! Do LU decomposition of P.
         call dgbtrf ( NEQ, NEQ, IWM(LML), IWM(LMU), WM(NPD), MEBAND,   &
                       IWM(LIPVT), IER )
         if( IER < 0 ) then
            print *, "(MUESLI:) ddajac: internal error"
            print *, "          illegal parameter in calling dgbtrf"
            print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            stop
         else if( IER > 0 ) then
            print "()"
            print *, "(MUESLI DaeSolve:) [BDF/ddajac] Warning"
            print *, "          The factorization using DGBTRF has been completed,"
            print *, "          but the factor U is exactly singular, and division"
            print *, "          by zero will occur..."
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            return
         end if
      end if

   end subroutine do_banded_factorization
!_______________________________________________________________________
!
   subroutine do_sparse_factorization()

      if( jac_symm_pos_def ) then
         ! Do sparse Cholesky decomposition of P. Factors are module data
         ! saved in 'mod_ddebd2.f90'
         call sp_chol_fact(neq)
      else
         ! Do sparse LU decomposition of P. LU factors are module data
         ! saved in 'mod_ddebd2.f90'
         call sp_lu_fact(neq)
      end if

   end subroutine do_sparse_factorization
!_______________________________________________________________________
!
end
