!DECK DDAINI
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 27 Feb 2019
!
   SUBROUTINE DDAINI( X, Y, YPRIME, NEQ, RES, JAC, SPJAC, H, WT, IDID,  &
                      PHI, DELTA, E, WM, IWM, HMIN, UROUND, NONNEG,     &
                      NN_IND, NTEMP, CHECK_JAC, PRINT_CHECK_JAC )

!***BEGIN PROLOGUE  DDAINI
!***SUBSIDIARY
!***PURPOSE  Initialization routine for DDASSL.
!***LIBRARY   SLATEC (DASSL)
!***TYPE      DOUBLE PRECISION (SDAINI-S, DDAINI-D)
!***AUTHOR  Petzold, Linda R., (LLNL)
!***DESCRIPTION
!-----------------------------------------------------------------------
!     DDAINI takes one step of size H or smaller with the backward
!     Euler method, to find YPRIME.  X and Y are updated to be
!     consistent with the new step.  A modified damped Newton iteration
!     is used to solve the corrector iteration.
!
!     The initial guess for YPRIME is used in the prediction, and in
!     forming the iteration matrix, but is not involved in the error
!     test. This may have trouble converging if the initial guess is
!     no good, or if G(X,Y,YPRIME) depends nonlinearly on YPRIME.
!
!     The parameters represent:
!     X         -- Independent variable
!     Y         -- Solution vector at X
!     YPRIME    -- Derivative of solution vector
!     NEQ       -- Number of equations
!     H         -- Stepsize. IMDER may use a stepsize smaller than H.
!     RES       -- External user-supplied subroutine to evaluate the
!                  residual. The call is:
!                         CALL RES(X,Y,YPRIME,DELTA,IRES)
!                  X,Y,YPRIME are input.  DELTA is output.
!                  On input, IRES = 0.  RES should alter IRES only
!                  if it encounters an illegal value of Y or a
!                  stop condition.  If IRES has been found to be a
!                  negative value, DDAINI returns control to the calling
!                  program with IDID = -102.
!     JAC       -- External user-supplied routine to evaluate the
!                  iteration matrix (this is optional). The call is:
!                         CALL JAC(X,Y,YPRIME,PD,CJ,LDJAC)
!                  PD is the matrix of partial derivatives,
!                  PD = DG/DY + CJ*DG/DYPRIME
!     SPJAC     -- External user-supplied routine to evaluate
!                  the iteration matrix (this is optional) under
!                  sparse storage. The call is of the form
!                    CALL SPJAC(X,Y,YPRIME,CJ,NROW,JOB,PD,IPD,JPD,NNZ)
!                  PD is the sparse matrix of partial derivatives,
!                  PD = DG/DY + CJ*DG/DYPRIME
!     WT        -- Vector of weights for error criterion
!     IDID      -- Completion code with the following meanings
!                    IDID =    1 -- YPRIME was found successfully
!                    IDID =   -8 -- The iteration matrix is singular
!                    IDID =  -14 -- A NaN value has been found in DELTA,
!                                   after calling RES().
!                    IDID = -102 -- DDAINI failed to find YPRIME
!     PHI       -- Work space for DDAINI
!     DELTA, E  -- Work space for DDAINI
!     WM, IWM   -- Real and integer arrays storing matrix information
!
!     NONNEG    -- Number of non-negativity constraints to be applied
!                  to some equations. Index of these equations are
!                  described in the NN_IND(*) integer array.
!     NN_IND(*) -- Integer array, of size at least NONNEG, which
!                  contains indexes of equations which must verified
!                  a non-negative constraint.
!-----------------------------------------------------------------------
!***ROUTINES CALLED  DDAJAC, DDANRM, DDASLV
!***REVISION HISTORY  (YYMMDD)
!   830315  DATE WRITTEN
!   901030  Last modification. For details see original file.
!***END PROLOGUE  DDAINI
!
      use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe
      use mod_mfdebug, only: mf_message_displayed, muesli_trace,        &
                             MF_NUMERICAL_CHECK
      use mod_ieee, only: MF_NAN

      IMPLICIT NONE

      INTEGER :: NEQ, IDID, IWM(*), NONNEG, NN_IND(*), NTEMP
      DOUBLE PRECISION :: X, Y(*), YPRIME(*), H, WT(*), PHI(NEQ,*),     &
                          DELTA(*), E(*), WM(*), HMIN, UROUND

      EXTERNAL :: RES, JAC, SPJAC
      INTEGER :: CHECK_JAC, PRINT_CHECK_JAC

      INTEGER :: I, IER, IRES, JCALC, M, NCF, NEF, NSF, II
      DOUBLE PRECISION :: CJ, DELNRM, ERR, OLDNRM, R, RATE, S,    &
                          XOLD, YNORM
      LOGICAL :: CONVGD

      integer, parameter :: LNRE=12, LNJE=13

      ! default maxit: 10
      integer, save :: MAXIT = 10, MJAC = 5
      double precision, save :: DAMP = 0.75D0

      double precision :: dt_min, dt_max
      integer :: nb_step, nb_resid
      common /slatec_daesolve_1/ dt_min, dt_max, nb_step, nb_resid

      double precision :: cpu_time_0
      integer :: inside_init, nb_resid_0, nb_jac_0, nb_solv_0
      common /slatec_daesolve_3/ cpu_time_0, inside_init, nb_resid_0,   &
                                 nb_jac_0, nb_solv_0

      integer ddaini_info
      common /slatec_ddaini_status/ ddaini_info

      integer :: i_eqn_group, i_group_eqn

      !-----------------------------------------------------------------------
      ! Block 1.
      ! Initializations.
      !-----------------------------------------------------------------------

      IDID = 1
      NEF = 0
      NCF = 0
      NSF = 0
      XOLD = X
      YNORM = DDANRM(NEQ,Y,WT)

      ! Save Y and YPRIME in PHI
      do I = 1, NEQ
         PHI(I,1) = Y(I)
         PHI(I,2) = YPRIME(I)
      end do

      !----------------------------------------------------
      ! Block 2.
      ! Do one backward euler step.
      !----------------------------------------------------

      ! Set up for start of corrector iteration
  200 CJ = 1.0D0/H
      X = X + H

      if( H < dt_min ) dt_min = H
      if( H > dt_max ) dt_max = H

      ! Predict solution and derivative
      do I = 1, NEQ
         Y(I) = Y(I)+H*YPRIME(I)
      end do

      JCALC = -1
      M = 0
      CONVGD = .TRUE.

      ! Corrector loop.
  300 IWM(LNRE) = IWM(LNRE)+1
      IRES = 0

      nb_resid = nb_resid + 1
      if( inside_init == 1 ) nb_resid_0 = nb_resid_0 + 1
      if( MF_NUMERICAL_CHECK ) then
         ! to be sure that, in case of use of an End Condition, all the
         ! coefficients of the vector RESID are effectively computed...
         DELTA(1:NEQ) = MF_NAN
      endif

      call mf_restore_fpe( )
      CALL RES( X, Y, YPRIME, DELTA, IRES )
      call mf_save_and_disable_fpe( )

      ! check NaN values
      ! Warning: in case of emergency of RESID (flag IRES = -1 or -2)
      ! we must avoid the following check
      if( MF_NUMERICAL_CHECK .and. IRES >= 0 ) then
         do i = 1, neq
            if( isnan(delta(i)) ) then
               print "(/,A)", "(MUESLI DaeSolve:) [BDF/ddaini] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied RESID routine."
               print "(20X,A,I0)", "This occured in delta(i) for i = ", i
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( i, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               IDID = -14
               return
            end if
         end do
      end if
      if( IRES < 0 ) then
         ! Got non-zero value of 'flag' from the RES user-routine
         ddaini_info = -1
         GO TO 430
      end if

      ! Evaluate the iteration matrix
      if( JCALC /= -1 ) GO TO 310
      IWM(LNJE) = IWM(LNJE) + 1
      JCALC = 0
      CALL DDAJAC( NEQ, X, Y, YPRIME, DELTA, CJ, H,                     &
                   IER, WT, E, WM, IWM, RES, IRES,                      &
                   UROUND, JAC, SPJAC, NTEMP,                           &
                   CHECK_JAC, PRINT_CHECK_JAC )
      if( ddajac_must_quit ) then
         ! singular Jacobian (case handled by dgefa_info)
         IDID = -8
         RETURN
      end if
      S = 1000000.D0
      if( IRES < 0 ) then
         ! Jacobian matrix was not completed (cf. DDAJAC)
         ddaini_info = -2
         GO TO 430
      end if
      if( IER /= 0 ) GO TO 430
      NSF = 0

      ! Multiply residual by damping factor
  310 CONTINUE
      do I = 1, NEQ
         DELTA(I) = DELTA(I)*DAMP
      end do

      ! Compute a new iterate (back substitution)
      ! Store the correction in DELTA

      CALL DDASLV( NEQ, DELTA, WM, IWM )

      ! Update Y and YPRIME
      do I = 1, NEQ
         Y(I) = Y(I) - DELTA(I)
         YPRIME(I) = YPRIME(I) - CJ*DELTA(I)
      end do

      ! Test for convergence of the iteration.

      DELNRM = DDANRM(NEQ,DELTA,WT)
      if( DELNRM <= 100.D0*UROUND*YNORM ) GO TO 400

      if( M > 0 ) GO TO 340
         OLDNRM = DELNRM
         GO TO 350

  340 RATE = (DELNRM/OLDNRM)**(1.0D0/M)
      if( RATE > 0.90D0 ) GO TO 430
      S = RATE/(1.0D0-RATE)

  350 if( S*DELNRM <= 0.33D0 ) GO TO 400

      ! The corrector has not yet converged. Update M and test whether
      ! the maximum number of iterations have been tried.
      ! Every MJAC iterations, get a new iteration matrix.

      M = M + 1
      if( M >= MAXIT ) GO TO 430

      if( (M/MJAC)*MJAC == M ) JCALC = -1
      GO TO 300

      ! The iteration has converged.
      ! Check nonnegativity constraints
      ! (modified by E.C. : generalized non-negativity constraints to be
      !  applied only to a subset of the NEQ equations)
  400 if( NONNEG == 0 ) GO TO 450
      do II = 1, NONNEG
         I = NN_IND(II)
         DELTA(I) = MIN(Y(I),0.0D0)
      end do
      DELNRM = DDANRM(NEQ,DELTA,WT)
      if( DELNRM > 0.33D0 ) GO TO 430
      do II = 1, NONNEG
         I = NN_IND(II)
         Y(I) = Y(I) - DELTA(I)
         YPRIME(I) = YPRIME(I) - CJ*DELTA(I)
      end do
      GO TO 450

      ! Exits from corrector loop.
  430 CONVGD = .FALSE.
  450 if( .NOT. CONVGD ) GO TO 600

      !-----------------------------------------------------
      ! Block 3.
      ! The corrector iteration converged.
      ! Do error test.
      !-----------------------------------------------------

      do I = 1, NEQ
         E(I) = Y(I) - PHI(I,1)
      end do
      ERR = DDANRM(NEQ,E,WT)

      if( ERR <= 1.0D0 ) RETURN

      !-----------------------------------------------------------------------
      ! Block 4.
      ! The backward Euler step failed. Restore X, Y and YPRIME to
      ! their original values. Reduce stepsize and try again, if
      ! possible.
      !-----------------------------------------------------------------------

  600 CONTINUE
      X = XOLD
      do I = 1, NEQ
         Y(I) = PHI(I,1)
         YPRIME(I) = PHI(I,2)
      end do

      if( CONVGD ) GO TO 640

      if( IER /= 0 ) then
         NSF = NSF + 1
         H = 0.25D0*H
         if( NSF < 3 .AND. ABS(H) >= HMIN ) GO TO 200
         IDID = -8
         RETURN
      end if

      if( IRES == -2 ) then
         IDID = -102
         RETURN
      end if

      NCF = NCF + 1
      H = 0.25D0*H
      if( NCF < 10 .AND. ABS(H) >= HMIN ) GO TO 200

      IDID = -102
      RETURN

  640 NEF = NEF + 1
      R = 0.90D0/(2.0D0*ERR+0.0001D0)
      R = MAX(0.1D0,MIN(0.5D0,R))
      H = H*R

      if( NEF < 10 .AND. ABS(H) >= HMIN ) GO TO 200

      IDID = -102

   END
