module rational_numbers

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

   ! version : december 1, 2008

   implicit none

   private

#include "rational_config.inc"

   ! try to select a integer kind such that : -10^r < i < 10^r
#if defined _RAT_1_BYTE
   integer, parameter :: prec = selected_int_kind(r=2) ! one byte
   integer, parameter :: prec_ext = selected_int_kind(r=4) ! two bytes
   double precision, parameter :: XX_THRES = -7.0d0
#elif defined _RAT_2_BYTES
   integer, parameter :: prec = selected_int_kind(r=4) ! two bytes
   integer, parameter :: prec_ext = selected_int_kind(r=6) ! four bytes
   double precision, parameter :: XX_THRES = -15.0d0
#elif defined _RAT_4_BYTES
   integer, parameter :: prec = selected_int_kind(r=6) ! four bytes
   integer, parameter :: prec_ext = selected_int_kind(r=12) ! eight bytes
   double precision, parameter :: XX_THRES = -31.0d0
#else
- '(MUESLI rational_numbers:) config not defined!'
#endif

   integer(kind=prec_ext), save :: int_max = huge(1_prec)

   ! only for exporting (avoid name clashing)
   integer, parameter :: rat_num_prec = prec

   type, public :: rational
      integer(kind=prec) :: num = 0_prec ! +/-
      integer(kind=prec) :: den = 1_prec ! > 0
   end type rational

   type(rational), public, parameter ::                                 &
                           RAT_0 = rational(   0_prec, 1_prec ),        &
                           RAT_1 = rational(   1_prec, 1_prec ),        &
                           RAT_M1 = rational( -1_prec, 1_prec ),        &
                           RAT_2 = rational(   2_prec, 1_prec ),        &
                           RAT_M2 = rational( -2_prec, 1_prec ),        &
                           RAT_3 = rational(   3_prec, 1_prec ),        &
                           RAT_M3 = rational( -3_prec, 1_prec ),        &
                           RAT_4 = rational(   4_prec, 1_prec )

   ! operator overload for comparison in 'mfUnit_equal_mfUnit' in the
   ! 'mod_physunits' module.
   interface operator(/=)
      module procedure rational_not_equal_rational
   end interface operator(/=)

   interface rational_mul
      module procedure rational_mul_int
      module procedure rational_mul_rat
   end interface rational_mul

   interface assignment(=)
      module procedure real_to_rational
   end interface assignment(=)

   public :: rat_num_prec
   public :: rational_add, rational_sub, rational_mul, rational_div
   public :: assignment(=), operator(/=)

#if defined _INTEL_IFC
! bug de INTEL-ifort !
! dans fml_matfun/Funm.inc, line 773, in the following statement:
!         G = ( mfConj(C) .hc. S ) .vc.                               &
!             (       -S  .hc. C )
! the operator(-) is misunderstood !
   public :: rational_neg
#else
   interface operator(-)
      module procedure rational_neg
   end interface operator(-)

   public :: operator(-)
#endif

   ! INFO: cannot introduce operator(+), operator(-), ... for
   !       procedures rational_add, rational_sub, rational_mul and
   !       rational_div, because they return two objects, not one.
   !       (the other returned object is a status flag)
   !
   !       In an ordinary framework, this status flag could be removed,
   !       but for MUESLI, I think it is better to have it: in case of
   !       error, we can raise the error in the 'mod_physunits' module
   !       instead in the 'rational_numbers' one. This make the error
   !       more clear for the developper and the user.

contains
!_______________________________________________________________________
!
   subroutine rational_add( r1, r2, res, status )

      type(rational), intent(in) :: r1, r2
      type(rational), intent(out) :: res
      integer, intent(out) :: status
      !------ API end ------

      ! if non-zero :
      !    status=1 indicates integer overflow
      !    status=-1 indicates bad arg. value (denominator null)

      integer(kind=prec) :: k, m1, m2
      integer(kind=prec_ext) :: k_ext, num_ext, den_ext

   !------ end of declarations -- execution starts hereafter  ------

      status = 0

      if( r1%den < 1_prec ) then
         print *
         print *, "(rational_numbers:) operator(+) : denominator must be > 0!"
         print *, '  r1%den = ', r1%den
         status = -1
         return
      end if

      if( r2%den < 1_prec ) then
         print *
         print *, "(rational_numbers:) operator(+) : denominator must be > 0!"
         print *, '  r2%den = ', r2%den
         status = -1
         return
      end if

      k = gcd( r1%den, r2%den )
      if( k /= 1_prec ) then
         m1 = r1%den / k
         m2 = r2%den / k
      else
         m1 = r1%den
         m2 = r2%den
      end if

      num_ext = int(r1%num,kind=prec_ext)*int(m2,kind=prec_ext) +       &
                int(r2%num,kind=prec_ext)*int(m1,kind=prec_ext)
      den_ext = int(r1%den,kind=prec_ext)*int(m2,kind=prec_ext)

      k_ext = gcd_ext( num_ext, den_ext )
      if( k_ext /= 1_prec_ext ) then
         num_ext = num_ext / k_ext
         den_ext = den_ext / k_ext
      end if

      ! integer overflow detection
      if( abs(num_ext) > int_max ) then
         status = 1
         return
      end if
      if( den_ext > int_max ) then
         status = 1
         return
      end if

      res%num = num_ext
      res%den = den_ext

   end subroutine rational_add
!_______________________________________________________________________
!
   subroutine rational_sub( r1, r2, res, status )

      type(rational), intent(in) :: r1, r2
      type(rational), intent(out) :: res
      integer, intent(out) :: status
      !------ API end ------

      ! if non-zero :
      !    status=1 indicates integer overflow
      !    status=-1 indicates bad arg. value (denominator null)

      integer(kind=prec) :: k, m1, m2
      integer(kind=prec_ext) :: k_ext, num_ext, den_ext

   !------ end of declarations -- execution starts hereafter  ------

      status = 0

      if( r1%den < 1_prec ) then
         print *
         print *, "(rational_numbers:) operator(-) : denominator must be > 0!"
         print *, '  r1%den = ', r1%den
         status = -1
         return
      end if

      if( r2%den < 1_prec ) then
         print *
         print *, "(rational_numbers:) operator(-) : denominator must be > 0!"
         print *, '  r2%den = ', r2%den
         status = -1
         return
      end if

      k = gcd( r1%den, r2%den )
      if( k /= 1_prec ) then
         m1 = r1%den / k
         m2 = r2%den / k
      else
         m1 = r1%den
         m2 = r2%den
      end if

      num_ext = int(r1%num,kind=prec_ext)*int(m2,kind=prec_ext) -       &
                int(r2%num,kind=prec_ext)*int(m1,kind=prec_ext)
      den_ext = int(r1%den,kind=prec_ext)*int(m2,kind=prec_ext)

      k_ext = gcd_ext( num_ext, den_ext )
      if( k_ext /= 1_prec_ext ) then
         num_ext = num_ext / k_ext
         den_ext = den_ext / k_ext
      end if

      ! integer overflow detection
      if( abs(num_ext) > int_max ) then
         status = 1
         return
      end if
      if( den_ext > int_max ) then
         status = 1
         return
      end if

      res%num = num_ext
      res%den = den_ext

   end subroutine rational_sub
!_______________________________________________________________________
!
   subroutine rational_mul_int( r, i, res, status )

      type(rational), intent(in) :: r
      integer, intent(in) :: i
      type(rational), intent(out) :: res
      integer, intent(out) :: status
      !------ API end ------

      ! if non-zero :
      !    status=1 indicates integer overflow
      !    status=-1 indicates bad arg. value (denominator null)

      integer(kind=prec_ext) :: k_ext, num_ext, den_ext

   !------ end of declarations -- execution starts hereafter  ------

      status = 0

      if( r%den < 1_prec ) then
         print *
         print *, "(rational_numbers:) operator(*) : denominator must be > 0!"
         print *, '  r%den = ', r%den
         status = -1
         return
      end if

      num_ext = int(i,kind=prec_ext)*int(r%num,kind=prec_ext)
      den_ext = r%den

      k_ext = gcd_ext( num_ext, den_ext )
      if( k_ext /= 1_prec_ext ) then
         num_ext = num_ext / k_ext
         den_ext = den_ext / k_ext
      end if

      ! integer overflow detection
      if( abs(num_ext) > int_max ) then
         status = 1
         return
      end if

      res%num = num_ext
      res%den = den_ext

   end subroutine rational_mul_int
!_______________________________________________________________________
!
   subroutine rational_mul_rat( r1, r2, res, status )

      type(rational), intent(in) :: r1, r2
      type(rational), intent(out) :: res
      integer, intent(out) :: status
      !------ API end ------

      ! if non-zero :
      !    status=1 indicates integer overflow
      !    status=-1 indicates bad arg. value (denominator null)

      integer(kind=prec_ext) :: k_ext, num_ext, den_ext

   !------ end of declarations -- execution starts hereafter  ------

      status = 0

      if( r1%den < 1_prec ) then
         print *
         print *, "(rational_numbers:) operator(*) : denominator must be > 0!"
         print *, '  r1%den = ', r1%den
         status = -1
         return
      end if

      if( r2%den < 1_prec ) then
         print *
         print *, "(rational_numbers:) operator(+) : denominator must be > 0!"
         print *, '  r2%den = ', r2%den
         status = -1
         return
      end if

      num_ext = int(r1%num,kind=prec_ext)*int(r2%num,kind=prec_ext)
      den_ext = int(r1%den,kind=prec_ext)*int(r2%den,kind=prec_ext)

      k_ext = gcd_ext( num_ext, den_ext )
      if( k_ext /= 1_prec_ext ) then
         num_ext = num_ext / k_ext
         den_ext = den_ext / k_ext
      end if

      ! integer overflow detection
      if( abs(num_ext) > int_max ) then
         status = 1
         return
      end if

      res%num = num_ext
      res%den = den_ext

   end subroutine rational_mul_rat
!_______________________________________________________________________
!
   subroutine rational_div( r, i, res, status )

      type(rational), intent(in) :: r
      integer, intent(in) :: i
      type(rational), intent(out) :: res
      integer, intent(out) :: status
      !------ API end ------

      ! i > 0

      ! if non-zero :
      !    status=1 indicates integer overflow
      !    status=-1 indicates bad arg. value (denominator null, or
      !                                        divisor null)

      integer(kind=prec_ext) :: k_ext, num_ext, den_ext

   !------ end of declarations -- execution starts hereafter  ------

      if( r%den < 1_prec ) then
         print *
         print *, "(rational_numbers:) operator(/) : denominator must be > 0!"
         print *, '  r%den = ', r%den
         status = -1
         return
      end if

      if( i <= 0 ) then
         print *
         print *, "(rational_numbers:) operator(/) : i must be > 0!"
         print *, '  divisor = ', i
         status = -1
         return
      end if

      status = 0

      num_ext = r%num
      den_ext = int(r%den,kind=prec_ext)*int(i,kind=prec_ext)

      k_ext = gcd_ext( num_ext, den_ext )
      if( k_ext /= 1_prec_ext ) then
         num_ext = num_ext / k_ext
         den_ext = den_ext / k_ext
      end if

      ! integer overflow detection
      if( abs(den_ext) > int_max ) then
         status = 1
         return
      end if

      res%num = num_ext
      res%den = den_ext

   end subroutine rational_div
!_______________________________________________________________________
!
   elemental function rational_neg( r ) result( res )

      type(rational), intent(in) :: r
      type(rational) :: res
      !------ API end ------

   !------ end of declarations -- execution starts hereafter  ------

      res%num = -r%num
      res%den =  r%den

   end function rational_neg
!_______________________________________________________________________
!
   subroutine real_to_rational( rat, x )

      type(rational), intent(in out) :: rat
      double precision, intent(in) :: x
      !------ API end ------

      ! conversion from a real number to its rational form
      ! (after Matlab 7.1 : specfun/rat.m, version 2004/07/05)
      !
      ! note: the conversion is of course almost inexact!

   !------ end of declarations -- execution starts hereafter  ------

      integer(kind=prec_ext) :: n_old, d_old, n, d, dd
      integer(kind=prec_ext) :: n_try, d_try
      double precision :: xx, xx_log2

      n_old = 0_prec_ext
      d_old = 1_prec_ext

      n = 1_prec_ext
      d = 0_prec_ext

      xx = x
      do
         dd = nint(xx)
         xx = xx - dd
         n_try = n*dd + n_old
         d_try = d*dd + d_old
         if( abs(n_try) > int_max .or. abs(d_try) > int_max ) then
            exit
         end if
         n_old = n
         d_old = d
         n = n_try
         d = d_try
         if( xx == 0.0d0 ) then
            exit
         end if
         xx_log2 = log(abs(xx))/log(2.0d0)
         ! to avoid integer overflow, we must have : log2(1/xx) <= k
         if( xx_log2 < XX_THRES ) then
            exit
         end if
         xx = 1.0d0 / xx
      end do
      if( d < 0_prec_ext ) then
         d = abs(d)
         n = -n
      end if

      rat%num = n
      rat%den = d

   end subroutine real_to_rational
!_______________________________________________________________________
!
   recursive function gcd( a, b ) result( divisor )

      integer(kind=prec), intent(in) :: a, b
      integer(kind=prec) :: divisor
      !------ API end ------

      integer(kind=prec) :: m, n

   !------ end of declarations -- execution starts hereafter  ------

      m = abs(a)
      n = abs(b)

      if ( m > n ) call swap( m, n )  ! Insure that m <= n.

      if ( m == 0_prec ) then
         divisor = n
      else
         divisor = gcd( mod( n, m ), m )
      end if

   contains

      subroutine swap( x, y )

         integer(kind=prec) :: x, y
         integer(kind=prec) :: tmp

         tmp = x
         x = y
         y = tmp

      end subroutine swap

   end function gcd
!_______________________________________________________________________
!
   recursive function gcd_ext( a, b ) result( divisor )

      integer(kind=prec_ext), intent(in) :: a, b
      integer(kind=prec_ext) :: divisor
      !------ API end ------

      integer(kind=prec_ext) :: m, n

   !------ end of declarations -- execution starts hereafter  ------

      m = abs(a)
      n = abs(b)

      if ( m > n ) call swap( m, n )  ! Insure that m <= n.

      if ( m == 0_prec_ext ) then
         divisor = n
      else
         divisor = gcd_ext( mod( n, m ), m )
      end if

   contains

      subroutine swap( x, y )

         integer(kind=prec_ext) :: x, y
         integer(kind=prec_ext) :: tmp

         tmp = x
         x = y
         y = tmp

      end subroutine swap

   end function gcd_ext
!_______________________________________________________________________
!
   function rational_not_equal_rational( a, b ) result( bool )

      type(rational), intent(in) :: a, b
      logical :: bool
      !------ API end ------

      type(rational) :: res
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call rational_sub( a, b, res, status )
      if( status /= 0 ) then
         print *
         print *, "(rational_numbers:) operator(/=) : error in substracting b from a!"
         print *
      end if

      if( res%num == 0_prec ) then
         bool = .false.
      else
         bool = .true.
      end if

   end function rational_not_equal_rational
!_______________________________________________________________________
!
end module rational_numbers
