! f90 include file
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 13 Jul 2017
!_______________________________________________________________________
!
   subroutine r1updt( m, n, s, ls, u, v, w, sing )

      integer :: m, n, ls
      logical :: sing
      double precision :: s(ls), u(m), v(n), w(m)
!     **********
!
!     subroutine r1updt
!
!     Given an m by n lower trapezoidal matrix S, an m-vector u,
!     and an n-vector v, the problem is to determine an
!     orthogonal matrix Q such that
!
!                   t
!           (S + u*v )*Q
!
!     is again lower trapezoidal.
!
!     This subroutine determines Q as the product of 2*(n - 1)
!     transformations
!
!           gv(n-1)*...*gv(1)*gw(1)*...*gw(n-1)
!
!     where gv(i), gw(i) are Givens rotations in the (i,n) plane
!     which eliminate elements in the i-th and n-th planes,
!     respectively. Q itself is not accumulated, rather the
!     information to recover the gv, gw rotations is returned.
!
!     The subroutine statement is
!
!       subroutine r1updt( m, n, s, ls, u, v, w, sing )
!
!     where
!
!       m is a positive integer input variable set to the number
!         of rows of S.
!
!       n is a positive integer input variable set to the number
!         of columns of S. n must not exceed m.
!
!       s is an array of length ls. On input s must contain the lower
!         trapezoidal matrix S stored by columns. On output s contains
!         the lower trapezoidal matrix produced as described above.
!
!       ls is a positive integer input variable not less than
!         (n*(2*m-n+1))/2.
!
!       u is an input array of length m which must contain the
!         vector u.
!
!       v is an array of length n. On input v must contain the vector
!         v. On output v(i) contains the information necessary to
!         recover the Givens rotation gv(i) described above.
!
!       w is an output array of length m. w(i) contains information
!         necessary to recover the Givens rotation gw(i) described
!         above.
!
!       sing is a logical output variable. sing is set true if any
!         of the diagonal elements of the output s are zero. Otherwise
!         sing is set false.
!
!     subprograms called
!
!       fortran-supplied ... abs, sqrt
!
!     Argonne National Laboratory. Minpack project. March 1980.
!     Burton S. Garbow, Kenneth E. Hillstrom, Jorge J. Moré,
!     John L. Nazareth
!
!     **********
      integer :: i, j, jj, l, nmj, nm1
      double precision :: cos, cotan, giant, sin, tan, tau, temp
      double precision, parameter :: one = 1.0d0, p5 = 0.5d0,           &
                                     p25 = 0.25d0, zero = 0.0d0

!--------- end of declarations -- execution starts hereafter  ------

      ! giant is the largest magnitude.
      giant = huge(1.0d0)

      ! Initialize the diagonal element pointer.
      jj = (n*(2*m - n + 1))/2 - (m - n)

      ! Move the nontrivial part of the last column of S into w.
      l = jj
      do i = n, m
         w(i) = s(l)
         l = l + 1
      end do

      ! Rotate the vector v into a multiple of the n-th unit vector
      ! in such a way that a spike is introduced into w.
      nm1 = n - 1
      if( nm1 < 1 ) go to 70
      do nmj = 1, nm1
         j = n - nmj
         jj = jj - (m - j + 1)
         w(j) = zero
         if( v(j) == zero ) cycle

         ! Determine a givens rotation which eliminates the j-th element of v.
         if( abs(v(n)) >= abs(v(j)) ) go to 20
            cotan = v(n)/v(j)
            sin = p5/sqrt(p25+p25*cotan**2)
            cos = sin*cotan
            tau = one
            if( abs(cos)*giant > one ) tau = one/cos
            go to 30
   20    continue
            tan = v(j)/v(n)
            cos = p5/sqrt(p25+p25*tan**2)
            sin = cos*tan
            tau = sin
   30    continue

         ! Apply the transformation to v and store the information
         ! necessary to recover the Givens rotation.
         v(n) = sin*v(j) + cos*v(n)
         v(j) = tau

         ! Apply the transformation to S and extend the spike in w.
         l = jj
         do i = j, m
            temp = cos*s(l) - sin*w(i)
            w(i) = sin*s(l) + cos*w(i)
            s(l) = temp
            l = l + 1
         end do
      end do
   70 continue

      ! Add the spike from the rank 1 update to w.
      do i = 1, m
         w(i) = w(i) + v(n)*u(i)
      end do

      ! Eliminate the spike.
      sing = .false.
      if( nm1 < 1 ) go to 140
      do j = 1, nm1
         if( w(j) == zero ) go to 120

         ! Determine a givens rotation which eliminates the
         ! j-th element of the spike.
         if( abs(s(jj)) >= abs(w(j)) ) go to 90
            cotan = s(jj)/w(j)
            sin = p5/sqrt(p25+p25*cotan**2)
            cos = sin*cotan
            tau = one
            if( abs(cos)*giant > one ) tau = one/cos
            go to 100
   90    continue
            tan = w(j)/s(jj)
            cos = p5/sqrt(p25+p25*tan**2)
            sin = cos*tan
            tau = sin
  100    continue

         ! Apply the transformation to S and reduce the spike in w.
         l = jj
         do i = j, m
            temp = cos*s(l) + sin*w(i)
            w(i) = -sin*s(l) + cos*w(i)
            s(l) = temp
            l = l + 1
         end do

         ! Store the information necessary to recover the Givens rotation.
         w(j) = tau
  120    continue

         ! Test for zero diagonal elements in the output S.
         if( s(jj) == zero ) sing = .true.
         jj = jj + (m - j + 1)
      end do
  140 continue

      ! Move w back into the last column of the output S.
      l = jj
      do i = n, m
         s(l) = w(i)
         l = l + 1
      end do
      if( s(jj) == zero ) sing = .true.

   end
