! f90 include file
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 13 Jul 2017
!_______________________________________________________________________
!
   subroutine qrsolv( n, r, ldr, ipvt, diag, qtb, x, sdiag, wa )

      integer :: n, ldr
      integer :: ipvt(n)
      double precision :: r(ldr,n), diag(n), qtb(n), x(n), sdiag(n), wa(n)
!     **********
!
!     subroutine qrsolv
!
!     Given an m by n matrix A, an n by n diagonal matrix D,
!     and an m-vector b, the problem is to determine an x which
!     solves the system
!
!           A*x = b ,     D*x = 0 ,
!
!     in the least squares sense.
!
!     This subroutine completes the solution of the problem
!     if it is provided with the necessary information from the
!     QR factorization, with column pivoting, of A. That is, if
!     A*P = Q*R, where P is a permutation matrix, Q has orthogonal
!     columns, and R is an upper triangular matrix with diagonal
!     elements of nonincreasing magnitude, then qrsolv expects
!     the full upper triangle of R, the permutation matrix P,
!     and the first n components of (Q transpose)*b. The system
!     A*x = b, D*x = 0, is then equivalent to
!
!                  t       t
!           r*z = q *b ,  p *d*p*z = 0 ,
!
!     where x = P*z. If this system does not have full rank,
!     then a least squares solution is obtained. On output qrsolv
!     also provides an upper triangular matrix S such that
!
!            t   t               t
!           p *(a *a + d*d)*p = s *s .
!
!     S is computed within qrsolv and may be of separate interest.
!
!     The subroutine statement is
!
!       subroutine qrsolv( n, r, ldr, ipvt, diag, qtb, x, sdiag, wa )
!
!     where
!
!       n is a positive integer input variable set to the order of R.
!
!       r is an n by n array. on input the full upper triangle
!         must contain the full upper triangle of the matrix R.
!         On output the full upper triangle is unaltered, and the
!         strict lower triangle contains the strict upper triangle
!         (transposed) of the upper triangular matrix S.
!
!       ldr is a positive integer input variable not less than n
!         which specifies the leading dimension of the array R.
!
!       ipvt is an integer input array of length n which defines the
!         permutation matrix P such that A*P = Q*R. Column j of P
!         is column ipvt(j) of the identity matrix.
!
!       diag is an input array of length n which must contain the
!         diagonal elements of the matrix D.
!
!       qtb is an input array of length n which must contain the first
!         n elements of the vector (Q transpose)*b.
!
!       x is an output array of length n which contains the least
!         squares solution of the system A*x = b, D*x = 0.
!
!       sdiag is an output array of length n which contains the
!         diagonal elements of the upper triangular matrix S.
!
!       wa is a work array of length n.
!
!     subprograms called
!
!       fortran-supplied ... abs, sqrt
!
!     Argonne National Laboratory. MINPACK project. March 1980.
!     Burton S. Garbow, Kenneth E. Hillstrom, Jorge J. Moré
!
!     **********
      integer :: i, j, jp1, k, kp1, l, nsing
      double precision :: cos, cotan, qtbpj, sin, sum, tan, temp
      double precision, parameter :: p5 = 0.5d0, p25 = 0.25d0, zero = 0.0d0

!--------- end of declarations -- execution starts hereafter  ------

      ! Copy R and (Q transpose)*b to preserve input and initialize S.
      ! in particular, save the diagonal elements of R in x.
      do j = 1, n
         do i = j, n
            r(i,j) = r(j,i)
         end do
         x(j) = r(j,j)
         wa(j) = qtb(j)
      end do

      ! Eliminate the diagonal matrix S using a givens rotation.
      do j = 1, n

         ! Prepare the row of S to be eliminated, locating the
         ! diagonal element using P from the QR factorization.
         l = ipvt(j)
         if( diag(l) == zero ) go to 90
         do k = j, n
            sdiag(k) = zero
         end do
         sdiag(j) = diag(l)

         ! The transformations to eliminate the row of D modify only
         ! a single element of (Q transpose)*b beyond the first n,
         ! which is initially zero.
         qtbpj = zero
         do k = j, n

            ! Determine a Givens rotation which eliminates the appropriate
            ! element in the current row of D.
            if( sdiag(k) == zero ) cycle
            if( abs(r(k,k)) >= abs(sdiag(k)) ) go to 40
               cotan = r(k,k)/sdiag(k)
               sin = p5/sqrt(p25+p25*cotan**2)
               cos = sin*cotan
               go to 50
   40       continue
               tan = sdiag(k)/r(k,k)
               cos = p5/sqrt(p25+p25*tan**2)
               sin = cos*tan
   50       continue

            ! Compute the modified diagonal element of R and the modified
            ! element of ((Q transpose)*b,0).
            r(k,k) = cos*r(k,k) + sin*sdiag(k)
            temp = cos*wa(k) + sin*qtbpj
            qtbpj = -sin*wa(k) + cos*qtbpj
            wa(k) = temp

            ! Accumulate the tranformation in the row of S.
            kp1 = k + 1
            if( n < kp1 ) cycle
            do i = kp1, n
               temp = cos*r(i,k) + sin*sdiag(i)
               sdiag(i) = -sin*r(i,k) + cos*sdiag(i)
               r(i,k) = temp
            end do
         end do
   90    continue

         ! Store the diagonal element of S and restore the corresponding
         ! diagonal element of R.
         sdiag(j) = r(j,j)
         r(j,j) = x(j)
      end do

      ! Solve the triangular system for z. If the system is singular,
      ! then obtain a least squares solution.
      nsing = n
      do j = 1, n
         if( sdiag(j) == zero .and. nsing == n ) nsing = j - 1
         if( nsing < n ) wa(j) = zero
      end do
      if( nsing < 1 ) go to 150
      do k = 1, nsing
         j = nsing - k + 1
         sum = zero
         jp1 = j + 1
         if( nsing < jp1 ) go to 130
         do i = jp1, nsing
            sum = sum + r(i,j)*wa(i)
         end do
  130    continue
         wa(j) = (wa(j) - sum)/sdiag(j)
      end do
  150 continue

      ! Permute the components of z back to components of x.
      do j = 1, n
         l = ipvt(j)
         x(l) = wa(j)
      end do

   end
