! f90 include file
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 14 Jul 2017
!_______________________________________________________________________
!
   subroutine qrfac( m, n, a, lda, pivot, ipvt, lipvt, rdiag, acnorm, wa )

      integer, intent(in) :: m, n, lda, lipvt
      integer, intent(out) :: ipvt(lipvt)
      logical, intent(in) :: pivot
      double precision, intent(in out) :: a(lda,n), wa(n)
      double precision, intent(out) :: rdiag(n), acnorm(n)
!     **********
!
!     subroutine qrfac
!
!     This subroutine uses Householder transformations with column
!     pivoting (optional) to compute a QR factorization of the
!     m by n matrix A. That is, qrfac determines an orthogonal
!     matrix Q, a permutation matrix P, and an upper trapezoidal
!     matrix R with diagonal elements of nonincreasing magnitude,
!     such that A*P = Q*R. The Householder transformation for
!     column k, k = 1,2,...,min(m,n), is of the form
!
!                           t
!           i - (1/u(k))*u*u
!
!     where u has zeros in the first k-1 positions. The form of
!     this transformation and the method of pivoting first
!     appeared in the corresponding LINPACK subroutine.
!
!     The subroutine statement is
!
!       subroutine qrfac( m, n, a, lda, pivot, ipvt, lipvt, rdiag, acnorm, wa )
!
!     where
!
!       m is a positive integer input variable set to the number
!         of rows of a.
!
!       n is a positive integer input variable set to the number
!         of columns of a.
!
!       A is an m by n array. On input A contains the matrix for
!         which the QR factorization is to be computed. On output
!         the strict upper trapezoidal part of A contains the strict
!         upper trapezoidal part of R, and the lower trapezoidal
!         part of A contains a factored form of Q (the non-trivial
!         elements of the u vectors described above).
!
!       lda is a positive integer input variable not less than m
!         which specifies the leading dimension of the array A.
!
!       pivot is a logical input variable. If pivot is set true,
!         then column pivoting is enforced. If pivot is set false,
!         then no column pivoting is done.
!
!       ipvt is an integer output array of length lipvt. ipvt
!         defines the permutation matrix P such that A*P = Q*R.
!         column j of p is column ipvt(j) of the identity matrix.
!         If pivot is false, ipvt is not referenced.
!
!       lipvt is a positive integer input variable. If pivot is false,
!         then lipvt may be as small as 1. If pivot is true, then
!         lipvt must be at least n.
!
!       rdiag is an output array of length n which contains the
!         diagonal elements of R.
!
!       acnorm is an output array of length n which contains the
!         norms of the corresponding columns of the input matrix A.
!         If this information is not needed, then acnorm can coincide
!         with rdiag.
!
!       wa is a work array of length n. If pivot is false, then wa
!         can coincide with rdiag.
!
!     subprograms called
!
!       minpack-supplied ... enorm
!
!       fortran-supplied ... max, sqrt, min
!
!     Argonne National Laboratory. MINPACK project. March 1980.
!     Burton S. Garbow, Kenneth E. Hillstrom, Jorge J. Moré
!
!     **********
      integer :: i, j, jp1, k, kmax, minmn
      double precision :: ajnorm, epsmch, sum, temp

      double precision, parameter :: one = 1.0d0, p05 = 0.05d0, zero = 0.0d0

!--------- end of declarations -- execution starts hereafter  ------

      ! epsmch is the machine precision.
      epsmch = epsilon(1.0d0)

      ! Compute the initial column norms and initialize several arrays.
      do j = 1, n
         acnorm(j) = enorm(m,a(1,j))
         rdiag(j) = acnorm(j)
         wa(j) = rdiag(j)
         if( pivot ) ipvt(j) = j
      end do

      ! Reduce a to r with Householder transformations.
      minmn = min(m,n)
      do j = 1, minmn
         if( .not. pivot ) go to 40

         ! Bring the column of largest norm into the pivot position.
         kmax = j
         do k = j, n
            if( rdiag(k) > rdiag(kmax) ) kmax = k
         end do
         if( kmax == j ) go to 40
         do i = 1, m
            temp = a(i,j)
            a(i,j) = a(i,kmax)
            a(i,kmax) = temp
         end do
         rdiag(kmax) = rdiag(j)
         wa(kmax) = wa(j)
         k = ipvt(j)
         ipvt(j) = ipvt(kmax)
         ipvt(kmax) = k
   40    continue

         ! Compute the Householder transformation to reduce the
         ! j-th column of a to a multiple of the j-th unit vector.
         ajnorm = enorm(m-j+1,a(j,j))
         if( ajnorm == zero ) go to 100
         if( a(j,j) < zero ) ajnorm = -ajnorm
         do i = j, m
            a(i,j) = a(i,j)/ajnorm
         end do
         a(j,j) = a(j,j) + one

         ! Apply the transformation to the remaining columns and update
         ! the norms.
         jp1 = j + 1
         if( n < jp1 ) go to 100
         do k = jp1, n
            sum = zero
            do i = j, m
               sum = sum + a(i,j)*a(i,k)
            end do
            temp = sum/a(j,j)
            do i = j, m
               a(i,k) = a(i,k) - temp*a(i,j)
            end do
            if( .not. pivot .or. rdiag(k) == zero ) cycle
            temp = a(j,k)/rdiag(k)
            rdiag(k) = rdiag(k)*sqrt(max(zero,one-temp**2))
            if( p05*(rdiag(k)/wa(k))**2 > epsmch ) cycle
            rdiag(k) = enorm(m-j,a(jp1,k))
            wa(k) = rdiag(k)
         end do
  100    continue
         rdiag(j) = -ajnorm
      end do

   end
