! f90 include file
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 13 Jul 2017
!_______________________________________________________________________
!
   subroutine lmpar( n, r, ldr, ipvt, diag, qtb, delta, par, x,         &
                     sdiag, wa1, wa2 )

      integer :: n, ldr
      integer :: ipvt(n)
      double precision :: delta, par
      double precision :: r(ldr,n), diag(n), qtb(n), x(n), sdiag(n),    &
                          wa1(n), wa2(n)
!     **********
!
!     subroutine lmpar
!
!     Given an m by n matrix A, an n by n nonsingular diagonal
!     matrix D, an m-vector b, and a positive number delta,
!     the problem is to determine a value for the parameter
!     par such that if x solves the system
!
!           A*x = b ,     sqrt(par)*D*x = 0 ,
!
!     in the least squares sense, and dxnorm is the euclidean
!     norm of D*x, then either par is zero and
!
!           (dxnorm-delta) <= 0.1*delta ,
!
!     or par is positive and
!
!           abs(dxnorm-delta) <= 0.1*delta .
!
!     this subroutine completes the solution of the problem
!     if it is provided with the necessary information from the
!     QR factorization, with column pivoting, of A. That is, if
!     A*P = Q*R, where P is a permutation matrix, Q has orthogonal
!     columns, and R is a upper triangular matrix with diagonal
!     elements of nonincreasing magnitude, then lmpar expects
!     the full upper triangle of R, the permutation matrix P,
!     and the first n components of (Q transpose)*b. On output
!     lmpar also provides an upper triangular matrix S such that
!
!            t   t                   t
!           P *(A *A + par*D*D)*P = S *S .
!
!     S is employed within lmpar and may be of separate interest.
!
!     Only a few iterations are generally needed for convergence
!     of the algorithm. If, however, the limit of 10 iterations
!     is reached, then the output par will contain the best
!     value obtained so far.
!
!     The subroutine statement is
!
!       subroutine lmpar( n, R, ldr, ipvt, diag, qtb, delta, par, x,
!                         sdiag, wa1, wa2 )
!
!     where
!
!       n is a positive integer input variable set to the order of R.
!
!       R is an n by n array. On input the full upper triangle
!         must contain the full upper triangle of the matrix R.
!         On output the full upper triangle is unaltered, and the
!         strict lower triangle contains the strict upper triangle
!         (transposed) of the upper triangular matrix S.
!
!       ldr is a positive integer input variable not less than n
!         which specifies the leading dimension of the array R.
!
!       ipvt is an integer input array of length n which defines the
!         permutation matrix P such that A*P = Q*R. Column j of P
!         is column ipvt(j) of the identity matrix.
!
!       diag is an input array of length n which must contain the
!         diagonal elements of the matrix D.
!
!       qtb is an input array of length n which must contain the first
!         n elements of the vector (Q transpose)*b.
!
!       delta is a positive input variable which specifies an upper
!         bound on the euclidean norm of D*x.
!
!       par is a nonnegative variable. On input par contains an
!         initial estimate of the Levenberg-Marquardt parameter.
!         On output par contains the final estimate.
!
!       x is an output array of length n which contains the least
!         squares solution of the system A*x = b, sqrt(par)*D*x = 0,
!         for the output par.
!
!       sdiag is an output array of length n which contains the
!         diagonal elements of the upper triangular matrix S.
!
!       wa1 and wa2 are work arrays of length n.
!
!     subprograms called
!
!       minpack-supplied ... enorm, qrsolv
!
!       fortran-supplied ... abs, max, min, sqrt
!
!     Argonne National Laboratory. MINPACK project. March 1980.
!     Burton S. Garbow, Kenneth E. Hillstrom, Jorge J. Moré
!
!     **********
      integer :: i, iter, j, jm1, jp1, k, l, nsing
      double precision :: dxnorm, dwarf, fp, gnorm, parc, parl, paru,   &
                          sum, temp

      double precision, parameter :: p1 = 0.1d0, p001 = 0.001d0, zero = 0.0d0

!--------- end of declarations -- execution starts hereafter  ------

      ! dwarf is the smallest positive magnitude.
      dwarf = tiny(1.0d0)

      ! Compute and store in x the gauss-newton direction. If the
      ! jacobian is rank-deficient, obtain a least squares solution.
      nsing = n
      do j = 1, n
         wa1(j) = qtb(j)
         if( r(j,j) == zero .and. nsing == n ) nsing = j - 1
         if( nsing < n ) wa1(j) = zero
      end do
      if( nsing >= 1 ) then
         do k = 1, nsing
            j = nsing - k + 1
            wa1(j) = wa1(j)/r(j,j)
            temp = wa1(j)
            jm1 = j - 1
            if( jm1 >= 1 ) then
               do i = 1, jm1
                  wa1(i) = wa1(i) - r(i,j)*temp
               end do
            end if
         end do
      end if
      do j = 1, n
         l = ipvt(j)
         x(l) = wa1(j)
      end do

      ! Initialize the iteration counter. Evaluate the function at the
      ! origin, and test for acceptance of the Gauss-Newton direction.
      iter = 0
      do j = 1, n
         wa2(j) = diag(j)*x(j)
      end do
      dxnorm = enorm(n,wa2)
      fp = dxnorm - delta
      if( fp <= p1*delta ) go to 99

      ! If the jacobian is not rank deficient, the Newton step provides
      ! a lower bound, parl, for the zero of the function. Otherwise set
      ! this bound to zero.
      parl = zero
      if( nsing >= n ) then
         do j = 1, n
            l = ipvt(j)
            wa1(j) = diag(l)*(wa2(l)/dxnorm)
         end do
         do j = 1, n
            sum = zero
            jm1 = j - 1
            if( jm1 >= 1 ) then
               do i = 1, jm1
                  sum = sum + r(i,j)*wa1(i)
               end do
            end if
            wa1(j) = (wa1(j) - sum)/r(j,j)
         end do
         temp = enorm(n,wa1)
         parl = ((fp/delta)/temp)/temp
      end if

      ! Calculate an upper bound, paru, for the zero of the function.
      do j = 1, n
         sum = zero
         do i = 1, j
            sum = sum + r(i,j)*qtb(i)
         end do
         l = ipvt(j)
         wa1(j) = sum/diag(l)
      end do
      gnorm = enorm(n,wa1)
      paru = gnorm/delta
      if( paru == zero ) paru = dwarf/min(delta,p1)

      ! If the input par lies outside of the interval (parl,paru),
      ! set par to the closer endpoint.
      par = max(par,parl)
      par = min(par,paru)
      if( par == zero ) par = gnorm/dxnorm

      ! Beginning of an iteration.
      do
         iter = iter + 1

         ! Evaluate the function at the current value of par.
         if( par == zero ) par = max(dwarf,p001*paru)
         temp = sqrt(par)
         do j = 1, n
            wa1(j) = temp*diag(j)
         end do
         call qrsolv(n,r,ldr,ipvt,wa1,qtb,x,sdiag,wa2)
         do j = 1, n
            wa2(j) = diag(j)*x(j)
         end do
         dxnorm = enorm(n,wa2)
         temp = fp
         fp = dxnorm - delta

         ! If the function is small enough, accept the current value
         ! of par. also test for the exceptional cases where parl
         ! is zero or the number of iterations has reached 10.

         if( abs(fp) <= p1*delta .or. parl == zero .and. fp <= temp     &
             .and. temp < zero .or. iter == 10 ) exit

         ! Compute the newton correction.
         do j = 1, n
            l = ipvt(j)
            wa1(j) = diag(l)*(wa2(l)/dxnorm)
         end do
         do j = 1, n
            wa1(j) = wa1(j)/sdiag(j)
            temp = wa1(j)
            jp1 = j + 1
            if( n >= jp1 ) then
               do i = jp1, n
                  wa1(i) = wa1(i) - r(i,j)*temp
               end do
            end if
         end do
         temp = enorm(n,wa1)
         parc = ((fp/delta)/temp)/temp

         ! Depending on the sign of the function, update parl or paru.
         if( fp > zero ) parl = max(parl,par)
         if( fp < zero ) paru = min(paru,par)

         ! Compute an improved estimate for par.
         par = max(parl,par+parc)

      end do

 99   continue

      if( iter == 0 ) par = zero

   end
