! f90 include file
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 13 Jul 2017
!
   subroutine lmdif_funfit( fcn, m, n, x, fvec, ftol, xtol, gtol,       &
                            maxfev, epsfcn, diag, mode, factor, nprint, &
                            info, nfev, fjac, ldfjac, ipvt, qtf,        &
                            wa1, wa2, wa3, wa4, user_fun )

      use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe

      integer :: m, n, maxfev, mode, nprint, info, nfev, ldfjac
      integer :: ipvt(n)
      double precision :: ftol, xtol, gtol, epsfcn, factor
      double precision :: x(n), fvec(m), diag(n), fjac(ldfjac,n),       &
                          qtf(n), wa1(n), wa2(n), wa3(n), wa4(m)
      interface
         subroutine fcn( m, n, x, fvec, user_fun )
            integer,          intent(in) :: m, n
            double precision, intent(in) :: x(n)
            double precision             :: fvec(m)
            interface
               function user_fun( x, p, n ) result( res )
                  double precision, intent(in) :: x, p(n)
                  integer,          intent(in) :: n
                  double precision             :: res
               end function
            end interface
         end subroutine
         function user_fun( x, p, n ) result( res )
            double precision, intent(in) :: x, p(n)
            integer,          intent(in) :: n
            double precision             :: res
         end function
      end interface
!     ******************************************************************
! EC 2012-03-23: the 'user_fun' argument has been added.
!                'user_fun' is the modelling function; its interface
!                is described below.
!                At a top level, the user only provides 'user_fun',
!                whereas 'fcn' is defined in MUESLI (module mod_datafun)
!     ******************************************************************
!
!     subroutine lmdif_funfit
!
!     The purpose of lmdif_funfit is to minimize the sum of the squares
!     of m nonlinear functions in n variables by a modification of
!     the Levenberg-Marquardt algorithm. The user must provide a
!     subroutine which calculates the functions. The jacobian is
!     then calculated by a forward-difference approximation.
!
!     The subroutine statement is
!
!       subroutine lmdif_funfit( fcn, m, n, x, fvec, ftol, xtol, gtol,
!                            maxfev, epsfcn, diag, mode, factor, nprint,
!                            info, nfev, fjac, ldfjac, ipvt, qtf,
!                            wa1, wa2, wa3, wa4, user_fun )
!
!     where
!
!       fcn is the name of a module subroutine defined by the Muesli
!         library (cf. module mod_datafun)
!
!       user_fun is the modelling user-function, as follows:
!         (n is the number of variables to be determined, p(n) a
!          vector containing these variables, xx the independant
!          variable) See the interface in the header.
!
!       m is a positive integer input variable set to the number
!         of functions.
!
!       n is a positive integer input variable set to the number
!         of variables. n must not exceed m.
!
!       x is an array of length n. On input x must contain
!         an initial estimate of the solution vector. On output x
!         contains the final estimate of the solution vector.
!
!       fvec is an output array of length m which contains
!         the functions evaluated at the output x.
!
!       ftol is a nonnegative input variable. Termination
!         occurs when both the actual and predicted relative
!         reductions in the sum of squares are at most ftol.
!         Therefore, ftol measures the relative error desired
!         in the sum of squares.
!
!       xtol is a nonnegative input variable. Termination
!         occurs when the relative error between two consecutive
!         iterates is at most xtol. Therefore, xtol measures the
!         relative error desired in the approximate solution.
!
!       gtol is a nonnegative input variable. Termination
!         occurs when the cosine of the angle between fvec and
!         any column of the jacobian is at most gtol in absolute
!         value. Therefore, gtol measures the orthogonality
!         desired between the function vector and the columns
!         of the jacobian.
!
!       maxfev is a positive integer input variable. Termination
!         occurs when the number of calls to fcn is at least
!         maxfev by the end of an iteration.
!
!       epsfcn is an input variable used in determining a suitable
!         step length for the forward-difference approximation. This
!         approximation assumes that the relative errors in the
!         functions are of the order of epsfcn. If epsfcn is less
!         than the machine precision, it is assumed that the relative
!         errors in the functions are of the order of the machine
!         precision.
!
!       diag is an array of length n. If mode = 1 (see
!         below), diag is internally set. If mode = 2, diag
!         must contain positive entries that serve as
!         multiplicative scale factors for the variables.
!
!       mode is an integer input variable. If mode = 1, the
!         variables will be scaled internally. If mode = 2,
!         the scaling is specified by the input diag. Other
!         values of mode are equivalent to mode = 1.
!
!       factor is a positive input variable used in determining the
!         initial step bound. This bound is set to the product of
!         factor and the euclidean norm of diag*x if nonzero, or else
!         to factor itself. In most cases factor should lie in the
!         interval (.1,100.). 100. is a generally recommended value.
!
!       nprint is an integer input variable that enables controlled
!         printing of iterates if it is positive. If nprint is not
!         positive, no special calls of fcn are made.
!
!       info is an integer output variable. If the user has
!         terminated execution, info is set to the (negative)
!         value of iflag. See description of fcn. Otherwise,
!         info is set as follows.
!
!         info = 0  improper input parameters.
!
!         info = 1  both actual and predicted relative reductions
!                   in the sum of squares are at most ftol.
!
!         info = 2  relative error between two consecutive iterates
!                   is at most xtol.
!
!         info = 3  conditions for info = 1 and info = 2 both hold.
!
!         info = 4  the cosine of the angle between fvec and any
!                   column of the jacobian is at most gtol in
!                   absolute value.
!
!         info = 5  number of calls to fcn has reached or
!                   exceeded maxfev.
!
!         info = 6  ftol is too small. No further reduction in
!                   the sum of squares is possible.
!
!         info = 7  xtol is too small. No further improvement in
!                   the approximate solution x is possible.
!
!         info = 8  gtol is too small. fvec is orthogonal to the
!                   columns of the jacobian to machine precision.
!
!       nfev is an integer output variable set to the number of
!         calls to fcn.
!
!       fjac is an output m by n array. The upper n by n submatrix
!         of fjac contains an upper triangular matrix r with
!         diagonal elements of nonincreasing magnitude such that
!
!                t     t           t
!               p *(jac *jac)*p = r *r,
!
!         where p is a permutation matrix and jac is the final
!         calculated jacobian. Column j of p is column ipvt(j)
!         (see below) of the identity matrix. The lower trapezoidal
!         part of fjac contains information generated during
!         the computation of r.
!
!       ldfjac is a positive integer input variable not less than m
!         which specifies the leading dimension of the array fjac.
!
!       ipvt is an integer output array of length n. ipvt
!         defines a permutation matrix p such that jac*p = q*r,
!         where jac is the final calculated jacobian, q is
!         orthogonal (not stored), and r is upper triangular
!         with diagonal elements of nonincreasing magnitude.
!         column j of p is column ipvt(j) of the identity matrix.
!
!       qtf is an output array of length n which contains
!         the first n elements of the vector (q transpose)*fvec.
!
!       wa1, wa2, and wa3 are work arrays of length n.
!
!       wa4 is a work array of length m.
!
!     subprograms called
!
!       user-supplied ...... fcn
!
!       minpack-supplied ... enorm, fdjac2, lmpar, qrfac
!
!       fortran-supplied ... abs, max, min, sqrt, mod
!
!     Argonne National Laboratory. MINPACK project. March 1980.
!     Burton S. Garbow, Kenneth E. Hillstrom, Jorge J. Moré
!
!     **********
      integer :: i, iflag, iter, j, l
      double precision :: actred, delta, dirder, epsmch, fnorm, fnorm1, &
                          gnorm, par, pnorm, prered, ratio, sum, temp,  &
                          temp1, temp2, xnorm

      double precision, parameter :: one = 1.0d0, p1 = 0.1d0, p5 = 0.5d0, &
         p25 = 0.25d0, p75 = 0.75d0, p0001 = 0.0001d0, zero = 0.0d0

!--------- end of declarations -- execution starts hereafter  ------

      ! epsmch is the machine precision.
      epsmch = epsilon(1.0d0)

      info = 0
      iflag = 0
      nfev = 0

      ! Check the input parameters for errors.
      if( n <= 0 .or. m < n .or. ldfjac < m .or. ftol < zero .or.       &
          xtol < zero .or. gtol < zero .or. maxfev <= 0 .or.            &
          factor <= zero ) go to 300
      if( mode /= 2 ) go to 20
      do j = 1, n
         if( diag(j) <= zero ) go to 300
      end do
   20 continue

      ! Evaluate the function at the starting point and calculate its norm.
      call mf_restore_fpe( )
      call fcn( m, n, x, fvec, user_fun )
      call mf_save_and_disable_fpe( )

      nfev = 1
      fnorm = enorm(m,fvec)

      ! Initialize levenberg-marquardt parameter and iteration counter.
      par = zero
      iter = 1

      ! Beginning of the outer loop.
   30 continue

         ! Calculate the jacobian matrix.
         iflag = 2
         call fdjac2_funfit( fcn, m, n, x, fvec, fjac, ldfjac,          &
                             epsfcn, wa4, user_fun )
         nfev = nfev + n
         if( iflag < 0 ) go to 300

         ! If requested, call fcn to enable printing of iterates.
         if( nprint <= 0 ) go to 40
         if( mod(iter-1,nprint) == 0 ) then

            call mf_restore_fpe( )
            call fcn( m, n, x, fvec, user_fun )
            call mf_save_and_disable_fpe( )

         end if
   40    continue

         ! Compute the qr factorization of the jacobian.
         call qrfac(m,n,fjac,ldfjac,.true.,ipvt,n,wa1,wa2,wa3)

         ! On the first iteration and if mode is 1, scale according
         ! to the norms of the columns of the initial jacobian.
         if( iter /= 1 ) go to 80
         if( mode == 2 ) go to 60
         do j = 1, n
            diag(j) = wa2(j)
            if( wa2(j) == zero ) diag(j) = one
         end do
   60    continue

         ! On the first iteration, calculate the norm of the scaled x
         ! and initialize the step bound delta.
         do j = 1, n
            wa3(j) = diag(j)*x(j)
         end do
         xnorm = enorm(n,wa3)
         delta = factor*xnorm
         if( delta == zero ) delta = factor
   80    continue

         ! Form (Q transpose)*fvec and store the first n components in qtf.
         do i = 1, m
            wa4(i) = fvec(i)
         end do
         do j = 1, n
            if( fjac(j,j) == zero ) go to 120
            sum = zero
            do i = j, m
               sum = sum + fjac(i,j)*wa4(i)
            end do
            temp = -sum/fjac(j,j)
            do i = j, m
               wa4(i) = wa4(i) + fjac(i,j)*temp
            end do
  120       continue
            fjac(j,j) = wa1(j)
            qtf(j) = wa4(j)
         end do

         ! Compute the norm of the scaled gradient.
         gnorm = zero
         if( fnorm == zero ) go to 170
         do j = 1, n
            l = ipvt(j)
            if( wa2(l) == zero ) go to 150
            sum = zero
            do i = 1, j
               sum = sum + fjac(i,j)*(qtf(i)/fnorm)
            end do
            gnorm = max(gnorm,abs(sum/wa2(l)))
  150       continue
         end do
  170    continue

         ! Test for convergence of the gradient norm.
         if( gnorm <= gtol ) info = 4
         if( info /= 0 ) go to 300

         ! Rescale if necessary.
         if( mode == 2 ) go to 190
         do j = 1, n
            diag(j) = max(diag(j),wa2(j))
         end do
  190    continue

         ! Beginning of the inner loop.
  200    continue

            ! Determine the levenberg-marquardt parameter.
            call lmpar( n, fjac, ldfjac, ipvt, diag, qtf, delta, par,   &
                        wa1, wa2, wa3, wa4 )

            ! Store the direction p and x + p. calculate the norm of p.
            do j = 1, n
               wa1(j) = -wa1(j)
               wa2(j) = x(j) + wa1(j)
               wa3(j) = diag(j)*wa1(j)
            end do
            pnorm = enorm(n,wa3)

            ! On the first iteration, adjust the initial step bound.
            if( iter == 1 ) delta = min(delta,pnorm)

            ! Evaluate the function at x + p and calculate its norm.
            call mf_restore_fpe( )
            call fcn( m, n, wa2, wa4, user_fun )
            call mf_save_and_disable_fpe( )

            nfev = nfev + 1
            fnorm1 = enorm(m,wa4)

            ! Compute the scaled actual reduction.
            actred = -one
            if( p1*fnorm1 < fnorm ) actred = one - (fnorm1/fnorm)**2

            ! Compute the scaled predicted reduction and the scaled
            ! directional derivative.
            do j = 1, n
               wa3(j) = zero
               l = ipvt(j)
               temp = wa1(l)
               do i = 1, j
                  wa3(i) = wa3(i) + fjac(i,j)*temp
               end do
            end do
            temp1 = enorm(n,wa3)/fnorm
            temp2 = (sqrt(par)*pnorm)/fnorm
            prered = temp1**2 + temp2**2/p5
            dirder = -(temp1**2 + temp2**2)

            ! Compute the ratio of the actual to the predicted reduction.
            ratio = zero
            if( prered /= zero ) ratio = actred/prered

            ! Update the step bound.
            if( ratio > p25 ) go to 240
               if( actred >= zero) temp = p5
               if( actred < zero) temp = p5*dirder/(dirder + p5*actred)
               if( p1*fnorm1 >= fnorm .or. temp < p1) temp = p1
               delta = temp*min(delta,pnorm/p1)
               par = par/temp
               go to 260
  240       continue
               if( par /= zero .and. ratio < p75 ) go to 250
               delta = pnorm/p5
               par = p5*par
  250          continue
  260       continue

            ! Test for successful iteration.
            if( ratio < p0001 ) go to 290

            ! Successful iteration. update x, fvec, and their norms.
            do j = 1, n
               x(j) = wa2(j)
               wa2(j) = diag(j)*x(j)
            end do
            do i = 1, m
               fvec(i) = wa4(i)
            end do
            xnorm = enorm(n,wa2)
            fnorm = fnorm1
            iter = iter + 1
  290       continue

            ! Tests for convergence.
            if( abs(actred) <= ftol .and. prered <= ftol                &
                .and. p5*ratio <= one ) info = 1
            if( delta <= xtol*xnorm ) info = 2
            if( abs(actred) <= ftol .and. prered <= ftol                &
                .and. p5*ratio <= one .and. info == 2 ) info = 3
            if( info /= 0 ) go to 300

            ! Tests for termination and stringent tolerances.
            if( nfev >= maxfev ) info = 5
            if( abs(actred) <= epsmch .and. prered <= epsmch            &
                .and. p5*ratio <= one ) info = 6
            if( delta <= epsmch*xnorm ) info = 7
            if( gnorm <= epsmch ) info = 8
            if( info /= 0 ) go to 300

            ! End of the inner loop. repeat if iteration unsuccessful.
            if( ratio < p0001 ) go to 200

            ! End of the outer loop.
         go to 30
  300 continue

      ! Termination, either normal or user imposed.
      if( iflag < 0 ) info = iflag
      if( nprint > 0 ) then

         call mf_restore_fpe( )
         call fcn( m, n, x, fvec, user_fun )
         call mf_save_and_disable_fpe( )

      end if

   end
