! f90 include file
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 17 Oct 2017
!_______________________________________________________________________
!
   subroutine hybrj_sp( fcn, jac_sp, n, x, fvec, xtol, ftol, maxfev,    &
                        info, nfev, njev, max_iter, print, format )

      use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe
      use mod_sparse

      interface
         subroutine fcn( n, x, fvec, flag )
            integer,          intent(in) :: n
            double precision, intent(in) :: x(n)
            double precision             :: fvec(n)
            integer                      :: flag
         end subroutine fcn
         subroutine jac_sp( n, x, job, pd, ipd, jpd, nnz )
            integer,          intent(in) :: n, job
            double precision, intent(in) :: x(n)
            double precision             :: pd(*)
            integer                      :: ipd(*), jpd(*), nnz
         end subroutine jac_sp
      end interface
      integer,          intent(in)  :: n, maxfev, max_iter
      double precision, intent(in)  :: xtol, ftol
      logical,          intent(in)  :: print
      character(len=*), intent(in)  :: format

      double precision :: x(n)

      integer,          intent(out) :: info, nfev, njev
      double precision, intent(out) :: fvec(n)

!     subroutine hybrj_sp (sparse version of hybrj_mf
!
!     The purpose of hybrj_sp is to find a zero of a system of
!     n nonlinear functions in n variables by a modification
!     of the Powell hybrid method. The user must provide two
!     subroutines which calculate the functions and the jacobian.
!
!     The subroutine statement is
!
!       subroutine hybrj_sp( fcn, jac_sp, n, x, fvec, xtol, ftol, maxfev,
!                            info, nfev, njev, max_iter, print, format )
!
!     where
!
!       fcn is the name of the user-supplied subroutine which
!         calculates the functions. See the interface in the header.
!         The value of flag should not be changed by fcn unless
!         the user wants to terminate execution of hybrj_mf; in this
!         case, set flag to a negative integer.
!
!       jac_sp is the name of the user-supplied subroutine which
!         calculates the jacobian using a sparse storage (CSC).
!         See the interface in the header.
!
!       n is a positive integer input variable set to the number
!         of functions and variables.
!
!       x is an array of length n. On input x must contain
!         an initial estimate of the solution vector. On output x
!         contains the final estimate of the solution vector.
!
!       fvec is an output array of length n which contains
!         the functions evaluated at the output x.
!
!       xtol is a nonnegative input variable. Termination
!         occurs when the relative error between two consecutive
!         iterates is at most xtol.
!
!       ftol is a nonnegative input variable. Termination
!         occurs when the norm of fvec is less or equal to ftol.
!
!       maxfev is a positive integer input variable. Termination
!         occurs when the number of calls to fcn is at least maxfev
!         by the end of an iteration.
!
!       factor is a positive input variable used in determining the
!         initial step bound. This bound is set to the product of
!         factor and the euclidean norm of diag*x if nonzero, or else
!         to factor itself. In most cases factor should lie in the
!         interval (0.1,100). 100 is a generally recommended value.
!
!       info is an integer output variable. If the user has
!         terminated execution, info is set to the (negative)
!         value of flag. See description of fcn. Otherwise,
!         info is set as follows.
!
!         info = 0   improper input parameters.
!
!         info = 1   relative error between two consecutive iterates
!                    is at most xtol.
!
!         info = 2   number of calls to fcn has reached or exceeded
!                    maxfev.
!
!         info = 3   xtol is too small. no further improvement in
!                    the approximate solution x is possible.
!
!         info = 4   iteration is not making good progress, as
!                    measured by the improvement from the last
!                    five jacobian evaluations.
!
!         info = 5   iteration is not making good progress, as
!                    measured by the improvement from the last
!                    ten iterations.
!
!         info = 6   function norm of fvec is at most ftol.
!
!       nfev is an integer output variable set to the number of
!         calls to fcn.
!
!       njev is an integer output variable set to the number of
!         calls to jac.
!
!       max_iter (integer) is the maximum number of iteration required
!         by the user.
!
!       print (logical)
!         if this variable is set to TRUE, then some information is written
!         on the screen at each iteration
!
!       format (character string)
!         fortran format used for writing information at each iteration
!
!     **********
      integer :: i, flag, iter, j, jm1, l, ncfail, ncsuc,               &
                 nslow1, nslow2, iwa(1)
      logical :: jeval, sing
      double precision :: actred, delta, epsmch, fnorm, fnorm1,         &
                          pnorm, prered, ratio, sum, temp, xnorm

      double precision, parameter :: one = 1.0d0, p1 = 0.1d0, p5 = 0.5d0, &
                          p001 = 0.001d0, p0001 = 0.0001d0, zero = 0.0d0, &
                          factor = 100.0d0

      integer :: nnz
      double precision, allocatable :: pd(:), jcn(:), diag(:)
      integer, allocatable :: ipd(:), jpd(:)
      double precision, allocatable :: wa1(:), wa2(:), wa3(:), wa4(:)

!--------- end of declarations -- execution starts hereafter  ------

      ! get only nnz, then allocate arrays for the CSC sparse jacobian
      ! (don't increase njev, because the call is very quick)
      call jac_sp( n, x, 0, pd, ipd, jpd, nnz )
      allocate( pd(nnz), ipd(nnz), jpd(n+1) )

      ! epsmch is the machine precision.
      epsmch = epsilon(1.0d0)

      info = 0
      flag = 0
      nfev = 0
      njev = 0

      ! Check the input parameters for errors.
      if( maxfev <= 0 .or. factor <= zero ) then
         go to 99
      end if

      ! Evaluate the function at the starting point and calculate its norm.
      flag = 0

      call mf_restore_fpe( )
      call fcn( n, x, fvec, flag )
      call mf_save_and_disable_fpe( )

      nfev = 1

      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. flag >= 0 ) then
         do i = 1, n
            if( isnan(fvec(i)) ) then
               print "(/,A)", "(MUESLI LsqNonLin:) [hybrj_sp] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied FCN routine."
               print "(20X,A,I0)", "This occured in fvec(i) for i = ", i
               call muesli_trace( pause="yes" )
               info = 14
               return
            end if
         end do
      end if

      if( flag < 0 ) go to 99
      fnorm = enorm(n,fvec)

      ! Initialize iteration counter and monitors.
      iter = 1
      ncsuc = 0
      ncfail = 0
      nslow1 = 0
      nslow2 = 0
      if( print ) then
         print format, iter, x(1:n), fnorm
      end if

      allocate( wa1(n), wa2(n), wa3(n), wa4(n) )
      allocate( jcn(n), diag(n) )

      outer:do

         jeval = .true.

         ! Calculate the jacobian sparse matrix.
         call mf_restore_fpe( )
         call jac_sp( n, x, 1, pd, ipd, jpd, nnz )
         call mf_save_and_disable_fpe( )
         njev = njev + 1

         ! Get column norms, for using them as scaling factors.
         call normcols( n, pd, ipd, jpd, jcn )

         if( iter == 1 ) then
            ! On the first iteration, scale according to the norms of
            ! the columns of the initial jacobian.
            do j = 1, n
               if( jcn(j) /= zero ) then
                  diag(j) = jcn(j)
               else
                  diag(j) = one
               end if
            end do

            ! On the first iteration, calculate the norm of the scaled x
            ! and initialize the step bound delta.
            do j = 1, n
               wa3(j) = diag(j)*x(j)
            end do
            xnorm = enorm(n,wa3)
            delta = factor*xnorm
            if( delta == zero ) delta = factor
         else
            do j = 1, n
               diag(j) = max(0.1d0*diag(j),jcn(j))
            end do
         end if

         inner:do

            ! Determine the direction p.
            call dogleg_sp( n, pd, ipd, jpd, diag, fvec, delta, wa1 )

            ! Store the direction p and x + p. Calculate the norm of p.
            do j = 1, n
               wa1(j) = -wa1(j)        ! p
               wa2(j) = x(j) + wa1(j)  ! x + p
               wa3(j) = diag(j)*wa1(j)
            end do
            pnorm = enorm(n,wa3)

            ! On the first iteration, adjust the initial step bound.
            if( iter == 1 ) delta = min(delta,pnorm)

            ! Evaluate the function at x + p and calculate its norm.
            flag = 0

            call mf_restore_fpe( )
            call fcn( n, wa2, wa4, flag )     ! wa4: fvec (new)
            call mf_save_and_disable_fpe( )
      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. flag >= 0 ) then
         do i = 1, n
            if( isnan(wa4(i)) ) then
               print "(/,A)", "(MUESLI LsqNonLin:) [hybrj_sp] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied FCN routine."
               print "(20X,A,I0)", "This occured in fvec(i) for i = ", i
               call muesli_trace( pause="yes" )
               info = 14
               return
            end if
         end do
      end if

            nfev = nfev + 1

            if( flag < 0 ) exit outer

            fnorm1 = enorm(n,wa4)

            ! Compute the scaled actual reduction.
            actred = -one
            if( fnorm1 < fnorm ) actred = one - (fnorm1/fnorm)**2

            ! Compute fvec + fjac*p   ! (old fvec, not new one)
            call amux( n, wa1, wa3, pd, ipd, jpd ) ! x: wa1, y: wa3
            wa3(1:n) = fvec(1:n) + wa3(1:n)
            ! Compute the scaled predicted reduction.
            temp = enorm(n,wa3)
            prered = zero
            if( temp < fnorm ) prered = one - (temp/fnorm)**2

            ! Compute the ratio of the actual to the predicted reduction.
            ratio = zero
            if( prered > zero ) ratio = actred/prered

            ! Update the step bound.
            if( ratio < p1 ) then
               ncsuc = 0
               ncfail = ncfail + 1
               delta = p5*delta
            else
               ncfail = 0
               ncsuc = ncsuc + 1
               if( ratio >= p5 .or. ncsuc > 1 ) delta = max(delta,pnorm/p5)
               if( abs(ratio-one) <= p1 ) delta = pnorm/p5
            end if

            ! Test for successful iteration.
            if( ratio >= p0001 ) then

               ! Successful iteration. Update x, fvec, and their norms.
               do j = 1, n
                  x(j) = wa2(j)
                  wa2(j) = diag(j)*x(j)
                  fvec(j) = wa4(j)
               end do
               xnorm = enorm(n,wa2)
               fnorm = fnorm1
               iter = iter + 1
               if( iter > max_iter ) then
                  print *
                  print *, "(MUESLI:) FSolve: internal error"
                  print *, "          in hybrd_sp(), line 291: iter > max_iter"
                  print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
                  print *
                  print *, "  pause for debugging purpose"
                  print *, "  [RETURN] to resume..."
                  read *
                  stop
               end if
               if( print ) then
                  print format, iter, x(1:n), fnorm
               end if

            end if

            ! Determine the progress of the iteration.
            nslow1 = nslow1 + 1
            if( actred >= p001 ) nslow1 = 0
            if( jeval ) nslow2 = nslow2 + 1
            if( actred >= p1 ) nslow2 = 0

            ! Test for convergence.
            ! (modified the fnorm test: previously, it was exactly zero,
            !  now it is the same as in octave-4.2.1)
            if( delta <= xtol*xnorm ) info = 1
            if( fnorm <= ftol*n*xnorm/25 ) info = 6
            if( info /= 0 ) exit outer

            ! Tests for termination and stringent tolerances.
            if( nfev >= maxfev ) info = 2
            if( p1*max(p1*delta,pnorm) <= epsmch*xnorm ) info = 3
            if( nslow2 == 10 ) info = 4 ! max was 5 in the original code
            if( nslow1 == 20 ) info = 5 ! max was 10 in the original code
            if( info /= 0 ) exit outer

            ! Criterion for recalculating jacobian.
            if( ncfail == 2 ) exit inner

            jeval = .false.
         end do inner

      end do outer

   99 continue

      ! Termination, either normal or user imposed.
      if( flag < 0 ) info = flag

   end
