! f90 include file
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 17 Oct 2017
!_______________________________________________________________________
!
   subroutine hybrd( fcn, n, x, fvec, xtol, ftol, maxfev, ml, mu,       &
                     epsfcn, diag, mode, factor, info, nfev,            &
                     fjac, r, lr, qtf, wa1, wa2, wa3, wa4,              &
                     max_iter, print, format )

      use mod_mfdebug, only: MF_NUMERICAL_CHECK, muesli_trace
      use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe

      interface
         subroutine fcn( n, x, fvec, flag )
            integer,          intent(in) :: n
            double precision, intent(in) :: x(n)
            double precision             :: fvec(n)
            integer                      :: flag
         end subroutine fcn
      end interface
      integer,          intent(in)  :: n, maxfev, ml, mu, mode, lr,     &
                                       max_iter
      double precision, intent(in)  :: xtol, ftol, factor, epsfcn
      logical,          intent(in)  :: print
      character(len=*), intent(in)  :: format

      double precision              :: x(n), diag(n),                   &
                                       wa1(n), wa2(n), wa3(n), wa4(n)

      integer,          intent(out) :: info, nfev
      double precision, intent(out) :: fvec(n), r(lr),                  &
                                       fjac(n,n), qtf(n)

!     **********
!
!     subroutine hybrd
!
!     The purpose of hybrd is to find a zero of a system of
!     n nonlinear functions in n variables by a modification
!     of the Powell hybrid method. The user must provide a
!     subroutine which calculates the functions. The jacobian is
!     then calculated by a forward-difference approximation.
!
!     The subroutine statement is
!
!       subroutine hybrd( fcn, n, x, fvec, xtol, ftol, maxfev, ml, mu,
!                         epsfcn, diag, mode, factor, info, nfev,
!                         fjac, r, lr, qtf, wa1, wa2, wa3, wa4,
!                         max_iter, print, format )
!
!     where
!
!       fcn is the name of the user-supplied subroutine which
!         calculates the functions. See the interface in the header.
!         The value of flag should not be changed by fcn unless
!         the user wants to terminate execution of hybrd; in this
!         case, set flag to a negative integer.
!
!       n is a positive integer input variable set to the number
!         of functions and variables.
!
!       x is an array of length n. On input x must contain
!         an initial estimate of the solution vector. On output x
!         contains the final estimate of the solution vector.
!
!       fvec is an output array of length n which contains
!         the functions evaluated at the output x.
!
!       xtol is a nonnegative input variable. Termination
!         occurs when the relative error between two consecutive
!         iterates is at most xtol.
!
!       ftol is a nonnegative input variable. Termination
!         occurs when the norm of fvec is less or equal to ftol.
!
!       maxfev is a positive integer input variable. Termination
!         occurs when the number of calls to fcn is at least maxfev
!         by the end of an iteration.
!
!       ml is a nonnegative integer input variable which specifies
!         the number of subdiagonals within the band of the
!         jacobian matrix. If the jacobian is not banded, set
!         ml to at least n - 1.
!
!       mu is a nonnegative integer input variable which specifies
!         the number of superdiagonals within the band of the
!         jacobian matrix. If the jacobian is not banded, set
!         mu to at least n - 1.
!
!       epsfcn is an input variable used in determining a suitable
!         step length for the forward-difference approximation. This
!         approximation assumes that the relative errors in the
!         functions are of the order of epsfcn. If epsfcn is less
!         than the machine precision, it is assumed that the relative
!         errors in the functions are of the order of the machine
!         precision.
!
!       diag is an array of length n. If mode = 1 (see
!         below), diag is internally set. If mode = 2, diag
!         must contain positive entries that serve as
!         multiplicative scale factors for the variables.
!
!       mode is an integer input variable. If mode = 1, the
!         variables will be scaled internally. If mode = 2,
!         the scaling is specified by the input diag. Other
!         values of mode are equivalent to mode = 1.
!
!       factor is a positive input variable used in determining the
!         initial step bound. This bound is set to the product of
!         factor and the euclidean norm of diag*x if nonzero, or else
!         to factor itself. In most cases factor should lie in the
!         interval (.1,100.). 100. is a generally recommended value.
!
!       info is an integer output variable. If the user has
!         terminated execution, info is set to the (negative)
!         value of flag (see description of fcn); otherwise,
!         info is set as follows.
!
!         info = 0   improper input parameters.
!
!         info = 1   relative error between two consecutive iterates
!                    is at most xtol.
!
!         info = 2   number of calls to fcn has reached or exceeded
!                    maxfev.
!
!         info = 3   xtol is too small. No further improvement in
!                    the approximate solution x is possible.
!
!         info = 4   iteration is not making good progress, as
!                    measured by the improvement from the last
!                    five jacobian evaluations.
!
!         info = 5   iteration is not making good progress, as
!                    measured by the improvement from the last
!                    ten iterations.
!
!         info = 6   function norm of fvec is at most ftol.
!
!       nfev is an integer output variable set to the number of
!         calls to fcn.
!
!       fjac is an output n by n array which contains the
!         orthogonal matrix Q produced by the QR factorization
!         of the final approximate jacobian.
!
!       r is an output array of length lr which contains the
!         upper triangular matrix produced by the QR factorization
!         of the final approximate jacobian, stored rowwise.
!
!       lr is a positive integer input variable not less than
!         (n*(n+1))/2.
!
!       qtf is an output array of length n which contains
!         the vector (Q transpose)*fvec.
!
!       wa1, wa2, wa3, and wa4 are work arrays of length n.
!
!       print (logical)
!         if this variable is set to TRUE, then some information is written
!         on the screen at each iteration
!
!       format (character string)
!         fortran format used for writing information at each iteration
!
!     subprograms called
!
!       user-supplied ...... fcn
!
!       minpack-supplied ... dogleg, enorm, fdjac1,
!                            qform, qrfac, r1mpyq, r1updt
!
!       fortran-supplied ... abs, max, min, mod
!
!     Argonne National Laboratory. Minpack project. March 1980.
!     Burton S. Garbow, Kenneth E. Hillstrom, Jorge J. Moré
!
!     **********
      integer :: i, flag, iter, j, jm1, l, msum, ncfail, ncsuc,         &
                 nslow1, nslow2, iwa(1)
      logical :: jeval, sing
      double precision :: actred, delta, epsmch, fnorm, fnorm1,         &
                          pnorm, prered, ratio, sum, temp, xnorm

      double precision, parameter :: one = 1.0d0, p1 = 0.1d0, p5 = 0.5d0, &
                          p001 = 0.001d0, p0001 = 0.0001d0, zero = 0.0d0

!--------- end of declarations -- execution starts hereafter  ------

      ! epsmch is the machine precision.
      epsmch = epsilon(1.0d0)

      info = 0
      flag = 0
      nfev = 0

      ! Check the input parameters for errors.
      if( maxfev <= 0 .or. factor <= zero .or. lr < (n*(n+1))/2 ) then
         go to 99
      end if
      if( mode == 2 ) then
         do j = 1, n
            if( diag(j) <= zero ) go to 99
         end do
      end if

      ! Evaluate the function at the starting point and calculate its norm.
      flag = 0

      call mf_restore_fpe( )
      call fcn( n, x, fvec, flag )
      call mf_save_and_disable_fpe( )

      nfev = 1

      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. flag >= 0 ) then
         do i = 1, n
            if( isnan(fvec(i)) ) then
               print "(/,A)", "(MUESLI LsqNonLin:) [hybrd] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied FCN routine."
               print "(20X,A,I0)", "This occured in fvec(i) for i = ", i
               call muesli_trace( pause="yes" )
               info = 14
               return
            end if
         end do
      end if

      if( flag < 0 ) go to 99
      fnorm = enorm(n,fvec)

      ! Determine the number of calls to fcn needed to compute
      ! the jacobian matrix.
      msum = min(ml+mu+1,n)

      ! Initialize iteration counter and monitors.
      iter = 1
      ncsuc = 0
      ncfail = 0
      nslow1 = 0
      nslow2 = 0
      if( print ) then
         print format, iter, x(1:n), fnorm
      end if

      outer:do

         jeval = .true.

         ! Calculate the jacobian matrix (finite differences approx.).
         flag = 0
         call fdjac1( fcn, n, x, fvec, fjac, flag, ml, mu,             &
                      epsfcn, wa1, wa2 )
         nfev = nfev + msum
         if( flag < 0 ) go to 99

         ! Compute the QR factorization of the jacobian.
         call qrfac( n, n, fjac, n, .false., iwa, 1, wa1, wa2, wa3 )

         ! On the first iteration and if mode is 1, scale according
         ! to the norms of the columns of the initial jacobian.
         if( iter == 1 ) then
            if( mode /= 2 ) then
               do j = 1, n
                  diag(j) = wa2(j)
                  if( wa2(j) == zero ) diag(j) = one
               end do
            end if

            ! Calculate the norm of the scaled x and initialize
            ! the step bound delta.
            do j = 1, n
               wa3(j) = diag(j)*x(j)
            end do
            xnorm = enorm(n,wa3)
            delta = factor*xnorm
            if( delta == zero ) delta = factor
         end if

         ! Form (Q transpose)*fvec and store in qtf.
         do i = 1, n
            qtf(i) = fvec(i)
         end do
         do j = 1, n
            if( fjac(j,j) /= zero ) then
               sum = zero
               do i = j, n
                  sum = sum + fjac(i,j)*qtf(i)
               end do
               temp = -sum/fjac(j,j)
               do i = j, n
                  qtf(i) = qtf(i) + fjac(i,j)*temp
               end do
            end if
         end do

         ! Copy the triangular factor of the QR factorization into R.
         sing = .false.
         do j = 1, n
            l = j
            jm1 = j - 1
            if( jm1 >= 1 ) then
               do i = 1, jm1
                  r(l) = fjac(i,j)
                  l = l + n - i
               end do
            end if
            r(l) = wa1(j)
            if( wa1(j) == zero ) sing = .true.
         end do

         ! Accumulate the orthogonal factor in fjac.
         call qform( n, n, fjac, n, wa1 )

         ! Rescale if necessary.
         if( mode /= 2 ) then
            do j = 1, n
               diag(j) = max(diag(j),wa2(j))
            end do
         end if

         inner:do

            ! Determine the direction p.
            call dogleg( n, r, lr, diag, qtf, delta, wa1, wa2, wa3 )

            ! Store the direction p and x + p. Calculate the norm of p.
            do j = 1, n
               wa1(j) = -wa1(j)
               wa2(j) = x(j) + wa1(j)
               wa3(j) = diag(j)*wa1(j)
            end do
            pnorm = enorm(n,wa3)

            ! On the first iteration, adjust the initial step bound.
            if( iter == 1 ) delta = min(delta,pnorm)

            ! Evaluate the function at x + p and calculate its norm.
            flag = 0

            call mf_restore_fpe( )
            call fcn( n, wa2, wa4, flag )
            call mf_save_and_disable_fpe( )
      ! check NaN values
      if( MF_NUMERICAL_CHECK .and. flag >= 0 ) then
         do i = 1, n
            if( isnan(wa4(i)) ) then
               print "(/,A)", "(MUESLI LsqNonLin:) [hybrd] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied FCN routine."
               print "(20X,A,I0)", "This occured in fvec(i) for i = ", i
               call muesli_trace( pause="yes" )
               info = 14
               return
            end if
         end do
      end if

            nfev = nfev + 1

            if( flag < 0 ) exit outer

            fnorm1 = enorm(n,wa4)

            ! Compute the scaled actual reduction.
            actred = -one
            if( fnorm1 < fnorm ) actred = one - (fnorm1/fnorm)**2

            ! Compute the scaled predicted reduction.
            l = 1
            do i = 1, n
               sum = zero
               do j = i, n
                  sum = sum + r(l)*wa1(j)
                  l = l + 1
               end do
               wa3(i) = qtf(i) + sum
            end do
            temp = enorm(n,wa3)
            prered = zero
            if( temp < fnorm ) prered = one - (temp/fnorm)**2

            ! Compute the ratio of the actual to the predicted reduction.
            ratio = zero
            if( prered > zero ) ratio = actred/prered

            ! Update the step bound.
            if( ratio < p1 ) then
               ncsuc = 0
               ncfail = ncfail + 1
               delta = p5*delta
            else
               ncfail = 0
               ncsuc = ncsuc + 1
               if( ratio >= p5 .or. ncsuc > 1 ) delta = max(delta,pnorm/p5)
               if( abs(ratio-one) <= p1 ) delta = pnorm/p5
            end if

            ! Test for successful iteration.
            if( ratio >= p0001 ) then

               ! Successful iteration. Update x, fvec, and their norm.
               do j = 1, n
                  x(j) = wa2(j)
                  wa2(j) = diag(j)*x(j)
                  fvec(j) = wa4(j)
               end do
               xnorm = enorm(n,wa2)
               fnorm = fnorm1
               iter = iter + 1
               if( iter > max_iter ) then
                  print *
                  print *, "(MUESLI:) FSolve: internal error"
                  print *, "          in hybrd(), line 383: iter > max_iter"
                  print *, "          Please report this bug to: Edouard.Canot@univ-rennes.fr"
                  print *
                  print *, "  pause for debugging purpose"
                  print *, "  [RETURN] to resume..."
                  read *
                  stop
               end if
               if( print ) then
                  print format, iter, x(1:n), fnorm
               end if

            end if

            ! Determine the progress of the iteration.
            nslow1 = nslow1 + 1
            if( actred >= p001 ) nslow1 = 0
            if( jeval ) nslow2 = nslow2 + 1
            if( actred >= p1 ) nslow2 = 0

            ! Test for convergence.
            ! (modified the fnorm test: previously, it was exactly zero,
            !  now it is the same as in octave-4.2.1)
            if( delta <= xtol*xnorm ) info = 1
            if( fnorm <= ftol*n*xnorm/25 ) info = 6
            if( info /= 0 ) exit outer

            ! Tests for termination and stringent tolerances.
            if( nfev >= maxfev ) info = 2
            if( p1*max(p1*delta,pnorm) <= epsmch*xnorm ) info = 3
            if( nslow2 == 10 ) info = 4 ! max was 5 in the original code
            if( nslow1 == 20 ) info = 5 ! max was 10 in the original code
            if( info /= 0 ) exit outer

            ! Criterion for recalculating jacobian approximation
            ! by forward differences.
            if( ncfail == 2 ) exit inner

            ! Calculate the rank one modification to the jacobian
            ! and update qtf if necessary.
            do j = 1, n
               sum = zero
               do i = 1, n
                  sum = sum + fjac(i,j)*wa4(i)
               end do
               wa2(j) = (sum - wa3(j))/pnorm
               wa1(j) = diag(j)*((diag(j)*wa1(j))/pnorm)
               if( ratio >= p0001) qtf(j) = sum
            end do

            ! Compute the qr factorization of the updated jacobian.
            call r1updt(n,n,r,lr,wa1,wa2,wa3,sing)
            call r1mpyq(n,n,fjac,n,wa2,wa3)
            call r1mpyq(1,n,qtf,1,wa2,wa3)

            jeval = .false.
         end do inner

      end do outer

   99 continue

      ! Termination, either normal or user imposed.
      if( flag < 0 ) info = flag

   end
