! f90 include file
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS --  13 Jul 2017
!_______________________________________________________________________
!
   subroutine full_chk_der( n, x, fcn, fvec, fjac, err, iflag )

      use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe

      interface
         subroutine fcn( n, x, fvec, flag )
            integer,          intent(in) :: n
            double precision, intent(in) :: x(n)
            double precision             :: fvec(n)
            integer                      :: flag
         end subroutine fcn
      end interface
      integer,          intent(in)  :: n
      double precision, intent(in)  :: x(n), fvec(n), fjac(n,n)
      real,             intent(out) :: err(n,n)
      integer,          intent(out) :: iflag

!     ******************************************************************
! EC 2014-06-05:
!
!     1) By using the current routine, the user can locate exactly
!        which component(s) of the jacobian matrix is (are) wrong.
!        This routine is more expensive than 'chkder' (see below), as
!        it requires n additional calls of the 'fcn' subroutine.
!        It is based on the approximation:
!
!          Fi( p1, ..., pj+dj, ..., pn ) - Fi( p1, ..., pn )
!         --------------------------------------------------- ~ jac(i,j)
!                                   dj
!
!         repeated for j = 1, n
!
!     2) The other routine 'chkder' doesn't make a complete check of
!        the jacobian matrix, but it is cheaper. Indeed, it merely makes
!        a global check by requiring only one additional call of the
!        'fcn' subroutine.
!
!     ******************************************************************
!
!     subroutine full_chk_der
!
!     This subroutine checks all components of the jacobian matrix fjac
!     evaluated at a point x.
!
!     The subroutine does not perform reliably if cancellation or
!     rounding errors cause a severe loss of significance in the
!     evaluation of a function.
!
!     The subroutine statement is
!
!       subroutine full_chk_der( n, x, fcn, fvec, fjac, err, iflag )
!
!     where
!
!       n is a positive integer input variable set to the number of
!         functions or variables
!
!       x is an input array of length n.
!
!       fcn is the name of the user-supplied subroutine which
!         calculates the functions Fi (see 'hybrj_mf')
!
!       fvec is an input array of length n. It contains the functions
!         evaluated at x.
!
!       fjac is an n by n array (and declared exactly of size (n,n)).
!         On input, the rows of fjac must contain the gradients of
!         the respective functions evaluated at x.
!
!       err is an array of size (n,n) which contains the relative error
!         of each component fjac(i,j).
!
!       iflag is an integer flag which is usually equal to zero. On the
!         contrary, it indicates an error during the 'fcn' evaluation.
!         Therefore, the jacobian check is done only if iflag is zero.
!
!     subprograms called
!
!       user-supplied ...... fcn
!
!       fortran supplied ... abs, log10, sqrt
!
!     Muesli numerical library. Jul 2017.
!     Édouard Canot.
!
!     **********
      integer :: j
      double precision :: epsmch, eps, dx, xp(n), fvecp(n), x_save
      double precision :: tmp(n)
      real :: abs_fjac

      logical :: switch_sign

!--------- end of declarations -- execution starts hereafter  ------

      ! epsmch is the machine precision.
      epsmch = epsilon(1.0d0)

      eps = sqrt(epsmch)

      abs_fjac = real( maxval(fjac) )
      if( abs_fjac < eps ) abs_fjac = 1.0

      ! it is preferable to initialize to zero; indeed, a return after
      ! the 'fcn' call may lead to NaN in the array err(:,:)
      err(:,:) = 0.0

      xp(:) = x(:)
      do j = 1, n
         x_save = xp(j)
         dx = eps*abs(x_save)
         if( dx < eps ) dx = eps

         xp(j) = x_save + dx

         iflag = 0

         call mf_restore_fpe( )
         call fcn( n, xp, fvecp, iflag )
         call mf_save_and_disable_fpe( )

         if( iflag /= 0 ) then
            print "(/,A)", " (MUESLI:) FSolve: Warning: non zero"       &
                           // " flag at exit of fcn when evaluating the jacobian!"
            print *, "          (Return from full_chk_der)"
            return
         end if

         xp(j) = x_save
         tmp(:) = (fvecp(:)-fvec(:))/dx
         err(:,j) = real( abs( fjac(:,j) - tmp(:) ) / abs_fjac )
      end do

   end
