! f90 include file
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 20 Jun 2018
!_______________________________________________________________________
!
   subroutine fdjac2( fcn, m, n, x, fvec, fjac, iflag, epsfcn, wa,      &
                      box_constrained, lbounds, ubounds )

      use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe

      interface
         subroutine fcn( m, n, p, fvec, flag )
            integer,          intent(in) :: m, n
            double precision, intent(in) :: p(n)
            double precision             :: fvec(m)
            integer                      :: flag
         end subroutine fcn
      end interface
      integer,          intent(in)  :: m, n
      double precision, intent(in)  :: epsfcn(n)
      double precision              :: x(n)
      double precision, intent(in)  :: fvec(m)
      double precision              :: fjac(m,n), wa(m)
      logical,          intent(in)  :: box_constrained
      double precision, intent(in)  :: lbounds(*), ubounds(*)
      integer,          intent(out) :: iflag
!     **********
!
!     subroutine fdjac2
!
!     This subroutine computes a forward-difference approximation
!     to the m by n jacobian matrix associated with a specified
!     problem of m functions in n variables.
!
!     The subroutine statement is
!
!       subroutine fdjac2( fcn, m, n, x, fvec, fjac, iflag,  epsfcn,
!                          wa, box_constrained, lbounds, ubounds )
!
!     where
!
!       fcn is the name of a module subroutine which calculates the
!         functions. See the interface in the header.
!         The value of iflag should not be changed by fcn unless
!         the user wants to terminate execution of fdjac2.
!         In this case set iflag to a negative integer.
!
!       m is a positive integer input variable set to the number
!         of functions.
!
!       n is a positive integer input variable set to the number
!         of variables. n must not exceed m.
!
!       x is an input array of length n.
!
!       fvec is an input array of length m which must contain the
!         functions evaluated at x.
!
!       fjac is an output m by n array which contains the
!         approximation to the jacobian matrix evaluated at x.
!
!       ldfjac is a positive integer input variable not less than m
!         which specifies the leading dimension of the array fjac.
!
!       iflag is an integer variable which can be used to terminate
!         the execution of fdjac2. See description of fcn.
!
!       epsfcn is a array of length n. On input, its components are
!         used in determining a suitable step length for the
!         forward-difference approximation. The user must know the
!         behavior of its function fcn in order to propose an optimal
!         increment value for each parameter. This increment cannot
!         be zero.
!
!       wa is a work array of length m.
!
!     Subprograms called
!
!       user-supplied ...... fcn
!
!       fortran-supplied ... abs, max, sqrt
!
!     Argonne National Laboratory. MINPACK project. March 1980.
!     Burton S. Garbow, Kenneth E. Hillstrom, Jorge J. Moré
!
!     Modified by É. Canot (implement constraints on parameters)
!
!     **********
      integer :: j
      double precision :: eps(n), epsmch, dx, x_save

      logical :: switch_sign

!--------- end of declarations -- execution starts hereafter  ------

      ! epsmch is the machine precision.
      epsmch = epsilon(1.0d0)

      do j = 1, n

         eps(j) = max( epsfcn(j), sqrt(epsmch) )

         x_save = x(j)
         dx = eps(j)*abs(x_save)
         if( dx == 0.0d0 ) dx = eps(j)

         x(j) = x_save + dx
         if( box_constrained ) then
            switch_sign = .false.
            if( ubounds(j) < x(j) ) then
               switch_sign = .true.
               x(j) = x_save - dx
            end if
            if( x(j) < lbounds(j) ) then
               ! Here, found a dead-end situation, and the user will be
               ! invited to:
               !  (1) decrease epsfcn(j) (by using another numerical
               !      method in the function fcn);
               !  (2) increase the interval [ubounds(j)-lbounds(j)] for
               !      the j-th parameters;
               !  (3) avoid using a finite-difference jacobian.
               print "(/,A)", " (MUESLI:) LsqNonLin: Error: when evaluating the jacobian, found the"
               print *,       "          following dead-end situation."
               print *,       "          The choice of 'dx' in the finite-difference approximation"
               print "(A,I0)","           violates the constraint of the j-th parameter, with j = ", j
               print *,       "          You are invited to:"
               print *,       "            (1) decrease epsfcn(j) (by using another numerical method"
               print *,       "                 in the function fcn);"
               print *,       "            (2) increase the interval [lbounds(j),ubounds(j)] for the"
               print *,       "                 j-th parameters;"
               print *,       "            (3) avoid using a finite-difference jacobian."
               print *, "          (Return from fdjac2)"
!### TODO ?: on pourrait aussi diminuer le dx, au risque d'accroître
!          les erreurs d'arrondis.
               iflag = 1
               return
            end if
         end if

         iflag = 0

         call mf_restore_fpe( )
         call fcn( m, n, x, wa, iflag )
         call mf_save_and_disable_fpe( )

         if( iflag /= 0 ) then
            print "(/,A)", " (MUESLI:) LsqNonLin: Warning: non zero"    &
                           // " flag at exit of fcn when evaluating the jacobian!"
            print *, "          (Return from fdjac2)"
            return
         end if

         x(j) = x_save
         fjac(:,j) = (wa(:) - fvec(:))/dx
         if( box_constrained ) then
            if( switch_sign ) then
               fjac(:,j) = - fjac(:,j)
            end if
         end if
      end do

   end
