! f90 include file
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 20 Jun 2018
!_______________________________________________________________________
!
   subroutine fdjac1( fcn, n, x, fvec, fjac, iflag, ml, mu,             &
                      epsfcn, wa1, wa2 )

      use mod_mfaux, only: mf_save_and_disable_fpe, mf_restore_fpe

      interface
         subroutine fcn( n, x, fvec, iflag )
            integer,          intent(in) :: n
            double precision, intent(in) :: x(n)
            double precision             :: fvec(n)
            integer                      :: iflag
         end subroutine fcn
      end interface
      integer,          intent(in)  :: n, ml, mu
      double precision, intent(in)  :: epsfcn
      double precision              :: x(n)
      double precision, intent(in)  :: fvec(n)
      double precision              :: fjac(n,n), wa1(n), wa2(n)
      integer,          intent(out) :: iflag
!     **********
!
!     subroutine fdjac1
!
!     This subroutine computes a forward-difference approximation
!     to the n by n jacobian matrix associated with a specified
!     problem of n functions in n variables. If the jacobian has
!     a banded form, then function evaluations are saved by only
!     approximating the nonzero terms.
!
!     The subroutine statement is
!
!       subroutine fdjac1( fcn, n, x, fvec, fjac, iflag, ml, mu,
!                          epsfcn, wa1, wa2 )
!
!     where
!
!       fcn is the name of the user-supplied subroutine which
!         calculates the functions. See the interface in the header.
!         The value of iflag should not be changed by fcn unless
!         the user wants to terminate execution of fdjac1.
!         In this case set iflag to a negative integer.
!
!       n is a positive integer input variable set to the number
!         of functions and variables.
!
!       x is an input array of length n.
!
!       fvec is an input array of length n which must contain the
!         functions evaluated at x.
!
!       fjac is an output n by n array which contains the
!         approximation to the jacobian matrix evaluated at x.
!
!       iflag is an integer variable which can be used to terminate
!         the execution of fdjac1. See description of fcn.
!
!       ml is a nonnegative integer input variable which specifies
!         the number of subdiagonals within the band of the
!         jacobian matrix. If the jacobian is not banded, set
!         ml to at least n - 1.
!
!       mu is a nonnegative integer input variable which specifies
!         the number of superdiagonals within the band of the
!         jacobian matrix. If the jacobian is not banded, set
!         mu to at least n - 1.
!
!       epsfcn is an input variable used in determining a suitable
!         step length for the forward-difference approximation. The
!         user must know the behavior of its function fcn in order
!         to propose an optimal increment value for each parameter.
!         This increment cannot be zero.
!
!       wa1 and wa2 are work arrays of length n. If ml + mu + 1 is at
!         least n, then the jacobian is considered dense, and wa2 is
!         not referenced.
!
!     subprograms called
!
!       user-supplied ...... fcn
!
!       fortran-supplied ... abs, max, sqrt
!
!     Argonne National Laboratory. Minpack project. March 1980.
!     Burton S. Garbow, Kenneth E. Hillstrom, Jorge J. Moré
!
!     Modified by É. Canot
!
!     **********
      integer :: i, j, k, msum
      double precision :: eps, epsmch, dx, x_save

!--------- end of declarations -- execution starts hereafter  ------

      ! epsmch is the machine precision.
      epsmch = epsilon(1.0d0)

      eps = max( epsfcn, sqrt(epsmch) )

      msum = ml + mu + 1
      if( msum < n ) then

         ! Computation of banded approximate jacobian.
         do k = 1, msum
            do j = k, n, msum
               wa2(j) = x(j)
               dx = eps*abs(wa2(j))
               if( dx == 0.0d0 ) dx = eps
               x(j) = wa2(j) + dx
            end do

            call mf_restore_fpe( )
            call fcn( n, x, wa1, iflag )
            call mf_save_and_disable_fpe( )

            if( iflag /= 0 ) then
               print "(/,A)", " (MUESLI:) FSolve: Warning: non zero"    &
                              // " flag at exit of fcn when evaluating the jacobian!"
               print *, "          (Return from fdjac1)"
               return
            end if

            do j = k, n, msum
               x(j) = wa2(j)
               dx = eps*abs(wa2(j))
               if( dx == 0.0d0 ) dx = eps
               do i = 1, n
                  fjac(i,j) = 0.0d0
                  if( i >= j - mu .and. i <= j + ml ) then
                     fjac(i,j) = (wa1(i) - fvec(i))/dx
                  end if
               end do
            end do
         end do

      else

         ! Computation of dense approximate jacobian.
         do j = 1, n
            x_save = x(j)
            dx = eps*abs(x_save)
            if( dx == 0.0d0 ) dx = eps

            x(j) = x_save + dx

            call mf_restore_fpe( )
            call fcn( n, x, wa1, iflag )
            call mf_save_and_disable_fpe( )

            if( iflag /= 0 ) then
               print "(/,A)", " (MUESLI:) FSolve: Warning: non zero"    &
                              // " flag at exit of fcn when evaluating the jacobian!"
               print *, "          (Return from fdjac1)"
               return
            end if

            x(j) = x_save
            fjac(:,j) = (wa1(:) - fvec(:))/dx
         end do

      end if

   end
