! f90 include file
!
! modified for MUESLI Fortran library
! (C) É. Canot -- IPR/CNRS -- 13 Jul 2017
!_______________________________________________________________________
!
   subroutine chkder( m, n, x, fvec, fjac, xp, dx, fvecp, mode,         &
                      quality, box_constrained, ubounds )

      use mod_core, only: MF_NAN

      integer,          intent(in)  :: m, n, mode
      double precision, intent(in)  :: x(n), fvec(m), fjac(m,n), fvecp(m)
      double precision              :: xp(n), dx(n)
      double precision, intent(out) :: quality(m)
      logical,          intent(in)  :: box_constrained
      double precision, intent(in)  :: ubounds(*)

!     ******************************************************************
! EC 2014-06-05:
!
!     1) The current routine doesn't make a complete check of the
!        jacobian matrix. Instead, it merely makes a global check by
!        requiring only one additional call of the 'fcn' subroutine.
!        Hence, it is cheap. It is based on the approximation:
!
!          Fi( p1+d1, ..., pn+dn ) - Fi( p1, ..., pn ) ~
!                                  + d1*jac(i,1) + ... + dn*jac(i,n)
!        By inspecting the returned vector 'quality', the user only
!        knows the wrong line of the jacobian matrix.
!
!     2) By using the other checking routine 'full_chk_der', the user
!        can locate exactly which component(s) of the jacobian matrix
!        is (are) wrong. This later routine is more expensive, as it
!        requires n additional calls of the 'fcn' subroutine.
!
!     ******************************************************************
!
!     subroutine chkder
!
!     This subroutine checks the gradients of m nonlinear functions
!     in n variables, evaluated at a point x, for consistency with
!     the functions themselves. The user must call chkder twice,
!     first with mode = 1 and then with mode = 2.
!
!     mode = 1. On input, x must contain the point of evaluation.
!               On output, xp is set to a neighboring point.
!               fjac is not referenced during this first call.
!
!     mode = 2. On input, fvec must contain the functions and the
!                         rows of fjac must contain the gradients
!                         of the respective functions each evaluated
!                         at x, and fvecp must contain the functions
!                         evaluated at xp.
!               On output, quality contains measures of correctness
!                          of the respective gradients.
!
!     The subroutine does not perform reliably if cancellation or
!     rounding errors cause a severe loss of significance in the
!     evaluation of a function. Therefore, none of the components
!     of x should be unusually small (in particular, zero) or any
!     other value which may cause loss of significance.
!
!     The subroutine statement is
!
!       subroutine chkder( m, n, x, fvec, fjac, xp, dx, fvecp, mode,
!                          quality, box_constrained, ubounds )
!
!     where
!
!       m is a positive integer input variable set to the number
!         of functions.
!
!       n is a positive integer input variable set to the number
!         of variables.
!
!       x is an input array of length n.
!
!       fvec is an array of length m. On input when mode = 2, fvec
!         must contain the functions evaluated at x.
!
!       fjac is an m by n array (and declared exactly of size (m,n)).
!         On input when mode = 2, the rows of fjac must contain the
!         gradients of the respective functions evaluated at x.
!
!       xp is an array of length n. On output when mode = 1, xp is
!         set to a neighboring point of x.
!
!       fvecp is an array of length m. On input when mode = 2, fvecp
!         must contain the functions evaluated at xp.
!
!       mode is an integer input variable set to 1 on the first call
!         and 2 on the second. Other values of mode are equivalent
!         to mode = 1.
!
!       quality is an array of length m. On output when mode = 2,
!         quality contains measures of correctness of the respective
!         gradients. If there is no severe loss of significance, then
!         if quality(i) is 1.0 the i-th gradient is correct, while if
!         quality(i) is 0.0 the i-th gradient is incorrect.
!         For values of quality between 0.0 and 1.0, the categorization
!         is less certain. In general, a value of quality(i) greater
!         than 0.5 indicates that the i-th gradient is probably correct,
!         while a value of quality(i) less than 0.5 indicates that the
!         i-th gradient is probably incorrect.
!         Contains NaN values when the numerical comparison cannot be
!         computed.
!
!
!     subprograms called
!
!       fortran supplied ... abs, log10, sqrt, epsilon
!
!     Argonne National Laboratory. Minpack project. March 1980.
!     Burton S. Garbow, Kenneth E. Hillstrom, Jorge J. Moré
!
!     Important modifications by É. Canot, in order to relax the
!     quality definition, which appears too strict.
!     Moreover, quality is set to NaN if the computation cannot be done.
!
!     **********
      integer :: i, j
      double precision :: eps, epsf, epslog, epsmch, temp, diff(m),     &
                          norm
      double precision, parameter :: one = 1.0d0, zero = 0.0d0,         &
                                     factor = 1.0d2, threshold = 0.1d0

!--------- end of declarations -- execution starts hereafter  ------

      ! epsmch is the machine precision.
      epsmch = epsilon(1.0d0)

      eps = sqrt(epsmch)

      if( mode == 1 ) then

         do j = 1, n
            dx(j) = eps*abs(x(j))
            if( dx(j) < eps ) dx(j) = eps
            xp(j) = x(j) + dx(j)
            if( box_constrained ) then
               if( ubounds(j) < xp(j) ) then
                  dx(j) = - dx(j)
                  xp(j) = x(j) + dx(j)
               end if
            end if
         end do

      else ! mode == 2

         epsf = factor*epsmch
         epslog = log10(eps)
         quality(1:m) = 0.0d0
         do j = 1, n
            quality(:) = quality(:) + dx(j)*fjac(:,j)
         end do

         do i = 1, m
            diff(i) = fvecp(i) - fvec(i)
         end do
         norm = maxval(abs(diff))

         do i = 1, m
            temp = abs( diff(i) - quality(i) )
            if( norm > epsf ) then
               temp = temp / norm
            else
               ! diff is too small, quality is estimated in an absolute
            end if
            if( temp <= eps ) then
               quality(i) = one
            else if( eps < temp .and. temp < threshold ) then
               quality(i) = log10(temp) / epslog
            else
               ! temp >= threshold
               quality(i) = zero
            end if
         end do

      end if

   end
