#if defined _GNU_GFC
! convert (3,2,0) to 30200
#define GCC_VERSION (__GNUC__*10000+__GNUC_MINOR__*100+__GNUC_PATCHLEVEL__)
#endif

module f90_gzlib

   ! Fortran 90 interface to the ZLIB-1.2
   !
   !   tested with zlib versions 1.2.1 and 1.2.2;
   !   but requires now version 1.2.3 (as of July 2005);
   !   should be ok with any 1.2.x version (test is done on "1.2" number)
   !
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

   use rational_numbers

   use mod_mfdebug

#if GCC_VERSION >= 40900
   use iso_C_binding, only : C_int, C_ptr, C_loc, C_size_t
#else
   ! While Gfortran 4.8 support the iso_C_binding intrinsic module, it is
   ! not mature enough to be employed in this case...
#endif

   implicit none

   private

   integer, parameter :: DOUBLE = kind(1.0d0)

   type :: gz_filedes
      private
      integer(kind=MF_ADDRESS) :: ptr = 0
   end type

#if GCC_VERSION >= 40900
   interface
      subroutine gateway_gzwrite( file, buf, n )                        &
                 bind(C, name="gateway_gzwrite_")
         import C_int, C_ptr, C_size_t
         integer(kind=C_size_t) :: file
         type(C_ptr), value     :: buf
         integer(kind=C_int)    :: n
      end subroutine
   end interface
#endif

   interface gzwrite
      module procedure gzwrite_string
      module procedure gzwrite_int
      module procedure gzwrite_int_dim_1
      module procedure gzwrite_int_dim_2
      module procedure gzwrite_real
      module procedure gzwrite_real_dim_1
      module procedure gzwrite_real_dim_2
      module procedure gzwrite_real_dble
      module procedure gzwrite_real_dble_dim_1
      module procedure gzwrite_real_dble_dim_2
      module procedure gzwrite_cmplx
      module procedure gzwrite_cmplx_dim_1
      module procedure gzwrite_cmplx_dim_2
      module procedure gzwrite_cmplx_dble
      module procedure gzwrite_cmplx_dble_dim_1
      module procedure gzwrite_cmplx_dble_dim_2
      module procedure gzwrite_rat_num_dim_1
      module procedure gzwrite_prop
   end interface

#if GCC_VERSION >= 40900
   interface
      subroutine gateway_gzread( file, buf, n, nbytes_read )            &
                 bind(C, name="gateway_gzread_")
         import C_int, C_ptr, C_size_t
         integer(kind=C_size_t) :: file
         type(C_ptr), value     :: buf
         integer(kind=C_int)    :: n
         integer(kind=C_int)    :: nbytes_read
      end subroutine
   end interface
#endif

   interface gzread
      module procedure gzread_string
      module procedure gzread_int
      module procedure gzread_int_dim_1
      module procedure gzread_int_dim_2
      module procedure gzread_real
      module procedure gzread_real_dim_1
      module procedure gzread_real_dim_2
      module procedure gzread_real_dble
      module procedure gzread_real_dble_dim_1
      module procedure gzread_real_dble_dim_2
      module procedure gzread_cmplx
      module procedure gzread_cmplx_dim_1
      module procedure gzread_cmplx_dim_2
      module procedure gzread_cmplx_dble
      module procedure gzread_cmplx_dble_dim_1
      module procedure gzread_cmplx_dble_dim_2
      module procedure gzread_rat_num
      module procedure gzread_rat_num_dim_1
      module procedure gzread_prop
   end interface

   public :: gz_filedes

   public :: gzopen, &
             gzwrite, &
             gzread, &
             gzclose

   public :: filedes_is_null

   logical, save :: zlib_first_call = .true.
   character(len=3), parameter :: zlib_required_version = "1.2"

contains
!_______________________________________________________________________
!
   function filedes_is_null( gzfile ) result ( bool )

      type(gz_filedes), intent(in) :: gzfile
      logical :: bool
      !------ API end ------

   !------ end of declarations -- execution starts hereafter  ------

      if( gzfile%ptr == 0 ) then
         bool = .true.
      else
         bool = .false.
      end if

   end function filedes_is_null
!_______________________________________________________________________
!
   function gzopen( path, mode ) result ( gzfile )

      character(len=*), intent(in) :: path, mode
      type(gz_filedes) :: gzfile
      !------ API end ------

      integer(kind=MF_ADDRESS), external :: gateway_gzopen
      character(len=6) :: zlib_version ! 5 car. + NULL

   !------ end of declarations -- execution starts hereafter  ------

      gzfile%ptr = gateway_gzopen( trim(path)//char(0),                 &
                                   trim(mode)//char(0),                 &
                                   zlib_version )

      if( zlib_first_call ) then
         if( zlib_version(1:3) /= zlib_required_version ) then
            print *, "(f90 gzopen:) the zlib version used must be : ",  &
                     zlib_required_version
            print *, "              but found version : ",              &
                     zlib_version(1:3)
            print *, "              f90_gzlib will not open gzipped file : ", trim(path)
            return
         end if
         zlib_first_call = .false.
      end if

      if( gzfile%ptr == 0 ) then
         print *, "(f90 gzopen:) cannot open gzipped file : ", trim(path)
      end if

   end function gzopen
!_______________________________________________________________________
!
   subroutine gzwrite_string( gzfile, string )

      type(gz_filedes),         intent(in) :: gzfile
      character(len=*), target, intent(in) :: string
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = len(string)

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(string), n )
#else
      call gateway_gzwrite( gzfile%ptr, string, n )
#endif

   end subroutine gzwrite_string
!_______________________________________________________________________
!
   subroutine gzwrite_int( gzfile, a )

      type(gz_filedes), intent(in) :: gzfile
      integer, target,  intent(in) :: a
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 4

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a), n )
#else
      call gateway_gzwrite( gzfile%ptr, a, n )
#endif

   end subroutine gzwrite_int
!_______________________________________________________________________
!
   subroutine gzwrite_int_dim_1( gzfile, a )

      type(gz_filedes), intent(in) :: gzfile
      integer, target,  intent(in) :: a(:)
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 4*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a(1)), n )
#else
      call gateway_gzwrite( gzfile%ptr, a(1), n )
#endif

   end subroutine gzwrite_int_dim_1
!_______________________________________________________________________
!
   subroutine gzwrite_int_dim_2( gzfile, a )

      type(gz_filedes), intent(in) :: gzfile
      integer, target,  intent(in) :: a(:,:)
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 4*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a(1,1)), n )
#else
      call gateway_gzwrite( gzfile%ptr, a(1,1), n )
#endif

   end subroutine gzwrite_int_dim_2
!_______________________________________________________________________
!
   subroutine gzwrite_real( gzfile, a )

      type(gz_filedes), intent(in) :: gzfile
      real, target,     intent(in) :: a
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 4

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a), n )
#else
      call gateway_gzwrite( gzfile%ptr, a, n )
#endif

   end subroutine gzwrite_real
!_______________________________________________________________________
!
   subroutine gzwrite_real_dim_1( gzfile, a )

      type(gz_filedes), intent(in) :: gzfile
      real, target,     intent(in) :: a(:)
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 4*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a(1)), n )
#else
      call gateway_gzwrite( gzfile%ptr, a(1), n )
#endif

   end subroutine gzwrite_real_dim_1
!_______________________________________________________________________
!
   subroutine gzwrite_real_dim_2( gzfile, a )

      type(gz_filedes), intent(in) :: gzfile
      real, target,     intent(in) :: a(:,:)
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 4*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a(1,1)), n )
#else
      call gateway_gzwrite( gzfile%ptr, a(1,1), n )
#endif

   end subroutine gzwrite_real_dim_2
!_______________________________________________________________________
!
   subroutine gzwrite_real_dble( gzfile, a )

      type(gz_filedes),          intent(in) :: gzfile
      real(kind=DOUBLE), target, intent(in) :: a
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 8

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a), n )
#else
      call gateway_gzwrite( gzfile%ptr, a, n )
#endif

   end subroutine gzwrite_real_dble
!_______________________________________________________________________
!
   subroutine gzwrite_real_dble_dim_1( gzfile, a )

      type(gz_filedes),          intent(in) :: gzfile
      real(kind=DOUBLE), target, intent(in) :: a(:)
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 8*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a(1)), n )
#else
      call gateway_gzwrite( gzfile%ptr, a(1), n )
#endif

   end subroutine gzwrite_real_dble_dim_1
!_______________________________________________________________________
!
   subroutine gzwrite_real_dble_dim_2( gzfile, a )

      type(gz_filedes),          intent(in) :: gzfile
      real(kind=DOUBLE), target, intent(in) :: a(:,:)
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 8*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a(1,1)), n )
#else
      call gateway_gzwrite( gzfile%ptr, a(1,1), n )
#endif

   end subroutine gzwrite_real_dble_dim_2
!_______________________________________________________________________
!
   subroutine gzwrite_cmplx( gzfile, a )

      type(gz_filedes), intent(in) :: gzfile
      complex, target,  intent(in) :: a
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 8

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a), n )
#else
      call gateway_gzwrite( gzfile%ptr, a, n )
#endif

   end subroutine gzwrite_cmplx
!_______________________________________________________________________
!
   subroutine gzwrite_cmplx_dim_1( gzfile, a )

      type(gz_filedes), intent(in) :: gzfile
      complex, target,  intent(in) :: a(:)
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 8*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a(1)), n )
#else
      call gateway_gzwrite( gzfile%ptr, a(1), n )
#endif

   end subroutine gzwrite_cmplx_dim_1
!_______________________________________________________________________
!
   subroutine gzwrite_cmplx_dim_2( gzfile, a )

      type(gz_filedes), intent(in) :: gzfile
      complex, target,  intent(in) :: a(:,:)
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 8*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a(1,1)), n )
#else
      call gateway_gzwrite( gzfile%ptr, a(1,1), n )
#endif

   end subroutine gzwrite_cmplx_dim_2
!_______________________________________________________________________
!
   subroutine gzwrite_cmplx_dble( gzfile, a )

      type(gz_filedes),             intent(in) :: gzfile
      complex(kind=DOUBLE), target, intent(in) :: a
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 16

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a), n )
#else
      call gateway_gzwrite( gzfile%ptr, a, n )
#endif

   end subroutine gzwrite_cmplx_dble
!_______________________________________________________________________
!
   subroutine gzwrite_cmplx_dble_dim_1( gzfile, a )

      type(gz_filedes),             intent(in) :: gzfile
      complex(kind=DOUBLE), target, intent(in) :: a(:)
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 16*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a(1)), n )
#else
      call gateway_gzwrite( gzfile%ptr, a(1), n )
#endif

   end subroutine gzwrite_cmplx_dble_dim_1
!_______________________________________________________________________
!
   subroutine gzwrite_cmplx_dble_dim_2( gzfile, a )

      type(gz_filedes),             intent(in) :: gzfile
      complex(kind=DOUBLE), target, intent(in) :: a(:,:)
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 16*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a(1,1)), n )
#else
      call gateway_gzwrite( gzfile%ptr, a(1,1), n )
#endif

   end subroutine gzwrite_cmplx_dble_dim_2
!_______________________________________________________________________
!
   subroutine gzwrite_rat_num_dim_1( gzfile, a )

      type(gz_filedes),       intent(in) :: gzfile
      type(rational), target, intent(in) :: a(:)
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = (2*2)*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a(1)), n )
#else
      call gateway_gzwrite( gzfile%ptr, a(1), n )
#endif

   end subroutine gzwrite_rat_num_dim_1
!_______________________________________________________________________
!
   subroutine gzwrite_prop( gzfile, a )

      type(gz_filedes),             intent(in) :: gzfile
      integer(kind=kind_1), target, intent(in) :: a
      !------ API end ------

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = 1

#if GCC_VERSION >= 40900
      call gateway_gzwrite( gzfile%ptr, C_loc(a), n )
#else
      call gateway_gzwrite( gzfile%ptr, a, n )
#endif

   end subroutine gzwrite_prop
!
!=======================================================================
!
   subroutine gzread_string( gzfile, string, status )

      type(gz_filedes), intent(in) :: gzfile
      character(len=*), target :: string
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = len(string)

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(string), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, string, n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_string
!_______________________________________________________________________
!
   subroutine gzread_int( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      integer, target :: a
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 4

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a, n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_int
!_______________________________________________________________________
!
   subroutine gzread_int_dim_1( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      integer, target :: a(:)
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 4*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a(1)), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a(1), n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_int_dim_1
!_______________________________________________________________________
!
   subroutine gzread_int_dim_2( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      integer, target :: a(:,:)
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 4*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a(1,1)), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a(1,1), n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_int_dim_2
!_______________________________________________________________________
!
   subroutine gzread_real( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      real, target :: a
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 4

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a, n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_real
!_______________________________________________________________________
!
   subroutine gzread_real_dim_1( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      real, target :: a(:)
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 4*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a(1)), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a(1), n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_real_dim_1
!_______________________________________________________________________
!
   subroutine gzread_real_dim_2( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      real, target :: a(:,:)
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 4*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a(1,1)), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a(1,1), n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_real_dim_2
!_______________________________________________________________________
!
   subroutine gzread_real_dble( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      real(kind=DOUBLE), target :: a
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 8

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a, n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_real_dble
!_______________________________________________________________________
!
   subroutine gzread_real_dble_dim_1( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      real(kind=DOUBLE), target :: a(:)
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 8*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a(1)), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a(1), n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_real_dble_dim_1
!_______________________________________________________________________
!
   subroutine gzread_real_dble_dim_2( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      real(kind=DOUBLE), target :: a(:,:)
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 8*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a(1,1)), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a(1,1), n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_real_dble_dim_2
!_______________________________________________________________________
!
   subroutine gzread_cmplx( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      complex, target :: a
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 8

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a, n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_cmplx
!_______________________________________________________________________
!
   subroutine gzread_cmplx_dim_1( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      complex, target :: a(:)
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 8*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a(1)), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a(1), n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_cmplx_dim_1
!_______________________________________________________________________
!
   subroutine gzread_cmplx_dim_2( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      complex, target :: a(:,:)
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 8*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a(1,1)), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a(1,1), n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_cmplx_dim_2
!_______________________________________________________________________
!
   subroutine gzread_cmplx_dble( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      complex(kind=DOUBLE), target :: a
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 16

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a, n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_cmplx_dble
!_______________________________________________________________________
!
   subroutine gzread_cmplx_dble_dim_1( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      complex(kind=DOUBLE), target :: a(:)
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 16*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a(1)), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a(1), n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_cmplx_dble_dim_1
!_______________________________________________________________________
!
   subroutine gzread_cmplx_dble_dim_2( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      complex(kind=DOUBLE), target :: a(:,:)
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 16*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a(1,1)), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a(1,1), n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_cmplx_dble_dim_2
!_______________________________________________________________________
!
   subroutine gzread_rat_num( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      type(rational), target :: a
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 2*2

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a, n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_rat_num
!_______________________________________________________________________
!
   subroutine gzread_rat_num_dim_1( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      type(rational), target :: a(:)
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = (2*2)*size(a)

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a(1)), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a(1), n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_rat_num_dim_1
!_______________________________________________________________________
!
   subroutine gzread_prop( gzfile, a, status )

      type(gz_filedes), intent(in) :: gzfile
      integer(kind=kind_1), target :: a
      integer, intent(out), optional :: status
      !------ API end ------

      integer :: n, nbytes_read

   !------ end of declarations -- execution starts hereafter  ------

      n = 1

#if GCC_VERSION >= 40900
      call gateway_gzread( gzfile%ptr, C_loc(a), n, nbytes_read )
#else
      call gateway_gzread( gzfile%ptr, a, n, nbytes_read )
#endif
      if( present(status) ) then
         if( nbytes_read == n ) then
            status = 0
         else
            status = -1
         end if
      end if

   end subroutine gzread_prop
!_______________________________________________________________________
!
   subroutine gzclose( gzfile )

      type(gz_filedes), intent(in) :: gzfile
      !------ API end ------

   !------ end of declarations -- execution starts hereafter  ------

      call gateway_gzclose( gzfile%ptr )

   end subroutine gzclose
!_______________________________________________________________________
!
end module f90_gzlib
