module bezier

   ! Automatic subdivision of a Bézier segment.
   ! Fortran implementation.
   !
   ! version 0.1   -- É. Canot -- IRISA/CNRS -- 11 Sept. 2001
   ! version 0.1.1 -- É. Canot -- IRISA/CNRS -- 19 Mars 2005
   !                  Corrections mineures : détection de mémoire
   !                    insuffisante et limitations d'échelles;
   ! version 0.1.2 -- É. Canot -- IRISA/CNRS -- 18 Mai 2006
   !                  Corrections mineures : utilisation de la routine
   !                    'PrintMessage' pour l'affichage des messages de
   !                    Warnings ou d'ERRORS.
   !                  Ajout de commentaires pour le point d'entrée
   !                    principal : 'cubic_bezier'
   !
   ! É. Canot -- IPR/CNRS -- 06 Juin 2006
   ! Ajout d'une version clippée : 'cubic_bezier_clipped'
   !
   ! version 0.2   -- É. Canot -- IPR/CNRS -- 17 Juin 2006
   !                  Ajout de routines d'intersection de segments
   !                  linéaires et/ou de Bézier
   !
   ! version 0.3   -- É. Canot -- IPR/CNRS --  1 Mar 2020
   !                  Passage en double precision
   !                  (MFPLOT est désormais en double)
   !
   ! version 0.4   -- É. Canot -- IPR/CNRS -- 30 Sep 2021
   !                  Ajout de la version quadratique: 'quadratic_bezier'
   !                  Certaines routines ont été renommées.
   !                  Toutes les routines liées au calcul d'intersection
   !                  ont été supprimées.

   use mod_mfarray

   implicit none

   private

   interface quadratic_bezier
      module procedure quadratic_bezier_pix
   end interface quadratic_bezier

   interface cubic_bezier
      module procedure cubic_bezier_pix
   end interface cubic_bezier

   public :: quadratic_bezier, cubic_bezier

   ! internal common data
   double precision :: clip_box(4)
   logical :: pending_last_pt
   double precision :: last_pt(0:1)

contains
!_______________________________________________________________________
!
   subroutine quadratic_bezier_pix ( ctrl, pa, pa_size )

      double precision, intent(in)  :: ctrl(0:5)
      double precision, pointer     :: pa(:)
      integer,          intent(out) :: pa_size
      !------ API end ------

      ! Bézier algorithm for drawing a pixel-based curved from
      ! the three control points of a segment.
      !
      ! in  : array 'ctrl', containing the (x,y) coordinates of the
      !       three control points, stored as [x1,y1,x2,y2,x3,y3]
      !
      ! out : array 'pa' of returned (x,y) coordinates of the
      !       interpolated points, stored in the same manner as 'ctrl'.
      !       The (pointer to) array is allocated in the current routine.
      !       'pa_size' is the size of the allocated array 'pa', so the
      !       number of returned points is: n = pa_size/2
      !
      ! control-points coordinates are supposed to be in pixels,
      ! but stored in real arrays.

      integer                       :: maxsize
      double precision              :: real_maxsize
      double precision, allocatable :: pt(:)
      integer                       :: pt_size

      integer :: i, alloc_stat
      double precision :: x_min, x_max, y_min, y_max

   !------ end of declarations -- execution starts hereafter  ------

      x_min = min( ctrl(0), ctrl(2), ctrl(4) )
      x_max = max( ctrl(0), ctrl(2), ctrl(4) )
      y_min = min( ctrl(1), ctrl(3), ctrl(5) )
      y_max = max( ctrl(1), ctrl(3), ctrl(5) )
      real_maxsize = 4.0 + 2.0*max( x_max-x_min, y_max-y_min )

      if( real_maxsize > huge(1)/10. ) then

         call PrintMessage( "quadratic_bezier_pix", "W",                &
                            "truncate real_maxsize to huge(1)/10!",    &
                            "(perhaps your control points contain Inf or very great values)" )
         maxsize = huge(1)/10.
      else
         maxsize = real_maxsize
      end if
      allocate( pt(0:maxsize-1), stat=alloc_stat )
      if( alloc_stat /= 0 ) then

         call PrintMessage( "quadratic_bezier_pix", "W",                &
                            "cannot allocate sufficient workspace!",   &
                            "(perhaps your control points contain NaN, Inf or very great values)",&
                            "-> returning initial control points without subdivisions." )

         ! return initial control points
         pa_size = 6
         allocate( pa(0:pa_size-1) )
         do i = 0, pa_size-1
            pa(i) = ctrl(i)
         end do

      else

         pt_size = 0

         call polygonize_quad_bezier( pt, pt_size, ctrl, maxsize )
         pt(pt_size) = ctrl(4); pt_size = pt_size + 1
         pt(pt_size) = ctrl(5); pt_size = pt_size + 1

         pa_size = pt_size
         allocate( pa(0:pa_size-1) )
         do i = 0, pa_size-1
            pa(i) = pt(i)
         end do

      end if

   end subroutine quadratic_bezier_pix
!_______________________________________________________________________
!
   subroutine cubic_bezier_pix ( ctrl, pa, pa_size )

      double precision, intent(in)  :: ctrl(0:7)
      double precision, pointer     :: pa(:)
      integer,          intent(out) :: pa_size
      !------ API end ------

      ! Bézier algorithm for drawing a pixel-based curved from
      ! the four control points of a segment.
      !
      ! in  : array 'ctrl', containing the (x,y) coordinates of the
      !       four control points, stored as [x1,y1,x2,y2,...,x4,y4]
      !
      ! out : array 'pa' of returned (x,y) coordinates of the
      !       interpolated points, stored in the same manner as 'ctrl'.
      !       The (pointer to) array is allocated in the current routine.
      !       'pa_size' is the size of the allocated array 'pa', so the
      !       number of returned points is: n = pa_size/2
      !
      ! control-points coordinates are supposed to be in pixels,
      ! but stored in real arrays.

      integer                       :: maxsize
      double precision              :: real_maxsize
      double precision, allocatable :: pt(:)
      integer                       :: pt_size

      integer :: i, alloc_stat
      double precision :: x_min, x_max, y_min, y_max

   !------ end of declarations -- execution starts hereafter  ------

      x_min = min( ctrl(0), ctrl(2), ctrl(4), ctrl(6) )
      x_max = max( ctrl(0), ctrl(2), ctrl(4), ctrl(6) )
      y_min = min( ctrl(1), ctrl(3), ctrl(5), ctrl(7) )
      y_max = max( ctrl(1), ctrl(3), ctrl(5), ctrl(7) )
      real_maxsize = 4.0 + 2.0*max( x_max-x_min, y_max-y_min )

      if( real_maxsize > huge(1)/10. ) then

         call PrintMessage( "cubic_bezier_pix", "W",                    &
                            "truncate real_maxsize to huge(1)/10!",    &
                            "(perhaps your control points contain Inf or very great values)" )
         maxsize = huge(1)/10.
      else
         maxsize = real_maxsize
      end if
      allocate( pt(0:maxsize-1), stat=alloc_stat )
      if( alloc_stat /= 0 ) then

         call PrintMessage( "cubic_bezier_pix", "W",                    &
                            "cannot allocate sufficient workspace!",   &
                            "(perhaps your control points contain NaN, Inf or very great values)",&
                            "-> returning initial control points without subdivisions." )

         ! return initial control points
         pa_size = 8
         allocate( pa(0:pa_size-1) )
         do i = 0, pa_size-1
            pa(i) = ctrl(i)
         end do

      else

         pt_size = 0

         call polygonize_cubic_bezier( pt, pt_size, ctrl, maxsize )
         pt(pt_size) = ctrl(6); pt_size = pt_size + 1
         pt(pt_size) = ctrl(7); pt_size = pt_size + 1

         pa_size = pt_size
         allocate( pa(0:pa_size-1) )
         do i = 0, pa_size-1
            pa(i) = pt(i)
         end do

      end if

   end subroutine cubic_bezier_pix
!_______________________________________________________________________
!
   recursive subroutine polygonize_quad_bezier ( acc, accsize, ctrl, maxsize )

      integer,          intent(in)     :: maxsize
      double precision, intent(in out) :: acc(0:maxsize-1)
      integer,          intent(in out) :: accsize
      double precision, intent(in)     :: ctrl(0:5)
      !------ API end ------

      integer :: c0(0:1), c1(0:1), c2(0:1)
      double precision :: l(0:5), r(0:5)

   !------ end of declarations -- execution starts hereafter  ------

      if ( accsize > maxsize / 2 ) then
         if ( accsize >= maxsize-4 ) then
            return
         end if
         !
         ! Running out of space - approximate by a line.
         !
         acc(accsize) = ctrl(0); accsize = accsize + 1
         acc(accsize) = ctrl(1); accsize = accsize + 1
         return
      end if

      !
      ! convert to integers for line condition check
      !
      c0(0) = ctrl(0); c0(1) = ctrl(1)
      c1(0) = ctrl(2); c1(1) = ctrl(3)
      c2(0) = ctrl(4); c2(1) = ctrl(5)

      !
      ! is control polygon inside a 2x2-pixels domain?
      !
      if ( abs(c1(0)-c0(0)) <= 1 .and. abs(c1(1)-c0(1)) <= 1 .and.      &
           abs(c2(0)-c0(0)) <= 1 .and. abs(c2(1)-c0(1)) <= 1 ) then
         !
         ! Approximate by one line.
         ! Dont need to write last pt as it is the same as first pt
         ! on the next segment
         !
         acc(accsize) = ctrl(0); accsize = accsize + 1
         acc(accsize) = ctrl(1); accsize = accsize + 1
         return
      end if

      !
      ! are the three control points aligned (up to a pixel)?
      !
      if ( pnt_on_line( c0, c2, c1 ) == 2       ) then
         !
         ! Approximate by one line.
         ! Don't need to write last pt as it is the same as first pt
         ! on the next segment
         !
         acc(accsize) = ctrl(0); accsize = accsize + 1
         acc(accsize) = ctrl(1); accsize = accsize + 1
         return
      end if

      !
      ! Too curved - recursively subdivide.
      !
      call split_quadratic( ctrl, l, r )
      call polygonize_quad_bezier( acc, accsize, l, maxsize )
      call polygonize_quad_bezier( acc, accsize, r, maxsize )

   end subroutine polygonize_quad_bezier
!_______________________________________________________________________
!
   recursive subroutine polygonize_cubic_bezier ( acc, accsize, ctrl, maxsize )

      integer,          intent(in)     :: maxsize
      double precision, intent(in out) :: acc(0:maxsize-1)
      integer,          intent(in out) :: accsize
      double precision, intent(in)     :: ctrl(0:7)
      !------ API end ------

      integer :: c0(0:1), c1(0:1), c2(0:1), c3(0:1)
      double precision :: l(0:7), r(0:7)

   !------ end of declarations -- execution starts hereafter  ------

      if ( accsize > maxsize / 2 ) then
         if ( accsize >= maxsize-4 ) then
            return
         end if
         !
         ! Running out of space - approximate by a line.
         !
         acc(accsize) = ctrl(0); accsize = accsize + 1
         acc(accsize) = ctrl(1); accsize = accsize + 1
         return
      end if

      !
      ! convert to integers for line condition check
      !
      c0(0) = ctrl(0); c0(1) = ctrl(1)
      c1(0) = ctrl(2); c1(1) = ctrl(3)
      c2(0) = ctrl(4); c2(1) = ctrl(5)
      c3(0) = ctrl(6); c3(1) = ctrl(7)

      !
      ! is control polygon inside a 2x2-pixels domain?
      !
      if ( abs(c1(0)-c0(0)) <= 1 .and. abs(c1(1)-c0(1)) <= 1 .and.      &
           abs(c2(0)-c0(0)) <= 1 .and. abs(c2(1)-c0(1)) <= 1 .and.      &
           abs(c3(0)-c1(0)) <= 1 .and. abs(c3(1)-c0(1)) <= 1 ) then
         !
         ! Approximate by one line.
         ! Dont need to write last pt as it is the same as first pt
         ! on the next segment
         !
         acc(accsize) = ctrl(0); accsize = accsize + 1
         acc(accsize) = ctrl(1); accsize = accsize + 1
         return
      end if

      !
      ! are the four control points aligned (up to a pixel)?
      !
      if ( pnt_on_line( c0, c3, c1 ) == 2 .and.                         &
           pnt_on_line( c0, c3, c2 ) == 2       ) then
         !
         ! Approximate by one line.
         ! Don't need to write last pt as it is the same as first pt
         ! on the next segment
         !
         acc(accsize) = ctrl(0); accsize = accsize + 1
         acc(accsize) = ctrl(1); accsize = accsize + 1
         return
      end if

      !
      ! Too curved - recursively subdivide.
      !
      call split_cubic( ctrl, l, r )
      call polygonize_cubic_bezier( acc, accsize, l, maxsize )
      call polygonize_cubic_bezier( acc, accsize, r, maxsize )

   end subroutine polygonize_cubic_bezier
!_______________________________________________________________________
!
   function pnt_on_line ( p, q, t )

      integer, intent(in) :: p(0:1), q(0:1), t(0:1)
      integer             :: pnt_on_line
      !------ API end ------

      !  given a line through P:(px,py) Q:(qx,qy) and T:(tx,ty)
      !  return 0 if T is not on the line through      <--P--Q-->
      !         1 if T is on the open ray ending at P: <--P
      !         2 if T is on the closed interior along:   P--Q
      !         3 if T is on the open ray beginning at Q:    Q-->
      !
      !  Example: consider the line P = (3,2), Q = (17,7). A plot
      !  of the test points T(x,y) (with 0 mapped onto '.') yields:
      !
      !      8| . . . . . . . . . . . . . . . . . 3 3
      !   Y  7| . . . . . . . . . . . . . . 2 2 Q 3 3    Q = 2
      !      6| . . . . . . . . . . . 2 2 2 2 2 . . .
      !   a  5| . . . . . . . . 2 2 2 2 2 2 . . . . .
      !   x  4| . . . . . 2 2 2 2 2 2 . . . . . . . .
      !   i  3| . . . 2 2 2 2 2 . . . . . . . . . . .
      !   s  2| 1 1 P 2 2 . . . . . . . . . . . . . .    P = 2
      !      1| 1 1 . . . . . . . . . . . . . . . . .
      !       +--------------------------------------
      !         1 2 3 4 5 X-axis 10        15      19
      !
      !  Point-Line distance is normalized with the Infinity Norm
      !  avoiding square-root code and tightening the test vs the
      !  Manhattan Norm. All math is done on the field of integers.
      !  The latter replaces the initial ">= MAX(...)" test with
      !  "> (ABS(qx-px) + ABS(qy-py))" loosening both inequality
      !  and norm, yielding a broader target line for selection.
      !  The tightest test is employed here for best discrimination
      !  in merging collinear (to grid coordinates) vertex chains
      !  into a larger, spanning vectors within the Lemming editor.

   !------ end of declarations -- execution starts hereafter  ------

      if ( q(0)==p(0) .and. q(1)==p(1) .and.                            &
           q(0)==t(0) .and. q(1)==t(1) ) then
         pnt_on_line = 2
         return
      end if

      if ( abs((q(1)-p(1))*(t(0)-p(0))-(t(1)-p(1))*(q(0)-p(0))) >=      &
           max( abs(q(0)-p(0)), abs(q(1)-p(1)) ) ) then
         pnt_on_line = 0
         return
      end if

      if ( ( q(0)<p(0) .and. p(0)<t(0) ) .or.                           &
           ( q(1)<p(1) .and. p(1)<t(1) ) ) then
         pnt_on_line = 1
         return
      end if

      if ( ( t(0)<p(0) .and. p(0)<q(0) ) .or.                           &
           ( t(1)<p(1) .and. p(1)<q(1) ) ) then
         pnt_on_line = 1
         return
      end if

      if ( ( p(0)<q(0) .and. q(0)<t(0) ) .or.                           &
           ( p(1)<q(1) .and. q(1)<t(1) ) ) then
         pnt_on_line = 3
         return
      end if

      if ( ( t(0)<q(0) .and. q(0)<p(0) ) .or.                           &
           ( t(1)<q(1) .and. q(1)<p(1) ) ) then
         pnt_on_line = 3
         return
      end if

      pnt_on_line = 2

   end function pnt_on_line
!_______________________________________________________________________
!
   subroutine split_quadratic ( p, l, r )

      double precision, intent(in)  :: p(0:5)
      double precision, intent(out) :: l(0:5), r(0:5)
      !------ API end ------

      ! basic (internal) Bézier algorithm for subdivision a quadratic segment
      !
      ! in  : array p
      !       3 initial control points
      !
      ! out : l[eft] and r[ight] arrays
      !       control points after subdivision
      !       (3 on left and 3 on right, sharing 1 point)

   !------ end of declarations -- execution starts hereafter  ------

      l(0) =  p(0)
      l(1) =  p(1)
      r(4) =  p(4)
      r(5) =  p(5)

      l(2) = (p(0)+ p(2))/2.0d0
      l(3) = (p(1)+ p(3))/2.0d0
      r(2) = (p(2)+ p(4))/2.0d0
      r(3) = (p(3)+ p(5))/2.0d0

      l(4) = (l(2)+ r(2))/2.0d0
      l(5) = (l(3)+ r(3))/2.0d0
      r(0) = l(4)
      r(1) = l(5)

   end subroutine split_quadratic
!_______________________________________________________________________
!
   subroutine split_cubic ( p, l, r )

      double precision, intent(in)  :: p(0:7)
      double precision, intent(out) :: l(0:7), r(0:7)
      !------ API end ------

      ! basic (internal) Bézier algorithm for subdivision a cubic segment
      !
      ! in  : array p
      !       4 initial control points
      !
      ! out : l[eft] and r[ight] arrays
      !       control points after subdivision
      !       (4 on left and 4 on right, sharing 1 point)

      double precision :: tmpx, tmpy

   !------ end of declarations -- execution starts hereafter  ------

      l(0) =  p(0)
      l(1) =  p(1)
      r(6) =  p(6)
      r(7) =  p(7)

      l(2) = (p(0)+ p(2))/2.0d0
      l(3) = (p(1)+ p(3))/2.0d0
      tmpx = (p(2)+ p(4))/2.0d0
      tmpy = (p(3)+ p(5))/2.0d0
      r(4) = (p(4)+ p(6))/2.0d0
      r(5) = (p(5)+ p(7))/2.0d0

      l(4) = (l(2)+ tmpx)/2.0d0
      l(5) = (l(3)+ tmpy)/2.0d0
      r(2) = (tmpx + r(4))/2.0d0
      r(3) = (tmpy + r(5))/2.0d0

      l(6) = (l(4)+ r(2))/2.0d0
      l(7) = (l(5)+ r(3))/2.0d0
      r(0) = l(6)
      r(1) = l(7)

   end subroutine split_cubic
!_______________________________________________________________________
!
end module bezier
