module convexhull_mod

   ! Algorithm for computing the convex hull of a set of points.
   ! Taken from the Graham Scan implementation of (c) Miguel Vieira, 2014
   ! (MIT license), translated from C++ to F90, improved and adapted to my
   ! needs.
   !
   ! v7
   !
   ! (c) Édouard Canot -- IPR/CNRS -- 2022-12-18

   implicit none

   private

   double precision :: p0_x, p0_y

   ! Makes public only what is necessary
   public :: CH_convex_hull, CH_ccw, CH_isLeftOf

contains
   !____________________________________________________________________
   !
   function CH_ccw( ax, ay, bx, by, cx, cy ) result( res)

      double precision, intent(in) :: ax, ay, bx, by, cx, cy
      double precision             :: res

      res = (bx - ax) * (cy - ay) - (by - ay) * (cx - ax)

   end function CH_ccw
   !____________________________________________________________________
   !
   function CH_isLeftOf( ax, ay, bx, by ) result( res )

      double precision, intent(in) :: ax, ay, bx, by
      logical                      :: res

      if( ax < bx .or. (ax == bx .and. ay < by) ) then
         res = .true.
      else
         res = .false.
      end if

   end function CH_isLeftOf
   !____________________________________________________________________
   !
   function isBefore( ax, ay, bx, by, cx, cy ) result( res)

      double precision, intent(in) :: ax, ay, bx, by, cx, cy
      logical                      :: res

      double precision :: det, dist2_ab, dist2_ac

      det = (bx - ax) * (cy - ay) - (by - ay) * (cx - ax)
      if( det < 0. ) then
         res = .true.
      else if( det == 0. ) then
         dist2_ab = (ax - bx)**2 + (ay - by)**2
         dist2_ac = (ax - cx)**2 + (ay - cy)**2
         if( dist2_ab < dist2_ac ) then
            res = .true.
         else
            res = .false.
         end if
      else
         res = .false.
      end if

   end function isBefore
   !____________________________________________________________________
   !
   recursive subroutine quick_sort( Ax, Ay, ind )

      ! isBefore( p, a, b )  ~  a < b
      !
      ! in all cases a /= b else it should mean that two points are the same

      double precision, intent(in)     :: Ax(:), Ay(:)
      integer,          intent(in out) :: ind(:)

      integer :: ipiv, n, itemp

      n = size(ind)

      if( n > 2 ) then
         call partition_sort( Ax, Ay, ind, ipiv )
         if( 2 <= ipiv-1 ) then
            call quick_sort( Ax, Ay, ind(:ipiv-1) )
         end if
         if( ipiv+1 <= n-1 ) then
            call quick_sort( Ax, Ay, ind(ipiv+1:) )
         end if
      else if( n == 2 ) then
         if( isBefore(p0_x,p0_y,Ax(ind(2)),Ay(ind(2)),Ax(ind(1)),Ay(ind(1))) ) then
            itemp  = ind(1)
            ind(1) = ind(2)
            ind(2) = itemp
         end if
      end if

   end subroutine quick_sort
   !____________________________________________________________________
   !
   subroutine partition_sort( Ax, Ay, ind, ipiv )

      double precision, intent(in)     :: Ax(:), Ay(:)
      integer,          intent(in out) :: ind(:)
      integer,          intent(out)    :: ipiv

      integer :: i, j, n, itemp, ip
      double precision :: piv_x, piv_y ! pivot

      n = size(ind)
      ip = n/2+1
      piv_x = Ax(ind(ip))
      piv_y = Ay(ind(ip))
      i = 1
      j = n

      do
         do
            if( i == ip ) exit
            if( isBefore(p0_x,p0_y,piv_x,piv_y,Ax(ind(i)),Ay(ind(i))) ) exit
            i = i + 1
         end do
         do
            if( j == ip ) exit
            if( isBefore(p0_x,p0_y,Ax(ind(j)),Ay(ind(j)),piv_x,piv_y) ) exit
            j = j - 1
         end do
         if( i < j ) then
            ! exchange indices for A(i) and A(j)
            itemp  = ind(i)
            ind(i) = ind(j)
            ind(j) = itemp
            if( i == ip ) then
               ip = j
            else if( j == ip ) then
               ip = i
            end if
         else
            if( i == ip ) then
               ipiv = i
            else
               ipiv = j
            end if
            return
         end if
     end do

   end subroutine partition_sort
   !____________________________________________________________________
   !
   subroutine CH_convex_hull( vx, vy, ind, ind_size )

      ! This is the only entry point from the user point-of-view.
      !
      ! Input : Two arrays vx(:) and vy(:) containing the (x,y) coordinates
      !         of points in the 2D plane.
      !
      ! Ouput : The convex hull of the input set of points, described by
      !         the array of indices ind(:).
      !
      ! Arrays 'vx', 'vy' and 'ind' must be allocated by the calling routine
      ! to the same size, that is the number of points.
      !
      ! On output, the effective number of points of the convex hull is
      ! stored in the integer ind_size.

      double precision, intent(in)  :: vx(:), vy(:)
      integer,          intent(out) :: ind(:), ind_size

      !------------------------

      integer :: i, j, n, v_it
      integer, allocatable :: perm(:)

      n = size(ind)

      ! Initialize the permutation vector.
      allocate( perm(n) )
      do i = 1, n
         perm(i) = i
      end do

      ! Put our leftmost point at first position.
      j = 1
      do i = 2, n
         if( CH_isLeftOf(vx(i),vy(i),vx(j),vy(j)) ) then
            j = i
         end if
      end do
      if( j /= 1 ) then
         perm(1) = j
         perm(j) = 1
      end if

      ! Sort the rest of points in counter-clockwise order (taking into
      ! account the collinear points), and get the associated permutation.
      p0_x = vx(j)
      p0_y = vy(j)
      call quick_sort( vx, vy, perm(2:n) )

      ! Add the first two points to the hull.
      ind(1) = 1
      ind(2) = 2
      ind_size = 2
      v_it = 3
      ! Try to find another point, not collinear to the two first.
      do while( CH_ccw( vx(perm(ind(1))), vy(perm(ind(1))),             &
                        vx(perm(ind(2))), vy(perm(ind(2))),             &
                        vx(perm(v_it)), vy(perm(v_it)) ) == 0. )
         ind(2) = v_it
         v_it = v_it + 1
      end do
      ind(3) = v_it
      v_it = v_it + 1
      ind_size = 3

      do while( v_it <= n )
         ! Pop off any point that makes a convex or null angle
         ! with point #i of v.
         do while( CH_ccw( vx(perm(ind(ind_size-1))), vy(perm(ind(ind_size-1))), &
                           vx(perm(ind(ind_size))), vy(perm(ind(ind_size))), &
                           vx(perm(v_it)), vy(perm(v_it)) ) >= 0. )
            if( ind_size == 2 ) exit
            ind_size = ind_size - 1
         end do
         ind_size = ind_size + 1
         ind(ind_size) = v_it
         v_it = v_it + 1
      end do

      ind(1:ind_size) = perm(ind(1:ind_size))

   end subroutine CH_convex_hull
   !____________________________________________________________________
   !
end module
