#include <fintrf.h>
! mssave_gateway.f90 - MATLAB gateway function for 'mbfwrite'
!
! This subroutine is the main gateway to MATLAB.  When a MEX function
! is executed MATLAB calls the MEXFUNCTION subroutine in the
! corresponding MEX file.
!
! Ref: apiext.pdf, apiref.pdf
!
! MATLAB 7.13 (R2011b) (unknown compatibility for previous versions)
! For both 32- and 64-bit machines.
!
! version: 2012-03-09
!
! Copyright É. Canot 2003-2025 -- IPR/CNRS

   subroutine mexfunction( nlhs, plhs, nrhs, prhs )

      implicit none

      mwPointer :: plhs(*), prhs(*)
      integer :: nlhs, nrhs

      mwPointer :: mxGetPr, mxGetPi, mxGetIr, mxGetJc
      mwPointer :: pr, pi, ir, jc

      mwSize :: mxGetM, mxGetN, mxGetNumberOfElements, mxGetNzmax
      mwSize :: m, n, nnz, str_len
      mwSize, pointer :: ia(:), ja(:)

      logical :: mxIsChar, mxIsDouble, mxIsComplex, mxIsSparse

      ! must match the main declaration in $MUESLI/src/mod_mfdebug.F90
      integer, parameter :: MF_DOUBLE = kind(1.0d0)

      integer, parameter :: len_max = 256
      character(len=len_max) :: filename, message
      real(kind=MF_DOUBLE), pointer :: double(:,:)
      complex(kind=MF_DOUBLE), pointer :: complx(:,:)
#ifdef _64_BITS
      integer :: m4, n4, nnz4
      integer, pointer :: ia4(:), ja4(:)
#endif
      real(kind=MF_DOUBLE), pointer :: ra(:)
      complex(kind=MF_DOUBLE), pointer :: ca(:)

      interface
         subroutine mbfwrite( filename, nrow, ncol, nnz, data_type,     &
                              double, complx, ia, ja, ra, ca )
            character(len=*), intent(in) :: filename
            integer, intent(in) :: nrow, ncol, nnz, data_type
            ! must match the main declaration in $MUESLI/src/mod_mfdebug.F90
            integer, parameter :: MF_DOUBLE = kind(1.0d0)
            real(kind=MF_DOUBLE), pointer :: double(:,:)
            complex(kind=MF_DOUBLE), pointer :: complx(:,:)
            integer, pointer :: ia(:), ja(:)
            real(kind=MF_DOUBLE), pointer :: ra(:)
            complex(kind=MF_DOUBLE), pointer :: ca(:)
         end subroutine mbfwrite
      end interface
!-----------------------------------------------------------------------

      ! must match the main declaration in $MUESLI/src/mod_mfarray.F90
      integer, parameter :: MF_DT_EMPTY    = 0,                         &
                            MF_DT_DBLE     = 1,                         &
                            MF_DT_CMPLX    = 2,                         &
                            MF_DT_BOOL     = 3,                         &
                            MF_DT_SP_DBLE  = 4,                         &
                            MF_DT_SP_CMPLX = 5,                         &
                            MF_DT_PERM_VEC = 6

      integer :: data_type

      ! Check for proper number of arguments
      if( nrhs /= 2 ) then
         call mexerrmsgtxt("  -> two input arguments must be present!")
      end if

      ! Check arguments
      if( .not. mxIsChar(prhs(1)) ) then
         call mexerrmsgtxt("  -> first argument must be a string!")
      end if

      m = mxGetM( prhs(1) )
      n = mxGetN( prhs(1) )

      if( m /= 1 .or. n == 0 ) then
         call mexerrmsgtxt("  -> empty string!")
      end if

      str_len = mxGetNumberOfElements( prhs(1) )
      if( str_len > len_max ) then
         write(message,"(A,I0,A)") "  -> first argument is too long! (max. is currently: ", &
                                   len_max, " chars)\n"
         message = trim(message)                                        &
                   // "  -> if you have to work with such a long string," &
                   // " please consider to recompile 'mssave_gateway.F90'" &
                   // " in MUESLI library."
         call mexerrmsgtxt(message)
      end if

      ! Copy the string data and place it into 'filename'
      filename = ""
      call mxGetString( prhs(1), filename, str_len )

      if( .not. mxIsDouble(prhs(2)) ) then
         call mexerrmsgtxt("  -> second arg must be a numeric array [Class double]!")
      end if

      if( mxIsSparse(prhs(2)) ) then
         if( mxIsComplex(prhs(2)) ) then
            data_type = MF_DT_SP_CMPLX
         else ! real
            data_type = MF_DT_SP_DBLE
         end if
      else ! dense
         if( mxIsComplex(prhs(2)) ) then
            data_type = MF_DT_CMPLX
         else ! real
            data_type = MF_DT_DBLE
         end if
      end if

      ! get array size
      m = mxGetM( prhs(2) )
      n = mxGetN( prhs(2) )

      if( data_type == MF_DT_DBLE ) then ! real case

         ! Assign pointer to the real part of RHS(2)
         pr = mxGetPr( prhs(2) )
         ! Copy output to matrix output
         allocate( double(m,n) )
         call mxCopyPtrToReal8( pr, double, m*n )

      else if( data_type == MF_DT_CMPLX ) then ! complex case

         ! Assign pointers to the real and imag part of RHS(2)
         pr = mxGetPr( prhs(2) )
         pi = mxGetPi( prhs(2) )
         ! Copy output to matrix output
         allocate( complx(m,n) )
         call mxCopyPtrToComplex16( pr, pi, complx, m*n )

      else if( data_type == MF_DT_SP_DBLE ) then ! complex case

         ! Assign pointer to each part of RHS(2)
         jc = mxGetJc( prhs(2) )
         ir = mxGetIr( prhs(2) )
         pr = mxGetPr( prhs(2) )
         nnz = mxGetNzmax( prhs(2) )
         ! Copy output to matrix output
         allocate( ja(n+1), ia(nnz), ra(nnz) )
         ! Be aware to Fortran-C offset for indexes!
#ifdef _64_BITS
         call mxCopyPtrToInteger8( jc, ja, n+1 )
         allocate( ja4(n+1) )
         ja4(:) = ja(:) + 1
         deallocate( ja )
         call mxCopyPtrToInteger8( ir, ia, nnz )
         allocate( ia4(nnz) )
         ia4(:) = ia(:) + 1
         deallocate( ia )
         nnz4 = nnz
#else
         call mxCopyPtrToInteger4( jc, ja, n+1 )
         call mxCopyPtrToInteger4( ir, ia, nnz )
         ja(:) = ja(:) + 1
         ia(:) = ia(:) + 1
#endif
         call mxCopyPtrToReal8( pr, ra, nnz )

      else if( data_type == MF_DT_SP_CMPLX ) then ! complex case

         ! Assign pointer to each part of RHS(2)
         jc = mxGetJc( prhs(2) )
         ir = mxGetIr( prhs(2) )
         pr = mxGetPr( prhs(2) )
         pi = mxGetPi( prhs(2) )
         nnz = mxGetNzmax( prhs(2) )
         ! Copy output to matrix output
         allocate( ja(n+1), ia(nnz), ca(nnz) )
         ! Be aware to Fortran-C offset for indexes!
#ifdef _64_BITS
         call mxCopyPtrToInteger8( jc, ja, n+1 )
         allocate( ja4(n+1) )
         ja4(:) = ja(:) + 1
         deallocate( ja )
         call mxCopyPtrToInteger8( ir, ia, nnz )
         allocate( ia4(nnz) )
         ia4(:) = ia(:) + 1
         deallocate( ia )
         nnz4 = nnz
#else
         call mxCopyPtrToInteger4( jc, ja, n+1 )
         call mxCopyPtrToInteger4( ir, ia, nnz )
         ja(:) = ja(:) + 1
         ia(:) = ia(:) + 1
#endif
         call mxCopyPtrToComplex16( pr, pi, ca, nnz )

      end if

#ifdef _64_BITS
      m4 = m
      n4 = n
      call mbfwrite( filename, m4, n4, nnz4, data_type,                 &
                     double, complx, ia4, ja4, ra, ca )
#else
      call mbfwrite( filename, m, n, nnz, data_type,                    &
                     double, complx, ia, ja, ra, ca )
#endif

   end subroutine mexfunction
