!----------------------------------------------------------------------!
!                                                                      !
! This subroutine comes from the MUESLI 'msSave()' subroutine.         !
!                                                                      !
! Part of MUESLI Numerical Library                                     !
! Copyright É. Canot 2003-2025 -- IPR/CNRS                           !
!                                                                      !
!----------------------------------------------------------------------!

!_______________________________________________________________________
!
   subroutine mbfwrite( filename, nrow, ncol, nnz, data_type,           &
                        double, complx, ia, ja, ra, ca )

      ! arg. in  : filename
      !
      ! arg. in  : double OR complx
      !               2-d real or complex array (double precision)
      !            nrow, ncol
      !               array shape
      !            data_type
      !               MF_DT_EMPTY    :
      !               MF_DT_DBLE     :        real
      !               MF_DT_CMPLX    :        complex
      !               MF_DT_SP_DBLE  : sparse real     {ia,ja,ra}
      !               MF_DT_SP_CMPLX : sparse complex  {ia,ja,ca}
      !
      ! MBF format written: 2.4
      !
      ! version: 2013-05-16
      !-------------------------------------------------------------

      use f90_gzlib

      use rational_numbers

      implicit none

      ! must match the main declaration in $MUESLI/src/mod_mfdebug.F90
      integer, parameter :: MF_DOUBLE = kind(1.0d0)
      integer, parameter :: kind_1 = selected_int_kind(r=2)
      integer(kind=kind_1), parameter :: UNKNOWN = -1

      ! must match the main declaration in $MUESLI/src/mod_mfarray.F90
      integer, parameter :: MF_DT_EMPTY    = 0,                         &
                            MF_DT_DBLE     = 1,                         &
                            MF_DT_CMPLX    = 2,                         &
                            MF_DT_BOOL     = 3,                         &
                            MF_DT_SP_DBLE  = 4,                         &
                            MF_DT_SP_CMPLX = 5,                         &
                            MF_DT_PERM_VEC = 6

      character(len=*), intent(in) :: filename
      integer, intent(in) :: nrow, ncol, nnz, data_type
      integer(kind=kind_1) :: byte(4)
      real(kind=MF_DOUBLE), pointer :: double(:,:)
      complex(kind=MF_DOUBLE), pointer :: complx(:,:)
      integer, pointer :: ia(:), ja(:)
      real(kind=MF_DOUBLE), pointer :: ra(:)
      complex(kind=MF_DOUBLE), pointer :: ca(:)

      !-----------------------------------------------------------------

      ! signature for MF release >= 2.8.0
      character(len=20), parameter :: MF_BIN_SIGN_24 = "MF-2.4_ECanot_CNRS"

      ! little endian
      integer, parameter :: MF_ENDIAN_NATIVE = 875770417 ! = "1234"

      integer :: unit, l, bytes_written

      logical :: gzipped
      type(gz_filedes) :: gz_file

      !-----------------------------------------------------------------

      ! trying to detect compress mode from filename suffix
      l = len_trim(filename)
      if( filename(l-2:l) == ".gz" ) then
         gzipped = .true.
      else
         gzipped = .false.
      end if

      if( gzipped ) then

         gz_file = gzopen( trim(filename), "wb" )

         if( filedes_is_null( gz_file ) ) then
            return
         end if

         bytes_written = 0

         ! remark : in sequential-unformatted write, each record
         !          are preceeded and followed by the record size.

         ! signature :  bytes = 4+20+4 = 28
         bytes_written = bytes_written + 28
         l = len(MF_BIN_SIGN_24)
         call gzwrite( gz_file, l )
         call gzwrite( gz_file, MF_BIN_SIGN_24 )
         call gzwrite( gz_file, l )

         ! endianness : bytes = 4+4+4 = 12
         bytes_written = bytes_written + 12
         l = 4
         call gzwrite( gz_file, l )
         call gzwrite( gz_file, MF_ENDIAN_NATIVE )
         call gzwrite( gz_file, l )

         ! data_type : bytes = 4+4+4 = 12
         bytes_written = bytes_written + 12
         l = 4
         call gzwrite( gz_file, l )
         call gzwrite( gz_file, data_type )
         call gzwrite( gz_file, l )
         if( data_type == MF_DT_EMPTY ) then ! MF_EMPTY
            call gzclose( gz_file )
            return
         end if

         ! array shape : bytes = 4+8+4 = 16
         bytes_written = bytes_written + 16
         l = 8
         call gzwrite( gz_file, l )
         call gzwrite( gz_file, nrow )
         call gzwrite( gz_file, ncol )
         call gzwrite( gz_file, l )

         ! stored size so far: 28+12+12+16=68
         ! adding 4 for next: 68+4=72 <---- must be a multiple of 8 !

         ! array data (column-wise in Fortran)
         if( data_type == MF_DT_DBLE ) then ! real case
            l = 8*nrow*ncol
            call gzwrite( gz_file, l )
            call gzwrite( gz_file, double )
            call gzwrite( gz_file, l )
            deallocate( double )
         else if( data_type == MF_DT_CMPLX ) then
            l = 16*nrow*ncol
            call gzwrite( gz_file, l )
            call gzwrite( gz_file, complx )
            call gzwrite( gz_file, l )
            deallocate( complx )
         else if( data_type == MF_DT_SP_DBLE ) then
            ! nnz : bytes = 4+4+4 = 12
            bytes_written = bytes_written + 12
            l = 4
            call gzwrite( gz_file, l )
            call gzwrite( gz_file, nnz )
            call gzwrite( gz_file, l )
            ! ja : bytes = 4 + 4*(ncol+1) + 4
            bytes_written = bytes_written + 4 + 4*(ncol+1) + 4
            l = 4*(ncol+1)
            call gzwrite( gz_file, l )
            call gzwrite( gz_file, ja(1:ncol+1) )
            call gzwrite( gz_file, l )
            ! ia : bytes = 4 + 4*(nnz) + 4
            bytes_written = bytes_written + 4 + 4*(nnz) + 4
            ! for reading 'ia' (in direct access and swap_bytes case)
            ! by 'mfLoad', bytes_written+4 must be a multiple of 8.
            if( mod(bytes_written+4,8) /= 0 ) then
               l = 4*nnz + 4
            else
               l = 4*nnz
            end if
            call gzwrite( gz_file, l )
            call gzwrite( gz_file, ia(1:nnz) )
            if( mod(bytes_written+4,8) /= 0 ) then
               call gzwrite( gz_file, 0 )
            end if
            call gzwrite( gz_file, l )
            l = 8*nnz
            call gzwrite( gz_file, l )
            call gzwrite( gz_file, ra(1:nnz) )
            call gzwrite( gz_file, l )
            deallocate( ia, ja, ra )
         else if( data_type == MF_DT_SP_CMPLX ) then
            ! nnz : bytes = 4+4+4 = 12
            bytes_written = bytes_written + 12
            l = 4
            call gzwrite( gz_file, l )
            call gzwrite( gz_file, nnz )
            call gzwrite( gz_file, l )
            ! ja : bytes = 4 + 4*(ncol+1) + 4
            bytes_written = bytes_written + 4 + 4*(ncol+1) + 4
            l = 4*(ncol+1)
            call gzwrite( gz_file, l )
            call gzwrite( gz_file, ja(1:ncol+1) )
            call gzwrite( gz_file, l )
            ! ia : bytes = 4 + 4*(nnz) + 4
            bytes_written = bytes_written + 4 + 4*(nnz) + 4
            ! for reading 'ia' (in direct access and swap_bytes case)
            ! by 'mfLoad', bytes_written+4 must be a multiple of 8.
            if( mod(bytes_written+4,8) /= 0 ) then
               l = 4*nnz + 4
            else
               l = 4*nnz
            end if
            call gzwrite( gz_file, l )
            call gzwrite( gz_file, ia(1:nnz) )
            if( mod(bytes_written+4,8) /= 0 ) then
               call gzwrite( gz_file, 0 )
            end if
            call gzwrite( gz_file, l )
            l = 16*nnz
            call gzwrite( gz_file, l )
            call gzwrite( gz_file, ca(1:nnz) )
            call gzwrite( gz_file, l )
            deallocate( ia, ja, ca )
         end if

         ! all properties written are unknown
         l = (4*1)
         call gzwrite( gz_file, l )
         call gzwrite( gz_file, UNKNOWN )
         call gzwrite( gz_file, UNKNOWN )
         call gzwrite( gz_file, UNKNOWN )
         call gzwrite( gz_file, UNKNOWN )
         call gzwrite( gz_file, l )

         ! below, 7 is the number of physical units used in MUESLI
         l = (2*2)*7
         call gzwrite( gz_file, l )

         ! physical unit (as it is not available in Matlab, stores a
         ! dimensionless mfArray)
         call gzwrite( gz_file, [ RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ] )
         call gzwrite( gz_file, l )

         call gzclose( gz_file )

      else ! not gzipped

         unit = 99

         open(unit=unit,file=trim(adjustl(filename)),form="unformatted")

         bytes_written = 0

         ! signature :  bytes = 4+20+4 = 28
         bytes_written = bytes_written + 28
         write(unit) MF_BIN_SIGN_24

         ! endianness : bytes = 4+4+4 = 12
         bytes_written = bytes_written + 12
         write(unit) MF_ENDIAN_NATIVE

         ! data_type : bytes = 4+4+4 = 12
         bytes_written = bytes_written + 12
         write(unit) data_type
         if( data_type == MF_DT_EMPTY ) then ! MF_EMPTY
            close(unit)
            return
         end if

         ! array shape : bytes = 4+8+4 = 16
         bytes_written = bytes_written + 16
         write(unit) nrow, ncol

         ! stored size so far: 28+12+12+16=68
         ! adding 4 from next: 68+4=72 <---- must be a multiple of 8 !

         ! array data (column-wise in Fortran)
         if( data_type == MF_DT_DBLE ) then ! real case
            write(unit) double(:,:)
            deallocate( double )
         else if( data_type == MF_DT_CMPLX ) then
            write(unit) complx(:,:)
            deallocate( complx )
         else if( data_type == MF_DT_SP_DBLE ) then
            ! nnz : bytes = 4+4+4 = 12
            bytes_written = bytes_written + 12
            write(unit) nnz
            ! ja : bytes = 4 + 4*(ncol+1) + 4
            bytes_written = bytes_written + 4 + 4*(ncol+1) + 4
            write(unit) ja(1:ncol+1)
            ! ia : bytes = 4 + 4*(nnz) + 4
            bytes_written = bytes_written + 4 + 4*(nnz) + 4
            ! for reading 'ia' (in direct access and swap_bytes case)
            ! by 'mfLoad', bytes_written+4 must be a multiple of 8.
            if( mod(bytes_written+4,8) == 0 ) then
               write(unit) ia(1:nnz)
            else
               write(unit) ia(1:nnz), 0
               bytes_written = bytes_written + 4
            end if
            write(unit) ra(1:nnz)
            deallocate( ia, ja, ra )
         else if( data_type == MF_DT_SP_CMPLX ) then
            ! nnz : bytes = 4+4+4 = 12
            bytes_written = bytes_written + 12
            write(unit) nnz
            ! ja : bytes = 4 + 4*(ncol+1) + 4
            bytes_written = bytes_written + 4 + 4*(ncol+1) + 4
            write(unit) ja(1:ncol+1)
            ! ia : bytes = 4 + 4*(nnz) + 4
            bytes_written = bytes_written + 4 + 4*(nnz) + 4
            ! for reading 'ia' (in direct access and swap_bytes case)
            ! by 'mfLoad', bytes_written+4 must be a multiple of 8.
            if( mod(bytes_written+4,8) == 0 ) then
               write(unit) ia(1:nnz)
            else
               write(unit) ia(1:nnz), 0
               bytes_written = bytes_written + 4
            end if
            write(unit) ca(1:nnz)
            deallocate( ia, ja, ca )
         end if

         ! matrix pattern and properties
         ! (all UNKNOWN in MATLAB)
         byte(1:4) = [ UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN ]
         write(unit) transfer(byte,0)

         ! physical unit (as it is not available in Matlab, stores a
         ! dimensionless mfArray)
         ! CHECK that 'num_base_units' in MUESLI is equal to 7
         write(unit) [ RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0, RAT_0 ]

         close(unit)

      end if

   end subroutine mbfwrite
