!----------------------------------------------------------------------!
!                                                                      !
! This subroutine comes from the MUESLI 'mfLoad()' function.           !
!                                                                      !
! Part of MUESLI Numerical Library                                     !
! Copyright É. Canot 2003-2025 -- IPR/CNRS                           !
!                                                                      !
!----------------------------------------------------------------------!

!_______________________________________________________________________
!
   subroutine mbfread( filename, nrow, ncol, nnz, data_type,            &
                       double, complx, ia, ja, ra, ca )

      ! arg. in  : filename
      !
      ! arg. out : double OR complx
      !               2-d real or complex array (double precision)
      !            nrow, ncol
      !               array shape
      !            data_type
      !               MF_DT_EMPTY    :
      !               MF_DT_DBLE     :        real
      !               MF_DT_CMPLX    :        complex
      !               MF_DT_SP_DBLE  : sparse real     {ia,ja,ra}
      !               MF_DT_SP_CMPLX : sparse complex  {ia,ja,ca}
      !               MF_DT_PERM_VEC : integer permutation vector: ia(nrow)

      ! warning : the array 'double' or 'complx' should be deallocated
      !           in the calling routine
      !           (i.e. the gateway routine to Matlab)
      !
      ! matrix properties (triangular patterns TRIL and TRIU, symmetry SYMM,
      ! positiveness POSD) are not read, nor the physical properties.
      !
      ! MBF format supported: 2.3 and 2.4
      !
      ! version: 2013-05-16
      !-------------------------------------------------------------

      use f90_gzlib

      implicit none

      ! must match the main declaration in $MUESLI/src/mod_mfdebug.F90
      integer, parameter :: MF_DOUBLE = kind(1.0d0)

      character(len=*), intent(in) :: filename
      integer, intent(out) :: nrow, ncol, nnz, data_type
      real(kind=MF_DOUBLE), pointer :: double(:,:)
      complex(kind=MF_DOUBLE), pointer :: complx(:,:)
      integer, pointer :: ia(:), ja(:)
      real(kind=MF_DOUBLE), pointer :: ra(:)
      complex(kind=MF_DOUBLE), pointer :: ca(:)

      ! must match the main declaration in $MUESLI/src/mod_mfarray.F90
      integer, parameter :: MF_DT_EMPTY    = 0,                         &
                            MF_DT_DBLE     = 1,                         &
                            MF_DT_CMPLX    = 2,                         &
                            MF_DT_BOOL     = 3,                         &
                            MF_DT_SP_DBLE  = 4,                         &
                            MF_DT_SP_CMPLX = 5,                         &
                            MF_DT_PERM_VEC = 6

      !-----------------------------------------------------------------

      ! signature for MF release >= 0.2.3
      character(len=20), parameter :: MF_BIN_SIGN_23 = "MF-2.3_ECanot_CNRS  "

      ! signature for MF release >= 2.8.0
      character(len=20), parameter :: MF_BIN_SIGN_24 = "MF-2.4_ECanot_CNRS  "

      ! little endian
      integer, parameter :: MF_ENDIAN_NATIVE = 875770417 ! = "1234"
      integer, parameter :: MF_ENDIAN_SWAP   = 825373492 ! = "4321"

      integer :: i, j, k, unit, l, rec_num, iostat, bytes_read
      character(len=20) :: signature_20
      integer :: endianness
      character(len=4) :: endianness_char

      logical :: gzipped, bytes_must_be_swapped
      type(gz_filedes) :: gz_file

      double precision :: tmp_re, tmp_im

      !-----------------------------------------------------------------

      bytes_must_be_swapped = .false.

      l = len_trim(filename)
      if( filename(l-2:l) == ".gz" ) then
         gzipped = .true.
      else
         gzipped = .false.
      end if

      if( gzipped ) then

         gz_file = gzopen( trim(filename), "rb" )

         if( filedes_is_null( gz_file ) ) then
            return
         end if

         ! remark : in sequential-unformatted write, each record
         !          are preceeded and followed by the record size.

         ! read and check signature
         call gzread( gz_file, l )
         call gzread( gz_file, signature_20 )
         call gzread( gz_file, l )

         if( signature_20 /= MF_BIN_SIGN_23 .and.                       &
             signature_20 /= MF_BIN_SIGN_24 ) then
            call mexerrmsgtxt( "  -> wrong MF signature!\n" //          &
                               "  found: '" // signature_20 // "'\n" // &
                               "  file: '" // trim(adjustl(filename)) // "'" )
            call gzclose( gz_file )
            return
         end if

         ! read and check endianness
         call gzread( gz_file, l )
         call gzread( gz_file, endianness )
         call gzread( gz_file, l )

         if( endianness == MF_ENDIAN_NATIVE ) then

            ! read data_type
            call gzread( gz_file, l )
            call gzread( gz_file, data_type )
            call gzread( gz_file, l )
            if( data_type == MF_DT_EMPTY ) then ! MF_EMPTY
               call gzclose( gz_file )
               return
            end if

            ! read array shape
            call gzread( gz_file, l )
            call gzread( gz_file, nrow )
            call gzread( gz_file, ncol )
            call gzread( gz_file, l )

            if( nrow < 0 .or. ncol < 0 ) then
               call mexerrmsgtxt( "  -> array shape cannot contain negative values!\n" // &
                                  "  file: '" // trim(adjustl(filename)) // "'" )
               call gzclose( gz_file )
               return
            end if
            if( nrow == 0 .and. ncol == 0 ) then
               call gzclose( gz_file )
               return
            end if

            ! read array data (column-wise)
            if( data_type == MF_DT_DBLE .or. data_type == MF_DT_BOOL ) then ! real or boolean case
               allocate( double(nrow,ncol) )
               call gzread( gz_file, l )
               call gzread( gz_file, double )
            else if( data_type == MF_DT_CMPLX ) then
               allocate( complx(nrow,ncol) )
               call gzread( gz_file, l )
               call gzread( gz_file, complx )
            else if( data_type == MF_DT_SP_DBLE ) then
               call gzread( gz_file, l )
               call gzread( gz_file, nnz )
               call gzread( gz_file, l )
               allocate( ja(ncol+1) )

               call gzread( gz_file, l )
               call gzread( gz_file, ja )
               call gzread( gz_file, l )
               allocate( ia(nnz) )

               call gzread( gz_file, l )
               call gzread( gz_file, ia )
               call gzread( gz_file, l )
               if( l == 0 ) then
                  call gzread( gz_file, l )
               end if
               allocate( ra(nnz) )

               call gzread( gz_file, l )
               call gzread( gz_file, ra )
            else if( data_type == MF_DT_SP_CMPLX ) then
               call gzread( gz_file, l )
               call gzread( gz_file, nnz )
               call gzread( gz_file, l )
               allocate( ja(ncol+1) )

               call gzread( gz_file, l )
               call gzread( gz_file, ja )
               call gzread( gz_file, l )
               allocate( ia(nnz) )

               call gzread( gz_file, l )
               call gzread( gz_file, ia )
               call gzread( gz_file, l )
               if( l == 0 ) then
                  call gzread( gz_file, l )
               end if
               allocate( ca(nnz) )

               call gzread( gz_file, l )
               call gzread( gz_file, ca )
            else if( data_type == MF_DT_PERM_VEC ) then
               allocate( ia(nrow) )

               call gzread( gz_file, l )
               call gzread( gz_file, ia )
            else
               call mexerrmsgtxt( "  -> internal error: unknown data type!\n" // &
                                  "  file: '" // trim(adjustl(filename)) // "'" )
               close(unit)
               return
            end if

            call gzclose( gz_file )

         else if( endianness == MF_ENDIAN_SWAP ) then
            ! bytes must be swapped

            ! read data_type
            call gzread( gz_file, l )
            call gzread( gz_file, data_type )
            call swap_4bytes( data_type )
            call gzread( gz_file, l )
            if( data_type == MF_DT_EMPTY ) then ! MF_EMPTY
               call gzclose( gz_file )
               return
            end if

            ! read array shape
            call gzread( gz_file, l )
            call gzread( gz_file, nrow )
            call swap_4bytes( nrow )
            call gzread( gz_file, ncol )
            call swap_4bytes( ncol )
            call gzread( gz_file, l )

            if( nrow < 0 .or. ncol < 0 ) then
               call mexerrmsgtxt( "  -> array shape cannot contain negative values!\n" // &
                                  "  file: '" // trim(adjustl(filename)) // "'" )
               call gzclose( gz_file )
               return
            end if
            if( nrow == 0 .and. ncol == 0 ) then
               call gzclose( gz_file )
               return
            end if

            ! read array data (column-wise)
            if( data_type == MF_DT_DBLE .or. data_type == MF_DT_BOOL ) then ! real or boolean case
               allocate( double(nrow,ncol) )
               call gzread( gz_file, l )
               call gzread( gz_file, double )
               do j = 1, ncol
                  do i = 1, nrow
                     call swap_8bytes( double(i,j) )
                  end do
               end do
            else if( data_type == MF_DT_CMPLX ) then
               allocate( complx(nrow,ncol) )
               call gzread( gz_file, l )
               call gzread( gz_file, complx )
               do j = 1, ncol
                  do i = 1, nrow
                     tmp_re = real(complx(i,j))
                     call swap_8bytes( tmp_re )
                     tmp_im = aimag(complx(i,j))
                     call swap_8bytes( tmp_im )
                     complx(i,j) = cmplx( tmp_re, tmp_im, kind=MF_DOUBLE )
                  end do
               end do
            else if( data_type == MF_DT_SP_DBLE ) then
               call gzread( gz_file, l )
               call gzread( gz_file, nnz )
               call swap_4bytes( nnz )
               call gzread( gz_file, l )
               allocate( ja(ncol+1) )

               call gzread( gz_file, l )
               call gzread( gz_file, ja )
               do k = 1, ncol+1
                  call swap_4bytes( ja(k) )
               end do
               call gzread( gz_file, l )
               allocate( ia(nnz) )

               call gzread( gz_file, l )
               call gzread( gz_file, ia )
               do k = 1, nnz
                  call swap_4bytes( ia(k) )
               end do
               call gzread( gz_file, l )
               if( l == 0 ) then
                  call gzread( gz_file, l )
               end if
               allocate( ra(nnz) )

               call gzread( gz_file, l )
               call gzread( gz_file, ra )
               do k = 1, nnz
                  call swap_8bytes( ra(k) )
               end do
            else if( data_type == MF_DT_SP_CMPLX ) then
               call gzread( gz_file, l )
               call gzread( gz_file, nnz )
               call swap_4bytes( nnz )
               call gzread( gz_file, l )
               allocate( ja(ncol+1) )

               call gzread( gz_file, l )
               call gzread( gz_file, ja )
               do k = 1, ncol+1
                  call swap_4bytes( ja(k) )
               end do
               call gzread( gz_file, l )
               allocate( ia(nnz) )

               call gzread( gz_file, l )
               call gzread( gz_file, ia )
               do k = 1, nnz
                  call swap_4bytes( ia(k) )
               end do
               call gzread( gz_file, l )
               if( l == 0 ) then
                  call gzread( gz_file, l )
               end if
               allocate( ca(nnz) )

               call gzread( gz_file, l )
               call gzread( gz_file, ca )
               do k = 1, nnz
                  tmp_re = real(ca(k))
                  call swap_8bytes( tmp_re )
                  tmp_im = aimag(ca(k))
                  call swap_8bytes( tmp_im )
                  ca(k) = cmplx( tmp_re, tmp_im, kind=MF_DOUBLE )
               end do
            else if( data_type == MF_DT_PERM_VEC ) then
               allocate( ia(nrow) )

               call gzread( gz_file, l )
               call gzread( gz_file, ia )
               do k = 1, nrow
                  call swap_4bytes( ia(k) )
               end do
            else
               call mexerrmsgtxt( "  -> internal error: unknown data type!\n" // &
                                  "  file: '" // trim(adjustl(filename)) // "'" )
               close(unit)
               return
            end if

            call gzclose( gz_file )

         end if

      else ! not gzipped

         unit = 99

         ! we must detect signature and endianness before any
         ! sequential read (this imply direct access)
         ! (reading each byte at a time, then endianness independant)
         open( unit=unit, file=trim(adjustl(filename)),                 &
               access="direct", recl=1, status="old" )

         do i = 1, 20
            rec_num = i + 4
            read(unit,rec=rec_num) signature_20(i:i)
         end do

         ! check signature

         if( signature_20 == MF_BIN_SIGN_23 .or.                        &
             signature_20 == MF_BIN_SIGN_24 ) then
            do i = 1, 4
               rec_num = i + 32
               read(unit,rec=rec_num) endianness_char(i:i)
            end do
            endianness = transfer(endianness_char,0)
            if( endianness == MF_ENDIAN_SWAP ) then
               ! different endianness
               bytes_must_be_swapped = .true.
            else if( endianness == MF_ENDIAN_NATIVE ) then
               ! nothing to do: just to verify
            else
               call mexerrmsgtxt( "  -> unknown value for endianness!\n" // &
                                  "  file: '" // trim(adjustl(filename)) // "'" )
               close(unit)
               return
            end if
         else
            call mexerrmsgtxt( "  -> wrong MF signature!\n" //          &
                               "  found: '" // signature_20 // "'\n" // &
                               "  file: '" // trim(adjustl(filename)) // "'" )
            close(unit)
            return
         end if

         close(unit)

         if( bytes_must_be_swapped ) then

            ! open in direct access
            ! (we cannot read a file having different endianness
            !  in sequential access => EOF)
            open( unit=unit, file=trim(adjustl(filename)),              &
                  access="direct", recl=4, status="old" )

            rec_num = 12
            read(unit,rec=rec_num) data_type
            call swap_4bytes( data_type )

            if( data_type == MF_DT_EMPTY ) then
               close(unit)
               return
            end if

            rec_num = 15
            read(unit,rec=rec_num) nrow
            call swap_4bytes( nrow )

            rec_num = 16
            read(unit,rec=rec_num) ncol
            call swap_4bytes( ncol )

            if( nrow < 0 .or. ncol < 0 ) then
               call mexerrmsgtxt( "  -> array shape cannot contain negative values!\n" // &
                                  "  file: '" // trim(adjustl(filename)) // "'" )
               close(unit)
               return
            end if
            if( nrow == 0 .and. ncol == 0 ) then
               close(unit)
               return
            end if

            if( data_type == MF_DT_DBLE .or. data_type == MF_DT_BOOL ) then ! real or boolean case
               close(unit)
               open( unit=unit, file=trim(adjustl(filename)),           &
                     access="direct", recl=8, status="old" )
               allocate( double(nrow,ncol) )

               rec_num = 10
outer2:        do j = 1, ncol
                  do i = 1, nrow
                     read(unit,rec=rec_num,iostat=iostat) double(i,j)
                     if( iostat /= 0 ) exit outer2
                     call swap_8bytes( double(i,j) )
                     rec_num = rec_num + 1
                  end do
               end do outer2
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: double, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
            else if( data_type == MF_DT_CMPLX ) then
               close(unit)
               open( unit=unit, file=trim(adjustl(filename)),           &
                     access="direct", recl=8, status="old" )
               allocate( complx(nrow,ncol) )

               rec_num = 10
outer3:        do j = 1, ncol
                  do i = 1, nrow
                     read(unit,rec=rec_num,iostat=iostat) tmp_re
                     if( iostat /= 0 ) exit outer3
                     call swap_8bytes( tmp_re )
                     rec_num = rec_num + 1
                     read(unit,rec=rec_num,iostat=iostat) tmp_im
                     if( iostat /= 0 ) exit outer3
                     call swap_8bytes( tmp_im )
                     rec_num = rec_num + 1
                     complx(i,j) = cmplx( tmp_re, tmp_im, kind=MF_DOUBLE )
                  end do
               end do outer3
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: complx, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
            else if( data_type == MF_DT_SP_DBLE ) then
               rec_num = 19
               read(unit,rec=rec_num,iostat=iostat) nnz
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: nnz, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
               call swap_4bytes( nnz )
               allocate( ja(ncol+1) )

               rec_num = 22
               do k = 1, ncol+1
                  read(unit,rec=rec_num,iostat=iostat) ja(k)
                  if( iostat /= 0 ) exit
                  call swap_4bytes( ja(k) )
                  rec_num = rec_num + 1
               end do
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ja, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
               allocate( ia(nnz) )

               rec_num = rec_num + 2
               do k = 1, nnz
                  read(unit,rec=rec_num,iostat=iostat) ia(k)
                  if( iostat /= 0 ) exit
                  call swap_4bytes( ia(k) )
                  rec_num = rec_num + 1
               end do
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ia, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
               allocate( ra(nnz) )

               close(unit)
               open( unit=unit, file=trim(adjustl(filename)),           &
                     access="direct", recl=8, status="old" )
               if( mod(rec_num+1,2) /= 0 ) then
                  ! fixing rec_num: cf. 'msSave'
                  rec_num = rec_num + 1
               end if
               rec_num = (rec_num+1)/2 + 1
               do k = 1, nnz
                  read(unit,rec=rec_num,iostat=iostat) ra(k)
                  if( iostat /= 0 ) exit
                  call swap_8bytes( ra(k) )
                  rec_num = rec_num + 1
               end do
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ra, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
            else if( data_type == MF_DT_SP_CMPLX ) then
               rec_num = 19
               read(unit,rec=rec_num,iostat=iostat) nnz
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: nnz, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
               call swap_4bytes( nnz )
               allocate( ja(ncol+1) )

               rec_num = 22
               do k = 1, ncol+1
                  read(unit,rec=rec_num,iostat=iostat) ja(k)
                  if( iostat /= 0 ) exit
                  call swap_4bytes( ja(k) )
                  rec_num = rec_num + 1
               end do
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ja, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
               allocate( ia(nnz) )

               rec_num = rec_num + 2
               do k = 1, nnz
                  read(unit,rec=rec_num,iostat=iostat) ia(k)
                  if( iostat /= 0 ) exit
                  call swap_4bytes( ia(k) )
                  rec_num = rec_num + 1
               end do
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ia, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
               allocate( ca(nnz) )

               close(unit)
               open( unit=unit, file=trim(adjustl(filename)),           &
                     access="direct", recl=8, status="old" )
               if( mod(rec_num+1,2) /= 0 ) then
                  ! fixing rec_num: cf. 'msSave'
                  rec_num = rec_num + 1
               end if
               rec_num = (rec_num+1)/2 + 1
               do k = 1, nnz
                  read(unit,rec=rec_num,iostat=iostat) tmp_re
                  if( iostat /= 0 ) exit
                  call swap_8bytes( tmp_re )
                  rec_num = rec_num + 1
                  read(unit,rec=rec_num,iostat=iostat) tmp_im
                  if( iostat /= 0 ) exit
                  call swap_8bytes( tmp_im )
                  rec_num = rec_num + 1
                  ca(k) = cmplx( tmp_re, tmp_im, kind=MF_DOUBLE )
               end do
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ca, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
            else if( data_type == MF_DT_PERM_VEC ) then
               rec_num = 19
               do k = 1, nrow
                  read(unit,rec=rec_num,iostat=iostat) ia(k)
                  if( iostat /= 0 ) exit
                  call swap_4bytes( ia(k) )
                  rec_num = rec_num + 1
               end do
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ia, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
            else
               call mexerrmsgtxt( "  -> internal error: unknown data type!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
               close(unit)
               return
            end if

         else

            ! open in sequential access
            open( unit=unit, file=trim(adjustl(filename)),              &
                  form="unformatted", status="old" )

            ! no need to verify read
            read(unit) signature_20

            ! no need to verify read
            read(unit) endianness

            read(unit) data_type

            if( data_type == MF_DT_EMPTY ) then
               close(unit)
               return
            end if

            read(unit) nrow, ncol
            if( nrow < 0 .or. ncol < 0 ) then
               call mexerrmsgtxt( "  -> array shape cannot contain negative values!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
               close(unit)
               return
            end if
            if( nrow == 0 .and. ncol == 0 ) then
               close(unit)
               return
            end if

            ! reading by column (quick memory access)
            if( data_type == MF_DT_DBLE .or. data_type == MF_DT_BOOL ) then ! real or boolean case
               allocate( double(nrow,ncol) )
               read(unit) double(:,:)
            else if( data_type == MF_DT_CMPLX ) then
               allocate( complx(nrow,ncol) )
               read(unit) complx(:,:)
            else if( data_type == MF_DT_SP_DBLE ) then
               read(unit,iostat=iostat) nnz
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: nnz, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
               allocate( ja(ncol+1) )

               read(unit,iostat=iostat) ja
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ja, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
               allocate( ia(nnz) )

               read(unit,iostat=iostat) ia
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ia, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
               allocate( ra(nnz) )

               read(unit,iostat=iostat) ra
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ra, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
            else if( data_type == MF_DT_SP_CMPLX ) then
               read(unit,iostat=iostat) nnz
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: nnz, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
               allocate( ja(ncol+1) )

               read(unit,iostat=iostat) ja
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ja, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
               allocate( ia(nnz) )

               read(unit,iostat=iostat) ia
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ia, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
               allocate( ca(nnz) )

               read(unit,iostat=iostat) ca
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ca, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
            else if( data_type == MF_DT_PERM_VEC ) then
               allocate( ia(nrow) )

               read(unit,iostat=iostat) ia
               if( iostat /= 0 ) then
                  call mexerrmsgtxt( "  -> reading: ia, incomplete data!\n" // &
                                     "  file: '" // trim(adjustl(filename)) // "'" )
                  close(unit)
                  return
               end if
            else
               call mexerrmsgtxt( "  -> internal error: unknown data type!\n" // &
                                  "  file: '" // trim(adjustl(filename)) // "'" )
               close(unit)
               return
            end if
            close(unit)

         end if

      end if

   end subroutine mbfread
