! f90 include file

!_______________________________________________________________________
!
   function mfErf( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Error function

      integer :: status
      integer :: i, j

#if defined _INTEL_IFC | defined _GNU_GFC
#else
      ! slatec:
      real(kind=MF_DOUBLE), external :: derf
#endif
      real(kind=MF_DOUBLE) :: u, v
      logical :: error
      complex(kind=MF_DOUBLE) :: z0, z1, erfc_val

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfErf", "E",                               &
                            "this function cannot (yet) be",            &
                            "applied to sparse matrix!" )
         go to 99
      end if

      if( mfIsReal(A) ) then

         out%data_type = MF_DT_DBLE
         out%shape = a%shape
         allocate( out%double(a%shape(1),a%shape(2)) )

#if defined _INTEL_IFC | defined _GNU_GFC
! 'derf' is an intrinsic (extension), so it is elemental
         out%double(:,:) = derf( a%double(:,:) )
#else
         do j = 1, a%shape(2)
            do i = 1, a%shape(1)
               out%double(i,j) = derf( a%double(i,j) )
            end do
         end do
#endif

      else

         out%data_type = MF_DT_CMPLX
         out%shape = a%shape
         allocate( out%cmplx(a%shape(1),a%shape(2)) )

         do j = 1, a%shape(2)
            do i = 1, a%shape(1)
               z0 = a%cmplx(i,j)
               z1 = (0.0d0,1.0d0)*z0
               call dwofz(real(z1),imag(z1),u,v,error)
               if( error ) then
                  call PrintMessage( "mfErf", "W",                      &
                                     "Overflow occured when computing erf(z) for complex value!", &
                                     "(Output not defined, so result is NaN)" )
                  out%cmplx(i,j) = cmplx(MF_NAN,MF_NAN)
               else
                  erfc_val = cmplx(u,v,kind=MF_DOUBLE)*exp(-z0**2)
                  out%cmplx(i,j) = (1.0d0,0.0d0) - erfc_val
               end if
            end do
         end do

      end if

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfErf", "E",                            &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfErf
!_______________________________________________________________________
!
#if defined _INTEL_IFC | defined _GNU_GFC
   elemental pure function erfinv( y )
#else
   function erfinv( y )
#endif

      real(kind=MF_DOUBLE), intent(in) :: y
      real(kind=MF_DOUBLE) :: erfinv
      !------ API end ------
#ifdef _DEVLP

      ! after Matlab 7.0 erfinv function, revision 5.15.4.2 (2004/07/05)

      ! Coefficients in rational approximations.

      real(kind=MF_DOUBLE), parameter :: a(4) =                         &
             [  0.886226899d0, -1.645349621d0,                          &
                0.914624893d0, -0.140543331d0 ]
      real(kind=MF_DOUBLE), parameter :: b(4) =                         &
             [ -2.118377725d0,  1.442710462d0,                          &
               -0.329097515d0,  0.012229801d0 ]
      real(kind=MF_DOUBLE), parameter :: c(4) =                         &
             [ -1.970840454d0, -1.624906493d0,                          &
                3.429567803d0,  1.641345311d0 ]
      real(kind=MF_DOUBLE), parameter :: d(2) =                         &
             [  3.543889200d0,  1.637067800d0 ]

      real(kind=MF_DOUBLE) :: x, y0, z, u

   !------ end of declarations -- execution starts hereafter  ------

      ! Exceptional cases

      if( mf_isnan(y) ) then
         x = MF_NAN
         goto 99
      else if( y == -1.0d0 ) then
         x = -MF_INF
         goto 99
      else if( y == +1.0d0 ) then
         x = MF_INF
         goto 99
      else if( abs(y) > +1.0d0 ) then
         x = MF_NAN
         goto 99
      end if

      y0 = 0.7d0

      if( y > y0 ) then

         ! Near end points of range
         z = sqrt(-log((1.0d0-y)/2.0d0))
         x = (((c(4)*z+c(3))*z+c(2))*z+c(1)) / ((d(2)*z+d(1))*z+1.0d0)

      else if( y < -y0 ) then

         z = sqrt(-log((1.0d0+y)/2.0d0))
         x = -(((c(4)*z+c(3))*z+c(2))*z+c(1)) / ((d(2)*z+d(1))*z+1.0d0)

      else ! abs(y) <= y0

         ! Central range
         z = y**2
         x = y*(((a(4)*z+a(3))*z+a(2))*z+a(1)) /                        &
             ((((b(4)*z+b(3))*z+b(2))*z+b(1))*z+1.0d0)

      end if

      ! The relative error of the approximation has absolute value less
      ! than 8.9e-7.  One iteration of Halley's rational method (third
      ! order) gives full machine precision.

      ! Newton's method: new x = x - f/f'
      ! Halley's method: new x = x - 1/(f'/f - (f"/f')/2)
      ! This function: f = erf(x) - y, f' = 2/sqrt(pi)*exp(-x^2), f" = -2*x*f'

      ! Newton's correction
      u = (derf(x) - y) / (2.0d0/sqrt(MF_PI) * exp(-x**2))

      ! Halley's step
      x = x - u/(1.0d0+x*u)

 99   continue

      erfinv = x

#endif
   end function erfinv
!_______________________________________________________________________
!
   function mfErfInv( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Inverse Error function

      ! Restriction: limited to real arg.

      integer :: status

#if defined _INTEL_IFC | defined _GNU_GFC
#else
      integer :: i, j
#endif


   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfErfInv", "E",                            &
                            "this function cannot (yet) be",            &
                            "applied to sparse matrix!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfErfInv", "E",                            &
                            "arg. must be real!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = a%shape
      allocate( out%double(a%shape(1),a%shape(2)) )

#if defined _INTEL_IFC | defined _GNU_GFC
      out%double(:,:) = erfinv( a%double(:,:) )
#else
      do j = 1, a%shape(2)
         do i = 1, a%shape(1)
            out%double(i,j) = erfinv( a%double(i,j) )
         end do
      end do
#endif

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfErfInv", "E",                         &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfErfInv
!_______________________________________________________________________
!
   function mfErfC( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Complementary Error function

      ! Restriction: limited to real arg.

      integer :: status
      integer :: i, j

#if defined _INTEL_IFC | defined _GNU_GFC
#else
      ! slatec:
      real(kind=MF_DOUBLE), external :: derfc
#endif
      real(kind=MF_DOUBLE) :: u, v
      logical :: error
      complex(kind=MF_DOUBLE) :: z0, z1

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfErfC", "E",                              &
                            "this function cannot (yet) be",            &
                            "applied to sparse matrix!" )
         go to 99
      end if

      if( mfIsReal(A) ) then

         out%data_type = MF_DT_DBLE
         out%shape = a%shape
         allocate( out%double(a%shape(1),a%shape(2)) )

#if defined _INTEL_IFC | defined _GNU_GFC
         out%double(:,:) = derfc( a%double(:,:) )
#else
         do j = 1, a%shape(2)
            do i = 1, a%shape(1)
               if( a%double(i,j) == -MF_INF ) then
                  out%double(i,j) = 2.0d0
               else if( a%double(i,j) == MF_INF ) then
                  out%double(i,j) = 0.0d0
               else
                  out%double(i,j) = derfc( a%double(i,j) )
               end if
            end do
         end do
#endif

      else

         out%data_type = MF_DT_CMPLX
         out%shape = a%shape
         allocate( out%cmplx(a%shape(1),a%shape(2)) )

         do j = 1, a%shape(2)
            do i = 1, a%shape(1)
               z0 = a%cmplx(i,j)
               z1 = (0.0d0,1.0d0)*z0
               call dwofz(real(z1),imag(z1),u,v,error)
               if( error ) then
                  call PrintMessage( "mfErfC", "W",                     &
                                     "Overflow occured when computing erfc(z) for complex value!", &
                                     "(Output not defined, so result is NaN)" )
                  out%cmplx(i,j) = cmplx(MF_NAN,MF_NAN)
               else
                  out%cmplx(i,j) = cmplx(u,v,kind=MF_DOUBLE)*exp(-z0**2)
               end if
            end do
         end do

      end if

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfErfC", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfErfC
!_______________________________________________________________________
!
#if defined _INTEL_IFC | defined _GNU_GFC
   elemental pure function erfcinv( y )
#else
   function erfcinv( y )
#endif

      real(kind=MF_DOUBLE), intent(in) :: y
      real(kind=MF_DOUBLE) :: erfcinv
      !------ API end ------
#ifdef _DEVLP

      ! use the relation: erfcinv(x) = erfinv(1-x)

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_isnan(y) ) then
         erfcinv = MF_NAN
      else if( y == 0.0d0 ) then
         erfcinv = MF_INF
      else if( y == 2.0d0 ) then
         erfcinv = -MF_INF
      else if( y < 0.0d0 .or. 2.0d0 < y ) then
         erfcinv = MF_NAN
      else
         erfcinv = erfinv( 1.0d0 - y )
      end if


#endif
   end function erfcinv
!_______________________________________________________________________
!
   function mfErfCInv( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Inverse Complementary Error function

      ! Restriction: limited to real arg.

      integer :: status

#if defined _INTEL_IFC | defined _GNU_GFC
#else
      integer :: i, j
#endif


   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfErfCInv", "E",                           &
                            "this function cannot (yet) be",            &
                            "applied to sparse matrix!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfErfCInv", "E",                           &
                            "arg. must be real!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = a%shape
      allocate( out%double(a%shape(1),a%shape(2)) )

#if defined _INTEL_IFC | defined _GNU_GFC
      out%double(:,:) = erfcinv( a%double(:,:) )
#else
      do j = 1, a%shape(2)
         do i = 1, a%shape(1)
            out%double(i,j) = erfcinv( a%double(i,j) )
         end do
      end do
#endif

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfErfCInv", "E",                        &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfErfCInv
!_______________________________________________________________________
!
   elemental pure function derfcx(x) result(res)
!------------------------------------------------------------------
!
!  Scaled complementary error function.
!  From CALERF : http://www.netlib.org/specfun/erf
!------------------------------------------------------------------
!  The main computation evaluates near-minimax approximations
!  from "Rational Chebyshev approximations for the error function"
!  by W. J. Cody, Math. Comp., 1969, PP. 631-638.
!
!  Author: W. J. Cody
!          Mathematics and Computer Science Division
!          Argonne National Laboratory
!          Argonne, IL 60439
!
!  Latest modification: March 19, 1990
!------------------------------------------------------------------
      real(kind=MF_DOUBLE), intent(in) :: x
      real(kind=MF_DOUBLE) :: res
      !------ API end ------
#ifdef _DEVLP

      integer :: i
      real(kind=MF_DOUBLE) :: xnum, xden, y, ysq, del
!------------------------------------------------------------------
!  Mathematical constants
!------------------------------------------------------------------
      real(kind=MF_DOUBLE), parameter :: four = 4.0d0,                  &
           one = 1.0d0, two = 2.0d0, zero = 0.0d0,                                   &
           sqrpi = 5.6418958354775628695d-1, thresh = 0.46875d0,        &
           sixten = 16.0d0
!------------------------------------------------------------------
!  Machine-dependent constants
!------------------------------------------------------------------
      real(kind=MF_DOUBLE), parameter :: xsmall = 1.11d-16,             &
           xbig = 26.543d0, xhuge = 6.71d7, xmax = 2.53d307,            &
           xinf = 1.79d308, xneg = -26.628d0
!------------------------------------------------------------------
!  Coefficients for approximation to  erf  in first interval
!------------------------------------------------------------------
      real(kind=MF_DOUBLE), parameter :: a(5) =                         &
             [ 3.16112374387056560d00, 1.13864154151050156d02,          &
               3.77485237685302021d02, 3.20937758913846947d03,          &
               1.85777706184603153d-1 ]
      real(kind=MF_DOUBLE), parameter :: b(4) =                         &
             [ 2.36012909523441209d01, 2.44024637934444173d02,          &
               1.28261652607737228d03, 2.84423683343917062d03 ]
!------------------------------------------------------------------
!  Coefficients for approximation to  erfc  in second interval
!------------------------------------------------------------------
      real(kind=MF_DOUBLE), parameter :: c(9) =                         &
             [ 5.64188496988670089d-1, 8.88314979438837594d0,           &
               6.61191906371416295d01, 2.98635138197400131d02,          &
               8.81952221241769090d02, 1.71204761263407058d03,          &
               2.05107837782607147d03, 1.23033935479799725d03,          &
               2.15311535474403846d-8 ]
      real(kind=MF_DOUBLE), parameter :: d(8) =                         &
             [ 1.57449261107098347d01, 1.17693950891312499d02,          &
               5.37181101862009858d02, 1.62138957456669019d03,          &
               3.29079923573345963d03, 4.36261909014324716d03,          &
               3.43936767414372164d03, 1.23033935480374942d03 ]
!------------------------------------------------------------------
!  Coefficients for approximation to  erfc  in third interval
!------------------------------------------------------------------
      real(kind=MF_DOUBLE), parameter :: p(6) =                         &
             [ 3.05326634961232344d-1, 3.60344899949804439d-1,          &
               1.25781726111229246d-1, 1.60837851487422766d-2,          &
               6.58749161529837803d-4, 1.63153871373020978d-2 ]
      real(kind=MF_DOUBLE), parameter :: q(5) =                         &
             [ 2.56852019228982242d00, 1.87295284992346047d00,          &
               5.27905102951428412d-1, 6.05183413124413191d-2,          &
               2.33520497626869185d-3 ]
!------------------------------------------------------------------

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_isnan(x) ) then
         res = MF_NAN
         return
      else if( x == -MF_INF ) then
         res = MF_INF
         return
      else if( x == MF_INF ) then
         res = 0.0d0
         return
      else if( x == 0.0d0 ) then
         res = 1.0d0
         return
      end if

      y = abs(x)
      if( y <= thresh ) then
!------------------------------------------------------------------
!  Evaluate for |x| <= 0.46875
!------------------------------------------------------------------
         ysq = zero
         if( y > xsmall ) ysq = y**2
         xnum = a(5)*ysq
         xden = ysq
         do i = 1, 3
            xnum = (xnum + a(i))*ysq
            xden = (xden + b(i))*ysq
         end do
         res = x*(xnum + a(4)) / (xden + b(4))
         res = one - res
         res = exp(ysq)*res
         return
!------------------------------------------------------------------
!  Evaluate for 0.46875 <= |x| <= 4.0
!------------------------------------------------------------------
      else if( y <= four ) then
         xnum = c(9)*y
         xden = y
         do i = 1, 7
            xnum = (xnum + c(i))*y
            xden = (xden + d(i))*y
         end do
         res = (xnum + c(8)) / (xden + d(8))
!------------------------------------------------------------------
!  Evaluate for |x| > 4.0
!------------------------------------------------------------------
      else
         res = zero
         if( y >= xbig ) then
            if( y >= xmax ) go to 300
            if( y >= xhuge ) then
               res = sqrpi / y
               go to 300
            end if
         end if
         ysq = one/y**2
         xnum = p(6)*ysq
         xden = ysq
         do i = 1, 4
            xnum = (xnum + p(i))*ysq
            xden = (xden + q(i))*ysq
         end do
         res = ysq *(xnum + p(5)) / (xden + q(5))
         res = (sqrpi -  res)/y
      end if

!------------------------------------------------------------------
!  Fix up for negative argument, etc.
!------------------------------------------------------------------
  300 if( x < zero ) then
         if( x < xneg ) then
            res = xinf
         else
            ysq = aint(x*sixten)/sixten
            del = (x-ysq)*(x+ysq)
            y = exp(ysq*ysq) * exp(del)
            res = two*y - res
         end if
      end if

#endif
   end function derfcx
!_______________________________________________________________________
!
   function mfErfCScaled( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Scaled complementary error function.
      !
      !   erfc_scaled(x) = exp(x**2) * erfc(x)

      ! Restriction: limited to real arg.

      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfErfCScaled", "E",                        &
                            "this function cannot (yet) be",            &
                            "applied to sparse matrix!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfErfCScaled", "E",                        &
                            "arg. must be real!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = a%shape
      allocate( out%double(a%shape(1),a%shape(2)) )

      out%double(:,:) = derfcx( a%double(:,:) )

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfErfCScaled", "E",                     &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfErfCScaled
