! f90 include file

!_______________________________________________________________________
!
   subroutine sqbanded2csc_nnz( ncol, ml, mu, a,                        &
                                nnz, threshold )

      integer,              intent(in)  :: ncol, ml, mu
      real(kind=MF_DOUBLE), intent(in)  :: a(:,:)
      integer,              intent(out) :: nnz
      real(kind=MF_DOUBLE), intent(in), optional :: threshold
      !------ API end ------

      !-----------------------------------------------------------------
      !  Preparation of 'sqbanded2csc' (see below), by computing only
      !  the number of non-zero elements.
      !  This special routine accepts a threshold value ('threshold')
      !  below which the elements are discarded.
      ! January 25, 2022
      !-----------------------------------------------------------------
      ! on entry:
      !---------
      ! ncol      = dimension of the square matrix
      ! ml        = number of subdiagonals
      ! mu        = number of upperdiagonals
      !             (main diagonal is supposed to be present)
      ! a         = real matrix in compressed banded format (cf. Blas/Lapack);
      !             element (i,j) of the matrix is stored in
      !                              a( i-j+bandwith, j )
      !             with bandwidth = ml + 1 + mu
      ! threshold = real positive value.
      !
      ! on return:
      !-----------
      ! nnz  = number of kept elements.
      !-----------------------------------------------------------------

      integer :: i, j, k, ii
      integer :: bandwidth
      logical :: economize

   !------ end of declarations -- execution starts hereafter  ------

      if( present(threshold) ) then
         economize = .true.
      else
         economize = .false.
      end if

      bandwidth = ml + 1 + mu

      ! Determine col-lengths, nnz, and fill the matrix by columns.
      k = 0
      do j = 1, mu
         do i = 1, j + ml
            ii = i - j + bandwidth
            if( economize ) then
               if( abs(a(ii,j)) > threshold ) then
                  k = k + 1
               end if
            else
               k = k + 1
            end if
         end do
      end do
      do j = mu+1, ncol-ml
         do i = j - mu, j + ml
            ii = i - j + bandwidth
            if( economize ) then
               if( abs(a(ii,j)) > threshold ) then
                  k = k + 1
               end if
            else
               k = k + 1
            end if
         end do
      end do
      do j = ncol-ml+1, ncol
         do i = j - mu, ncol
            ii = i - j + bandwidth
            if( economize ) then
               if( abs(a(ii,j)) > threshold ) then
                  k = k + 1
               end if
            else
               k = k + 1
            end if
         end do
      end do

      nnz = k

   end subroutine sqbanded2csc_nnz
!_______________________________________________________________________
!
   subroutine sqbanded2csc( ncol, ml, mu, a,                            &
                            nnz, ao, iao, jao, threshold )

      integer,              intent(in)           :: ncol, ml, mu
      real(kind=MF_DOUBLE), intent(in)           :: a(:,:)
      real(kind=MF_DOUBLE), intent(out)          :: ao(:)
      integer,              intent(out)          :: nnz, iao(:), jao(:)
      real(kind=MF_DOUBLE), intent(in), optional :: threshold
      !------ API end ------

      !-----------------------------------------------------------------
      !  Compressed Banded  to  Compressed Sparse Column
      ! (Compressed Banded is the usual storage scheme of Blas/Lapack
      !  for banded matrices)
      ! January 25, 2022
      !-----------------------------------------------------------------
      ! Converts a square, banded matrix that is stored in compressed
      ! (dense) format into a CSC format.
      ! No check for size of ao and iao! Assumed they are larger enough to
      ! contain the nnz elements.
      ! We must have: ml < ncol, mu < ncol
      !-----------------------------------------------------------------
      ! on entry:
      !---------
      ! ncol      = dimension of the square matrix
      ! ml        = number of subdiagonals
      ! mu        = number of upperdiagonals
      !             (main diagonal is supposed to be present)
      ! a         = real matrix in compressed banded format (cf. Blas/Lapack);
      !             element (i,j) of the matrix is stored in
      !                              a( i-j+bandwith, j )
      !             with bandwidth = ml + 1 + mu
      ! threshold (optional) = real positive value (see 'sqbanded2csc_nnz').
      !
      ! on return:
      !-----------
      ! nnz  = number of elements in all specified diagonals.
      ! ao, iao, jao = matrix in general sparse matrix format with ao
      ! containing the real values, iao containing the row indexes,
      ! and jao being the pointer to the beginning of the column,
      ! in arrays ao, iao.
      !
      ! The resulting matrix is row sorted.
      !
      ! Entries containing a null value are kept, except if the optional
      ! argument 'threshold' is present.
      !-----------------------------------------------------------------

      integer :: i, j, k, ii
      integer :: bandwidth
      logical :: economize, keep

   !------ end of declarations -- execution starts hereafter  ------

      if( present(threshold) ) then
         economize = .true.
      else
         economize = .false.
      end if

      bandwidth = ml + 1 + mu

      ! Determine pointers to columns, nnz, and fill the matrix by columns.
      jao(1) = 1
      k = 0
      do j = 1, mu
         do i = 1, j + ml
            ii = i - j + bandwidth
            if( economize ) then
               if( abs(a(ii,j)) > threshold ) then
                  keep = .true.
               else
                  keep = .false.
               end if
            else
               keep = .true.
            end if
            if( keep ) then
               k = k + 1
               ao(k) = a(ii,j)
               iao(k) = i
            end if
         end do
         jao(j+1) = k + 1
      end do
      do j = mu+1, ncol-ml
         do i = j - mu, j + ml
            ii = i - j + bandwidth
            if( economize ) then
               if( abs(a(ii,j)) > threshold ) then
                  keep = .true.
               else
                  keep = .false.
               end if
            else
               keep = .true.
            end if
            if( keep ) then
               k = k + 1
               ao(k) = a(ii,j)
               iao(k) = i
            end if
         end do
         jao(j+1) = k + 1
      end do
      do j = ncol-ml+1, ncol
         do i = j - mu, ncol
            ii = i - j + bandwidth
            if( economize ) then
               if( abs(a(ii,j)) > threshold ) then
                  keep = .true.
               else
                  keep = .false.
               end if
            else
               keep = .true.
            end if
            if( keep ) then
               k = k + 1
               ao(k) = a(ii,j)
               iao(k) = i
            end if
         end do
         jao(j+1) = k + 1
      end do

      nnz = k

   end subroutine sqbanded2csc
