! f90 include file

!_______________________________________________________________________
!
   function row_sorted( ncol, ia, ja ) result( bool )

      integer, intent(in) :: ncol, ia(:), ja(:)
      logical :: bool
      !------ API end ------

      !-----------------------------------------------------------------
      ! Checks if matrix in CSC format has columns sorted by rows.
      ! (adapted from SPARSKIT2 (Mar 8 2005) 'csorted' by É. Canot)
      !-----------------------------------------------------------------
      ! On entry:
      !--------------
      ! ncol    = number of cols in matrix
      ! ia, ja  = sparsity structure of matrix in CSC format
      !
      ! On return:
      !---------------
      ! bool = TRUE : indicates that matrix is sorted by columns and
      !               has no duplicated entries
      !-----------------------------------------------------------------

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      do j = 1, ncol
         do i = ja(j)+1, ja(j+1)-1
            if( ia(i-1) >= ia(i) ) then
               bool = .false.
               return
            end if
         end do
      end do
      bool = .true.

   end function row_sorted
!_______________________________________________________________________
!
   function weak_row_sorted( ncol, ia, ja ) result( bool )

      integer, intent(in) :: ncol, ia(:), ja(:)
      logical :: bool
      !------ API end ------

      !-----------------------------------------------------------------
      ! Checks if matrix in CSC format has columns sorted by rows.
      ! (adapted from SPARSKIT2 (Mar 8 2005) 'csorted' by É. Canot)
      !-----------------------------------------------------------------
      ! On entry:
      !--------------
      ! ncol    = number of cols in matrix
      ! ia, ja  = sparsity structure of matrix in CSC format
      !
      ! On return:
      !---------------
      ! bool = TRUE : indicates that matrix is sorted by columns, but
      !               has perhaps duplicated entries
      !-----------------------------------------------------------------

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      do j = 1, ncol
         do i = ja(j)+1, ja(j+1)-1
            if( ia(i-1) > ia(i) ) then
               bool = .false.
               return
            end if
         end do
      end do
      bool = .true.

   end function weak_row_sorted
!_______________________________________________________________________
!
   function unique_entries( ncol, ia, ja ) result( bool )

      integer, intent(in) :: ncol, ia(:), ja(:)
      logical :: bool
      !------ API end ------

      !-----------------------------------------------------------------
      ! Checks if a sorted matrix in CSC format has no duplicate entries
      ! (adapted from SPARSKIT2 (Mar 8 2005) 'csorted' by É. Canot)
      !-----------------------------------------------------------------
      ! On entry:
      !--------------
      ! ncol    = number of cols in matrix
      ! ia, ja  = sparsity structure of matrix in CSC format
      !
      ! On return:
      !---------------
      ! bool = TRUE : indicates that matrix has no duplicate entries.
      !
      ! note: the matrix must be 'row sorted'
      !-----------------------------------------------------------------

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      do j = 1, ncol
         do i = ja(j)+1, ja(j+1)-1
            if( ia(i-1) == ia(i) ) then
               bool = .false.
               return
            end if
         end do
      end do
      bool = .true.

   end function unique_entries
!_______________________________________________________________________
!
   subroutine remove_dupl_entries( ncol, a, ia, ja, nb_removed )

      integer, intent(in)  :: ncol
      real(kind=MF_DOUBLE) :: a(:)
      integer              :: ia(:), ja(:)
      integer, intent(out) :: nb_removed
      !------ API end ------

      !-----------------------------------------------------------------
      ! Remove duplicated entries in a sorted matrix in CSC format
      ! (É. Canot -- March 2009) (improved version 2015-03-31)
      !-----------------------------------------------------------------
      ! On entry:
      !--------------
      ! ncol    = number of cols in matrix
      ! a, ia, ja  = matrix in CSC format
      !
      ! note: the matrix must be 'row sorted';
      !       duplicated entries are not added: first found is kept,
      !       others are discarded.
      !-----------------------------------------------------------------

      integer :: i, j, k, off, rem
      integer :: offset(ncol)

   !------ end of declarations -- execution starts hereafter  ------

      ! loop over columns
      do j = 1, ncol
         rem = (ja(j+1)-1) - ja(j)
         offset(j) = 0
         if( rem <= 0 ) cycle ! if 0 or 1 elem
         i = ja(j) + 1 ! starting with the second element
         rem = rem - 1
         ! loop inside column j
         do
            if( ia(i-1) == ia(i) ) then
               ! must remove i, without using a(i)
               offset(j) = offset(j) + 1
               ! update of current col; update for others is delayed
               do k = i, i+rem-1
                  ia(k) = ia(k+1)
                  a(k)  = a(k+1)
               end do
            else
               i = i + 1
            end if
            if( rem == 0 ) exit
            rem = rem - 1
         end do
      end do

      ! cumulate the offsets
      do j = 2, ncol
         offset(j) = offset(j) + offset(j-1)
      end do
      nb_removed = offset(ncol)

      ! shift the columns' chunks
      do j = 2, ncol
         off = offset(j-1)
         if( off /= 0 ) then
            do k = ja(j), ja(j+1)-1
               ia(k-off) = ia(k)
               a(k-off)  = a(k)
            end do
            ja(j) = ja(j) - off
         end if
      end do
      ja(ncol+1) = ja(ncol+1) - nb_removed

   end subroutine remove_dupl_entries
!_______________________________________________________________________
!
   subroutine remove_dupl_entries_cmplx( ncol, a, ia, ja, nb_removed )

      integer, intent(in)     :: ncol
      complex(kind=MF_DOUBLE) :: a(:)
      integer                 :: ia(:), ja(:)
      integer, intent(out)    :: nb_removed
      !------ API end ------

      !-----------------------------------------------------------------
      ! complex version of 'remove_dupl_entries'
      ! (improved version 2015-03-31)
      !-----------------------------------------------------------------

      integer :: i, j, k, off, rem
      integer :: offset(ncol)

   !------ end of declarations -- execution starts hereafter  ------

      ! loop over columns
      do j = 1, ncol
         rem = (ja(j+1)-1) - ja(j)
         offset(j) = 0
         if( rem <= 0 ) cycle ! if 0 or 1 elem
         i = ja(j) + 1 ! starting with the second element
         rem = rem - 1
         ! loop inside column j
         do
            if( ia(i-1) == ia(i) ) then
               ! must remove i, without using a(i)
               offset(j) = offset(j) + 1
               ! update of current col; update for others is delayed
               do k = i, i+rem-1
                  ia(k) = ia(k+1)
                  a(k)  = a(k+1)
               end do
            else
               i = i + 1
            end if
            if( rem == 0 ) exit
            rem = rem - 1
         end do
      end do

      ! cumulate the offsets
      do j = 2, ncol
         offset(j) = offset(j) + offset(j-1)
      end do
      nb_removed = offset(ncol)

      ! shift the columns' chunks
      do j = 2, ncol
         off = offset(j-1)
         if( off /= 0 ) then
            do k = ja(j), ja(j+1)-1
               ia(k-off) = ia(k)
               a(k-off)  = a(k)
            end do
            ja(j) = ja(j) - off
         end if
      end do
      ja(ncol+1) = ja(ncol+1) - nb_removed

   end subroutine remove_dupl_entries_cmplx
!_______________________________________________________________________
!
   subroutine remove_dupl_entries_struct_only( ncol, ia, ja, nb_removed )

      integer, intent(in)  :: ncol
      integer              :: ia(:), ja(:)
      integer, intent(out) :: nb_removed
      !------ API end ------

      !-----------------------------------------------------------------
      ! (as remove_dupl_entries, but for structure only)
      ! (improved version 2015-03-31)
      !-----------------------------------------------------------------
      ! On entry:
      !--------------
      ! ncol    = number of cols in matrix
      ! ia, ja  = matrix in CSC format
      !
      ! note: the matrix must be 'row sorted';
      !       duplicated entries are removed.
      !-----------------------------------------------------------------

      integer :: i, j, k, off, rem
      integer :: offset(ncol)

   !------ end of declarations -- execution starts hereafter  ------

      ! loop over columns
      do j = 1, ncol
         rem = (ja(j+1)-1) - ja(j)
         offset(j) = 0
         if( rem <= 0 ) cycle ! if 0 or 1 elem
         i = ja(j) + 1 ! starting with the second element
         rem = rem - 1
         ! loop inside column j
         do
            if( ia(i-1) == ia(i) ) then
               ! must remove i, without using a(i)
               offset(j) = offset(j) + 1
               ! update of current col; update for others is delayed
               do k = i, i+rem-1
                  ia(k) = ia(k+1)
               end do
            else
               i = i + 1
            end if
            if( rem == 0 ) exit
            rem = rem - 1
         end do
      end do

      ! cumulate the offsets
      do j = 2, ncol
         offset(j) = offset(j) + offset(j-1)
      end do
      nb_removed = offset(ncol)

      ! shift the columns' chunks
      do j = 2, ncol
         off = offset(j-1)
         if( off /= 0 ) then
            do k = ja(j), ja(j+1)-1
               ia(k-off) = ia(k)
            end do
            ja(j) = ja(j) - off
         end if
      end do
      ja(ncol+1) = ja(ncol+1) - nb_removed

   end subroutine remove_dupl_entries_struct_only
!_______________________________________________________________________
!
   subroutine add_dupl_entries( ncol, a, ia, ja, nb_added )

      integer,              intent(in)     :: ncol
      real(kind=MF_DOUBLE), intent(in out) :: a(:)
      integer,              intent(in out) :: ia(:), ja(:)
      integer,              intent(out)    :: nb_added
      !------ API end ------

      !-----------------------------------------------------------------
      ! Add duplicated entries in a sorted matrix in CSC format
      ! (É. Canot -- July 2011) (improved version 2015-03-31)
      !-----------------------------------------------------------------
      ! On entry:
      !--------------
      ! ncol    = number of cols in matrix
      ! a, ia, ja  = matrix in CSC format
      !
      ! note: the matrix must be 'row sorted';
      !       duplicated entries are added.
      !-----------------------------------------------------------------

      integer :: i, j, k, off, rem
      integer :: offset(ncol)

   !------ end of declarations -- execution starts hereafter  ------

      ! loop over columns
      do j = 1, ncol
         rem = (ja(j+1)-1) - ja(j)
         offset(j) = 0
         if( rem <= 0 ) cycle ! if 0 or 1 elem
         i = ja(j) + 1 ! starting with the second element
         rem = rem - 1
         ! loop inside column j
         do
            if( ia(i-1) == ia(i) ) then
               ! must remove i but must add corresponding value
               ! to previous element
               offset(j) = offset(j) + 1
               a(i-1) = a(i-1) + a(i)
               ! update of current col; update for others is delayed
               do k = i, i+rem-1
                  ia(k) = ia(k+1)
                  a(k)  = a(k+1)
               end do
            else
               i = i + 1
            end if
            if( rem == 0 ) exit
            rem = rem - 1
         end do
      end do

      ! cumulate the offsets
      do j = 2, ncol
         offset(j) = offset(j) + offset(j-1)
      end do
      nb_added = offset(ncol)

      ! shift the columns' chunks
      do j = 2, ncol
         off = offset(j-1)
         if( off /= 0 ) then
            do k = ja(j), ja(j+1)-1
               ia(k-off) = ia(k)
               a(k-off)  = a(k)
            end do
            ja(j) = ja(j) - off
         end if
      end do
      ja(ncol+1) = ja(ncol+1) - nb_added

   end subroutine add_dupl_entries
!_______________________________________________________________________
!
   subroutine add_dupl_entries_cmplx( ncol, a, ia, ja, nb_added )

      integer,                 intent(in)     :: ncol
      complex(kind=MF_DOUBLE), intent(in out) :: a(:)
      integer,                 intent(in out) :: ia(:), ja(:)
      integer,                 intent(out)    :: nb_added
      !------ API end ------

      !-----------------------------------------------------------------
      ! complex version of 'add_dupl_entries'
      ! (improved version 2015-03-31)
      !-----------------------------------------------------------------

      integer :: i, j, k, off, rem
      integer :: offset(ncol)

   !------ end of declarations -- execution starts hereafter  ------

      ! loop over columns
      do j = 1, ncol
         rem = (ja(j+1)-1) - ja(j)
         offset(j) = 0
         if( rem <= 0 ) cycle ! if 0 or 1 elem
         i = ja(j) + 1 ! starting with the second element
         rem = rem - 1
         ! loop inside column j
         do
            if( ia(i-1) == ia(i) ) then
               ! must remove i but must add corresponding value
               ! to previous element
               offset(j) = offset(j) + 1
               a(i-1) = a(i-1) + a(i)
               ! update of current col; update for others is delayed
               do k = i, i+rem-1
                  ia(k) = ia(k+1)
                  a(k)  = a(k+1)
               end do
            else
               i = i + 1
            end if
            if( rem == 0 ) exit
            rem = rem - 1
         end do
      end do

      ! cumulate the offsets
      do j = 2, ncol
         offset(j) = offset(j) + offset(j-1)
      end do
      nb_added = offset(ncol)

      ! shift the columns' chunks
      do j = 2, ncol
         off = offset(j-1)
         if( off /= 0 ) then
            do k = ja(j), ja(j+1)-1
               ia(k-off) = ia(k)
               a(k-off)  = a(k)
            end do
            ja(j) = ja(j) - off
         end if
      end do
      ja(ncol+1) = ja(ncol+1) - nb_added

   end subroutine add_dupl_entries_cmplx
!_______________________________________________________________________
!
   subroutine replace_dupl_entries( ncol, a, ia, ja, nb_replaced )

      integer, intent(in) :: ncol
      real(kind=MF_DOUBLE), intent(in out) :: a(:)
      integer, intent(in out) :: ia(:), ja(:)
      integer, intent(out) :: nb_replaced
      !------ API end ------

      !-----------------------------------------------------------------
      ! Replace duplicated entries in a sorted matrix in CSC format
      ! (É. Canot -- Sept 2011) (improved version 2015-03-31)
      !-----------------------------------------------------------------
      ! On entry:
      !--------------
      ! ncol    = number of cols in matrix
      ! a, ia, ja  = matrix in CSC format
      !
      ! note: the matrix must be 'row sorted';
      !       duplicated entries are replaced (last entries overwrite
      !       previous ones).
      !-----------------------------------------------------------------

      integer :: i, j, k, off, rem
      integer :: offset(ncol)

   !------ end of declarations -- execution starts hereafter  ------

      ! loop over columns
      do j = 1, ncol
         rem = (ja(j+1)-1) - ja(j)
         offset(j) = 0
         if( rem <= 0 ) cycle ! if 0 or 1 elem
         i = ja(j) + 1 ! starting with the second element
         rem = rem - 1
         ! loop inside column j
         do
            if( ia(i-1) == ia(i) ) then
               ! must remove i but must replace the previous element
               ! by the corresponding value
               offset(j) = offset(j) + 1
               a(i-1) = a(i)
               ! update of current col; update for others is delayed
               do k = i, i+rem-1
                  ia(k) = ia(k+1)
                  a(k)  = a(k+1)
               end do
            else
               i = i + 1
            end if
            if( rem == 0 ) exit
            rem = rem - 1
         end do
      end do

      ! cumulate the offsets
      do j = 2, ncol
         offset(j) = offset(j) + offset(j-1)
      end do
      nb_replaced = offset(ncol)

      ! shift the columns' chunks
      do j = 2, ncol
         off = offset(j-1)
         if( off /= 0 ) then
            do k = ja(j), ja(j+1)-1
               ia(k-off) = ia(k)
               a(k-off)  = a(k)
            end do
            ja(j) = ja(j) - off
         end if
      end do
      ja(ncol+1) = ja(ncol+1) - nb_replaced

   end subroutine replace_dupl_entries
!_______________________________________________________________________
!
   subroutine replace_dupl_entries_cmplx( ncol, a, ia, ja, nb_replaced )

      integer, intent(in) :: ncol
      complex(kind=MF_DOUBLE), intent(in out) :: a(:)
      integer, intent(in out) :: ia(:), ja(:)
      integer, intent(out) :: nb_replaced
      !------ API end ------

      !-----------------------------------------------------------------
      ! complex version of 'replace_dupl_entries'
      ! (improved version 2015-03-31)
      !-----------------------------------------------------------------

      integer :: i, j, k, off, rem
      integer :: offset(ncol)

   !------ end of declarations -- execution starts hereafter  ------

      ! loop over columns
      do j = 1, ncol
         rem = (ja(j+1)-1) - ja(j)
         offset(j) = 0
         if( rem <= 0 ) cycle ! if 0 or 1 elem
         i = ja(j) + 1 ! starting with the second element
         rem = rem - 1
         ! loop inside column j
         do
            if( ia(i-1) == ia(i) ) then
               ! must remove i but must replace the previous element
               ! by the corresponding value
               offset(j) = offset(j) + 1
               a(i-1) = a(i)
               ! update of current col; update for others is delayed
               do k = i, i+rem-1
                  ia(k) = ia(k+1)
                  a(k)  = a(k+1)
               end do
            else
               i = i + 1
            end if
            if( rem == 0 ) exit
            rem = rem - 1
         end do
      end do

      ! cumulate the offsets
      do j = 2, ncol
         offset(j) = offset(j) + offset(j-1)
      end do
      nb_replaced = offset(ncol)

      ! shift the columns' chunks
      do j = 2, ncol
         off = offset(j-1)
         if( off /= 0 ) then
            do k = ja(j), ja(j+1)-1
               ia(k-off) = ia(k)
               a(k-off)  = a(k)
            end do
            ja(j) = ja(j) - off
         end if
      end do
      ja(ncol+1) = ja(ncol+1) - nb_replaced

   end subroutine replace_dupl_entries_cmplx
!_______________________________________________________________________
!
   subroutine row_sort( nrow, ncol, a, ia, ja)

      integer,              intent(in)     :: nrow, ncol
      integer,              intent(in out) :: ia(:), ja(:)
      real(kind=MF_DOUBLE), intent(in out) :: a(:)
      !------ API end ------

      !-----------------------------------------------------------------
      ! This routine sorts the elements of a matrix (stored in CSC
      ! format) in increasing order of their row indexes within each
      ! col. It uses a form of bucket sort with a cost of O(nnz) where
      ! nnz is the number of nonzero elements.
      ! (adapted from SPARSKIT2 (Mar 8 2005) 'csort' by É. Canot)
      ! (few bugs fixed -- Mar 2010)
      !-----------------------------------------------------------------
      ! on entry:
      !---------
      ! nrow  = the row number of the matrix
      ! ncol  = the col number of the matrix
      ! a     = the matrix A in CSC format.
      ! ia    = the array of row indexes of the elements in array a.
      ! ja    = the array of pointers to the rows.
      !
      ! note: input matrix may have many duplicated entries!
      !
      ! on return:
      !----------
      ! the matrix stored in the structure a, ia, ja is permuted in such
      ! a way that the row indexes are in increasing order within each
      ! col.
      !
      ! note: doesn't remove duplicated entries.
      !-----------------------------------------------------------------

      !   iwork(1:nnz)  contains the permutation used to rearrange the
      !                 elements; but this array must have a size
      !                 greater (see below).
      integer, allocatable :: iwork(:)

      integer :: i, k, j, jfirst, nnz, next, jcol, ko
      integer :: i_shift

   !------ end of declarations -- execution starts hereafter  ------

      jfirst = ja(1)
      if( jfirst /= 1 ) then
         write(STDERR,*) "(MUESLI row_sort:) internal error :"
         write(STDERR,*) "                   ja(1) must be equal to 1!"
         mf_message_displayed = .true.
         call muesli_trace( pause="yes" )
         stop
      end if
      ! in original routine jfirst may be /= 1
      ! we should replace all '1' by 'jfirst' !
      nnz = ja(ncol+1)-1
      i_shift = max(nrow+1,nnz)
      allocate( iwork(i_shift+nnz) )

      ! count the number of elements in each row
      do i = 1, nrow+1
         iwork(i) = 0
      end do
      do j = 1, ncol
         do k = ja(j), ja(j+1)-1
            i = ia(k) + 1
            iwork(i) = iwork(i) + 1
         end do
      end do
      ! compute pointers from lengths.
      iwork(1) = 1
      do i = 1, nrow
         iwork(i+1) = iwork(i) + iwork(i+1)
      end do

      ! get the positions of the nonzero elements in order of rows.
      do j = 1, ncol
         do k = ja(j), ja(j+1)-1
            i = ia(k)
            next = iwork(i)
            iwork(i_shift+next) = k
            iwork(i) = next + 1
         end do
      end do

      ! convert to coordinate format
      do j = 1, ncol
         do k = ja(j), ja(j+1)-1
            iwork(k) = j
         end do
      end do

      ! loop to find permutation: for each element find the correct
      ! position in (sorted) arrays a, ia. Record this in iwork.
      do k = 1, nnz
         ko = iwork(i_shift+k)
         jcol = iwork(ko)
         next = ja(jcol)

         ! the current element should go in next position in col.
         ! iwork records this position.
         iwork(ko) = next
         ja(jcol)  = next + 1
      end do

      ! perform an in-place inverse permutation of the arrays.
      call ivpinv( nnz, ia, iwork )
      call dvpinv( nnz, a, iwork )

      ! reshift the pointers of the original matrix back.
      do j = ncol, 1, -1
         ja(j+1) = ja(j)
      end do
      ja(1) = 1

   end subroutine row_sort
!_______________________________________________________________________
!
   subroutine row_sort_cmplx( nrow, ncol, a, ia, ja)

      integer,                 intent(in)     :: nrow, ncol
      integer,                 intent(in out) :: ia(:), ja(:)
      complex(kind=MF_DOUBLE), intent(in out) :: a(:)
      !------ API end ------

      !-----------------------------------------------------------------
      ! complex version of 'row_sort'
      !-----------------------------------------------------------------

      integer, allocatable :: iwork(:)

      integer :: i, k, j, jfirst, nnz, next, jcol, ko
      integer :: i_shift

   !------ end of declarations -- execution starts hereafter  ------

      jfirst = ja(1)
      if( jfirst /= 1 ) then
         write(STDERR,*) "(MUESLI row_sort:) internal error :"
         write(STDERR,*) "                   ja(1) must be equal to 1!"
         mf_message_displayed = .true.
         call muesli_trace( pause="yes" )
         stop
      end if
      ! in original routine jfirst may be /= 1
      ! we should replace all '1' by 'jfirst' !
      nnz = ja(ncol+1)-1
      i_shift = max(nrow+1,nnz)
      allocate( iwork(i_shift+nnz) )

      ! count the number of elements in each row
      do i = 1, nrow+1
         iwork(i) = 0
      end do
      do j = 1, ncol
         do k = ja(j), ja(j+1)-1
            i = ia(k) + 1
            iwork(i) = iwork(i) + 1
         end do
      end do
      ! compute pointers from lengths.
      iwork(1) = 1
      do i = 1, nrow
         iwork(i+1) = iwork(i) + iwork(i+1)
      end do

      ! get the positions of the nonzero elements in order of rows.
      do j = 1, ncol
         do k = ja(j), ja(j+1)-1
            i = ia(k)
            next = iwork(i)
            iwork(i_shift+next) = k
            iwork(i) = next + 1
         end do
      end do

      ! convert to coordinate format
      do j = 1, ncol
         do k = ja(j), ja(j+1)-1
            iwork(k) = j
         end do
      end do

      ! loop to find permutation: for each element find the correct
      ! position in (sorted) arrays a, ia. Record this in iwork.
      do k = 1, nnz
         ko = iwork(i_shift+k)
         jcol = iwork(ko)
         next = ja(jcol)

         ! the current element should go in next position in col.
         ! iwork records this position.
         iwork(ko) = next
         ja(jcol)  = next + 1
      end do

      ! perform an in-place inverse permutation of the arrays.
      call ivpinv( nnz, ia, iwork )
      call zvpinv( nnz, a, iwork )

      ! reshift the pointers of the original matrix back.
      do j = ncol, 1, -1
         ja(j+1) = ja(j)
      end do
      ja(1) = 1

   end subroutine row_sort_cmplx
!_______________________________________________________________________
!
   subroutine row_sort_struct_only( nrow, ncol, ia, ja)

      integer,              intent(in)     :: nrow, ncol
      integer,              intent(in out) :: ia(:), ja(:)
      !------ API end ------

      !-----------------------------------------------------------------
      ! (as row_sort, but for the structure only)
      !-----------------------------------------------------------------
      ! on entry:
      !---------
      ! nrow  = the row number of the matrix
      ! ncol  = the col number of the matrix
      ! ia    = the array of row indexes of the elements in array a.
      ! ja    = the array of pointers to the rows.
      !
      ! on return:
      !----------
      ! the matrix stored in the structure ia, ja is permuted in such
      ! a way that the row indexes are in increasing order within each
      ! col.
      !
      ! note: doesn't remove duplicated entries.
      !-----------------------------------------------------------------

      ! iwork(1:nnz) contains the permutation used to rearrange the
      ! elements.
      integer, allocatable :: iwork(:)

      integer :: i, k, j, jfirst, nnz, next, jcol, ko
      integer :: i_shift

   !------ end of declarations -- execution starts hereafter  ------

      jfirst = ja(1)
      if( jfirst /= 1 ) then
         write(STDERR,*) "(MUESLI row_sort:) internal error :"
         write(STDERR,*) "                   ja(1) must be equal to 1!"
         mf_message_displayed = .true.
         call muesli_trace( pause="yes" )
         stop
      end if
      ! in original routine jfirst may be /= 1
      ! we should replace all '1' by 'jfirst' !
      nnz = ja(ncol+1)-1
      i_shift = max(nrow+1,nnz)
      allocate( iwork(i_shift+nnz) )

      ! count the number of elements in each row
      do i = 1, nrow+1
         iwork(i) = 0
      end do
      do j = 1, ncol
         do k = ja(j), ja(j+1)-1
            i = ia(k) + 1
            iwork(i) = iwork(i) + 1
         end do
      end do
      ! compute pointers from lengths.
      iwork(1) = 1
      do i = 1, nrow
         iwork(i+1) = iwork(i) + iwork(i+1)
      end do

      ! get the positions of the nonzero elements in order of rows.
      do j = 1, ncol
         do k = ja(j), ja(j+1)-1
            i = ia(k)
            next = iwork(i)
            iwork(i_shift+next) = k
            iwork(i) = next + 1
         end do
      end do

      ! convert to coordinate format
      do j = 1, ncol
         do k = ja(j), ja(j+1)-1
            iwork(k) = j
         end do
      end do

      ! loop to find permutation: for each element find the correct
      ! position in (sorted) arrays a, ia. Record this in iwork.
      do k = 1, nnz
         ko = iwork(i_shift+k)
         jcol = iwork(ko)
         next = ja(jcol)

         ! the current element should go in next position in col.
         ! iwork records this position.
         iwork(ko) = next
         ja(jcol)  = next + 1
      end do

      ! perform an in-place inverse permutation of the array.
      call ivpinv( nnz, ia, iwork )

      ! reshift the pointers of the original matrix back.
      do j = ncol, 1, -1
         ja(j+1) = ja(j)
      end do
      ja(1) = 1

   end subroutine row_sort_struct_only
!_______________________________________________________________________
!
   subroutine ivpinv( n, ix, pinv )

      integer :: n, pinv(:), ix(:)
      !------ API end ------

      !-----------------------------------------------------------------------
      ! this subroutine performs an in-place permutation of an integer vector
      ! ix according to the inverse permutation array pinv(*), i.e., on return,
      ! the vector x satisfies,
      !
      ! ix(pinv(j)) := ix(j), j=1,2,.., n
      !
      !-----------------------------------------------------------------------
      ! on entry:
      !---------
      ! n  = length of vector x.
      ! pinv  = integer array of length n containing the permutation array.
      ! ix = input vector
      !
      ! on return:
      !----------
      ! ix = vector x permuted according to ix(pinv(*)) :=  ix(*)
      !
      !-----------------------------------------------------------------------
      ! (from ivperm of SPARSKIT2: Y. Saad, Sep. 21 1989)
      !-----------------------------------------------------------------------
      ! local variables
      integer :: tmp, tmp1
      integer :: init, ii, k, next, j

   !------ end of declarations -- execution starts hereafter  ------

      init = 1
      tmp = ix(init)
      ii = pinv(init)
      pinv(init) = -pinv(init)
      k = 0

      ! loop
 6    k = k+1

      ! save the chased element
      tmp1 = ix(ii)
      ix(ii) = tmp
      next = pinv(ii)
      if( next < 0 ) goto 65

      ! test for end
      if( k > n ) goto 101
      tmp = tmp1
      pinv(ii) = - pinv(ii)
      ii = next

      ! end loop
      goto 6

      ! reinitialize cycle
 65   init = init+1
      if(init > n) goto 101
      if(pinv(init) < 0) goto 65
      tmp = ix(init)
      ii = pinv(init)
      pinv(init) = -pinv(init)
      goto 6

 101  continue
      do j = 1, n
         pinv(j) = -pinv(j)
      end do

   end subroutine ivpinv
!_______________________________________________________________________
!
   subroutine dvpinv(n, x, pinv)

      integer :: n, pinv(:)
      real(kind=MF_DOUBLE) :: x(:)
      !------ API end ------

      !-----------------------------------------------------------------------
      ! real version of 'ivpinv'
      !-----------------------------------------------------------------------
      ! local variables
      real(kind=MF_DOUBLE) :: tmp, tmp1
      integer :: init, ii, k, next, j

   !------ end of declarations -- execution starts hereafter  ------

      init = 1
      tmp = x(init)
      ii = pinv(init)
      pinv(init) = -pinv(init)
      k = 0

      ! loop
 6    k = k+1

      ! save the chased element
      tmp1 = x(ii)
      x(ii) = tmp
      next = pinv(ii)
      if( next < 0 ) goto 65

      ! test for end
      if( k > n ) goto 101
      tmp = tmp1
      pinv(ii) = - pinv(ii)
      ii = next

      ! end loop
      goto 6

      ! reinitialize cycle
 65   init = init+1
      if(init > n) goto 101
      if(pinv(init) < 0) goto 65
      tmp = x(init)
      ii = pinv(init)
      pinv(init) = -pinv(init)
      goto 6

 101  continue
      do j = 1, n
         pinv(j) = -pinv(j)
      end do

   end subroutine dvpinv
!_______________________________________________________________________
!
   subroutine zvpinv(n, x, pinv)

      integer :: n, pinv(:)
      complex(kind=MF_DOUBLE) :: x(:)
      !------ API end ------

      !-----------------------------------------------------------------------
      ! complex version of 'dvpinv'
      !-----------------------------------------------------------------------
      ! local variables
      complex(kind=MF_DOUBLE) :: tmp, tmp1
      integer :: init, ii, k, next, j

   !------ end of declarations -- execution starts hereafter  ------

      init = 1
      tmp = x(init)
      ii = pinv(init)
      pinv(init) = -pinv(init)
      k = 0

      ! loop
 6    k = k+1

      ! save the chased element
      tmp1 = x(ii)
      x(ii) = tmp
      next = pinv(ii)
      if( next < 0 ) goto 65

      ! test for end
      if( k > n ) goto 101
      tmp = tmp1
      pinv(ii) = - pinv(ii)
      ii = next

      ! end loop
      goto 6

      ! reinitialize cycle
 65   init = init+1
      if(init > n) goto 101
      if(pinv(init) < 0) goto 65
      tmp = x(init)
      ii = pinv(init)
      pinv(init) = -pinv(init)
      goto 6

 101  continue
      do j = 1, n
         pinv(j) = -pinv(j)
      end do

   end subroutine zvpinv
!_______________________________________________________________________
!
   subroutine matcolperm(m, n, mat, perm)

      integer, intent(in) :: m, n
      integer :: perm(:)
      real(kind=MF_DOUBLE) :: mat(:,:)
      !------ API end ------

      !-----------------------------------------------------------------------
      ! this subroutine performs an in-place permutation of the columns
      ! of a real matrix mat
      ! according to the permutation array perm(*), i.e., on return,
      ! the matrix mat satisfies,
      !
      ! mat(:,j) := mat(:,perm(j)), j = 1, 2, ..., n
      !
      !-----------------------------------------------------------------------
      ! on entry:
      !---------
      ! m, n  = nb of rows and columns of matrix mat.
      ! perm  = integer array of length n containing the permutation.
      ! mat = input matrix
      !
      ! on return:
      !----------
      ! mat = output matrix
      !
      !-----------------------------------------------------------------------
      ! É. Canot, Feb. 15 2012
      !-----------------------------------------------------------------------
      ! local variables
      real(kind=MF_DOUBLE), allocatable :: val_tmp_1(:), val_tmp_2(:)
      integer, allocatable :: pinv(:)
      integer :: i

   !------ end of declarations -- execution starts hereafter  ------

      ! first compute the inverse permutation
      allocate( pinv(n) )
      pinv(perm(:)) = [ ( i, i = 1, n ) ]

      allocate( val_tmp_1(m), val_tmp_2(m) )

      i = 1
      val_tmp_1(:) = mat(:,i)
loop: do
         if ( pinv(i) < 0 ) then
            ! we search the first next which is non negative
            do
               i = i + 1
               if( i == n+1 ) exit loop
               if( pinv(i) > 0 ) then
                  val_tmp_1(:) = mat(:,i)
                  exit
               else
                  cycle
               end if
            end do
         else
            if( i /= pinv(i) ) then
               ! column i moves to pinv(i)
               val_tmp_2(:) = mat(:,pinv(i))
               mat(:,pinv(i)) = val_tmp_1(:)
               pinv(i) = -pinv(i) ! mark this value in order to not reuse it
               i = -pinv(i)
               val_tmp_1(:) = val_tmp_2(:)
            else
               ! no move : just mark pinv(i)
               mat(:,pinv(i)) = val_tmp_1(:)
               pinv(i) = -pinv(i) ! mark this value in order to not reuse it
            end if
         end if
      end do loop

   end subroutine matcolperm
!_______________________________________________________________________
!
   subroutine matcolperm_cmplx(m, n, mat, perm)

      integer, intent(in) :: m, n
      integer :: perm(:)
      complex(kind=MF_DOUBLE) :: mat(:,:)
      !------ API end ------

      !-----------------------------------------------------------------------
      ! complex version of matcolperm
      !-----------------------------------------------------------------------
      ! local variables
      complex(kind=MF_DOUBLE), allocatable :: val_tmp_1(:), val_tmp_2(:)
      integer, allocatable :: pinv(:)
      integer :: i

   !------ end of declarations -- execution starts hereafter  ------

      ! first compute the inverse permutation
      allocate( pinv(n) )
      pinv(perm(:)) = [ ( i, i = 1, n ) ]

      allocate( val_tmp_1(m), val_tmp_2(m) )

      i = 1
      val_tmp_1(:) = mat(:,i)
loop: do
         if ( pinv(i) < 0 ) then
            ! we search the first next which is non negative
            do
               i = i + 1
               if( i == n+1 ) exit loop
               if( pinv(i) > 0 ) then
                  val_tmp_1(:) = mat(:,i)
                  exit
               else
                  cycle
               end if
            end do
         else
            if( i /= pinv(i) ) then
               ! column i moves to pinv(i)
               val_tmp_2(:) = mat(:,pinv(i))
               mat(:,pinv(i)) = val_tmp_1(:)
               pinv(i) = -pinv(i) ! mark this value in order to not reuse it
               i = -pinv(i)
               val_tmp_1(:) = val_tmp_2(:)
            else
               ! no move : just mark pinv(i)
               mat(:,pinv(i)) = val_tmp_1(:)
               pinv(i) = -pinv(i) ! mark this value in order to not reuse it
            end if
         end if
      end do loop

   end subroutine matcolperm_cmplx
!_______________________________________________________________________
!
   subroutine matrowperm(m, n, mat, perm)

      integer, intent(in) :: m, n
      integer :: perm(:)
      real(kind=MF_DOUBLE) :: mat(:,:)
      !------ API end ------

      !-----------------------------------------------------------------------
      ! this subroutine performs an in-place permutation of the rows
      ! of a real matrix mat
      ! according to the permutation array perm(*), i.e., on return,
      ! the matrix mat satisfies,
      !
      ! mat(j,:) := mat(perm(j),:), j = 1, 2, ..., m
      !
      !-----------------------------------------------------------------------
      ! on entry:
      !---------
      ! m, n  = nb of rows and columns of matrix mat.
      ! perm  = integer array of length n containing the permutation.
      ! mat = input matrix
      !
      ! on return:
      !----------
      ! mat = output matrix
      !
      !-----------------------------------------------------------------------
      ! É. Canot, Feb. 15 2012
      !-----------------------------------------------------------------------
      ! local variables
      real(kind=MF_DOUBLE), allocatable :: val_tmp_1(:), val_tmp_2(:)
      integer, allocatable :: pinv(:)
      integer :: i

   !------ end of declarations -- execution starts hereafter  ------

      ! first compute the inverse permutation
      allocate( pinv(m) )
      pinv(perm(:)) = [ ( i, i = 1, m ) ]

      allocate( val_tmp_1(n), val_tmp_2(n) )

      i = 1
      val_tmp_1(:) = mat(i,:)
loop: do
         if ( pinv(i) < 0 ) then
            ! we search the first next which is non negative
            do
               i = i + 1
               if( i == m+1 ) exit loop
               if( pinv(i) > 0 ) then
                  val_tmp_1(:) = mat(i,:)
                  exit
               else
                  cycle
               end if
            end do
         else
            if( i /= pinv(i) ) then
               ! row i moves to pinv(i)
               val_tmp_2(:) = mat(pinv(i),:)
               mat(pinv(i),:) = val_tmp_1(:)
               pinv(i) = -pinv(i) ! mark this value in order to not reuse it
               i = -pinv(i)
               val_tmp_1(:) = val_tmp_2(:)
            else
               ! no move : just mark pinv(i)
               mat(pinv(i),:) = val_tmp_1(:)
               pinv(i) = -pinv(i) ! mark this value in order to not reuse it
            end if
         end if
      end do loop

   end subroutine matrowperm
!_______________________________________________________________________
!
   subroutine matrowperm_cmplx(m, n, mat, perm)

      integer, intent(in) :: m, n
      integer :: perm(:)
      complex(kind=MF_DOUBLE) :: mat(:,:)
      !------ API end ------

      !-----------------------------------------------------------------------
      ! complex version of matrowperm
      !-----------------------------------------------------------------------
      ! local variables
      complex(kind=MF_DOUBLE), allocatable :: val_tmp_1(:), val_tmp_2(:)
      integer, allocatable :: pinv(:)
      integer :: i

   !------ end of declarations -- execution starts hereafter  ------

      ! first compute the inverse permutation
      allocate( pinv(m) )
      pinv(perm(:)) = [ ( i, i = 1, m ) ]

      allocate( val_tmp_1(n), val_tmp_2(n) )

      i = 1
      val_tmp_1(:) = mat(i,:)
loop: do
         if ( pinv(i) < 0 ) then
            ! we search the first next which is non negative
            do
               i = i + 1
               if( i == m+1 ) exit loop
               if( pinv(i) > 0 ) then
                  val_tmp_1(:) = mat(i,:)
                  exit
               else
                  cycle
               end if
            end do
         else
            if( i /= pinv(i) ) then
               ! row i moves to pinv(i)
               val_tmp_2(:) = mat(pinv(i),:)
               mat(pinv(i),:) = val_tmp_1(:)
               pinv(i) = -pinv(i) ! mark this value in order to not reuse it
               i = -pinv(i)
               val_tmp_1(:) = val_tmp_2(:)
            else
               ! no move : just mark pinv(i)
               mat(pinv(i),:) = val_tmp_1(:)
               pinv(i) = -pinv(i) ! mark this value in order to not reuse it
            end if
         end if
      end do loop

   end subroutine matrowperm_cmplx
