! f90 include file

!_______________________________________________________________________
!
   subroutine readmt( nmax, nzmax, job, iounit, a, ja, ia, rhs, nrhs,   &
                      guesol, nrow, ncol, nnz, title, key, type, ierr )

      integer, intent(in) :: nmax, nzmax, iounit

      integer, intent(in out) :: job, nrhs

      integer, intent(out) :: ja(:), ia(:), nrow, ncol, nnz, ierr
      real(kind=MF_DOUBLE), intent(out) :: a(:), rhs(:)

      character(len=2), intent(out) :: guesol
      character(len=72), intent(out) :: title
      character(len=8), intent(out) :: key
      character(len=3), intent(out) :: type
      !------ API end ------

!-----------------------------------------------------------------------
! This subroutine reads a Harwell-Boeing matrix. Handles right hand
! sides in full format only (no sparse RHS). Also the matrix must be in
! assembled form.
! Author: Youcef Saad - Date: Sept., 1989
!         updated Oct 31, 1989
!         modified EC Sep 27, 2017
!-----------------------------------------------------------------------
! on entry:
!---------
! nmax  = max column dimension allowed for matrix. The array ia should
!         be of length at least ncol+1 (see below) if job>0
! nzmax  = max number of nonzeros elements allowed. the arrays a,
!          and ja should be of length equal to nnz (see below) if these
!          arrays are to be read (see job).
!
! job = integer to indicate what is to be read. (note: job is an
!          input and output parameter, it can be modified on return)
!          job = 0    read the values of nrow, ncol, nnz, title, key,
!                     type and return. matrix is not read and arrays
!                     a, ja, ia, rhs are not touched.
!          job = 1    read structure only, i.e., the arrays ja and ia.
!          job = 2    read matrix including values, i.e., a, ja, ia
!          job = 3    read matrix and right hand sides: a,ja,ia,rhs.
!        rhs may contain initial guesses and exact
!                     solutions appended to the actual right hand sides.
!        this will be indicated by the output parameter
!                     guesol [see below].
!
! nrhs   = integer. nrhs is an input as well as ouput parameter.
!          at input nrhs contains the total length of the array rhs.
!          See also ierr and nrhs in output parameters.
!
! iounit = logical unit number where to read the matrix from.
!
! on return:
!----------
! job    = on return job may be modified to the highest job it could
!          do: if job=2 on entry but no matrix values are available it
!          is reset to job=1 on return. Similarly of job=3 but no rhs
!          is provided then it is rest to job=2 or job=1 depending on
!          whether or not matrix values are provided.
!          Note that no error message is triggered (i.e. ierr = 0
!          on return in these cases. It is therefore important to
!          compare the values of job on entry and return ).
!
! a    = the a matrix in the a, ia, ja (column) storage format
! ja   = row number of element a(i,j) in array a.
! ia     = pointer  array. ia(i) points to the beginning of column i.
!
! rhs    = real array of size nrow + 1 if available (see job)
!
! nrhs   = integer containing the number of right-hand sides found
!          each right hand side may be accompanied with an initial guess
!          and also the exact solution.
!
! guesol = a 2-character string indicating whether an initial guess
!          (1-st character) and / or the exact solution (2-nd
!          character) is provided with the right hand side.
!    if the first character of guesol is 'G' it means that an
!          an initial guess is provided for each right-hand side.
!          These are appended to the right hand-sides in the array rhs.
!    if the second character of guesol is 'X' it means that an
!          exact solution is provided for each right-hand side.
!          These are  appended to the right hand-sides
!          and the initial guesses (if any) in the array rhs.
!
! nrow   = number of rows in matrix
! ncol  = number of columns in matrix
! nnz  = number of nonzero elements in A. This info is returned
!          even if there is not enough space in a, ja, ia, in order
!          to determine the minimum storage needed.
!
! title  = character*72 = title of matrix test ( character a*72).
! key    = character*8  = key of matrix
! type   = charatcer*3  = type of matrix.
!          for meaning of title, key and type refer to documentation
!          Harwell/Boeing matrices.
!
! ierr   = integer used for error messages
!         * ierr  =  0 means that the matrix has been read normally.
!         * ierr  =  1 means that the array matrix could not be read
!         because ncol+1 > nmax
!         * ierr  =  2 means that the array matrix could not be read
!         because nnz > nzmax
!         * ierr  =  3 means that the array matrix could not be read
!         because both (ncol+1 > nmax) and  (nnz > nzmax )
!         * ierr  =  4 means that the right hand side (s) initial
!         guesse (s) and exact solution (s)   could  not be
!         read because they are stored in sparse format (not handled
!         by this routine ...)
!         * ierr  =  5 means that the right-hand-sides, initial guesses
!         and exact solutions could not be read because the length of
!         rhs as specified by the input value of nrhs is not
!         sufficient to store them. The rest of the matrix may have
!         been read normally.
!
! Notes:
!-------
! 1) The file inout must be open (and possibly rewound if necessary)
!    prior to calling readmt.
! 2) Refer to the documentation on the Harwell-Boeing formats
!    for details on the format assumed by readmt.
!    We summarize the format here for convenience.
!
!    a) all lines in inout are assumed to be 80 character long.
!    b) the file consists of a header followed by the block of the
!       column start pointers followed by the block of the
!       row indexes, followed by the block of the real values and
!       finally the numerical values of the right-hand-side if a
!       right hand side is supplied.
!    c) the file starts by a header which contains four lines if no
!       right hand side is supplied and five lines otherwise.
!       * first line contains the title (72 characters long) followed by
!         the 8-character identifier (name of the matrix, called key)
!        [ A72,A8 ]
!       * second line contains the number of lines for each
!         of the following data blocks (4 of them) and the total number
!         of lines excluding the header.
!        [5i4]
!       * the third line contains a 3-character string identifying
!         the type of matrices as they are referenced in the Harwell
!         Boeing documentation [e.g., rua, rsa,..] and the number of
!         rows, columns, nonzero entries.
!         [A3,11X,4I14]
!       * The fourth line contains the variable fortran format
!         for the following data blocks.
!         [2A16,2A20]
!       * The fifth line is only present if right-hand-sides are
!         supplied. It consists of three one character-strings containing
!         the storage format for the right-hand-sides
!         ('F'= full,'M'=sparse=same as matrix), an initial guess
!         indicator ('G' for yes), an exact solution indicator
!         ('X' for yes), followed by the number of right-hand-sides
!         and then the number of row indexes.
!         [A3,11X,2I14]
!     d) The three following blocks follow the header as described
!        above.
!     e) In case the right hand-side are in sparse formats then
!        the fourth block uses the same storage format as for the matrix
!        to describe the NRHS right hand sides provided, with a column
!        being replaced by a right hand side.
!-----------------------------------------------------------------------

      character(len=16) :: ptrfmt, indfmt
      character(len=20) :: valfmt, rhsfmt
      character(len=3) :: rhstyp
      integer :: totcrd, ptrcrd, indcrd, valcrd, rhscrd, neltvl, nrwindx
      integer :: lenrhs, i, n, len, nvec, next, iend

   !------ end of declarations -- execution starts hereafter  ------

      ierr = 0
      lenrhs = nrhs

      read(iounit,"(a72,a8,/,5i14,/,a3,11x,4i14,/,2a16,2a20)")          &
           title, key, totcrd, ptrcrd, indcrd, valcrd, rhscrd, type,    &
           nrow, ncol, nnz, neltvl, ptrfmt, indfmt, valfmt, rhsfmt

      if( rhscrd > 0 ) then
         read (iounit,"(a3,11x,i14,i14)") rhstyp, nrhs, nrwindx
         guesol = rhstyp(2:3)
      end if

      ! anything else to read ?

      if( job <= 0 ) return
      ! check whether matrix is readable
      n = ncol
      if( ncol > nmax ) ierr = 1
      if( nnz > nzmax ) ierr = ierr + 2
      if( ierr /= 0 ) return
      ! read pointer and row numbers
      read(iounit,ptrfmt) (ia (i), i = 1, n+1)
      read(iounit,indfmt) (ja (i), i = 1, nnz)
      ! reading values of matrix if required...
      if( job <= 1 )  return
      ! and if available
      if( valcrd <= 0 ) then
         job = 1
         return
      end if
      read(iounit,valfmt) (a(i), i = 1, nnz)
      ! reading rhs if required...
      if( job <= 2 )  return
      ! and if available
      if( rhscrd <= 0 ) then
         job = 2
         nrhs = 0
         return
      end if

      ! read right-hand-side.
      if( to_upper(rhstyp(1:1)) == 'M' ) then
         ierr = 4
         return
      end if

      nvec = 1
      if( to_upper(guesol(1:1)) == 'G' ) nvec = nvec + 1
      if( to_upper(guesol(2:2)) == 'X' ) nvec = nvec + 1

      len = nrhs*nrow

      if( len*nvec > lenrhs ) then
         ierr = 5
         return
      end if

      ! read right-hand-sides
      next = 1
      iend = len
      read(iounit,rhsfmt) (rhs(i), i = next, iend)

      ! read initial guesses if available
      if( to_upper(guesol(1:1)) == 'G' ) then
         next = next+len
         iend = iend+ len
         read(iounit,valfmt) (rhs(i), i = next, iend)
      end if

      ! read exact solutions if available
      if( to_upper(guesol(2:2)) == 'X' ) then
         next = next+len
         iend = iend+ len
         read(iounit,valfmt) (rhs(i), i = next, iend)
      end if

   end subroutine readmt
